# SPDX-License-Identifier: AGPL-3.0-or-later
# Copyright (C) 2020-2022 Andrew Rechnitzer
# Copyright (C) 2020-2024 Colin B. Macdonald
# Copyright (C) 2022 Edith Coates

import hashlib
from io import BytesIO
import json

import requests
from requests_toolbelt import MultipartDecoder, MultipartEncoder

from plom import undo_json_packing_of_version_map
from plom.plom_exceptions import PlomBenignException, PlomSeriousException
from plom.plom_exceptions import (
    PlomAuthenticationException,
    PlomConflict,
    PlomDatabaseCreationError,
    PlomExistingDatabase,
    PlomNoMoreException,
    PlomNoPaper,
    PlomOwnersLoggedInException,
    PlomServerNotReady,
    PlomRangeException,
    PlomTakenException,
    PlomUnidentifiedPaperException,
)
from plom.baseMessenger import BaseMessenger


# TODO:
# _userName = "manager"


class ManagerMessenger(BaseMessenger):
    """Management-related communications."""

    def __init__(self, *args, **kwargs) -> None:
        super().__init__(*args, **kwargs)

    def InitialiseDB(self, version_map={}):
        """Instruct the server to initialise the DB (but not yet populate with tests).

        Returns:
            dict: the version map (if none supplied then generated by
                server, else the one supplied).  Note: you can change
                this version map if you wish before instantiating rows;
                its a *suggestion*, not a requirement.
                See ``plom/create/buildDatabaseAndPapers.py` for how
                this is currently used (which does not yet take advantage
                of the above flexibility, see e.g., Issues #1745, #1995).

        Raises:
            PlomExistingDatabase: already has a populated database.
            PlomServerNotReady: e.g., has no spec.
            PlomAuthenticationException: login troubles.
            PlomSeriousException: unexpected errors.
        """
        self.SRmutex.acquire()
        try:
            response = self.put(
                "/plom/admin/initialiseDB",
                json={
                    "user": self.user,
                    "token": self.token,
                    "version_map": version_map,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code == 409:
                raise PlomExistingDatabase(response.reason) from None
            if response.status_code == 400:
                raise PlomServerNotReady(response.reason) from None
            raise PlomSeriousException("Unexpected {}".format(e)) from None
        finally:
            self.SRmutex.release()

        # JSON casts dict keys to str, force back to ints
        return undo_json_packing_of_version_map(response.json())

    def appendTestToDB(self, test_number, vmap_for_test):
        """Ask server to append given test with the given versions.

        Returns:
            str: summary

        Raises:
            PlomAuthenticationException: login troubles.
            PlomServerNotReady: server has no spec.
            PlomDatabaseCreationError: problems with version map or spec.
            PlomConflict: row already exists or otherwise cannot create.
            PlomSeriousException: unexpected errors.
        """
        with self.SRmutex:
            try:
                response = self.put(
                    "/plom/admin/appendTestToDB",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "test_number": test_number,
                        "vmap_for_test": vmap_for_test,
                    },
                )
                response.raise_for_status()
                return response.text
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 406:
                    raise PlomDatabaseCreationError(response.reason) from None
                if response.status_code == 409:
                    raise PlomConflict(response.reason) from None
                if response.status_code == 400:
                    raise PlomServerNotReady(response.reason) from None
                raise PlomSeriousException("Unexpected {}".format(e)) from None

    def getGlobalPageVersionMap(self):
        with self.SRmutex:
            try:
                response = self.get(
                    "/plom/admin/pageVersionMap",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code == 401:
                    raise PlomAuthenticationException() from None
                elif response.status_code == 409:
                    raise PlomConflict(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None
        # JSON casts dict keys to str, force back to ints
        return undo_json_packing_of_version_map(response.json())

    def pre_id_paper(
        self, paper_number, studentID, *, certainty=0.9, predictor="prename"
    ):
        """Pre-id a paper.

        Args:
            paper_number (str): which paper.
            studentID (str): student ID to predict for that paper.

        Keyword Args:
            predictor: defaults to "prename"
            certainty: defaults to 0.9

        Returns:
            None

        Exceptions:
            PlomConflict: `studentID` already used on a different paper.
                Currently does not do this.  Could enable again in the
                future.
            PlomAuthenticationException: login problems.
            PlomSeriousException: other errors.
        """
        with self.SRmutex:
            try:
                response = self.put(
                    f"/ID/preid/{paper_number}",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "sid": studentID,
                        "predictor": predictor,
                        "certainty": certainty,
                    },
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 404:
                    raise PlomSeriousException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def remove_pre_id(self, paper_number):
        """Remove the predicted "pre-id" for a particular paper.

        Exceptions:
            PlomAuthenticationException: login problems.
            PlomSeriousException: other errors.
        """
        with self.SRmutex:
            try:
                response = self.delete(
                    f"/ID/preid/{paper_number}",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 404:
                    raise PlomSeriousException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def id_paper(self, paper_number, studentID, studentName):
        """Identify a paper directly, not as part of a IDing task.

        Exceptions:
            PlomConflict: `studentID` already used on a different paper.
            PlomAuthenticationException: login problems.
            PlomSeriousException: other errors.
        """
        with self.SRmutex:
            try:
                response = self.put(
                    f"/ID/{paper_number}",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "sid": studentID,
                        "sname": studentName,
                    },
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 409:
                    raise PlomConflict(response.reason) from None
                if response.status_code == 404:
                    raise PlomSeriousException(e) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def un_id_paper(self, paper_number):
        """Remove the identify of a paper directly.

        Exceptions:
            PlomAuthenticationException: login problems.
            PlomSeriousException: other errors.
        """
        with self.SRmutex:
            try:
                response = self.delete(
                    f"/ID/{paper_number}",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 406:
                    raise PlomSeriousException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def upload_classlist(self, classdict, force):
        """Give the server a classlist.

        Args:
            classdict (list): list of dict.  Each dict is one student.
                It MUST have keys `"id"` and `"studentNumber"` (case
                matters).  There may be other keys included as well.
                Keys should probably be homogeneous between rows (TODO?).
            force (bool): Force uploading if a classlist already exists.

        Exceptions:
            PlomConflict: server already has one.
            PlomRangeException: this classlist causes an invalid server
                spec.  Most likely numberToProduce is too small but
                check error message to be sure.
            PlomServerNotReady: e.g., it has no spec.
            PlomAuthenticationException: login problems.
            PlomSeriousException: other errors.
        """
        self.SRmutex.acquire()
        try:
            response = self.put(
                "/ID/classlist",
                json={
                    "user": self.user,
                    "token": self.token,
                    "classlist": classdict,
                    "force": force,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 409:
                raise PlomConflict(response.reason) from None
            if response.status_code == 400 and "no spec" in response.reason:
                raise PlomServerNotReady(response.reason) from None
            if response.status_code == 406:
                raise PlomRangeException(response.reason) from None
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def upload_spec(self, specdata):
        """Give the server a specification.

        Args:
            specdata (dict): see :func:`plom.SpecVerifier`.

        Exceptions:
            PlomConflict: server already has a database, cannot accept spec.
            PlomAuthenticationException: login problems.
            ValueError: invalid spec
            PlomSeriousException: other errors.

        Returns:
            None
        """
        with self.SRmutex:
            try:
                response = self.put(
                    "/info/spec",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "spec": specdata,
                    },
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 400:
                    raise ValueError(response.reason) from None
                if response.status_code == 409:
                    raise PlomConflict(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def RgetCompletionStatus(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/completionStatus",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return response.json()

    def RgetStatus(self, test):
        self.SRmutex.acquire()
        try:
            response = self.get(
                f"/REP/status/{test}",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            if response.status_code == 404:
                raise PlomSeriousException(f"Could not find test {test}.") from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return response.json()

    def getScannedTests(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/scanned",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Server could not find the spec - this should not happen!"
                ) from None
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getIncompleteTests(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/incomplete",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Server could not find the spec - this should not happen!"
                ) from None
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getDanglingPages(self):
        self.SRmutex.acquire()
        # Note: long timeout, slow for large (1000s) of papers
        timeout = (self.default_timeout[0], 10 * self.default_timeout[1])
        try:
            response = self.get(
                "/REP/dangling",
                json={"user": self.user, "token": self.token},
                timeout=timeout,
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Server could not find the spec - this should not happen!"
                ) from None
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def RgetSpreadsheet(self):
        with self.SRmutex:
            try:
                response = self.get(
                    "/REP/spreadsheet",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def IDprogressCount(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/ID/progress",
                json={"user": self.user, "token": self.token},
            )
            # throw errors when response code != 200.
            response.raise_for_status()
            # convert the content of the response to a textfile for identifier
            progress = response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return progress

    def IDgetImageList(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/TMP/imageList",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            # TODO: print(response.encoding) autodetected
            imageList = response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return imageList

    def IDgetImageFromATest(self):
        """Get any ID image.

        DEPRECATED: only legacy servers do this.
        """
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/ID/randomImage",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            imageList = []
            for img in MultipartDecoder.from_response(response).parts:
                imageList.append(
                    BytesIO(img.content).getvalue()
                )  # pass back image as bytes
            return imageList
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code == 410:
                raise PlomNoMoreException("Cannot find ID image.") from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getProgress(self, q, v):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/progress",
                json={"user": self.user, "token": self.token, "q": q, "v": v},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Server could not find the spec - this should not happen!"
                ) from None
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getQuestionUserProgress(self, q, v):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/questionUserProgress",
                json={"user": self.user, "token": self.token, "q": q, "v": v},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Server could not find the spec - this should not happen!"
                ) from None
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getMarkHistogram(self, q, v):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/markHistogram",
                json={"user": self.user, "token": self.token, "q": q, "v": v},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Server could not find the spec - this should not happen!"
                ) from None
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def replaceMissingTestPage(self, t, p, v):
        """Replace a do-not-mark page with a server-generated placeholder.

        Returns:
            str: diagnostic message, largely unnecessary

        Raises:
            PlomConflict: collisions, duplicates
            PlomNoPaper: no paper or also invalid page
            PlomAuthenticationException: auth or invalid user
            PlomSeriousException: anything unexpected
        """
        with self.SRmutex:
            try:
                response = self.put(
                    "/plom/admin/missingTestPage",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "test": t,
                        "page": p,
                        "version": v,
                    },
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 404:
                    raise PlomNoPaper(response.reason) from None
                if response.status_code == 409:
                    raise PlomConflict(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def replaceMissingDNMPage(self, t, p):
        """Replace a do-not-mark page with a server-generated placeholder.

        Returns:
            str: diagnostic message, largely unnecessary

        Raises:
            PlomConflict: collisions, duplicates
            PlomNoPaper: no paper or also invalid page
            PlomAuthenticationException: auth or invalid user
            PlomSeriousException: anything unexpected
        """
        with self.SRmutex:
            try:
                response = self.put(
                    "/plom/admin/missingDNMPage",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "test": t,
                        "page": p,
                    },
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 404:
                    raise PlomNoPaper(response.reason) from None
                if response.status_code == 409:
                    raise PlomConflict(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def replaceMissingIDPage(self, t):
        with self.SRmutex:
            try:
                response = self.put(
                    "/plom/admin/missingIDPage",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "test": t,
                    },
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 410:
                    raise PlomUnidentifiedPaperException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def replaceMissingHWQuestion(self, student_id=None, test=None, question=None):
        # can replace by SID or by test-number
        self.SRmutex.acquire()
        try:
            response = self.put(
                "/plom/admin/missingHWQuestion",
                json={
                    "user": self.user,
                    "token": self.token,
                    "question": question,
                    "sid": student_id,
                    "test": test,
                },
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Server could not find the TPV - this should not happen!"
                ) from None
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            if response.status_code == 405:  # that question already has pages
                raise PlomTakenException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def removeSinglePage(self, test_number, page_name):
        self.SRmutex.acquire()
        try:
            response = self.delete(
                "/plom/admin/singlePage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "test": test_number,
                    "page_name": page_name,
                },
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code in (406, 409, 410):
                raise PlomConflict(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def removeAllScannedPages(self, test_number):
        self.SRmutex.acquire()
        try:
            response = self.delete(
                "/plom/admin/scannedPages",
                json={
                    "user": self.user,
                    "token": self.token,
                    "test": test_number,
                },
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Server could not find the page - this should not happen!"
                ) from None
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getTPageImageData(self, t, p, v):
        with self.SRmutex:
            try:
                response = self.get(
                    "/plom/admin/scannedTPage",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "test": t,
                        "page": p,
                        "version": v,
                    },
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 400:
                    # TODO? do something else?
                    return None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def getHWPageImageData(self, t, q, o):
        with self.SRmutex:
            try:
                response = self.get(
                    "/plom/admin/scannedHWPage",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "test": t,
                        "question": q,
                        "order": o,
                    },
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 400:
                    # TODO? do something else?
                    return None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def getEXPageImageData(self, t, q, o):
        with self.SRmutex:
            try:
                response = self.get(
                    "/plom/admin/scannedEXPage",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "test": t,
                        "question": q,
                        "order": o,
                    },
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 400:
                    # TODO? do something else?
                    return None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def getCollidingImage(self, fname):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/plom/admin/collidingImage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "fileName": fname,
                },
            )
            response.raise_for_status()
            image = BytesIO(response.content).getvalue()
            return image
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            if response.status_code == 404:
                return None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def removeUnknownImage(self, fname):
        """Discard an UnknownPage.

        Args:
            fname (str): a filename/code that the server should have
                associated with an UnknownPage.

        Returns:
            None

        Raises:
            PlomAuthenticationException: login troubles.
            PlomNoPaper: could not find an UnknownPage associated with
                that filename.
            PlomSeriousException: somethung unexpected.
        """
        with self.SRmutex:
            try:
                response = self.delete(
                    "/plom/admin/unknownImage",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "fileName": fname,
                        "reason": "",
                    },
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 404:
                    raise PlomNoPaper(response.reason)
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def removeCollidingImage(self, fname):
        self.SRmutex.acquire()
        try:
            response = self.delete(
                "/plom/admin/collidingImage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "fileName": fname,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            if response.status_code == 404:
                return False
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()
        return True

    def checkTPage(self, testNumber, pageNumber):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/plom/admin/checkTPage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "test": testNumber,
                    "page": pageNumber,
                },
            )
            response.raise_for_status()
            # either ["scanned", version] or ["collision", version, image]
            vimg = MultipartDecoder.from_response(response).parts
            ver = int(vimg[1].content)
            if len(vimg) == 3:  # just look at length - sufficient for now?
                rval = [ver, BytesIO(vimg[2].content).getvalue()]
            else:
                rval = [ver, None]
            return rval  # [v, None] or [v, image1]
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            if response.status_code == 404:
                raise PlomSeriousException(
                    "Cannot find image file for {}.".format(testNumber)
                ) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def unknownToTestPage(self, fname, test, page, theta):
        """Map UnknownPage onto a TestPage.

        This is most likely used for QR-coded page where the QR
        codes were not read clearly, but in principle it can
        push any UnknownPage to a TestPage.

        Returns:
            str: "collision" if a colliding page created, something
            else otherwise ("testPage" currently but this is not
            guaranteed.

        Raises:
            PlomAuthenticationException
            PlomConflict
        """
        self.SRmutex.acquire()
        try:
            response = self.put(
                "/plom/admin/unknownToTestPage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "fileName": fname,
                    "test": test,
                    "page": page,
                    "rotation": theta,
                },
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            # Currently the route does not do this
            # if response.status_code == 406:
            #    raise PlomOwnersLoggedInException(response.reason) from None
            if response.status_code == 409:
                raise PlomConflict(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def unknownToExtraPage(self, fname, test, questions, theta):
        """Map Unknown Page to an Extra Page.

        Returns:
            None

        Raises:
            PlomAuthenticationException
            PlomConflict
        """
        self.SRmutex.acquire()
        try:
            response = self.put(
                "/plom/admin/unknownToExtraPage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "fileName": fname,
                    "test": test,
                    "questions": questions,
                    "rotation": theta,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            # Currently the route does not do this
            # if response.status_code == 406:
            #     raise PlomOwnersLoggedInException(response.reason) from None
            if response.status_code == 409:
                raise PlomConflict(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def unknownToHWPage(self, fname, test, questions, theta):
        """Map Unknown Page to a Homework Page.

        Returns:
            None

        Raises:
            PlomAuthenticationException
            PlomConflict
        """
        self.SRmutex.acquire()
        try:
            response = self.put(
                "/plom/admin/unknownToHWPage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "fileName": fname,
                    "test": test,
                    "questions": questions,
                    "rotation": theta,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            # Currently the route does not do this
            # if response.status_code == 406:
            #     raise PlomOwnersLoggedInException(response.reason) from None
            if response.status_code == 409:
                raise PlomConflict(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def collidingToTestPage(self, fname, test, page, version):
        self.SRmutex.acquire()
        try:
            response = self.put(
                "/plom/admin/collidingToTestPage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "fileName": fname,
                    "test": test,
                    "page": page,
                    "version": version,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code == 406:
                raise PlomOwnersLoggedInException(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def discardToUnknown(self, fname):
        self.SRmutex.acquire()
        try:
            response = self.put(
                "/plom/admin/discardToUnknown",
                json={
                    "user": self.user,
                    "token": self.token,
                    "fileName": fname,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code == 406:
                return False
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()
        return True

    def ID_delete_machine_predictions(self):
        """Deletes the machine-learning predicted IDs for all papers."""
        with self.SRmutex:
            try:
                response = self.delete(
                    "/ID/predictedID",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def ID_delete_all_predictions(self):
        """Deletes the predicted IDs for all predictors including prenamed."""
        with self.SRmutex:
            try:
                response = self.delete(
                    "/ID/predictions",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def ID_delete_predictions_from_predictor(self, *, predictor):
        """Deletes the predicted IDs for one particular predictor."""
        with self.SRmutex:
            try:
                response = self.delete(
                    f"/ID/predictions/{predictor}",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def run_predictor(self):
        """Match the results of the id digit reader with unidentified papers.

        Returns:
            str: status information about the solve.

        Raises:
            PlomAuthenticationException:
            PlomConflict: failed, in a non-fatal way, message explains more.
        """
        with self.SRmutex:
            try:
                # increase timeout as unsure how long this will take on real data
                timeout = (self.default_timeout[0], 5 * self.default_timeout[1])
                response = self.post_auth("/ID/predictedID", timeout=timeout)
                response.raise_for_status()
                return response.text
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 406:
                    raise PlomConflict(response.reason) from None
                if response.status_code == 409:
                    raise PlomConflict(response.reason) from None
                if response.status_code == 412:
                    raise PlomConflict(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def id_reader_get_logs(self):
        """Get the logs for the background id digit reader.

        Returns:
            list: TODO.

        Raises:
            PlomAuthenticationException:
            PlomSeriousException: something unexpected.
        """
        with self.SRmutex:
            try:
                response = self.get(
                    "/ID/id_reader",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def id_reader_run(self, top, bottom, *, ignore_timestamp):
        """Runs the id digit reader on the ID pages of all papers.

        Returns:
            list: ``[True, True]`` ID reader was (probably) started.
                ``[True, False]`` ID reader is still running OR has
                crashed (unfortunately we cannot yet tell difference)
                ``[False, last_time]`` ID reader was previously run,
                where ``last_time`` is a time-stamp.

        Raises:
            PlomAuthenticationException:
        """
        self.SRmutex.acquire()
        try:
            response = self.post_auth(
                "/ID/id_reader",
                json={
                    "crop_top": top,
                    "crop_bottom": bottom,
                    "ignore_timestamp": ignore_timestamp,
                },
            )
            response.raise_for_status()
            if response.status_code == 202:
                return [True, False, *response.json()]
            if response.status_code == 205:
                return [False, None, *response.json()]
            return [True, True, *response.json()]
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def id_reader_kill(self):
        """Kill a running background id digit reader job.

        Returns:
            list: TODO

        Raises:
            PlomAuthenticationException:
            PlomSeriousException: something unexpected.
        """
        with self.SRmutex:
            try:
                response = self.delete(
                    "/ID/id_reader",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def getIdentified(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/identified",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getNotAutoIdentified(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/notautoid",
                json={
                    "user": self.user,
                    "token": self.token,
                },
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getUserList(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/userList",
                json={
                    "user": self.user,
                    "token": self.token,
                },
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getUserDetails(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/userDetails",
                json={
                    "user": self.user,
                    "token": self.token,
                },
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getMarkReview(
        self, *, filterPaperNumber, filterQ, filterV, filterUser, filterMarked
    ):
        with self.SRmutex:
            try:
                response = self.get(
                    "/REP/markReview",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "filterPaperNumber": filterPaperNumber,
                        "filterQ": filterQ,
                        "filterV": filterV,
                        "filterUser": filterUser,
                        "filterMarked": filterMarked,
                    },
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def getIDReview(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/idReview",
                json={
                    "user": self.user,
                    "token": self.token,
                },
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def clearAuthorisationUser(self, someuser):
        self.SRmutex.acquire()
        try:
            response = self.delete(
                f"/authorisation/{someuser}",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def enableUser(self, someuser):
        self.SRmutex.acquire()
        try:
            response = self.put(
                f"/enable/{someuser}",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code == 400:
                raise PlomConflict(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def disableUser(self, someuser):
        self.SRmutex.acquire()
        try:
            response = self.put(
                f"/disable/{someuser}",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code == 400:
                raise PlomConflict(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def createUser(self, someuser, password):
        self.SRmutex.acquire()
        try:
            response = self.post_auth(
                f"/authorisation/{someuser}", json={"password": password}
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 406:
                return [False, response.text]
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()
        if response.status_code == 200:
            return [True, "User created."]
        raise PlomSeriousException(f"Unexpected {response.status_code}") from None

    def changeUserPassword(self, someuser, password):
        self.SRmutex.acquire()
        try:
            response = self.patch(
                f"/authorisation/{someuser}",
                json={
                    "user": self.user,
                    "token": self.token,
                    "password": password,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 406:
                return [False, response.text]
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()
        if response.status_code == 200:
            return [True, "User password updated."]
        raise PlomSeriousException(f"Unexpected {response.status_code}") from None

    def MrevertTask(self, code):
        self.SRmutex.acquire()
        try:
            response = self.patch(
                f"/MK/revert/{code}",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            if response.status_code == 204:
                raise PlomBenignException("No action to be taken.")

        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            elif response.status_code == 409:
                raise PlomConflict(response.reason) from None
            else:
                raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def MreviewQuestion(self, paper_number, question):
        with self.SRmutex:
            try:
                response = self.patch(
                    "/MK/review",
                    json={
                        "user": self.user,
                        "token": self.token,
                        "paper_number": paper_number,
                        "question": question,
                    },
                )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                if response.status_code == 409:
                    raise PlomConflict(response.reason) from None
                if response.status_code == 404:
                    raise PlomSeriousException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def IDreviewID(self, testNumber):
        self.SRmutex.acquire()
        try:
            response = self.patch(
                "/ID/review",
                json={
                    "user": self.user,
                    "token": self.token,
                    "testNumber": testNumber,
                },
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code == 404:
                raise PlomSeriousException(response.reason) from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def RgetOutToDo(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/outToDo",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def RgetMarked(self, q, v):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/marked",
                json={"user": self.user, "token": self.token, "q": q, "v": v},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return response.json()

    def putSolutionImage(self, question, version, fileName):
        with self.SRmutex:
            try:
                with open(fileName, "rb") as fh:
                    param = {
                        "user": self.user,
                        "token": self.token,
                        "question": question,
                        "version": version,
                        "md5sum": hashlib.md5(fh.read()).hexdigest(),
                    }
                    # reset stream position to start before reading again
                    fh.seek(0)
                    dat = MultipartEncoder(
                        fields={
                            "param": json.dumps(param),
                            "image": fh,
                        }
                    )
                    response = self.put(
                        "/plom/admin/solution",
                        json={"user": self.user, "token": self.token},
                        data=dat,
                        headers={"Content-Type": dat.content_type},
                    )
                response.raise_for_status()
            except requests.HTTPError as e:
                if response.status_code == 401:
                    raise PlomAuthenticationException() from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None

    def deleteSolutionImage(self, question, version):
        self.SRmutex.acquire()
        try:
            response = self.delete(
                "/plom/admin/solution",
                json={
                    "user": self.user,
                    "token": self.token,
                    "question": question,
                    "version": version,
                },
            )
            response.raise_for_status()
            if response.status_code == 200:
                return True
            # if response.status_code == 204:
            return False
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    # =====
    # Rubric analysis stuff

    def RgetTestRubricMatrix(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/test_rubric_matrix",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return response.json()

    def RgetRubricCounts(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/rubric/counts",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return response.json()

    def RgetRubricDetails(self, key):
        self.SRmutex.acquire()
        try:
            response = self.get(
                f"/REP/rubric/{key}",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return response.json()

    # =====
    # Bundle image stuff

    def getBundleFromImage(self, filename):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/plom/admin/bundleFromImage",
                json={"user": self.user, "token": self.token, "filename": filename},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code in (401, 403):
                raise PlomAuthenticationException(response.reason) from None
            if response.status_code == 410:
                raise PlomNoMoreException("Cannot find that image.") from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return response.json()

    def getImagesInBundle(self, bundle_name):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/plom/admin/imagesInBundle",
                json={"user": self.user, "token": self.token, "bundle": bundle_name},
            )
            response.raise_for_status()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            if response.status_code == 410:
                raise PlomNoMoreException("Cannot find that bundle.") from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

        return response.json()

    def getPageFromBundle(self, bundle_name, image_position):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/plom/admin/bundlePage",
                json={
                    "user": self.user,
                    "token": self.token,
                    "bundle_name": bundle_name,
                    "bundle_order": image_position,
                },
            )
            response.raise_for_status()
            image = BytesIO(response.content).getvalue()
            return image
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            if response.status_code == 410:
                raise PlomNoMoreException("Cannot find that image / bundle.") from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def RgetCompletions(self):
        self.SRmutex.acquire()
        try:
            response = self.get(
                "/REP/completions",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def RgetCoverPageInfo(self, test):
        self.SRmutex.acquire()
        try:
            response = self.get(
                f"/REP/coverPageInfo/{test}",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def RgetOriginalFiles(self, testNumber):
        self.SRmutex.acquire()
        try:
            response = self.get(
                f"/REP/originalFiles/{testNumber}",
                json={"user": self.user, "token": self.token},
            )
            response.raise_for_status()
            return response.json()
        except requests.HTTPError as e:
            if response.status_code == 401:
                raise PlomAuthenticationException() from None
            raise PlomSeriousException(f"Some other sort of error {e}") from None
        finally:
            self.SRmutex.release()

    def getFilesInAllTests(self):
        with self.SRmutex:
            try:
                response = self.get(
                    "/REP/filesInAllTests",
                    json={"user": self.user, "token": self.token},
                )
                response.raise_for_status()
                return response.json()
            except requests.HTTPError as e:
                if response.status_code in (401, 403):
                    raise PlomAuthenticationException(response.reason) from None
                raise PlomSeriousException(f"Some other sort of error {e}") from None
