"""Utility functions for DNA sequence manipulation"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_dna_utils.ipynb.

# %% auto 0
__all__ = ['COMPLEMENT', 'is_valid_codon', 'reverse_complement', 'open_fastq', 'fastq_iterator', 'fastq_pair_iterator']

# %% ../nbs/01_dna_utils.ipynb 4
from pathlib import Path
from Bio.Seq import Seq
from Bio.SeqIO.QualityIO import FastqGeneralIterator
import gzip
from .core import logger

# %% ../nbs/01_dna_utils.ipynb 6
# Constants for complement mapping
COMPLEMENT = {
    'A': 'T', 'C': 'G', 'G': 'C', 'T': 'A',
    '[': ']', ']': '[', '(': ')', ')': '(',
    '.': '.', 'N': 'N'
}

# %% ../nbs/01_dna_utils.ipynb 9
def is_valid_codon(codon: str) -> bool:
    """
    Check if a string represents a valid DNA codon.
    
    A valid codon is exactly 3 nucleotides long and contains only A, C, G, T.
    
    Args:
        codon: String to check
        
    Returns:
        True if valid codon, False otherwise
    """
    if len(codon) != 3:
        return False
    return all(base in "ACGT" for base in codon)

# %% ../nbs/01_dna_utils.ipynb 10
def reverse_complement(sequence: str) -> str:
    """
    Generate the reverse complement of a DNA sequence.
    
    DNA bases are complemented (A→T, C→G, G→C, T→A), and the sequence is reversed.
    Special characters like brackets and parentheses are also handled with their complements.
    
    Args:
        sequence: DNA sequence string
        
    Returns:
        Reverse complement of the input sequence
    """
    return ''.join(COMPLEMENT.get(char, char) for char in reversed(sequence))

# %% ../nbs/01_dna_utils.ipynb 11
def open_fastq(filename: str):
    """
    Open a FASTQ file (gzipped or not) for reading.
    
    Args:
        filename: Path to FASTQ file
        
    Returns:
        File handle for reading
    
    Raises:
        FileNotFoundError: If file doesn't exist
    """
    if filename.endswith(('.gz', '.gzip')):
        return gzip.open(filename, "rt")
    else:
        return open(filename, "r")

# %% ../nbs/01_dna_utils.ipynb 12
def fastq_iterator(filename: str):
    """
    Create an iterator over FASTQ records.
    
    Uses Biopython's FastqGeneralIterator for efficient parsing.
    
    Args:
        filename: Path to FASTQ file
        
    Returns:
        Iterator yielding (title, sequence, quality) tuples
    """
    with open_fastq(filename) as handle:
        for record in FastqGeneralIterator(handle):
            yield record

# %% ../nbs/01_dna_utils.ipynb 13
def fastq_pair_iterator(filename1: str, filename2: str):
    """
    Create an iterator over paired FASTQ records.
    
    Args:
        filename1: Path to first FASTQ file in pair
        filename2: Path to second FASTQ file in pair
        
    Returns:
        Iterator yielding pairs of (title, sequence, quality) tuples
    """
    with open_fastq(filename1) as handle1, open_fastq(filename2) as handle2:
        for read1, read2 in zip(FastqGeneralIterator(handle1), FastqGeneralIterator(handle2)):
            yield read1, read2
