"""Functions for processing and saving results"""

# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/02_output.ipynb.

# %% auto 0
__all__ = ['ResultsFormatter', 'ResultsSaver']

# %% ../nbs/02_output.ipynb 4
import os
import pandas as pd
import json
from pathlib import Path
from collections import Counter
from typing import Dict, List, Tuple, Set, Any
from .core import OligoCounter

# %% ../nbs/02_output.ipynb 6
class ResultsFormatter:
    """Formats oligo codon counting results for output."""
    
    DEFAULT_GENCODE = {
        'ATA':'I', 'ATC':'I', 'ATT':'I', 'ATG':'M',
        'ACA':'T', 'ACC':'T', 'ACG':'T', 'ACT':'T',
        'AAC':'N', 'AAT':'N', 'AAA':'K', 'AAG':'K',
        'AGC':'S', 'AGT':'S', 'AGA':'R', 'AGG':'R',
        'CTA':'L', 'CTC':'L', 'CTG':'L', 'CTT':'L',
        'CCA':'P', 'CCC':'P', 'CCG':'P', 'CCT':'P',
        'CAC':'H', 'CAT':'H', 'CAA':'Q', 'CAG':'Q',
        'CGA':'R', 'CGC':'R', 'CGG':'R', 'CGT':'R',
        'GTA':'V', 'GTC':'V', 'GTG':'V', 'GTT':'V',
        'GCA':'A', 'GCC':'A', 'GCG':'A', 'GCT':'A',
        'GAC':'D', 'GAT':'D', 'GAA':'E', 'GAG':'E',
        'GGA':'G', 'GGC':'G', 'GGG':'G', 'GGT':'G',
        'TCA':'S', 'TCC':'S', 'TCG':'S', 'TCT':'S',
        'TTC':'F', 'TTT':'F', 'TTA':'L', 'TTG':'L',
        'TAC':'Y', 'TAT':'Y', 'TAA':'*', 'TAG':'*',
        'TGC':'C', 'TGT':'C', 'TGA':'*', 'TGG':'W'
    }
    
    @classmethod
    def process_dataframe(cls, df: pd.DataFrame, gencode: Dict[str, str] = None) -> pd.DataFrame:
        """
        Process the dataframe by consolidating non-standard codons.
        
        Args:
            df (pd.DataFrame): Input dataframe of codon counts
            gencode (Dict[str, str], optional): Custom genetic code dictionary
        
        Returns:
            pd.DataFrame: Processed dataframe with non-standard codons consolidated
        """
        # Use default gencode if not provided
        if gencode is None:
            gencode = cls.DEFAULT_GENCODE
        
        # Create a copy to avoid modifying the original
        df_processed = df.copy()
        
        # Identify rows that are not in gencode keys
        non_gencode_rows = [idx for idx in df_processed.index 
                             if idx not in gencode.keys() and idx != 'none']
        
        # If no non-standard codons, return the original dataframe
        if not non_gencode_rows:
            return df_processed
        
        # Calculate sum of counts for non-gencode rows
        non_gencode_sum = df_processed.loc[non_gencode_rows].sum()
        
        # Add these counts to 'none' row if it exists, or create it
        if 'none' in df_processed.index:
            df_processed.loc['none'] += non_gencode_sum
        else:
            df_processed.loc['none'] = non_gencode_sum
        
        # Remove the non-gencode rows
        df_processed = df_processed.drop(non_gencode_rows)
        
        return df_processed

    @classmethod
    def to_dataframe(cls, results: Dict[int, Counter], 
                      oligos: List[str], 
                      offset: int = 1, 
                      gencode: Dict[str, str] = None) -> pd.DataFrame:
        """
        Convert results to a pandas DataFrame.
        
        Args:
            results: Dictionary of oligo index to codon counter
            oligos: List of oligo sequences
            offset: Value to add to oligo index in column names (default: 1)
            gencode: Optional custom genetic code dictionary
            
        Returns:
            DataFrame with columns for each oligo and rows for each codon
        """
        try:
            df_dict = {}
            for i, oligo in enumerate(oligos):
                col_name = f"{i + offset}_{oligo}"
                col_data = results.get(i, Counter())
                df_dict[col_name] = col_data
            
            # Create DataFrame and fill missing values with 0
            df = pd.DataFrame(df_dict).fillna(0)
            
            # Reorder columns based on oligo index
            df = df.reindex(sorted(df.columns, key=lambda x: int(x.split('_')[0])), axis=1)

            # Process dataframe with optional custom genetic code
            df = cls.process_dataframe(df, gencode)
            
            return df
        
        except Exception as e:
            raise ValueError(f"Error converting results to DataFrame: {str(e)}")
    
    @classmethod
    def summarize_results(cls, df: pd.DataFrame) -> Dict[str, Any]:
        """
        Generate a summary of the results.
        
        Args:
            df: DataFrame containing oligo codon counts
            
        Returns:
            Dictionary with summary statistics
        """
        summary = {}
        
        # Total reads and matched reads
        for col in df.columns:
            total_reads = df[col].sum()
            matched_reads = total_reads - df[col].get('none', 0)
            
            summary[f"{col}_total_reads"] = total_reads
            summary[f"{col}_matched_reads"] = matched_reads
            
            # Avoid division by zero
            summary[f"{col}_match_rate"] = (matched_reads / total_reads) if total_reads > 0 else 0
                
            # Top codons
            if matched_reads > 0:
                # Filter out 'none' and get top 5 codons
                codon_counts = df[col].drop('none', errors='ignore')
                top_codons = codon_counts.nlargest(5)
                summary[f"{col}_top_codons"] = top_codons.to_dict()
            else:
                summary[f"{col}_top_codons"] = {}
        
        return summary

# %% ../nbs/02_output.ipynb 8
class ResultsSaver:
    """Saves processing results to files."""
    
    @staticmethod
    def save_csv(df: pd.DataFrame, output_path: str, 
                 filename: str = 'oligo_counts.csv', create_dir: bool = True) -> str:
        """Save results to a CSV file.
        
        Args:
            df: DataFrame containing oligo codon counts
            output_path: Directory to save the file
            filename: Name of the output file (default: 'oligo_counts.csv')
            create_dir: Whether to create the output directory if it doesn't exist (default: True)
            
        Returns:
            Full path to the saved file
        """
        output_dir = Path(output_path)
        if create_dir:
            output_dir.mkdir(exist_ok=True, parents=True)
            
        output_file = output_dir / filename
        df.to_csv(output_file)
        return str(output_file)
    
    @staticmethod
    def save_summary(results_summary: Dict, output_path: str, 
                     filename: str = 'oligo_summary.json', create_dir: bool = True) -> str:
        """Save a summary of results to a JSON file.
        
        Args:
            results_summary: Dictionary containing summary information
            output_path: Directory to save the file
            filename: Name of the output file (default: 'oligo_summary.json')
            create_dir: Whether to create the output directory if it doesn't exist (default: True)
            
        Returns:
            Full path to the saved file
        """
        output_dir = Path(output_path)
        if create_dir:
            output_dir.mkdir(exist_ok=True, parents=True)
            
        output_file = output_dir / filename
        
        with open(output_file, 'w') as f:
            json.dump(results_summary, f, indent=2)
            
        return str(output_file)
    
    @staticmethod
    def save_text_report(results_summary: Dict, df: pd.DataFrame, output_path: str,
                         filename: str = 'oligo_report.txt', create_dir: bool = True) -> str:
        """Save a human-readable text report of the results.
        
        Args:
            results_summary: Dictionary containing summary information
            df: DataFrame containing oligo codon counts
            output_path: Directory to save the file
            filename: Name of the output file (default: 'oligo_report.txt')
            create_dir: Whether to create the output directory if it doesn't exist (default: True)
            
        Returns:
            Full path to the saved file
        """
        output_dir = Path(output_path)
        if create_dir:
            output_dir.mkdir(exist_ok=True, parents=True)
            
        output_file = output_dir / filename
        
        with open(output_file, 'w') as f:
            f.write("=== OligoSeeker Results Report ===\n\n")
            
            # Write overall stats
            f.write("Overall Statistics:\n")
            f.write(f"Total oligos processed: {len(df.columns)}\n")
            f.write("\n")
            
            # Write per-oligo stats
            f.write("=== Per-Oligo Statistics ===\n\n")
            
            for col in df.columns:
                f.write(f"Oligo: {col}\n")
                f.write(f"  Total reads: {results_summary.get(f'{col}_total_reads', 0)}\n")
                f.write(f"  Matched reads: {results_summary.get(f'{col}_matched_reads', 0)}\n")
                
                match_rate = results_summary.get(f'{col}_match_rate', 0) * 100
                f.write(f"  Match rate: {match_rate:.2f}%\n")
                
                # Top codons
                top_codons = results_summary.get(f'{col}_top_codons', {})
                if top_codons:
                    f.write("  Top codons:\n")
                    for codon, count in top_codons.items():
                        f.write(f"    {codon}: {count}\n")
                
                f.write("\n")
                
            # Write a note about other output files
            f.write("\nNote: More detailed results are available in the CSV, Excel, and JSON output files.\n")
            
        return str(output_file)
