import abc
import logbook

import pandas as pd


log = logbook.Logger("TradingControl")

class TradingPolicy(metaclass=abc.ABCMeta):
    """
    Abstract base class representing a fail-safe control on the behavior of any
    algorithm.
    """

    def __init__(self, **kwargs):
        """
        Track any arguments that should be printed in the error message
        generated by self.handle_violation.
        """
        self.__fail_args = kwargs

    @abc.abstractmethod
    def validate(self, asset, algo_datetime, algo_current_data):
        """
        Before any transcation is executed by TradingAlgorithm, this method should be
        called *exactly once* on each registered TradingPolicy object.

        If the specified asset do not violate this TradingPolicy's
        restraint, this method should return False.

        If the desired transcation violates this TradingPolicy's contraint, this
        method should call self.handle_violation.
        """
        raise NotImplementedError

    def _constraint_msg(self, metadata, constraint=None):
        if not constraint:
            constraint = repr(self)

        if metadata:
            constraint = "{constraint} (Metadata: {metadata})".format(
                constraint=constraint, metadata=metadata
            )
        return constraint
       
    def handle_violation(self, asset, datetime, metadata=None, constraint=None):
        """
        Handle a TradingPolicyViolation, either by logging with information about the violation.

        If dynamic information should be displayed as well, pass it in via `metadata`.
        """
        constraint = self._constraint_msg(metadata, constraint)

        log.info(
                "{asset} violates trading constraint {constraint} at {dt} ",
                asset=asset,
                constraint=constraint,
                dt=datetime,
                )

    def __repr__(self):
        return "{name}({attrs})".format(
            name=self.__class__.__name__, attrs=self.__fail_args
        )

class ExcludeIlliquidAsset(TradingPolicy):

    """TradingPolicy representing a limit on the illiquid assets allowed
    by the algorithm.

    Parameters
    ----------
    rules : list
        see：zipline.sources.TEJ_Api_Data.LIQUIDITY_RISK_COLUMNS
    log : bool
        whether the logging is active or inactive.

    For example, *rules=Full_Delivery limits an algorithm to trading on Full Delivery assets.
    """

    def __init__(self, rules, log):

        super(ExcludeIlliquidAsset, self).__init__(rules=rules,
                                                   log=log)

        self.log = log
        self.rules = rules

    def validate(self, asset, algo_datetime, algo_current_data):
        """
        Temporarily suspended the trading if data.current(asset, LIQUIDITY_RISK_COLUMNS.get(rule))=='Y'.
        """
        # If the order is for 0 shares, then silently pass through.
        # if amount == 0:
        #    return

        normalized_algo_dt = pd.Timestamp(algo_datetime).normalize()

        for rule in self.rules:
            if rule in algo_current_data.current(asset, 'annotation'):
            # if any(s in algo_current_data.current(asset, 'annotation') for s in rule):

                metadata = {"asset":asset}

                if self.log:
                    self.handle_violation(asset,
                                          normalized_algo_dt,
                                          metadata=metadata,
                                          constraint=str('"{}"').format(rule))

                return False

            else:
                continue

        return True