# Copyright (c) 2017 Qumulo, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may not
# use this file except in compliance with the License. You may obtain a copy of
# the License at http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations under
# the License.


import argparse

import qumulo.lib.opts
import qumulo.lib.util as util
import qumulo.rest.fs as fs
import qumulo.rest.ftp as ftp

from qumulo.lib.opts import str_decode
from qumulo.rest_client import RestClient


class FtpGetStatus(qumulo.lib.opts.Subcommand):
    NAME = 'ftp_get_status'
    SYNOPSIS = 'Get FTP server settings and status'

    @staticmethod
    def main(rest_client: RestClient, _args: argparse.Namespace) -> None:
        print(ftp.get_status(rest_client.conninfo, rest_client.credentials))


class FtpModifySettings(qumulo.lib.opts.Subcommand):
    NAME = 'ftp_modify_settings'
    SYNOPSIS = 'Set FTP server settings'

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '--enabled', type=util.bool_from_string, metavar='{true,false}', required=False
        )

        parser.add_argument(
            '--check-remote-host',
            type=util.bool_from_string,
            metavar='{true,false}',
            required=False,
        )

        parser.add_argument(
            '--log-operations', type=util.bool_from_string, metavar='{true,false}', required=False
        )

        parser.add_argument(
            '--chroot-users', type=util.bool_from_string, metavar='{true,false}', required=False
        )

        parser.add_argument(
            '--allow-unencrypted-connections',
            type=util.bool_from_string,
            metavar='{true,false}',
            required=False,
        )

        parser.add_argument(
            '--expand-wildcards', type=util.bool_from_string, metavar='{true,false}', required=False
        )

        group = parser.add_mutually_exclusive_group()

        group.add_argument('--anonymous-user-as-local-user', type=fs.LocalUser, required=False)

        group.add_argument('--anonymous-user-none', action='store_true', required=False)

        group.add_argument('--greeting', type=str_decode, required=False)

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        anonymous_user = None
        if args.anonymous_user_none:
            anonymous_user = 'none'
        else:
            anonymous_user = args.anonymous_user_as_local_user

        if (
            args.enabled is None
            and args.check_remote_host is None
            and args.log_operations is None
            and args.chroot_users is None
            and args.allow_unencrypted_connections is None
            and args.expand_wildcards is None
            and anonymous_user is None
            and args.greeting is None
        ):
            parser = argparse.ArgumentParser(description='ftp_modify_settings')
            FtpModifySettings.options(parser)
            parser.error('Must provide at least one argument')

        print(
            ftp.modify_settings(
                rest_client.conninfo,
                rest_client.credentials,
                enabled=args.enabled,
                check_remote_host=args.check_remote_host,
                log_operations=args.log_operations,
                chroot_users=args.chroot_users,
                allow_unencrypted_connections=args.allow_unencrypted_connections,
                expand_wildcards=args.expand_wildcards,
                anonymous_user=anonymous_user,
                greeting=args.greeting,
            )
        )
