# Copyright (c) 2023 Qumulo, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may not
# use this file except in compliance with the License. You may obtain a copy of
# the License at http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations under
# the License.

import argparse
import sys
import textwrap

from typing import Dict, List, Sequence

import qumulo.lib.auth
import qumulo.lib.opts
import qumulo.rest.fs as fs

from qumulo.lib.request import pretty_json
from qumulo.lib.util import tabulate
from qumulo.rest_client import RestClient


def print_pretty_output(entries: List[Dict[str, str]]) -> None:
    print(
        tabulate(
            [
                [
                    entry['id'],
                    entry['name'],
                    entry['disabled'],
                    entry['comment'],
                    entry['public_key'],
                ]
                for entry in entries
            ],
            ['id', 'name', 'disabled', 'comment', 'public_key'],
        )
    )


def mutually_exclusive(a: Sequence[str], b: Sequence[str]) -> bool:
    insufficient = not all(a) and not all(b)
    excessive = any(a) and any(b)
    return not (insufficient or excessive)


class AddKeyCommand(qumulo.lib.opts.Subcommand):
    NAME = 'fs_security_add_key'
    SYNOPSIS = 'Add a key to the file system key-store.'
    DESCRIPTION = textwrap.dedent(
        f"""
        {SYNOPSIS}

        The key store lets you lock objects such as snapshots. You can unlock a locked object only by creating a verification signature with the private key.

        Perform the following steps:
        1. Generate an ECDSA P-256 public-private key pair.
        2. Use the private key to sign the key name.
        3. Run the command by using the public key, the verification signature, and the key name that you specify.

        If your system has the Python cryptography library, you can use the --private-key-file flag and let the command determine the public key and verification signature to send to Qumulo Core, rather than calculate these elements.

        Examples:
            - To add a key to the key store by using a verification signature:
            qq fs_security_add_key --name <key name> --public-key <Base64-encoded public key content> --verification-signature <a signed key name> [--comment <comment>]

            - To add a key to the key store by using the private key file:
            qq fs_security_add_key --name <key name> --private-key-file <the path to the private key> [--comment <comment>]

        """  # noqa: E501
    )

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '--name',
            type=str,
            required=True,
            help="""
                The unique name to associate with the key to add to the key store. The name must
                not contain only numbers.
                """,
        )
        parser.add_argument(
            '--private-key-file',
            type=argparse.FileType('r'),
            help='The ECDSA private key file for the key to add to the key store.',
        )
        parser.add_argument(
            '--public-key',
            type=str,
            help='The ECDSA public key for the key to add to the key store.',
        )
        parser.add_argument(
            '--verification-signature',
            type=str,
            help='A signature generated by signing the key name with the ECDSA private key.',
        )
        parser.add_argument(
            '--comment',
            type=str,
            default='',
            help="""
                An optional comment that the system stores alongside the key to add to the key
                store.
                """,
        )
        parser.add_argument(
            '--json',
            action='store_true',
            help='Print the output in JSON format. The default output is in a table.',
        )

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        if not mutually_exclusive(
            (args.private_key_file,), (args.public_key, args.verification_signature)
        ):
            sys.stderr.write(
                'Either the --private-key-file argument or both the [ --public-key and'
                ' --verification-signature ] arguments should be passed to this command.'
            )
            sys.exit(1)

        if args.private_key_file:
            key = args.private_key_file.read()
            args.public_key, args.verification_signature = fs.get_verified_public_key(
                key, args.name
            )

        response = fs.security_add_key(
            rest_client.conninfo,
            rest_client.credentials,
            name=args.name,
            public_key_data=args.public_key,
            verification_signature=args.verification_signature,
            comment=args.comment,
        )
        if args.json:
            print(response)
        else:
            print_pretty_output([response.data])


class GetKeyCommand(qumulo.lib.opts.Subcommand):
    NAME = 'fs_security_get_key'
    SYNOPSIS = 'Get information for a key in the file system key store.'

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '-k',
            '--key',
            required=True,
            help='The identifier or name of the key for which to show information.',
        )
        parser.add_argument(
            '--json',
            action='store_true',
            help='Print the output in JSON format. By default, the output is in a table.',
        )

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        response = fs.security_get_key(rest_client.conninfo, rest_client.credentials, args.key)
        if args.json:
            print(response)
        else:
            print_pretty_output([response.data])


class GetKeyUsageCommand(qumulo.lib.opts.Subcommand):
    NAME = 'fs_security_get_key_usage'
    SYNOPSIS = (
        'Show information about snapshot and snapshot policy usage for a key from the file system'
        ' key store.'
    )

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '-k',
            '--key',
            required=True,
            help='The key identifier or name for which to show usage information.',
        )
        parser.add_argument(
            '--json',
            action='store_true',
            help='Print the output in JSON format. By default, the output is in a table.',
        )

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        key_ref = args.key

        iterator = fs.security_get_key_usage(rest_client.conninfo, rest_client.credentials, key_ref)

        if args.json:
            usages = [u for r in iterator for u in r.data['usages']]
            print(pretty_json(usages))
        else:
            usages = [(u['type'], u['id']) for r in iterator for u in r.data['usages']]
            usages.sort()
            print(tabulate(usages, ['usage_type', 'usage_id']))


class ModifyKeyCommand(qumulo.lib.opts.Subcommand):
    NAME = 'fs_security_modify_key'
    SYNOPSIS = textwrap.dedent(
        """\
        Modify the name or comment of a key in the file system key store.
        Enable or disable a key.
        """
    )

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '-k', '--key', required=True, help='The identifier or name of the key to modify.'
        )

        parser.add_argument('--new-name', type=str, help='The new name for the specified key.')
        parser.add_argument('--comment', type=str, help='The comment for the specified key.')

        disabled_group = parser.add_mutually_exclusive_group(required=False)
        disabled_group.set_defaults(disabled=None)
        disabled_group.add_argument(
            '--enable',
            dest='disabled',
            action='store_false',
            help="""
                Enable the specified key. You can use an enabled key to lock snapshots.
                You can also associate an enabled key with policies that take snapshots. Note: The
                system enables keys upon creation.
                """,
        )
        disabled_group.add_argument(
            '--disable',
            dest='disabled',
            action='store_true',
            help="""
                Disable the specified key. You cannot use a disabled key to lock new snapshots.
                Existing snapshots which use the disabled key remain locked. However, you can still
                use a disabled key to unlock the snapshots that it locked. Important: You cannot
                disable a key if any snapshot policy uses it.
                """,
        )

        parser.add_argument(
            '--json',
            action='store_true',
            help='Print the output in JSON format. By default, the output is in a table.',
        )

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        response = fs.security_modify_key(
            rest_client.conninfo,
            rest_client.credentials,
            key_ref=args.key,
            name=args.new_name,
            comment=args.comment,
            disabled=args.disabled,
        )
        if args.json:
            print(response)
        else:
            print_pretty_output([response.data])


class ListKeysCommand(qumulo.lib.opts.Subcommand):
    NAME = 'fs_security_list_keys'
    SYNOPSIS = 'List information for all keys in the file system key store.'

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '--json',
            action='store_true',
            help='Print the output in JSON format. By default, the output is in a table.',
        )

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        response = fs.security_list_keys(rest_client.conninfo, rest_client.credentials)
        if args.json:
            print(response)
        else:
            print_pretty_output(response.data['entries'])


class GetReplaceKeyChallengeCommand(qumulo.lib.opts.Subcommand):
    NAME = 'fs_security_get_key_replace_challenge'
    SYNOPSIS = (
        'Get a security challenge for replacing the specified key without affecting the snapshots'
        ' and snapshot policies associated with it.'
    )

    DESCRIPTION = textwrap.dedent(
        f"""
        {SYNOPSIS}

        Perform the following steps:
        1. Use the qq fs_security_get_key_replace_challenge command to get the security challenge.
        2. Use the qq fs_security_replace_key command to replace the key. For more information, run the qq fs_security_replace_key -h command.
        """  # noqa: E501
    )

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '-k',
            '--key',
            required=True,
            help='The identifier or name of the key for which to get a security challenge.',
        )

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        print(
            fs.security_get_key_replace_challenge(
                rest_client.conninfo, rest_client.credentials, args.key
            )
        )


class ReplaceKeyCommand(qumulo.lib.opts.Subcommand):
    NAME = 'fs_security_replace_key'
    SYNOPSIS = (
        'Replace the specified key without affecting the snapshots and snapshot policies associated'
        ' with it.'
    )
    DESCRIPTION = textwrap.dedent(
        f"""
        {SYNOPSIS}

        Examples:
            - To replace a key by using the verification signature, perform the following steps:
            1. Run the qq fs_security_get_key_replace_challenge command and specify the identifier or name of the key to replace.
            The command outputs a challenge.
            2. Use the existing private key and replacement private key to sign the challenge.
            3. Generate two verification signatures.
            4. Run the qq fs_security_replace_key command and specify the two verification signatures, the replacement public key, and the identifier or name of the key to replace. For example:
                qq fs_security_replace_key --key <key id> --replacement-public-key <Base64-encoded replacement public key contents> –replacement-key-verification-signature <key replacement challenge signed by using the replacement private key and encoded in Base64 format> --old-key-verification-signature <key replacement challenge signed by using the old private key and encoded in Base64 format>

            - To replace a key by using the private key files:
            qq fs_security_replace_key --key <key id> --old-private-key-file <the path to the old private key> --replacement-private-key-file <the path to the replacement private key>
        """  # noqa: E501
    )

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '-k',
            '--key',
            required=True,
            help='The identifier or name of the key to replace in the key store.',
        )

        parser.add_argument(
            '--old-private-key-file',
            type=argparse.FileType('r'),
            help='The ECDSA private key file that corresponds to the existing public key.',
        )
        parser.add_argument(
            '--replacement-private-key-file',
            type=argparse.FileType('r'),
            help='The ECDSA private key file with which to replace the existing key.',
        )
        parser.add_argument(
            '--replacement-public-key',
            type=str,
            help='The ECDSA public key with which to replace the existing key.',
        )
        parser.add_argument(
            '--old-key-verification-signature',
            type=str,
            help="""
                A signature that you generate when you sign the challenge with the existing ECDSA
                private key.
                """,
        )
        parser.add_argument(
            '--replacement-key-verification-signature',
            type=str,
            help="""
                A signature that you generate when you sign the challenge with the replacement
                ECDSA private key.
                """,
        )
        parser.add_argument(
            '--json',
            action='store_true',
            help='Print the output in JSON format. By default, the output is in a table.',
        )

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        if not mutually_exclusive(
            (args.replacement_private_key_file,),
            (args.replacement_public_key, args.replacement_key_verification_signature),
        ):
            sys.stderr.write(
                'Either the --replacement-private-key-file argument or both the ['
                ' --replacement-public-key and --replacement-key-verification-signature ] arguments'
                ' should be passed to this command.'
            )
            sys.exit(1)

        if not mutually_exclusive(
            (args.old_private_key_file,), (args.old_key_verification_signature,)
        ):
            sys.stderr.write(
                'Either the --old-private-key-file arguments or the'
                ' --old-key-verification-signature arguments should be passed to this command.'
            )
            sys.exit(2)

        if args.old_private_key_file or args.replacement_private_key_file:
            challenge = fs.security_get_key_replace_challenge(
                rest_client.conninfo, rest_client.credentials, args.key
            ).data['challenge']
        # These messages are necessary for differentiating the keys if they're password protected.
        if args.old_private_key_file:
            key = args.old_private_key_file.read()
            print('Signing challenge with old key.', file=sys.stderr)
            _, args.old_key_verification_signature = fs.get_verified_public_key(key, challenge)

        if args.replacement_private_key_file:
            key = args.replacement_private_key_file.read()
            print('Signing challenge with new key.', file=sys.stderr)
            (
                args.replacement_public_key,
                args.replacement_key_verification_signature,
            ) = fs.get_verified_public_key(key, challenge)

        response = fs.security_replace_key(
            rest_client.conninfo,
            rest_client.credentials,
            args.key,
            replacement_key=args.replacement_public_key,
            old_key_verification_signature=args.old_key_verification_signature,
            replacement_key_verification_signature=args.replacement_key_verification_signature,
        )
        if args.json:
            print(response)
        else:
            print_pretty_output([response.data])


class DeleteKeyCommand(qumulo.lib.opts.Subcommand):
    NAME = 'fs_security_delete_key'
    SYNOPSIS = 'Delete a key from the file system key store.'

    @staticmethod
    def options(parser: argparse.ArgumentParser) -> None:
        parser.add_argument(
            '-k',
            '--key',
            required=True,
            help='The identifier or name of the key to delete from the key store.',
        )

    @staticmethod
    def main(rest_client: RestClient, args: argparse.Namespace) -> None:
        fs.security_delete_key(rest_client.conninfo, rest_client.credentials, args.key)
