"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.withResolved = exports.isResolvableObject = exports.Tokenization = exports.Token = exports.TokenComparison = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * An enum-like class that represents the result of comparing two Tokens.
 * The return type of {@link Token.compareStrings}.
 */
class TokenComparison {
    constructor() {
    }
}
exports.TokenComparison = TokenComparison;
_a = JSII_RTTI_SYMBOL_1;
TokenComparison[_a] = { fqn: "@aws-cdk/core.TokenComparison", version: "1.145.0" };
/**
 * This means we're certain the two components are NOT
 * Tokens, and identical.
 */
TokenComparison.SAME = new TokenComparison();
/**
 * This means we're certain the two components are NOT
 * Tokens, and different.
 */
TokenComparison.DIFFERENT = new TokenComparison();
/** This means exactly one of the components is a Token. */
TokenComparison.ONE_UNRESOLVED = new TokenComparison();
/** This means both components are Tokens. */
TokenComparison.BOTH_UNRESOLVED = new TokenComparison();
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    constructor() {
    }
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_core_EncodingOptions(options);
        if (typeof value === 'string') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === 'number') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_core_EncodingOptions(options);
        if (Array.isArray(value) && value.every(x => typeof x === 'string')) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return isResolvableObject(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    /** Compare two strings that might contain Tokens with each other. */
    static compareStrings(possibleToken1, possibleToken2) {
        const firstIsUnresolved = Token.isUnresolved(possibleToken1);
        const secondIsUnresolved = Token.isUnresolved(possibleToken2);
        if (firstIsUnresolved && secondIsUnresolved) {
            return TokenComparison.BOTH_UNRESOLVED;
        }
        if (firstIsUnresolved || secondIsUnresolved) {
            return TokenComparison.ONE_UNRESOLVED;
        }
        return possibleToken1 === possibleToken2 ? TokenComparison.SAME : TokenComparison.DIFFERENT;
    }
}
exports.Token = Token;
_b = JSII_RTTI_SYMBOL_1;
Token[_b] = { fqn: "@aws-cdk/core.Token", version: "1.145.0" };
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    constructor() {
    }
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a string which is either a complete encoded token, or doesn't contain tokens at all
     *
     * It's illegal for the string to be a concatenation of an encoded token and something else.
     */
    static reverseCompleteString(s) {
        const fragments = Tokenization.reverseString(s);
        if (fragments.length !== 1) {
            throw new Error(`Tokenzation.reverseCompleteString: argument must not be a concatentation, got '${s}'`);
        }
        return fragments.firstToken;
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Reverse any value into a Resolvable, if possible
     *
     * In case of a string, the string must not be a concatenation.
     */
    static reverse(x, options = {}) {
        jsiiDeprecationWarnings._aws_cdk_core_ReverseOptions(options);
        if (Tokenization.isResolvable(x)) {
            return x;
        }
        if (typeof x === 'string') {
            if (options.failConcat === false) {
                // Handle this specially because reverseCompleteString might fail
                const fragments = Tokenization.reverseString(x);
                return fragments.length === 1 ? fragments.firstToken : undefined;
            }
            return Tokenization.reverseCompleteString(x);
        }
        if (Array.isArray(x)) {
            return Tokenization.reverseList(x);
        }
        if (typeof x === 'number') {
            return Tokenization.reverseNumber(x);
        }
        return undefined;
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        var _d;
        jsiiDeprecationWarnings._aws_cdk_core_ResolveOptions(options);
        return resolve_1.resolve(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: ((_d = options.preparing) !== null && _d !== void 0 ? _d : false),
            removeEmpty: options.removeEmpty,
        });
    }
    /**
     * Return whether the given object is an IResolvable object
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     */
    static isResolvable(obj) {
        return isResolvableObject(obj);
    }
    /**
     * Stringify a number directly or lazily if it's a Token. If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.uncachedString({
                produce: context => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== 'number' ? resolved : `${resolved}`;
                },
            });
        }
        else {
            return typeof x !== 'number' ? x : `${x}`;
        }
    }
}
exports.Tokenization = Tokenization;
_c = JSII_RTTI_SYMBOL_1;
Tokenization[_c] = { fqn: "@aws-cdk/core.Tokenization", version: "1.145.0" };
function isResolvableObject(x) {
    return typeof (x) === 'object' && x !== null && typeof x.resolve === 'function';
}
exports.isResolvableObject = isResolvableObject;
function withResolved(...args) {
    if (args.length < 2) {
        return;
    }
    const argArray = args.slice(0, args.length - 1);
    if (argArray.some(Token.isUnresolved)) {
        return;
    }
    args[args.length - 1].apply(arguments, argArray);
}
exports.withResolved = withResolved;
//# sourceMappingURL=data:application/json;base64,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