/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import '../style/index.css';
import { ReactWidget, UseSignal, Clipboard } from '@jupyterlab/apputils';
import { Signal } from '@phosphor/signaling';
import React from 'react';
import { CodeSnippetManager } from './CodeSnippet';
import { ExpandableComponent } from './ExpandableComponent';
/**
 * The CSS class added to code snippet widget.
 */
const CODE_SNIPPETS_CLASS = 'elyra-CodeSnippets';
const CODE_SNIPPETS_HEADER_CLASS = 'elyra-codeSnippetsHeader';
const CODE_SNIPPET_ITEM = 'elyra-codeSnippet-item';
const COPY_ICON_CLASS = 'elyra-copy-icon';
const INSERT_ICON_CLASS = 'elyra-add-icon';
/**
 * A React Component for code-snippets display list.
 */
class CodeSnippetTable extends React.Component {
    constructor() {
        // TODO: Use code mirror to display code
        // TODO: implement copy to clipboard command
        // TODO: implement insert code to file editor command (first check for code language matches file editor kernel language)
        super(...arguments);
        this.renderCodeSnippet = (codeSnippet) => {
            const displayName = '[' + codeSnippet.language + '] ' + codeSnippet.displayName;
            const actionButtons = [
                {
                    title: 'Copy',
                    iconClass: COPY_ICON_CLASS,
                    onClick: () => {
                        Clipboard.copyToSystem(codeSnippet.code.join('\n'));
                    }
                },
                {
                    title: 'Insert',
                    iconClass: INSERT_ICON_CLASS,
                    onClick: () => {
                        console.log('INSERT CODE BUTTON CLICKED');
                    }
                }
            ];
            return (React.createElement("div", { key: codeSnippet.name, className: CODE_SNIPPET_ITEM },
                React.createElement(ExpandableComponent, { displayName: displayName, tooltip: codeSnippet.description, actionButtons: actionButtons },
                    React.createElement("textarea", { defaultValue: codeSnippet.code.join('\n') }))));
        };
    }
    render() {
        return (React.createElement("div", null,
            React.createElement("div", { id: "codeSnippets" },
                React.createElement("div", null, this.props.codeSnippets.map(this.renderCodeSnippet)))));
    }
}
/**
 * A widget for Code Snippets.
 */
export class CodeSnippetWidget extends ReactWidget {
    constructor() {
        super();
        this.codeSnippetManager = new CodeSnippetManager();
        this.renderCodeSnippetsSignal = new Signal(this);
    }
    // Request code snippets from server
    async fetchData() {
        return await this.codeSnippetManager.findAll();
    }
    // Triggered when the widget button on side palette is clicked
    onAfterShow(msg) {
        this.fetchData().then((codeSnippets) => {
            this.renderCodeSnippetsSignal.emit(codeSnippets);
        });
    }
    render() {
        return (React.createElement("div", { className: CODE_SNIPPETS_CLASS },
            React.createElement("header", { className: CODE_SNIPPETS_HEADER_CLASS }, '</> Code Snippets'),
            React.createElement(UseSignal, { signal: this.renderCodeSnippetsSignal, initialArgs: [] }, (_, codeSnippets) => (React.createElement(CodeSnippetTable, { codeSnippets: codeSnippets })))));
    }
}
//# sourceMappingURL=CodeSnippetWidget.js.map