"use strict";
/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const application_1 = require("@elyra/application");
const canvas_1 = require("@elyra/canvas");
const apputils_1 = require("@jupyterlab/apputils");
const docregistry_1 = require("@jupyterlab/docregistry");
const algorithm_1 = require("@phosphor/algorithm");
require("@elyra/canvas/dist/common-canvas.min.css");
require("@elyra/canvas/dist/common-canvas.min.css");
require("carbon-components/css/carbon-components.min.css");
const React = __importStar(require("react"));
const react_intl_1 = require("react-intl");
const i18nData = __importStar(require("./en.json"));
const palette = __importStar(require("./palette.json"));
const PipelineSubmissionDialog_1 = require("./PipelineSubmissionDialog");
const properties = __importStar(require("./properties.json"));
const PIPELINE_ICON_CLASS = 'jp-MaterialIcon elyra-PipelineIcon';
const PIPELINE_CLASS = 'elyra-PipelineEditor';
exports.commandIDs = {
    openPipelineEditor: 'pipeline-editor:open',
    openDocManager: 'docmanager:open',
    newDocManager: 'docmanager:new-untitled'
};
/**
 * Wrapper Class for Common Canvas React Component
 */
class PipelineEditorWidget extends apputils_1.ReactWidget {
    constructor(props) {
        super(props);
        this.app = props.app;
        this.browserFactory = props.browserFactory;
        this.context = props.context;
        this.iconRegistry = props.iconRegistry;
    }
    render() {
        return (React.createElement(PipelineEditor, { app: this.app, browserFactory: this.browserFactory, iconRegistry: this.iconRegistry, widgetContext: this.context }));
    }
}
exports.PipelineEditorWidget = PipelineEditorWidget;
/**
 * Class for Common Canvas React Component
 */
class PipelineEditor extends React.Component {
    constructor(props) {
        super(props);
        this.position = 10;
        this.propertiesInfo = {
            parameterDef: this.addDockerImages(properties),
            appData: { id: '' }
        };
        this.app = props.app;
        this.browserFactory = props.browserFactory;
        this.iconRegistry = props.iconRegistry;
        this.canvasController = new canvas_1.CanvasController();
        this.canvasController.setPipelineFlowPalette(palette);
        this.widgetContext = props.widgetContext;
        this.widgetContext.ready.then(() => {
            this.canvasController.setPipelineFlow(this.widgetContext.model.toJSON());
        });
        this.toolbarMenuActionHandler = this.toolbarMenuActionHandler.bind(this);
        this.contextMenuHandler = this.contextMenuHandler.bind(this);
        this.contextMenuActionHandler = this.contextMenuActionHandler.bind(this);
        this.editActionHandler = this.editActionHandler.bind(this);
        this.state = { showPropertiesDialog: false, propertiesInfo: {} };
        this.applyPropertyChanges = this.applyPropertyChanges.bind(this);
        this.closePropertiesDialog = this.closePropertiesDialog.bind(this);
        this.openPropertiesDialog = this.openPropertiesDialog.bind(this);
        this.node = React.createRef();
        this.handleEvent = this.handleEvent.bind(this);
    }
    render() {
        const style = { height: '100%' };
        const emptyCanvasContent = (React.createElement("div", null,
            React.createElement("div", { className: "dragdrop" }),
            React.createElement("h1", null,
                ' ',
                "Start your new pipeline by dragging files from the file browser pane.",
                ' ')));
        const canvasConfig = {
            enableInternalObjectModel: true,
            emptyCanvasContent: emptyCanvasContent,
            enablePaletteLayout: 'Modal',
            paletteInitialState: false
        };
        const toolbarConfig = [
            { action: 'run', label: 'Run Pipeline', enable: true },
            { divider: true },
            { action: 'save', label: 'Save Pipeline', enable: true },
            { divider: true },
            // { action: 'open', label: 'Open Pipeline', enable: true },
            // { divider: true },
            { action: 'new', label: 'New Pipeline', enable: true },
            { divider: true },
            { action: 'clear', label: 'Clear Pipeline', enable: true },
            { divider: true },
            { action: 'undo', label: 'Undo', enable: true },
            { action: 'redo', label: 'Redo', enable: true },
            { action: 'cut', label: 'Cut', enable: false },
            { action: 'copy', label: 'Copy', enable: false },
            { action: 'paste', label: 'Paste', enable: false },
            { action: 'addComment', label: 'Add Comment', enable: true },
            { action: 'delete', label: 'Delete', enable: true },
            {
                action: 'arrangeHorizontally',
                label: 'Arrange Horizontally',
                enable: true
            },
            { action: 'arrangeVertically', label: 'Arrange Vertically', enable: true }
        ];
        const propertiesCallbacks = {
            applyPropertyChanges: this.applyPropertyChanges,
            closePropertiesDialog: this.closePropertiesDialog
        };
        const commProps = this.state.showPropertiesDialog ? (React.createElement(react_intl_1.IntlProvider, { key: "IntlProvider2", locale: 'en', messages: i18nData.messages },
            React.createElement(canvas_1.CommonProperties, { propertiesInfo: this.propertiesInfo, propertiesConfig: {}, callbacks: propertiesCallbacks }))) : null;
        return (React.createElement("div", { style: style, ref: this.node },
            React.createElement(canvas_1.CommonCanvas, { canvasController: this.canvasController, toolbarMenuActionHandler: this.toolbarMenuActionHandler, contextMenuHandler: this.contextMenuHandler, contextMenuActionHandler: this.contextMenuActionHandler, editActionHandler: this.editActionHandler, toolbarConfig: toolbarConfig, config: canvasConfig }),
            commProps));
    }
    addDockerImages(properties) {
        let firstImage = true;
        const dockerImages = application_1.FrontendServices.getDockerImages();
        const imageEnum = [];
        for (const image in dockerImages) {
            if (firstImage) {
                properties.current_parameters.image = image;
                firstImage = false;
            }
            imageEnum.push(image);
            properties.resources['image.' + image + '.label'] = dockerImages[image];
        }
        properties.parameters[0].enum = imageEnum;
        return properties;
    }
    openPropertiesDialog(source) {
        console.log('Opening properties dialog');
        const node_id = source.targetObject.id;
        const app_data = this.canvasController.getNode(node_id).app_data;
        const node_props = this.propertiesInfo;
        node_props.appData.id = node_id;
        node_props.parameterDef.current_parameters.image = app_data.image;
        node_props.parameterDef.current_parameters.outputs = app_data.outputs;
        node_props.parameterDef.current_parameters.vars = app_data.vars;
        node_props.parameterDef.current_parameters.dependencies =
            app_data.dependencies;
        node_props.parameterDef.current_parameters.recursive_dependencies =
            app_data.recursive_dependencies;
        this.setState({ showPropertiesDialog: true, propertiesInfo: node_props });
    }
    applyPropertyChanges(propertySet, appData) {
        console.log('Applying changes to properties');
        const app_data = this.canvasController.getNode(appData.id).app_data;
        app_data.image = propertySet.image;
        app_data.outputs = propertySet.outputs;
        app_data.vars = propertySet.vars;
        app_data.dependencies = propertySet.dependencies;
        app_data.recursive_dependencies = propertySet.recursive_dependencies;
    }
    closePropertiesDialog() {
        console.log('Closing properties dialog');
        this.setState({ showPropertiesDialog: false, propertiesInfo: {} });
    }
    contextMenuHandler(source, defaultMenu) {
        let customMenu = defaultMenu;
        if (source.type === 'node') {
            if (source.selectedObjectIds.length > 1) {
                customMenu = customMenu.concat({
                    action: 'openNotebook',
                    label: 'Open Notebooks'
                });
            }
            else {
                customMenu = customMenu.concat({
                    action: 'openNotebook',
                    label: 'Open Notebook'
                });
            }
            customMenu = customMenu.concat({
                action: 'properties',
                label: 'Properties'
            });
        }
        return customMenu;
    }
    contextMenuActionHandler(action, source) {
        if (action === 'openNotebook' && source.type === 'node') {
            const nodes = source.selectedObjectIds;
            for (let i = 0; i < nodes.length; i++) {
                const path = this.canvasController.getNode(nodes[i]).app_data.artifact;
                this.app.commands.execute(exports.commandIDs.openDocManager, { path });
            }
        }
        else if (action === 'properties' && source.type === 'node') {
            if (this.state.showPropertiesDialog) {
                this.closePropertiesDialog();
            }
            else {
                this.openPropertiesDialog(source);
            }
        }
    }
    /*
     * Handles creating new nodes in the canvas
     */
    editActionHandler(data) {
        this.widgetContext.model.fromJSON(this.canvasController.getPipelineFlow());
    }
    handleAdd(x, y) {
        let failedAdd = 0;
        let position = 0;
        const missingXY = !(x && y);
        // if either x or y is undefined use the default coordinates
        if (missingXY) {
            position = this.position;
            x = 75;
            y = 85;
        }
        const fileBrowser = this.browserFactory.defaultBrowser;
        algorithm_1.toArray(fileBrowser.selectedItems()).map(item => {
            // if the selected item is a file
            if (item.type == 'notebook') {
                //add each selected notebook
                console.log('Adding ==> ' + item.path);
                const nodeTemplate = this.canvasController.getPaletteNode('execute-notebook-node');
                if (nodeTemplate) {
                    const data = {
                        editType: 'createNode',
                        offsetX: x + position,
                        offsetY: y + position,
                        nodeTemplate: this.canvasController.convertNodeTemplate(nodeTemplate)
                    };
                    // create a notebook widget to get a string with the node content then dispose of it
                    const notebookWidget = fileBrowser.model.manager.open(item.path);
                    const notebookStr = notebookWidget.content.model.toString();
                    notebookWidget.dispose();
                    const vars = application_1.NotebookParser.getEnvVars(notebookStr).map(str => str + '=');
                    data.nodeTemplate.label = item.path.replace(/^.*[\\/]/, '');
                    data.nodeTemplate.label = data.nodeTemplate.label.replace(/\.[^/.]+$/, '');
                    data.nodeTemplate.image =
                        'data:image/svg+xml;utf8,' +
                            encodeURIComponent(this.iconRegistry.svg('notebook'));
                    data.nodeTemplate.app_data['artifact'] = item.path;
                    data.nodeTemplate.app_data['image'] = this.propertiesInfo.parameterDef.current_parameters.image;
                    data.nodeTemplate.app_data['vars'] = vars;
                    data.nodeTemplate.app_data['recursive_dependencies'] = this.propertiesInfo.parameterDef.current_parameters.recursive_dependencies;
                    this.canvasController.editActionHandler(data);
                    position += 20;
                }
            }
            else {
                failedAdd++;
            }
        });
        // update position if the default coordinates were used
        if (missingXY) {
            this.position = position;
        }
        if (failedAdd) {
            return apputils_1.showDialog({
                title: 'Unsupported File(s)',
                body: 'Currently, only selected notebook files can be added to a pipeline',
                buttons: [apputils_1.Dialog.okButton()]
            });
        }
    }
    handleRun() {
        application_1.SubmissionHandler.makeGetRequest('api/metadata/runtimes', 'pipeline', (response) => {
            if (Object.keys(response.runtimes).length === 0) {
                return application_1.SubmissionHandler.noMetadataError('runtimes');
            }
            apputils_1.showDialog({
                title: 'Run pipeline',
                body: new PipelineSubmissionDialog_1.PipelineSubmissionDialog({ runtimes: response.runtimes }),
                buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton()],
                focusNodeSelector: '#pipeline_name'
            }).then(result => {
                if (result.value == null) {
                    // When Cancel is clicked on the dialog, just return
                    return;
                }
                // prepare pipeline submission details
                const pipelineFlow = this.canvasController.getPipelineFlow();
                pipelineFlow.pipelines[0]['app_data']['title'] =
                    result.value.pipeline_name;
                // TODO: Be more flexible and remove hardcoded runtime type
                pipelineFlow.pipelines[0]['app_data']['runtime'] = 'kfp';
                pipelineFlow.pipelines[0]['app_data']['runtime-config'] =
                    result.value.runtime_config;
                application_1.SubmissionHandler.submitPipeline(pipelineFlow, result.value.runtime_config, 'pipeline');
            });
        });
    }
    handleSave() {
        this.widgetContext.model.fromJSON(this.canvasController.getPipelineFlow());
        this.widgetContext.save();
    }
    handleOpen() {
        algorithm_1.toArray(this.browserFactory.defaultBrowser.selectedItems()).map(item => {
            // if the selected item is a file
            if (item.type != 'directory') {
                console.log('Opening ==> ' + item.path);
                this.app.commands.execute(exports.commandIDs.openDocManager, {
                    path: item.path
                });
            }
        });
    }
    handleNew() {
        // Clears the canvas, then creates a new file and sets the pipeline_name field to the new name.
        this.app.commands.execute(exports.commandIDs.openPipelineEditor);
    }
    handleClear() {
        return apputils_1.showDialog({
            title: 'Clear Pipeline?',
            body: 'Are you sure you want to clear? You can not undo this.',
            buttons: [apputils_1.Dialog.cancelButton(), apputils_1.Dialog.okButton({ label: 'Clear' })]
        }).then(result => {
            if (result.button.accept) {
                this.canvasController.clearPipelineFlow();
                this.widgetContext.model.fromJSON(this.canvasController.getPipelineFlow());
                this.position = 10;
            }
        });
    }
    /**
     * Handles submitting pipeline runs
     */
    toolbarMenuActionHandler(action, source) {
        console.log('Handling action: ' + action);
        if (action == 'run') {
            // When executing the pipeline
            this.handleRun();
        }
        else if (action == 'save') {
            this.handleSave();
        }
        else if (action == 'open') {
            this.handleOpen();
        }
        else if (action == 'new') {
            this.handleNew();
        }
        else if (action == 'clear') {
            this.handleClear();
        }
    }
    componentDidMount() {
        const node = this.node.current;
        node.addEventListener('dragenter', this.handleEvent);
        node.addEventListener('dragover', this.handleEvent);
        node.addEventListener('p-dragover', this.handleEvent);
        node.addEventListener('p-drop', this.handleEvent);
    }
    componentWillUnmount() {
        const node = this.node.current;
        node.removeEventListener('p-drop', this.handleEvent);
        node.removeEventListener('p-dragover', this.handleEvent);
        node.removeEventListener('dragover', this.handleEvent);
        node.removeEventListener('dragenter', this.handleEvent);
    }
    /**
     * Handle the DOM events.
     *
     * @param event - The DOM event.
     */
    handleEvent(event) {
        switch (event.type) {
            case 'dragenter':
                event.preventDefault();
                break;
            case 'dragover':
                event.preventDefault();
                break;
            case 'p-dragover':
                event.preventDefault();
                event.stopPropagation();
                event.dropAction = event.proposedAction;
                break;
            case 'p-drop':
                event.preventDefault();
                event.stopPropagation();
                this.handleAdd(event.offsetX, event.offsetY);
                break;
            default:
                break;
        }
    }
}
exports.PipelineEditor = PipelineEditor;
class PipelineEditorFactory extends docregistry_1.ABCWidgetFactory {
    constructor(options) {
        super(options);
        this.app = options.app;
        this.browserFactory = options.browserFactory;
        this.iconRegistry = options.iconRegistry;
    }
    createNewWidget(context) {
        // Creates a blank widget with a DocumentWidget wrapper
        const props = {
            app: this.app,
            browserFactory: this.browserFactory,
            iconRegistry: this.iconRegistry,
            context: context
        };
        const content = new PipelineEditorWidget(props);
        const widget = new docregistry_1.DocumentWidget({
            content,
            context,
            node: document.createElement('div')
        });
        widget.addClass(PIPELINE_CLASS);
        widget.title.iconClass = PIPELINE_ICON_CLASS;
        return widget;
    }
}
exports.PipelineEditorFactory = PipelineEditorFactory;
//# sourceMappingURL=PipelineEditorWidget.js.map