"""
Switches on Zigbee Home Automation networks.

For more details on this platform, please refer to the documentation
at https://home-assistant.io/components/switch.zha/
"""
import logging

from homeassistant.components.switch import DOMAIN, SwitchDevice
from homeassistant.helpers.dispatcher import async_dispatcher_connect
from .core.const import (
    DATA_ZHA, DATA_ZHA_DISPATCHERS, ZHA_DISCOVERY_NEW, LISTENER_ON_OFF,
    SIGNAL_ATTR_UPDATED
)
from .entity import ZhaEntity

_LOGGER = logging.getLogger(__name__)

DEPENDENCIES = ['zha']


async def async_setup_platform(hass, config, async_add_entities,
                               discovery_info=None):
    """Old way of setting up Zigbee Home Automation switches."""
    pass


async def async_setup_entry(hass, config_entry, async_add_entities):
    """Set up the Zigbee Home Automation switch from config entry."""
    async def async_discover(discovery_info):
        await _async_setup_entities(hass, config_entry, async_add_entities,
                                    [discovery_info])

    unsub = async_dispatcher_connect(
        hass, ZHA_DISCOVERY_NEW.format(DOMAIN), async_discover)
    hass.data[DATA_ZHA][DATA_ZHA_DISPATCHERS].append(unsub)

    switches = hass.data.get(DATA_ZHA, {}).get(DOMAIN)
    if switches is not None:
        await _async_setup_entities(hass, config_entry, async_add_entities,
                                    switches.values())
        del hass.data[DATA_ZHA][DOMAIN]


async def _async_setup_entities(hass, config_entry, async_add_entities,
                                discovery_infos):
    """Set up the ZHA switches."""
    entities = []
    for discovery_info in discovery_infos:
        entities.append(Switch(**discovery_info))

    async_add_entities(entities, update_before_add=True)


class Switch(ZhaEntity, SwitchDevice):
    """ZHA switch."""

    _domain = DOMAIN

    def __init__(self, **kwargs):
        """Initialize the ZHA switch."""
        super().__init__(**kwargs)
        self._on_off_listener = self.cluster_listeners.get(LISTENER_ON_OFF)

    @property
    def is_on(self) -> bool:
        """Return if the switch is on based on the statemachine."""
        if self._state is None:
            return False
        return self._state

    async def async_turn_on(self, **kwargs):
        """Turn the entity on."""
        await self._on_off_listener.on()

    async def async_turn_off(self, **kwargs):
        """Turn the entity off."""
        await self._on_off_listener.off()

    def async_set_state(self, state):
        """Handle state update from listener."""
        self._state = bool(state)
        self.async_schedule_update_ha_state()

    @property
    def device_state_attributes(self):
        """Return state attributes."""
        return self.state_attributes

    async def async_added_to_hass(self):
        """Run when about to be added to hass."""
        await super().async_added_to_hass()
        await self.async_accept_signal(
            self._on_off_listener, SIGNAL_ATTR_UPDATED, self.async_set_state)
