"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const kms = require("aws-cdk-lib/aws-kms");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("@aws-solutions-constructs/core");
test('Default construct has all expected properties', () => {
    const stack = new cdk.Stack();
    const construct = new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    // By default, we don't create a VPC
    expect(construct.vpc).toBeUndefined();
});
test('New VPC is created when deployVpc flag is true', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        deployVpc: true
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get the VPC created by the construct
    template.resourceCountIs('AWS::EC2::VPC', 1);
    const vpcResource = template.findResources('AWS::EC2::VPC');
    const [vpcId] = Object.keys(vpcResource);
    verifyLambdaFunctionVpcProps(template, vpcId, "Isolated");
});
test('Existing VPC is used when specified', () => {
    const stack = new cdk.Stack();
    const existingVpc = new ec2.Vpc(stack, 'test-vpc', { vpcName: 'my-vpc-name' });
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        existingVpc
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get the VPC by its name
    template.resourceCountIs('AWS::EC2::VPC', 1);
    const vpcResource = template.findResources('AWS::EC2::VPC');
    const [vpcId] = Object.keys(vpcResource);
    verifyLambdaFunctionVpcProps(template, vpcId, "Private");
});
test('New VPC is created from user-provided vpcProps', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        vpcProps: {
            vpcName: 'my-vpc-name'
        },
        deployVpc: true
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get the VPC by its name
    template.resourceCountIs('AWS::EC2::VPC', 1);
    const vpcResource = template.findResources('AWS::EC2::VPC');
    const [vpcId] = Object.keys(vpcResource);
    verifyLambdaFunctionVpcProps(template, vpcId, "Isolated");
});
test('Lambda Function has default stream environment variable', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                KINESIS_DATASTREAM_NAME: {
                    Ref: kinesisStreamId
                }
            }
        }
    });
});
test('Lambda Function stream name environment variable can be overridden', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        streamEnvironmentVariableName: 'CUSTOM_ENV_VAR_NAME'
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                CUSTOM_ENV_VAR_NAME: {
                    Ref: kinesisStreamId
                }
            }
        }
    });
});
test('Kinesis Stream is encrypted with AWS-managed CMK by default', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
});
test('CloudWatch Alarms are created for Kinesis Stream by default', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Kinesis',
        MetricName: 'GetRecords.IteratorAgeMilliseconds'
    });
    template.hasResourceProperties('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Kinesis',
        MetricName: 'ReadProvisionedThroughputExceeded'
    });
});
test('CloudWatch Alarms are not created when createCloudWatchAlarms property is set to false', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        createCloudWatchAlarms: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::CloudWatch::Alarm', 0);
});
test('Error is thrown when vpc is specified and existing lambda function is not associated with it', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'test-lamba', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    });
    const existingVpc = new ec2.Vpc(stack, 'test-vpc', {});
    const app = () => {
        new lib_1.LambdaToKinesisStreams(stack, 'lambda-to-sqs-stack', {
            existingLambdaObj,
            existingVpc
        });
    };
    expect(app).toThrowError();
});
test('Construct uses existing Lambda Function', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'test-lamba', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        functionName: 'my-lambda-function'
    });
    new lib_1.LambdaToKinesisStreams(stack, 'lambda-to-sqs-stack', {
        existingLambdaObj
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'my-lambda-function'
    });
});
test('Construct uses existing Kinesis Stream', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingStreamObj = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'my-stream',
    });
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        existingStreamObj
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        Name: 'my-stream'
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('Construct uses unencrypted existing stream', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingStreamObj = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'my-stream',
        encryption: kinesis.StreamEncryption.UNENCRYPTED
    });
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        existingStreamObj
    });
    const template = assertions_1.Template.fromStack(stack);
    defaults.expectNonexistence(stack, 'AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('Construct uses unencrypted streams from stream props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        kinesisStreamProps: {
            streamName: 'my-stream',
            encryption: kinesis.StreamEncryption.UNENCRYPTED
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    defaults.expectNonexistence(stack, 'AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('Construct grants PutRecord permission for the Lambda Function to write to the Kinesis Stream', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'xray:PutTraceSegments',
                        'xray:PutTelemetryRecords'
                    ],
                    Effect: 'Allow',
                    Resource: '*'
                },
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            kinesisStreamId,
                            'Arn'
                        ]
                    }
                }
            ]
        }
    });
});
test('When a Customer-manged CMK is used on an existing stream, construct grants the Lambda Function permission to use the encryption key so it can publish messages to it', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        description: 'my-key-description'
    });
    const existingStreamObj = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'my-stream',
        encryptionKey
    });
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        existingStreamObj
    });
    const template = assertions_1.Template.fromStack(stack);
    const resource = template.findResources('AWS::KMS::Key', {
        Properties: {
            Description: assertions_1.Match.exact('my-key-description')
        }
    });
    const [kmsKey] = Object.keys(resource);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'xray:PutTraceSegments',
                        'xray:PutTelemetryRecords'
                    ],
                    Effect: 'Allow',
                    Resource: '*'
                },
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'teststream04374A09',
                            'Arn'
                        ]
                    }
                },
                {
                    Action: [
                        'kms:Encrypt',
                        'kms:ReEncrypt*',
                        'kms:GenerateDataKey*'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            kmsKey,
                            'Arn'
                        ]
                    }
                },
                {
                    Action: [
                        'kms:Decrypt',
                        'kms:GenerateDataKey*'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            kmsKey,
                            'Arn'
                        ]
                    }
                }
            ]
        }
    });
});
function verifyLambdaFunctionVpcProps(template, vpcId, subnetType) {
    // Get the Security Group associated with the VPC
    const securityGroupResource = template.findResources('AWS::EC2::SecurityGroup', {
        Properties: {
            VpcId: {
                Ref: assertions_1.Match.exact(vpcId)
            }
        }
    });
    const [securityGroupId] = Object.keys(securityGroupResource);
    // Get the Private or Isolated Subnets associated with the VPC
    const subnetResources = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            Tags: assertions_1.Match.arrayWith([
                {
                    Key: "aws-cdk:subnet-type",
                    Value: subnetType
                }
            ])
        }
    });
    const [subnet1, subnet2] = Object.keys(subnetResources);
    // Verify the Lambda Function has the same Security Group
    template.hasResourceProperties('AWS::Lambda::Function', {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    'Fn::GetAtt': [
                        securityGroupId,
                        'GroupId'
                    ]
                }
            ],
            SubnetIds: [
                {
                    Ref: subnet1
                },
                {
                    Ref: subnet2
                }
            ]
        }
    });
    // Verify the VPC has an interface endpoint for Kinesis Streams
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.kinesis-streams'
                ]
            ]
        },
        VpcId: {
            Ref: vpcId
        },
    });
    // Verify the VPC has dns hostnames and support enabled
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
}
test('Confirm CheckVpcProps() is being called', () => {
    const stack = new cdk.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: lambda.Runtime.NODEJS_18_X,
                handler: 'index.handler'
            },
            deployVpc: true,
            existingVpc: vpc
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToKinesisStreams(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
test('Confirm call to CheckKinesisStreamProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_16_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingStreamObj: new kinesis.Stream(stack, 'test', {}),
        kinesisStreamProps: {}
    };
    const app = () => {
        new lib_1.LambdaToKinesisStreams(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingStreamObj or kinesisStreamProps, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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