# -*- coding: utf-8 -*-
#
# Copyright (c) 2016 -- Lars Heuer - Semagia <http://www.semagia.com/>.
# All rights reserved.
#
# License: BSD License
#
"""\
Constants.

Internal module. May change without further warning.
"""
from collections import namedtuple

ALPHANUMERIC_CHARS = br'0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ $%*+-./:'

# ISO/IEC 18004:2015(E) -- Table 2 — Mode indicators for QR Code (page 23)
MODE_NUMERIC = 0x1
MODE_ALPHANUMERIC = 0x2
MODE_BYTE = 0x4
MODE_ECI = 0x7
MODE_KANJI = 0x8

# Micro QR Code uses different mode indicators
# ISO/IEC 18004:2015(E) -- Table 2 — Mode indicators for QR Code (page 23)
MODE_TO_MICRO_MODE_MAPPING = {
    MODE_NUMERIC: 0x0,
    MODE_ALPHANUMERIC: 0x1,
    MODE_BYTE: 0x2,
    MODE_KANJI: 0x3,
}

# Note: These versions must be comparable: Version 1 > M4 > M3 > M2 > M1
VERSION_M4 = 0
VERSION_M3 = -1
VERSION_M2 = -2
VERSION_M1 = -3

MICRO_VERSION_MAPPING = {
    'M1': VERSION_M1,
    'M2': VERSION_M2,
    'M3': VERSION_M3,
    'M4': VERSION_M4,
}

MICRO_VERSIONS = tuple(sorted(MICRO_VERSION_MAPPING.values()))

# ISO/IEC 18004:2015(E)
# Table 12 — Error correction level indicators for QR Code symbols (page 55)
ERROR_LEVEL_L = 1
ERROR_LEVEL_M = 0
ERROR_LEVEL_Q = 3
ERROR_LEVEL_H = 2

ERROR_LEVEL_TO_MICRO_MAPPING = {
    VERSION_M1: {None: 0},
    VERSION_M2: {ERROR_LEVEL_L: 1,
                 ERROR_LEVEL_M: 2},
    VERSION_M3: {ERROR_LEVEL_L: 3,
                 ERROR_LEVEL_M: 4},
    VERSION_M4: {ERROR_LEVEL_L: 5,
                 ERROR_LEVEL_M: 6,
                 ERROR_LEVEL_Q: 7},
}

DEFAULT_BYTE_ENCODING = 'iso-8859-1'
KANJI_ENCODING = 'shift_jis'

MODE_MAPPING = {
    'numeric': MODE_NUMERIC,
    'alphanumeric': MODE_ALPHANUMERIC,
    'byte': MODE_BYTE,
    'kanji': MODE_KANJI,
}

ERROR_MAPPING = {
    'L': ERROR_LEVEL_L,
    'M': ERROR_LEVEL_M,
    'Q': ERROR_LEVEL_Q,
    'H': ERROR_LEVEL_H,
}

#
# ISO/IEC 18004:2015(E) -- 7.3.2 Extended Channel Interpretation (ECI) mode (page 20)
#
# <https://strokescribe.com/en/ECI.html>
# ECI       Reference
# ------    ---------
# 000000    Represents the default encodation scheme
# 000001    Represents the GLI encodation scheme of a number of symbologies
#           with characters 0 to 127 being identical to those of
#           ISO/IEC 646 : 1991 IRV (equivalent to ANSI X3.4) and characters
#           128 to 255 being identical to those values of ISO 8859-1
# 000002    An equivalent code table to ECI 000000, without the return-to-GLI 0
#           logic. It is the default encodation scheme for encoders fully
#           compliant with this standard.
# 000003    ISO/IEC 8859-1 Latin alphabet No. 1
# 000004    ISO/IEC 8859-2 Latin alphabet No. 2
# 000005    ISO/IEC 8859-3 Latin alphabet No. 3
# 000006    ISO/IEC 8859-4 Latin alphabet No. 4
# 000007    ISO/IEC 8859-5 Latin/Cyrillic alphabet
# 000008    ISO/IEC 8859-6 Latin/Arabic alphabet
# 000009    ISO/IEC 8859-7 Latin/Greek alphabet
# 000010    ISO/IEC 8859-8 Latin/Hebrew alphabet
# 000011    ISO/IEC 8859-9 Latin alphabet No. 5
# 000012    ISO/IEC 8859-10 Latin alphabet No. 6
# 000013    ISO/IEC 8859-11 Latin/Thai alphabet
# 000014    Reserved
# 000015    ISO/IEC 8859-13 Latin alphabet No. 7 (Baltic Rim)
# 000016    ISO/IEC 8859-14 Latin alphabet No. 8 (Celtic)
# 000017    ISO/IEC 8859-15 Latin alphabet No. 9
# 000018    ISO/IEC 8859-16 Latin alphabet No. 10
# 000019    Reserved
# 000020    Shift JIS (JIS X 0208 Annex 1 + JIS X 0201)
# 000021    Windows 1250 Latin 2 (Central Europe)
# 000022    Windows 1251 Cyrillic
# 000023    Windows 1252 Latin 1
# 000024    Windows 1256 Arabic
# 000025    ISO/IEC 10646 UCS-2 (High order byte first)
# 000026    ISO/IEC 10646 UTF-8 (See information above)
# 000027    ISO/IEC 646:1991 International Reference Version of ISO 7-bit
#           coded character set
# 000028    Big 5 (Taiwan) Chinese Character Set
# 000029    GB (PRC) Chinese Character Set
# 000030    Korean Character Set
ECI_ASSIGNMENT_NUM = {
    # Codecs name (``codecs.lookup(some-charset).name``) -> ECI designator
    'cp437': 1,
    'iso8859-1': 3,
    'iso8859-2': 4,
    'iso8859-3': 5,
    'iso8859-4': 6,
    'iso8859-5': 7,
    'iso8859-6': 8,
    'iso8859-7': 9,
    'iso8859-8': 10,
    'iso8859-9': 11,
    'iso8859-10': 12,
    'iso8859-11': 13,
    'iso8859-13': 15,
    'iso8859-14': 16,
    'iso8859-15': 17,
    'iso8859-16': 18,
    'shift_jis': 20,
    'cp1250': 21,
    'cp1251': 22,
    'cp1252': 23,
    'cp1256': 24,
    'utf-16-be': 25,
    'utf-8': 26,
    'ascii': 27,
    'big5': 28,
    'gb18030': 29, 'gbk': 29,  # GBK is treated as GB-18030
    'euc_kr': 30,
}


# ISO/IEC 18004:2015(E) -- Table 2 — Mode indicators for QR Code (page 23)
SUPPORTED_MODES = {
    MODE_NUMERIC: (None, VERSION_M1, VERSION_M2, VERSION_M3, VERSION_M4),
    MODE_ALPHANUMERIC: (None, VERSION_M2, VERSION_M3, VERSION_M4),
    MODE_BYTE: (None, VERSION_M3, VERSION_M4),
    MODE_ECI: (None,),
    MODE_KANJI: (None, VERSION_M3, VERSION_M4),
}

# ISO/IEC 18004:2015(E) -- Table 2 — Mode indicators for QR Code (page 23)
TERMINATOR_LENGTH = {
    None: 4,  # QR Codes, all versions
    VERSION_M1: 3,
    VERSION_M2: 5,
    VERSION_M3: 7,
    VERSION_M4: 9
}

VERSION_RANGE_01_09 = 1  # Version  1 ..  9
VERSION_RANGE_10_26 = 2  # Version 10 .. 26
VERSION_RANGE_27_40 = 3  # Version 27 .. 40

# ISO/IEC 18004:2015(E)
# Table 3 — Number of bits in character count indicator for QR Code (page 23)
CHAR_COUNT_INDICATOR_LENGTH = {
    MODE_NUMERIC: {
        VERSION_RANGE_01_09: 10,
        VERSION_RANGE_10_26: 12,
        VERSION_RANGE_27_40: 14, VERSION_M1: 3, VERSION_M2: 4, VERSION_M3: 5, VERSION_M4: 6},
    MODE_ALPHANUMERIC: {
        VERSION_RANGE_01_09:  9,
        VERSION_RANGE_10_26: 11,
        VERSION_RANGE_27_40: 13,                VERSION_M2: 3, VERSION_M3: 4, VERSION_M4: 5},
    MODE_BYTE: {
        VERSION_RANGE_01_09:  8,
        VERSION_RANGE_10_26: 16,
        VERSION_RANGE_27_40: 16,                               VERSION_M3: 4, VERSION_M4: 5},
    MODE_KANJI: {
        VERSION_RANGE_01_09:  8,
        VERSION_RANGE_10_26: 10,
        VERSION_RANGE_27_40: 12,                               VERSION_M3: 3, VERSION_M4: 4},
}


# ISO/IEC 18004:2015(E) - 6.4.10 Bit stream to codeword conversion (page 33)
# Table 7 — Number of symbol characters and input data capacity for QR Code
SYMBOL_CAPACITY = {
    VERSION_M1: {
           None:          {0:    20, MODE_NUMERIC:    5}},
    VERSION_M2: {
           ERROR_LEVEL_L: {0:    40, MODE_NUMERIC:   10, MODE_ALPHANUMERIC:    6},
           ERROR_LEVEL_M: {0:    32, MODE_NUMERIC:    8, MODE_ALPHANUMERIC:    5}},
    VERSION_M3: {
           ERROR_LEVEL_L: {0:    84, MODE_NUMERIC:   23, MODE_ALPHANUMERIC:   14, MODE_BYTE:    9, MODE_KANJI:    6},
           ERROR_LEVEL_M: {0:    68, MODE_NUMERIC:   18, MODE_ALPHANUMERIC:   11, MODE_BYTE:    7, MODE_KANJI:    4}},
    VERSION_M4: {
           ERROR_LEVEL_L: {0:   128, MODE_NUMERIC:   35, MODE_ALPHANUMERIC:   21, MODE_BYTE:   15, MODE_KANJI:    9},
           ERROR_LEVEL_M: {0:   112, MODE_NUMERIC:   30, MODE_ALPHANUMERIC:   18, MODE_BYTE:   13, MODE_KANJI:    8},
           ERROR_LEVEL_Q: {0:    80, MODE_NUMERIC:   21, MODE_ALPHANUMERIC:   13, MODE_BYTE:    9, MODE_KANJI:    5}},
    1:    {ERROR_LEVEL_L: {0:   152, MODE_NUMERIC:   41, MODE_ALPHANUMERIC:   25, MODE_BYTE:   17, MODE_KANJI:   10},
           ERROR_LEVEL_M: {0:   128, MODE_NUMERIC:   34, MODE_ALPHANUMERIC:   20, MODE_BYTE:   14, MODE_KANJI:    8},
           ERROR_LEVEL_Q: {0:   104, MODE_NUMERIC:   27, MODE_ALPHANUMERIC:   16, MODE_BYTE:   11, MODE_KANJI:    7},
           ERROR_LEVEL_H: {0:    72, MODE_NUMERIC:   17, MODE_ALPHANUMERIC:   10, MODE_BYTE:    7, MODE_KANJI:    4}},
    2:    {ERROR_LEVEL_L: {0:   272, MODE_NUMERIC:   77, MODE_ALPHANUMERIC:   47, MODE_BYTE:   32, MODE_KANJI:   20},
           ERROR_LEVEL_M: {0:   224, MODE_NUMERIC:   63, MODE_ALPHANUMERIC:   38, MODE_BYTE:   26, MODE_KANJI:   16},
           ERROR_LEVEL_Q: {0:   176, MODE_NUMERIC:   48, MODE_ALPHANUMERIC:   29, MODE_BYTE:   20, MODE_KANJI:   12},
           ERROR_LEVEL_H: {0:   128, MODE_NUMERIC:   34, MODE_ALPHANUMERIC:   20, MODE_BYTE:   14, MODE_KANJI:    8}},
    3:    {ERROR_LEVEL_L: {0:   440, MODE_NUMERIC:  127, MODE_ALPHANUMERIC:   77, MODE_BYTE:   53, MODE_KANJI:   32},
           ERROR_LEVEL_M: {0:   352, MODE_NUMERIC:  101, MODE_ALPHANUMERIC:   61, MODE_BYTE:   42, MODE_KANJI:   26},
           ERROR_LEVEL_Q: {0:   272, MODE_NUMERIC:   77, MODE_ALPHANUMERIC:   47, MODE_BYTE:   32, MODE_KANJI:   20},
           ERROR_LEVEL_H: {0:   208, MODE_NUMERIC:   58, MODE_ALPHANUMERIC:   35, MODE_BYTE:   24, MODE_KANJI:   15}},
    4:    {ERROR_LEVEL_L: {0:   640, MODE_NUMERIC:  187, MODE_ALPHANUMERIC:  114, MODE_BYTE:   78, MODE_KANJI:   48},
           ERROR_LEVEL_M: {0:   512, MODE_NUMERIC:  149, MODE_ALPHANUMERIC:   90, MODE_BYTE:   62, MODE_KANJI:   38},
           ERROR_LEVEL_Q: {0:   384, MODE_NUMERIC:  111, MODE_ALPHANUMERIC:   67, MODE_BYTE:   46, MODE_KANJI:   28},
           ERROR_LEVEL_H: {0:   288, MODE_NUMERIC:   82, MODE_ALPHANUMERIC:   50, MODE_BYTE:   34, MODE_KANJI:   21}},
    5:    {ERROR_LEVEL_L: {0:   864, MODE_NUMERIC:  255, MODE_ALPHANUMERIC:  154, MODE_BYTE:  106, MODE_KANJI:   65},
           ERROR_LEVEL_M: {0:   688, MODE_NUMERIC:  202, MODE_ALPHANUMERIC:  122, MODE_BYTE:   84, MODE_KANJI:   52},
           ERROR_LEVEL_Q: {0:   496, MODE_NUMERIC:  144, MODE_ALPHANUMERIC:   87, MODE_BYTE:   60, MODE_KANJI:   37},
           ERROR_LEVEL_H: {0:   368, MODE_NUMERIC:  106, MODE_ALPHANUMERIC:   64, MODE_BYTE:   44, MODE_KANJI:   27}},
    6:    {ERROR_LEVEL_L: {0:  1088, MODE_NUMERIC:  322, MODE_ALPHANUMERIC:  195, MODE_BYTE:  134, MODE_KANJI:   82},
           ERROR_LEVEL_M: {0:   864, MODE_NUMERIC:  255, MODE_ALPHANUMERIC:  154, MODE_BYTE:  106, MODE_KANJI:   65},
           ERROR_LEVEL_Q: {0:   608, MODE_NUMERIC:  178, MODE_ALPHANUMERIC:  108, MODE_BYTE:   74, MODE_KANJI:   45},
           ERROR_LEVEL_H: {0:   480, MODE_NUMERIC:  139, MODE_ALPHANUMERIC:   84, MODE_BYTE:   58, MODE_KANJI:   36}},
    7:    {ERROR_LEVEL_L: {0:  1248, MODE_NUMERIC:  370, MODE_ALPHANUMERIC:  224, MODE_BYTE:  154, MODE_KANJI:   95},
           ERROR_LEVEL_M: {0:   992, MODE_NUMERIC:  293, MODE_ALPHANUMERIC:  178, MODE_BYTE:  122, MODE_KANJI:   75},
           ERROR_LEVEL_Q: {0:   704, MODE_NUMERIC:  207, MODE_ALPHANUMERIC:  125, MODE_BYTE:   86, MODE_KANJI:   53},
           ERROR_LEVEL_H: {0:   528, MODE_NUMERIC:  154, MODE_ALPHANUMERIC:   93, MODE_BYTE:   64, MODE_KANJI:   39}},
    8:    {ERROR_LEVEL_L: {0:  1552, MODE_NUMERIC:  461, MODE_ALPHANUMERIC:  279, MODE_BYTE:  192, MODE_KANJI:  118},
           ERROR_LEVEL_M: {0:  1232, MODE_NUMERIC:  365, MODE_ALPHANUMERIC:  221, MODE_BYTE:  152, MODE_KANJI:   93},
           ERROR_LEVEL_Q: {0:   880, MODE_NUMERIC:  259, MODE_ALPHANUMERIC:  157, MODE_BYTE:  108, MODE_KANJI:   66},
           ERROR_LEVEL_H: {0:   688, MODE_NUMERIC:  202, MODE_ALPHANUMERIC:  122, MODE_BYTE:   84, MODE_KANJI:   52}},
    9:    {ERROR_LEVEL_L: {0:  1856, MODE_NUMERIC:  552, MODE_ALPHANUMERIC:  335, MODE_BYTE:  230, MODE_KANJI:  141},
           ERROR_LEVEL_M: {0:  1456, MODE_NUMERIC:  432, MODE_ALPHANUMERIC:  262, MODE_BYTE:  180, MODE_KANJI:  111},
           ERROR_LEVEL_Q: {0:  1056, MODE_NUMERIC:  312, MODE_ALPHANUMERIC:  189, MODE_BYTE:  130, MODE_KANJI:   80},
           ERROR_LEVEL_H: {0:   800, MODE_NUMERIC:  235, MODE_ALPHANUMERIC:  143, MODE_BYTE:   98, MODE_KANJI:   60}},
    10:   {ERROR_LEVEL_L: {0:  2192, MODE_NUMERIC:  652, MODE_ALPHANUMERIC:  395, MODE_BYTE:  271, MODE_KANJI:  167},
           ERROR_LEVEL_M: {0:  1728, MODE_NUMERIC:  513, MODE_ALPHANUMERIC:  311, MODE_BYTE:  213, MODE_KANJI:  131},
           ERROR_LEVEL_Q: {0:  1232, MODE_NUMERIC:  364, MODE_ALPHANUMERIC:  221, MODE_BYTE:  151, MODE_KANJI:   93},
           ERROR_LEVEL_H: {0:   976, MODE_NUMERIC:  288, MODE_ALPHANUMERIC:  174, MODE_BYTE:  119, MODE_KANJI:   74}},
    11:   {ERROR_LEVEL_L: {0:  2592, MODE_NUMERIC:  772, MODE_ALPHANUMERIC:  468, MODE_BYTE:  321, MODE_KANJI:  198},
           ERROR_LEVEL_M: {0:  2032, MODE_NUMERIC:  604, MODE_ALPHANUMERIC:  366, MODE_BYTE:  251, MODE_KANJI:  155},
           ERROR_LEVEL_Q: {0:  1440, MODE_NUMERIC:  427, MODE_ALPHANUMERIC:  259, MODE_BYTE:  177, MODE_KANJI:  109},
           ERROR_LEVEL_H: {0:  1120, MODE_NUMERIC:  331, MODE_ALPHANUMERIC:  200, MODE_BYTE:  137, MODE_KANJI:   85}},
    12:   {ERROR_LEVEL_L: {0:  2960, MODE_NUMERIC:  883, MODE_ALPHANUMERIC:  535, MODE_BYTE:  367, MODE_KANJI:  226},
           ERROR_LEVEL_M: {0:  2320, MODE_NUMERIC:  691, MODE_ALPHANUMERIC:  419, MODE_BYTE:  287, MODE_KANJI:  177},
           ERROR_LEVEL_Q: {0:  1648, MODE_NUMERIC:  489, MODE_ALPHANUMERIC:  296, MODE_BYTE:  203, MODE_KANJI:  125},
           ERROR_LEVEL_H: {0:  1264, MODE_NUMERIC:  374, MODE_ALPHANUMERIC:  227, MODE_BYTE:  155, MODE_KANJI:   96}},
    13:   {ERROR_LEVEL_L: {0:  3424, MODE_NUMERIC: 1022, MODE_ALPHANUMERIC:  619, MODE_BYTE:  425, MODE_KANJI:  262},
           ERROR_LEVEL_M: {0:  2672, MODE_NUMERIC:  796, MODE_ALPHANUMERIC:  483, MODE_BYTE:  331, MODE_KANJI:  204},
           ERROR_LEVEL_Q: {0:  1952, MODE_NUMERIC:  580, MODE_ALPHANUMERIC:  352, MODE_BYTE:  241, MODE_KANJI:  149},
           ERROR_LEVEL_H: {0:  1440, MODE_NUMERIC:  427, MODE_ALPHANUMERIC:  259, MODE_BYTE:  177, MODE_KANJI:  109}},
    14:   {ERROR_LEVEL_L: {0:  3688, MODE_NUMERIC: 1101, MODE_ALPHANUMERIC:  667, MODE_BYTE:  458, MODE_KANJI:  282},
           ERROR_LEVEL_M: {0:  2920, MODE_NUMERIC:  871, MODE_ALPHANUMERIC:  528, MODE_BYTE:  362, MODE_KANJI:  223},
           ERROR_LEVEL_Q: {0:  2088, MODE_NUMERIC:  621, MODE_ALPHANUMERIC:  376, MODE_BYTE:  258, MODE_KANJI:  159},
           ERROR_LEVEL_H: {0:  1576, MODE_NUMERIC:  468, MODE_ALPHANUMERIC:  283, MODE_BYTE:  194, MODE_KANJI:  120}},
    15:   {ERROR_LEVEL_L: {0:  4184, MODE_NUMERIC: 1250, MODE_ALPHANUMERIC:  758, MODE_BYTE:  520, MODE_KANJI:  320},
           ERROR_LEVEL_M: {0:  3320, MODE_NUMERIC:  991, MODE_ALPHANUMERIC:  600, MODE_BYTE:  412, MODE_KANJI:  254},
           ERROR_LEVEL_Q: {0:  2360, MODE_NUMERIC:  703, MODE_ALPHANUMERIC:  426, MODE_BYTE:  292, MODE_KANJI:  180},
           ERROR_LEVEL_H: {0:  1784, MODE_NUMERIC:  530, MODE_ALPHANUMERIC:  321, MODE_BYTE:  220, MODE_KANJI:  136}},
    16:   {ERROR_LEVEL_L: {0:  4712, MODE_NUMERIC: 1408, MODE_ALPHANUMERIC:  854, MODE_BYTE:  586, MODE_KANJI:  361},
           ERROR_LEVEL_M: {0:  3624, MODE_NUMERIC: 1082, MODE_ALPHANUMERIC:  656, MODE_BYTE:  450, MODE_KANJI:  277},
           ERROR_LEVEL_Q: {0:  2600, MODE_NUMERIC:  775, MODE_ALPHANUMERIC:  470, MODE_BYTE:  322, MODE_KANJI:  198},
           ERROR_LEVEL_H: {0:  2024, MODE_NUMERIC:  602, MODE_ALPHANUMERIC:  365, MODE_BYTE:  250, MODE_KANJI:  154}},
    17:   {ERROR_LEVEL_L: {0:  5176, MODE_NUMERIC: 1548, MODE_ALPHANUMERIC:  938, MODE_BYTE:  644, MODE_KANJI:  397},
           ERROR_LEVEL_M: {0:  4056, MODE_NUMERIC: 1212, MODE_ALPHANUMERIC:  734, MODE_BYTE:  504, MODE_KANJI:  310},
           ERROR_LEVEL_Q: {0:  2936, MODE_NUMERIC:  876, MODE_ALPHANUMERIC:  531, MODE_BYTE:  364, MODE_KANJI:  224},
           ERROR_LEVEL_H: {0:  2264, MODE_NUMERIC:  674, MODE_ALPHANUMERIC:  408, MODE_BYTE:  280, MODE_KANJI:  173}},
    18:   {ERROR_LEVEL_L: {0:  5768, MODE_NUMERIC: 1725, MODE_ALPHANUMERIC: 1046, MODE_BYTE:  718, MODE_KANJI:  442},
           ERROR_LEVEL_M: {0:  4504, MODE_NUMERIC: 1346, MODE_ALPHANUMERIC:  816, MODE_BYTE:  560, MODE_KANJI:  345},
           ERROR_LEVEL_Q: {0:  3176, MODE_NUMERIC:  948, MODE_ALPHANUMERIC:  574, MODE_BYTE:  394, MODE_KANJI:  243},
           ERROR_LEVEL_H: {0:  2504, MODE_NUMERIC:  746, MODE_ALPHANUMERIC:  452, MODE_BYTE:  310, MODE_KANJI:  191}},
    19:   {ERROR_LEVEL_L: {0:  6360, MODE_NUMERIC: 1903, MODE_ALPHANUMERIC: 1153, MODE_BYTE:  792, MODE_KANJI:  488},
           ERROR_LEVEL_M: {0:  5016, MODE_NUMERIC: 1500, MODE_ALPHANUMERIC:  909, MODE_BYTE:  624, MODE_KANJI:  384},
           ERROR_LEVEL_Q: {0:  3560, MODE_NUMERIC: 1063, MODE_ALPHANUMERIC:  644, MODE_BYTE:  442, MODE_KANJI:  272},
           ERROR_LEVEL_H: {0:  2728, MODE_NUMERIC:  813, MODE_ALPHANUMERIC:  493, MODE_BYTE:  338, MODE_KANJI:  208}},
    20:   {ERROR_LEVEL_L: {0:  6888, MODE_NUMERIC: 2061, MODE_ALPHANUMERIC: 1249, MODE_BYTE:  858, MODE_KANJI:  528},
           ERROR_LEVEL_M: {0:  5352, MODE_NUMERIC: 1600, MODE_ALPHANUMERIC:  970, MODE_BYTE:  666, MODE_KANJI:  410},
           ERROR_LEVEL_Q: {0:  3880, MODE_NUMERIC: 1159, MODE_ALPHANUMERIC:  702, MODE_BYTE:  482, MODE_KANJI:  297},
           ERROR_LEVEL_H: {0:  3080, MODE_NUMERIC:  919, MODE_ALPHANUMERIC:  557, MODE_BYTE:  382, MODE_KANJI:  235}},
    21:   {ERROR_LEVEL_L: {0:  7456, MODE_NUMERIC: 2232, MODE_ALPHANUMERIC: 1352, MODE_BYTE:  929, MODE_KANJI:  572},
           ERROR_LEVEL_M: {0:  5712, MODE_NUMERIC: 1708, MODE_ALPHANUMERIC: 1035, MODE_BYTE:  711, MODE_KANJI:  438},
           ERROR_LEVEL_Q: {0:  4096, MODE_NUMERIC: 1224, MODE_ALPHANUMERIC:  742, MODE_BYTE:  509, MODE_KANJI:  314},
           ERROR_LEVEL_H: {0:  3248, MODE_NUMERIC:  969, MODE_ALPHANUMERIC:  587, MODE_BYTE:  403, MODE_KANJI:  248}},
    22:   {ERROR_LEVEL_L: {0:  8048, MODE_NUMERIC: 2409, MODE_ALPHANUMERIC: 1460, MODE_BYTE: 1003, MODE_KANJI:  618},
           ERROR_LEVEL_M: {0:  6256, MODE_NUMERIC: 1872, MODE_ALPHANUMERIC: 1134, MODE_BYTE:  779, MODE_KANJI:  480},
           ERROR_LEVEL_Q: {0:  4544, MODE_NUMERIC: 1358, MODE_ALPHANUMERIC:  823, MODE_BYTE:  565, MODE_KANJI:  348},
           ERROR_LEVEL_H: {0:  3536, MODE_NUMERIC: 1056, MODE_ALPHANUMERIC:  640, MODE_BYTE:  439, MODE_KANJI:  270}},
    23:   {ERROR_LEVEL_L: {0:  8752, MODE_NUMERIC: 2620, MODE_ALPHANUMERIC: 1588, MODE_BYTE: 1091, MODE_KANJI:  672},
           ERROR_LEVEL_M: {0:  6880, MODE_NUMERIC: 2059, MODE_ALPHANUMERIC: 1248, MODE_BYTE:  857, MODE_KANJI:  528},
           ERROR_LEVEL_Q: {0:  4912, MODE_NUMERIC: 1468, MODE_ALPHANUMERIC:  890, MODE_BYTE:  611, MODE_KANJI:  376},
           ERROR_LEVEL_H: {0:  3712, MODE_NUMERIC: 1108, MODE_ALPHANUMERIC:  672, MODE_BYTE:  461, MODE_KANJI:  284}},
    24:   {ERROR_LEVEL_L: {0:  9392, MODE_NUMERIC: 2812, MODE_ALPHANUMERIC: 1704, MODE_BYTE: 1171, MODE_KANJI:  721},
           ERROR_LEVEL_M: {0:  7312, MODE_NUMERIC: 2188, MODE_ALPHANUMERIC: 1326, MODE_BYTE:  911, MODE_KANJI:  561},
           ERROR_LEVEL_Q: {0:  5312, MODE_NUMERIC: 1588, MODE_ALPHANUMERIC:  963, MODE_BYTE:  661, MODE_KANJI:  407},
           ERROR_LEVEL_H: {0:  4112, MODE_NUMERIC: 1228, MODE_ALPHANUMERIC:  744, MODE_BYTE:  511, MODE_KANJI:  315}},
    25:   {ERROR_LEVEL_L: {0: 10208, MODE_NUMERIC: 3057, MODE_ALPHANUMERIC: 1853, MODE_BYTE: 1273, MODE_KANJI:  784},
           ERROR_LEVEL_M: {0:  8000, MODE_NUMERIC: 2395, MODE_ALPHANUMERIC: 1451, MODE_BYTE:  997, MODE_KANJI:  614},
           ERROR_LEVEL_Q: {0:  5744, MODE_NUMERIC: 1718, MODE_ALPHANUMERIC: 1041, MODE_BYTE:  715, MODE_KANJI:  440},
           ERROR_LEVEL_H: {0:  4304, MODE_NUMERIC: 1286, MODE_ALPHANUMERIC:  779, MODE_BYTE:  535, MODE_KANJI:  330}},
    26:   {ERROR_LEVEL_L: {0: 10960, MODE_NUMERIC: 3283, MODE_ALPHANUMERIC: 1990, MODE_BYTE: 1367, MODE_KANJI:  842},
           ERROR_LEVEL_M: {0:  8496, MODE_NUMERIC: 2544, MODE_ALPHANUMERIC: 1542, MODE_BYTE: 1059, MODE_KANJI:  652},
           ERROR_LEVEL_Q: {0:  6032, MODE_NUMERIC: 1804, MODE_ALPHANUMERIC: 1094, MODE_BYTE:  751, MODE_KANJI:  462},
           ERROR_LEVEL_H: {0:  4768, MODE_NUMERIC: 1425, MODE_ALPHANUMERIC:  864, MODE_BYTE:  593, MODE_KANJI:  365}},
    27:   {ERROR_LEVEL_L: {0: 11744, MODE_NUMERIC: 3514, MODE_ALPHANUMERIC: 2132, MODE_BYTE: 1465, MODE_KANJI:  902},
           ERROR_LEVEL_M: {0:  9024, MODE_NUMERIC: 2701, MODE_ALPHANUMERIC: 1637, MODE_BYTE: 1125, MODE_KANJI:  692},
           ERROR_LEVEL_Q: {0:  6464, MODE_NUMERIC: 1933, MODE_ALPHANUMERIC: 1172, MODE_BYTE:  805, MODE_KANJI:  496},
           ERROR_LEVEL_H: {0:  5024, MODE_NUMERIC: 1501, MODE_ALPHANUMERIC:  910, MODE_BYTE:  625, MODE_KANJI:  385}},
    28:   {ERROR_LEVEL_L: {0: 12248, MODE_NUMERIC: 3669, MODE_ALPHANUMERIC: 2223, MODE_BYTE: 1528, MODE_KANJI:  940},
           ERROR_LEVEL_M: {0:  9544, MODE_NUMERIC: 2857, MODE_ALPHANUMERIC: 1732, MODE_BYTE: 1190, MODE_KANJI:  732},
           ERROR_LEVEL_Q: {0:  6968, MODE_NUMERIC: 2085, MODE_ALPHANUMERIC: 1263, MODE_BYTE:  868, MODE_KANJI:  534},
           ERROR_LEVEL_H: {0:  5288, MODE_NUMERIC: 1581, MODE_ALPHANUMERIC:  958, MODE_BYTE:  658, MODE_KANJI:  405}},
    29:   {ERROR_LEVEL_L: {0: 13048, MODE_NUMERIC: 3909, MODE_ALPHANUMERIC: 2369, MODE_BYTE: 1628, MODE_KANJI: 1002},
           ERROR_LEVEL_M: {0: 10136, MODE_NUMERIC: 3035, MODE_ALPHANUMERIC: 1839, MODE_BYTE: 1264, MODE_KANJI:  778},
           ERROR_LEVEL_Q: {0:  7288, MODE_NUMERIC: 2181, MODE_ALPHANUMERIC: 1322, MODE_BYTE:  908, MODE_KANJI:  559},
           ERROR_LEVEL_H: {0:  5608, MODE_NUMERIC: 1677, MODE_ALPHANUMERIC: 1016, MODE_BYTE:  698, MODE_KANJI:  430}},
    30:   {ERROR_LEVEL_L: {0: 13880, MODE_NUMERIC: 4158, MODE_ALPHANUMERIC: 2520, MODE_BYTE: 1732, MODE_KANJI: 1066},
           ERROR_LEVEL_M: {0: 10984, MODE_NUMERIC: 3289, MODE_ALPHANUMERIC: 1994, MODE_BYTE: 1370, MODE_KANJI:  843},
           ERROR_LEVEL_Q: {0:  7880, MODE_NUMERIC: 2358, MODE_ALPHANUMERIC: 1429, MODE_BYTE:  982, MODE_KANJI:  604},
           ERROR_LEVEL_H: {0:  5960, MODE_NUMERIC: 1782, MODE_ALPHANUMERIC: 1080, MODE_BYTE:  742, MODE_KANJI:  457}},
    31:   {ERROR_LEVEL_L: {0: 14744, MODE_NUMERIC: 4417, MODE_ALPHANUMERIC: 2677, MODE_BYTE: 1840, MODE_KANJI: 1132},
           ERROR_LEVEL_M: {0: 11640, MODE_NUMERIC: 3486, MODE_ALPHANUMERIC: 2113, MODE_BYTE: 1452, MODE_KANJI:  894},
           ERROR_LEVEL_Q: {0:  8264, MODE_NUMERIC: 2473, MODE_ALPHANUMERIC: 1499, MODE_BYTE: 1030, MODE_KANJI:  634},
           ERROR_LEVEL_H: {0:  6344, MODE_NUMERIC: 1897, MODE_ALPHANUMERIC: 1150, MODE_BYTE:  790, MODE_KANJI:  486}},
    32:   {ERROR_LEVEL_L: {0: 15640, MODE_NUMERIC: 4686, MODE_ALPHANUMERIC: 2840, MODE_BYTE: 1952, MODE_KANJI: 1201},
           ERROR_LEVEL_M: {0: 12328, MODE_NUMERIC: 3693, MODE_ALPHANUMERIC: 2238, MODE_BYTE: 1538, MODE_KANJI:  947},
           ERROR_LEVEL_Q: {0:  8920, MODE_NUMERIC: 2670, MODE_ALPHANUMERIC: 1618, MODE_BYTE: 1112, MODE_KANJI:  684},
           ERROR_LEVEL_H: {0:  6760, MODE_NUMERIC: 2022, MODE_ALPHANUMERIC: 1226, MODE_BYTE:  842, MODE_KANJI:  518}},
    33:   {ERROR_LEVEL_L: {0: 16568, MODE_NUMERIC: 4965, MODE_ALPHANUMERIC: 3009, MODE_BYTE: 2068, MODE_KANJI: 1273},
           ERROR_LEVEL_M: {0: 13048, MODE_NUMERIC: 3909, MODE_ALPHANUMERIC: 2369, MODE_BYTE: 1628, MODE_KANJI: 1002},
           ERROR_LEVEL_Q: {0:  9368, MODE_NUMERIC: 2805, MODE_ALPHANUMERIC: 1700, MODE_BYTE: 1168, MODE_KANJI:  719},
           ERROR_LEVEL_H: {0:  7208, MODE_NUMERIC: 2157, MODE_ALPHANUMERIC: 1307, MODE_BYTE:  898, MODE_KANJI:  553}},
    34:   {ERROR_LEVEL_L: {0: 17528, MODE_NUMERIC: 5253, MODE_ALPHANUMERIC: 3183, MODE_BYTE: 2188, MODE_KANJI: 1347},
           ERROR_LEVEL_M: {0: 13800, MODE_NUMERIC: 4134, MODE_ALPHANUMERIC: 2506, MODE_BYTE: 1722, MODE_KANJI: 1060},
           ERROR_LEVEL_Q: {0:  9848, MODE_NUMERIC: 2949, MODE_ALPHANUMERIC: 1787, MODE_BYTE: 1228, MODE_KANJI:  756},
           ERROR_LEVEL_H: {0:  7688, MODE_NUMERIC: 2301, MODE_ALPHANUMERIC: 1394, MODE_BYTE:  958, MODE_KANJI:  590}},
    35:   {ERROR_LEVEL_L: {0: 18448, MODE_NUMERIC: 5529, MODE_ALPHANUMERIC: 3351, MODE_BYTE: 2303, MODE_KANJI: 1417},
           ERROR_LEVEL_M: {0: 14496, MODE_NUMERIC: 4343, MODE_ALPHANUMERIC: 2632, MODE_BYTE: 1809, MODE_KANJI: 1113},
           ERROR_LEVEL_Q: {0: 10288, MODE_NUMERIC: 3081, MODE_ALPHANUMERIC: 1867, MODE_BYTE: 1283, MODE_KANJI:  790},
           ERROR_LEVEL_H: {0:  7888, MODE_NUMERIC: 2361, MODE_ALPHANUMERIC: 1431, MODE_BYTE:  983, MODE_KANJI:  605}},
    36:   {ERROR_LEVEL_L: {0: 19472, MODE_NUMERIC: 5836, MODE_ALPHANUMERIC: 3537, MODE_BYTE: 2431, MODE_KANJI: 1496},
           ERROR_LEVEL_M: {0: 15312, MODE_NUMERIC: 4588, MODE_ALPHANUMERIC: 2780, MODE_BYTE: 1911, MODE_KANJI: 1176},
           ERROR_LEVEL_Q: {0: 10832, MODE_NUMERIC: 3244, MODE_ALPHANUMERIC: 1966, MODE_BYTE: 1351, MODE_KANJI:  832},
           ERROR_LEVEL_H: {0:  8432, MODE_NUMERIC: 2524, MODE_ALPHANUMERIC: 1530, MODE_BYTE: 1051, MODE_KANJI:  647}},
    37:   {ERROR_LEVEL_L: {0: 20528, MODE_NUMERIC: 6153, MODE_ALPHANUMERIC: 3729, MODE_BYTE: 2563, MODE_KANJI: 1577},
           ERROR_LEVEL_M: {0: 15936, MODE_NUMERIC: 4775, MODE_ALPHANUMERIC: 2894, MODE_BYTE: 1989, MODE_KANJI: 1224},
           ERROR_LEVEL_Q: {0: 11408, MODE_NUMERIC: 3417, MODE_ALPHANUMERIC: 2071, MODE_BYTE: 1423, MODE_KANJI:  876},
           ERROR_LEVEL_H: {0:  8768, MODE_NUMERIC: 2625, MODE_ALPHANUMERIC: 1591, MODE_BYTE: 1093, MODE_KANJI:  673}},
    38:   {ERROR_LEVEL_L: {0: 21616, MODE_NUMERIC: 6479, MODE_ALPHANUMERIC: 3927, MODE_BYTE: 2699, MODE_KANJI: 1661},
           ERROR_LEVEL_M: {0: 16816, MODE_NUMERIC: 5039, MODE_ALPHANUMERIC: 3054, MODE_BYTE: 2099, MODE_KANJI: 1292},
           ERROR_LEVEL_Q: {0: 12016, MODE_NUMERIC: 3599, MODE_ALPHANUMERIC: 2181, MODE_BYTE: 1499, MODE_KANJI:  923},
           ERROR_LEVEL_H: {0:  9136, MODE_NUMERIC: 2735, MODE_ALPHANUMERIC: 1658, MODE_BYTE: 1139, MODE_KANJI:  701}},
    39:   {ERROR_LEVEL_L: {0: 22496, MODE_NUMERIC: 6743, MODE_ALPHANUMERIC: 4087, MODE_BYTE: 2809, MODE_KANJI: 1729},
           ERROR_LEVEL_M: {0: 17728, MODE_NUMERIC: 5313, MODE_ALPHANUMERIC: 3220, MODE_BYTE: 2213, MODE_KANJI: 1362},
           ERROR_LEVEL_Q: {0: 12656, MODE_NUMERIC: 3791, MODE_ALPHANUMERIC: 2298, MODE_BYTE: 1579, MODE_KANJI:  972},
           ERROR_LEVEL_H: {0:  9776, MODE_NUMERIC: 2927, MODE_ALPHANUMERIC: 1774, MODE_BYTE: 1219, MODE_KANJI:  750}},
    40:   {ERROR_LEVEL_L: {0: 23648, MODE_NUMERIC: 7089, MODE_ALPHANUMERIC: 4296, MODE_BYTE: 2953, MODE_KANJI: 1817},
           ERROR_LEVEL_M: {0: 18672, MODE_NUMERIC: 5596, MODE_ALPHANUMERIC: 3391, MODE_BYTE: 2331, MODE_KANJI: 1435},
           ERROR_LEVEL_Q: {0: 13328, MODE_NUMERIC: 3993, MODE_ALPHANUMERIC: 2420, MODE_BYTE: 1663, MODE_KANJI: 1024},
           ERROR_LEVEL_H: {0: 10208, MODE_NUMERIC: 3057, MODE_ALPHANUMERIC: 1852, MODE_BYTE: 1273, MODE_KANJI:  784}}
}


# ISO/IEC 18004:2015(E) -- Table 9 — Error correction characteristics for QR Code (page 38)
EC = namedtuple('EC', 'num_blocks num_total num_data')

ECC = {
    VERSION_M1: {None: (EC(1, 5, 3),)},
    VERSION_M2: {ERROR_LEVEL_L: (EC(1, 10, 5),), ERROR_LEVEL_M: (EC(1, 10, 4),)},
    VERSION_M3: {ERROR_LEVEL_L: (EC(1, 17, 11),), ERROR_LEVEL_M: (EC(1, 17, 9),)},
    VERSION_M4: {ERROR_LEVEL_L: (EC(1, 24, 16),), ERROR_LEVEL_M: (EC(1, 24, 14),),
                 ERROR_LEVEL_Q: (EC(1, 24, 10),)},
     1: {
        ERROR_LEVEL_L: (EC(1, 26, 19),), ERROR_LEVEL_M: (EC(1, 26, 16),),
        ERROR_LEVEL_Q: (EC(1, 26, 13),), ERROR_LEVEL_H: (EC(1, 26, 9),), },
     2: {
        ERROR_LEVEL_L: (EC(1, 44, 34),), ERROR_LEVEL_M: (EC(1, 44, 28),),
        ERROR_LEVEL_Q: (EC(1, 44, 22),), ERROR_LEVEL_H: (EC(1, 44, 16),), },
     3: {
        ERROR_LEVEL_L: (EC(1, 70, 55),), ERROR_LEVEL_M: (EC(1, 70, 44),),
        ERROR_LEVEL_Q: (EC(2, 35, 17),), ERROR_LEVEL_H: (EC(2, 35, 13),), },
     4: {
        ERROR_LEVEL_L: (EC(1, 100, 80),), ERROR_LEVEL_M: (EC(2, 50, 32),),
        ERROR_LEVEL_Q: (EC(2, 50, 24),),  ERROR_LEVEL_H: (EC(4, 25, 9),), },
     5: {
        ERROR_LEVEL_L: (EC(1, 134, 108),), ERROR_LEVEL_M: (EC(2, 67, 43),),
        ERROR_LEVEL_Q: (EC(2, 33, 15),  EC(2, 34, 16)),
        ERROR_LEVEL_H: (EC(2, 33, 11),  EC(2, 34, 12)), },
     6: {
        ERROR_LEVEL_L: (EC(2, 86, 68),), ERROR_LEVEL_M: (EC(4, 43, 27),),
        ERROR_LEVEL_Q: (EC(4, 43, 19),), ERROR_LEVEL_H: (EC(4, 43, 15),), },
     7: {
        ERROR_LEVEL_L: (EC(2, 98, 78),), ERROR_LEVEL_M: (EC(4, 49, 31),),
        ERROR_LEVEL_Q: (EC(2, 32, 14),  EC(4, 33, 15)),
        ERROR_LEVEL_H: (EC(4, 39, 13),  EC(1, 40, 14)), },
     8: {
        ERROR_LEVEL_L: (EC(2, 121, 97),),
        ERROR_LEVEL_M: (EC(2, 60, 38),  EC(2, 61, 39)),
        ERROR_LEVEL_Q: (EC(4, 40, 18),  EC(2, 41, 19)),
        ERROR_LEVEL_H: (EC(4, 40, 14),  EC(2, 41, 15)), },
     9: {
        ERROR_LEVEL_L: (EC(2, 146, 116),),
        ERROR_LEVEL_M: (EC(3, 58, 36),  EC(2, 59, 37)),
        ERROR_LEVEL_Q: (EC(4, 36, 16),  EC(4, 37, 17)),
        ERROR_LEVEL_H: (EC(4, 36, 12),  EC(4, 37, 13)), },
    10: {
        ERROR_LEVEL_L: (EC(2, 86, 68),  EC(2, 87, 69)),
        ERROR_LEVEL_M: (EC(4, 69, 43),  EC(1, 70, 44)),
        ERROR_LEVEL_Q: (EC(6, 43, 19),  EC(2, 44, 20)),
        ERROR_LEVEL_H: (EC(6, 43, 15),  EC(2, 44, 16)), },
    11 : {
        ERROR_LEVEL_L: (EC(4, 101, 81),),
        ERROR_LEVEL_M: (EC(1, 80, 50),  EC(4, 81, 51)),
        ERROR_LEVEL_Q: (EC(4, 50, 22),  EC(4, 51, 23)),
        ERROR_LEVEL_H: (EC(3, 36, 12),  EC(8, 37, 13)), },
    12 : {
        ERROR_LEVEL_L: (EC(2, 116, 92),  EC(2, 117, 93)),
        ERROR_LEVEL_M: (EC(6, 58, 36),  EC(2, 59, 37)),
        ERROR_LEVEL_Q: (EC(4, 46, 20),  EC(6, 47, 21)),
        ERROR_LEVEL_H: (EC(7, 42, 14),  EC(4, 43, 15)), },
    13 : {
        ERROR_LEVEL_L: (EC(4, 133, 107),),
        ERROR_LEVEL_M: (EC(8, 59, 37),  EC(1, 60, 38)),
        ERROR_LEVEL_Q: (EC(8, 44, 20),  EC(4, 45, 21)),
        ERROR_LEVEL_H: (EC(12, 33, 11),  EC(4, 34, 12)), },
    14 : {
        ERROR_LEVEL_L: (EC(3, 145, 115),  EC(1, 146, 116)),
        ERROR_LEVEL_M: (EC(4, 64, 40),  EC(5, 65, 41)),
        ERROR_LEVEL_Q: (EC(11, 36, 16),  EC(5, 37, 17)),
        ERROR_LEVEL_H: (EC(11, 36, 12),  EC(5, 37, 13)), },
    15 : {
        ERROR_LEVEL_L: (EC(5, 109, 87),  EC(1, 110, 88)),
        ERROR_LEVEL_M: (EC(5, 65, 41),  EC(5, 66, 42)),
        ERROR_LEVEL_Q: (EC(5, 54, 24),  EC(7, 55, 25)),
        ERROR_LEVEL_H: (EC(11, 36, 12),  EC(7, 37, 13)), },
    16 : {
        ERROR_LEVEL_L: (EC(5, 122, 98),  EC(1, 123, 99)),
        ERROR_LEVEL_M: (EC(7, 73, 45),  EC(3, 74, 46)),
        ERROR_LEVEL_Q: (EC(15, 43, 19),  EC(2, 44, 20)),
        ERROR_LEVEL_H: (EC(3, 45, 15),  EC(13, 46, 16)), },
    17 : {
        ERROR_LEVEL_L: (EC(1, 135, 107),  EC(5, 136, 108)),
        ERROR_LEVEL_M: (EC(10, 74, 46),  EC(1, 75, 47)),
        ERROR_LEVEL_Q: (EC(1, 50, 22),  EC(15, 51, 23)),
        ERROR_LEVEL_H: (EC(2, 42, 14),  EC(17, 43, 15)), },
    18 : {
        ERROR_LEVEL_L: (EC(5, 150, 120),  EC(1, 151, 121)),
        ERROR_LEVEL_M: (EC(9, 69, 43),  EC(4, 70, 44)),
        ERROR_LEVEL_Q: (EC(17, 50, 22),  EC(1, 51, 23)),
        ERROR_LEVEL_H: (EC(2, 42, 14),  EC(19, 43, 15)), },
    19 : {
        ERROR_LEVEL_L: (EC(3, 141, 113),  EC(4, 142, 114)),
        ERROR_LEVEL_M: (EC(3, 70, 44),  EC(11, 71, 45)),
        ERROR_LEVEL_Q: (EC(17, 47, 21),  EC(4, 48, 22)),
        ERROR_LEVEL_H: (EC(9, 39, 13),  EC(16, 40, 14)), },
    20 : {
        ERROR_LEVEL_L: (EC(3, 135, 107),  EC(5, 136, 108)),
        ERROR_LEVEL_M: (EC(3, 67, 41),  EC(13, 68, 42)),
        ERROR_LEVEL_Q: (EC(15, 54, 24),  EC(5, 55, 25)),
        ERROR_LEVEL_H: (EC(15, 43, 15),  EC(10, 44, 16)), },
    21 : {
        ERROR_LEVEL_L: (EC(4, 144, 116),  EC(4, 145, 117)),
        ERROR_LEVEL_M: (EC(17, 68, 42),),
        ERROR_LEVEL_Q: (EC(17, 50, 22),  EC(6, 51, 23)),
        ERROR_LEVEL_H: (EC(19, 46, 16),  EC(6, 47, 17)), },
    22 : {
        ERROR_LEVEL_L: (EC(2, 139, 111),  EC(7, 140, 112)),
        ERROR_LEVEL_M: (EC(17, 74, 46),),
        ERROR_LEVEL_Q: (EC(7, 54, 24),  EC(16, 55, 25)),
        ERROR_LEVEL_H: (EC(34, 37, 13),), },
    23 : {
        ERROR_LEVEL_L: (EC(4, 151, 121),  EC(5, 152, 122)),
        ERROR_LEVEL_M: (EC(4, 75, 47),  EC(14, 76, 48)),
        ERROR_LEVEL_Q: (EC(11, 54, 24),  EC(14, 55, 25)),
        ERROR_LEVEL_H: (EC(16, 45, 15),  EC(14, 46, 16)), },
    24 : {
        ERROR_LEVEL_L: (EC(6, 147, 117),  EC(4, 148, 118)),
        ERROR_LEVEL_M: (EC(6, 73, 45),  EC(14, 74, 46)),
        ERROR_LEVEL_Q: (EC(11, 54, 24),  EC(16, 55, 25)),
        ERROR_LEVEL_H: (EC(30, 46, 16),  EC(2, 47, 17)), },
    25 : {
        ERROR_LEVEL_L: (EC(8, 132, 106),  EC(4, 133, 107)),
        ERROR_LEVEL_M: (EC(8, 75, 47),  EC(13, 76, 48)),
        ERROR_LEVEL_Q: (EC(7, 54, 24),  EC(22, 55, 25)),
        ERROR_LEVEL_H: (EC(22, 45, 15),  EC(13, 46, 16)), },
    26 : {
        ERROR_LEVEL_L: (EC(10, 142, 114),  EC(2, 143, 115)),
        ERROR_LEVEL_M: (EC(19, 74, 46),  EC(4, 75, 47)),
        ERROR_LEVEL_Q: (EC(28, 50, 22),  EC(6, 51, 23)),
        ERROR_LEVEL_H: (EC(33, 46, 16),  EC(4, 47, 17)), },
    27 : {
        ERROR_LEVEL_L: (EC(8, 152, 122),  EC(4, 153, 123)),
        ERROR_LEVEL_M: (EC(22, 73, 45),  EC(3, 74, 46)),
        ERROR_LEVEL_Q: (EC(8, 53, 23),  EC(26, 54, 24)),
        ERROR_LEVEL_H: (EC(12, 45, 15),  EC(28, 46, 16)), },
    28 : {
        ERROR_LEVEL_L: (EC(3, 147, 117),  EC(10, 148, 118)),
        ERROR_LEVEL_M: (EC(3, 73, 45),  EC(23, 74, 46)),
        ERROR_LEVEL_Q: (EC(4, 54, 24),  EC(31, 55, 25)),
        ERROR_LEVEL_H: (EC(11, 45, 15),  EC(31, 46, 16)), },
    29 : {
        ERROR_LEVEL_L: (EC(7, 146, 116),  EC(7, 147, 117)),
        ERROR_LEVEL_M: (EC(21, 73, 45),  EC(7, 74, 46)),
        ERROR_LEVEL_Q: (EC(1, 53, 23),  EC(37, 54, 24)),
        ERROR_LEVEL_H: (EC(19, 45, 15),  EC(26, 46, 16)), },
    30 : {
        ERROR_LEVEL_L: (EC(5, 145, 115),  EC(10, 146, 116)),
        ERROR_LEVEL_M: (EC(19, 75, 47),  EC(10, 76, 48)),
        ERROR_LEVEL_Q: (EC(15, 54, 24),  EC(25, 55, 25)),
        ERROR_LEVEL_H: (EC(23, 45, 15),  EC(25, 46, 16)), },
    31 : {
        ERROR_LEVEL_L: (EC(13, 145, 115),  EC(3, 146, 116)),
        ERROR_LEVEL_M: (EC(2, 74, 46),  EC(29, 75, 47)),
        ERROR_LEVEL_Q: (EC(42, 54, 24),  EC(1, 55, 25)),
        ERROR_LEVEL_H: (EC(23, 45, 15),  EC(28, 46, 16)), },
    32 : {
        ERROR_LEVEL_L: (EC(17, 145, 115),),
        ERROR_LEVEL_M: (EC(10, 74, 46),  EC(23, 75, 47)),
        ERROR_LEVEL_Q: (EC(10, 54, 24),  EC(35, 55, 25)),
        ERROR_LEVEL_H: (EC(19, 45, 15),  EC(35, 46, 16)), },
    33 : {
        ERROR_LEVEL_L: (EC(17, 145, 115),  EC(1, 146, 116)),
        ERROR_LEVEL_M: (EC(14, 74, 46),  EC(21, 75, 47)),
        ERROR_LEVEL_Q: (EC(29, 54, 24),  EC(19, 55, 25)),
        ERROR_LEVEL_H: (EC(11, 45, 15),  EC(46, 46, 16)), },
    34 : {
        ERROR_LEVEL_L: (EC(13, 145, 115),  EC(6, 146, 116)),
        ERROR_LEVEL_M: (EC(14, 74, 46),  EC(23, 75, 47)),
        ERROR_LEVEL_Q: (EC(44, 54, 24),  EC(7, 55, 25)),
        ERROR_LEVEL_H: (EC(59, 46, 16),  EC(1, 47, 17)), },
    35 : {
        ERROR_LEVEL_L: (EC(12, 151, 121),  EC(7, 152, 122)),
        ERROR_LEVEL_M: (EC(12, 75, 47),  EC(26, 76, 48)),
        ERROR_LEVEL_Q: (EC(39, 54, 24),  EC(14, 55, 25)),
        ERROR_LEVEL_H: (EC(22, 45, 15),  EC(41, 46, 16)), },
    36 : {
        ERROR_LEVEL_L: (EC(6, 151, 121),  EC(14, 152, 122)),
        ERROR_LEVEL_M: (EC(6, 75, 47),  EC(34, 76, 48)),
        ERROR_LEVEL_Q: (EC(46, 54, 24),  EC(10, 55, 25)),
        ERROR_LEVEL_H: (EC(2, 45, 15),  EC(64, 46, 16)), },
    37 : {
        ERROR_LEVEL_L: (EC(17, 152, 122),  EC(4, 153, 123)),
        ERROR_LEVEL_M: (EC(29, 74, 46),  EC(14, 75, 47)),
        ERROR_LEVEL_Q: (EC(49, 54, 24),  EC(10, 55, 25)),
        ERROR_LEVEL_H: (EC(24, 45, 15),  EC(46, 46, 16)), },
    38 : {
        ERROR_LEVEL_L: (EC(4, 152, 122),  EC(18, 153, 123)),
        ERROR_LEVEL_M: (EC(13, 74, 46),  EC(32, 75, 47)),
        ERROR_LEVEL_Q: (EC(48, 54, 24),  EC(14, 55, 25)),
        ERROR_LEVEL_H: (EC(42, 45, 15),  EC(32, 46, 16)), },
    39 : {
        ERROR_LEVEL_L: (EC(20, 147, 117),  EC(4, 148, 118)),
        ERROR_LEVEL_M: (EC(40, 75, 47),  EC(7, 76, 48)),
        ERROR_LEVEL_Q: (EC(43, 54, 24),  EC(22, 55, 25)),
        ERROR_LEVEL_H: (EC(10, 45, 15),  EC(67, 46, 16)), },
    40 : {
        ERROR_LEVEL_L: (EC(19, 148, 118),  EC(6, 149, 119)),
        ERROR_LEVEL_M: (EC(18, 75, 47),  EC(31, 76, 48)),
        ERROR_LEVEL_Q: (EC(34, 54, 24),  EC(34, 55, 25)),
        ERROR_LEVEL_H: (EC(20, 45, 15),  EC(61, 46, 16)), },
}


# ISO/IEC 18004:2015 -- Annex C - D.1 Error correction bit calculation
# Table C.1 — Valid format information bit sequences (page 80)
FORMAT_INFO = (
    # M: mask 0, mask 1 .. 7
    0x5412, 0x5125, 0x5e7c, 0x5b4b, 0x45f9, 0x40ce, 0x4f97, 0x4aa0,
    # L
    0x77c4, 0x72f3, 0x7daa, 0x789d, 0x662f, 0x6318, 0x6c41, 0x6976,
    # H
    0x1689, 0x13be, 0x1ce7, 0x19d0, 0x0762, 0x0255, 0x0d0c, 0x083b,
    # Q
    0x355f, 0x3068, 0x3f31, 0x3a06, 0x24b4, 0x2183, 0x2eda, 0x2bed,
)

FORMAT_INFO_MICRO = (
    0x4445, 0x4172, 0x4e2b, 0x4b1c, 0x55ae, 0x5099, 0x5fc0, 0x5af7,
    0x6793, 0x62a4, 0x6dfd, 0x68ca, 0x7678, 0x734f, 0x7c16, 0x7921,
    0x06de, 0x03e9, 0x0cb0, 0x0987, 0x1735, 0x1202, 0x1d5b, 0x186c,
    0x2508, 0x203f, 0x2f66, 0x2a51, 0x34e3, 0x31d4, 0x3e8d, 0x3bba,
)


# ISO/IEC 18004:2015 -- Annex D - D.1 Error correction bit calculation
# Table D.1 — Version information bit stream for each version (page 82)
VERSION_INFO = (
    # Version 7, 8, 9 .. 40
    0x07c94, 0x085bc, 0x09a99, 0x0a4d3, 0x0bbf6, 0x0c762, 0x0d847, 0x0e60d,
    0x0f928, 0x10b78, 0x1145d, 0x12a17, 0x13532, 0x149a6, 0x15683, 0x168c9,
    0x177ec, 0x18ec4, 0x191e1, 0x1afab, 0x1b08e, 0x1cc1a, 0x1d33f, 0x1ed75,
    0x1f250, 0x209d5, 0x216f0, 0x228ba, 0x2379f, 0x24b0b, 0x2542e, 0x26a64,
    0x27541, 0x28c69,
)


# ISO/IEC 18004:2015 -- Annex E - Position of alignment patterns
# Table E.1 — Row/column coordinates of center module of alignment patterns (page 83)
ALIGNMENT_POS = (
    (6, 18),  # Version 2 (version 1 has no additional alignment patterns)
    (6, 22),  # Version 3
    (6, 26),  # ..
    (6, 30),
    (6, 34),
    (6, 22, 38),  # Version 7
    (6, 24, 42),
    (6, 26, 46),
    (6, 28, 50),
    (6, 30, 54),
    (6, 32, 58),
    (6, 34, 62),
    (6, 26, 46, 66),  # Version 14
    (6, 26, 48, 70),
    (6, 26, 50, 74),
    (6, 30, 54, 78),
    (6, 30, 56, 82),
    (6, 30, 58, 86),
    (6, 34, 62, 90),
    (6, 28, 50, 72, 94),  # Version 21
    (6, 26, 50, 74, 98),
    (6, 30, 54, 78, 102),
    (6, 28, 54, 80, 106),
    (6, 32, 58, 84, 110),
    (6, 30, 58, 86, 114),
    (6, 34, 62, 90, 118),
    (6, 26, 50, 74, 98, 122),  # Version 28
    (6, 30, 54, 78, 102, 126),
    (6, 26, 52, 78, 104, 130),
    (6, 30, 56, 82, 108, 134),
    (6, 34, 60, 86, 112, 138),
    (6, 30, 58, 86, 114, 142),
    (6, 34, 62, 90, 118, 146),
    (6, 30, 54, 78, 102, 126, 150),  # Version 35
    (6, 24, 50, 76, 102, 128, 154),
    (6, 28, 54, 80, 106, 132, 158),
    (6, 32, 58, 84, 110, 136, 162),
    (6, 26, 54, 82, 110, 138, 166),
    (6, 30, 58, 86, 114, 142, 170),  # Version 40
)


# ISO/IEC 18004:2015 -- Annex A - Error detection and correction generator polynomials
# Table A.1 — Generator polynomials for Reed-Solomon error correction codewords (page 73)
GEN_POLY = {
    2: (25, 2),
    5: (113, 164, 166, 119, 10),
    6: (166, 0, 134, 5, 176, 15),
    7: (87, 229, 146, 149, 238, 102, 21),
    8: (175, 238, 208, 249, 215, 252, 196, 28),
   10: (251, 67, 46, 61, 118, 70, 64, 94, 32, 45),
   13: (74, 152, 176, 100, 86, 100, 106, 104, 130, 218, 206, 140, 78),
   14: (199, 249, 155, 48, 190, 124, 218, 137, 216, 87, 207, 59, 22, 91),
   15: (8, 183, 61, 91, 202, 37, 51, 58, 58, 237, 140, 124, 5, 99, 105),
   16: (120, 104, 107, 109, 102, 161, 76, 3, 91, 191, 147, 169, 182, 194, 225, 120),
   17: (43, 139, 206, 78, 43, 239, 123, 206, 214, 147, 24, 99, 150, 39, 243, 163, 136),
   18: (215, 234, 158, 94, 184, 97, 118, 170, 79, 187, 152, 148, 252, 179, 5, 98, 96, 153),
   20: (17, 60, 79, 50, 61, 163, 26, 187, 202, 180, 221, 225, 83, 239, 156, 164, 212, 212, 188, 190),
   22: (210, 171, 247, 242, 93, 230, 14, 109, 221, 53, 200, 74, 8, 172, 98, 80, 219, 134, 160, 105, 165, 231),
   24: (229, 121, 135, 48, 211, 117, 251, 126, 159, 180, 169, 152, 192, 226, 228, 218, 111, 0, 117, 232, 87, 96, 227, 21),
   26: (173, 125, 158, 2, 103, 182, 118, 17, 145, 201, 111, 28, 165, 53, 161, 21, 245, 142, 13, 102, 48, 227, 153, 145, 218, 70),
   28: (168, 223, 200, 104, 224, 234, 108, 180, 110, 190, 195, 147, 205, 27, 232, 201, 21, 43, 245, 87, 42, 195, 212, 119, 242, 37, 9, 123),
   30: (41, 173, 145, 152, 216, 31, 179, 182, 50, 48, 110, 86, 239, 96, 222, 125, 42, 173, 226, 193, 224, 130, 156, 37, 251, 216, 238, 40, 192, 180)
}

# Precomputed Galios Log tables
#
# prime polynomial: 0x11d (285) / generator: 2

# GF(256) log
GALIOS_LOG = (
    0, 0, 1, 25, 2, 50, 26, 198, 3, 223, 51, 238, 27, 104, 199, 75, 4, 100,
    224, 14, 52, 141, 239, 129, 28, 193, 105, 248, 200, 8, 76, 113, 5, 138,
    101, 47, 225, 36, 15, 33, 53, 147, 142, 218, 240, 18, 130, 69, 29, 181,
    194, 125, 106, 39, 249, 185, 201, 154, 9, 120, 77, 228, 114, 166, 6, 191,
    139, 98, 102, 221, 48, 253, 226, 152, 37, 179, 16, 145, 34, 136, 54, 208,
    148, 206, 143, 150, 219, 189, 241, 210, 19, 92, 131, 56, 70, 64, 30, 66,
    182, 163, 195, 72, 126, 110, 107, 58, 40, 84, 250, 133, 186, 61, 202, 94,
    155, 159, 10, 21, 121, 43, 78, 212, 229, 172, 115, 243, 167, 87, 7, 112,
    192, 247, 140, 128, 99, 13, 103, 74, 222, 237, 49, 197, 254, 24, 227, 165,
    153, 119, 38, 184, 180, 124, 17, 68, 146, 217, 35, 32, 137, 46, 55, 63,
    209, 91, 149, 188, 207, 205, 144, 135, 151, 178, 220, 252, 190, 97, 242,
    86, 211, 171, 20, 42, 93, 158, 132, 60, 57, 83, 71, 109, 65, 162, 31, 45,
    67, 216, 183, 123, 164, 118, 196, 23, 73, 236, 127, 12, 111, 246, 108,
    161, 59, 82, 41, 157, 85, 170, 251, 96, 134, 177, 187, 204, 62, 90, 203,
    89, 95, 176, 156, 169, 160, 81, 11, 245, 22, 235, 122, 117, 44, 215, 79,
    174, 213, 233, 230, 231, 173, 232, 116, 214, 244, 234, 168, 80, 88, 175
)

# GF(256) antilog
# Inverse of the logarithm table.  Maps integer logarithms to members
# of the field.
GALIOS_EXP = ([
    1, 2, 4, 8, 16, 32, 64, 128, 29, 58, 116, 232, 205, 135, 19, 38, 76, 152,
    45, 90, 180, 117, 234, 201, 143, 3, 6, 12, 24, 48, 96, 192, 157, 39, 78,
    156, 37, 74, 148, 53, 106, 212, 181, 119, 238, 193, 159, 35, 70, 140, 5,
    10, 20, 40, 80, 160, 93, 186, 105, 210, 185, 111, 222, 161, 95, 190, 97,
    194, 153, 47, 94, 188, 101, 202, 137, 15, 30, 60, 120, 240, 253, 231, 211,
    187, 107, 214, 177, 127, 254, 225, 223, 163, 91, 182, 113, 226, 217, 175,
    67, 134, 17, 34, 68, 136, 13, 26, 52, 104, 208, 189, 103, 206, 129, 31,
    62, 124, 248, 237, 199, 147, 59, 118, 236, 197, 151, 51, 102, 204, 133, 23,
    46, 92, 184, 109, 218, 169, 79, 158, 33, 66, 132, 21, 42, 84, 168, 77, 154,
    41, 82, 164, 85, 170, 73, 146, 57, 114, 228, 213, 183, 115, 230, 209, 191,
    99, 198, 145, 63, 126, 252, 229, 215, 179, 123, 246, 241, 255, 227, 219,
    171, 75, 150, 49, 98, 196, 149, 55, 110, 220, 165, 87, 174, 65, 130, 25,
    50, 100, 200, 141, 7, 14, 28, 56, 112, 224, 221, 167, 83, 166, 81, 162, 89,
    178, 121, 242, 249, 239, 195, 155, 43, 86, 172, 69, 138, 9, 18, 36, 72,
    144, 61, 122, 244, 245, 247, 243, 251, 235, 203, 139, 11, 22, 44, 88, 176,
    125, 250, 233, 207, 131, 27, 54, 108, 216, 173, 71, 142] * 2
)

