# ruff: noqa
"""Level of theory

Contains classes for identifying and representing the level of theory (LOT) used
to calculate a given property.
"""


import dataclasses
import hashlib
import json
import os
from dataclasses import dataclass, field
from string import Template
from typing import ClassVar, Union

from chemtrayzer.core._serialization import IJsonConvertible

# this can be used as field metadata, to ensure that the string passed is
# converted to all lower case
_LOWER_CASE = {"__to_lower__": True}

# the following two functions are used for hasing the level of theory objects
def _dict2dataclass(
    dict_obj: dict, cls: type = None, cls_table: dict[str, type] = None
):
    """Converts a dictionary to a dataclass instance. This is supposed to do the
    opposite of dataclass.asdict().
    If cls_table is supplied, the function tries to look up the class name expected at dict_obj['__cls__'] and uses that instead of cls. Then, cls can be None.

    :param cls: dataclass, an instance of which is to be created
    :param dict_obj: dictionary generated by dataclasses.asdict()
    :return: instance of cls with data from dict_obj"""

    if dict_obj is None:
        return None

    # always remove __cls__ even if cls is not None
    cls_str = dict_obj.pop("__cls__") if "__cls__" in dict_obj else None

    if cls_table is not None:
        try:
            cls = cls_table[cls_str]
        except KeyError:
            if cls is None:
                raise KeyError(
                    Template(
                        "cls_table does not contain {cls} but " "cls is None"
                    ).substitute(cls=cls_str)
                )
    elif cls is None:
        raise ValueError("Must supply cls or cls_table.")

    if not dataclasses.is_dataclass(cls):
        raise TypeError("cls/cls_table[__cls__] must be a dataclass")

    values = {}
    non_init_values = {}

    for field in dataclasses.fields(cls):
        field: dataclasses.Field

        if field.name not in dict_obj:
            if field.default != dataclasses.MISSING:
                # field has a default value
                continue
            else:
                raise KeyError(f"expected {field.name} for {cls}, but it does "
                               "not appear in the provided dictionary.")

        # check if this field is a list, tuple, etc. of dataclasses
        if hasattr(field.type, "__args__"):
            for arg in field.type.__args__:
                if dataclasses.is_dataclass(arg):
                    raise NotImplementedError(
                        f"Recursion for {str(field.type)}" " is not supported"
                    )

        # recursion
        if dataclasses.is_dataclass(field.type):
            value = _dict2dataclass(
                cls=field.type, dict_obj=dict_obj[field.name], cls_table=cls_table
            )
        else:
            value = dict_obj[field.name]

        if field.init:
            values[field.name] = value
        else:
            non_init_values[field.name] = value

    # create dataclass object
    obj = cls(**values)

    # fields where init=False need to be set after initialization
    for name, value in non_init_values.items():
        setattr(obj, name, value)

    return obj

def _dataclass2dict(
    datacls_obj,
    exclude_non_hash_fields: bool = False,
    exclude_default_values: bool = False,
):
    """dataclass.asdict() is too unflexible, b/c we cannot skip elements. This
    function only does a shallow conversion but it excludes fields where
    hash=False or field that store default values, if requested."""
    fields = dataclasses.fields(datacls_obj)

    if datacls_obj is None:
        return None

    dict_obj = {"__cls__": type(datacls_obj).__name__}

    for field in fields:
        value = getattr(datacls_obj, field.name)

        if field.default != dataclasses.MISSING:
            is_default_value = value == field.default
        elif field.default_factory != dataclasses.MISSING:
            is_default_value = value == field.default_factory()
        else:
            is_default_value = False

        is_hash_field = field.compare if field.hash is None else field.hash

        exclude = (exclude_non_hash_fields and not is_hash_field) or (
            exclude_default_values and is_default_value
        )

        if not exclude:
            dict_obj[field.name] = value

    return dict_obj

@dataclass(frozen=True, eq=True)
class BasisSet:
    """
    class containing basic information about a quantum chemistry basis set
    This set was generated with data from https://www._basissetexchange.org [1]:

    .. data:: basis_sets

    dictionary containing a lot of basis sets sorted by their name in the Basis Set Exchange database [1]. Example:

    .. code::

        from ctydata._basissets import basis_sets

        basis_set = basis_sets['def2-QZVP']

    Two BasisSet objects are equal and have the same hash, iff they have the
    same name and type.

    [1] Basis Set Exchange database
    A New Basis Set Exchange: An Open, Up-to-date Resource for the Molecular
    Sciences Community. Benjamin P. Pritchard, Doaa Altarawy, Brett Didier,
    Tara D. Gibson, Theresa L. Windus. J. Chem. Inf. Model. 2019, 59(11),
    4814-4820, https://doi.org/10.1021/acs.jcim.9b00725
    """

    name: str
    description: str = field(compare=False)
    _warn_on_creation = True

    # def __post_init__(self):
    #     if self._warn_on_creation:
    #         logging.warning(
    #             "_BasisSet objects should not be initialized manually! "
    #             "Instead import the basis_set dictionary and use it to retrieve "
    #             "_BasisSet objects. This is done to avoid typos and keep the naming "
    #             "of basis sets. consistent."
    #         )

    def __eq__(self, __o: object) -> bool:
        # the basis sets should be compared by their name, to allow the creation
        # of special basis set types via inheritance, we also check the type
        return type(__o) is type(self) and __o.name == self.name

    def __hash__(self) -> int:
        # __hash__ and __eq__ must be consistent-> use name and type for hashing
        return int(
            hashlib.sha1(bytes(self.name, encoding='utf-8')).hexdigest(), 16
        ) ^ hash(type(self))


# use frozen and equal to ensure that a __hash__ function is generated which is
# needed for fast lookup in the database
@dataclass(frozen=True, eq=True)
class PESMethod:
    """
    Computational chemistry method, e.g. specific force field or the type of
    hybrid functional.

    When looking up a level of theory/PESMethod in the database, the hash of
    this object is used
    """

    citation: str = field(compare=False, hash=False, default=None, kw_only=True)
    """A DOI, Bibtex string or similar for the method. This variable will not be
    used in comparisions or hashing.

    It will always be set to None and can be changed after initialization."""
    description: str = field(compare=False, hash=False, default=None, kw_only=True)
    """description of the method (ignored during comparisons)"""

    def __post_init__(self):
        for field in dataclasses.fields(type(self)):
            # turn all fields where __to_lower__ is True to lowercase
            if "__to_lower__" in field.metadata:
                if field.metadata["__to_lower__"]:
                    object.__setattr__(
                        self, field.name, getattr(self, field.name).lower()
                    )


@dataclass(frozen=True, eq=True)
class MolecularMechanics(PESMethod):
    """molecular mechanics (Force Field) methods"""

    name: str = field(metadata=_LOWER_CASE)
    """user defined name.

    .. note:: This is used when determining if two methods are
              equal, so be careful to avoid typos!"""

    GFN_FF: ClassVar["MolecularMechanics"]
    REAX_FF: ClassVar["MolecularMechanics"]


@dataclass(frozen=True, eq=True)
class ReaxFF(MolecularMechanics):
    """reactive force fields"""

    file: Union[os.PathLike, str] = field(compare=False, hash=False, default=None, kw_only=True)
    """Path to the ReaxFF file. (not used for comparison)"""


@dataclass(frozen=True, eq=True)
class QCMethod(PESMethod):
    """base class for quantum mechanical methods"""

    # semi-empirical methods
    # DFTB methods
    GFN_xTB: ClassVar["SemiEmpiricalMethod"]
    GFN2_xTB: ClassVar["SemiEmpiricalMethod"]
    DFTB3ob: ClassVar["SemiEmpiricalMethod"]

    # NDDO methods
    PM6: ClassVar["SemiEmpiricalMethod"]

    # ab-initio methods
    HF: ClassVar["WavefunctionMethod"]
    DLPNO_CCSD_T: ClassVar["DLPNO_CCSD_T"]

    # DFT
    # (meta-)GGA functionals
    B97M_D3BJ: ClassVar["Dft"]
    B97M_V: ClassVar["Dft"]
    TPSS_D3: ClassVar["Dft"]
    TPSS_D3BJ: ClassVar["Dft"]
    TPSS_D4: ClassVar["Dft"]

    # hybrid functionals
    wB97M_V: ClassVar["Dft"]
    wB97M_D3BJ: ClassVar["Dft"]
    wB97M_D4: ClassVar["Dft"]
    wB97X_V: ClassVar["Dft"]
    wB97X_D3: ClassVar["Dft"]
    wB97X_D3BJ: ClassVar["Dft"]
    wB97X_D4: ClassVar["Dft"]
    B3LYP_G_D3: ClassVar["Dft"]
    B3LYP_G_D3BJ: ClassVar["Dft"]
    B3LYP_G_D4: ClassVar["Dft"]
    B3LYP_TM_D3: ClassVar["Dft"]
    B3LYP_TM_D3BJ: ClassVar["Dft"]
    B3LYP_TM_D4: ClassVar["Dft"]


@dataclass(frozen=True, eq=True)
class SemiEmpiricalMethod(QCMethod):
    """semi-empirical quantum mechanical methods"""

    name: str = field(metadata=_LOWER_CASE)
    """user defined name

    .. note:: This is used when determining if two methods are
              equal, so be careful to avoid typos!"""

@dataclass(frozen=True, eq=True)
class WavefunctionMethod(QCMethod):
    """wave function (ab-initio) methods, like Hartree-Fock and post-HF methods"""

    name: str = field(metadata=_LOWER_CASE)
    """user defined name

    .. note:: This is used when determining if two methods are
              equal, so be careful to avoid typos!"""


@dataclass(frozen=True, eq=True)
class CoupledCluster(WavefunctionMethod):
    """base class for coupled cluster classes"""


@dataclass(frozen=True, eq=True)
class DLPNO_CCSD_T(CoupledCluster):
    """DLPNO-CCSD(T)"""

    aux_basis: BasisSet = field(compare=False, hash=False)
    """ORCA uses RI for integral transformations, so an auxiliary basis set is
    required. """


class _RI_KEYWORDS(BasisSet):
    """used to define special keywords when specifying auxiliary basis sets"""

    def __post_init__(self):
        # override to avoid triggering a warning
        pass


RI_CORRESPONDING_BS = _RI_KEYWORDS("RI_CORRESPONDING_BS", description=None)
"""Choose this special value as auxiliary basis set to indicate that the density
fitting basis set should correpsond to the basis set speciefied in
LevelOfTheory.el_struc"""
RI_GENERATE_BS = _RI_KEYWORDS("RI_GENERATE_BS", description=None)
"""Choose this special value as auxiliary basis set to indicate that the QM
Software should generate a density fitting basis set"""


@dataclass(frozen=True, eq=True)
class Dft(QCMethod):
    """density-functional theory"""

    functional: str = field(metadata=_LOWER_CASE)
    """name of the functional (case-sensitive)"""
    dispersion_correction: str = field(default=None, metadata=_LOWER_CASE)
    """name of the dispersion correction method"""
    density_fit_J: BasisSet = field(default=None, compare=False, hash=False)
    """indicates whether to use density fitting for the Coulomb matrix. Since density fitting should not significantly change the results, this options is not included when comparing or hashing level of theory objects."""

    # these constants are supplied to avoid typos/ambiguities
    D3: ClassVar[str] = "d3"
    """Grimme's D3 disperion correction with zero-damping"""
    D3BJ: ClassVar[str] = "d3bj"
    """Grimme's D3 disperion correction using Becke-Johnson damping"""
    D4: ClassVar[str] = "d4"
    """Grimme's D4 disperion correction with zero-damping"""


class SolvationMethod(PESMethod):
    """Solvation model used to get the solvation energy"""

    ...


@dataclass(frozen=True, eq=True)
class QMSoftware:
    """used to indicate with which program data was or is supposed to be
    computed."""

    name: str
    """name of the program"""
    version: str = None

    ORCA: ClassVar["QMSoftware"]
    GAUSSIAN: ClassVar["QMSoftware"]
    TURBOMOLE: ClassVar["QMSoftware"]
    DFTBPLUS: ClassVar["QMSoftware"]

QMSoftware.ORCA = QMSoftware(name="ORCA")
QMSoftware.GAUSSIAN = QMSoftware(name="GAUSSIAN")
QMSoftware.TURBOMOLE = QMSoftware(name="TURBOMOLE")
QMSoftware.DFTBPLUS = QMSoftware(name="DFTBPLUS")


@dataclass(frozen=True, eq=True)
class ElectronicStructureProperties:
    """Contains metadata of computed geometries and energies

    :param charge: total charge of system
    :param multiplicity: total spin multiplicity of the system
    :basis_set: basis set used in QM calculations
    """

    basis_set: BasisSet
    charge: int = 0
    multiplicity: int = 1


# use frozen and equal to ensure that a __hash__ function is generated which is
# needed for fast lookup in the database
@dataclass(frozen=True, eq=True)
class LevelOfTheory(IJsonConvertible):
    """
    Describes how PES data (geometries or energies) was or is supposed to be
    generated, i.e. method, basis set, multiplicity, ...
    If the specified method is an electronic structure method, el_struc has to
    be specified.

    This class and all its fields are either simple data types or data classes.
    The level of theory may be used to look up data in a database which is done
    via its hash value.

    .. note::

        Be careful when setting string type values of e.g. PESMethod-type
        objects like DFT.functional or ReaxFF.name. Typos and inconsistencies
        can lead to problems, e.g. when looking up geometries in a database at a
        specified level of theory, because 'omegaB97' and 'wB97' or 'GFN1-xTB'
        and 'GFN-xTB' are considered different methods by this class (and
        consequently a database). To help you to stay consistent, some
        subclasses of PESMethod contain class variables that follow a consistent
        naming convention. It is recommended to use these variables or names
        consistent with them. If you want to slightly change one of those class
        variables, you can use the replace method from Python's dataclasses
        module.

    Not all fields are included when comparing two level of theory objects. For
    example

    >>> lot1 = LevelOfTheory(Dft.B3LYP_G_D3, ElectronicStructureProperties(basis_sets['def2-SVP']), QMSoftware.ORCA)
    >>> lot2 = LevelOfTheory(Dft.B3LYP_G_D3, ElectronicStructureProperties(basis_sets['def2-SVP']), QMSoftware.GAUSSIAN)
    >>> lot1 == lot2
    True

    :param method: computational chemistry method of generation
    :param el_struc:  contains basis set, charge and multiplicity
    :type el_struc: ElectronicStructureProperties
    :param software: quantum mechanics software (not used in comparison of two
                     LevelOfTheory objects)
    """

    method: PESMethod
    el_struc: ElectronicStructureProperties = None
    software: QMSoftware = field(compare=False, hash=False, default=None)

    def __post_init__(self):
        # some sanity check to help avoid bugs
        if (isinstance(self.method, QCMethod)
            and not isinstance(self.method, SemiEmpiricalMethod)
            and self.el_struc is None):
            raise TypeError(
                "A basis set and a multiplicity must be defined for"
                " any quantum-mechanical method"
            )

        if not isinstance(self.method, QCMethod) and self.el_struc is not None:
            raise TypeError(
                "Electronic structure properties (basis set, charge"
                " and multiplicity) cannot be defined for a non-"
                "QM method"
            )

    def to_json(
        self,
        exclude_non_hash_fields: bool = False,
        exclude_default_values: bool = False,
    ) -> str:
        """converts the level of theory into a json representation.

        The function is designed such that it produces the same JSON string for the same class, so the string or its hash can be used for equality comparisons of two LevelOfTheory objects

        :param exclude_non_hash_fields: if True, all fields that are not considered when computing the hash are also not included in the JSON representation.
        :param exclude_default_values: if True, all fields that are set to their default value will be excluded from the JOSN.
        """
        return json.dumps(
            self,
            default=lambda datacls_obj: _dataclass2dict(
                datacls_obj,
                exclude_non_hash_fields=exclude_non_hash_fields,
                exclude_default_values=exclude_default_values,
            ),
            separators=(",", ":"),
            indent=None,
            ensure_ascii=False,
            sort_keys=True,
        )

    @classmethod
    def from_json(cls, json_str: str) -> "LevelOfTheory":
        as_dict = json.loads(json_str)

        return _dict2dataclass(
            as_dict,
            cls=LevelOfTheory,
            # list all possible classes that may have to be instantiated
            cls_table={
                _cls.__name__: _cls
                for _cls in [
                    LevelOfTheory,
                    Dft,
                    MolecularMechanics,
                    ReaxFF,
                    QCMethod,
                    QMSoftware,
                    CoupledCluster,
                    DLPNO_CCSD_T,
                    SemiEmpiricalMethod,
                    WavefunctionMethod,
                    SolvationMethod,
                ]
            },
        )


MolecularMechanics.GFN_FF = MolecularMechanics(
    name="gfn-ff", description="force field based on the GFN-xtb methods"
)
MolecularMechanics.REAX_FF = _dict2dataclass(
    dict(
        name="reax-ff",
        desciption="The first reactive force field by van Duin",
        citation="doi:10.1021/jp004368u",
    ),
    ReaxFF,
)

# semi-empirical methods
# DFTB methods
QCMethod.GFN_xTB = SemiEmpiricalMethod(name="gfn-xtb")
QCMethod.GFN2_xTB = SemiEmpiricalMethod(name="gfn2-xtb")
QCMethod.DFTB3ob = _dict2dataclass(
    dict(name="dftb3ob", description="DFTB3 version for bio and organic molecules"),
    SemiEmpiricalMethod,
)

# NDDO methods
QCMethod.PM6 = SemiEmpiricalMethod(name="pm6")


# ab-initio methods
QCMethod.HF = WavefunctionMethod(name="hf")
QCMethod.DLPNO_CCSD_T = DLPNO_CCSD_T(
    name="dlpno-ccsd(t)",
    description="Domain based local pair natural orbtial coupled cluster method with single and doulbe excitations and perturbative triple excitations",
    aux_basis=RI_CORRESPONDING_BS,
)

# DFT
# (meta-)GGA functionals
QCMethod.B97M_D3BJ = Dft(
    functional="b97m-v",
    dispersion_correction=Dft.D3BJ,
    density_fit_J=RI_CORRESPONDING_BS,
    description="Head-Gordon's B97M-V functional with Grimme's disperion correction by Najibi and Goerigk.",
)
QCMethod.B97M_V = Dft(
    functional="b97m-v",
    dispersion_correction="b97m-v",
    density_fit_J=RI_CORRESPONDING_BS,
    description="Head-Gordon's B97M-V functional with VV10 nonlocal correlation.",
)
QCMethod.TPSS_D3 = Dft(
    functional="tpss",
    dispersion_correction=Dft.D3,
    density_fit_J=RI_CORRESPONDING_BS,
    description="TPSS functional with Grimme's disperion correction",
)
QCMethod.TPSS_D3BJ = Dft(
    functional="tpss",
    dispersion_correction=Dft.D3BJ,
    density_fit_J=RI_CORRESPONDING_BS,
    description="TPSS functional with Grimme's disperion correction and Becke-Johnson damping.",
)
QCMethod.TPSS_D4 = Dft(
    functional="tpss",
    dispersion_correction=Dft.D4,
    density_fit_J=RI_CORRESPONDING_BS,
    description="TPSS functional with Grimme's D4 disperion correction.",
)

# hybrid functionals
QCMethod.wB97M_V = Dft(functional="wb97m-v", dispersion_correction="wb97m-v")
QCMethod.wB97M_D3BJ = Dft(functional="wb97m-v", dispersion_correction=Dft.D3BJ)
QCMethod.wB97M_D4 = Dft(functional="wb97m-v", dispersion_correction=Dft.D4)
QCMethod.wB97X_V = Dft(functional="wb97x-v", dispersion_correction="wb97m-v")
QCMethod.wB97X_D3 = Dft(functional="wb97x-v", dispersion_correction=Dft.D3)
QCMethod.wB97X_D3BJ = Dft(functional="wb97x-v", dispersion_correction=Dft.D3BJ)
QCMethod.wB97X_D4 = Dft(functional="wb97x-v", dispersion_correction=Dft.D4)
QCMethod.B3LYP_G_D3 = _dict2dataclass(
    cls=Dft,
    dict_obj={
        "functional": "b3lyp/g",
        "dispersion_correction": Dft.D3,
        "description": "GAUSSIAN's version of the B3LYP functional with Grimme's D3 dispersion correction with zero damping",
    },
)
QCMethod.B3LYP_G_D3BJ = _dict2dataclass(
    cls=Dft,
    dict_obj={
        "functional": "b3lyp/g",
        "dispersion_correction": Dft.D3BJ,
        "description": "GAUSSIAN's version of the B3LYP functional with Grimme's D3 dispersion correction using Becke-Johnson damping",
    },
)
QCMethod.B3LYP_G_D4 = _dict2dataclass(
    cls=Dft,
    dict_obj={
        "functional": "b3lyp/g",
        "dispersion_correction": Dft.D4,
        "description": "GAUSSIAN's version of the B3LYP functional with Grimme's D4 dispersion correction",
    },
)
QCMethod.B3LYP_TM_D3 = _dict2dataclass(
    cls=Dft,
    dict_obj={
        "functional": "b3lyp/tm",
        "dispersion_correction": Dft.D3,
        "description": "TURBOMOLES's version of the B3LYP functional with Grimme's D3 dispersion correction with zero damping",
    },
)
QCMethod.B3LYP_TM_D3BJ = _dict2dataclass(
    cls=Dft,
    dict_obj={
        "functional": "b3lyp/tm",
        "dispersion_correction": Dft.D3BJ,
        "description": "TURBOMOLES's version of the B3LYP functional with Grimme's D3 dispersion correction using Becke-Johnson damping",
    },
)
QCMethod.B3LYP_TM_D4 = _dict2dataclass(
    cls=Dft,
    dict_obj={
        "functional": "b3lyp/tm",
        "dispersion_correction": Dft.D4,
        "description": "TURBOMOLES's version of the B3LYP functional with Grimme's D4 dispersion correction",
    },
)


# Since the list of basis sets is so long, it is in another module. To avoid
# circular imports, we need to import it after BasisSet is defined. We also
# want the name basis_sets as part of this modules namespace, so we reassign
# it.
from chemtrayzer.core._basissets import _basis_sets as basis_sets  # noqa: E402
basis_sets = basis_sets