# ruff: noqa
"""Fits for rate constant models

This module contains classes derived from core.kinetics.ReactionRate that represent fits of heuristic models
to computed or measured data.
"""

__all__ = ['ModifiedArrheniusRate']

from numbers import Number
from typing import Union, Tuple

import numpy as np
from scipy.optimize import lsq_linear

from chemtrayzer.core.constants import R
from chemtrayzer.core.kinetics import ReactionRate


class ModifiedArrheniusRate(ReactionRate):
    r'''modified version of Arrhenius' rate law

    .. math::

        k = AT^b\exp\left( - \frac{E_a}{R T} \right)

    :param A: pre-exponential factor, same unit as k
    :param b: temperature exponent [-]
    :param E_a: activation energy [Joule/mol]
    '''

    def __init__(self, A: Number, b: Number, E_a: Number) -> None:
        super().__init__()

        self.A = A
        self.b = b
        self.E_a = E_a


    def k(self, T: Union[Number, np.ndarray]) -> Union[Number, np.ndarray]:
        return self.A * T**self.b * np.exp(-self.E_a/(R*T))


    @classmethod
    def fit(cls, T: np.ndarray, k: np.ndarray)\
            -> Tuple['ModifiedArrheniusRate', Number]:
        '''fit the Arrhenius parameters to given data.

        :param T: temperatures [K]
        :param k: rate constants (unit depends on order of the reaction)
        :return: fitted model, residuals
        '''

        b = np.log(k)
        A = np.zeros((T.shape[0], 3))

        A[:, 0] = 1
        A[:, 1] = np.log(T)
        A[:, 2] = -1/(R*T)

        result = lsq_linear(A, b)

        lnA, b, E_a = result.x
        residuals = result.fun

        model = cls(A=np.exp(lnA), b=b, E_a=E_a)

        return model, residuals


    @classmethod
    def fit_to(cls, k_model: ReactionRate,
               T_interval: Tuple[float, float] = (298, 3000),
               n_points = 100)\
            -> Tuple['ModifiedArrheniusRate', Number]:
        '''fit the Arrhenius parameters to data generated by another rate constant model

        :param k_model: other model used to generate data to fit to
        :param T_interval: interval over which to fit
        :param n_points: number of points on a log scale on which k_model is
                         evaluated
        :return: fitted model, residuals
        '''
        # evenly distributed points on logarithmic scale because the fitting
        # procedure is also logarithmic
        Ts = np.geomspace(T_interval[0], T_interval[1], num=n_points)

        return cls.fit(Ts, k_model.k(Ts))
