"""
Spec-Kit bidirectional sync implementation.

This module provides bidirectional synchronization between Spec-Kit markdown artifacts
and SpecFact plans/protocols. It detects changes, merges updates, and resolves conflicts.
"""

from __future__ import annotations

import hashlib
from dataclasses import dataclass
from pathlib import Path
from typing import Any

from beartype import beartype
from icontract import ensure, require

from specfact_cli.importers.speckit_converter import SpecKitConverter
from specfact_cli.importers.speckit_scanner import SpecKitScanner


@dataclass
class SyncResult:
    """
    Result of sync operation.

    Attributes:
        status: Sync status ("success" | "conflict" | "error")
        changes: List of detected changes
        conflicts: List of conflicts (if any)
        merged: Merged artifacts
    """

    status: str
    changes: list[dict[str, Any]]
    conflicts: list[dict[str, Any]]
    merged: dict[str, Any]

    @beartype
    def __post_init__(self) -> None:
        """Validate SyncResult after initialization."""
        valid_statuses = ["success", "conflict", "error"]
        if self.status not in valid_statuses:
            msg = f"Status must be one of {valid_statuses}, got {self.status}"
            raise ValueError(msg)


class SpecKitSync:
    """
    Bidirectional sync between Spec-Kit and SpecFact.

    Synchronizes changes between Spec-Kit markdown artifacts (generated by Spec-Kit
    slash commands) and SpecFact plan bundles/protocols.
    """

    @beartype
    def __init__(self, repo_path: Path) -> None:
        """
        Initialize Spec-Kit sync.

        Args:
            repo_path: Path to repository root
        """
        self.repo_path = Path(repo_path).resolve()
        self.scanner = SpecKitScanner(self.repo_path)
        self.converter = SpecKitConverter(self.repo_path)
        self.hash_store: dict[str, str] = {}

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @require(lambda repo_path: repo_path.is_dir(), "Repository path must be a directory")
    @ensure(lambda result: isinstance(result, SyncResult), "Must return SyncResult")
    @ensure(lambda result: result.status in ["success", "conflict", "error"], "Status must be valid")
    def sync_bidirectional(self, repo_path: Path | None = None) -> SyncResult:
        """
        Sync changes between Spec-Kit and SpecFact artifacts bidirectionally.

        Note: Spec-Kit is a workflow tool that generates markdown artifacts through
        slash commands. This method synchronizes the **artifacts that Spec-Kit commands
        have already generated**, not run Spec-Kit commands ourselves.

        Args:
            repo_path: Path to repository (default: self.repo_path)

        Returns:
            Sync result with changes, conflicts, and merged artifacts
        """
        if repo_path is None:
            repo_path = self.repo_path

        # 1. Detect changes in Spec-Kit artifacts
        speckit_changes = self.detect_speckit_changes(repo_path)

        # 2. Detect changes in SpecFact artifacts
        specfact_changes = self.detect_specfact_changes(repo_path)

        # 3. Merge bidirectional changes
        merged = self.merge_changes(speckit_changes, specfact_changes)

        # 4. Detect conflicts
        conflicts = self.detect_conflicts(speckit_changes, specfact_changes)

        # 5. Resolve conflicts if any
        if conflicts:
            resolved = self.resolve_conflicts(conflicts)
            merged = self.apply_resolved_conflicts(merged, resolved)

        return SyncResult(
            status="conflict" if conflicts else "success",
            changes=[speckit_changes, specfact_changes],
            conflicts=conflicts,
            merged=merged,
        )

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @ensure(lambda result: isinstance(result, dict), "Must return dict")
    def detect_speckit_changes(self, repo_path: Path) -> dict[str, Any]:
        """
        Detect changes in Spec-Kit artifacts.

        Monitors modern Spec-Kit format:
        - `.specify/memory/constitution.md` (from `/speckit.constitution`)
        - `specs/[###-feature-name]/spec.md` (from `/speckit.specify`)
        - `specs/[###-feature-name]/plan.md` (from `/speckit.plan`)
        - `specs/[###-feature-name]/tasks.md` (from `/speckit.tasks`)

        Args:
            repo_path: Path to repository

        Returns:
            Dictionary of detected changes keyed by file path
        """
        changes: dict[str, Any] = {}

        # Check for modern Spec-Kit format (.specify directory)
        specify_dir = repo_path / ".specify"
        if specify_dir.exists():
            # Monitor .specify/memory/ files
            memory_dir = repo_path / ".specify" / "memory"
            if memory_dir.exists():
                for memory_file in memory_dir.glob("*.md"):
                    relative_path = str(memory_file.relative_to(repo_path))
                    current_hash = self._get_file_hash(memory_file)
                    stored_hash = self.hash_store.get(relative_path, "")

                    if current_hash != stored_hash:
                        changes[relative_path] = {
                            "file": memory_file,
                            "hash": current_hash,
                            "type": "modified" if stored_hash else "new",
                        }

            # Monitor specs/ directory for feature specifications
            specs_dir = repo_path / "specs"
            if specs_dir.exists():
                for spec_dir in specs_dir.iterdir():
                    if spec_dir.is_dir():
                        for spec_file in spec_dir.glob("*.md"):
                            relative_path = str(spec_file.relative_to(repo_path))
                            current_hash = self._get_file_hash(spec_file)
                            stored_hash = self.hash_store.get(relative_path, "")

                            if current_hash != stored_hash:
                                changes[relative_path] = {
                                    "file": spec_file,
                                    "hash": current_hash,
                                    "type": "modified" if stored_hash else "new",
                                }

        return changes

    @beartype
    @require(lambda repo_path: repo_path.exists(), "Repository path must exist")
    @ensure(lambda result: isinstance(result, dict), "Must return dict")
    def detect_specfact_changes(self, repo_path: Path) -> dict[str, Any]:
        """
        Detect changes in SpecFact artifacts.

        Monitors:
        - `.specfact/plans/*.yaml`
        - `.specfact/protocols/*.yaml`

        Args:
            repo_path: Path to repository

        Returns:
            Dictionary of detected changes keyed by file path
        """
        changes: dict[str, Any] = {}

        # Monitor .specfact/plans/ files
        plans_dir = repo_path / ".specfact" / "plans"
        if plans_dir.exists():
            for plan_file in plans_dir.glob("*.yaml"):
                relative_path = str(plan_file.relative_to(repo_path))
                current_hash = self._get_file_hash(plan_file)
                stored_hash = self.hash_store.get(relative_path, "")

                if current_hash != stored_hash:
                    changes[relative_path] = {
                        "file": plan_file,
                        "hash": current_hash,
                        "type": "modified" if stored_hash else "new",
                    }

        # Monitor .specfact/protocols/ files
        protocols_dir = repo_path / ".specfact" / "protocols"
        if protocols_dir.exists():
            for protocol_file in protocols_dir.glob("*.yaml"):
                relative_path = str(protocol_file.relative_to(repo_path))
                current_hash = self._get_file_hash(protocol_file)
                stored_hash = self.hash_store.get(relative_path, "")

                if current_hash != stored_hash:
                    changes[relative_path] = {
                        "file": protocol_file,
                        "hash": current_hash,
                        "type": "modified" if stored_hash else "new",
                    }

        return changes

    @beartype
    @ensure(lambda result: isinstance(result, dict), "Must return dict")
    def merge_changes(self, speckit_changes: dict[str, Any], specfact_changes: dict[str, Any]) -> dict[str, Any]:
        """
        Merge changes from both sources.

        Args:
            speckit_changes: Spec-Kit detected changes
            specfact_changes: SpecFact detected changes

        Returns:
            Merged changes dictionary
        """
        merged: dict[str, Any] = {}

        # Merge Spec-Kit changes
        for key, change in speckit_changes.items():
            merged[key] = {
                "source": "speckit",
                **change,
            }

        # Merge SpecFact changes
        for key, change in specfact_changes.items():
            if key in merged:
                # Conflict detected
                merged[key]["conflict"] = True
                merged[key]["specfact_change"] = change
            else:
                merged[key] = {
                    "source": "specfact",
                    **change,
                }

        return merged

    @beartype
    @ensure(lambda result: isinstance(result, list), "Must return list")
    def detect_conflicts(
        self, speckit_changes: dict[str, Any], specfact_changes: dict[str, Any]
    ) -> list[dict[str, Any]]:
        """
        Detect conflicts between Spec-Kit and SpecFact changes.

        Args:
            speckit_changes: Spec-Kit detected changes
            specfact_changes: SpecFact detected changes

        Returns:
            List of conflict dictionaries
        """
        conflicts: list[dict[str, Any]] = []

        for key in set(speckit_changes.keys()) & set(specfact_changes.keys()):
            conflicts.append(
                {
                    "key": key,
                    "speckit_change": speckit_changes[key],
                    "specfact_change": specfact_changes[key],
                }
            )

        return conflicts

    @beartype
    @ensure(lambda result: isinstance(result, dict), "Must return dict")
    def resolve_conflicts(self, conflicts: list[dict[str, Any]]) -> dict[str, Any]:
        """
        Resolve conflicts with merge strategy.

        Strategy:
        - Priority: SpecFact > Spec-Kit for artifacts (specs/*)
        - Priority: Spec-Kit > SpecFact for memory files (.specify/memory/)

        Args:
            conflicts: List of conflict dictionaries

        Returns:
            Resolved conflicts dictionary
        """
        resolved: dict[str, Any] = {}

        for conflict in conflicts:
            file_key = conflict["key"]
            file_type = self._get_file_type(file_key)

            if file_type == "artifact":
                # SpecFact takes priority for artifacts
                resolved[file_key] = {
                    "resolution": "specfact_priority",
                    "source": "specfact",
                    "data": conflict["specfact_change"],
                }
            elif file_type == "memory":
                # Spec-Kit takes priority for memory files
                resolved[file_key] = {
                    "resolution": "speckit_priority",
                    "source": "speckit",
                    "data": conflict["speckit_change"],
                }
            else:
                # Default: SpecFact priority
                resolved[file_key] = {
                    "resolution": "specfact_priority",
                    "source": "specfact",
                    "data": conflict["specfact_change"],
                }

        return resolved

    @beartype
    @ensure(lambda result: isinstance(result, dict), "Must return dict")
    def apply_resolved_conflicts(self, merged: dict[str, Any], resolved: dict[str, Any]) -> dict[str, Any]:
        """
        Apply resolved conflicts to merged changes.

        Args:
            merged: Merged changes dictionary
            resolved: Resolved conflicts dictionary

        Returns:
            Updated merged changes dictionary
        """
        for key, resolution in resolved.items():
            if key in merged:
                merged[key]["conflict"] = False
                merged[key]["resolution"] = resolution["resolution"]
                merged[key]["source"] = resolution["source"]

        return merged

    @beartype
    def _get_file_hash(self, file_path: Path) -> str:
        """
        Get file hash for change detection.

        Args:
            file_path: Path to file

        Returns:
            SHA256 hash of file contents
        """
        if not file_path.exists():
            return ""

        with file_path.open("rb") as f:
            content = f.read()
            return hashlib.sha256(content).hexdigest()

    @beartype
    def _get_file_type(self, file_path: str) -> str:
        """
        Determine file type for conflict resolution.

        Args:
            file_path: Relative file path

        Returns:
            File type ("artifact" | "memory" | "other")
        """
        if "/memory/" in file_path or file_path.startswith(".specify/memory/"):
            return "memory"
        if "/specs/" in file_path or file_path.startswith("specs/"):
            return "artifact"
        return "other"
