"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiPipelineLib = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const codepipeline = require("aws-cdk-lib/aws-codepipeline");
const codepipeline_actions = require("aws-cdk-lib/aws-codepipeline-actions");
const ec2 = require("aws-cdk-lib/aws-ec2");
const events = require("aws-cdk-lib/aws-events");
const event_targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const imagebuilder = require("aws-cdk-lib/aws-imagebuilder");
const kms = require("aws-cdk-lib/aws-kms");
const sns = require("aws-cdk-lib/aws-sns");
const constructs_1 = require("constructs");
const cleaner_1 = require("./cleaner");
const component_builder_1 = require("./component_builder");
const image_recipe_version_bump_1 = require("./image_recipe_version_bump");
const slack_notification_1 = require("./slack_notification");
const source_action_builder_1 = require("./source_action_builder");
const ssmupdate = require("./ssm_update");
const utils = require("./utils");
/**
 * Construct for creating a Codepipeline, EC2 Image builder pipeline from 1 pipeline configuration.
 */
class AmiPipelineLib extends constructs_1.Construct {
    /**
     * Constructor
     * @param scope
     * @param id
     * @param pipelineConfig
     * @param componentDepsConfig
     */
    constructor(scope, id, pipelineConfig, componentDepsConfig, componentBuilder, optionalParams) {
        super(scope, id);
        this.id = id;
        this.pipelineConfig = pipelineConfig;
        this.pipelineConfig.name = this.id;
        this.componentDepsConfig = componentDepsConfig;
        this.slackConfig = {
            channel: optionalParams.channel,
            slackWebhookUrl: optionalParams.slackWebhookUrl,
            username: optionalParams.username,
        };
        this.componentBuilder = componentBuilder ?? new component_builder_1.ComponentBuilder(this, componentDepsConfig);
        this.sourceActionBuilder = new source_action_builder_1.SourceActionBuilder(this, pipelineConfig.sources, this.id);
        this.extraParameters = optionalParams.extraParams;
        this.createImagebuilderPipeline();
        this.createCodepipelineProject();
        this.createScheduledTask();
        this.createCleanerTask();
        if (this.topic) {
            new ssmupdate.SsmUpdateConstruct(this, 'SSMUpdate', this.topic, this.pipelineConfig);
            if (this.slackConfig && this.slackConfig.channel && this.slackConfig.slackWebhookUrl && this.slackConfig.username) {
                new slack_notification_1.SlackNotification(this, 'SlackNotification', this.topic, this.slackConfig, `${this.id}Recipe`);
            }
        }
    }
    createCleanerTask() {
        if (this.pipelineConfig.cleaning_schedule) {
            new cleaner_1.Cleaner(this, 'Cleaner', this.pipelineConfig.cleaning_schedule, this.id);
        }
    }
    createScheduledTask() {
        if (this.codepipeline && this.pipelineConfig.schedule) {
            const pipelineTarget = new event_targets.CodePipeline(this.codepipeline);
            new events.Rule(this, 'ScheduleRule', {
                schedule: events.Schedule.expression(this.pipelineConfig.schedule),
                targets: [pipelineTarget],
            });
        }
    }
    createImagebuilderPipeline() {
        let image_id;
        let parentImage;
        let cpuType;
        if ('disk_size' in this.pipelineConfig && this.pipelineConfig.disk_size) {
            this.diskSize = this.pipelineConfig.disk_size;
        }
        if ('cpu_type' in this.pipelineConfig && this.pipelineConfig.cpu_type == 'ARM_64') {
            cpuType = ec2.AmazonLinuxCpuType.ARM_64;
        }
        else {
            cpuType = ec2.AmazonLinuxCpuType.X86_64;
        }
        if ('image_id' in this.pipelineConfig && this.pipelineConfig.image_id) {
            image_id = this.pipelineConfig.image_id;
        }
        else {
            switch (this.pipelineConfig.parent_image) {
                case 'AmazonLinux2':
                    parentImage = ec2.MachineImage.latestAmazonLinux({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        edition: ec2.AmazonLinuxEdition.STANDARD,
                        virtualization: ec2.AmazonLinuxVirt.HVM,
                        storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                        cpuType: cpuType,
                    });
                    break;
                default:
                    parentImage = ec2.MachineImage.lookup(this.getLookupCriteria(this.pipelineConfig.parent_image, cpuType));
                    break;
            }
        }
        // Use a custom block device mapping if encryption is required.
        let blockDeviceMappings = this.createBlockDeviceMapping();
        this.recipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            name: `${this.id}Recipe`,
            parentImage: image_id ? image_id : parentImage ? parentImage.getImage(this).imageId : null,
            version: this.getNextRecipeVersion(`${this.id}Recipe`),
            components: this.getComponents(this.pipelineConfig.recipe ? this.pipelineConfig.recipe.components : []),
            blockDeviceMappings: blockDeviceMappings,
            tags: {
                ShortName: `${this.id}Recipe`,
            },
        });
        const builderRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
            ],
            inlinePolicies: this.getInlinePolicies(this.pipelineConfig),
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [builderRole.roleName],
        });
        this.topic = new sns.Topic(this, 'AmiPipelineTopic');
        this.infrastructure = new imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: `${this.id} - EC2 IB Infrastructure`,
            instanceProfileName: instanceProfile.ref,
            snsTopicArn: this.topic.topicArn,
            instanceTypes: this.pipelineConfig.instance_type ? [this.pipelineConfig.instance_type] : undefined,
            terminateInstanceOnFailure: !('terminate_on_failure' in this.pipelineConfig) || this.pipelineConfig.terminate_on_failure,
            subnetId: this.pipelineConfig.subnet_id,
            securityGroupIds: this.pipelineConfig.security_group_ids ? this.pipelineConfig.security_group_ids.split(',') : undefined,
        });
        this.distributionConfig = this.createDistributionConfig();
        this.imagePipeline = new imagebuilder.CfnImagePipeline(this, 'AmiPipeline', {
            name: `${this.id} - AMI Pipeline`,
            imageRecipeArn: this.recipe.ref,
            infrastructureConfigurationArn: this.infrastructure.ref,
            distributionConfigurationArn: this.distributionConfig ? this.distributionConfig.ref : undefined,
        });
    }
    getComponents(components) {
        let compMap = this.componentBuilder.componentDependenciesMap;
        if (this.extraParameters) {
            for (let param of this.extraParameters) {
                let component = components.find((c) => c.name === param.componentName);
                if (component) {
                    component.parameters = component.parameters || {};
                    component.parameters = {
                        ...component.parameters,
                        ...param.parameters,
                    };
                }
            }
        }
        let compList = components.map((c) => ({
            componentArn: compMap[c.name].ref,
            parameters: 'parameters' in c ? Object.keys(c.parameters).map((k) => ({
                name: k,
                value: [c.parameters[k]],
            })) : [],
        }));
        for (let c of compList) {
            if (c.parameters && c.parameters.length === 0) {
                delete c.parameters;
            }
        }
        return compList;
    }
    createDistributionConfig() {
        // Get current region
        const region = cdk.Stack.of(this).region;
        const accountId = cdk.Stack.of(this).account;
        const amiTags = {
            Name: this.id,
            CreatedBy: 'EC2 Image Builder',
            Ec2ImageBuilderArn: '{{imagebuilder:buildVersion}}',
            AutoClean: 'True',
        };
        let sharedWithDistributions = [];
        if (this.pipelineConfig.shared_with) {
            sharedWithDistributions = this.pipelineConfig.shared_with.map((s) => ({
                region: s.region,
                amiDistributionConfiguration: {
                    Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                    Description: `${this.id} Shared image`,
                    AmiTags: amiTags,
                    LaunchPermissionConfiguration: {
                        UserIds: s.account_ids,
                    },
                },
            }));
            // current region might already be included
            if (sharedWithDistributions.some(s => s.region === region)) {
                const distribution = sharedWithDistributions.find(s => s.region === region);
                //Push current account ID as distribute
                distribution?.amiDistributionConfiguration.LaunchPermissionConfiguration.UserIds.push(accountId);
            }
        }
        if (this.pipelineConfig.copy_to) {
            console.log('Pipeline config copy to', this.pipelineConfig.copy_to);
            sharedWithDistributions = [...sharedWithDistributions, ...this.pipelineConfig.copy_to.map((s) => ({
                    region: s.region,
                    amiDistributionConfiguration: {
                        Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                        Description: `${this.id} Copied image`,
                        AmiTags: amiTags,
                        TargetAccountIds: s.account_ids,
                    },
                }))];
            // current region might already be included
            if (sharedWithDistributions.some(s => s.region === region)) {
                const distribution = sharedWithDistributions.find(s => s.region === region);
                //Push current account ID as distribute
                distribution?.amiDistributionConfiguration.TargetAccountIds.push(accountId);
            }
        }
        if (sharedWithDistributions.length == 0) {
            sharedWithDistributions = [{
                    region: region,
                    amiDistributionConfiguration: {
                        Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                        Description: `${this.id} Shared image`,
                        AmiTags: amiTags,
                        LaunchPermissionConfiguration: {
                            UserIds: [accountId],
                        },
                    },
                }];
        }
        return new imagebuilder.CfnDistributionConfiguration(this, 'DistributionConfig', {
            name: `${this.id} - Distribution`,
            distributions: sharedWithDistributions,
        });
    }
    getInlinePolicies(pipelineConfig) {
        let result = {};
        const statements = pipelineConfig.policy_statements;
        if (statements) {
            for (let statement in statements) {
                result[statement] = new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            sid: statement,
                            actions: statements[statement].actions,
                            effect: iam.Effect.ALLOW,
                            resources: statements[statement].resources,
                        }),
                    ],
                });
            }
        }
        return result;
    }
    createBlockDeviceMapping() {
        const key = this.pipelineConfig.encrypted ? new kms.Key(this, 'PipelineEncryption', {
            description: `Image encryption for ${this.pipelineConfig.name}`,
            alias: `${utils.pascalize(this.pipelineConfig.name)}_key`,
        }) : undefined;
        this.ebsEncryptionKey = key;
        let blockDeviceMappings = [
            {
                deviceName: this.pipelineConfig.parent_image && this.pipelineConfig.parent_image.includes('AmazonLinux') ? '/dev/xvda' : '/dev/sda1',
                ebs: {
                    encrypted: key ? true : false,
                    volumeSize: this.pipelineConfig.disk_size ? this.pipelineConfig.disk_size : 8,
                    volumeType: 'gp3',
                    iops: this.pipelineConfig.iops ? this.pipelineConfig.iops : 3000,
                    kmsKeyId: key?.keyArn,
                },
            },
        ];
        return blockDeviceMappings;
    }
    getNextRecipeVersion(recipeName) {
        const recipeBumpCustomResource = new image_recipe_version_bump_1.ImageRecipeVersionBump(this, 'ImageRecipeVersionBump', recipeName);
        return recipeBumpCustomResource.nextVersion;
    }
    createCodepipelineProject() {
        if (this.imagePipeline) {
            // Create CodePipeline project
            let envVariables = {
                IMAGE_PIPELINE_ARN: { value: this.imagePipeline.ref },
                PIPELINE_NAME: { value: this.imagePipeline.name },
            };
            if (this.slackConfig && this.slackConfig.slackWebhookUrl && this.slackConfig.channel && this.slackConfig.username) {
                envVariables.SLACK_WEBHOOK_URL = { value: this.slackConfig.slackWebhookUrl };
                envVariables.SLACK_CHANNEL = { value: this.slackConfig.channel };
                envVariables.SLACK_USERNAME = { value: this.slackConfig.username };
            }
            let imagebuilderBuild = new codebuild.PipelineProject(this, 'AmiPipelineBuild', {
                buildSpec: codebuild.BuildSpec.fromObject({
                    version: '0.2',
                    phases: {
                        install: {
                            commands: [
                                'apt-get update',
                                'apt-get install awscli -y',
                            ],
                        },
                        pre_build: {
                            commands: [
                                'if [ -n "$SLACK_WEBHOOK_URL" ]; then curl -X POST -H \'Content-type: application/json\' --data "{\\"text\\":\\"$PIPELINE_NAME build has started.\\"}" $SLACK_WEBHOOK_URL; fi',
                            ],
                        },
                        build: {
                            commands: 'aws imagebuilder start-image-pipeline-execution --image-pipeline-arn $IMAGE_PIPELINE_ARN',
                        },
                    },
                }),
                environment: {
                    buildImage: codebuild.LinuxBuildImage.STANDARD_7_0,
                    environmentVariables: envVariables,
                },
            });
            imagebuilderBuild.addToRolePolicy(iam.PolicyStatement.fromJson({
                Sid: 'imagebuilderstart',
                Effect: 'Allow',
                Action: ['imagebuilder:StartImagePipelineExecution'],
                Resource: '*',
            }));
            const imageBuilderOutput = new codepipeline.Artifact('CdkBuildOutput');
            let sourceActions = this.sourceActionBuilder.createPipelineSources();
            const buildAction = new codepipeline_actions.CodeBuildAction({
                actionName: 'StartImageBuilder',
                project: imagebuilderBuild,
                input: sourceActions.find(a => a && a.sourceOutput)?.sourceOutput,
                extraInputs: sourceActions.length > 1 ? sourceActions.slice(1).map(a => a.sourceOutput) : [],
                outputs: [imageBuilderOutput],
            });
            // Complete Pipeline Project
            this.codepipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: this.pipelineConfig.name,
                restartExecutionOnUpdate: true,
                stages: [
                    {
                        stageName: 'Source',
                        actions: sourceActions.map(c => c.action).filter((c) => c != null),
                    },
                    {
                        stageName: 'Build',
                        actions: [buildAction],
                    },
                ],
            });
        }
    }
    getLookupCriteria(parentImage, cpuType) {
        switch (parentImage) {
            case 'Ubuntu1804':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-*',
                    owners: ['099720109477'],
                    filters: {
                        architecture: [cpuType.toLowerCase()],
                    },
                };
            case 'Ubuntu2004':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-focal-20.04-*',
                    owners: ['099720109477'],
                    filters: {
                        architecture: [cpuType.toLowerCase()],
                    },
                };
            case 'CentOS7':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['cvugziknvmxgqna9noibqnnsy'],
                    },
                };
            case 'CentOS8':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['47k9ia2igxpcce2bzo8u3kj03'],
                    },
                };
            default:
                return {
                    name: 'amzn-linux',
                    filters: {
                        architecture: [cpuType.toLowerCase()],
                    },
                };
        }
    }
}
exports.AmiPipelineLib = AmiPipelineLib;
_a = JSII_RTTI_SYMBOL_1;
AmiPipelineLib[_a] = { fqn: "halloumi-ami-pipelines.AmiPipelineLib", version: "0.0.40" };
//# sourceMappingURL=data:application/json;base64,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