"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: "us-east-1" },
});
(0, core_1.SetConsistentFeatureFlags)(stack);
stack.templateOptions.description =
    "Integration Test for public HTTP API with a existing function and ALB";
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, "lambda-sg", {
    vpc: myVpc,
    allowAllOutbound: false,
});
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: "index.handler",
        vpc: myVpc,
        securityGroups: [testSg],
    },
});
const loadBalancer = defaults.ObtainAlb(stack, "existing-alb", {
    vpc: myVpc,
    publicApi: false,
    loadBalancerProps: {
        internetFacing: false,
        vpc: myVpc,
    },
    logAccessLogs: true,
    loggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    }
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: "HTTP",
    },
    publicApi: true,
};
const albToLambda = new lib_1.AlbToLambda(stack, "test-one", props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    {
        id: "W56",
        reason: "All integration tests must be HTTP because of certificate limitations.",
    },
]);
defaults.addCfnGuardSuppressRules(albToLambda.listener, ["ELBV2_LISTENER_SSL_POLICY_RULE"]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0]
    .node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
    { id: "W2", reason: "Rule does not apply for ELB." },
    { id: "W9", reason: "Rule does not apply for ELB." },
]);
defaults.addCfnSuppressRules(testSg, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
]);
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::VpcRestrictDefaultSGCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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