"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? false;
        this.user = props.user;
        this.group = props.group;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-23.ContainerSecurityContext", version: "2.0.0-rc.52" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-23.EnvValue", version: "2.0.0-rc.52" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.port = props.port;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup;
        this._lifecycle = props.lifecycle;
        this.resources = props.resources;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        if (this.port) {
            ports.push({
                containerPort: this.port,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this._imagePullPolicyToKube(this.imagePullPolicy),
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
    _imagePullPolicyToKube(imagePullPolicy) {
        switch (imagePullPolicy) {
            case ImagePullPolicy.ALWAYS:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.ALWAYS;
            case ImagePullPolicy.IF_NOT_PRESENT:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.IF_NOT_PRESENT;
            case ImagePullPolicy.NEVER:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.NEVER;
            default:
                throw new Error(`Unsupported image pull policy: ${imagePullPolicy}`);
        }
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-23.Container", version: "2.0.0-rc.52" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-23.Cpu", version: "2.0.0-rc.52" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-23.EnvFrom", version: "2.0.0-rc.52" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    const ports = [];
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers;
    for (const con of containers ?? []) {
        if (con.port) {
            ports.push(con.port);
        }
    }
    return ports;
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-23.Env", version: "2.0.0-rc.52" };
//# sourceMappingURL=data:application/json;base64,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