"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const container = require("./container");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = props.minReady ?? cdk8s_1.Duration.seconds(0);
        this.progressDeadline = props.progressDeadline ?? cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = props.replicas ?? 1;
        this.strategy = props.strategy ?? DeploymentStrategy.rollingUpdate();
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        const ports = options.ports ?? this.extractPorts();
        if (ports.length === 0) {
            throw new Error(`Unable to expose deployment ${this.name} via a service: `
                + 'Deployment port cannot be determined.'
                + 'Either pass \'ports\', or configure ports on the containers of the deployment');
        }
        const metadata = { namespace: this.metadata.namespace };
        if (options.name) {
            metadata.name = options.name;
        }
        return new service.Service(this, `${options.name ?? ''}Service`, {
            selector: this,
            ports,
            metadata,
            type: options.serviceType ?? service.ServiceType.CLUSTER_IP,
        });
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    extractPorts() {
        return container.extractContainerPorts(this).map(port => ({ targetPort: port, port }));
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-23.Deployment", version: "2.0.0-rc.52" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-23.PercentOrAbsolute", version: "2.0.0-rc.52" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: k8s.IoK8SApiAppsV1DeploymentStrategyType.RECREATE,
        });
    }
    static rollingUpdate(options = {}) {
        const maxSurge = options.maxSurge ?? PercentOrAbsolute.percent(25);
        const maxUnavailable = options.maxUnavailable ?? PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: k8s.IoK8SApiAppsV1DeploymentStrategyType.ROLLING_UPDATE,
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-23.DeploymentStrategy", version: "2.0.0-rc.52" };
//# sourceMappingURL=data:application/json;base64,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