# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, wildcard-import, unused-wildcard-import
"""
.. _`Williams (1969)`: \
https://elibrary.asabe.org/abstract.asp?aid=38772&confalias=
.. _`LARSIM`: http://www.larsim.de/en/the-model/
.. _`Todini (2007)`: https://www.hydrol-earth-syst-sci.net/11/1645/2007

Version 1 of *HydPy-L-Stream* is a kinematic wave routing approach.
Initially, we designed it to agree with the variable storage-coefficient
procedure `Williams (1969)`_.  After recognising some accuracy and stability
issues, we reimplemented it in a more hydrodynamical fashion.  It is now
similar to the "DV/DQ FUER WILLIAMS" option of `LARSIM`_, which is also a
more hydrodynamical version of the pure "WILLIAMS" option.

While `Williams (1969)`_ uses precalculated tabulated data, both `LARSIM`_
and *HydPy* calculate the discharge dynamically based on the actualchannel
storage or water stage.  For this purpose, they apply the
Gauckler-Manning-Strickler equation on a triple trapezoid profile of the
following structure:

.. image:: HydPy-L-Stream_Version-1.png

A running |lstream_v001| model consists of a several equally long
channel-subsections, which one can understand as an array of nonlinear
storages.  This nonlinearity is computationally more demanding than using
linear approaches (for example, the linear storage cascade, see |arma_v1|
and |LinearStorageCascade|). Still, it allows to model changes in velocity
due to rising or falling water stages more easily.

|lstream_v001| is not a fully dynamical 1D-approach considering all energy
terms of the Saint-Venant equation.  Instead, as all kinematic approaches,
it takes only the local water stage into account explicitly and ignores
changes in depth or velocity over the channel length. Hence, do not use
|lstream_v001| when you expect substantial backwater effects or other
hydraulic complications.

While |lstream_v001| is kinematic, it still allows for diffusion (or
attenuation) of flood waves. You can control the strength of diffusivity
via parameter |GTS|, which determines the number of channel-subsections.
The more subsections, the smaller the numerical diffusivity: peak flows
decrease less over the complete channel length. Contrary, when you set
|GTS| to one, you get the highest possible flood attenuation.  You can
use method |Model.calculate_characteristiclength| to gain reasonable
estimates for |GTS|.

Please note that a high number of channel-subsections does not only result
in high peak-flows but often also in high computation times.  This
unfavourable aspect is due to increased stiffness, which the explicit
integration method implemented in |ELSModel| cannot handle very efficiently.
Usually, |ELSModel| still returns good results (more correctly: results
meeting the acceptable numerical tolerance), but it may need to decrease
the internal numerical integration step size considerably.

The default values of the numerical tolerance parameters (|AbsErrorMax| and
|RelErrorMax|) should work well for most situations.  However, you are
always free to modify them to either gain more computational speed or
numerical accuracy.

One implementation detail of |lstream_v001| stems from the fact that the
triple trapezoid profile depicted above introduces discontinuities into the
relationships between water stage and some other variables.  Generally,
discontinuities decrease the speed and reliability of numerical integration
algorithms.  As a solution, we "smooth" the affected relationships or, put
more concretely, we apply the function |smooth_logistic2| to calculate
different modified water stages for the locations around the individual
discontinuities.

On the downside, this approach introduces one more control parameter that is
not too easy to understand: |HR|.  The higher the value of |HR|, the smoother
the affected relationships.  Hence, high |HR| values are beneficial for numerical
reasons, but at some point, they caninfluence the calculated results notably.
In our experience, 0.1 m serves as a good compromise value in most cases.

Integration tests
=================

.. how_to_understand_integration_tests::

.. _lstream_v001_main_channel_flow:

main channel flow
_________________

For our first test, we take up one of the examples given by
`Todini (2007)`_, focussing on the routing of a unimodal flood wave
through a 100 km long channel exhibiting the trapezoidal cross-section
shown in figure 5.

>>> from hydpy.models.lstream_v001 import *
>>> parameterstep('1d')

The simulation period spans 96 hours; the simulation step is 30 minutes:

>>> from hydpy import pub, Nodes, Element
>>> pub.timegrids = '2000-01-01', '2000-01-05', '30m'

For testing purposes, the model retrieves its input data from two nodes
(`input1` and `input2`) and passes its output to node `output`.  First,
we define these nodes:

>>> nodes = Nodes('input1', 'input2', 'output')

Second, we define the element `stream` and build the connections between
the prepared nodes and the |lstream_v001| model instance:

>>> stream = Element('stream',
...                  inlets=['input1', 'input2'],
...                  outlets='output')
>>> stream.model = model

Next, we prepare a test function object which sets the intial stage
to 3.71783276 m, which results in an initial outflow of 100 m³/s:

>>> from hydpy.core.testtools import IntegrationTest
>>> IntegrationTest.plotting_options.activated = fluxes.qz, fluxes.qa
>>> test = IntegrationTest(stream, inits=[[states.h, 3.71783276]])

Now we define the geometry and roughness values for the main channel
given by `Todini (2007)`_:

>>> laen(100.0)
>>> gef(0.00025)
>>> bm(15.0)
>>> bnm(5.0)
>>> skm(1.0/0.035)

To enforce that all water flows through the main channel only, we set
its height to infinity:

>>> hm(inf)

Hence, the following values, describing the forelands and outer
embankments, are irrelevant in this first example:

>>> bv(100.0)
>>> bbv(20.0)
>>> bnv(10.0)
>>> bnvr(100.0)
>>> skv(10.0)
>>> ekm(1.0)
>>> ekv(1.0)


The same holds for the value of parameter |HR|, which, for example, defines
the degree of smoothness of the differential equations as soon as water
leaves the main channel:

>>> hr(0.1)

`Todini (2007)`_ divides the channel into different numbers of subsections
in his experiments.  Here, we define eight subsections, which is in
agreement with the characteristic length of the Kalinin-Milyukov method:

>>> gts(8)

As we do not change the default solver parameter values, the accuracy of
the following results should be driven by the relative error estimates,
which must be lower than the tolerance defined by parameter |RelErrorMax|:

>>> parameters.update()
>>> solver.relerrormax
relerrormax(0.001)

Node `input1` supplies constant base flow and node `input2` supplies
the actual flood wave.  The peak occurs after 24 hours and with a
(total) height of 900 m³/s:

>>> import numpy
>>> q_base = 100.0
>>> q_peak = 900.0
>>> t_peak = 24.0
>>> β = 16.0
>>> ts = pub.timegrids.init.to_timepoints()
>>> nodes.input1.sequences.sim.series = q_base
>>> nodes.input2.sequences.sim.series = (
...     (q_peak-q_base)*((ts/t_peak)*numpy.exp(1.0-ts/t_peak))**β)

The following table and figure show all relevant input and output data
as well as all relevant variables calculated internally.  |QG| shows
the typical storage cascade pattern with translation and retention
increasing from subsection to subsection. |lstream_v001| keeps the water
balance with high accuracy.  The height and timing of the calculated peak flow
agree very well with the results of `Todini (2007)`_ (see figure 13):

.. integration-test::

    >>> test('lstream_v001_main_channel_flow')
    |                date |         qz |                                                                                             qg |         qa |                                                                                     dh |                                                                              h | input1 |     input2 |     output |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 100.000029 | 100.000003       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 100.000175 | 100.000017  100.000001       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 100.000826 | 100.000089  100.000008       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717835  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 100.003194 | 100.000372  100.000037  100.000003       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717843  3.717834  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 100.010515 | 100.001326  100.000144  100.000014  100.000001       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717868  3.717837  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 100.030315 | 100.004133  100.000492  100.000054  100.000006       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717941  3.717847  3.717834  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 100.078179 | 100.011493  100.001492   100.00018   100.00002  100.000002       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.718125  3.717875  3.717838  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 100.183351 | 100.028995  100.004079  100.000535  100.000066  100.000007  100.000001       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.718551  3.717944  3.717848  3.717835  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 100.396214 | 100.067234  100.010209  100.001444  100.000192  100.000024  100.000003       100.0       100.0 |      100.0 |  0.000001        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.719458  3.718101  3.717872  3.717838  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.396214 |      100.0 |
    | 2000-01-01 08:00:00 | 100.797305 | 100.144828  100.023644  100.003593  100.000512  100.000069  100.000009  100.000001       100.0 |      100.0 |  0.000001        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.721258  3.718435  3.717927  3.717847  3.717835  3.717833  3.717833  3.717833 |  100.0 |   0.797305 |      100.0 |
    | 2000-01-01 08:30:00 | 101.507032 | 100.292378  100.051143   100.00832  100.001268  100.000182  100.000024  100.000003       100.0 |      100.0 |  0.000002        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.724607  3.719101  3.718046  3.717866  3.717838  3.717833  3.717833  3.717833 |  100.0 |   1.507032 |      100.0 |
    | 2000-01-01 09:00:00 | 102.694853 | 100.557256   100.10411  100.018071  100.002935  100.000448  100.000064  100.000009  100.000001 | 100.000001 |  0.000003   0.000001        0.0        0.0        0.0        0.0        0.0        0.0 | 3.730495  3.720351  3.718283  3.717908  3.717845  3.717835  3.717833  3.717833 |  100.0 |   2.694853 | 100.000001 |
    | 2000-01-01 09:30:00 | 104.586356 | 101.008966  100.200763  100.037071  100.006398  100.001037  100.000159  100.000023  100.000003 | 100.000003 |  0.000005   0.000001        0.0        0.0        0.0        0.0        0.0        0.0 | 3.740335   3.72258  3.718735  3.717993  3.717859  3.717837  3.717833  3.717833 |  100.0 |   4.586356 | 100.000003 |
    | 2000-01-01 10:00:00 | 107.466641 | 101.744688  100.368788  100.072246  100.013214  100.002269  100.000367  100.000056  100.000008 | 100.000008 |  0.000009   0.000002        0.0        0.0        0.0        0.0        0.0        0.0 | 3.756035  3.726372  3.719553  3.718155   3.71789  3.717842  3.717834  3.717833 |  100.0 |   7.466641 | 100.000008 |
    | 2000-01-01 10:30:00 | 111.678559 | 102.894484  100.648428  100.134433  100.025995  100.004715  100.000805   100.00013   100.00002 |  100.00002 |  0.000013   0.000003   0.000001        0.0        0.0        0.0        0.0        0.0 | 3.780046  3.732557  3.720971  3.718455  3.717948  3.717853  3.717836  3.717833 |  100.0 |  11.678559 |  100.00002 |
    | 2000-01-01 11:00:00 | 117.614836 | 104.625672  101.095905  100.239895  100.048929  100.009353  100.001683  100.000286  100.000046 | 100.000046 |   0.00002   0.000005   0.000001        0.0        0.0        0.0        0.0        0.0 | 3.815353  3.742262  3.723331  3.718982  3.718057  3.717874   3.71784  3.717834 |  100.0 |  17.614836 | 100.000046 |
    | 2000-01-01 11:30:00 | 125.703736 | 107.145873  101.786998  100.412139  100.088474  100.017786  100.003365  100.000601  100.000102 | 100.000102 |  0.000028   0.000008   0.000002        0.0        0.0        0.0        0.0        0.0 | 3.865393  3.756962   3.72712  3.719875  3.718253  3.717914  3.717848  3.717835 |  100.0 |  25.703736 | 100.000102 |
    | 2000-01-01 12:00:00 | 136.388647 | 110.704308   102.82069  100.683991  100.154245  100.032544  100.006459   100.00121  100.000215 | 100.000215 |  0.000038   0.000012   0.000003   0.000001        0.0        0.0        0.0        0.0 | 3.933889  3.778513  3.733003  3.721336  3.718588  3.717986  3.717862  3.717838 |  100.0 |  36.388647 | 100.000215 |
    | 2000-01-01 12:30:00 | 150.102726 | 115.590949  104.322601  101.100048  100.260049  100.057487  100.011946  100.002343  100.000435 | 100.000435 |   0.00005   0.000017   0.000005   0.000001        0.0        0.0        0.0        0.0 | 4.024589  3.809164  3.741863  3.723652  3.719147  3.718112  3.717889  3.717843 |  100.0 |  50.102726 | 100.000435 |
    | 2000-01-01 13:00:00 | 167.240349 | 122.132947  106.448326   101.71915  100.425246  100.098313  100.021356  100.004377   100.00085 |  100.00085 |  0.000065   0.000024   0.000007   0.000002   0.000001        0.0        0.0        0.0 | 4.140938  3.851536  3.754839  3.727218  3.720049  3.718324  3.717935  3.717853 |  100.0 |  67.240349 |  100.00085 |
    | 2000-01-01 13:30:00 | 188.127569 | 130.687737  109.386166  102.617261  100.676215  100.163208  100.036998  100.007911  100.001601 | 100.001601 |   0.00008   0.000032    0.00001   0.000003   0.000001        0.0        0.0        0.0 | 4.285711  3.908551  3.773355  3.732562  3.721464  3.718672  3.718016   3.71787 |  100.0 |  88.127569 | 100.001601 |
    | 2000-01-01 14:00:00 | 212.993962 | 141.631926  113.359162  103.890442  101.048112  100.263642   100.06227  100.013869  100.002924 | 100.002924 |  0.000097   0.000042   0.000014   0.000004   0.000001        0.0        0.0        0.0 | 4.460677  3.983309  3.799148  3.740378  3.723627  3.719228  3.718149  3.717901 |  100.0 | 112.993962 | 100.002924 |
    | 2000-01-01 14:30:00 | 241.948195 | 155.345173   118.62584  105.657958  101.586862  100.415315  100.102044  100.023639  100.005182 | 100.005182 |  0.000114   0.000053    0.00002   0.000006   0.000002        0.0        0.0        0.0 | 4.666348  4.078907  3.834271  3.751553  3.726856  3.720092  3.718366  3.717952 |  100.0 | 141.948195 | 100.005182 |
    | 2000-01-01 15:00:00 |  274.95935 | 172.188724  125.478902  108.065412  102.351411  100.639301  100.163151  100.039254  100.008935 | 100.008935 |  0.000131   0.000066   0.000026   0.000009   0.000003   0.000001        0.0        0.0 | 4.901846  4.198196  3.881074  3.767198  3.731571  3.721405  3.718708  3.718035 |  100.0 |  174.95935 | 100.008935 |
    | 2000-01-01 15:30:00 | 311.845554 | 192.479103  134.240813  111.287675  103.416239   100.96338   100.25498  100.063624   100.01502 |  100.01502 |  0.000146   0.000081   0.000034   0.000012   0.000004   0.000001        0.0        0.0 | 5.164937  4.343518  3.942151  3.788674  3.738318  3.723359  3.719235  3.718169 |  100.0 | 211.845554 |  100.01502 |
    | 2000-01-01 16:00:00 | 352.270855 | 216.458641  145.255127  115.531232  104.874124  101.423603  100.390198  100.100837  100.024655 | 100.024655 |   0.00016   0.000096   0.000043   0.000016   0.000005   0.000002        0.0        0.0 | 5.452178  4.516424  4.020241  3.817612  3.747795  3.726207  3.720034   3.71838 |  100.0 | 252.270855 | 100.024655 |
    | 2000-01-01 16:30:00 | 395.750601 |  244.26564  158.872491  121.035351  106.839104  102.066103  100.585624  100.156526  100.039587 | 100.039587 |  0.000171   0.000112   0.000054   0.000021   0.000007   0.000002   0.000001        0.0 |  5.75917   4.71745  4.118075  3.855917  3.760875  3.730285  3.721217  3.718702 |  100.0 | 295.750601 | 100.039587 |
    | 2000-01-01 17:00:00 | 441.664962 | 275.907668  175.430781  128.071216  109.449491  102.949179  100.863271  100.238324   100.06227 |  100.06227 |  0.000179   0.000127   0.000067   0.000028    0.00001   0.000003   0.000001        0.0 | 6.080855  4.945961  4.238181  3.905752  3.778638  3.736024  3.722941  3.719188 |  100.0 | 341.664962 |  100.06227 |
    | 2000-01-01 17:30:00 | 489.279607 | 311.241523  195.229714  136.937985  112.870689  104.145642   101.25158  100.356425  100.096093 | 100.096093 |  0.000184   0.000141    0.00008   0.000035   0.000013   0.000004   0.000001        0.0 | 6.411818  5.200116  4.382646  3.969487  3.802386  3.743977  3.725408  3.719906 |  100.0 | 389.279607 | 100.096093 |
    | 2000-01-01 18:00:00 |  537.77215 |  349.96279  218.501512   147.95463  117.297407  105.745422  101.786877  100.524262  100.145666 | 100.145666 |  0.000186   0.000154   0.000095   0.000045   0.000017   0.000006   0.000002   0.000001 | 6.746561  5.476936  4.552874  4.049614  3.833666  3.754836  3.728886  3.720951 |  100.0 |  437.77215 | 100.145666 |
    | 2000-01-01 18:30:00 | 586.262607 | 391.606697  245.380362  161.446551  122.954655  107.858336  102.515068  100.759321  100.217175 | 100.217175 |  0.000185   0.000164   0.000109   0.000055   0.000023   0.000008   0.000003   0.000001 | 7.079736  5.772479  4.749366  4.148609  3.874269  3.769458  3.733719  3.722446 |  100.0 | 486.262607 | 100.217175 |
    | 2000-01-01 19:00:00 | 633.846018 | 435.560584  275.874359  177.726492   130.09669   110.61688  103.493589  101.084121  100.318814 | 100.318814 |  0.000181   0.000172   0.000123   0.000067   0.000029   0.000011   0.000004   0.000001 | 7.406317  6.082078  4.971572  4.268754  3.926213  3.788889  3.740345  3.724556 |  100.0 | 533.846018 | 100.318814 |
    | 2000-01-01 19:30:00 | 679.625318 | 481.086906  309.843841  197.070098  139.002931  114.178772  104.793614  101.527381   100.46132 |  100.46132 |  0.000175   0.000177   0.000137    0.00008   0.000036   0.000014   0.000005   0.000002 | 7.721735  6.400614  5.217822  4.411925  3.991696   3.81438  3.749312  3.727493 |  100.0 | 579.625318 |  100.46132 |
    | 2000-01-01 20:00:00 | 722.742733 | 527.354551  346.989543  219.687653  149.969792  118.728817  106.502466  102.125381  100.658609 | 100.658609 |  0.000167   0.000179   0.000149   0.000093   0.000045   0.000018   0.000007   0.000002 | 8.021952  6.722786  5.485372  4.579369  4.073012  3.847406  3.761306  3.731532 |  100.0 | 622.742733 | 100.658609 |
    | 2000-01-01 20:30:00 | 762.408181 | 573.475597  386.852781  245.694679  163.297578  124.479508  108.726163  102.923566  100.928557 | 100.928557 |  0.000156   0.000178   0.000158   0.000107   0.000055   0.000023   0.000009   0.000003 | 8.303511  7.043357  5.770542  4.771504  4.172427  3.889661  3.777163   3.73702 |  100.0 | 662.408181 | 100.928557 |
    | 2000-01-01 21:00:00 | 797.923464 | 618.544348  428.828299  275.085012  179.272062  131.669585  111.591916  103.978367  101.293924 | 101.293924 |  0.000144   0.000174   0.000166    0.00012   0.000066    0.00003   0.000012   0.000004 | 8.563546  7.357356  6.068923  4.987772   4.29201   3.94304  3.797898  3.744396 |  100.0 | 697.923464 | 101.293924 |
    | 2000-01-01 21:30:00 | 828.701418 | 661.675646  472.188683  307.710232    198.1412  140.559638  115.250305  105.359243  101.783451 | 101.783451 |  0.000131   0.000168    0.00017   0.000133   0.000079   0.000037   0.000015   0.000005 | 8.799772  7.660239  6.375632  5.226572  4.433441  4.009591  3.824715  3.754206 |  100.0 | 728.701418 | 101.783451 |
    | 2000-01-01 22:00:00 |  854.27953 | 702.039892  516.117892  343.268879   220.08852  151.423847   119.87673  107.150895  102.433153 | 102.433153 |  0.000117    0.00016   0.000172   0.000144   0.000091   0.000045   0.000019   0.000007 | 9.010461  7.947991  6.685574  5.485296  4.597809  4.091436   3.85902   3.76712 |  100.0 |  754.27953 | 102.433153 |
    | 2000-01-01 22:30:00 | 874.327871 | 738.892845  559.750472  381.307675  245.205792   164.53715  125.671555  109.455545   103.28781 |  103.28781 |  0.000102    0.00015   0.000171   0.000153   0.000104   0.000055   0.000024   0.000009 | 9.194406  8.217193  6.993691  5.760447  4.785422  4.190649  3.902417  3.783955 |  100.0 | 774.327871 |  103.28781 |
    | 2000-01-01 23:00:00 | 888.651531 | 771.598936  602.212752  421.235358  273.468405  180.157616  132.858242   112.39511  104.402654 | 104.402654 |  0.000087   0.000138   0.000167    0.00016   0.000117   0.000066    0.00003   0.000012 | 9.350873  8.465047  7.295165  6.047833  4.995681  4.309105  3.956689  3.805691 |  100.0 | 788.651531 | 104.402654 |
    | 2000-01-01 23:30:00 | 897.187983 | 799.652572   642.65583  462.348836  304.716693  198.504825   141.67905   116.11316  105.845243 | 105.845243 |  0.000071   0.000125   0.000161   0.000164   0.000129   0.000077   0.000037   0.000015 | 9.479547   8.68939   7.58558  6.342805  5.227019  4.448301  4.023744  3.833486 |  100.0 | 797.187983 | 105.845243 |
    | 2000-01-02 00:00:00 |      900.0 | 822.664819  680.321963  503.861271  338.649032  219.734145  152.385206  120.775356  107.697516 | 107.697516 |  0.000056   0.000111   0.000153   0.000165   0.000139   0.000089   0.000045    0.00002 |  9.58051  8.888607  7.861081  6.640498  5.476928  4.609168  4.105543  3.868679 |  100.0 |      800.0 | 107.697516 |
    | 2000-01-02 00:30:00 | 897.264896 | 840.403602  714.517746  544.955105  374.821469  243.914069  165.226813  126.569879  110.057628 | 110.057628 |  0.000041   0.000096   0.000143   0.000164   0.000147   0.000102   0.000055   0.000025 | 9.654161  9.061661  8.118381  6.936087  5.742076  4.791909  4.203984  3.912788 |  100.0 | 797.264896 | 110.057628 |
    | 2000-01-02 01:00:00 | 889.260928 | 852.762318  744.678879  584.808913  412.665273  271.002779  180.434731  133.704835   113.04194 |  113.04194 |  0.000026   0.000081   0.000131    0.00016   0.000154   0.000113   0.000065    0.00003 | 9.701182   9.20801  8.354822  7.224981   6.01848  4.995882   4.32076  3.967482 |  100.0 | 789.260928 |  113.04194 |
    | 2000-01-02 01:30:00 | 876.351729 | 859.759614  770.369855   622.64076  451.511082  300.833147   198.20074   142.40346  116.786393 | 116.786393 |  0.000012   0.000066   0.000119   0.000154   0.000157   0.000124   0.000076   0.000037 | 9.722492  9.327553  8.568372   7.50299  6.301735  5.219546  4.457198   4.03454 |  100.0 | 776.351729 | 116.786393 |
    | 2000-01-02 02:00:00 | 858.969595 | 861.527622  791.285287  657.745314  490.622391  333.106654  218.655189   152.89563  121.446924 | 121.446924 | -0.000002   0.000052   0.000105   0.000146   0.000159   0.000134   0.000087   0.000045 | 9.719209  9.420571  8.757586  7.766443  6.587244  5.460495  4.614089  4.115769 |  100.0 | 758.969595 | 121.446924 |
    | 2000-01-02 02:30:00 |   837.5984 | 858.297729  807.263044  689.502301  529.232787  367.396112   241.84567  165.407551  127.199594 | 127.199594 | -0.000015   0.000037   0.000091   0.000137   0.000157   0.000142   0.000099   0.000054 |  9.69261   9.48766  8.921592  8.012241  6.870449  5.715563  4.791542  4.212902 |  100.0 |   737.5984 | 127.199594 |
    | 2000-01-02 03:00:00 | 812.756821 | 850.382302  818.256811  717.421605  566.583746  403.162814  267.716825  180.144026  134.237066 | 134.237066 | -0.000027   0.000023   0.000077   0.000125   0.000154   0.000147    0.00011   0.000064 | 9.644097  9.529682  9.060001  8.237909  7.147019  5.980989  4.988883  4.327467 |  100.0 | 712.756821 | 134.237066 |
    | 2000-01-02 03:30:00 | 784.982427 | 838.160441   824.33643  741.133855  601.961213  439.779732  296.098592  197.271635  142.764291 | 142.764291 | -0.000038    0.00001   0.000063   0.000113   0.000148   0.000151    0.00012   0.000074 | 9.575169  9.547716  9.172856  8.441581  7.413011  6.252626  5.204611  4.460636 |  100.0 | 684.982427 | 142.764291 |
    | 2000-01-02 04:00:00 | 754.817086 | 822.059896  825.671812  760.396558  634.725125  476.563263  326.701401  216.898521  152.990564 | 152.990564 | -0.000049  -0.000003   0.000049     0.0001    0.00014   0.000152   0.000129   0.000085 | 9.487392  9.543007  9.260566  8.621975  7.664978  6.526159  5.436436  4.613078 |  100.0 | 654.817086 | 152.990564 |
    | 2000-01-02 04:30:00 |  722.79402 | 802.541555  822.513925  775.093599  664.330357  512.808351   359.12134  239.055072  165.118452 | 165.118452 | -0.000058  -0.000014   0.000035   0.000087   0.000131   0.000151   0.000136   0.000095 | 9.382373  9.516932  9.323839  8.778344  7.900035  6.797314  5.681368  4.784831 |  100.0 |  622.79402 | 165.118452 |
    | 2000-01-02 05:00:00 | 689.426711 | 780.083129  815.184102  785.217667  690.345151  547.822812  392.855076  263.677955   179.32976 |  179.32976 | -0.000067  -0.000026   0.000022   0.000073    0.00012   0.000147   0.000141   0.000106 | 9.261744  9.470957  9.363633  8.910412  8.115888  7.062038  5.935882  4.975209 |  100.0 | 589.426711 |  179.32976 |
    | 2000-01-02 05:30:00 | 655.199767 | 755.167014  804.051535  790.865623  712.453974  580.959874  427.322824  290.599443   195.76902 |  195.76902 | -0.000075  -0.000036    0.00001    0.00006   0.000108   0.000141   0.000145   0.000115 | 9.127139  9.406604  9.381097  9.018315  8.310827  7.316645  6.196102  5.182772 |  100.0 | 555.199767 |  195.76902 |
    | 2000-01-02 06:00:00 | 620.561767 | 728.268553  789.518293  792.221926  730.458507   611.64422  461.897332   319.54451  214.525945 | 214.525945 | -0.000082  -0.000045  -0.000002   0.000047   0.000096   0.000134   0.000145   0.000124 | 8.980178  9.325434  9.377519  9.102534  8.483696  7.557922  6.457999  5.405359 |  100.0 | 520.561767 | 214.525945 |
    | 2000-01-02 06:30:00 | 585.920022 | 699.845573  772.007859  789.538654  744.273637  639.391189  495.935491  350.136516  235.618797 | 235.618797 | -0.000088  -0.000054  -0.000013   0.000034   0.000083   0.000125   0.000144    0.00013 | 8.822455  9.229014  9.354286  9.163834  8.633844  7.783182  6.717588  5.640174 |  100.0 | 485.920022 | 235.618797 |
    | 2000-01-02 07:00:00 |  551.63713 | 670.330613  751.951862   783.12031  753.916658  663.819098  528.809571  381.911203  258.980686 | 258.980686 | -0.000093  -0.000061  -0.000023   0.000022   0.000071   0.000115   0.000141   0.000135 | 8.655526  9.118901  9.312857  9.203197  8.761073  7.990297  6.971094  5.883935 |  100.0 |  451.63713 | 258.980686 |
    | 2000-01-02 07:30:00 | 518.029154 | 640.125744  729.775657  773.312906  759.492225  684.654526  559.935484  414.337813   284.45087 |  284.45087 | -0.000097  -0.000068  -0.000032    0.00001   0.000058   0.000104   0.000136   0.000138 | 8.480898  8.996625   9.25472  9.221775  8.865565  8.177688  7.215081  6.133036 |  100.0 | 418.029154 |  284.45087 |
    | 2000-01-02 08:00:00 | 485.365238 | 609.597431  705.892871  760.484926  761.180213  701.730588  588.795844  446.845147  311.772914 | 311.772914 |   -0.0001  -0.000074  -0.000041  -0.000001   0.000046   0.000093   0.000129   0.000139 | 8.300022  8.863678  9.181374  9.220841  8.947827  8.344297  7.446553  6.383737 |  100.0 | 385.365238 | 311.772914 |
    | 2000-01-02 08:30:00 |  453.86844 | 579.074654  680.694641  745.019366  759.218441  714.981005  614.956794  478.849813  340.600225 | 340.600225 | -0.000103  -0.000079  -0.000048  -0.000011   0.000034   0.000081    0.00012   0.000138 | 8.114286  8.721505    9.0943  9.201752  9.008621  8.489538  7.663007  6.632337 |  100.0 |  353.86844 | 340.600225 |
    | 2000-01-02 09:00:00 | 423.717566 | 548.846213  654.549738  727.295787  753.891148  724.429695   638.07803  509.783877  370.508969 | 370.508969 | -0.000105  -0.000083  -0.000055   -0.00002   0.000022   0.000069   0.000111   0.000135 | 7.925009   8.57149  8.994957  9.165906  9.048915  8.613249  7.862455  6.875322 |  100.0 | 323.717566 | 370.508969 |
    | 2000-01-02 09:30:00 | 395.049792 |  519.16199  627.794696  707.689263    745.5114  730.179039  657.916701  539.119753  401.016944 | 401.016944 | -0.000106  -0.000087  -0.000061  -0.000028   0.000012   0.000057   0.000101    0.00013 | 7.733437  8.414953  8.884756   9.11472  9.069824  8.715623  8.043423  7.109494 |  100.0 | 295.049792 | 401.016944 |
    | 2000-01-02 10:00:00 | 367.963873 | 490.233033  600.734523  686.558925  734.410939  732.395885  674.325675  566.390401  431.606709 | 431.606709 | -0.000107   -0.00009  -0.000066  -0.000036   0.000002   0.000045    0.00009   0.000124 | 7.540742  8.253145  8.765058  9.049604  9.072569  8.797154  8.204919  7.332052 |  100.0 | 267.963873 | 431.606709 |
    | 2000-01-02 10:30:00 | 342.523746 | 462.233322  573.640753  664.243483  720.928869  731.297914   687.24744  591.203951  461.750491 | 461.750491 | -0.000107  -0.000092  -0.000071  -0.000043  -0.000008   0.000034   0.000079   0.000116 | 7.348015  8.087247  8.637162  8.971944  9.058437  8.858573  8.346396  7.540653 |  100.0 | 242.523746 | 461.750491 |
    | 2000-01-02 11:00:00 | 318.762374 | 435.301991  546.751233  641.057298  705.402613  727.140374  696.704852  613.252429  490.934583 | 490.934583 | -0.000107  -0.000094  -0.000075  -0.000049  -0.000016   0.000023   0.000067   0.000107 | 7.156271  7.918363  8.502302  8.883085  9.028748  8.900797  8.467695  7.733426 |  100.0 | 218.762374 | 490.934583 |
    | 2000-01-02 11:30:00 |  296.68568 |  409.54409  520.275687  617.284749  688.160978   720.20391  702.789861   632.31476  518.680838 | 518.680838 | -0.000105  -0.000095  -0.000078  -0.000055  -0.000024   0.000013   0.000056   0.000098 | 6.966446  7.747514  8.361652  8.784319  8.984832  8.924879  8.568993  7.908976 |  100.0 |  196.68568 | 518.680838 |
    | 2000-01-02 12:00:00 | 276.276442 | 385.041601  494.375397  593.192853  669.516289  710.783596  705.651418  648.255854  544.566016 | 544.566016 | -0.000104  -0.000095  -0.000081   -0.00006  -0.000032   0.000004   0.000045   0.000087 | 6.779388  7.575666  8.216287  8.676891  8.928001  8.931966  8.650745  8.066352 |  100.0 | 176.276442 | 544.566016 |
    | 2000-01-02 12:30:00 | 257.498082 | 361.843497  469.200695  569.006067  649.764767  699.179748  705.482942  661.019836  568.232264 | 568.232264 | -0.000102  -0.000096  -0.000083  -0.000064  -0.000038  -0.000005   0.000035   0.000077 |  6.59587  7.403698  8.067227  8.561971  8.859539  8.923261  8.713626  8.205018 |  100.0 | 157.498082 | 568.232264 |
    | 2000-01-02 13:00:00 | 240.298231 | 339.978288  444.866616  544.926287  629.179029  685.690364  702.510796  670.622745  589.396626 | 589.396626 |   -0.0001  -0.000095  -0.000084  -0.000067  -0.000044  -0.000013   0.000025   0.000067 | 6.416585  7.232414  7.915418  8.440663  8.780687  8.899997  8.758486    8.3248 |  100.0 | 140.298231 | 589.396626 |
    | 2000-01-02 13:30:00 | 224.612057 | 319.454544  421.462777  521.127642  608.008089  670.604756  696.983467  677.142288  607.853515 | 607.853515 | -0.000097  -0.000094  -0.000085   -0.00007  -0.000049   -0.00002   0.000015   0.000056 | 6.242148  7.062548  7.761736  8.314001  8.692632  8.863409  8.786299  8.425835 |  100.0 | 124.612057 | 607.853515 |
    | 2000-01-02 14:00:00 | 210.365288 | 300.263877  399.055874  497.757892  586.476429  654.198845   689.16206  680.707046  623.473398 | 623.473398 | -0.000094  -0.000093  -0.000086  -0.000073  -0.000053  -0.000027   0.000007   0.000046 | 6.073095  6.894762  7.606983   8.18295  8.596501  8.814713  8.798129  8.508523 |  100.0 | 110.365288 | 623.473398 |
    | 2000-01-02 14:30:00 |  197.47692 | 282.383756  377.692218  474.940139  564.783841  636.731733  679.312202   681.48561  636.198111 | 636.198111 | -0.000091  -0.000092  -0.000086  -0.000075  -0.000057  -0.000033  -0.000002   0.000036 | 5.909889   6.72965  7.451893  8.048405   8.49336  8.755095  8.795089  8.573471 |  100.0 |   97.47692 | 636.198111 |
    | 2000-01-02 15:00:00 | 185.861607 | 265.780112  357.400231  452.774756  543.105881  618.443364  667.697368  679.676213  646.033719 | 646.033719 | -0.000087   -0.00009  -0.000086  -0.000076  -0.000061  -0.000039  -0.000009   0.000027 | 5.752916  6.567744  7.297136   7.91119  8.384206  8.685699  8.778321  8.621448 |  100.0 |  85.861607 | 646.033719 |
    | 2000-01-02 15:30:00 | 175.431719 | 250.409721   338.19287  431.341424  521.594787  599.553129  654.573563  675.497246  653.041829 | 653.041829 | -0.000084  -0.000088  -0.000085  -0.000077  -0.000063  -0.000043  -0.000016   0.000018 | 5.602494   6.40951  7.143316  7.772065   8.26997  8.607615  8.748965  8.653346 |  100.0 |  75.431719 | 653.041829 |
    | 2000-01-02 16:00:00 | 166.099094 | 236.222336  320.069908  410.701229   500.38074   580.25924  640.185263  669.178884  657.330157 | 657.330157 |  -0.00008  -0.000086  -0.000085  -0.000078  -0.000066  -0.000048  -0.000023   0.000009 |  5.45887  6.255355   6.99098  7.631726  8.151519  8.521876  8.708147  8.670136 |  100.0 |  66.099094 | 657.330157 |
    | 2000-01-02 16:30:00 | 157.776495 | 223.162573  303.020063  390.898731  479.573353  560.738725  624.762459  660.955954  659.042987 | 659.042987 | -0.000076  -0.000083  -0.000084  -0.000078  -0.000068  -0.000051  -0.000028   0.000002 | 5.322225  6.105632  6.840613  7.490806  8.029651  8.429457  8.656962  8.672845 |  100.0 |  57.776495 | 659.042987 |
    | 2000-01-02 17:00:00 | 150.378786 | 211.171561  287.022947  371.963979  459.263309  541.147906  608.518661  651.062035  658.352037 | 658.352037 | -0.000072  -0.000081  -0.000082  -0.000078  -0.000069  -0.000055  -0.000034  -0.000006 | 5.192677  5.960639  6.692651   7.34988  7.905104  8.331269  8.596467   8.66252 |  100.0 |  50.378786 | 658.352037 |
    | 2000-01-02 17:30:00 | 143.823868 | 200.188344  272.050836  353.914423  439.524065  521.623237  591.649714  639.724759  655.448069 | 655.448069 | -0.000068  -0.000078  -0.000081  -0.000078   -0.00007  -0.000057  -0.000038  -0.000012 | 5.070283  5.820621  6.547475  7.209467  7.778552  8.228159  8.527665  8.640216 |  100.0 |  43.823868 | 655.448069 |
    | 2000-01-02 18:00:00 | 138.033385 | 190.151076  258.070239  336.756717  420.413575  502.282408  574.333266  627.162201  650.533446 | 650.533446 | -0.000064  -0.000075  -0.000079  -0.000077  -0.000071   -0.00006  -0.000042  -0.000018 | 4.955047  5.685779  6.405421  7.070035  7.650612  8.120916  8.451513  8.606972 |  100.0 |  38.033385 | 650.533446 |
    | 2000-01-02 18:30:00 | 132.933235 | 180.998001  245.043293  320.488369  401.975986   483.22562  556.728765  613.580239  643.815767 | 643.815767 |  -0.00006  -0.000072  -0.000077  -0.000077  -0.000072  -0.000061  -0.000046  -0.000024 | 4.846918  5.556268  6.266779  6.932002  7.521846  8.010264  8.368905  8.563802 |  100.0 |  32.933235 | 643.815767 |
    | 2000-01-02 19:00:00 | 128.453911 | 172.668261  232.928964  305.099264  384.243264  464.536962  538.977862  599.170753  635.502571 | 635.502571 | -0.000056  -0.000069  -0.000075  -0.000076  -0.000072  -0.000063  -0.000049  -0.000029 | 4.745798  5.432201  6.131799  6.795741  7.392764  7.896873  8.280681  8.511681 |  100.0 |  28.453911 | 635.502571 |
    | 2000-01-02 19:30:00 | 124.530701 | 165.102523  221.684092  290.573023  367.236739   446.28584  521.205118  584.110531  625.797085 | 625.797085 | -0.000052  -0.000066  -0.000073  -0.000075  -0.000072  -0.000064  -0.000052  -0.000033 | 4.651546  5.313653   6.00069  6.661581  7.263826  7.781355  8.187623  8.451542 |  100.0 |  24.530701 | 625.797085 |
    | 2000-01-02 20:00:00 | 121.103761 | 158.243474  211.264269  276.888227  350.968549  428.528403  503.518926   568.56074  614.894927 | 614.894927 | -0.000049  -0.000063  -0.000071  -0.000073  -0.000071  -0.000065  -0.000054  -0.000037 | 4.563984  5.200662  5.873628  6.529812  7.135445  7.664269  8.090456  8.384268 |  100.0 |  21.103761 | 614.894927 |
    | 2000-01-02 20:30:00 | 118.118088 | 152.036183  201.624579  264.019489  335.442969  411.308948   486.01258  552.666871  602.981671 | 602.981671 | -0.000045   -0.00006  -0.000068  -0.000072  -0.000071  -0.000066  -0.000056  -0.000041 | 4.482897  5.093233  5.750752  6.400689  7.007995  7.546124  7.989847  8.310693 |  100.0 |  18.118088 | 602.981671 |
    | 2000-01-02 21:00:00 | 115.523427 | 146.428357    192.7202  251.938399  320.657622  394.661253  468.765426  536.559026  590.231145 | 590.231145 | -0.000042  -0.000057  -0.000066   -0.00007   -0.00007  -0.000066  -0.000057  -0.000044 | 4.408045  4.991342  5.632175  6.274431  6.881806  7.427383  7.886413  8.231595 |  100.0 |  15.523427 | 590.231145 |
    | 2000-01-02 21:30:00 | 113.274094 | 141.370508  184.506902  240.614336  306.604577   378.60986  451.844052  520.352477  576.804357 | 576.804357 | -0.000038  -0.000054  -0.000063  -0.000068  -0.000069  -0.000066  -0.000059  -0.000047 | 4.339162  4.894935  5.517979  6.151226  6.757174  7.308463  7.780717    8.1477 |  100.0 |  13.274094 | 576.804357 |
    | 2000-01-02 22:00:00 | 111.328777 | 136.816041  176.941433  230.015164  293.271329  363.171247  435.303484  504.148419   562.84893 |  562.84893 | -0.000035  -0.000051  -0.000061  -0.000067  -0.000068  -0.000066   -0.00006  -0.000049 | 4.275966  4.803934   5.40822  6.031233  6.634359  7.189742  7.673275  8.059683 |  100.0 |  11.328777 |  562.84893 |
    | 2000-01-02 22:30:00 | 109.650285 | 132.721286  169.981821  220.107826  280.641668  348.354934  419.188352  488.034846  548.498951 | 548.498951 | -0.000032  -0.000048  -0.000058  -0.000065  -0.000067  -0.000066   -0.00006  -0.000051 | 4.218159  4.718237  5.302932  5.914587  6.513592  7.071558  7.564557  7.968167 |  100.0 |   9.650285 | 548.498951 |
    | 2000-01-02 23:00:00 | 108.205287 | 129.045478  163.587601  210.858837  268.696446  334.164476  403.534013   472.08751  533.875143 | 533.875143 | -0.000029  -0.000045  -0.000056  -0.000063  -0.000066  -0.000065  -0.000061  -0.000052 | 4.165434  4.637725  5.202124  5.801396  6.395073  6.954216   7.45499  7.873724 |  100.0 |   8.205287 | 533.875143 |
    | 2000-01-02 23:30:00 | 106.964036 | 125.750706  157.719976  202.234693  257.414243  320.598372  388.367613  456.370911  519.085277 | 519.085277 | -0.000027  -0.000042  -0.000054  -0.000061  -0.000064  -0.000065  -0.000061  -0.000054 |  4.11748  4.562256  5.105788  5.691746  6.278977  6.837986  7.344962  7.776881 |  100.0 |   6.964036 | 519.085277 |
    | 2000-01-03 00:00:00 | 105.900084 | 122.801821  152.341929  194.202206  246.771943  307.650879  373.709086  440.939305   504.22476 |  504.22476 | -0.000024  -0.000039  -0.000051  -0.000059  -0.000063  -0.000064  -0.000061  -0.000055 | 4.073986  4.491677  5.013898  5.585704  6.165453   6.72311  7.234822  7.678121 |  100.0 |   5.900084 |  504.22476 |
    | 2000-01-03 00:30:00 |  104.99001 | 120.166338   147.41828  186.728769  236.745232  295.312738  359.572065   425.83768  489.377348 | 489.377348 | -0.000022  -0.000037  -0.000049  -0.000057  -0.000062  -0.000063  -0.000061  -0.000056 |  4.03464   4.42582  4.926408  5.483318  6.054629  6.609801  7.124887  7.577883 |  100.0 |    4.99001 | 489.377348 |
    | 2000-01-03 01:00:00 | 104.213141 | 117.814315  142.915718   179.78257   227.30902  283.571819  345.964738  411.102711  474.615935 | 474.615935 |  -0.00002  -0.000034  -0.000046  -0.000055   -0.00006  -0.000062  -0.000061  -0.000056 | 3.999138  4.364505  4.843261  5.384618  5.946613  6.498247  7.015439  7.476568 |  100.0 |   4.213141 | 474.615935 |
    | 2000-01-03 01:30:00 | 103.551298 |  115.71822  138.802803   173.33275  218.437793  272.413683  332.890607  396.763658  460.003377 | 460.003377 | -0.000018  -0.000032  -0.000044  -0.000053  -0.000058  -0.000061   -0.00006  -0.000057 | 3.967185  4.307545  4.764383   5.28962  5.841495  6.388613  6.906733  7.374541 |  100.0 |   3.551298 | 460.003377 |
    | 2000-01-03 02:00:00 | 102.988541 | 113.852802  135.049937  167.349523  210.105909  261.822081  320.349185  382.843217  445.593338 | 445.593338 | -0.000016  -0.000029  -0.000041  -0.000051  -0.000057   -0.00006   -0.00006  -0.000057 | 3.938495  4.254746   4.68969  5.198325  5.739345  6.281043  6.798996   7.27213 |  100.0 |   2.988541 | 445.593338 |
    | 2000-01-03 02:30:00 |  102.51094 | 112.194953  131.629326  161.804262  202.287846  251.779376  308.336617  369.358306  431.431123 | 431.431123 | -0.000014  -0.000027  -0.000039  -0.000049  -0.000055  -0.000059  -0.000059  -0.000057 | 3.912793  4.205909  4.619087  5.110721  5.640222  6.175662   6.69243  7.169635 |  100.0 |    2.51094 | 431.431123 |
    | 2000-01-03 03:00:00 | 102.106358 | 110.723572  128.514927  156.669553  194.958397   242.26692  296.846239   356.32079  417.554487 | 417.554487 | -0.000013  -0.000025  -0.000037  -0.000047  -0.000053  -0.000057  -0.000058  -0.000057 | 3.889819  4.160831  4.552469  5.026784  5.544166  6.072576  6.587214  7.067324 |  100.0 |   2.106358 | 417.554487 |
    | 2000-01-03 03:30:00 | 101.764251 | 109.419427  125.682386  151.919224  188.092829  233.265365  285.869062  343.738145  403.994408 | 403.994408 | -0.000011  -0.000023  -0.000035  -0.000045  -0.000052  -0.000056  -0.000058  -0.000057 | 3.869328  4.119309  4.489721   4.94648  5.451209  5.971875  6.483508  6.965439 |  100.0 |   1.764251 | 403.994408 |
    | 2000-01-03 04:00:00 |  101.47548 | 108.265033  123.108966  147.528362   181.66701   224.75493  275.394211   331.61406  390.775817 | 390.775817 |  -0.00001  -0.000021  -0.000033  -0.000043   -0.00005  -0.000055  -0.000057  -0.000056 | 3.851088   4.08114  4.430723  4.869766  5.361369  5.873636  6.381451  6.864198 |  100.0 |    1.47548 | 390.775817 |
    | 2000-01-03 04:30:00 | 101.232151 | 107.244518  120.773477  143.473301  175.657504  216.715631  265.409301  319.948982  377.918284 | 377.918284 | -0.000009  -0.000019  -0.000031  -0.000041  -0.000048  -0.000053  -0.000056  -0.000056 | 3.834884  4.046124  4.375349  4.796587  5.274652   5.77792  6.281165  6.763794 |  100.0 |   1.232151 | 377.918284 |
    | 2000-01-03 05:00:00 | 101.027458 | 106.343504  118.656199  139.731613   170.04164  209.127467  255.900767  308.740601  365.436649 | 365.436649 | -0.000008  -0.000018  -0.000029  -0.000039  -0.000046  -0.000052  -0.000055  -0.000055 | 3.820515  4.014063  4.323465  4.726884  5.191056  5.684779  6.182756  6.664401 |  100.0 |   1.027458 | 365.436649 |
    | 2000-01-03 05:30:00 | 100.855551 | 105.548994  116.738807  136.282077  164.797566  201.970575  246.854152  297.984284  353.341602 | 353.341602 | -0.000007  -0.000016  -0.000027  -0.000037  -0.000045   -0.00005  -0.000054  -0.000055 | 3.807797  3.984763  4.274938  4.660587  5.110571  5.594252  6.086316  6.566173 |  100.0 |   0.855551 | 353.341602 |
    | 2000-01-03 06:00:00 | 100.711411 | 104.849257  115.004299  133.104649  159.904278  195.225356  238.254355  287.673468  341.640216 | 341.640216 | -0.000006  -0.000015  -0.000025  -0.000035  -0.000043  -0.000049  -0.000052  -0.000054 | 3.796559  3.958036  4.229626  4.597623  5.033177  5.506366  5.991924  6.469248 |  100.0 |   0.711411 | 341.640216 |
    | 2000-01-03 06:30:00 | 100.590745 | 104.233731  113.436916  130.180423  155.341641  188.872579  230.085834   277.79999  330.336425 | 330.336425 | -0.000006  -0.000014  -0.000023  -0.000033  -0.000041  -0.000047  -0.000051  -0.000053 | 3.786646  3.933701   4.18739  4.537911  4.958848  5.421144  5.899646  6.373745 |  100.0 |   0.590745 | 330.336425 |
    | 2000-01-03 07:00:00 | 100.489885 | 103.692916  112.022075  127.491587  151.090397  182.893461  222.332793  268.354398  319.431465 | 319.431465 | -0.000005  -0.000012  -0.000022  -0.000031   -0.00004  -0.000046   -0.00005  -0.000052 | 3.777914  3.911582  4.148086  4.481364  4.887549  5.338595  5.809537  6.279771 |  100.0 |   0.489885 | 319.431465 |
    | 2000-01-03 07:30:00 | 100.405708 | 103.218291  110.746299  125.021384  147.132158  177.269727  214.979328  259.326203  308.924259 | 308.924259 | -0.000004  -0.000011   -0.00002   -0.00003  -0.000038  -0.000044  -0.000049  -0.000051 | 3.770235  3.891513  4.111573  4.427894  4.819242  5.258723  5.721644  6.187419 |  100.0 |   0.405708 | 308.924259 |
    | 2000-01-03 08:00:00 |  100.33556 | 102.802221  109.597146   122.75406  143.449399  171.983659  208.009557  250.704116  298.811766 | 298.811766 | -0.000004   -0.00001  -0.000019  -0.000028  -0.000036  -0.000043  -0.000048   -0.00005 | 3.763491  3.873335  4.077709  4.377406  4.753882  5.181526  5.636003   6.09677 |  100.0 |    0.33556 | 298.811766 |
    | 2000-01-03 08:30:00 | 100.277186 |  102.43788   108.56315  120.674827  140.025441  167.018129  201.407721  242.476242  289.089295 | 289.089295 | -0.000003  -0.000009  -0.000017  -0.000026  -0.000035  -0.000041  -0.000046  -0.000049 | 3.757576  3.856894  4.046352  4.329803  4.691417  5.106994  5.552643  6.007893 |  100.0 |   0.277186 | 289.089295 |
    | 2000-01-03 09:00:00 | 100.228681 | 102.119175  107.633752   118.76981   136.84443  162.356617  195.158271  234.630247  279.750779 | 279.750779 | -0.000003  -0.000008  -0.000016  -0.000025  -0.000033   -0.00004  -0.000045  -0.000048 | 3.752396   3.84205  4.017364  4.284985  4.631794  5.035112  5.471584  5.920848 |  100.0 |   0.228681 | 279.750779 |
    | 2000-01-03 09:30:00 | 100.188431 | 101.840678  106.799249  117.026008  133.891315  157.983229  189.245938  227.153512  270.789014 | 270.789014 | -0.000003  -0.000007  -0.000015  -0.000023  -0.000032  -0.000038  -0.000044  -0.000047 | 3.747864  3.828667  3.990608  4.242851  4.574952  4.965858   5.39284  5.835686 |  100.0 |   0.188431 | 270.789014 |
    | 2000-01-03 10:00:00 | 100.155078 | 101.597558   106.05073  115.431248  131.151818    153.8827  183.655787   220.03325   262.19588 |  262.19588 | -0.000002  -0.000007  -0.000014  -0.000022   -0.00003  -0.000037  -0.000042  -0.000046 | 3.743904  3.816619   3.96595  4.203296   4.52083  4.899207  5.316419  5.752448 |  100.0 |   0.155078 |  262.19588 |
    | 2000-01-03 10:30:00 | 100.127477 | 101.385528  105.380027  113.974142   128.61241  150.040394  178.373261  213.256617  253.962519 | 253.962519 | -0.000002  -0.000006  -0.000013  -0.000021  -0.000029  -0.000036  -0.000041  -0.000045 | 3.740448  3.805789  3.943262  4.166216  4.469361   4.83513  5.242324  5.671169 |  100.0 |   0.127477 | 253.962519 |
    | 2000-01-03 11:00:00 | 100.104666 | 101.200785  104.779663  112.644048  126.260282  146.442294  173.384214  206.810795  246.079506 | 246.079506 | -0.000002  -0.000005  -0.000012  -0.000019  -0.000027  -0.000034   -0.00004  -0.000044 | 3.737436  3.796067  3.922417  4.131505  4.420475  4.773591   5.17055  5.591878 |  100.0 |   0.104666 | 246.079506 |
    | 2000-01-03 11:30:00 | 100.085838 | 101.039964    104.2428  111.431032  124.083315     143.075   168.67493  200.683072  238.536984 | 238.536984 | -0.000001  -0.000005  -0.000011  -0.000018  -0.000026  -0.000033  -0.000039  -0.000043 | 3.734812  3.787351  3.903294   4.09906  4.374102  4.714553  5.101091  5.514595 |  100.0 |   0.085838 | 238.536984 |
    | 2000-01-03 12:00:00 | 100.070317 | 100.900088  103.763199  110.325825  122.070048  139.925706  164.232145  194.860894  231.324789 | 231.324789 | -0.000001  -0.000004   -0.00001  -0.000017  -0.000024  -0.000031  -0.000037  -0.000042 | 3.732529  3.779548  3.885776  4.068773  4.330165  4.657974  5.033934  5.439336 |  100.0 |   0.070317 | 231.324789 |
    | 2000-01-03 12:30:00 | 100.057539 | 100.778533  103.335169  109.319792  120.209656  136.982194  160.043055  189.331925  224.432555 | 224.432555 | -0.000001  -0.000004  -0.000009  -0.000016  -0.000023   -0.00003  -0.000036  -0.000041 | 3.730544   3.77257  3.869751  4.040542   4.28859  4.603809  4.969063  5.366111 |  100.0 |   0.057539 | 224.432555 |
    | 2000-01-03 13:00:00 | 100.047032 | 100.672986  102.953532  108.404896  118.491919  134.232811  156.095319  184.084077  217.849803 | 217.849803 | -0.000001  -0.000003  -0.000008  -0.000015  -0.000022  -0.000029  -0.000035   -0.00004 | 3.728821  3.766338  3.855112  4.014263  4.249299  4.552012  4.906458  5.294927 |  100.0 |   0.047032 | 217.849803 |
    | 2000-01-03 13:30:00 | 100.038402 | 100.581408   102.61358  107.573661  116.907191  131.666455  152.377058  179.105548  211.566016 | 211.566016 | -0.000001  -0.000003  -0.000007  -0.000014  -0.000021  -0.000027  -0.000034  -0.000038 | 3.727325  3.760778  3.841758  3.989836  4.212213  4.502531  4.846095  5.225784 |  100.0 |   0.038402 | 211.566016 |
    | 2000-01-03 14:00:00 | 100.031322 | 100.502012   102.31104  106.819145  115.446381   129.27255  148.876859  174.384841  205.570705 | 205.570705 | -0.000001  -0.000003  -0.000007  -0.000013  -0.000019  -0.000026  -0.000032  -0.000037 | 3.726028  3.755824  3.829591  3.967159  4.177251  4.455315  4.787948   5.15868 |  100.0 |   0.031322 | 205.570705 |
    | 2000-01-03 14:30:00 | 100.025521 | 100.433227  102.042039  106.134907   114.10092  127.041035  145.583758  169.910785  199.853462 | 199.853462 | -0.000001  -0.000002  -0.000006  -0.000012  -0.000018  -0.000025  -0.000031  -0.000036 | 3.724905  3.751414  3.818521  3.946137  4.144334  4.410308  4.731988  5.093608 |  100.0 |   0.025521 | 199.853462 |
    | 2000-01-03 15:00:00 | 100.020773 | 100.373675  101.803072  105.514978  112.862742  124.962338  142.487244  165.672546  194.404004 | 194.404004 | -0.000001  -0.000002  -0.000006  -0.000011  -0.000017  -0.000024   -0.00003  -0.000035 | 3.723932  3.747493  3.808462  3.926674   4.11338  4.367455  4.678182  5.030559 |  100.0 |   0.020773 | 194.404004 |
    | 2000-01-03 15:30:00 | 100.016892 | 100.322152  101.590969  104.953831  111.724258   123.02736  139.577237  161.659634  189.212212 | 189.212212 |       0.0  -0.000002  -0.000005   -0.00001  -0.000016  -0.000023  -0.000029  -0.000034 |  3.72309   3.74401  3.799332  3.908679  4.084309  4.326696  4.626497   4.96952 |  100.0 |   0.016892 | 189.212212 |
    | 2000-01-03 16:00:00 | 100.013722 | 100.277603   101.40287   104.44636  110.678334  121.227455  136.844085  157.861912  184.268155 | 184.268155 |       0.0  -0.000002  -0.000005  -0.000009  -0.000015  -0.000022  -0.000028  -0.000033 | 3.722363   3.74092  3.791055  3.892061   4.05704  4.287973  4.576895  4.910475 |  100.0 |   0.013722 | 184.268155 |
    | 2000-01-03 16:30:00 | 100.011136 | 100.239109  101.236197  103.987846  109.718267  119.554413  134.278551  154.269588  179.562122 | 179.562122 |       0.0  -0.000002  -0.000004  -0.000009  -0.000014   -0.00002  -0.000026  -0.000032 | 3.721734  3.738179  3.783561  3.876736  4.031493  4.251224  4.529337  4.853405 |  100.0 |   0.011136 | 179.562122 |
    | 2000-01-03 17:00:00 | 100.009028 | 100.205864  101.088628  103.573939  108.837768  118.000441  131.871796  150.873225  175.084631 | 175.084631 |       0.0  -0.000001  -0.000004  -0.000008  -0.000013  -0.000019  -0.000025  -0.000031 | 3.721191  3.735752  3.776783   3.86262  4.007588  4.216387  4.483784  4.798291 |  100.0 |   0.009028 | 175.084631 |
    | 2000-01-03 17:30:00 | 100.007313 |  100.17717  100.958077  103.200632   108.03094  116.558144  129.615369  147.663726  170.826451 | 170.826451 |       0.0  -0.000001  -0.000003  -0.000007  -0.000012  -0.000018  -0.000024   -0.00003 | 3.720723  3.733604   3.77066  3.849633  3.985248    4.1834  4.440192  4.745109 |  100.0 |   0.007313 | 170.826451 |
    | 2000-01-03 18:00:00 | 100.005918 | 100.152417  100.842671  102.864238  107.292256  115.220512  127.501193  144.632338  166.778605 | 166.778605 |       0.0  -0.000001  -0.000003  -0.000007  -0.000012  -0.000017  -0.000023  -0.000028 | 3.720319  3.731704  3.765134  3.837701  3.964395  4.152199  4.398517  4.693833 |  100.0 |   0.005918 | 166.778605 |
    | 2000-01-03 18:30:00 | 100.004785 | 100.131074  100.740728   102.56137  106.616546    113.9809  125.521552  141.770639  162.932383 | 162.932383 |       0.0  -0.000001  -0.000003  -0.000006  -0.000011  -0.000016  -0.000022  -0.000027 |  3.71997  3.730026  3.760152   3.82675  3.944953   4.12272  4.358714  4.644438 |  100.0 |   0.004785 | 162.932383 |
    | 2000-01-03 19:00:00 | 100.003865 | 100.112681  100.650743   102.28892  105.998973  112.833017  123.669078  139.070535   159.27934 |  159.27934 |       0.0  -0.000001  -0.000002  -0.000006   -0.00001  -0.000015  -0.000021  -0.000026 |  3.71967  3.728544  3.755665  3.816711  3.926849    4.0949  4.320736  4.596893 |  100.0 |   0.003865 |  159.27934 |
    | 2000-01-03 19:30:00 | 100.003119 | 100.096836   100.57137   102.04404  105.435022  111.770905  121.936737  136.524249  155.811303 | 155.811303 |       0.0  -0.000001  -0.000002  -0.000005  -0.000009  -0.000015   -0.00002  -0.000025 | 3.719412  3.727237  3.751629  3.807518  3.910011  4.068674  4.284535  4.551168 |  100.0 |   0.003119 | 155.811303 |
    | 2000-01-03 20:00:00 | 100.002515 | 100.083194  100.501406  101.824122  104.920479  110.788932  120.317816  134.124313  152.520366 | 152.520366 |       0.0  -0.000001  -0.000002  -0.000005  -0.000009  -0.000014  -0.000019  -0.000024 | 3.719189  3.726085  3.748001  3.799111  3.894368  4.043978  4.250062  4.507231 |  100.0 |   0.002515 | 152.520366 |
    | 2000-01-03 20:30:00 | 100.002026 | 100.071452  100.439777  101.626783  104.451415  109.881771  118.805915  131.863558   149.39889 |  149.39889 |       0.0  -0.000001  -0.000002  -0.000004  -0.000008  -0.000013  -0.000018  -0.000023 | 3.718997   3.72507  3.744743  3.791429  3.879853   4.02075  4.217268  4.465049 |  100.0 |   0.002026 |  149.39889 |
    | 2000-01-03 21:00:00 | 100.001631 |  100.06135  100.385525  101.449846  104.024174  109.044392  117.394928  129.735105  146.439501 | 146.439501 |       0.0        0.0  -0.000002  -0.000004  -0.000007  -0.000012  -0.000017  -0.000022 | 3.718833  3.724176  3.741819  3.784419  3.866398  3.998926  4.186102  4.424586 |  100.0 |   0.001631 | 146.439501 |
    | 2000-01-03 21:30:00 | 100.001312 | 100.052662  100.337797  101.291327  103.635353  108.272049  116.079039  127.732357  143.635083 | 143.635083 |       0.0        0.0  -0.000001  -0.000004  -0.000007  -0.000011  -0.000016  -0.000022 | 3.718691   3.72339  3.739199  3.778028  3.853942  3.978443  4.156513  4.385807 |  100.0 |   0.001312 | 143.635083 |
    | 2000-01-03 22:00:00 | 100.001054 | 100.045193  100.295834  101.149418  103.281795  107.560264  114.852705  125.848989  140.978776 | 140.978776 |       0.0        0.0  -0.000001  -0.000003  -0.000006  -0.000011  -0.000016  -0.000021 | 3.718569  3.722699  3.736853  3.772207  3.842422  3.959242   4.12845  4.348674 |  100.0 |   0.001054 | 140.978776 |
    | 2000-01-03 22:30:00 | 100.000846 | 100.038774  100.258962  101.022475  102.960567  106.904817  113.710649  124.078934  138.463965 | 138.463965 |       0.0        0.0  -0.000001  -0.000003  -0.000006   -0.00001  -0.000015   -0.00002 | 3.718464  3.722092  3.734753  3.766911   3.83178   3.94126  4.101861  4.313149 |  100.0 |   0.000846 | 138.463965 |
    | 2000-01-03 23:00:00 | 100.000679 | 100.033259  100.226582  100.909004  102.668952  106.301739   112.64785  122.416383  136.084282 | 136.084282 |       0.0        0.0  -0.000001  -0.000003  -0.000005  -0.000009  -0.000014  -0.000019 | 3.718375  3.721559  3.732875  3.762097   3.82196  3.924439  4.076693  4.279192 |  100.0 |   0.000679 | 136.084282 |
    | 2000-01-03 23:30:00 | 100.000544 | 100.028523  100.198163   100.80765  102.404436  105.747292   111.65953  120.855767  133.833591 | 133.833591 |       0.0        0.0  -0.000001  -0.000002  -0.000005  -0.000009  -0.000013  -0.000018 | 3.718297  3.721091  3.731198  3.757725  3.812907  3.908722  4.052896  4.246764 |  100.0 |   0.000544 | 133.833591 |
    | 2000-01-04 00:00:00 | 100.000436 | 100.024456  100.173234  100.717182  102.164692  105.237967  110.741149  119.391755  131.705985 | 131.705985 |       0.0        0.0  -0.000001  -0.000002  -0.000005  -0.000008  -0.000012  -0.000017 | 3.718231  3.720681    3.7297  3.753759  3.804571   3.89405  4.030418  4.215825 |  100.0 |   0.000436 | 131.705985 |
    | 2000-01-04 00:30:00 | 100.000349 | 100.020965  100.151377  100.636488  101.947572  104.770468  109.888392  118.019242  129.695781 | 129.695781 |       0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000008  -0.000012  -0.000016 | 3.718174  3.720321  3.728364  3.750164  3.796902   3.88037  4.009206  4.186332 |  100.0 |   0.000349 | 129.695781 |
    | 2000-01-04 01:00:00 | 100.000279 | 100.017969  100.132224   100.56456  101.751096  104.341706  109.097163  116.733342  127.797509 | 127.797509 |       0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000007  -0.000011  -0.000016 | 3.718125  3.720006  3.727173  3.746909  3.789855  3.867628   3.98921  4.158243 |  100.0 |   0.000279 | 127.797509 |
    | 2000-01-04 01:30:00 | 100.000223 | 100.015399  100.115449   100.50049  101.573438  103.948784  108.363577   115.52938  126.005909 | 126.005909 |       0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000007   -0.00001  -0.000015 | 3.718084   3.71973  3.726113  3.743964  3.783385  3.855771   3.97038  4.131518 |  100.0 |   0.000223 | 126.005909 |
    | 2000-01-04 02:00:00 | 100.000178 | 100.013194  100.100764  100.443456  101.412917  103.588993  107.683949  114.402886  124.315921 | 124.315921 |       0.0        0.0  -0.000001  -0.000001  -0.000003  -0.000006   -0.00001  -0.000014 | 3.718048  3.719488  3.725168  3.741301   3.77745  3.844749  3.952664  4.106113 |  100.0 |   0.000178 | 124.315921 |
    | 2000-01-04 02:30:00 | 100.000142 | 100.011303  100.087914  100.392718   101.26799  103.259801  107.054788  113.349586  122.722681 | 122.722681 |       0.0        0.0        0.0  -0.000001  -0.000003  -0.000006  -0.000009  -0.000013 | 3.718017  3.719277  3.724328  3.738895  3.772013  3.834515  3.936015  4.081985 |  100.0 |   0.000142 | 122.722681 |
    | 2000-01-04 03:00:00 | 100.000113 | 100.009682  100.076674  100.347609  101.137237   102.95884   106.47279  112.365396  121.221513 | 121.221513 |       0.0        0.0        0.0  -0.000001  -0.000003  -0.000005  -0.000009  -0.000013 |  3.71799  3.719092  3.723581  3.736725  3.767035  3.825021  3.920383  4.059091 |  100.0 |   0.000113 | 121.221513 |
    | 2000-01-04 03:30:00 |  100.00009 | 100.008292  100.066849  100.307529  101.019358  102.683904  105.934832  111.446416  119.807924 | 119.807924 |       0.0        0.0        0.0  -0.000001  -0.000003  -0.000005  -0.000008  -0.000012 | 3.717968  3.718931  3.722918  3.734767  3.762482  3.816222  3.905721   4.03739 |  100.0 |    0.00009 | 119.807924 |
    | 2000-01-04 04:00:00 | 100.000072 | 100.007101  100.058262  100.271938  100.913162  102.432937   105.43796  110.588923  118.477595 | 118.477595 |       0.0        0.0        0.0  -0.000001  -0.000002  -0.000005  -0.000008  -0.000011 | 3.717948   3.71879  3.722329  3.733003  3.758322  3.808077  3.891982  4.016837 |  100.0 |   0.000072 | 118.477595 |
    | 2000-01-04 04:30:00 | 100.000057 |  100.00608  100.050762  100.240352  100.817557  102.204024  104.979386  109.789364   117.22638 |  117.22638 |       0.0        0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000007  -0.000011 | 3.717932  3.718666  3.721806  3.731414  3.754524  3.800542  3.879121  3.997391 |  100.0 |   0.000057 |  117.22638 |
    | 2000-01-04 05:00:00 | 100.000046 | 100.005206  100.044213  100.212337  100.731546  101.995385  104.556481  109.044354  116.050297 | 116.050297 |       0.0        0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000007   -0.00001 | 3.717917  3.718559  3.721342  3.729985  3.751059   3.79358  3.867094  3.979009 |  100.0 |   0.000046 | 116.050297 |
    | 2000-01-04 05:30:00 | 100.000036 | 100.004456  100.038496  100.187502   100.65422  101.805366  104.166766  108.350666  114.945523 | 114.945523 |       0.0        0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000006   -0.00001 | 3.717905  3.718465  3.720931  3.728699  3.747901  3.787152  3.855856  3.961648 |  100.0 |   0.000036 | 114.945523 |
    | 2000-01-04 06:00:00 | 100.000029 | 100.003815  100.033509  100.165499  100.584747  101.632435  103.807909  107.705226  113.908391 | 113.908391 |       0.0        0.0        0.0  -0.000001  -0.000002  -0.000003  -0.000006  -0.000009 | 3.717895  3.718383  3.720567  3.727545  3.745026  3.781224  3.845368  3.945269 |  100.0 |   0.000029 | 113.908391 |
    | 2000-01-04 06:30:00 | 100.000023 | 100.003265  100.029159  100.146014  100.522371  101.475169  103.477716   107.10511  112.935381 | 112.935381 |       0.0        0.0        0.0  -0.000001  -0.000001  -0.000003  -0.000005  -0.000009 | 3.717886  3.718311  3.720245  3.726508  3.742409  3.775761  3.835587  3.929829 |  100.0 |   0.000023 | 112.935381 |
    | 2000-01-04 07:00:00 | 100.000018 | 100.002794  100.025367  100.128769  100.466403  101.332254  103.174125  106.547539  112.023122 | 112.023122 |       0.0        0.0        0.0  -0.000001  -0.000001  -0.000003  -0.000005  -0.000008 | 3.717878  3.718249   3.71996  3.725578   3.74003  3.770731  3.826475  3.915288 |  100.0 |   0.000018 | 112.023122 |
    | 2000-01-04 07:30:00 | 100.000014 | 100.002391  100.022061  100.113513  100.416215  101.202471  102.895201  106.029872  111.168379 | 111.168379 |       0.0        0.0        0.0        0.0  -0.000001  -0.000003  -0.000005  -0.000008 | 3.717872  3.718195  3.719707  3.724743  3.737869  3.766104  3.817993  3.901607 |  100.0 |   0.000014 | 111.168379 |
    | 2000-01-04 08:00:00 | 100.000011 | 100.002046  100.019182  100.100024  100.371239  101.084697  102.639129    105.5496  110.368056 | 110.368056 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000007 | 3.717866  3.718147  3.719484  3.723996  3.735907  3.761852  3.810106  3.888746 |  100.0 |   0.000011 | 110.368056 |
    | 2000-01-04 08:30:00 | 100.000009 | 100.001751  100.016673  100.088104  100.330958  100.977896  102.404211  105.104346  109.619187 | 109.619187 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000007 | 3.717861  3.718106  3.719287  3.723326  3.734128  3.757947  3.802779  3.876668 |  100.0 |   0.000009 | 109.619187 |
    | 2000-01-04 09:00:00 | 100.000007 | 100.001498  100.014489  100.077573  100.294904   100.88111  102.188856  104.691855  108.918936 | 108.918936 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000006 | 3.717857   3.71807  3.719113  3.722727  3.732515  3.754364  3.795978  3.865335 |  100.0 |   0.000007 | 108.918936 |
    | 2000-01-04 09:30:00 | 100.000006 | 100.001282  100.012588  100.068276   100.26265   100.79346  101.991578  104.309994  108.264589 | 108.264589 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000004  -0.000006 | 3.717854  3.718039   3.71896  3.722191  3.731054   3.75108   3.78967   3.85471 |  100.0 |   0.000006 | 108.264589 |
    | 2000-01-04 10:00:00 | 100.000004 | 100.001096  100.010934  100.060071  100.233814  100.714136  101.810989  103.956743  107.653553 | 107.653553 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000003  -0.000006 | 3.717851  3.718012  3.718824  3.721712  3.729731  3.748071  3.783826   3.84476 |  100.0 |   0.000004 | 107.653553 |
    | 2000-01-04 10:30:00 | 100.000004 | 100.000938  100.009495  100.052832  100.208048  100.642394  101.645795  103.630195  107.083352 | 107.083352 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000003  -0.000005 | 3.717848  3.717988  3.718704  3.721284  3.728535  3.745317  3.778415  3.835448 |  100.0 |   0.000004 | 107.083352 |
    | 2000-01-04 11:00:00 | 100.000003 | 100.000802  100.008243   100.04645  100.185036  100.577551  101.494788  103.328547  106.551622 | 106.551622 |       0.0        0.0        0.0        0.0  -0.000001  -0.000001  -0.000003  -0.000005 | 3.717846  3.717968  3.718599  3.720902  3.727454  3.742799   3.77341  3.826742 |  100.0 |   0.000003 | 106.551622 |
    | 2000-01-04 11:30:00 | 100.000002 | 100.000686  100.007155  100.040824  100.164497   100.51898  101.356848    103.0501  106.056109 | 106.056109 |       0.0        0.0        0.0        0.0  -0.000001  -0.000001  -0.000003  -0.000005 | 3.717844   3.71795  3.718506  3.720561  3.726477  3.740497  3.768784  3.818609 |  100.0 |   0.000002 | 106.056109 |
    | 2000-01-04 12:00:00 | 100.000002 | 100.000587  100.006209  100.035867  100.146173  100.466109  101.230929  102.793252  105.594666 | 105.594666 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000004 | 3.717842  3.717935  3.718424  3.720256  3.725596  3.738395  3.764512  3.811019 |  100.0 |   0.000002 | 105.594666 |
    | 2000-01-04 12:30:00 | 100.000001 | 100.000502  100.005387  100.031502  100.129835   100.41841  101.116062  102.556495  105.165246 | 105.165246 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000004 | 3.717841  3.717921  3.718352  3.719985  3.724801  3.736477  3.760571  3.803941 |  100.0 |   0.000001 | 105.165246 |
    | 2000-01-04 13:00:00 | 100.000001 | 100.000429  100.004673  100.027659  100.115275  100.375405  101.011348  102.338412  104.765905 | 104.765905 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000004 |  3.71784  3.717909  3.718289  3.719743  3.724084  3.734727  3.756937  3.797347 |  100.0 |   0.000001 | 104.765905 |
    | 2000-01-04 13:30:00 | 100.000001 | 100.000367  100.004052  100.024277  100.102305  100.336655  100.915951  102.137669  104.394791 | 104.394791 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000003 | 3.717839  3.717899  3.718233  3.719528  3.723438  3.733134  3.753589  3.791208 |  100.0 |   0.000001 | 104.394791 |
    | 2000-01-04 14:00:00 | 100.000001 | 100.000314  100.003514  100.021302  100.090757  100.301758    100.8291  101.953016  104.050148 | 104.050148 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000003 | 3.717838   3.71789  3.718184  3.719337  3.722856  3.731682  3.750508  3.785497 |  100.0 |   0.000001 | 104.050148 |
    | 2000-01-04 14:30:00 | 100.000001 | 100.000269  100.003046  100.018686  100.080481  100.270349  100.750081  101.783279  103.730309 | 103.730309 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000003 | 3.717837  3.717883  3.718141  3.719166  3.722332  3.730362  3.747673   3.78019 |  100.0 |   0.000001 | 103.730309 |
    | 2000-01-04 15:00:00 |      100.0 |  100.00023   100.00264  100.016386  100.071341  100.242097  100.678232   101.62736  103.433692 | 103.433692 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000001  -0.000003 | 3.717837  3.717876  3.718103  3.719014  3.721861  3.729161  3.745068  3.775261 |  100.0 |        0.0 | 103.433692 |
    | 2000-01-04 15:30:00 |      100.0 | 100.000196  100.002288  100.014365  100.063214  100.216696  100.612945   101.48423    103.1588 |   103.1588 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000001  -0.000003 | 3.717836   3.71787  3.718069   3.71888  3.721438   3.72807  3.742676  3.770688 |  100.0 |        0.0 |   103.1588 |
    | 2000-01-04 16:00:00 |      100.0 | 100.000168  100.001982   100.01259  100.055992  100.193873  100.553656  101.352929  102.904216 | 102.904216 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000001  -0.000002 | 3.717835  3.717865   3.71804   3.71876  3.721058  3.727079   3.74048  3.766448 |  100.0 |        0.0 | 102.904216 |
    | 2000-01-04 16:30:00 |      100.0 | 100.000144  100.001717  100.011031  100.049577  100.173376  100.499847  101.232558    102.6686 |   102.6686 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002 | 3.717835  3.717861  3.718014  3.718653  3.720716  3.726179  3.738466   3.76252 |  100.0 |        0.0 |   102.6686 |
    | 2000-01-04 17:00:00 |      100.0 | 100.000123  100.001488  100.009662   100.04388  100.154978  100.451041   101.12228  102.450686 | 102.450686 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002 | 3.717835  3.717857  3.717992  3.718559   3.72041  3.725363   3.73662  3.758884 |  100.0 |        0.0 | 102.450686 |
    | 2000-01-04 17:30:00 |      100.0 | 100.000105  100.001288  100.008461  100.038825  100.138473  100.406799  101.021313  102.249279 | 102.249279 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002 | 3.717834  3.717854  3.717972  3.718475  3.720135  3.724624   3.73493  3.755521 |  100.0 |        0.0 | 102.249279 |
    | 2000-01-04 18:00:00 |      100.0 |  100.00009  100.001115  100.007408   100.03434  100.123673  100.366718  100.928932  102.063254 | 102.063254 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002 | 3.717834  3.717851  3.717955  3.718401  3.719889  3.723954  3.733384  3.752412 |  100.0 |        0.0 | 102.063254 |
    | 2000-01-04 18:30:00 |      100.0 | 100.000077  100.000966  100.006483  100.030362  100.110409  100.330427  100.844458   101.89155 |  101.89155 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002 | 3.717834  3.717849  3.717939  3.718335  3.719668  3.723347  3.731969   3.74954 |  100.0 |        0.0 |  101.89155 |
    | 2000-01-04 19:00:00 |      100.0 | 100.000066  100.000836  100.005673  100.026837  100.098526  100.297587  100.767264  101.733168 | 101.733168 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000001 | 3.717834  3.717846  3.717926  3.718277   3.71947  3.722798  3.730676   3.74689 |  100.0 |        0.0 | 101.733168 |
    | 2000-01-04 19:30:00 |      100.0 | 100.000056  100.000723  100.004963  100.023712  100.087888  100.267885  100.696765   101.58717 |  101.58717 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000001 | 3.717834  3.717845  3.717914  3.718225  3.719293  3.722302  3.729495  3.744446 |  100.0 |        0.0 |  101.58717 |
    | 2000-01-04 20:00:00 |      100.0 | 100.000048  100.000626   100.00434  100.020945  100.078366  100.241037  100.632421  101.452676 | 101.452676 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000001 | 3.717834  3.717843  3.717904  3.718179  3.719135  3.721854  3.728418  3.742193 |  100.0 |        0.0 | 101.452676 |
    | 2000-01-04 20:30:00 |      100.0 | 100.000041  100.000542  100.003795  100.018495  100.069849  100.216781  100.573729  101.328858 | 101.328858 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000001 | 3.717833  3.717842  3.717895  3.718139  3.718993  3.721448  3.727434  3.740118 |  100.0 |        0.0 | 101.328858 |
    | 2000-01-04 21:00:00 |      100.0 | 100.000035  100.000469  100.003318  100.016326  100.062234  100.194879  100.520224  101.214941 | 101.214941 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0  -0.000001 | 3.717833   3.71784  3.717887  3.718103  3.718866  3.721083  3.726538  3.738209 |  100.0 |        0.0 | 101.214941 |
    | 2000-01-04 21:30:00 |      100.0 |  100.00003  100.000405    100.0029  100.014407  100.055428  100.175113  100.471477    101.1102 |   101.1102 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0  -0.000001 | 3.717833  3.717839   3.71788  3.718071  3.718753  3.720752  3.725721  3.736453 |  100.0 |        0.0 |   101.1102 |
    | 2000-01-04 22:00:00 |      100.0 | 100.000026  100.000351  100.002534   100.01271  100.049348  100.157283   100.42709  101.013955 | 101.013955 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0  -0.000001 | 3.717833  3.717838  3.717874  3.718043  3.718652  3.720455  3.724978  3.734839 |  100.0 |        0.0 | 101.013955 |
    | 2000-01-04 22:30:00 |      100.0 | 100.000022  100.000303  100.002213  100.011209  100.043918  100.141209  100.386697  100.925573 | 100.925573 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0  -0.000001 | 3.717833  3.717838  3.717869  3.718018  3.718562  3.720186  3.724301  3.733356 |  100.0 |        0.0 | 100.925573 |
    | 2000-01-04 23:00:00 |      100.0 | 100.000019  100.000262  100.001933  100.009883  100.039072  100.126724  100.349958   100.84446 |  100.84446 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0  -0.000001 | 3.717833  3.717837  3.717865  3.717996  3.718481  3.719945  3.723686  3.731995 |  100.0 |        0.0 |  100.84446 |
    | 2000-01-04 23:30:00 |      100.0 | 100.000016  100.000227  100.001688  100.008711  100.034749  100.113677  100.316562  100.770063 | 100.770063 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0  -0.000001 | 3.717833  3.717836  3.717861  3.717977  3.718409  3.719727  3.723127  3.730747 |  100.0 |        0.0 | 100.770063 |

.. _lstream_v001_overbank_flow:

overbank flow
_____________

In the above example, water flows in the main channel only.  Next, we
set the height of the main channel to 6 m:

>>> hm(6.0)
>>> parameters.update()

Now the calculated flood wave shows a considerable increase in translation
and retention.  Inspecting the time series of the stages reveals that this
effect is actually due to the (originally discontinuous) transition from
pure channel flow to a mixture of channel and overbank flow, where
additional storage capacities come into play:

.. integration-test::

    >>> test('lstream_v001_overbank_flow')
    |                date |         qz |                                                                                             qg |         qa |                                                                                     dh |                                                                              h | input1 |     input2 |     output |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 100.000029 | 100.000003       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 100.000175 | 100.000017  100.000001       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 100.000826 | 100.000089  100.000008       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717835  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 100.003194 | 100.000372  100.000037  100.000003       100.0       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717843  3.717834  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 100.010515 | 100.001326  100.000144  100.000014  100.000001       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717868  3.717837  3.717833  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 100.030315 | 100.004133  100.000492  100.000054  100.000006       100.0       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.717941  3.717847  3.717834  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 100.078179 | 100.011493  100.001492   100.00018   100.00002  100.000002       100.0       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.718125  3.717875  3.717838  3.717833  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 100.183351 | 100.028995  100.004079  100.000535  100.000066  100.000007  100.000001       100.0       100.0 |      100.0 |       0.0        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.718551  3.717944  3.717848  3.717835  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 100.396214 | 100.067234  100.010209  100.001444  100.000192  100.000024  100.000003       100.0       100.0 |      100.0 |  0.000001        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.719458  3.718101  3.717872  3.717838  3.717833  3.717833  3.717833  3.717833 |  100.0 |   0.396214 |      100.0 |
    | 2000-01-01 08:00:00 | 100.797305 | 100.144828  100.023644  100.003593  100.000512  100.000069  100.000009  100.000001       100.0 |      100.0 |  0.000001        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.721258  3.718435  3.717927  3.717847  3.717835  3.717833  3.717833  3.717833 |  100.0 |   0.797305 |      100.0 |
    | 2000-01-01 08:30:00 | 101.507032 | 100.292378  100.051143   100.00832  100.001268  100.000182  100.000024  100.000003       100.0 |      100.0 |  0.000002        0.0        0.0        0.0        0.0        0.0        0.0        0.0 | 3.724607  3.719101  3.718046  3.717866  3.717838  3.717833  3.717833  3.717833 |  100.0 |   1.507032 |      100.0 |
    | 2000-01-01 09:00:00 | 102.694853 | 100.557256   100.10411  100.018071  100.002935  100.000448  100.000064  100.000009  100.000001 | 100.000001 |  0.000003   0.000001        0.0        0.0        0.0        0.0        0.0        0.0 | 3.730495  3.720351  3.718283  3.717908  3.717845  3.717835  3.717833  3.717833 |  100.0 |   2.694853 | 100.000001 |
    | 2000-01-01 09:30:00 | 104.586356 | 101.008966  100.200763  100.037071  100.006398  100.001037  100.000159  100.000023  100.000003 | 100.000003 |  0.000005   0.000001        0.0        0.0        0.0        0.0        0.0        0.0 | 3.740335   3.72258  3.718735  3.717993  3.717859  3.717837  3.717833  3.717833 |  100.0 |   4.586356 | 100.000003 |
    | 2000-01-01 10:00:00 | 107.466641 | 101.744688  100.368788  100.072246  100.013214  100.002269  100.000367  100.000056  100.000008 | 100.000008 |  0.000009   0.000002        0.0        0.0        0.0        0.0        0.0        0.0 | 3.756035  3.726372  3.719553  3.718155   3.71789  3.717842  3.717834  3.717833 |  100.0 |   7.466641 | 100.000008 |
    | 2000-01-01 10:30:00 | 111.678559 | 102.894484  100.648428  100.134433  100.025995  100.004715  100.000805   100.00013   100.00002 |  100.00002 |  0.000013   0.000003   0.000001        0.0        0.0        0.0        0.0        0.0 | 3.780046  3.732557  3.720971  3.718455  3.717948  3.717853  3.717836  3.717833 |  100.0 |  11.678559 |  100.00002 |
    | 2000-01-01 11:00:00 | 117.614836 | 104.625672  101.095905  100.239895  100.048929  100.009353  100.001683  100.000286  100.000046 | 100.000046 |   0.00002   0.000005   0.000001        0.0        0.0        0.0        0.0        0.0 | 3.815353  3.742262  3.723331  3.718982  3.718057  3.717874   3.71784  3.717834 |  100.0 |  17.614836 | 100.000046 |
    | 2000-01-01 11:30:00 | 125.703736 | 107.145873  101.786998  100.412139  100.088474  100.017786  100.003365  100.000601  100.000102 | 100.000102 |  0.000028   0.000008   0.000002        0.0        0.0        0.0        0.0        0.0 | 3.865393  3.756962   3.72712  3.719875  3.718253  3.717914  3.717848  3.717835 |  100.0 |  25.703736 | 100.000102 |
    | 2000-01-01 12:00:00 | 136.388647 | 110.704308   102.82069  100.683991  100.154245  100.032544  100.006459   100.00121  100.000215 | 100.000215 |  0.000038   0.000012   0.000003   0.000001        0.0        0.0        0.0        0.0 | 3.933889  3.778513  3.733003  3.721336  3.718588  3.717986  3.717862  3.717838 |  100.0 |  36.388647 | 100.000215 |
    | 2000-01-01 12:30:00 | 150.102726 | 115.590949  104.322601  101.100048  100.260049  100.057487  100.011946  100.002343  100.000435 | 100.000435 |   0.00005   0.000017   0.000005   0.000001        0.0        0.0        0.0        0.0 | 4.024589  3.809164  3.741863  3.723652  3.719147  3.718112  3.717889  3.717843 |  100.0 |  50.102726 | 100.000435 |
    | 2000-01-01 13:00:00 | 167.240349 | 122.132947  106.448326   101.71915  100.425246  100.098313  100.021356  100.004377   100.00085 |  100.00085 |  0.000065   0.000024   0.000007   0.000002   0.000001        0.0        0.0        0.0 | 4.140938  3.851536  3.754839  3.727218  3.720049  3.718324  3.717935  3.717853 |  100.0 |  67.240349 |  100.00085 |
    | 2000-01-01 13:30:00 | 188.127569 | 130.687737  109.386166  102.617261  100.676215  100.163208  100.036998  100.007911  100.001601 | 100.001601 |   0.00008   0.000032    0.00001   0.000003   0.000001        0.0        0.0        0.0 | 4.285711  3.908551  3.773355  3.732562  3.721464  3.718672  3.718016   3.71787 |  100.0 |  88.127569 | 100.001601 |
    | 2000-01-01 14:00:00 | 212.993962 | 141.631926  113.359162  103.890442  101.048112  100.263642   100.06227  100.013869  100.002924 | 100.002924 |  0.000097   0.000042   0.000014   0.000004   0.000001        0.0        0.0        0.0 | 4.460677  3.983309  3.799148  3.740378  3.723627  3.719228  3.718149  3.717901 |  100.0 | 112.993962 | 100.002924 |
    | 2000-01-01 14:30:00 | 241.948195 | 155.345173   118.62584  105.657958  101.586862  100.415315  100.102044  100.023639  100.005182 | 100.005182 |  0.000114   0.000053    0.00002   0.000006   0.000002        0.0        0.0        0.0 | 4.666348  4.078907  3.834271  3.751553  3.726856  3.720092  3.718366  3.717952 |  100.0 | 141.948195 | 100.005182 |
    | 2000-01-01 15:00:00 |  274.95935 | 172.188724  125.478902  108.065412  102.351411  100.639301  100.163151  100.039254  100.008935 | 100.008935 |  0.000131   0.000066   0.000026   0.000009   0.000003   0.000001        0.0        0.0 | 4.901846  4.198196  3.881074  3.767198  3.731571  3.721405  3.718708  3.718035 |  100.0 |  174.95935 | 100.008935 |
    | 2000-01-01 15:30:00 | 311.845554 | 192.479101  134.240813  111.287675  103.416239   100.96338   100.25498  100.063624   100.01502 |  100.01502 |  0.000146   0.000081   0.000034   0.000012   0.000004   0.000001        0.0        0.0 | 5.164937  4.343518  3.942151  3.788674  3.738318  3.723359  3.719235  3.718169 |  100.0 | 211.845554 |  100.01502 |
    | 2000-01-01 16:00:00 | 352.270855 | 216.458338  145.255128   115.53123  104.874124  101.423603  100.390198  100.100837  100.024655 | 100.024655 |   0.00016   0.000096   0.000043   0.000016   0.000005   0.000002        0.0        0.0 | 5.452163  4.516424  4.020241  3.817612  3.747795  3.726207  3.720034   3.71838 |  100.0 | 252.270855 | 100.024655 |
    | 2000-01-01 16:30:00 | 395.750601 | 244.213717  158.873023  121.034838  106.839105  102.066103  100.585624  100.156526  100.039587 | 100.039587 |  0.000169   0.000112   0.000054   0.000021   0.000007   0.000002   0.000001        0.0 | 5.756641  4.717327  4.118078  3.855916  3.760875  3.730285  3.721217  3.718702 |  100.0 | 295.750601 | 100.039587 |
    | 2000-01-01 17:00:00 | 441.664962 |  272.63583  175.262862  128.065162  109.449125  102.949168   100.86327  100.238325  100.062271 | 100.062271 |   0.00013   0.000123   0.000066   0.000028    0.00001   0.000003   0.000001        0.0 | 5.990478  4.938905  4.237778  3.905735  3.778637  3.736024  3.722941  3.719188 |  100.0 | 341.664962 | 100.062271 |
    | 2000-01-01 17:30:00 | 489.279607 | 292.218408  193.320548  136.803161  112.863013  104.145265  101.251563  100.356425  100.096093 | 100.096093 |  0.000073   0.000121   0.000078   0.000035   0.000013   0.000004   0.000001        0.0 | 6.121346  5.156406  4.377907  3.969136  3.802366  3.743976  3.725408  3.719906 |  100.0 | 389.279607 | 100.096093 |
    | 2000-01-01 18:00:00 |  537.77215 | 309.172032  211.666167  147.176524  117.230213  105.740646  101.786585  100.524246  100.145665 | 100.145665 |  0.000068   0.000115   0.000087   0.000043   0.000017   0.000006   0.000002   0.000001 | 6.243952    5.3641  4.533829  4.047423  3.833477  3.754823  3.728886  3.720951 |  100.0 |  437.77215 | 100.145665 |
    | 2000-01-01 18:30:00 | 586.262607 | 328.410593  230.316165  159.028651  122.665656   107.83042  102.512773  100.759155  100.217164 | 100.217164 |  0.000074   0.000113   0.000093   0.000052   0.000022   0.000008   0.000003   0.000001 | 6.376601  5.566787  4.701626  4.141025  3.873382  3.769375  3.733713  3.722445 |  100.0 | 486.262607 | 100.217164 |
    | 2000-01-01 19:00:00 | 633.846018 | 350.729287   249.62895  172.258595  129.254094  110.513135  103.482791   101.08314  100.318735 | 100.318735 |   0.00008   0.000111   0.000098    0.00006   0.000028   0.000011   0.000004   0.000001 | 6.520236  5.766865  4.878748  4.249756  3.923373  3.788553  3.740311  3.724553 |  100.0 | 533.846018 | 100.318735 |
    | 2000-01-01 19:30:00 | 679.625318 | 376.060385  268.699812  186.759615  137.062073  113.887579  104.756869   101.52334  100.460925 | 100.460925 |  0.000085   0.000093   0.000101   0.000068   0.000034   0.000014   0.000005   0.000002 | 6.672591    5.9344  5.061121  4.372902   3.98456  3.813356  3.749189   3.72748 |  100.0 | 579.625318 | 100.460925 |
    | 2000-01-01 20:00:00 | 722.742733 | 404.090558  283.436832  201.967952  146.102396  118.049454  106.401946  102.112426  100.657124 | 100.657124 |  0.000088   0.000059   0.000098   0.000075   0.000041   0.000017   0.000007   0.000002 |   6.8308  6.039798  5.237552   4.50832  4.057727  3.844829  3.760943  3.731487 |  100.0 | 622.742733 | 100.657124 |
    | 2000-01-01 20:30:00 | 762.408181 | 434.375816   294.99175  217.011322  156.259482  123.077281  108.489835  102.888778  100.923993 | 100.923993 |  0.000089   0.000047   0.000091    0.00008   0.000047   0.000022   0.000009   0.000003 | 6.991893  6.125119  5.402199    4.6522  4.143046  3.883987  3.776252  3.736893 |  100.0 | 662.408181 | 100.923993 |
    | 2000-01-01 21:00:00 | 797.923464 | 466.373269  306.759263  231.672705  167.333507  129.017221  111.093374   103.89628  101.281819 | 101.281819 |  0.000089   0.000048   0.000086   0.000083   0.000054   0.000027   0.000011   0.000004 | 7.152944  6.211367  5.557026  4.800995   4.23999  3.931715  3.795858  3.744077 |  100.0 | 697.923464 | 101.281819 |
    | 2000-01-01 21:30:00 | 828.701418 | 499.463162  320.029092  246.122381  179.154572  135.881286   114.28084  105.183429  101.754887 | 101.754887 |  0.000088   0.000052   0.000082   0.000084    0.00006   0.000032   0.000014   0.000005 | 7.311167  6.304526   5.70523  4.952225  4.347542  3.988681  3.820537  3.753483 |  100.0 | 728.701418 | 101.754887 |
    | 2000-01-01 22:00:00 |  854.27953 | 532.980419  335.198723  260.407919  191.590019  143.659125  118.114356  106.802242  102.371345 | 102.371345 |  0.000085   0.000056   0.000077   0.000084   0.000065   0.000037   0.000017   0.000007 | 7.463989  6.405685  5.843303  5.104025  4.464408  4.055295  3.851066  3.765614 |  100.0 |  754.27953 | 102.371345 |
    | 2000-01-01 22:30:00 | 874.327871 | 566.238625  352.333812  273.313709  204.433753  152.318308  122.645962  108.806751  103.163362 | 103.163362 |  0.000081    0.00006    0.00006   0.000083   0.000069   0.000042   0.000021   0.000009 |   7.6091  6.514106  5.950507  5.252547  4.588932  4.131685  3.888179  3.781025 |  100.0 | 774.327871 | 103.163362 |
    | 2000-01-01 23:00:00 | 888.651531 | 598.561152  371.338346  283.390492  217.222761  161.772527  127.914629  111.251431  104.166756 | 104.166756 |  0.000075   0.000063   0.000042   0.000078   0.000072   0.000048   0.000025   0.000011 | 7.744491  6.628348  6.026871  5.392202  4.718686  4.217603  3.932529  3.800308 |  100.0 | 788.651531 | 104.166756 |
    | 2000-01-01 23:30:00 | 897.187983 | 629.312807  392.017662  291.940809  229.543675  171.866177  133.936719  114.188929   105.42057 |  105.42057 |  0.000069   0.000066   0.000036   0.000072   0.000073   0.000053   0.000029   0.000013 | 7.868461  6.746688  6.091889  5.521313   4.85084  4.312344  3.984625  3.824077 |  100.0 | 797.187983 |  105.42057 |
    | 2000-01-02 00:00:00 |      900.0 | 657.931356  414.105621  300.528344  241.320749  182.420812   140.70005  117.666092  106.966171 | 106.966171 |  0.000062   0.000067   0.000036   0.000067   0.000073   0.000057   0.000033   0.000016 | 7.979382  6.867287  6.156081  5.641375  4.983015  4.414804  4.044771  3.852938 |  100.0 |      800.0 | 106.966171 |
    | 2000-01-02 00:30:00 | 897.264896 | 683.800116  437.274323  309.904768  252.648729   193.28471  148.168137  121.721368  108.846105 | 108.846105 |  0.000053   0.000067   0.000038   0.000062   0.000073    0.00006   0.000038   0.000019 | 8.074798  6.988217  6.223853  5.753774   5.11356  4.523642  4.113034  3.887463 |  100.0 | 797.264896 | 108.846105 |
    | 2000-01-02 01:00:00 | 889.260928 | 706.117996  461.118734  320.414446  263.446405   204.32652   156.28621  126.382714  111.102919 | 111.102919 |  0.000044   0.000066   0.000041   0.000056   0.000071   0.000063   0.000042   0.000023 | 8.153161  7.107423  6.296837  5.855172  5.241054  4.637412  4.189233   3.92815 |  100.0 | 789.260928 | 111.102919 |
    | 2000-01-02 01:30:00 | 876.351729 | 724.523813  485.183197    332.1892  273.035415  215.367983  164.981658  131.665314  113.777303 | 113.777303 |  0.000035   0.000064   0.000044   0.000045   0.000068   0.000065   0.000047   0.000026 | 8.215467   7.22296  6.375244  5.936548  5.363133  4.754514  4.272944  3.975399 |  100.0 | 776.351729 | 113.777303 |
    | 2000-01-02 02:00:00 | 858.969595 | 739.128312  509.016553  345.235399   280.88542  226.125872  174.150721  137.569761  116.906765 | 116.906765 |  0.000026   0.000061   0.000046   0.000035   0.000063   0.000066    0.00005    0.00003 |   8.2631  7.333207  6.458539  5.998868  5.477061  4.873055  4.363482  4.029474 |  100.0 | 758.969595 | 116.906765 |
    | 2000-01-02 02:30:00 |   837.5984 | 750.094402  532.205678  359.480678   287.62884  236.370851  183.653443  144.077287  120.523174 | 120.523174 |  0.000019   0.000058   0.000048   0.000029   0.000058   0.000066   0.000054   0.000034 | 8.297094  7.436861  6.545811  6.051584  5.581944  4.991078  4.459883  4.090478 |  100.0 |   737.5984 | 120.523174 |
    | 2000-01-02 03:00:00 | 812.756821 | 757.553788    554.3792  374.798599  294.125932  246.038996   193.34492  151.148379  124.650906 | 124.650906 |  0.000012   0.000053    0.00005   0.000028   0.000054   0.000064   0.000056   0.000038 |  8.31822  7.532881  6.635974  6.101465   5.67848  5.106942  4.560979  4.158318 |  100.0 | 712.756821 | 124.650906 |
    | 2000-01-02 03:30:00 | 784.982427 |  761.62197  575.209049  391.022834  300.912965  255.155069  203.097074  158.725685  129.304279 | 129.304279 |  0.000005   0.000049   0.000051   0.000028   0.000049   0.000063   0.000058   0.000041 | 8.327088  7.620452  6.727865  6.152144  5.767252  5.219446  4.665498  4.232705 |  100.0 | 684.982427 | 129.304279 |
    | 2000-01-02 04:00:00 | 754.817086 |  762.41403  594.411747  407.954774  308.292596  263.658776   212.79458  166.738791  134.486649 | 134.486649 | -0.000002   0.000044   0.000051    0.00003   0.000044    0.00006   0.000059   0.000045 | 8.324208  7.698949  6.820297  6.205498  5.846671  5.327578  4.772143  4.313155 |  100.0 | 654.817086 | 134.486649 |
    | 2000-01-02 04:30:00 |  722.79402 | 760.052079  611.748125  425.373165   316.43206  271.256554  222.308922   175.10569  140.189051 | 140.189051 | -0.000008   0.000038   0.000051   0.000032   0.000037   0.000057    0.00006   0.000048 | 8.310022  7.767912  6.912106  6.262388  5.913165  5.430035  4.879595  4.399014 |  100.0 |  622.79402 | 140.189051 |
    | 2000-01-02 05:00:00 | 689.426711 |  754.66807  627.023793  443.040348  325.409359  277.718919  231.474917  183.729079  146.389666 | 146.389666 | -0.000014   0.000033    0.00005   0.000034    0.00003   0.000053   0.000059    0.00005 | 8.284917  7.827022  7.002176   6.32302  5.966717  5.525425  4.986471  4.489469 |  100.0 | 589.426711 | 146.389666 |
    | 2000-01-02 05:30:00 | 655.199767 | 746.402837  640.087605  460.709613  335.240032   283.29387  240.150529  192.496131  153.052362 | 153.052362 |  -0.00002   0.000027   0.000048   0.000036   0.000025   0.000049   0.000058   0.000052 |  8.24923  7.876075  7.089466  6.387185  6.011755  5.613121  5.091415  4.583562 |  100.0 | 555.199767 | 153.052362 |
    | 2000-01-02 06:00:00 | 620.561767 | 735.403102  650.824716  478.134305  345.893744  288.457655  248.275378  201.291298  160.126424 | 160.126424 | -0.000026   0.000022   0.000046   0.000037   0.000023   0.000045   0.000057   0.000054 | 8.203242  7.914965  7.173019  6.454416   6.05269  5.693327  5.193272  4.680232 |  100.0 | 520.561767 | 160.126424 |
    | 2000-01-02 06:30:00 | 585.920022 | 721.816708  659.154206  495.070989  357.305929  293.594961  255.841873  210.009893  167.549028 | 167.549028 | -0.000031   0.000016   0.000044   0.000039   0.000022   0.000041   0.000054   0.000055 | 8.147171  7.943679  7.251976   6.52409  6.092454   5.76632  5.291159  4.778387 |  100.0 | 485.920022 | 167.549028 |
    | 2000-01-02 07:00:00 |  551.63713 | 705.796687  665.026469   511.28751  369.384644   298.96193     262.815  218.561222  175.248557 | 175.248557 | -0.000037    0.00001   0.000041    0.00004   0.000022   0.000036   0.000052   0.000055 | 8.081212  7.962298  7.325581  6.595493  6.132862  5.831591  5.384352  4.876954 |  100.0 |  451.63713 | 175.248557 |
    | 2000-01-02 07:30:00 | 518.029154 | 687.556149  668.431339  526.565645  382.015704   304.72559  269.090723  226.850082  183.147191 | 183.147191 | -0.000042   0.000005   0.000038    0.00004   0.000023   0.000031   0.000049   0.000054 | 8.005924  7.971012  7.393183  6.667861  6.175021  5.887885  5.472129  4.974889 |  100.0 | 418.029154 | 183.147191 |
    | 2000-01-02 08:00:00 | 485.365238 | 667.525856  669.405032  540.709525  395.066507  310.993158  274.573176  234.778744   191.16035 |  191.16035 | -0.000046        0.0   0.000034    0.00004   0.000025   0.000026   0.000045   0.000053 | 7.922959  7.970156   7.45425   6.74041   6.21955  5.935048  5.553774   5.07118 |  100.0 | 385.365238 |  191.16035 |
    | 2000-01-02 08:30:00 |  453.86844 | 646.311029  668.047379  553.549354  408.390322  317.826153  279.352222   242.25863  199.198267 | 199.198267 | -0.000049  -0.000006    0.00003    0.00004   0.000026   0.000022   0.000042   0.000052 | 7.834357   7.96025  7.508374  6.812359   6.26671  5.975013  5.628873  5.164873 |  100.0 |  353.86844 | 199.198267 |
    | 2000-01-02 09:00:00 | 423.717566 | 624.340072  664.511452  564.947083  421.830671  325.249679  283.673585  249.238384   207.17097 |  207.17097 | -0.000052   -0.00001   0.000026   0.000039   0.000028    0.00002   0.000038    0.00005 | 7.741395  7.941898  7.555284  6.882948  6.316497  6.010437  5.697422  5.255152 |  100.0 | 323.717566 |  207.17097 |
    | 2000-01-02 09:30:00 | 395.049792 | 601.877496    658.9738  574.800826  435.226317  333.259005  287.793312   255.69884  214.996173 | 214.996173 | -0.000054  -0.000015   0.000022   0.000038   0.000029   0.000018   0.000035   0.000048 | 7.644967  7.915733   7.59483  6.951464  6.368711  6.043519  5.759555  5.341366 |  100.0 | 295.049792 | 214.996173 |
    | 2000-01-02 10:00:00 | 367.963873 |   579.1579  651.628324  583.041234   448.41603  341.825247  291.912513  261.623035  222.600627 | 222.600627 | -0.000055  -0.000019   0.000018   0.000037    0.00003   0.000018   0.000031   0.000045 | 7.545875  7.882421  7.626978  7.017249  6.423014   6.07582   5.81519  5.423007 |  100.0 | 267.963873 | 222.600627 |
    | 2000-01-02 10:30:00 | 342.523746 | 556.401302  642.676395  589.632821  461.243254  350.899794  296.177475  266.973003  229.918685 | 229.918685 | -0.000056  -0.000022   0.000014   0.000035   0.000031   0.000018   0.000027   0.000043 | 7.444867  7.842645  7.651795  7.079715  6.478979  6.108415  5.864015  5.499631 |  100.0 | 242.523746 | 229.918685 |
    | 2000-01-02 11:00:00 | 318.762374 | 533.804667  632.321956  594.569202   473.56004  360.417991  300.693862  271.724261  236.883443 | 236.883443 | -0.000057  -0.000025    0.00001   0.000033   0.000032   0.000019   0.000023    0.00004 | 7.342626  7.797091  7.669433  7.138351  6.536123  6.142042  5.906052  5.570861 |  100.0 | 218.762374 | 236.883443 |
    | 2000-01-02 11:30:00 |  296.68568 |  511.54026  620.762972  597.872051  485.230741  370.302105  305.537477  275.917053  243.439817 | 243.439817 | -0.000057  -0.000028   0.000006    0.00003   0.000032    0.00002    0.00002   0.000036 | 7.239778  7.746428  7.680119  7.192725  6.593931  6.177185  5.942206  5.636459 |  100.0 |  196.68568 | 243.439817 |
    | 2000-01-02 12:00:00 | 276.276442 | 489.755626  608.189804  599.587077  496.134915  380.464165  310.759577  279.674323  249.550816 | 249.550816 | -0.000057  -0.000031   0.000002   0.000028   0.000032   0.000021   0.000018   0.000033 | 7.136885  7.691302  7.684132  7.242489  6.651877  6.214127  5.973993  5.696372 |  100.0 | 176.276442 | 249.550816 |
    | 2000-01-02 12:30:00 | 257.498082 | 468.573324  594.788274  599.777781  506.168784  390.808398  316.390065  283.157664  255.195346 | 255.195346 | -0.000057  -0.000033  -0.000001   0.000025   0.000032   0.000022   0.000016    0.00003 | 7.034447  7.632338  7.681804  7.287377  6.709435  6.252982  6.002931  5.750679 |  100.0 | 157.498082 | 255.195346 |
    | 2000-01-02 13:00:00 | 240.298231 | 448.093811  580.733192  598.525106  515.246911  401.233968  322.440109  286.512046  260.364696 | 260.364696 | -0.000056  -0.000035  -0.000005   0.000022   0.000031   0.000023   0.000015   0.000027 | 6.932902  7.570128  7.673501  7.327198  6.766096  6.293729  6.030257  5.799435 |  100.0 | 140.298231 | 260.364696 |
    | 2000-01-02 13:30:00 | 224.612057 | 428.396886  566.187701  595.923017  523.302755  411.637656   328.90425  289.855105  265.048022 | 265.048022 | -0.000056  -0.000036  -0.000008   0.000019   0.000031   0.000024   0.000015   0.000024 |  6.83263  7.505233  7.659614  7.361838  6.821377  6.336232  6.056903  5.842662 |  100.0 | 124.612057 | 265.048022 |
    | 2000-01-02 14:00:00 | 210.365288 | 409.543169  551.303546   592.07423  530.288345  421.916224  335.762767  293.278549  269.244486 | 269.244486 | -0.000055  -0.000037  -0.000011   0.000016    0.00003   0.000024   0.000015   0.000021 | 6.733955  7.438179  7.640552  7.391247  6.874828  6.380275  6.083563  5.880524 |  100.0 | 110.365288 | 269.244486 |
    | 2000-01-02 14:30:00 |  197.47692 | 391.576692  536.218633  587.088143  536.173867  431.968956  342.983799  296.853653  272.983166 | 272.983166 | -0.000054  -0.000038  -0.000013   0.000013   0.000028   0.000025   0.000015   0.000018 | 6.637143  7.369456  7.616736  7.415438   6.92604  6.425581  6.110753  5.913561 |  100.0 |   97.47692 | 272.983166 |
    | 2000-01-02 15:00:00 | 185.861607 | 374.526926  521.057126  581.077788  540.946593  441.699827  350.525354   300.63577  276.334744 | 276.334744 | -0.000053  -0.000039  -0.000016   0.000011   0.000027   0.000026   0.000016   0.000016 | 6.542411  7.299515  7.588588  7.434479  6.974649  6.471831  6.138852  5.942662 |  100.0 |  85.861607 | 276.334744 |
    | 2000-01-02 15:30:00 | 175.431719 | 358.410791  505.929511  574.157412  544.609516   451.01941  358.337113  304.666888  279.396987 | 279.396987 | -0.000051  -0.000039  -0.000018   0.000008   0.000025   0.000026   0.000016   0.000015 | 6.449917  7.228773  7.556528  7.448483  7.020342  6.518684  6.168126  5.968842 |  100.0 |  75.431719 | 279.396987 |
    | 2000-01-02 16:00:00 | 166.099094 | 343.233755  490.932919  566.440422  547.179747  459.846448  366.362082   308.97696   282.27096 |  282.27096 |  -0.00005   -0.00004   -0.00002   0.000005   0.000024   0.000026   0.000017   0.000013 |  6.35974  7.157608  7.520969  7.457605  7.062852  6.565782  6.198739  5.993026 |  100.0 |  66.099094 |  282.27096 |
    | 2000-01-02 16:30:00 | 157.776495 | 328.986918  476.151439  558.037649  548.686768  468.109076  374.538163  313.584646  285.046305 | 285.046305 | -0.000049   -0.00004  -0.000021   0.000002   0.000022   0.000026   0.000018   0.000013 |  6.27177  7.086361  7.482312  7.462032  7.101967  6.612767  6.230768  6.015976 |  100.0 |  57.776495 | 285.046305 |
    | 2000-01-02 17:00:00 | 150.378786 | 315.626757  461.656037  549.055863  549.170627  475.745713  382.799698  318.497934  287.797389 | 287.797389 | -0.000048  -0.000039  -0.000023        0.0    0.00002   0.000026   0.000019   0.000012 | 6.185264  7.015323  7.440944  7.461977  7.137519  6.659286   6.26421  6.038292 |  100.0 |  50.378786 | 287.797389 |
    | 2000-01-02 17:30:00 | 143.823868 | 302.991534  447.496987  539.596571  548.680042  482.705347  391.078854  323.714905  290.584887 | 290.584887 | -0.000049  -0.000039  -0.000024  -0.000002   0.000018   0.000025   0.000019   0.000012 |  6.09702  6.944689  7.397229  7.457677  7.169392  6.704998  6.298991  6.060443 |  100.0 |  43.823868 | 290.584887 |
    | 2000-01-02 18:00:00 | 138.033385 | 290.389097  433.692813  529.752652  547.270655  488.948185  399.307336  329.224582  293.458317 | 293.458317 | -0.000058  -0.000039  -0.000025  -0.000005   0.000016   0.000025    0.00002   0.000012 | 5.992654  6.874306  7.351499  7.449381  7.197512  6.749579  6.334985  6.082799 |  100.0 |  38.033385 | 293.458317 |
    | 2000-01-02 18:30:00 | 132.933235 | 274.631604  420.097429  519.603539  545.002749  494.444905  407.417439  335.008039  296.458395 | 296.458395 | -0.000101   -0.00004  -0.000026  -0.000007   0.000014   0.000024   0.000021   0.000013 | 5.811712  6.802512  7.303982  7.437345  7.221845   6.79273  6.372019   6.10565 |  100.0 |  32.933235 | 296.458395 |
    | 2000-01-02 19:00:00 | 128.453911 | 251.872903  406.198922  509.186453   541.93802  499.176642  415.343615  341.039404  299.618403 | 299.618403 | -0.000134  -0.000043  -0.000027  -0.000009   0.000011   0.000023   0.000021   0.000013 | 5.571135  6.725957  7.254645   7.42181  7.242395  6.834176  6.409889  6.129222 |  100.0 |  28.453911 | 299.618403 |
    | 2000-01-02 19:30:00 | 124.530701 | 229.746597  391.743353  498.488308  538.133939  503.133766  423.023444  347.286952  302.964887 | 302.964887 | -0.000121  -0.000045  -0.000029   -0.00001   0.000009   0.000022   0.000021   0.000014 | 5.353639  6.645157  7.203335  7.402984  7.259194  6.873672  6.448367  6.153684 |  100.0 |  24.530701 | 302.964887 |
    | 2000-01-02 20:00:00 | 121.103761 |  211.14753   377.07599  487.510742  533.641612  506.314582  430.398719   353.71412  306.517977 | 306.517977 | -0.000107  -0.000046   -0.00003  -0.000012   0.000007   0.000021   0.000022   0.000014 | 5.161767  6.561922  7.150066  7.381049  7.272297  6.911001  6.487211  6.179149 |  100.0 |  21.103761 | 306.517977 |
    | 2000-01-02 20:30:00 | 118.118088 |  195.51391  362.532745  476.289372  528.510004  508.724011  437.415948  360.280457  310.291417 | 310.291417 | -0.000094  -0.000047  -0.000031  -0.000014   0.000005   0.000019   0.000022   0.000015 | 4.992314  6.477641  7.094995   7.35618  7.281778  6.945979  6.526172  6.205681 |  100.0 |  18.118088 | 310.291417 |
    | 2000-01-02 21:00:00 | 115.523427 | 182.296829  348.353258  464.883885  522.788422  510.372897  444.027017  366.942546  314.292663 | 314.292663 | -0.000083  -0.000047  -0.000031  -0.000015   0.000003   0.000018   0.000022   0.000015 | 4.842481  6.393315  7.038368  7.328555  7.287723  6.978448  6.564998  6.233296 |  100.0 |  15.523427 | 314.292663 |
    | 2000-01-02 21:30:00 | 113.274094 | 171.065392  334.700529  453.365889  516.528183   511.27744  450.189518  373.654877  318.523014 | 318.523014 | -0.000074  -0.000047  -0.000032  -0.000017   0.000001   0.000017   0.000021   0.000016 | 4.709886  6.309595  6.980481  7.298365  7.290237  7.008281  6.603443  6.261966 |  100.0 |  13.274094 | 318.523014 |
    | 2000-01-02 22:00:00 | 111.328777 | 161.478716  321.670411  441.810419  509.782951  511.458777  455.867007  380.370702  322.977897 | 322.977897 | -0.000065  -0.000046  -0.000033  -0.000018        0.0   0.000015   0.000021   0.000016 | 4.592484  6.226631  6.921641  7.265808  7.289435  7.035377  6.641264  6.291625 |  100.0 |  11.328777 | 322.977897 |
    | 2000-01-02 22:30:00 | 109.650285 |  153.26442  309.266727  430.288157  502.608265  510.942589  461.029138  387.042848  327.647274 | 327.647274 | -0.000058  -0.000046  -0.000033  -0.000019  -0.000002   0.000013   0.000021   0.000017 | 4.488512  6.143319   6.86213  7.231091  7.285444  7.059663  6.678233   6.32217 |  100.0 |   9.650285 | 327.647274 |
    | 2000-01-02 23:00:00 | 108.205287 | 146.203017  297.247018  418.855157   495.05996  509.758632  465.651718  393.624517  332.516146 | 332.516146 | -0.000051   -0.00005  -0.000033   -0.00002  -0.000004   0.000012    0.00002   0.000017 | 4.396434   6.05399  6.802085  7.194422  7.278404  7.081091  6.714131  6.353477 |  100.0 |   8.205287 | 332.516146 |
    | 2000-01-02 23:30:00 | 106.964036 | 140.115166  284.365438  407.500133  487.191501  507.940082  469.716571  400.069939   337.56513 |  337.56513 | -0.000045  -0.000068  -0.000034  -0.000021  -0.000006    0.00001   0.000019   0.000018 | 4.314911  5.931415  6.741033  7.155978   7.26846  7.099636  6.748759  6.385395 |  100.0 |   6.964036 |  337.56513 |
    | 2000-01-03 00:00:00 | 105.900084 | 134.854424  266.025317  395.973203  479.040596  505.522296  473.211469  406.335162  342.771031 | 342.771031 |  -0.00004   -0.00012  -0.000036  -0.000022  -0.000007   0.000009   0.000018   0.000018 | 4.242761  5.716182  6.676322    7.1158  7.255759  7.115294  6.781931  6.417761 |  100.0 |   5.900084 | 342.771031 |
    | 2000-01-03 00:30:00 |  104.99001 | 130.299271  243.138348  383.785435  470.599441  502.539613  476.129629  412.378544  348.107447 | 348.107447 | -0.000035  -0.000126  -0.000039  -0.000023  -0.000009   0.000007   0.000018   0.000018 | 4.178943  5.488747  6.605954  7.073693  7.240426   7.12808  6.813483  6.450401 |  100.0 |    4.99001 | 348.107447 |
    | 2000-01-03 01:00:00 | 104.213141 | 126.348934  222.921852  371.002181  461.842415  499.021129  478.468962   418.16134  353.545363 | 353.545363 | -0.000031  -0.000112  -0.000041  -0.000025   -0.00001   0.000006   0.000017   0.000018 | 4.122533  5.286876   6.53149  7.029506  7.222558  7.138022  6.843268  6.483135 |  100.0 |   4.213141 | 353.545363 |
    | 2000-01-03 01:30:00 | 103.551298 | 122.918433  205.924134  358.003539  452.771837  494.992694   480.23112  423.647929  359.053703 | 359.053703 | -0.000028  -0.000099  -0.000043  -0.000026  -0.000011   0.000004   0.000015   0.000018 | 4.072711  5.108414  6.454599  6.983266  7.202241   7.14516  6.871159  6.515778 |  100.0 |   3.551298 | 359.053703 |
    | 2000-01-03 02:00:00 | 102.988541 | 119.936245  191.560897  345.085123  443.423337  490.479993  481.421013   428.80613  364.599885 | 364.599885 | -0.000024  -0.000088  -0.000043  -0.000027  -0.000013   0.000002   0.000014   0.000018 | 4.028744  4.950453  6.376513   6.93513  7.179565  7.149538  6.897045  6.548149 |  100.0 |   2.988541 | 364.599885 |
    | 2000-01-03 02:30:00 |  102.51094 | 117.341697  179.358778  332.457517    433.8522  485.511136   482.04656  433.607342  370.150348 | 370.150348 | -0.000022  -0.000078  -0.000044  -0.000028  -0.000014   0.000001   0.000013   0.000018 | 3.989981  4.810487  6.298063  6.885333  7.154631  7.151212  6.920837   6.58007 |  100.0 |    2.51094 | 370.150348 |
    | 2000-01-03 03:00:00 | 102.106358 | 115.083064  168.942886  320.255096  424.123362  480.117704  482.118642  438.026648   375.67106 |  375.67106 | -0.000019  -0.000069  -0.000044  -0.000028  -0.000015  -0.000001   0.000012   0.000017 | 3.955837  4.686363  6.219541  6.834144  7.127557  7.150245   6.94246  6.611367 |  100.0 |   2.106358 |  375.67106 |
    | 2000-01-03 03:30:00 | 101.764251 | 113.116057  160.013861  308.513534  414.302291  474.334761  481.651104  442.042891  381.127995 | 381.127995 | -0.000017  -0.000061  -0.000044  -0.000029  -0.000016  -0.000002   0.000011   0.000017 | 3.925793  4.576218  6.139991  6.781824  7.098476  7.146709  6.961857  6.641874 |  100.0 |   1.764251 | 381.127995 |
    | 2000-01-03 04:00:00 |  101.47548 | 111.402659  152.330632  297.028858  404.443831  468.199662  480.660696  445.638721  386.487604 | 386.487604 | -0.000015  -0.000054  -0.000048   -0.00003  -0.000017  -0.000003    0.00001   0.000016 | 3.899382  4.478437  6.054196  6.728506  7.067528  7.140685  6.978987  6.671436 |  100.0 |    1.47548 | 386.487604 |
    | 2000-01-03 04:30:00 | 101.232151 | 109.909907  145.697277  284.688987  394.543746  461.749671  479.166824  448.800551  391.717195 | 391.717195 | -0.000013  -0.000048  -0.000065   -0.00003  -0.000018  -0.000005   0.000008   0.000016 | 3.876189  4.391607  5.937882  6.673772  7.034836  7.132261  6.993823  6.699908 |  100.0 |   1.232151 | 391.717195 |
    | 2000-01-03 05:00:00 | 101.027458 | 108.609464  139.953627  267.328542   384.38716  455.010247  477.190767  451.518563  396.785402 | 396.785402 | -0.000011  -0.000043  -0.000113  -0.000033  -0.000019  -0.000006   0.000007   0.000015 | 3.855842  4.314492  5.734382   6.61522  7.000404  7.121527  7.006355  6.727154 |  100.0 |   1.027458 | 396.785402 |
    | 2000-01-03 05:30:00 | 100.855551 | 107.476657  134.967579  245.230857  373.521974  447.968199  474.753156  453.786426  401.662468 | 401.662468 |  -0.00001  -0.000038  -0.000123  -0.000036   -0.00002  -0.000007   0.000006   0.000014 |  3.83801  4.246005  5.513309  6.550772   6.96402  7.108553  7.016582  6.753054 |  100.0 |   0.855551 | 401.662468 |
    | 2000-01-03 06:00:00 | 100.711411 | 106.490156  130.629625  225.396809  361.976841  440.592655  471.870335  455.600786  406.320598 | 406.320598 | -0.000009  -0.000034   -0.00011  -0.000038  -0.000021  -0.000008   0.000004   0.000014 | 3.822399  4.185188  5.315997  6.481834    6.9255  7.093389  7.024512  6.777499 |  100.0 |   0.711411 | 406.320598 |
    | 2000-01-03 06:30:00 | 100.590745 | 105.631273  126.848215  208.633876  350.115375  432.878132  468.555964  456.960597  410.734076 | 410.734076 | -0.000008   -0.00003  -0.000097   -0.00004  -0.000023   -0.00001   0.000003   0.000013 | 3.808744  4.131191  5.140965  6.410047  6.884839  7.076073  7.030162  6.800391 |  100.0 |   0.590745 | 410.734076 |
    | 2000-01-03 07:00:00 | 100.489885 | 104.883757  123.546478  194.404593  338.228484  424.851849  464.823923  457.866821   414.87944 |  414.87944 | -0.000007  -0.000027  -0.000086  -0.000041  -0.000024  -0.000011   0.000002   0.000012 | 3.796812  4.083263  4.985499  6.336625  6.842161  7.056649  7.033551  6.821647 |  100.0 |   0.489885 |  414.87944 |
    | 2000-01-03 07:30:00 | 100.405708 | 104.233426  120.659491  182.262592  326.524907  416.561132  460.690832  458.322366  418.735558 | 418.735558 | -0.000006  -0.000024  -0.000077  -0.000041  -0.000025  -0.000012   0.000001   0.000011 | 3.786396  4.040736  4.847246  6.262336  6.797667  7.035174  7.034704  6.841194 |  100.0 |   0.405708 | 418.735558 |
    | 2000-01-03 08:00:00 |  100.33556 | 103.667894  118.132141  171.852544   315.13161  408.063511  456.177129  458.332203  422.283697 | 422.283697 | -0.000005  -0.000021  -0.000068  -0.000042  -0.000026  -0.000013  -0.000001    0.00001 | 3.777311  4.003016  4.724177  6.187259    6.7516  7.011725  7.033655  6.858971 |  100.0 |    0.33556 | 422.283697 |
    | 2000-01-03 08:30:00 | 100.277186 | 103.176337  115.917418  162.889534  304.049986   399.41649  451.307108  457.903508  425.507582 | 425.507582 | -0.000004  -0.000019  -0.000061  -0.000043  -0.000026  -0.000014  -0.000002   0.000009 | 3.769394  3.969574  4.614539  6.109687  6.704173  6.986394  7.030442  6.874929 |  100.0 |   0.277186 | 425.507582 |
    | 2000-01-03 09:00:00 | 100.228681 | 102.749304  113.975078  155.143281  292.936436  390.662565  446.107471  457.045747  428.393462 | 428.393462 | -0.000004  -0.000016  -0.000054  -0.000049  -0.000027  -0.000015  -0.000003   0.000008 | 3.762501  3.939939  4.516805  6.021505  6.655411   6.95928  7.025112  6.889028 |  100.0 |   0.228681 | 428.393462 |
    | 2000-01-03 09:30:00 | 100.188431 |  102.37849  112.270457  148.425872  280.068307  381.758185  440.604143  455.770498  430.930111 | 430.930111 | -0.000003  -0.000015  -0.000048  -0.000075  -0.000028  -0.000016  -0.000004   0.000007 | 3.756505  3.913691  4.429643  5.886894  6.604494  6.930447   7.01772   6.90124 |  100.0 |   0.188431 | 430.930111 |
    | 2000-01-03 10:00:00 | 100.155078 |  102.05668  110.773693  142.583303  261.202169  372.404967  434.806293  454.090532  433.108854 | 433.108854 | -0.000003  -0.000013  -0.000043  -0.000118  -0.000031  -0.000017  -0.000005   0.000006 | 3.751293  3.890455  4.351884  5.673985  6.548605   6.89981  7.008315  6.911548 |  100.0 |   0.155078 | 433.108854 |
    | 2000-01-03 10:30:00 | 100.127477 | 101.777539  109.458905  137.488118  240.005577  362.243855  428.685937  452.017122  434.923335 | 434.923335 | -0.000003  -0.000011  -0.000039  -0.000116  -0.000034  -0.000018  -0.000006   0.000005 | 3.746766  3.869896    4.2825  5.465833  6.486912  6.867119  6.996928   6.91994 |  100.0 |   0.127477 | 434.923335 |
    | 2000-01-03 11:00:00 | 100.104666 |  101.53556  108.303663  133.034429  221.592886  351.467034   422.21444  449.557583  436.369064 | 436.369064 | -0.000002   -0.00001  -0.000034  -0.000103  -0.000037  -0.000019  -0.000007   0.000004 | 3.742838  3.851715  4.220582   5.28031  6.421056  6.832229  6.983572  6.926411 |  100.0 |   0.104666 | 436.369064 |
    | 2000-01-03 11:30:00 | 100.085838 | 101.325901    107.2884  129.133416  205.948099  340.410163  415.392859  446.717971  437.442987 | 437.442987 | -0.000002  -0.000009  -0.000031  -0.000092  -0.000038  -0.000021  -0.000009   0.000003 | 3.739431  3.835648  4.165326  5.115161  6.352498  6.795159  6.968256  6.930961 |  100.0 |   0.085838 | 437.442987 |
    | 2000-01-03 12:00:00 | 100.070317 | 101.144343  106.396063  125.710353  192.589129  329.329863  408.250211  443.505631  438.143382 | 438.143382 | -0.000002  -0.000008  -0.000027  -0.000082  -0.000039  -0.000022   -0.00001   0.000001 | 3.736478  3.821455  4.116019   4.96794  6.282263   6.75604  6.950999   6.93359 |  100.0 |   0.070317 | 438.143382 |
    | 2000-01-03 12:30:00 | 100.057539 | 100.987206  105.611744  122.701984  181.124738  318.401862   400.83215  439.931201  438.469953 | 438.469953 | -0.000001  -0.000007  -0.000024  -0.000073   -0.00004  -0.000023  -0.000011        0.0 | 3.733921  3.808926  4.072026  4.836535  6.210815  6.715069  6.931838  6.934307 |  100.0 |   0.057539 | 438.469953 |
    | 2000-01-03 13:00:00 | 100.047032 | 100.851276  104.922393  120.054491  171.241657  307.701768  393.191084  436.009328  438.424032 | 438.424032 | -0.000001  -0.000006  -0.000022  -0.000065  -0.000041  -0.000024  -0.000012  -0.000001 | 3.731708  3.797872  4.032781  4.719122  6.137461  6.672453  6.910824  6.933123 |  100.0 |   0.047032 | 438.424032 |
    | 2000-01-03 13:30:00 | 100.038402 | 100.733754  104.316567  117.721875  162.687132  297.092527  385.372672  431.758283   438.00879 |  438.00879 | -0.000001  -0.000005  -0.000019  -0.000058  -0.000044  -0.000025  -0.000013  -0.000002 | 3.729793  3.788125  3.997781  4.614122  6.057937  6.628314  6.888027  6.930057 |  100.0 |   0.038402 |  438.00879 |
    | 2000-01-03 14:00:00 | 100.031322 | 100.632201  103.784229  115.664674   155.25567  285.710797  377.387196    427.1973  437.229295 | 437.229295 | -0.000001  -0.000005  -0.000017  -0.000052  -0.000058  -0.000026  -0.000014  -0.000003 | 3.728139  3.779535  3.966575  4.520155  5.953486  6.582332  6.863506  6.925136 |  100.0 |   0.031322 | 437.229295 |
    | 2000-01-03 14:30:00 | 100.025521 | 100.544484  103.316538  113.848808  148.778666  270.197491  369.069715  422.340189  436.092132 | 436.092132 | -0.000001  -0.000004  -0.000015  -0.000047    -0.0001  -0.000028  -0.000015  -0.000004 | 3.726709  3.771969   3.93876  4.436016  5.772874  6.532573  6.837236  6.918388 |  100.0 |   0.025521 | 436.092132 |
    | 2000-01-03 15:00:00 | 100.020773 |  100.46876  102.905739  112.244873  143.117203  249.549551  360.025046    417.1708  434.603761 | 434.603761 | -0.000001  -0.000004  -0.000014  -0.000042  -0.000117  -0.000031  -0.000016  -0.000005 | 3.725475  3.765308  3.913976  4.360645  5.562452  6.476764  6.809003  6.909828 |  100.0 |   0.020773 | 434.603761 |
    | 2000-01-03 15:30:00 | 100.016892 | 100.403417  102.544998  110.827309  138.155747  230.295562  350.206358  411.652781  432.767805 | 432.767805 | -0.000001  -0.000003  -0.000012  -0.000038  -0.000106  -0.000034  -0.000017  -0.000006 | 3.724409  3.759448  3.891902  4.293108  5.372117  6.415924  6.778583  6.899454 |  100.0 |   0.016892 | 432.767805 |
    | 2000-01-03 16:00:00 | 100.013722 | 100.347062  102.228314  109.573919  133.797859  213.850864  339.931681   405.77106  430.585223 | 430.585223 | -0.000001  -0.000003  -0.000011  -0.000034  -0.000094  -0.000036  -0.000018  -0.000007 |  3.72349  3.754295  3.872249  4.232578  5.202277  6.351623  6.745918  6.887253 |  100.0 |   0.013722 | 430.585223 |
    | 2000-01-03 16:30:00 | 100.011136 | 100.298478  101.950382  108.465276  129.962294  199.786509  329.483039  399.539988  428.057575 | 428.057575 |       0.0  -0.000003   -0.00001   -0.00003  -0.000084  -0.000037  -0.000019  -0.000008 | 3.722698  3.749765  3.854758  4.178322  5.050541  6.285015  6.711079  6.873219 |  100.0 |   0.011136 | 428.057575 |
    | 2000-01-03 17:00:00 | 100.009028 | 100.256612  101.706539  107.484401  126.580408  187.696897  319.061347  392.994638  425.188979 | 425.188979 |       0.0  -0.000002  -0.000009  -0.000027  -0.000075  -0.000038   -0.00002  -0.000009 | 3.722016  3.745787  3.839197  4.129687  4.914782  6.216726  6.674222  6.857359 |  100.0 |   0.009028 | 425.188979 |
    | 2000-01-03 17:30:00 | 100.007313 | 100.220549  101.492675  106.616404  123.593852  177.256153  308.773199  386.180489  421.987336 | 421.987336 |       0.0  -0.000002  -0.000008  -0.000024  -0.000068  -0.000039  -0.000021   -0.00001 | 3.721427  3.742293   3.82536  4.086092  4.793167  6.146388  6.635531  6.839699 |  100.0 |   0.007313 | 421.987336 |
    | 2000-01-03 18:00:00 | 100.005918 | 100.189498  101.305167  105.848207  120.952796  168.201382  298.547745  379.141127  418.464364 | 418.464364 |       0.0  -0.000002  -0.000007  -0.000022  -0.000061  -0.000042  -0.000022  -0.000011 | 3.720921  3.739228  3.813062  4.047018  4.684108  6.070846  6.595136   6.82028 |  100.0 |   0.005918 | 418.464364 |
    | 2000-01-03 18:30:00 | 100.004785 | 100.162773  101.140828  105.168304  118.614508    160.3189  287.767827  371.895723  414.633723 | 414.633723 |       0.0  -0.000001  -0.000006  -0.000019  -0.000054  -0.000052  -0.000023  -0.000012 | 3.720485  3.736539  3.802135  4.012001  4.586223  5.976419  6.552847  6.799145 |  100.0 |   0.004785 | 414.633723 |
    | 2000-01-03 19:00:00 | 100.003865 | 100.139778  100.996841  104.566531  116.542106  153.433434  273.803002  364.332505   410.50639 |  410.50639 |       0.0  -0.000001  -0.000005  -0.000017  -0.000049  -0.000089  -0.000025  -0.000013 |  3.72011  3.734182  3.792431  3.980626  4.498303   5.81709  6.507198  6.776277 |  100.0 |   0.003865 |  410.50639 |
    | 2000-01-03 19:30:00 | 100.003119 | 100.120001  100.870733  104.033943   114.70376  147.400589  254.216196  356.097298  406.070453 | 406.070453 |       0.0  -0.000001  -0.000005  -0.000016  -0.000044  -0.000115  -0.000029  -0.000014 | 3.719788  3.732116  3.783817   3.95252  4.419288  5.610915  6.455642  6.751486 |  100.0 |   0.003119 | 406.070453 |
    | 2000-01-03 20:00:00 | 100.002515 | 100.102996  100.760323  103.562625  113.071802  142.100306  234.864276  347.035532   401.28885 |  401.28885 |       0.0  -0.000001  -0.000004  -0.000014  -0.000039  -0.000106  -0.000032  -0.000015 | 3.719511  3.730307  3.776175  3.927348  4.348243  5.420741  6.398644  6.724524 |  100.0 |   0.002515 |  401.28885 |
    | 2000-01-03 20:30:00 | 100.002026 |  100.08838  100.663696  103.145592  111.622183  137.432363  218.209908  337.424953   396.13881 |  396.13881 |       0.0  -0.000001  -0.000004  -0.000013  -0.000036  -0.000095  -0.000034  -0.000016 | 3.719273  3.728722  3.769396  3.904811  4.284342  5.250584  6.337757  6.695289 |  100.0 |   0.002026 |  396.13881 |
    | 2000-01-03 21:00:00 | 100.001631 | 100.075822   100.57916  102.776647  110.333849  133.312349  203.930113   327.55129  390.626206 | 390.626206 |       0.0  -0.000001  -0.000003  -0.000011  -0.000032  -0.000085  -0.000035  -0.000017 | 3.719068  3.727336  3.763388  3.884639  4.226852  5.098227  6.274163  6.663816 |  100.0 |   0.001631 | 390.626206 |
    | 2000-01-03 21:30:00 | 100.001312 | 100.065034  100.505229  102.450306  109.188374  129.668935   191.62582  317.621284  384.778875 | 384.778875 |       0.0  -0.000001  -0.000003   -0.00001  -0.000029  -0.000076  -0.000036  -0.000019 | 3.718892  3.726123  3.758064  3.866589   4.17512  4.961598  6.208501   6.63023 |  100.0 |   0.001312 | 384.778875 |
    | 2000-01-03 22:00:00 | 100.001054 |  100.05577  100.440596  102.161712  108.169578  126.441491  180.973906  307.745757  378.636276 | 378.636276 |       0.0  -0.000001  -0.000003  -0.000009  -0.000026  -0.000068  -0.000038   -0.00002 | 3.718741  3.725063  3.753348  3.850444  4.128567  4.838905  6.140389  6.594691 |  100.0 |   0.001054 | 378.636276 |
    | 2000-01-03 22:30:00 | 100.000846 | 100.047816  100.384112  101.906558  107.263223  123.578222  171.713246  297.855314  372.237047 | 372.237047 |       0.0  -0.000001  -0.000002  -0.000008  -0.000023  -0.000061  -0.000041  -0.000021 | 3.718612  3.724136  3.749173  3.836008  4.086671  4.728597  6.066632  6.557306 |  100.0 |   0.000846 | 372.237047 |
    | 2000-01-03 23:00:00 | 100.000679 |  100.04099  100.334767  101.681031  106.456759  121.034703  163.631311  287.336373    365.5963 |   365.5963 |       0.0        0.0  -0.000002  -0.000007  -0.000021  -0.000055  -0.000052  -0.000022 |   3.7185  3.723326  3.745479  3.823104  4.048968  4.629326  5.973663  6.517865 |  100.0 |   0.000679 |   365.5963 |
    | 2000-01-03 23:30:00 | 100.000544 | 100.035132  100.291673  101.481736  105.739075  118.772577  156.553562  273.644284  358.602865 | 358.602865 |       0.0        0.0  -0.000002  -0.000006  -0.000019   -0.00005  -0.000087  -0.000024 | 3.718405  3.722619  3.742211  3.811575  4.015041  4.539913  5.817676  6.474921 |  100.0 |   0.000544 | 358.602865 |
    | 2000-01-04 00:00:00 | 100.000436 | 100.030106  100.254051  101.305676  105.100364  116.758718   150.33603  254.544569  350.919079 | 350.919079 |       0.0        0.0  -0.000002  -0.000006  -0.000017  -0.000045  -0.000112  -0.000027 | 3.718323  3.722001  3.739321  3.801277  3.984516  4.459322  5.616843  6.426039 |  100.0 |   0.000436 | 350.919079 |
    | 2000-01-04 00:30:00 | 100.000349 | 100.025795  100.221217  101.150185  104.531923  114.964314  144.858874  235.635611   342.40138 |  342.40138 |       0.0        0.0  -0.000001  -0.000005  -0.000015   -0.00004  -0.000103   -0.00003 | 3.718253  3.721462  3.736767  3.792083  3.957057  4.386642  5.430985  6.371661 |  100.0 |   0.000349 |  342.40138 |
    | 2000-01-04 01:00:00 | 100.000279 | 100.022098  100.192573  101.012906  104.026048  113.364301  140.021941  219.298915  333.314462 | 333.314462 |       0.0        0.0  -0.000001  -0.000005  -0.000014  -0.000036  -0.000093  -0.000032 | 3.718193  3.720992  3.734511  3.783878  3.932359  4.321067  5.264283  6.313276 |  100.0 |   0.000279 | 333.314462 |
    | 2000-01-04 01:30:00 | 100.000223 | 100.018929  100.167591  100.891739  103.575878   111.93671  135.740789  205.244928  323.932097 | 323.932097 |       0.0        0.0  -0.000001  -0.000004  -0.000012  -0.000033  -0.000083  -0.000034 | 3.718141  3.720582  3.732519  3.776558  3.910151   4.26188  5.114667  6.251995 |  100.0 |   0.000223 | 323.932097 |
    | 2000-01-04 02:00:00 | 100.000178 | 100.016212  100.145811  100.784829  103.175321  110.662295  131.943986  193.096618  314.448931 | 314.448931 |       0.0        0.0  -0.000001  -0.000004  -0.000011   -0.00003  -0.000075  -0.000035 | 3.718097  3.720225   3.73076   3.77003  3.890186  4.208445  4.980171   6.18832 |  100.0 |   0.000178 | 314.448931 |
    | 2000-01-04 02:30:00 | 100.000142 | 100.013883  100.126827  100.690526  102.818958  109.524131  128.570747  182.547052  304.954224 | 304.954224 |       0.0        0.0  -0.000001  -0.000003   -0.00001  -0.000027  -0.000067  -0.000037 | 3.718059  3.719913  3.729208  3.764211  3.872243  4.160195  4.859096  6.121522 |  100.0 |   0.000142 | 304.954224 |
    | 2000-01-04 03:00:00 | 100.000113 | 100.011888  100.110287  100.607371  102.501963  108.507297  125.569094  173.347414  295.319393 | 295.319393 |       0.0        0.0  -0.000001  -0.000003  -0.000009  -0.000024  -0.000061  -0.000041 | 3.718026  3.719642  3.727839  3.759026  3.856122   4.11662  4.749967  6.047325 |  100.0 |   0.000113 | 295.319393 |
    | 2000-01-04 03:30:00 |  100.00009 | 100.010178  100.095879  100.534071  102.220044   107.59862  122.894391   165.29462  284.723203 | 284.723203 |       0.0        0.0  -0.000001  -0.000003  -0.000008  -0.000022  -0.000055  -0.000055 | 3.717998  3.719405  3.726631  3.754407  3.841642  4.077267    4.6515  5.948135 |  100.0 |    0.00009 | 284.723203 |
    | 2000-01-04 04:00:00 | 100.000072 | 100.008713  100.083333  100.469476  101.969364  106.786415  120.508062  158.221451  270.233905 | 270.233905 |       0.0        0.0  -0.000001  -0.000002  -0.000007   -0.00002  -0.000049  -0.000093 | 3.717975  3.719199  3.725567  3.750295  3.828641  4.041726  4.562575  5.781052 |  100.0 |   0.000072 | 270.233905 |
    | 2000-01-04 04:30:00 | 100.000057 | 100.007459   100.07241  100.412571  101.746512  106.060335  118.376731  151.989577  251.156022 | 251.156022 |       0.0        0.0  -0.000001  -0.000002  -0.000006  -0.000018  -0.000045  -0.000108 | 3.717954   3.71902   3.72463  3.746635   3.81697  4.009629  4.482204  5.586135 |  100.0 |   0.000057 | 251.156022 |
    | 2000-01-04 05:00:00 | 100.000046 | 100.006384  100.062905  100.362458  101.548451  105.411204    116.4714   146.48376  233.106266 | 233.106266 |       0.0        0.0        0.0  -0.000002  -0.000006  -0.000016   -0.00004  -0.000099 | 3.717937  3.718864  3.723804  3.743379  3.806498  3.980646  4.409518  5.408013 |  100.0 |   0.000046 | 233.106266 |
    | 2000-01-04 05:30:00 | 100.000036 | 100.005464  100.054634  100.318339  101.372466  104.830837   114.76671  141.607204  217.525356 | 217.525356 |       0.0        0.0        0.0  -0.000002  -0.000005  -0.000015  -0.000037  -0.000089 | 3.717922  3.718728  3.723077  3.740484  3.797105  3.954477  4.343749  5.247911 |  100.0 |   0.000036 | 217.525356 |
    | 2000-01-04 06:00:00 | 100.000029 | 100.004676  100.047441  100.279511  101.216136  104.311951  113.240471    137.2783  204.067467 | 204.067467 |       0.0        0.0        0.0  -0.000001  -0.000005  -0.000013  -0.000033   -0.00008 | 3.717909   3.71861  3.722437  3.737909  3.788683  3.930855  4.284212  5.103845 |  100.0 |   0.000029 | 204.067467 |
    | 2000-01-04 06:30:00 | 100.000023 | 100.004001  100.041185   100.24535  101.077305  103.848054  111.873197  133.427825   192.38841 |  192.38841 |       0.0        0.0        0.0  -0.000001  -0.000004  -0.000012   -0.00003  -0.000072 | 3.717898  3.718508  3.721874  3.735622  3.781133  3.909534    4.2303  4.974003 |  100.0 |   0.000023 |  192.38841 |
    | 2000-01-04 07:00:00 | 100.000018 | 100.003424  100.035746  100.215303  100.954048  103.433348  110.647722  129.996755  182.207865 | 182.207865 |       0.0        0.0        0.0  -0.000001  -0.000004  -0.000011  -0.000027  -0.000065 | 3.717888  3.718419  3.721379  3.733589  3.774369  3.890295  4.181469   4.85681 |  100.0 |   0.000018 | 182.207865 |
    | 2000-01-04 07:30:00 | 100.000014 |  100.00293  100.031019  100.188884  100.844649  103.062659  109.548889  126.934518  173.297625 | 173.297625 |       0.0        0.0        0.0  -0.000001  -0.000003   -0.00001  -0.000025  -0.000059 |  3.71788  3.718341  3.720944  3.731785   3.76831   3.87294  4.137231  4.750902 |  100.0 |   0.000014 | 173.297625 |
    | 2000-01-04 08:00:00 | 100.000011 | 100.002506  100.026912  100.165661  100.747579  102.731358  108.563284  124.197586  165.470587 | 165.470587 |       0.0        0.0        0.0  -0.000001  -0.000003  -0.000009  -0.000022  -0.000053 | 3.717874  3.718274  3.720561  3.730183  3.762884  3.857287  4.097152  4.655088 |  100.0 |   0.000011 | 165.470587 |
    | 2000-01-04 08:30:00 | 100.000009 | 100.002144  100.023344  100.145255  100.661474  102.435307  107.679006  121.748329  158.572261 | 158.572261 |       0.0        0.0        0.0  -0.000001  -0.000003  -0.000008   -0.00002  -0.000048 | 3.717868  3.718215  3.720225  3.728761  3.758029  3.843173  4.060838  4.568326 |  100.0 |   0.000009 | 158.572261 |
    | 2000-01-04 09:00:00 | 100.000007 | 100.001834  100.020245  100.127329  100.585118  102.170804   106.88548  119.554083  152.474225 | 152.474225 |       0.0        0.0        0.0  -0.000001  -0.000002  -0.000007  -0.000018  -0.000044 | 3.717863  3.718164   3.71993    3.7275  3.753684  3.830452  4.027934  4.489698 |  100.0 |   0.000007 | 152.474225 |
    | 2000-01-04 09:30:00 | 100.000006 | 100.001569  100.017554  100.111586  100.517427  101.934533  106.173291  117.586392  147.069052 | 147.069052 |       0.0        0.0        0.0  -0.000001  -0.000002  -0.000006  -0.000017   -0.00004 | 3.717858   3.71812   3.71967  3.726382  3.749799  3.818989  3.998123  4.418393 |  100.0 |   0.000006 | 147.069052 |
    | 2000-01-04 10:00:00 | 100.000004 | 100.001342  100.015218  100.097764  100.457436  101.723527  105.534044  115.820373  142.266334 | 142.266334 |       0.0        0.0        0.0  -0.000001  -0.000002  -0.000006  -0.000015  -0.000036 | 3.717855  3.718082  3.719442  3.725391  3.746325  3.808663  3.971116  4.353694 |  100.0 |   0.000004 | 142.266334 |
    | 2000-01-04 10:30:00 | 100.000004 | 100.001148   100.01319  100.085633  100.404286  101.535126  104.960241  114.234196  137.989561 | 137.989561 |       0.0        0.0        0.0        0.0  -0.000002  -0.000005  -0.000014  -0.000033 | 3.717851  3.718049  3.719243  3.724513   3.74322  3.799364  3.946653  4.294962 |  100.0 |   0.000004 | 137.989561 |
    | 2000-01-04 11:00:00 | 100.000003 | 100.000982   100.01143  100.074988  100.357211   101.36695  104.445182  112.808653  134.173634 | 134.173634 |       0.0        0.0        0.0        0.0  -0.000002  -0.000005  -0.000012   -0.00003 | 3.717849   3.71802  3.719067  3.723735  3.740446  3.790993  3.924496  4.241626 |  100.0 |   0.000003 | 134.173634 |
    | 2000-01-04 11:30:00 | 100.000002 |  100.00084  100.009904   100.06565  100.315529  101.216865  103.982868  111.526792  130.762886 | 130.762886 |       0.0        0.0        0.0        0.0  -0.000001  -0.000004  -0.000011  -0.000027 | 3.717846  3.717995  3.718913  3.723046  3.737969  3.783459  3.904433  4.193177 |  100.0 |   0.000002 | 130.762886 |
    | 2000-01-04 12:00:00 | 100.000002 | 100.000718   100.00858  100.057462  100.278633   101.08296  103.567925  110.373616  127.709487 | 127.709487 |       0.0        0.0        0.0        0.0  -0.000001  -0.000004   -0.00001  -0.000024 | 3.717844  3.717973  3.718778  3.722436  3.735757  3.776682  3.886268  4.149159 |  100.0 |   0.000002 | 127.709487 |
    | 2000-01-04 12:30:00 | 100.000001 | 100.000614  100.007431  100.050282  100.245984  100.963522  103.195536  109.335821  124.972158 | 124.972158 |       0.0        0.0        0.0        0.0  -0.000001  -0.000003  -0.000009  -0.000022 | 3.717843  3.717954   3.71866  3.721897  3.733783  3.770588  3.869828  4.109159 |  100.0 |   0.000001 | 124.972158 |
    | 2000-01-04 13:00:00 | 100.000001 | 100.000525  100.006436   100.04399  100.217101  100.857018  102.861377  108.401583  122.515121 | 122.515121 |       0.0        0.0        0.0        0.0  -0.000001  -0.000003  -0.000008   -0.00002 | 3.717841  3.717938  3.718557  3.721419  3.732022  3.765109   3.85495  4.072809 |  100.0 |   0.000001 | 122.515121 |
    | 2000-01-04 13:30:00 | 100.000001 | 100.000449  100.005573  100.038476  100.191558  100.762074  102.561568  107.560372  120.307243 | 120.307243 |       0.0        0.0        0.0        0.0  -0.000001  -0.000003  -0.000007  -0.000018 |  3.71784  3.717924  3.718466  3.720997  3.730452  3.760185  3.841491  4.039774 |  100.0 |   0.000001 | 120.307243 |
    | 2000-01-04 14:00:00 | 100.000001 | 100.000384  100.004824  100.033645  100.168974  100.677458  102.292622  106.802789  118.321331 | 118.321331 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000007  -0.000017 | 3.717839  3.717912  3.718386  3.720624  3.729052  3.755761  3.829318  4.009751 |  100.0 |   0.000001 | 118.321331 |
    | 2000-01-04 14:30:00 | 100.000001 | 100.000329  100.004176  100.029415  100.149014  100.602069  102.051407  106.120437  116.533553 | 116.533553 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000006  -0.000015 | 3.717838  3.717901  3.718317  3.720294  3.727804  3.751789  3.818312  3.982468 |  100.0 |   0.000001 | 116.533553 |
    | 2000-01-04 15:00:00 |      100.0 | 100.000281  100.003614  100.025711  100.131377  100.534921  101.835108  105.505795  114.922951 | 114.922951 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000006  -0.000014 | 3.717837  3.717892  3.718256  3.720003  3.726692  3.748223  3.808364  3.957678 |  100.0 |        0.0 | 114.922951 |
    | 2000-01-04 15:30:00 |      100.0 |  100.00024  100.003128  100.022468  100.115798  100.475129  101.641192  104.952127  113.471039 | 113.471039 |       0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000005  -0.000013 | 3.717837  3.717884  3.718202  3.719745  3.725702  3.745022  3.799375  3.935154 |  100.0 |        0.0 | 113.471039 |
    | 2000-01-04 16:00:00 |      100.0 | 100.000205  100.002706   100.01963   100.10204  100.421903  101.467383  104.453386  112.161466 | 112.161466 |       0.0        0.0        0.0        0.0        0.0  -0.000002  -0.000005  -0.000011 | 3.717836  3.717877  3.718156  3.719518  3.724821  3.742151  3.791255  3.914694 |  100.0 |        0.0 | 112.161466 |
    | 2000-01-04 16:30:00 |      100.0 | 100.000176  100.002341  100.017147  100.089894  100.374537  101.311634  104.004139  110.979732 | 110.979732 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000004   -0.00001 | 3.717836  3.717871  3.718115  3.719317  3.724036  3.739576  3.783922  3.896111 |  100.0 |        0.0 | 110.979732 |
    | 2000-01-04 17:00:00 |      100.0 |  100.00015  100.002025  100.014975  100.079174  100.332398  101.172101  103.599502  109.912946 | 109.912946 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000004  -0.000009 | 3.717835  3.717866  3.718079   3.71914  3.723338  3.737268  3.777303  3.879236 |  100.0 |        0.0 | 109.912946 |
    | 2000-01-04 17:30:00 |      100.0 | 100.000128  100.001751  100.013076  100.069715  100.294919   101.04713  103.235082  108.949621 | 108.949621 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000003  -0.000009 | 3.717835  3.717861  3.718048  3.718984  3.722717  3.735199  3.771329  3.863917 |  100.0 |        0.0 | 108.949621 |
    | 2000-01-04 18:00:00 |      100.0 |  100.00011  100.001515  100.011415  100.061371  100.261595  100.935229  102.906922  108.079504 | 108.079504 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000003  -0.000008 | 3.717835  3.717858   3.71802  3.718846  3.722165  3.733346   3.76594  3.850013 |  100.0 |        0.0 | 108.079504 |
    | 2000-01-04 18:30:00 |      100.0 | 100.000094   100.00131  100.009963  100.054013  100.231974  100.835058  102.611455  107.293422 | 107.293422 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000003  -0.000007 | 3.717834  3.717854  3.717997  3.718724  3.721674  3.731686   3.76108  3.837396 |  100.0 |        0.0 | 107.293422 |
    | 2000-01-04 19:00:00 |      100.0 |  100.00008  100.001132  100.008694  100.047526  100.205653  100.745413  102.345469  106.583157 | 106.583157 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000006 | 3.717834  3.717851  3.717976  3.718617  3.721238  3.730201  3.756699   3.82595 |  100.0 |        0.0 | 106.583157 |
    | 2000-01-04 19:30:00 |      100.0 | 100.000069  100.000979  100.007586  100.041808   100.18227   100.66521  102.106064  105.941336 | 105.941336 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000006 | 3.717834  3.717849  3.717957  3.718523  3.720851  3.728871   3.75275  3.815571 |  100.0 |        0.0 | 105.941336 |
    | 2000-01-04 20:00:00 |      100.0 | 100.000059  100.000846  100.006617  100.036769  100.161503  100.593473  101.890628  105.361332 | 105.361332 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000005 | 3.717834  3.717847  3.717941  3.718439  3.720507  3.727681  3.749193   3.80616 |  100.0 |        0.0 | 105.361332 |
    | 2000-01-04 20:30:00 |      100.0 |  100.00005  100.000731  100.005771  100.032331  100.143065  100.529329  101.696802  104.837183 | 104.837183 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000005 | 3.717834  3.717845  3.717928  3.718366  3.720201  3.726617  3.745989  3.797631 |  100.0 |        0.0 | 104.837183 |
    | 2000-01-04 21:00:00 |      100.0 | 100.000043  100.000632  100.005032  100.028421  100.126699  100.471988  101.522455  104.363521 | 104.363521 |       0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000002  -0.000004 | 3.717833  3.717843  3.717915  3.718302   3.71993  3.725666  3.743105  3.789903 |  100.0 |        0.0 | 104.363521 |
    | 2000-01-04 21:30:00 |      100.0 | 100.000037  100.000546  100.004388  100.024979  100.112177  100.420745  101.365668  103.935504 | 103.935504 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000004 | 3.717833  3.717842  3.717905  3.718245   3.71969  3.724815  3.740509  3.782902 |  100.0 |        0.0 | 103.935504 |
    | 2000-01-04 22:00:00 |      100.0 | 100.000031  100.000472  100.003825  100.021949  100.099295  100.374964  101.224705  103.548765 | 103.548765 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000004 | 3.717833   3.71784  3.717896  3.718195  3.719476  3.724055  3.738174  3.776564 |  100.0 |        0.0 | 103.548765 |
    | 2000-01-04 22:30:00 |      100.0 | 100.000027  100.000407  100.003333  100.019282   100.08787  100.334074  101.098001   103.19936 |  103.19936 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000003 | 3.717833  3.717839  3.717888  3.718151  3.719287  3.723377  3.736073  3.770826 |  100.0 |        0.0 |  103.19936 |
    | 2000-01-04 23:00:00 |      100.0 | 100.000023  100.000352  100.002905  100.016936   100.07774  100.297564  100.984143  102.883726 | 102.883726 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000003 | 3.717833  3.717839   3.71788  3.718112  3.719119  3.722771  3.734185  3.765634 |  100.0 |        0.0 | 102.883726 |
    | 2000-01-04 23:30:00 |      100.0 |  100.00002  100.000304  100.002531  100.014872  100.068761  100.264973  100.881856  102.598641 | 102.598641 |       0.0        0.0        0.0        0.0        0.0        0.0  -0.000001  -0.000003 | 3.717833  3.717838  3.717874  3.718078   3.71897   3.72223  3.732487  3.760937 |  100.0 |        0.0 | 102.598641 |

.. _lstream_v001_negative_water_stage:

negative water stages
_____________________

Finally, we show that |lstream_v001| allows for negative stages.
Therefore, we reset |HM| to infinity, set the base flow provided by
node `input1` to zero, and set the initial water stage to -1.0 m:

>>> hm(inf)
>>> nodes.input1.sequences.sim.series = 0.0
>>> test.inits.h = -1.0

In usual simulations with reasonable smoothing parameterisations, one is
unlikely to encounter negative stages.  However, they are principally
allowed for the sake of numerical robustness.  Careful inspection of the
stage (|H|) and the discharge (|QG|) of a cross-section at the beginning
of the rising limb reveals that the first discharge occurs for stages
slightly below zero (due to setting the regularisation parameter |HR|
to 0.1 m):

.. integration-test::

    >>> with pub.options.reprdigits(2):
    ...     test('lstream_v001_negative_water_stage')
    |                date |     qz |                                                             qg |     qa |                                     dh |                                                      h | input1 | input2 | output |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 00:30:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 01:00:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 01:30:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 02:00:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 02:30:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.99   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 03:00:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.93   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 03:30:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.73   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 04:00:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.61   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 04:30:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.51   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 05:00:00 |    0.0 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.43   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.0 |    0.0 |
    | 2000-01-01 05:30:00 |   0.01 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.36  -0.97   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |   0.01 |    0.0 |
    | 2000-01-01 06:00:00 |   0.03 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.29  -0.84   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |   0.03 |    0.0 |
    | 2000-01-01 06:30:00 |   0.08 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.23  -0.69   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |   0.08 |    0.0 |
    | 2000-01-01 07:00:00 |   0.18 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.18  -0.59   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |   0.18 |    0.0 |
    | 2000-01-01 07:30:00 |    0.4 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.13   -0.5   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.4 |    0.0 |
    | 2000-01-01 08:00:00 |    0.8 |    0.0     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.08  -0.42   -1.0   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |    0.8 |    0.0 |
    | 2000-01-01 08:30:00 |   1.51 |   0.01     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 | -0.03  -0.35  -0.97   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |   1.51 |    0.0 |
    | 2000-01-01 09:00:00 |   2.69 |   0.03     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.02  -0.28  -0.83   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |   2.69 |    0.0 |
    | 2000-01-01 09:30:00 |   4.59 |   0.09     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.08  -0.22  -0.68   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |   4.59 |    0.0 |
    | 2000-01-01 10:00:00 |   7.47 |   0.22     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.15  -0.17  -0.58   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |   7.47 |    0.0 |
    | 2000-01-01 10:30:00 |  11.68 |   0.51     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.25  -0.11  -0.49   -1.0   -1.0   -1.0   -1.0   -1.0 |    0.0 |  11.68 |    0.0 |
    | 2000-01-01 11:00:00 |  17.61 |   1.06     0.0     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.38  -0.06  -0.41  -0.99   -1.0   -1.0   -1.0   -1.0 |    0.0 |  17.61 |    0.0 |
    | 2000-01-01 11:30:00 |   25.7 |   2.07    0.02     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.56  -0.01  -0.33  -0.95   -1.0   -1.0   -1.0   -1.0 |    0.0 |   25.7 |    0.0 |
    | 2000-01-01 12:00:00 |  36.39 |   3.79    0.05     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.77   0.05  -0.26  -0.77   -1.0   -1.0   -1.0   -1.0 |    0.0 |  36.39 |    0.0 |
    | 2000-01-01 12:30:00 |   50.1 |   6.54    0.15     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.03   0.12   -0.2  -0.64   -1.0   -1.0   -1.0   -1.0 |    0.0 |   50.1 |    0.0 |
    | 2000-01-01 13:00:00 |  67.24 |  10.71    0.37     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.34   0.22  -0.14  -0.53   -1.0   -1.0   -1.0   -1.0 |    0.0 |  67.24 |    0.0 |
    | 2000-01-01 13:30:00 |  88.13 |  16.75    0.85     0.0     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.68   0.35  -0.08  -0.44   -1.0   -1.0   -1.0   -1.0 |    0.0 |  88.13 |    0.0 |
    | 2000-01-01 14:00:00 | 112.99 |  25.15    1.77    0.01     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.05   0.52  -0.02  -0.36  -0.98   -1.0   -1.0   -1.0 |    0.0 | 112.99 |    0.0 |
    | 2000-01-01 14:30:00 | 141.95 |  36.41    3.41    0.04     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.46   0.74   0.04  -0.28  -0.83   -1.0   -1.0   -1.0 |    0.0 | 141.95 |    0.0 |
    | 2000-01-01 15:00:00 | 174.96 |  51.03    6.14    0.12     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.89   1.02   0.11  -0.21  -0.67   -1.0   -1.0   -1.0 |    0.0 | 174.96 |    0.0 |
    | 2000-01-01 15:30:00 | 211.85 |  69.43   10.39    0.32     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  3.33   1.33    0.2  -0.15  -0.56   -1.0   -1.0   -1.0 |    0.0 | 211.85 |    0.0 |
    | 2000-01-01 16:00:00 | 252.27 |  91.94   16.67    0.77     0.0     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  3.79   1.69   0.33  -0.09  -0.46   -1.0   -1.0   -1.0 |    0.0 | 252.27 |    0.0 |
    | 2000-01-01 16:30:00 | 295.75 | 118.75   25.53    1.68    0.01     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  4.25   2.09   0.51  -0.03  -0.37  -0.98   -1.0   -1.0 |    0.0 | 295.75 |    0.0 |
    | 2000-01-01 17:00:00 | 341.66 | 149.91   37.54    3.36    0.04     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  4.71   2.51   0.74   0.03  -0.29  -0.86   -1.0   -1.0 |    0.0 | 341.66 |    0.0 |
    | 2000-01-01 17:30:00 | 389.28 | 185.24   53.21     6.2    0.11     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  5.17   2.96   1.03    0.1  -0.22  -0.68   -1.0   -1.0 |    0.0 | 389.28 |    0.0 |
    | 2000-01-01 18:00:00 | 437.77 | 224.42   72.98    10.7    0.32     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  5.61   3.43   1.36    0.2  -0.15  -0.56   -1.0   -1.0 |    0.0 | 437.77 |    0.0 |
    | 2000-01-01 18:30:00 | 486.26 | 266.91   97.14    17.4    0.79     0.0     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  6.04    3.9   1.74   0.34  -0.09  -0.46   -1.0   -1.0 |    0.0 | 486.26 |    0.0 |
    | 2000-01-01 19:00:00 | 533.85 | 312.04  125.81   26.92    1.76    0.01     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  6.46   4.38   2.15   0.53  -0.03  -0.37  -0.99   -1.0 |    0.0 | 533.85 |    0.0 |
    | 2000-01-01 19:30:00 | 579.63 | 358.98   158.9   39.84    3.56    0.04     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  6.85   4.85   2.59   0.77   0.04  -0.28  -0.86   -1.0 |    0.0 | 579.63 |    0.0 |
    | 2000-01-01 20:00:00 | 622.74 | 406.84   196.1   56.68    6.64    0.12     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  7.21   5.31   3.06   1.07   0.11  -0.21  -0.67   -1.0 |    0.0 | 622.74 |    0.0 |
    | 2000-01-01 20:30:00 | 662.41 | 454.66  236.88   77.83   11.52    0.35     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  7.55   5.76   3.53   1.42   0.22  -0.14  -0.55   -1.0 |    0.0 | 662.41 |    0.0 |
    | 2000-01-01 21:00:00 | 697.92 | 501.47  280.54  103.53    18.8    0.87     0.0     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  7.86   6.18   4.02   1.81   0.36  -0.08  -0.45   -1.0 |    0.0 | 697.92 |    0.0 |
    | 2000-01-01 21:30:00 |  728.7 | 546.33  326.21  133.76   29.09    1.96    0.01     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.13   6.58    4.5   2.24   0.56  -0.02  -0.36  -0.98 |    0.0 |  728.7 |    0.0 |
    | 2000-01-01 22:00:00 | 754.28 | 588.39  372.95  168.29    43.0    3.96    0.04     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.38   6.96   4.97   2.69   0.82   0.05  -0.27  -0.83 |    0.0 | 754.28 |    0.0 |
    | 2000-01-01 22:30:00 | 774.33 | 626.86  419.73  206.64   60.99    7.36    0.14     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.59    7.3   5.43   3.16   1.13   0.13   -0.2  -0.66 |    0.0 | 774.33 |    0.0 |
    | 2000-01-01 23:00:00 | 788.65 |  661.1  465.55  248.12   83.39   12.72    0.41     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.77   7.62   5.87   3.65   1.49   0.24  -0.13  -0.54 |    0.0 | 788.65 |    0.0 |
    | 2000-01-01 23:30:00 | 797.19 | 690.58  509.46  291.88  110.32   20.65    1.02     0.0     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.92    7.9   6.28   4.13    1.9   0.39  -0.07  -0.43 |    0.0 | 797.19 |    0.0 |
    | 2000-01-02 00:00:00 |  800.0 | 714.93  550.58  336.94  141.63   31.76    2.26    0.01     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  9.04   8.15   6.67   4.61   2.33   0.61  -0.01  -0.34 |    0.0 |  800.0 |    0.0 |
    | 2000-01-02 00:30:00 | 797.26 | 733.89  588.15  382.27  176.94    46.6    4.52    0.06     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  9.12   8.36   7.02   5.08    2.8   0.88   0.06  -0.26 |    0.0 | 797.26 |    0.0 |
    | 2000-01-02 01:00:00 | 789.26 | 747.38  621.55  426.85  215.63   65.58     8.3    0.18     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  9.18   8.55   7.35   5.53   3.27    1.2   0.15  -0.19 |    0.0 | 789.26 |    0.0 |
    | 2000-01-02 01:30:00 | 776.35 | 755.42   650.3  469.69  256.88   88.94   14.17     0.5     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  9.21    8.7   7.63   5.95   3.75   1.57   0.26  -0.12 |    0.0 | 776.35 |    0.0 |
    | 2000-01-02 02:00:00 | 758.97 | 758.13  674.08  509.92  299.76  116.65   22.74    1.21     0.0 |    0.0 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  9.21   8.81   7.89   6.34   4.23   1.99   0.43  -0.06 |    0.0 | 758.97 |    0.0 |
    | 2000-01-02 02:30:00 |  737.6 | 755.77  692.72  546.79  343.23  148.46   34.58    2.63    0.02 |   0.02 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  9.18    8.9   8.11   6.71    4.7   2.43   0.66    0.0 |    0.0 |  737.6 |   0.02 |
    | 2000-01-02 03:00:00 | 712.76 | 748.66  706.16   579.7  386.28  183.84   50.18    5.16    0.07 |   0.07 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  9.13   8.96    8.3   7.04   5.16   2.89   0.94   0.08 |    0.0 | 712.76 |   0.07 |
    | 2000-01-02 03:30:00 | 684.98 | 737.17  714.48  608.18  427.92  222.08   69.88    9.32    0.22 |   0.22 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  9.06   8.99   8.45   7.34   5.59   3.37   1.27   0.17 |    0.0 | 684.98 |   0.22 |
    | 2000-01-02 04:00:00 | 654.82 | 721.75  717.86  631.95  467.28   262.3   93.78   15.67     0.6 |    0.6 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.97    9.0   8.58   7.61   5.99   3.84   1.65    0.3 |    0.0 | 654.82 |    0.6 |
    | 2000-01-02 04:30:00 | 622.79 | 702.86  716.57  650.85  503.59  303.52  121.77   24.78    1.43 |   1.43 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.86   8.98   8.67   7.84   6.37   4.31   2.07   0.47 |    0.0 | 622.79 |   1.43 |
    | 2000-01-02 05:00:00 | 589.43 | 680.98  710.95  664.87  536.26  344.75  153.46   37.19    3.03 |   3.03 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.73   8.94   8.74   8.03   6.71   4.77   2.51   0.71 |    0.0 | 589.43 |   3.03 |
    | 2000-01-02 05:30:00 |  555.2 | 656.62  701.37   674.1  564.82  385.01  188.28    53.3    5.82 |   5.82 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.59   8.87   8.77    8.2   7.02    5.2   2.97    1.0 |    0.0 |  555.2 |   5.82 |
    | 2000-01-02 06:00:00 | 520.56 | 630.23  688.26  678.72  588.99  423.43  225.43   73.35   10.31 |  10.31 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.43   8.79   8.79   8.33    7.3   5.62   3.44   1.34 |    0.0 | 520.56 |  10.31 |
    | 2000-01-02 06:30:00 | 485.92 | 602.29  672.05  679.02  608.62  459.23  264.02   97.36   17.03 |  17.03 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.26    8.7   8.78   8.43   7.54    6.0    3.9   1.72 |    0.0 | 485.92 |  17.03 |
    | 2000-01-02 07:00:00 | 451.64 | 573.23  653.19  675.32  623.67   491.8  303.11  125.12   26.53 |  26.53 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  8.08   8.58   8.75    8.5   7.75   6.35   4.36   2.14 |    0.0 | 451.64 |  26.53 |
    | 2000-01-02 07:30:00 | 418.03 | 543.46  632.12  667.97  634.25  520.66  341.76  156.18   39.26 |  39.26 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  7.89   8.45    8.7   8.55   7.92   6.68    4.8   2.57 |    0.0 | 418.03 |  39.26 |
    | 2000-01-02 08:00:00 | 385.37 | 513.34  609.26  657.38  640.52   545.5  379.09  189.91   55.57 |  55.57 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  7.69   8.31   8.63   8.58   8.06   6.96   5.21   3.02 |    0.0 | 385.37 |  55.57 |
    | 2000-01-02 08:30:00 | 353.87 | 483.21  585.01  643.95  642.75  566.15  414.32  225.52   75.61 |  75.61 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  7.49   8.16   8.54   8.58   8.18   7.22   5.61   3.48 |    0.0 | 353.87 |  75.61 |
    | 2000-01-02 09:00:00 | 323.72 | 453.36  559.74  628.08  641.23  582.55   446.8  262.14   99.31 |  99.31 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  7.28    8.0   8.45   8.56   8.27   7.44   5.97   3.93 |    0.0 | 323.72 |  99.31 |
    | 2000-01-02 09:30:00 | 295.05 | 424.03  533.81  610.16   636.3  594.76  476.04  298.88  126.42 | 126.42 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  7.06   7.84   8.33   8.52   8.33   7.63   6.31   4.37 |    0.0 | 295.05 | 126.42 |
    | 2000-01-02 10:00:00 | 267.96 | 395.45  507.53  590.58  628.33  602.95  501.68  334.89  156.45 | 156.45 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  6.85   7.66   8.21   8.47   8.36   7.79   6.61   4.79 |    0.0 | 267.96 | 156.45 |
    | 2000-01-02 10:30:00 | 242.52 | 367.78  481.16  569.68  617.69  607.32   523.5  369.37  188.74 | 188.74 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  6.63   7.48   8.08    8.4   8.38   7.92   6.88   5.19 |    0.0 | 242.52 | 188.74 |
    | 2000-01-02 11:00:00 | 218.76 | 341.16  454.95   547.8  604.73  608.15  541.41  401.67  222.55 | 222.55 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  6.41    7.3   7.94   8.32   8.37   8.02   7.11   5.57 |    0.0 | 218.76 | 222.55 |
    | 2000-01-02 11:30:00 | 196.69 | 315.71  429.12  525.24  589.82  605.74  555.44  431.23  257.05 | 257.05 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  6.19   7.11   7.79   8.22   8.35   8.09   7.32   5.91 |    0.0 | 196.69 | 257.05 |
    | 2000-01-02 12:00:00 | 176.28 | 291.49  403.83  502.26  573.29  600.43  565.67  457.66  291.42 | 291.42 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  5.97   6.92   7.64   8.11   8.31   8.15   7.49   6.23 |    0.0 | 176.28 | 291.42 |
    | 2000-01-02 12:30:00 |  157.5 | 268.57  379.23  479.11  555.46  592.54  572.31  480.69  324.89 | 324.89 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  5.75   6.73   7.48    8.0   8.26   8.18   7.64   6.51 |    0.0 |  157.5 | 324.89 |
    | 2000-01-02 13:00:00 |  140.3 | 246.97  355.44   456.0  536.63   582.4  575.57  500.16  356.78 | 356.78 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  5.54   6.54   7.32   7.88   8.19   8.19   7.76   6.77 |    0.0 |  140.3 | 356.78 |
    | 2000-01-02 13:30:00 | 124.61 | 226.69  332.55   433.1  517.07  570.35  575.72  516.04   386.5 |  386.5 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  5.32   6.34   7.15   7.75   8.11   8.18   7.85   6.99 |    0.0 | 124.61 |  386.5 |
    | 2000-01-02 14:00:00 | 110.37 | 207.73  310.63  410.57  497.01  556.69  573.06  528.39  413.59 | 413.59 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  5.11   6.15   6.98   7.61   8.02   8.15   7.92   7.18 |    0.0 | 110.37 | 413.59 |
    | 2000-01-02 14:30:00 |  97.48 | 190.07  289.73  388.53  476.67   541.7  567.88  537.33  437.73 | 437.73 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  4.91   5.96   6.81   7.47   7.92   8.11   7.96   7.35 |    0.0 |  97.48 | 437.73 |
    | 2000-01-02 15:00:00 |  85.86 | 173.67  269.87   367.1  456.24  525.67  560.49  543.04   458.7 |  458.7 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  4.71   5.78   6.64   7.33   7.81   8.06   7.99   7.48 |    0.0 |  85.86 |  458.7 |
    | 2000-01-02 15:30:00 |  75.43 | 158.48  251.07  346.36  435.87  508.82  551.18  545.76  476.41 | 476.41 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  4.51   5.59   6.47   7.18    7.7    8.0    8.0   7.59 |    0.0 |  75.43 | 476.41 |
    | 2000-01-02 16:00:00 |   66.1 | 144.46  233.32  326.36  415.72  491.39  540.23  545.73  490.83 | 490.83 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  4.32   5.41    6.3   7.03   7.58   7.92   7.99   7.68 |    0.0 |   66.1 | 490.83 |
    | 2000-01-02 16:30:00 |  57.78 | 131.55  216.62  307.16   395.9  473.56  527.92  543.22  502.06 | 502.06 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  4.14   5.23   6.13   6.88   7.46   7.84   7.97   7.74 |    0.0 |  57.78 | 502.06 |
    | 2000-01-02 17:00:00 |  50.38 |  119.7  200.95  288.79   376.5  455.52  514.48   538.5   510.2 |  510.2 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  3.96   5.05   5.97   6.73   7.33   7.75   7.93   7.79 |    0.0 |  50.38 |  510.2 |
    | 2000-01-02 17:30:00 |  43.82 | 108.83  186.28  271.26  357.61  437.41  500.15  531.83  515.45 | 515.45 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  3.78   4.88    5.8   6.57    7.2   7.65   7.88   7.81 |    0.0 |  43.82 | 515.45 |
    | 2000-01-02 18:00:00 |  38.03 |  98.88  172.57  254.59  339.29  419.38  485.13  523.49  518.02 | 518.02 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  3.62   4.71   5.64   6.42   7.06   7.55   7.82   7.82 |    0.0 |  38.03 | 518.02 |
    | 2000-01-02 18:30:00 |  32.93 |   89.8  159.78  238.77  321.58  401.51  469.61  513.72  518.12 | 518.12 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  3.45   4.55   5.47   6.27   6.93   7.44   7.76   7.81 |    0.0 |  32.93 | 518.12 |
    | 2000-01-02 19:00:00 |  28.45 |  81.53  147.88   223.8  304.53  383.93  453.75  502.75  516.01 | 516.01 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   3.3   4.38   5.32   6.12   6.79   7.32   7.68   7.79 |    0.0 |  28.45 | 516.01 |
    | 2000-01-02 19:30:00 |  24.53 |  73.99  136.82  209.65  288.15  366.69   437.7  490.79  511.93 | 511.93 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  3.14   4.23   5.16   5.97   6.66   7.21    7.6   7.76 |    0.0 |  24.53 | 511.93 |
    | 2000-01-02 20:00:00 |   21.1 |  67.14  126.56  196.31  272.46  349.87  421.59  478.04  506.12 | 506.12 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   3.0   4.08   5.01   5.82   6.52   7.09   7.51   7.72 |    0.0 |   21.1 | 506.12 |
    | 2000-01-02 20:30:00 |  18.12 |  60.93  117.05  183.75  257.47  333.51  405.51  464.67   498.8 |  498.8 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.86   3.93   4.86   5.67   6.38   6.97   7.41   7.66 |    0.0 |  18.12 |  498.8 |
    | 2000-01-02 21:00:00 |  15.52 |  55.29  108.25  171.95  243.18  317.66  389.58  450.85   490.2 |  490.2 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.73   3.79   4.71   5.53   6.24   6.84   7.31    7.6 |    0.0 |  15.52 |  490.2 |
    | 2000-01-02 21:30:00 |  13.27 |  50.18   100.1  160.87  229.58  302.34  373.86  436.72  480.51 | 480.51 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   2.6   3.65   4.57   5.39   6.11   6.72   7.21   7.53 |    0.0 |  13.27 | 480.51 |
    | 2000-01-02 22:00:00 |  11.33 |  45.56   92.58  150.48  216.65  287.58  358.42   422.4  469.93 | 469.93 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.48   3.52   4.43   5.25   5.97   6.59    7.1   7.46 |    0.0 |  11.33 | 469.93 |
    | 2000-01-02 22:30:00 |   9.65 |  41.37   85.63  140.75   204.4  273.38  343.31  407.99  458.62 | 458.62 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.36   3.39    4.3   5.11   5.84   6.47   6.99   7.37 |    0.0 |   9.65 | 458.62 |
    | 2000-01-02 23:00:00 |   8.21 |  37.59   79.22  131.65  192.79  259.76  328.58   393.6  446.75 | 446.75 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.25   3.26   4.16   4.98    5.7   6.34   6.88   7.29 |    0.0 |   8.21 | 446.75 |
    | 2000-01-02 23:30:00 |   6.96 |  34.17   73.31  123.13   181.8  246.72  314.27   379.3  434.44 | 434.44 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.14   3.15   4.04   4.84   5.57   6.22   6.77    7.2 |    0.0 |   6.96 | 434.44 |
    | 2000-01-03 00:00:00 |    5.9 |  31.08   67.86  115.18  171.42  234.25  300.39  365.16  421.83 | 421.83 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  2.04   3.03   3.91   4.71   5.44   6.09   6.65    7.1 |    0.0 |    5.9 | 421.83 |
    | 2000-01-03 00:30:00 |   4.99 |  28.29   62.84  107.75  161.61  222.34  286.97  351.24  409.01 | 409.01 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.95   2.92   3.79   4.59   5.32   5.97   6.54   7.01 |    0.0 |   4.99 | 409.01 |
    | 2000-01-03 01:00:00 |   4.21 |  25.77   58.21  100.82  152.37  210.99  274.02  337.59   396.1 |  396.1 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.86   2.81   3.67   4.47   5.19   5.85   6.43   6.91 |    0.0 |   4.21 |  396.1 |
    | 2000-01-03 01:30:00 |   3.55 |  23.49   53.95   94.36  143.65  200.18  261.54  324.24  383.16 | 383.16 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.77   2.71   3.56   4.35   5.07   5.73   6.31    6.8 |    0.0 |   3.55 | 383.16 |
    | 2000-01-03 02:00:00 |   2.99 |  21.43   50.03   88.33  135.44   189.9  249.55  311.22  370.27 | 370.27 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.69   2.61   3.45   4.23   4.95   5.61    6.2    6.7 |    0.0 |   2.99 | 370.27 |
    | 2000-01-03 02:30:00 |   2.51 |  19.57   46.41   82.71  127.71  180.13  238.03  298.56  357.49 | 357.49 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.61   2.51   3.34   4.11   4.83   5.49   6.08    6.6 |    0.0 |   2.51 | 357.49 |
    | 2000-01-03 03:00:00 |   2.11 |  17.89   43.08   77.47  120.43  170.85  226.99  286.28  344.88 | 344.88 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.53   2.42   3.24    4.0   4.71   5.37   5.97   6.49 |    0.0 |   2.11 | 344.88 |
    | 2000-01-03 03:30:00 |   1.76 |  16.37   40.02   72.59  113.59  162.05  216.42  274.38  332.47 | 332.47 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.46   2.33   3.14   3.89    4.6   5.26   5.86   6.39 |    0.0 |   1.76 | 332.47 |
    | 2000-01-03 04:00:00 |   1.48 |   15.0   37.19   68.04  107.16   153.7  206.31  262.89  320.29 | 320.29 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   1.4   2.25   3.04   3.79   4.49   5.14   5.74   6.28 |    0.0 |   1.48 | 320.29 |
    | 2000-01-03 04:30:00 |   1.23 |  13.76   34.59   63.81  101.11  145.79  196.65   251.8  308.39 | 308.39 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.34   2.17   2.95   3.69   4.38   5.03   5.63   6.18 |    0.0 |   1.23 | 308.39 |
    | 2000-01-03 05:00:00 |   1.03 |  12.63   32.19   59.86   95.42   138.3  187.42  241.11  296.78 | 296.78 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.28   2.09   2.86   3.59   4.28   4.92   5.53   6.07 |    0.0 |   1.03 | 296.78 |
    | 2000-01-03 05:30:00 |   0.86 |  11.61   29.98   56.18   90.08  131.21  178.62  230.82  285.48 | 285.48 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.22   2.02   2.77   3.49   4.17   4.82   5.42   5.97 |    0.0 |   0.86 | 285.48 |
    | 2000-01-03 06:00:00 |   0.71 |  10.69   27.94   52.75   85.07   124.5  170.23  220.93  274.51 | 274.51 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.17   1.95   2.69    3.4   4.07   4.71   5.31   5.86 |    0.0 |   0.71 | 274.51 |
    | 2000-01-03 06:30:00 |   0.59 |   9.85   26.06   49.55   80.35  118.14  162.24  211.44  263.86 | 263.86 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.12   1.88   2.61    3.3   3.97   4.61   5.21   5.76 |    0.0 |   0.59 | 263.86 |
    | 2000-01-03 07:00:00 |   0.49 |   9.08   24.32   46.57   75.92  112.14  154.62  202.33  253.56 | 253.56 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.07   1.81   2.53   3.22   3.88   4.51    5.1   5.66 |    0.0 |   0.49 | 253.56 |
    | 2000-01-03 07:30:00 |   0.41 |   8.39   22.72   43.79   71.76  106.46  147.38  193.59  243.59 | 243.59 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  1.02   1.75   2.45   3.13   3.78   4.41    5.0   5.56 |    0.0 |   0.41 | 243.59 |
    | 2000-01-03 08:00:00 |   0.34 |   7.76   21.23    41.2   67.85  101.08  140.48  185.23  233.97 | 233.97 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.98   1.69   2.38   3.05   3.69   4.31    4.9   5.45 |    0.0 |   0.34 | 233.97 |
    | 2000-01-03 08:30:00 |   0.28 |   7.19   19.86   38.78   64.18   96.01  133.92  177.22  224.69 | 224.69 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.94   1.63   2.31   2.96    3.6   4.22    4.8   5.36 |    0.0 |   0.28 | 224.69 |
    | 2000-01-03 09:00:00 |   0.23 |   6.66    18.6   36.53   60.73    91.2  127.69  169.56  215.75 | 215.75 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   0.9   1.58   2.24   2.89   3.52   4.12   4.71   5.26 |    0.0 |   0.23 | 215.75 |
    | 2000-01-03 09:30:00 |   0.19 |   6.19   17.42   34.42   57.48   86.66  121.76  162.24  207.14 | 207.14 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.86   1.52   2.17   2.81   3.43   4.03   4.61   5.16 |    0.0 |   0.19 | 207.14 |
    | 2000-01-03 10:00:00 |   0.16 |   5.75   16.34   32.45   54.43   82.37  116.13  155.23  198.85 | 198.85 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.83   1.47   2.11   2.73   3.35   3.95   4.52   5.07 |    0.0 |   0.16 | 198.85 |
    | 2000-01-03 10:30:00 |   0.13 |   5.35   15.33   30.62   51.57   78.32  110.77  148.54  190.89 | 190.89 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   0.8   1.42   2.05   2.66   3.27   3.86   4.43   4.97 |    0.0 |   0.13 | 190.89 |
    | 2000-01-03 11:00:00 |    0.1 |   4.99    14.4    28.9   48.87   74.48  105.68  142.16  183.24 | 183.24 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.77   1.38   1.99   2.59   3.19   3.77   4.34   4.88 |    0.0 |    0.1 | 183.24 |
    | 2000-01-03 11:30:00 |   0.09 |   4.65   13.53   27.29   46.34   70.86  100.85  136.05   175.9 |  175.9 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.74   1.33   1.93   2.53   3.11   3.69   4.25   4.79 |    0.0 |   0.09 |  175.9 |
    | 2000-01-03 12:00:00 |   0.07 |   4.35   12.72   25.79   43.95   67.43   96.26  130.23  168.85 | 168.85 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.71   1.29   1.87   2.46   3.04   3.61   4.17    4.7 |    0.0 |   0.07 | 168.85 |
    | 2000-01-03 12:30:00 |   0.06 |   4.06   11.98   24.38    41.7   64.19    91.9  124.67  162.09 | 162.09 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.68   1.25   1.82    2.4   2.97   3.53   4.08   4.61 |    0.0 |   0.06 | 162.09 |
    | 2000-01-03 13:00:00 |   0.05 |    3.8   11.28   23.07   39.59   61.12   87.75  119.37  155.61 | 155.61 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.66   1.21   1.77   2.34    2.9   3.46    4.0   4.53 |    0.0 |   0.05 | 155.61 |
    | 2000-01-03 13:30:00 |   0.04 |   3.57   10.64   21.83    37.6   58.22   83.81  114.31   149.4 |  149.4 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.64   1.17   1.72   2.28   2.83   3.38   3.92   4.44 |    0.0 |   0.04 |  149.4 |
    | 2000-01-03 14:00:00 |   0.03 |   3.35   10.03   20.68   35.73   55.48   80.07  109.48  143.44 | 143.44 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.61   1.13   1.67   2.22   2.76   3.31   3.84   4.36 |    0.0 |   0.03 | 143.44 |
    | 2000-01-03 14:30:00 |   0.03 |   3.14    9.47    19.6   33.96   52.88   76.52  104.88  137.74 | 137.74 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.59    1.1   1.63   2.16    2.7   3.24   3.76   4.28 |    0.0 |   0.03 | 137.74 |
    | 2000-01-03 15:00:00 |   0.02 |   2.95    8.95   18.58    32.3   50.42   73.14  100.48  132.28 | 132.28 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.57   1.07   1.58   2.11   2.64   3.17   3.69    4.2 |    0.0 |   0.02 | 132.28 |
    | 2000-01-03 15:30:00 |   0.02 |   2.78    8.46   17.63   30.73    48.1   69.93   96.29  127.05 | 127.05 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.55   1.03   1.54   2.05   2.58    3.1   3.62   4.12 |    0.0 |   0.02 | 127.05 |
    | 2000-01-03 16:00:00 |   0.01 |   2.62    8.01   16.73   29.25   45.89   66.88   92.29  122.04 | 122.04 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.53    1.0    1.5    2.0   2.52   3.03   3.54   4.04 |    0.0 |   0.01 | 122.04 |
    | 2000-01-03 16:30:00 |   0.01 |   2.47    7.58   15.89   27.85   43.81   63.98   88.48  117.24 | 117.24 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.52   0.97   1.46   1.95   2.46   2.97   3.47   3.97 |    0.0 |   0.01 | 117.24 |
    | 2000-01-03 17:00:00 |   0.01 |   2.33    7.19    15.1   26.53   41.83   61.23   84.85  112.65 | 112.65 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   0.5   0.95   1.42   1.91    2.4    2.9    3.4    3.9 |    0.0 |   0.01 | 112.65 |
    | 2000-01-03 17:30:00 |   0.01 |   2.21    6.81   14.36   25.29   39.96   58.61   81.38  108.26 | 108.26 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.48   0.92   1.38   1.86   2.35   2.84   3.34   3.82 |    0.0 |   0.01 | 108.26 |
    | 2000-01-03 18:00:00 |   0.01 |   2.09    6.46   13.66   24.11   38.18   56.11   78.07  104.05 | 104.05 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.47   0.89   1.34   1.81    2.3   2.78   3.27   3.75 |    0.0 |   0.01 | 104.05 |
    | 2000-01-03 18:30:00 |    0.0 |   1.98    6.14    13.0    23.0    36.5   53.74   74.91  100.03 | 100.03 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.45   0.87   1.31   1.77   2.25   2.73   3.21   3.68 |    0.0 |    0.0 | 100.03 |
    | 2000-01-03 19:00:00 |    0.0 |   1.87    5.83   12.38   21.95    34.9   51.49   71.89   96.18 |  96.18 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.44   0.84   1.27   1.73    2.2   2.67   3.14   3.62 |    0.0 |    0.0 |  96.18 |
    | 2000-01-03 19:30:00 |    0.0 |   1.78    5.55    11.8   20.96   33.38   49.34   69.02   92.49 |  92.49 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.43   0.82   1.24   1.69   2.15   2.61   3.08   3.55 |    0.0 |    0.0 |  92.49 |
    | 2000-01-03 20:00:00 |    0.0 |   1.69    5.28   11.25   20.02   31.94    47.3   66.27   88.96 |  88.96 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.41    0.8   1.21   1.65    2.1   2.56   3.02   3.48 |    0.0 |    0.0 |  88.96 |
    | 2000-01-03 20:30:00 |    0.0 |    1.6    5.02   10.73   19.13   30.58   45.35   63.65   85.58 |  85.58 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   0.4   0.78   1.18   1.61   2.05   2.51   2.96   3.42 |    0.0 |    0.0 |  85.58 |
    | 2000-01-03 21:00:00 |    0.0 |   1.52    4.79   10.24   18.29   29.28    43.5   61.15   82.35 |  82.35 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.39   0.75   1.15   1.57   2.01   2.45   2.91   3.36 |    0.0 |    0.0 |  82.35 |
    | 2000-01-03 21:30:00 |    0.0 |   1.45    4.56    9.78   17.49   28.05   41.74   58.76   79.25 |  79.25 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.38   0.73   1.12   1.53   1.96    2.4   2.85    3.3 |    0.0 |    0.0 |  79.25 |
    | 2000-01-03 22:00:00 |    0.0 |   1.38    4.35    9.34   16.74   26.88   40.06   56.48   76.29 |  76.29 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.37   0.72   1.09    1.5   1.92   2.36    2.8   3.24 |    0.0 |    0.0 |  76.29 |
    | 2000-01-03 22:30:00 |    0.0 |   1.31    4.15    8.93   16.02   25.77   38.46    54.3   73.45 |  73.45 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.36    0.7   1.07   1.47   1.88   2.31   2.74   3.18 |    0.0 |    0.0 |  73.45 |
    | 2000-01-03 23:00:00 |    0.0 |   1.25    3.96    8.54   15.34   24.71   36.93   52.22   70.73 |  70.73 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.35   0.68   1.04   1.43   1.84   2.26   2.69   3.12 |    0.0 |    0.0 |  70.73 |
    | 2000-01-03 23:30:00 |    0.0 |    1.2    3.79    8.17    14.7   23.71   35.48   50.23   68.13 |  68.13 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.34   0.66   1.02    1.4    1.8   2.22   2.64   3.07 |    0.0 |    0.0 |  68.13 |
    | 2000-01-04 00:00:00 |    0.0 |   1.14    3.62    7.82   14.09   22.75   34.09   48.33   65.63 |  65.63 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.33   0.64   0.99   1.37   1.76   2.17   2.59   3.01 |    0.0 |    0.0 |  65.63 |
    | 2000-01-04 00:30:00 |    0.0 |   1.09    3.46    7.49    13.5   21.84   32.77   46.51   63.25 |  63.25 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.32   0.63   0.97   1.34   1.73   2.13   2.54   2.96 |    0.0 |    0.0 |  63.25 |
    | 2000-01-04 01:00:00 |    0.0 |   1.04    3.32    7.17   12.95   20.97   31.51   44.78   60.96 |  60.96 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.31   0.61   0.95   1.31   1.69   2.09   2.49    2.9 |    0.0 |    0.0 |  60.96 |
    | 2000-01-04 01:30:00 |    0.0 |    1.0    3.18    6.87   12.43   20.15    30.3   43.11   58.77 |  58.77 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.31    0.6   0.93   1.28   1.66   2.05   2.45   2.85 |    0.0 |    0.0 |  58.77 |
    | 2000-01-04 02:00:00 |    0.0 |   0.96    3.04    6.59   11.93   19.36   29.15   41.53   56.67 |  56.67 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   0.3   0.58    0.9   1.25   1.62   2.01    2.4    2.8 |    0.0 |    0.0 |  56.67 |
    | 2000-01-04 02:30:00 |    0.0 |   0.92    2.92    6.32   11.46   18.61   28.05   40.01   54.65 |  54.65 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.29   0.57   0.88   1.23   1.59   1.97   2.36   2.75 |    0.0 |    0.0 |  54.65 |
    | 2000-01-04 03:00:00 |    0.0 |   0.88     2.8    6.07   11.01    17.9   27.01   38.55   52.72 |  52.72 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.28   0.56   0.86    1.2   1.56   1.93   2.31    2.7 |    0.0 |    0.0 |  52.72 |
    | 2000-01-04 03:30:00 |    0.0 |   0.84    2.68    5.83   10.58   17.22    26.0   37.16   50.87 |  50.87 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.28   0.54   0.84   1.17   1.52   1.89   2.27   2.66 |    0.0 |    0.0 |  50.87 |
    | 2000-01-04 04:00:00 |    0.0 |   0.81    2.58     5.6   10.17   16.57   25.05   35.83   49.09 |  49.09 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.27   0.53   0.83   1.15   1.49   1.86   2.23   2.61 |    0.0 |    0.0 |  49.09 |
    | 2000-01-04 04:30:00 |    0.0 |   0.78    2.48    5.38    9.78   15.95   24.13   34.55   47.39 |  47.39 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.26   0.52   0.81   1.12   1.46   1.82   2.19   2.57 |    0.0 |    0.0 |  47.39 |
    | 2000-01-04 05:00:00 |    0.0 |   0.75    2.38    5.17    9.41   15.36   23.26   33.33   45.76 |  45.76 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.26   0.51   0.79    1.1   1.43   1.79   2.15   2.52 |    0.0 |    0.0 |  45.76 |
    | 2000-01-04 05:30:00 |    0.0 |   0.72    2.29    4.98    9.06   14.79   22.42   32.16   44.19 |  44.19 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.25    0.5   0.77   1.08   1.41   1.75   2.11   2.48 |    0.0 |    0.0 |  44.19 |
    | 2000-01-04 06:00:00 |    0.0 |   0.69     2.2    4.79    8.72   14.25   21.62   31.04   42.69 |  42.69 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.25   0.49   0.76   1.06   1.38   1.72   2.07   2.44 |    0.0 |    0.0 |  42.69 |
    | 2000-01-04 06:30:00 |    0.0 |   0.66    2.12    4.61     8.4   13.74   20.85   29.96   41.24 |  41.24 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.24   0.48   0.74   1.04   1.35   1.69   2.04   2.39 |    0.0 |    0.0 |  41.24 |
    | 2000-01-04 07:00:00 |    0.0 |   0.64    2.04    4.44    8.09   13.25   20.12   28.93   39.86 |  39.86 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.23   0.46   0.73   1.01   1.33   1.66    2.0   2.35 |    0.0 |    0.0 |  39.86 |
    | 2000-01-04 07:30:00 |    0.0 |   0.62    1.96    4.28     7.8   12.77   19.42   27.94   38.53 |  38.53 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.23   0.45   0.71   0.99    1.3   1.63   1.96   2.31 |    0.0 |    0.0 |  38.53 |
    | 2000-01-04 08:00:00 |    0.0 |   0.59    1.89    4.12    7.52   12.32   18.75   26.99   37.25 |  37.25 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.22   0.45    0.7   0.97   1.28    1.6   1.93   2.27 |    0.0 |    0.0 |  37.25 |
    | 2000-01-04 08:30:00 |    0.0 |   0.57    1.82    3.98    7.26   11.89    18.1   26.09   36.02 |  36.02 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.22   0.44   0.68   0.96   1.25   1.57    1.9   2.24 |    0.0 |    0.0 |  36.02 |
    | 2000-01-04 09:00:00 |    0.0 |   0.55    1.76    3.84     7.0   11.48   17.49   25.21   34.84 |  34.84 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.22   0.43   0.67   0.94   1.23   1.54   1.86    2.2 |    0.0 |    0.0 |  34.84 |
    | 2000-01-04 09:30:00 |    0.0 |   0.53     1.7     3.7    6.76   11.09    16.9   24.38   33.71 |  33.71 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.21   0.42   0.65   0.92   1.21   1.51   1.83   2.16 |    0.0 |    0.0 |  33.71 |
    | 2000-01-04 10:00:00 |    0.0 |   0.51    1.64    3.57    6.52   10.71   16.33   23.57   32.62 |  32.62 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.21   0.41   0.64    0.9   1.18   1.48    1.8   2.13 |    0.0 |    0.0 |  32.62 |
    | 2000-01-04 10:30:00 |    0.0 |    0.5    1.58    3.45     6.3   10.35   15.78    22.8   31.57 |  31.57 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   0.2    0.4   0.63   0.88   1.16   1.46   1.77   2.09 |    0.0 |    0.0 |  31.57 |
    | 2000-01-04 11:00:00 |    0.0 |   0.48    1.53    3.33    6.09    10.0   15.26   22.06   30.56 |  30.56 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |   0.2   0.39   0.62   0.87   1.14   1.43   1.74   2.06 |    0.0 |    0.0 |  30.56 |
    | 2000-01-04 11:30:00 |    0.0 |   0.46    1.48    3.22    5.88    9.67   14.76   21.35   29.59 |  29.59 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.19   0.39    0.6   0.85   1.12   1.41   1.71   2.02 |    0.0 |    0.0 |  29.59 |
    | 2000-01-04 12:00:00 |    0.0 |   0.45    1.43    3.11    5.69    9.35   14.28   20.67   28.66 |  28.66 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.19   0.38   0.59   0.83    1.1   1.38   1.68   1.99 |    0.0 |    0.0 |  28.66 |
    | 2000-01-04 12:30:00 |    0.0 |   0.43    1.38    3.01     5.5    9.04   13.82   20.01   27.77 |  27.77 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.19   0.37   0.58   0.82   1.08   1.36   1.65   1.96 |    0.0 |    0.0 |  27.77 |
    | 2000-01-04 13:00:00 |    0.0 |   0.42    1.34    2.91    5.32    8.75   13.38   19.38    26.9 |   26.9 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.18   0.36   0.57    0.8   1.06   1.33   1.63   1.93 |    0.0 |    0.0 |   26.9 |
    | 2000-01-04 13:30:00 |    0.0 |   0.41    1.29    2.82    5.15    8.47   12.95   18.77   26.07 |  26.07 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.18   0.36   0.56   0.79   1.04   1.31    1.6    1.9 |    0.0 |    0.0 |  26.07 |
    | 2000-01-04 14:00:00 |    0.0 |   0.39    1.25    2.73    4.98     8.2   12.55   18.18   25.27 |  25.27 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.17   0.35   0.55   0.78   1.02   1.29   1.57   1.87 |    0.0 |    0.0 |  25.27 |
    | 2000-01-04 14:30:00 |    0.0 |   0.38    1.21    2.64    4.83    7.94   12.15   17.62    24.5 |   24.5 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.17   0.34   0.54   0.76   1.01   1.27   1.55   1.84 |    0.0 |    0.0 |   24.5 |
    | 2000-01-04 15:00:00 |    0.0 |   0.37    1.17    2.56    4.67    7.69   11.78   17.08   23.76 |  23.76 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.17   0.34   0.53   0.75   0.99   1.25   1.52   1.81 |    0.0 |    0.0 |  23.76 |
    | 2000-01-04 15:30:00 |    0.0 |   0.36    1.14    2.48    4.53    7.45   11.41   16.56   23.05 |  23.05 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.16   0.33   0.52   0.73   0.97   1.23    1.5   1.78 |    0.0 |    0.0 |  23.05 |
    | 2000-01-04 16:00:00 |    0.0 |   0.35     1.1     2.4    4.39    7.22   11.06   16.06   22.36 |  22.36 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.16   0.33   0.51   0.72   0.95   1.21   1.47   1.75 |    0.0 |    0.0 |  22.36 |
    | 2000-01-04 16:30:00 |    0.0 |   0.34    1.07    2.33    4.25     7.0   10.73   15.58    21.7 |   21.7 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.16   0.32    0.5   0.71   0.94   1.19   1.45   1.73 |    0.0 |    0.0 |   21.7 |
    | 2000-01-04 17:00:00 |    0.0 |   0.33    1.04    2.26    4.13    6.79   10.41   15.11   21.06 |  21.06 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.16   0.31   0.49    0.7   0.92   1.17   1.43    1.7 |    0.0 |    0.0 |  21.06 |
    | 2000-01-04 17:30:00 |    0.0 |   0.32    1.01    2.19     4.0    6.59   10.09   14.67   20.44 |  20.44 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.15   0.31   0.49   0.69   0.91   1.15    1.4   1.67 |    0.0 |    0.0 |  20.44 |
    | 2000-01-04 18:00:00 |    0.0 |   0.31    0.98    2.12    3.88    6.39     9.8   14.24   19.85 |  19.85 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.15    0.3   0.48   0.67   0.89   1.13   1.38   1.65 |    0.0 |    0.0 |  19.85 |
    | 2000-01-04 18:30:00 |    0.0 |    0.3    0.95    2.06    3.77     6.2    9.51   13.82   19.28 |  19.28 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.15    0.3   0.47   0.66   0.88   1.11   1.36   1.62 |    0.0 |    0.0 |  19.28 |
    | 2000-01-04 19:00:00 |    0.0 |   0.29    0.92     2.0    3.66    6.02    9.23   13.42   18.72 |  18.72 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.14   0.29   0.46   0.65   0.86   1.09   1.34    1.6 |    0.0 |    0.0 |  18.72 |
    | 2000-01-04 19:30:00 |    0.0 |   0.28     0.9    1.94    3.55    5.85    8.96   13.03   18.19 |  18.19 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.14   0.29   0.45   0.64   0.85   1.07   1.32   1.57 |    0.0 |    0.0 |  18.19 |
    | 2000-01-04 20:00:00 |    0.0 |   0.28    0.87    1.89    3.45    5.68     8.7   12.66   17.68 |  17.68 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.14   0.28   0.45   0.63   0.83   1.06    1.3   1.55 |    0.0 |    0.0 |  17.68 |
    | 2000-01-04 20:30:00 |    0.0 |   0.27    0.85    1.83    3.35    5.51    8.46    12.3   17.18 |  17.18 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.14   0.28   0.44   0.62   0.82   1.04   1.28   1.53 |    0.0 |    0.0 |  17.18 |
    | 2000-01-04 21:00:00 |    0.0 |   0.26    0.82    1.78    3.25    5.36    8.22   11.96    16.7 |   16.7 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.13   0.27   0.43   0.61   0.81   1.02   1.26    1.5 |    0.0 |    0.0 |   16.7 |
    | 2000-01-04 21:30:00 |    0.0 |   0.25     0.8    1.73    3.16    5.21    7.99   11.62   16.24 |  16.24 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.13   0.27   0.42    0.6    0.8   1.01   1.24   1.48 |    0.0 |    0.0 |  16.24 |
    | 2000-01-04 22:00:00 |    0.0 |   0.25    0.78    1.69    3.08    5.06    7.76    11.3   15.79 |  15.79 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.13   0.27   0.42   0.59   0.78   0.99   1.22   1.46 |    0.0 |    0.0 |  15.79 |
    | 2000-01-04 22:30:00 |    0.0 |   0.24    0.76    1.64    2.99    4.92    7.55   10.99   15.36 |  15.36 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.13   0.26   0.41   0.58   0.77   0.98    1.2   1.44 |    0.0 |    0.0 |  15.36 |
    | 2000-01-04 23:00:00 |    0.0 |   0.23    0.74    1.59    2.91    4.79    7.34   10.69   14.94 |  14.94 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.12   0.26    0.4   0.57   0.76   0.96   1.18   1.42 |    0.0 |    0.0 |  14.94 |
    | 2000-01-04 23:30:00 |    0.0 |   0.23    0.72    1.55    2.83    4.66    7.14    10.4   14.54 |  14.54 | 0.0  0.0  0.0  0.0  0.0  0.0  0.0  0.0 |  0.12   0.25    0.4   0.56   0.75   0.95   1.17    1.4 |    0.0 |    0.0 |  14.54 |

We printed the last test results with a lower precision than usual.
Otherwise, there were some differences when performing the test
in Python and in Cython mode and possibly also when working with
different compilers, which indicates that |lstream_v001| reaches
limited numerical accuracy when negative water stages occur.  In our
real applications, we did not realise any problems stemming from this.
However, we cannot guarantee that they will never occur.  Any
suggestions on solving this issue are welcome.
"""
# import...
# ...from standard library
from typing import *
# ...from HydPy
from hydpy.core import objecttools
from hydpy.exe.modelimports import *
# ...from lstream
from hydpy.models.lstream import lstream_fluxes
from hydpy.models.lstream import lstream_model
from hydpy.models.lstream import lstream_solver


class Characteristics(NamedTuple):
    """Data class for holding the results of method
    |Model.calculate_characteristiclength|."""
    waterstage: float
    discharge: float
    derivative: float
    length_orig: float
    nmb_subsections: int
    length_adj: float

    def __repr__(self) -> str:
        return (
            f'Characteristics(\n'
            f'    waterstage={objecttools.repr_(self.waterstage)},\n'
            f'    discharge={objecttools.repr_(self.discharge)},\n'
            f'    derivative={objecttools.repr_(self.derivative)},\n'
            f'    length_orig={objecttools.repr_(self.length_orig)},\n'
            f'    nmb_subsections={self.nmb_subsections},\n'
            f'    length_adj={objecttools.repr_(self.length_adj)},\n'
            f')'
        )

    def __str__(self):
        return objecttools.flatten_repr(self)


class Model(lstream_model.Model, lstream_model.ProfileMixin):
    """Version 1 of HydPy-L-Stream."""
    SOLVERPARAMETERS = (
        lstream_solver.AbsErrorMax,
        lstream_solver.RelErrorMax,
        lstream_solver.RelDTMin,
        lstream_solver.RelDTMax,
    )
    SOLVERSEQUENCES = (
        lstream_fluxes.QG,
        lstream_fluxes.DH,
    )
    INLET_METHODS = (
        lstream_model.Pick_Q_V1,
    )
    RECEIVER_METHODS = ()
    ADD_METHODS = (
        lstream_model.Return_QF_V1,
        lstream_model.Return_H_V1,
    )
    PART_ODE_METHODS = (
        lstream_model.Calc_RHM_V1,
        lstream_model.Calc_RHMDH_V1,
        lstream_model.Calc_RHV_V1,
        lstream_model.Calc_RHVDH_V1,
        lstream_model.Calc_RHLVR_RHRVR_V1,
        lstream_model.Calc_RHLVRDH_RHRVRDH_V1,
        lstream_model.Calc_AM_UM_V1,
        lstream_model.Calc_ALV_ARV_ULV_URV_V1,
        lstream_model.Calc_ALVR_ARVR_ULVR_URVR_V1,
        lstream_model.Calc_QM_V1,
        lstream_model.Calc_QLV_QRV_V1,
        lstream_model.Calc_QLVR_QRVR_V1,
        lstream_model.Calc_AG_V1,
        lstream_model.Calc_QG_V1,
        lstream_model.Calc_QA_V1,
        lstream_model.Calc_WBM_V1,
        lstream_model.Calc_WBLV_WBRV_V1,
        lstream_model.Calc_WBLVR_WBRVR_V1,
        lstream_model.Calc_WBG_V1,
        lstream_model.Calc_DH_V1,
    )
    FULL_ODE_METHODS = (
        lstream_model.Update_H_V1,
    )
    OUTLET_METHODS = (
        lstream_model.Pass_Q_V1,
    )
    SENDER_METHODS = ()
    SUBMODELS = (
        lstream_model.PegasusH,
    )

    def calculate_characteristiclength(
            self,
            *,
            h: Optional[float] = None,
            q: Optional[float] = None,
            dx: float = 1e-6,
            lenmin: float = 0.1,
            nmbmax: int = 50,
    ) -> Characteristics:
        """Approximate the characteristic length after the Kalinin-Milyukov
        method.

        Method |Model.calculate_characteristiclength| determines the
        characteristic length based on a finite difference approach and
        returns it, accompanied by some intermediate results, within a
        |Characteristics| object.  You need to pass in either the stage
        or the discharge (`q`) value of interest (`h`).  Optionally, you
        can define alternative values for the finite-difference step size
        (`dx`), the smallest allowed subsection length (`minlen`) and the
        maximum number of cross-sections (`nmbmax`).  The returned
        `length_orig` value is the initial approximation of the ideal
        characteristic length, and `length_adj` is its adjustment to the
        actual channel length.  All length value must be given and are
        returned in kilometres.

        We reuse the example given in the main documentation on module
        |lstream_v001|:

        >>> from hydpy.models.lstream_v001 import *
        >>> parameterstep('1d')
        >>> simulationstep('30m')

        >>> gts(1)
        >>> laen(100.0)
        >>> gef(0.00025)
        >>> bm(15.0)
        >>> bnm(5.0)
        >>> skm(1.0/0.035)
        >>> hm(inf)
        >>> bv(100.0)
        >>> bbv(20.0)
        >>> bnv(10.0)
        >>> bnvr(100.0)
        >>> skv(10.0)
        >>> ekm(1.0)
        >>> ekv(1.0)
        >>> hr(0.1)
        >>> parameters.update()

        When passing a water stage of 7.5 m, method
        |Model.calculate_characteristiclength| determines a characteristic
        length of 13 km and suggest to split the total channel length of
        100 km length into eight subsections:

        >>> model.calculate_characteristiclength(h=7.5)
        Characteristics(
            waterstage=7.5,
            discharge=470.654321,
            derivative=0.006929,
            length_orig=13.044707,
            nmb_subsections=8,
            length_adj=12.5,
        )

        Passing the discharge value corresponding to the lastly defined
        water stage returns equal results (method
        |Model.calculate_characteristiclength| uses first method
        |lstream_model.Return_H_V1| to calculate the water stage
        and then precedes as usual):

        >>> model.calculate_characteristiclength(q=470.654321)
        Characteristics(
            waterstage=7.5,
            discharge=470.654321,
            derivative=0.006929,
            length_orig=13.044707,
            nmb_subsections=8,
            length_adj=12.5,
        )

        The relevant discharge value must be greater than zero:

        >>> model.calculate_characteristiclength(q=0.0)
        Traceback (most recent call last):
        ...
        ValueError: While trying to calculate the characteristic length for \
the river channel of `lstream_v001` of element `?`, the following error \
occurred: The given values result in a mean discharge of 0.0 m³/s and a \
discharge gradient of 0.0 m³/s/m.

        The same holds for the discharge gradient:

        >>> model.calculate_characteristiclength(q=100.0, dx=0.0)
        Traceback (most recent call last):
        ...
        ValueError: While trying to calculate the characteristic length for \
the river channel of `lstream_v001` of element `?`, the following error \
occurred: The given values result in a mean discharge of 100.0 m³/s and a \
discharge gradient of 0.0 m³/s/m.

        One must define at either `h` or `q` but not both:

        >>> model.calculate_characteristiclength()
        Traceback (most recent call last):
        ...
        ValueError: While trying to calculate the characteristic length for \
the river channel of `lstream_v001` of element `?`, the following error \
occurred: Calculating the characteristic length requires either a reference \
stage or a discharge value, but neither is given.

        >>> model.calculate_characteristiclength(h=7.5, q=470.654321)
        Traceback (most recent call last):
        ...
        ValueError: While trying to calculate the characteristic length for \
the river channel of `lstream_v001` of element `?`, the following error \
occurred: Calculating the characteristic length requires either a reference \
stage or a discharge value, but both are given.

        You can modify the allowed channel length or increase the allowed
        number of subsections:

        >>> model.calculate_characteristiclength(h=7.5, lenmin=20.0)
        Characteristics(
            waterstage=7.5,
            discharge=470.654321,
            derivative=0.006929,
            length_orig=13.044707,
            nmb_subsections=5,
            length_adj=20.0,
        )

        >>> model.calculate_characteristiclength(
        ...     h=7.5, lenmin=20.0, nmbmax=4)
        Characteristics(
            waterstage=7.5,
            discharge=470.654321,
            derivative=0.006929,
            length_orig=13.044707,
            nmb_subsections=4,
            length_adj=25.0,
        )

        The lowest returned number of subsections is one:

        >>> laen(1.0)
        >>> model.calculate_characteristiclength(h=7.5)
        Characteristics(
            waterstage=7.5,
            discharge=470.654321,
            derivative=0.006929,
            length_orig=13.044707,
            nmb_subsections=1,
            length_adj=1.0,
        )
        """
        try:
            if h is None:
                if q is None:
                    raise ValueError(
                        'Calculating the characteristic length requires either '
                        'a reference stage or a discharge value, but neither is '
                        'given.')
                self.sequences.fluxes.qg = q
                h = self.return_h_v1()
            elif q is not None:
                raise ValueError(
                    'Calculating the characteristic length requires either '
                    'a reference stage or a discharge value, but both are given.')
            qs = []
            for h_ in (h-dx/2.0, h+dx/2.0):
                self.sequences.states.h(h_)
                self.calculate_single_terms()
                qs.append(self.sequences.fluxes.qg[0])
            qmean = float(numpy.mean(qs))
            dq = qs[1]-qs[0]
            if (qmean == 0.) or (dq == 0.):
                raise ValueError(
                    f'The given values result in a mean discharge of '
                    f'{objecttools.repr_(qmean)} m³/s and a discharge '
                    f'gradient of {objecttools.repr_(dq)} m³/s/m.'
                )
            length_orig = (qmean*dx)/(self.parameters.control.gef*dq)/1000.0
            length_adj = max(length_orig, lenmin)
            number = int(min(max(round(
                self.parameters.control.laen/length_adj), 1), nmbmax))
            return Characteristics(
                waterstage=h,
                discharge=qmean,
                derivative=dx/dq,
                length_orig=length_orig,
                nmb_subsections=number,
                length_adj=self.parameters.control.laen/number
            )
        except BaseException:
            objecttools.augment_excmessage(
                f'While trying to calculate the characteristic length for '
                f'the river channel of {objecttools.elementphrase(self)}'
            )


tester = Tester()
cythonizer = Cythonizer()
cythonizer.finalise()
