# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, wildcard-import, unused-wildcard-import
"""Version 2 of HydPy-L-Stream is a computationally more efficient
alternative to application model |lstream_v001|.  The higher efficiency
is due to not calculating discharges based on channel geometries but based
on preprocessed relationships between discharge and storage, similar to the
"V/Q-BEZIEHUNG EXTERN" option of `LARSIM`_.  An artificial neural network,
available as "parameter" |VG2QG|, serves to describe this relationship.
Principally, the neural network can fit any relationship provided by
|lstream_v001| very accurately.  However, high accuracy might require a
considerable number of neurons, which also can come with a relevant
performance cost.

.. _`LARSIM`: http://www.larsim.de/en/the-model/

Integration tests
=================

.. how_to_understand_integration_tests::

.. _lstream_v002_main_channel_flow:

main channel flow
_________________

The following integration test repeats the :ref:`lstream_v001_main_channel_flow`
example of the documentation on application model |lstream_v001|.  The spatial
end temporal settings are identical:

>>> from hydpy import pub, Nodes, Element
>>> pub.timegrids = '2000-01-01', '2000-01-05', '30m'
>>> from hydpy.models.lstream_v002 import *
>>> parameterstep('1d')
>>> nodes = Nodes('input1', 'input2', 'output')
>>> stream = Element('stream',
...                  inlets=['input1', 'input2'],
...                  outlets='output')
>>> stream.model = model

We again divide the channel into eight subsections:

>>> gts(8)

Next, we define a relatively small neural network consisting of three neurons
in a single hidden layer.  This network roughly approximates the discharge
calculated by the Gauckler-Manning-Strickler equation on the triple trapezoid
profile defined in the :ref:`lstream_v001_main_channel_flow` example:

>>> vg2qg(nmb_inputs=1,
...       nmb_neurons=(3,),
...       nmb_outputs=1,
...       weights_input=[[0.610346, 1.360399, 0.35465]],
...       weights_output=[[432.958081],
...                       [94.674914],
...                       [1778.576841]],
...       intercepts_hidden=[[-2.681908, -2.237483, -3.790469]],
...       intercepts_output=[-78.006394])

In contrast to application model |lstream_v001|, |lstream_v002| uses the
stored water volume (|VG|) as its state variable instead of the water
stage (|H|).  Hence, we now must set |VG| to a value resulting in an
initial outflow of 100 m³/s for the defined parameterisation of |VG2QG|,
which holds for 1.570929405 million m³:

>>> from hydpy.core.testtools import IntegrationTest
>>> IntegrationTest.plotting_options.activated = fluxes.qz, fluxes.qa
>>> test = IntegrationTest(stream, inits=[[states.vg, 1.570929405]])

Finally, we define two identical inflow time series:

>>> import numpy
>>> q_base = 100.0
>>> q_peak = 900.0
>>> t_peak = 24.0
>>> β = 16.0
>>> ts = pub.timegrids.init.to_timepoints()
>>> nodes.input1.sequences.sim.series = q_base
>>> nodes.input2.sequences.sim.series = (
...     (q_peak-q_base)*((ts/t_peak)*numpy.exp(1.0-ts/t_peak))**β)

Our approximation of the discharge-storage relationship is far from
perfect.  Still, it is, at least in the range relevant for the selected
event, sufficient to reproduce the original results of application
model |lstream_v001| with reasonable accuracy (for example, peak
flow is 660 m³/s instead of 659 m³/s):

.. integration-test::

    >>> test('lstream_v002_main_channel_flow')
    |                date |         qz |        qza |                                                                                             qg |         qa |                                                                             vg | input1 |     input2 |     output |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2000-01-01 00:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 00:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 01:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:00:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 02:30:00 |      100.0 |      100.0 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |        0.0 |      100.0 |
    | 2000-01-01 03:00:00 | 100.000003 | 100.000003 |      100.0       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |   0.000003 |      100.0 |
    | 2000-01-01 03:30:00 | 100.000029 | 100.000029 | 100.000003       100.0       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |   0.000029 |      100.0 |
    | 2000-01-01 04:00:00 | 100.000175 | 100.000175 | 100.000018  100.000001       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 |  1.57093  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |   0.000175 |      100.0 |
    | 2000-01-01 04:30:00 | 100.000826 | 100.000826 | 100.000094  100.000005       100.0       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570931   1.57093  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |   0.000826 |      100.0 |
    | 2000-01-01 05:00:00 | 100.003194 | 100.003194 | 100.000392  100.000028  100.000002       100.0       100.0       100.0       100.0       100.0 |      100.0 | 1.570936   1.57093  1.570929  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |   0.003194 |      100.0 |
    | 2000-01-01 05:30:00 | 100.010515 | 100.010515 | 100.001392  100.000118  100.000011  100.000001       100.0       100.0       100.0       100.0 |      100.0 | 1.570953  1.570933   1.57093  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |   0.010515 |      100.0 |
    | 2000-01-01 06:00:00 | 100.030315 | 100.030315 | 100.004326  100.000423  100.000043  100.000004       100.0       100.0       100.0       100.0 |      100.0 | 1.570999   1.57094   1.57093  1.570929  1.570929  1.570929  1.570929  1.570929 |  100.0 |   0.030315 |      100.0 |
    | 2000-01-01 06:30:00 | 100.078179 | 100.078179 | 100.011998  100.001333  100.000149  100.000014  100.000001       100.0       100.0       100.0 |      100.0 | 1.571118  1.570959  1.570932   1.57093  1.570929  1.570929  1.570929  1.570929 |  100.0 |   0.078179 |      100.0 |
    | 2000-01-01 07:00:00 | 100.183351 | 100.183351 | 100.030196  100.003757  100.000461  100.000049  100.000005       100.0       100.0       100.0 |      100.0 | 1.571394  1.571006  1.570938   1.57093   1.57093  1.570929  1.570929  1.570929 |  100.0 |   0.183351 |      100.0 |
    | 2000-01-01 07:30:00 | 100.396214 | 100.396214 | 100.069866  100.009628  100.001287  100.000152  100.000017  100.000002       100.0       100.0 |      100.0 | 1.571982  1.571115  1.570953  1.570933   1.57093  1.570929  1.570929  1.570929 |  100.0 |   0.396214 |      100.0 |
    | 2000-01-01 08:00:00 | 100.797305 | 100.797305 | 100.150201  100.022713  100.003292  100.000427  100.000051  100.000006       100.0       100.0 |      100.0 | 1.573146  1.571344  1.570988  1.570938   1.57093   1.57093  1.570929  1.570929 |  100.0 |   0.797305 |      100.0 |
    | 2000-01-01 08:30:00 | 101.507032 | 101.507032 | 100.297052  100.054779   100.00835  100.001216   100.00016   100.00002  100.000002       100.0 |      100.0 | 1.575324   1.57178  1.571072  1.570951  1.570932   1.57093  1.570929  1.570929 |  100.0 |   1.507032 |      100.0 |
    | 2000-01-01 09:00:00 | 102.694853 | 102.694853 | 100.566876  100.109714  100.018847  100.002973  100.000431  100.000057  100.000007  100.000001 | 100.000001 | 1.579155  1.572603  1.571236  1.570979  1.570937   1.57093   1.57093  1.570929 |  100.0 |   2.694853 | 100.000001 |
    | 2000-01-01 09:30:00 | 104.586356 | 104.586356 | 101.026882  100.210142  100.039036  100.006684  100.001052  100.000153  100.000021  100.000003 | 100.000003 | 1.585562  1.574073  1.571543  1.571037  1.570947  1.570932   1.57093  1.570929 |  100.0 |   4.586356 | 100.000003 |
    | 2000-01-01 10:00:00 | 107.466641 | 107.466641 | 101.776031  100.384789  100.076246  100.014007  100.002376  100.000373  100.000054  100.000007 | 100.000007 | 1.595805  1.576578  1.572099  1.571149  1.570968  1.570936   1.57093   1.57093 |  100.0 |   7.466641 | 100.000007 |
    | 2000-01-01 10:30:00 | 111.678559 | 111.678559 | 102.946883  100.675411  100.141937  100.027735  100.005027  100.000846  100.000133  100.000019 | 100.000019 | 1.611522  1.580666  1.573059  1.571355  1.571009  1.570943  1.570932   1.57093 |  100.0 |  11.678559 | 100.000019 |
    | 2000-01-01 11:00:00 | 117.614836 | 117.614836 | 104.710185  101.140371   100.25329   100.05236  100.010071  100.001805  100.000302  100.000047 | 100.000047 |  1.63475  1.587092  1.574656  1.571717  1.571085  1.570958  1.570934   1.57093 |  100.0 |  17.614836 | 100.000047 |
    | 2000-01-01 11:30:00 | 125.703736 | 125.703736 | 107.278151  101.858435  100.435142  100.094812  100.019258  100.003654  100.000649  100.000108 | 100.000108 | 1.667916  1.596847  1.577218  1.572329  1.571221  1.570986   1.57094  1.570931 |  100.0 |  25.703736 | 100.000108 |
    | 2000-01-01 12:00:00 | 136.388647 | 136.388647 |  110.91141  102.923358   100.72522   100.16599  100.035517  100.007114  100.001336  100.000236 | 100.000236 | 1.713775  1.611226  1.581174  1.573336  1.571456  1.571037   1.57095  1.570933 |  100.0 |  36.388647 | 100.000236 |
    | 2000-01-01 12:30:00 | 150.102726 | 150.102726 | 115.896767  104.483442  101.163991  100.280047  100.062884  100.013231  100.002615  100.000486 | 100.000486 | 1.775346   1.63177  1.587149  1.574927  1.571847  1.571127  1.570969  1.570937 |  100.0 |  50.102726 | 100.000486 |
    | 2000-01-01 13:00:00 | 167.240349 | 167.240349 | 122.571405  106.692694  101.817703  100.457806  100.107649   100.02373  100.004918  100.000961 | 100.000961 |  1.85575  1.660352  1.595924  1.577375  1.572477  1.571278  1.571003  1.570944 |  100.0 |  67.240349 | 100.000961 |
    | 2000-01-01 13:30:00 | 188.127569 | 188.127569 | 131.292887  109.747419  102.766972  100.727721  100.178742  100.041178  100.008926  100.001825 | 100.001825 | 1.958053  1.699133  1.608489  1.581045  1.573465  1.571525  1.571061  1.570957 |  100.0 |  88.127569 | 100.001825 |
    | 2000-01-01 14:00:00 | 212.993962 | 212.993962 | 142.425369  113.878332  104.113795   101.12774  100.288703  100.069354  100.015686  100.003349 | 100.003349 | 2.085076  1.750518  1.626065   1.58642  1.574975   1.57192  1.571158  1.570979 |  100.0 | 112.993962 | 100.003349 |
    | 2000-01-01 14:30:00 | 241.948195 | 241.948195 | 156.314703  119.348015  105.984474  101.707456  100.454723  100.113673  100.026768  100.005954 | 100.005954 | 2.239216  1.817058   1.65012  1.594119   1.57723  1.572534  1.571314  1.571017 |  100.0 | 141.948195 | 100.005954 |
    | 2000-01-01 15:00:00 |  274.95935 |  274.95935 | 173.262326  126.443966  108.531934  102.530462  100.699898  100.181737  100.044472  100.010284 | 100.010284 | 2.422271  1.901331  1.682361  1.604922  1.580525  1.573467  1.571561  1.571078 |  100.0 |  174.95935 | 100.010284 |
    | 2000-01-01 15:30:00 | 311.845554 | 311.845554 | 193.508739   135.46699  111.936854  103.676803  101.054675  100.283993  100.072092  100.017303 | 100.017303 | 2.635277  2.005806  1.724716   1.61979  1.585245  1.574854  1.571943  1.571177 |  100.0 | 211.845554 | 100.017303 |
    | 2000-01-01 16:00:00 | 352.270855 | 352.270855 | 217.236756  146.716799  116.406984  105.245379  101.558498  100.434542  100.114252  100.028414 | 100.028414 | 2.878339  2.132742  1.779273  1.639881  1.591882  1.576877  1.572519  1.571331 |  100.0 | 252.270855 | 100.028414 |
    | 2000-01-01 16:30:00 | 395.750601 | 395.750601 | 244.590445  160.479098  122.174304  107.356088  102.261634  100.652088  100.177319  100.045625 | 100.045625 | 3.150427  2.284143  1.848222  1.666553  1.601052  1.579774  1.573374  1.571568 |  100.0 | 295.750601 | 100.045625 |
    | 2000-01-01 17:00:00 | 441.664962 | 441.664962 | 275.682156  177.018613    129.4901  110.151475  103.227125  100.961053  100.269923  100.071758 | 100.071758 | 3.449196  2.461737  1.933773  1.701363  1.613515  1.583853  1.574618  1.571925 |  100.0 | 341.664962 | 100.071758 |
    | 2000-01-01 17:30:00 | 489.279607 | 489.279607 |  310.55212  196.579273  138.618456  113.797651  104.532841  101.392853  100.403562  100.110704 | 100.110704 | 3.770905  2.666888  2.038103   1.74604  1.630192  1.589505  1.576399  1.572452 |  100.0 | 389.279607 | 100.110704 |
    | 2000-01-01 18:00:00 |  537.77215 |  537.77215 | 349.076068  219.387366  149.828899    118.4842   106.27352  101.987345   100.59335  100.167754 | 100.167754 | 4.110558  2.900328  2.163308  1.802461  1.652171   1.59722  1.578908  1.573218 |  100.0 |  437.77215 | 100.167754 |
    | 2000-01-01 18:30:00 | 586.262607 | 586.262607 | 390.868408  245.647955  163.389138  124.422743  108.562715  102.794439  100.858872  100.249986 | 100.249986 | 4.462268  3.161725  2.311374    1.8726  1.680719  1.607603  1.582392  1.574314 |  100.0 | 486.262607 | 100.249986 |
    | 2000-01-01 19:00:00 | 633.846018 | 633.846018 |  435.24892  275.519594  179.558831  131.843834  111.534437  103.875861  101.225204  100.366752 | 100.366752 | 4.819743  3.449237  2.484103  1.958487  1.717276  1.621389  1.587163  1.575859 |  100.0 | 533.846018 | 100.366752 |
    | 2000-01-01 19:30:00 | 679.625318 | 679.625318 | 481.302938  309.053185  198.584477  140.992018  115.344235  105.307022  101.724071  100.530248 | 100.530248 | 5.176723  3.759287  2.682947  2.062154  1.763442  1.639456  1.593612  1.578008 |  100.0 | 579.625318 | 100.530248 |
    | 2000-01-01 20:00:00 | 722.742733 | 722.742733 |  528.00335  346.101042  220.693309  152.119189  120.169321  107.178937  102.395183  100.756201 | 100.756201 | 5.527254  4.086711  2.908681  2.185587  1.820952  1.662838  1.602223  1.580958 |  100.0 | 622.742733 | 100.756201 |
    | 2000-01-01 20:30:00 | 762.408181 | 762.408181 | 574.332042  386.237769  246.080582  165.476907  126.207306  109.600031  103.287721  101.064672 | 101.064672 | 5.865791  4.425281  3.160964  2.330674  1.891637  1.692731  1.613585   1.58496 |  100.0 | 662.408181 | 101.064672 |
    | 2000-01-01 21:00:00 | 797.923464 | 797.923464 | 619.355372  428.748227  274.880275  181.308693  133.673153  112.697641  104.461975  101.480999 | 101.480999 | 6.187213  4.768374  3.437926  2.499103  1.977381  1.730487  1.628409  1.590326 |  100.0 | 697.923464 | 101.480999 |
    | 2000-01-01 21:30:00 | 828.701418 | 828.701418 | 662.246682  472.702646  307.109452  199.842771  142.794176  116.618889  105.991069  102.036889 | 102.036889 | 6.486832  5.109553  3.735994  2.692183  2.080069  1.777603  1.647539  1.597443 |  100.0 | 728.701418 | 102.036889 |
    | 2000-01-01 22:00:00 |  854.27953 |  854.27953 |  702.27674  517.083089  342.592028  221.283824  153.803325  121.530558  107.962696  102.771649 | 102.771649 | 6.760437  5.442902  4.050078  2.910537  2.201534  1.835694  1.671962  1.606787 |  100.0 |  754.27953 | 102.771649 |
    | 2000-01-01 22:30:00 | 874.327871 | 874.327871 | 738.798495   560.90094  380.895725  245.799148  166.931431  127.617562  110.480698  103.733579 | 103.733579 |  7.00439  5.763117  4.374087  3.153711  2.343496  1.906459  1.702808  1.618932 |  100.0 | 774.327871 | 103.733579 |
    | 2000-01-01 23:00:00 | 888.651531 | 888.651531 | 771.242533  603.265869  421.326778  273.491086  182.399412  135.079699  113.666287  104.981456 | 104.981456 | 7.215726  6.065475  4.701578  3.419815  2.507461  1.991634  1.741352  1.634565 |  100.0 | 788.651531 | 104.981456 |
    | 2000-01-01 23:30:00 | 897.187983 | 897.187983 | 799.126157  643.404231  462.998573  304.348272  200.410805  144.126602  117.658598  106.586101 | 106.586101 | 7.392237  6.345775  5.026308  3.705386  2.694548  2.092946  1.788995  1.654495 |  100.0 | 797.187983 | 106.586101 |
    | 2000-01-02 00:00:00 |      900.0 |      900.0 | 822.070077  680.648677  504.944768  338.181261  221.143325  154.971161  122.614242  108.631884 | 108.631884 | 7.532511  6.600333  5.342575   4.00556  2.905216  2.212056  1.847237  1.679663 |  100.0 |      800.0 | 108.631884 |
    | 2000-01-02 00:30:00 | 897.264896 | 897.264896 | 839.814373  714.423121  546.226532  374.570342  244.735421  167.822108  128.705519  111.218049 | 111.218049 | 7.635922  6.826037  5.645329  4.314541  3.138919    2.3505  1.917647  1.711141 |  100.0 | 797.264896 | 111.218049 |
    | 2000-01-02 01:00:00 | 889.260928 | 889.260928 | 852.227698  744.236945  585.999477  412.861994  271.261251  182.876571  136.117041  114.459632 | 114.459632 | 7.702582  7.020421  5.930156  4.626189    3.3938  2.509592  2.001814  1.750124 |  100.0 | 789.260928 | 114.459632 |
    | 2000-01-02 01:30:00 | 876.351729 | 876.351729 | 859.307412  769.690509  623.536568  452.228123  300.688566    200.3129  145.040743  118.487709 | 118.487709 | 7.733262  7.181731  6.193233  4.934544  3.666572  2.690268  2.101304   1.79792 |  100.0 | 776.351729 | 118.487709 |
    | 2000-01-02 02:00:00 | 858.969595 | 858.969595 | 861.171397  790.486754  658.223704  491.765264  332.824622  220.282548  155.669607  123.448672 | 123.448672 | 7.729298  7.308964  6.431307  5.234169  3.952665  2.892844  2.217607  1.855917 |  100.0 | 758.969595 | 123.448672 |
    | 2000-01-02 02:30:00 |   837.5984 |   837.5984 | 858.043758  806.441904  689.547814  530.592242   367.27268   242.89764  168.190686  129.502246 | 129.502246 | 7.692497  7.401847  6.641716  5.520289   4.24664  3.116719   2.35208  1.925556 |  100.0 |   737.5984 | 129.502246 |
    | 2000-01-02 03:00:00 | 812.756821 | 812.756821 | 850.236826  817.490209  717.089675  567.915664  403.428655  268.208947  182.778174  136.818079 | 136.818079 | 7.625033  7.460791  6.822437  5.788803  4.542717  3.360115  2.505855  2.008285 |  100.0 | 712.756821 | 136.818079 |
    | 2000-01-02 03:30:00 | 784.982427 | 784.982427 | 838.131498  823.680796   740.52434   603.05718  440.530645  296.170244  199.586705   145.57092 |  145.57092 | 7.529365  7.486802  6.972119  6.036243  4.835264  3.619963  2.679705  2.105513 |  100.0 | 684.982427 |  145.57092 |
    | 2000-01-02 04:00:00 | 754.817086 | 754.817086 | 822.157419   825.16722  759.625973  635.454347   477.74451   326.59338  218.743875  155.934701 | 155.934701 | 7.408152  7.481384  7.090093  6.259752  5.119142  3.892035  2.873835   2.21857 |  100.0 | 654.817086 | 155.934701 |
    | 2000-01-02 04:30:00 |  722.79402 |  722.79402 | 802.774144  822.191729  774.272161  664.651187  514.251842   359.11186  240.339167  168.076088 | 168.076088 | 7.264188  7.446433  7.176348   6.45707  5.389861  4.171287  3.087625  2.348643 |  100.0 |  622.79402 | 168.076088 |
    | 2000-01-02 05:00:00 | 689.426711 | 689.426711 | 780.454205  815.066573   784.44387  690.289464  549.313143  393.176751  264.405154  182.148121 | 182.148121 | 7.100338   7.38413  7.231469  6.626548  5.643618  4.452333  3.319414  2.496706 |  100.0 | 589.426711 | 182.148121 |
    | 2000-01-02 05:30:00 | 655.199767 | 655.199767 | 755.668881  804.154511  790.219534  712.104484  582.298183   428.09655  290.887933  198.284114 | 198.284114 | 6.919494  7.296856  7.256552  6.767155   5.87727  4.729896   3.56639  2.663393 |  100.0 | 555.199767 | 198.284114 |
    | 2000-01-02 06:00:00 | 620.561767 | 620.561767 | 728.877238  789.850222  791.763727  729.924135  612.691172  463.109515  319.610155  216.590965 | 216.590965 | 6.724526  7.187105  7.253108  6.878466  6.088289  4.999143  3.824689  2.848827 |  100.0 | 520.561767 | 216.590965 |
    | 2000-01-02 06:30:00 | 585.920022 | 585.920022 | 700.518686  772.563933  789.312071  743.668004  640.083533  497.461733  350.240348   237.13962 |  237.13962 | 6.518248  7.057424  7.222961  6.960626  6.274741  5.255862  4.089687  3.052409 |  100.0 | 485.920022 |  237.13962 |
    | 2000-01-02 07:00:00 |  551.63713 |  551.63713 |   671.0088  752.708114  783.154402  753.343914  664.164278  530.467105  382.287485  259.949429 | 259.949429 | 6.303379  6.910365  7.168158  7.014285  6.435264  5.496517   4.35641  3.272617 |  100.0 |  451.63713 | 259.949429 |
    | 2000-01-02 07:30:00 | 518.029154 | 518.029154 | 640.737633  730.687694  773.618082    759.0407  684.712521  561.539759  415.131609  284.964028 | 284.964028 | 6.082504  6.748455  7.090883  7.040524  6.569055  5.718228  4.619945  3.506919 |  100.0 | 418.029154 | 284.964028 |
    | 2000-01-02 08:00:00 | 485.365238 | 485.365238 | 610.069326  706.893751  761.052888  760.917496   701.59241  590.202951   448.08348  312.021153 | 312.021153 | 5.858037  6.574171  6.993396  7.040768   6.67584  5.918729   4.87576  3.751831 |  100.0 | 385.365238 | 312.021153 |
    | 2000-01-02 08:30:00 |  453.86844 |  453.86844 | 579.341696  681.700176  745.818423  759.190714  714.748769  616.084334  480.452752  340.826594 | 340.826594 | 5.632185  6.389925  6.877984  7.016697  6.755836  6.096325  5.119897  4.003158 |  100.0 |  353.86844 | 340.826594 |
    | 2000-01-02 09:00:00 | 423.717566 | 423.717566 | 548.864793  655.462387  728.274463  754.120194  724.201675  638.906481  511.604077  370.947274 | 370.947274 |  5.40692   6.19805  6.746922  6.970175  6.809689  6.249856  5.349041   4.25634 |  100.0 | 323.717566 | 370.947274 |
    | 2000-01-02 09:30:00 | 395.049792 | 395.049792 | 518.918002  628.516968  708.774195  745.995862  730.039026  658.477672  540.990819  401.833364 | 401.833364 | 5.183957  6.000772  6.602459  6.903176  6.838411  6.378667  5.560518  4.506824 |  100.0 | 295.049792 | 401.833364 |
    | 2000-01-02 10:00:00 | 367.963873 | 367.963873 | 489.746062  601.181099  687.659953  735.125905  732.407105  674.684366  568.167408   432.86615 |  432.86615 | 4.964749  5.800189  6.446797  6.817737  6.843305  6.482568  5.752248  4.750366 |  100.0 | 267.963873 |  432.86615 |
    | 2000-01-02 10:30:00 | 342.523746 | 342.523746 | 461.554942  573.750945  665.260736  721.826989  731.499827  687.484765  592.787545    463.4165 |   463.4165 | 4.750493  5.598236  6.282079  6.715918  6.825894  6.561795  5.922703  4.983234 |  100.0 | 242.523746 |   463.4165 |
    | 2000-01-02 11:00:00 | 318.762374 | 318.762374 |  434.50873  546.498683  641.890721  706.416699  727.547582   696.90236  614.595923  492.896592 | 492.896592 |  4.54215  5.396654  6.110374  6.599771  6.787859  6.616956  6.070855  5.202293 |  100.0 | 218.762374 | 492.896592 |
    | 2000-01-02 11:30:00 |  296.68568 |  296.68568 | 408.728472  519.668437  617.847968  689.208015  720.806566  703.018666  633.418624  520.794264 | 520.794264 | 4.340473  5.196962   5.93365  6.471323  6.730981  6.648974  6.196135  5.405016 |  100.0 |  196.68568 | 520.794264 |
    | 2000-01-02 12:00:00 | 276.276442 | 276.276442 | 384.293438  493.471882  593.412715  670.505444  711.549248  705.964973  649.154375  546.688554 | 546.688554 | 4.146042  5.000441  5.753757  6.332556  6.657102  6.659026  6.298394  5.589455 |  100.0 | 176.276442 | 546.688554 |
    | 2000-01-02 12:30:00 | 257.498082 | 257.498082 | 361.244738  468.084509  568.844967  650.602299  700.056323  705.913379  661.766941  570.251174 | 570.251174 | 3.959298  4.808129  5.572388  6.185393  6.568085  6.648483  6.377858  5.754183 |  100.0 | 157.498082 | 570.251174 |
    | 2000-01-02 13:00:00 | 240.298231 | 240.298231 | 339.590802  443.643488  544.381428  629.778595  686.610246  703.067639  671.278073  591.240255 | 591.240255 | 3.780571  4.620834   5.39106  6.031678  6.465788   6.61886  6.435079  5.898251 |  100.0 | 140.298231 | 591.240255 |
    | 2000-01-02 13:30:00 | 224.612057 | 224.612057 | 319.314005  420.247662  520.232165  608.299119  671.490234  697.654395  677.760412  609.491325 | 609.491325 | 3.610108  4.439154  5.211088  5.873158  6.352044  6.571765  6.470888  6.021136 |  100.0 | 124.612057 | 609.491325 |
    | 2000-01-02 14:00:00 | 210.365288 | 210.365288 | 300.377677  397.959795  496.577591  586.411392  654.968464  689.915212  681.330078  624.908252 | 624.908252 | 3.448085  4.263506  5.033576  5.711457  6.228641   6.50886  6.486341  6.122695 |  100.0 | 110.365288 | 624.908252 |
    | 2000-01-02 14:30:00 |  197.47692 |  197.47692 |  282.73276  376.810719  473.566392  564.343402  637.307164  680.099684  682.138961  637.455041 | 637.455041 | 3.294625  4.094166  4.859416  5.548058  6.097307  6.431834   6.48267  6.203126 |  100.0 |   97.47692 | 637.455041 |
    | 2000-01-02 15:00:00 | 185.861607 | 185.861607 | 266.323572  356.804787  451.314849  542.301248  618.756263  668.459716  680.366999  647.148425 | 647.148425 | 3.149793  3.931299  4.689297  5.384283  5.959688  6.342368  6.461237  6.262919 |  100.0 |  85.861607 | 647.148425 |
    | 2000-01-02 15:30:00 | 175.431719 | 175.431719 | 251.092244  337.925893  429.907745  520.466922  599.551351  655.244912  676.214759   654.05085 |  654.05085 | 3.013605  3.774999   4.52373  5.221276  5.817336  6.242119  6.423492  6.302814 |  100.0 |  75.431719 |  654.05085 |
    | 2000-01-02 16:00:00 | 166.099094 | 166.099094 |  236.98167  320.143444  409.400784  498.996583  579.911775  640.698926  669.896632  658.263599 | 658.263599 | 2.886016  3.625308  4.363067  5.060004  5.671688  6.132702  6.370936  6.323754 |  100.0 |  66.099094 | 658.263599 |
    | 2000-01-02 16:30:00 | 157.776495 | 157.776495 | 223.937004  303.417733  389.824172  478.019629  560.038822  625.056582  661.634835  659.919937 | 659.919937 | 2.766927  3.482242  4.207535  4.901252  5.524054   6.01567  6.305095  6.326841 |  100.0 |  57.776495 | 659.919937 |
    | 2000-01-02 17:00:00 | 150.378786 | 150.378786 | 211.905958  287.704424  371.186927  457.638756  540.114049  608.541548  651.654307  659.178404 | 659.178404 | 2.656178  3.345805  4.057267  4.745639  5.375598  5.892501  6.227492  6.313297 |  100.0 |  50.378786 | 659.178404 |
    | 2000-01-02 17:30:00 | 143.823868 | 143.823868 | 200.838292  272.957968  353.481451  437.931067  520.297882    591.3644  640.178495  656.216395 | 656.216395 | 2.553552   3.21599  3.912324  4.593629  5.227338  5.764581  6.139627  6.284429 |  100.0 |  43.823868 | 656.216395 |
    | 2000-01-02 18:00:00 | 138.033385 | 138.033385 | 190.684932  259.133947  336.687963  418.950124  500.728668  573.720967  627.425939  651.224256 | 651.224256 | 2.458779  3.092781  3.772727  4.445558  5.080137  5.633195  6.042958  6.241592 |  100.0 |  38.033385 | 651.224256 |
    | 2000-01-02 18:30:00 | 132.933235 | 132.933235 | 181.397097  246.190422  320.778512  400.728723  481.522314  555.790883  613.607536  644.400006 | 644.400006 | 2.371544  2.976153  3.638469  4.301647  4.934708  5.499512  5.938888  6.186166 |  100.0 |  32.933235 | 644.400006 |
    | 2000-01-02 19:00:00 | 128.453911 | 128.453911 | 172.925686  234.088439  305.720374  383.282113  462.772606  537.736385  598.924336  635.944785 | 635.944785 | 2.291495   2.86606  3.509531  4.162036  4.791625  5.364577  5.828749  6.119529 |  100.0 |  28.453911 | 635.944785 |
    | 2000-01-02 19:30:00 | 124.530701 | 124.530701 | 165.221042  222.791892  291.478775   366.61138  444.552229    519.7014  583.565759   626.05902 |  626.05902 | 2.218253  2.762433  3.385895  4.026797  4.651332  5.229308  5.713793  6.043041 |  100.0 |  24.530701 |  626.05902 |
    | 2000-01-02 20:00:00 | 121.103761 | 121.103761 | 158.233068  212.266948  278.018906  350.706755  426.914384  501.810994  567.708135  614.939268 | 614.939268 |  2.15142  2.665172  3.267541  3.895959  4.514158  5.094494  5.595179  5.958025 |  100.0 |  21.103761 | 614.939268 |
    | 2000-01-02 20:30:00 | 118.118088 | 118.118088 | 151.911611  202.481266  265.307298  335.550663  409.894906  484.171277  551.513518   602.77566 |  602.77566 | 2.090591  2.574147  3.154454  3.769521  4.380338  4.960797  5.473963  5.865753 |  100.0 |  18.118088 |  602.77566 |
    | 2000-01-02 21:00:00 | 115.523427 | 115.523427 | 146.207009  193.403192  253.312612  321.120382  393.514675  466.869773  535.128758  589.749838 | 589.749838 | 2.035361  2.489193  3.046617  3.647467  4.250029  4.828758  5.351096  5.767435 |  100.0 |  15.523427 | 589.749838 |
    | 2000-01-02 21:30:00 | 113.274094 | 113.274094 | 141.070659  185.001082  242.005932  307.390271   377.78216  449.976276  518.684856  576.033305 | 576.033305 | 1.985327  2.410119  2.944009  3.529775  4.123323  4.698808  5.227421  5.664208 |  100.0 |  13.274094 | 576.033305 |
    | 2000-01-02 22:00:00 | 111.328777 | 111.328777 | 136.461306  177.240521  231.360436  294.332346  362.695202  433.543186  502.294084  561.781496 | 561.781496 | 1.940089  2.336716  2.846593  3.416426   4.00027  4.571282  5.103669  5.557131 |  100.0 |  11.328777 | 561.781496 |
    | 2000-01-02 22:30:00 | 109.650285 | 109.650285 | 132.320953  170.095174  221.352002  281.922564   348.24623  417.611102  486.060973  547.152255 | 547.152255 | 1.899281  2.268722  2.754331  3.307399  3.880887  4.446425   4.98046  5.447166 |  100.0 |   9.650285 | 547.152255 |
    | 2000-01-02 23:00:00 | 108.205287 | 108.205287 | 128.614683  163.526887  211.957568  270.133977  334.420361  402.204236  470.064722   532.27419 |  532.27419 | 1.862545  2.205881  2.667155  3.202681  3.765172  4.324414  4.858311  5.335189 |  100.0 |   8.205287 |  532.27419 |
    | 2000-01-02 23:30:00 | 106.964036 | 106.964036 | 125.302473  157.501969   203.15477  258.942673  321.199735  387.337102  454.372995  517.267835 | 517.267835 | 1.829535  2.147921   2.58498  3.102263  3.653109  4.205367  4.737646  5.221979 |  100.0 |   6.964036 | 517.267835 |
    | 2000-01-03 00:00:00 | 105.900084 | 105.900084 | 122.347031  151.986541  194.921648  248.326575  308.564478  373.015122  439.039334  502.239752 | 502.239752 | 1.799931   2.09457  2.507697  3.006134  3.544681  4.089356  4.618803  5.108218 |  100.0 |   5.900084 | 502.239752 |
    | 2000-01-03 00:30:00 |  104.99001 |  104.99001 | 119.713819  146.946901  187.236176  238.265441  296.493918  359.236502  424.104399  487.282557 | 487.282557 | 1.773428  2.045551  2.435176  2.914282   3.43987  3.976419   4.50204  4.994497 |  100.0 |    4.99001 | 487.282557 |
    | 2000-01-03 01:00:00 | 104.213141 | 104.213141 |  117.37102  142.349906  180.075942  228.740673  284.967511  345.993966  409.597403  472.475217 | 472.475217 | 1.749744  2.000589   2.36727  2.826685  3.338661  3.866572  4.387554  4.881317 |  100.0 |   4.213141 | 472.475217 |
    | 2000-01-03 01:30:00 | 103.551298 | 103.551298 | 115.289448  138.163306  173.417969  219.734996  273.965495  333.276294  395.537628  457.883613 | 457.883613 | 1.728615  1.959416  2.303811  2.743314  3.241047  3.759812  4.275484  4.769094 |  100.0 |   3.551298 | 457.883613 |
    | 2000-01-03 02:00:00 | 102.988541 | 102.988541 | 113.442426  134.356023  167.238694  211.232062  263.469306  321.069642   381.93598  443.561327 | 443.561327 | 1.709798  1.921771  2.244622  2.664126   3.14702  3.656131  4.165924  4.658169 |  100.0 |   2.988541 | 443.561327 |
    | 2000-01-03 02:30:00 |  102.51094 |  102.51094 | 111.805648  130.898372  161.514049  203.216038  253.461786  309.358651  368.796495  429.550623 | 429.550623 | 1.693068  1.887404  2.189514  2.589063  3.056577  3.555517  4.058936  4.548811 |  100.0 |    2.51094 | 429.550623 |
    | 2000-01-03 03:00:00 | 102.106358 | 102.106358 | 110.357028  127.762211  156.219638  195.671239  243.927215  298.127329  356.117758  415.883572 | 415.883572 | 1.678216  1.856075  2.138291   2.51805  2.969716  3.457957  3.954553  4.441233 |  100.0 |   2.106358 | 415.883572 |
    | 2000-01-03 03:30:00 | 101.764251 | 101.764251 | 109.076541  124.921026  151.330958  188.581815  234.851207  287.359734  343.894195   402.58326 |  402.58326 | 1.665054  1.827555  2.090753  2.450998  2.886431  3.363442  3.852791  4.335592 |  100.0 |   1.764251 |  402.58326 |
    | 2000-01-03 04:00:00 |  101.47548 |  101.47548 | 107.946073   122.34998  146.823636  181.931539  226.220507  277.040468  332.117213  389.665032 | 389.665032 | 1.653407  1.801628    2.0467  2.387804  2.806711  3.271966  3.753653  4.232006 |  100.0 |    1.47548 | 389.665032 |
    | 2000-01-03 04:30:00 | 101.232151 | 101.232151 | 106.949265  120.025909  142.673664   175.70368  218.022721  267.155009  320.776183  377.137719 | 377.137719 | 1.643116   1.77809  2.005934   2.32835  2.730537  3.183528  3.657135  4.130556 |  100.0 |   1.232151 | 377.137719 |
    | 2000-01-03 05:00:00 | 101.027458 | 101.027458 | 106.071374  117.927292  138.857605  169.880974  210.246024  257.689884  309.859262  365.004811 | 365.004811 | 1.634037  1.756749   1.96826  2.272508   2.65788  3.098129   3.56323  4.031294 |  100.0 |   1.027458 | 365.004811 |
    | 2000-01-03 05:30:00 | 100.855551 | 100.855551 | 105.299136  116.034197  135.352771  164.445671  202.878864  248.632733  299.354061  353.265548 | 353.265548 | 1.626039  1.737426  1.933486  2.220141    2.5887  3.015772  3.471932  3.934253 |  100.0 |   0.855551 | 353.265548 |
    | 2000-01-03 06:00:00 | 100.711411 | 100.711411 | 104.620632  114.328217  132.137357  159.379637  195.909698  239.972259   289.24816  341.915902 | 341.915902 | 1.619002  1.719953   1.90143  2.171105  2.522946  2.936459  3.383235  3.839451 |  100.0 |   0.711411 | 341.915902 |
    | 2000-01-03 06:30:00 | 100.590745 | 100.590745 | 104.025177  112.792384  129.190547    154.6645  189.326782  231.698108    279.5295   330.94945 |  330.94945 |  1.61282  1.704172  1.871913  2.125252  2.460554  2.860191  3.297139  3.746895 |  100.0 |   0.590745 |  330.94945 |
    | 2000-01-03 07:00:00 | 100.489885 | 100.489885 | 103.503201  111.411093  126.492575  150.281807  183.118014  223.800686  270.186645  320.358121 | 320.358121 | 1.607396  1.689937  1.844767  2.082431  2.401449  2.786962  3.213644  3.656586 |  100.0 |   0.489885 | 320.358121 |
    | 2000-01-03 07:30:00 | 100.405708 | 100.405708 |  103.04615  110.170014  124.024767  146.213196  177.270845   216.27095  261.208944  310.132826 | 310.132826 | 1.602644  1.677115  1.819828  2.042492  2.345545  2.716762  3.132756  3.568523 |  100.0 |   0.405708 | 310.132826 |
    | 2000-01-03 08:00:00 |  100.33556 |  100.33556 | 102.646388  109.056006  121.769545  142.440546   171.77225  209.100189  252.586598  300.263973 | 300.263973 | 1.598484  1.665577  1.796944  2.005284  2.292748  2.649571   3.05448  3.482704 |  100.0 |    0.33556 | 300.263973 |
    | 2000-01-03 08:30:00 | 100.277186 | 100.277186 | 102.297111  108.057036  119.710424  138.946123  166.608754  202.279806  244.310652  290.741875 | 290.741875 | 1.594848  1.655209  1.775968   1.97066  2.242955  2.585363  2.978825  3.399128 |  100.0 |   0.277186 | 290.741875 |
    | 2000-01-03 09:00:00 | 100.228681 | 100.228681 | 101.992265  107.162099  117.831985  135.712693  161.766498   195.80113   236.37292  281.557064 | 281.557064 | 1.591674  1.645904  1.756762  1.938475  2.196058  2.524101  2.905795  3.317797 |  100.0 |   0.228681 | 281.557064 |
    | 2000-01-03 09:30:00 | 100.188431 | 100.188431 | 101.726473  106.361141  116.119844  132.723618  157.231334  189.655269  228.765872  272.700507 | 272.700507 | 1.588905  1.637561  1.739196  1.908588  2.151945  2.465738  2.835396  3.238714 |  100.0 |   0.188431 | 272.700507 |
    | 2000-01-03 10:00:00 | 100.155078 | 100.155078 | 101.494962  105.644991  114.560604  129.962928  152.988943  183.832989  221.482483  264.163748 | 264.163748 | 1.586494  1.630091  1.723148  1.880864  2.110498  2.410219  2.767627  3.161888 |  100.0 |   0.155078 | 264.163748 |
    | 2000-01-03 10:30:00 | 100.127477 | 100.127477 | 101.293509  105.005287  113.141816  127.415366   149.02495  178.324658  214.516066  255.938985 | 255.938985 | 1.584395   1.62341  1.708502  1.855171    2.0716  2.357479  2.702483  3.087327 |  100.0 |   0.127477 | 255.938985 |
    | 2000-01-03 11:00:00 | 100.104666 | 100.104666 | 101.118378  104.434421  111.851922  125.066422  145.325047  173.120217  207.860109  248.019077 | 248.019077 |  1.58257  1.617441  1.695151  1.831385  2.035135  2.307448  2.639951  3.015041 |  100.0 |   0.104666 | 248.019077 |
    | 2000-01-03 11:30:00 | 100.085838 | 100.085838 | 100.966269  103.925471  110.680208  122.902346  141.875101  168.209195  201.508112  240.397508 | 240.397508 | 1.580985  1.612115  1.682992  1.809385  2.000984  2.260047  2.580013   2.94504 |  100.0 |   0.085838 | 240.397508 |
    | 2000-01-03 12:00:00 | 100.070317 | 100.070317 | 100.834274  103.472154  109.616755  120.910146  138.661253   163.58076  195.453452  233.068316 | 233.068316 |  1.57961  1.607366  1.671932  1.789057  1.969032  2.215191  2.522642  2.877333 |  100.0 |   0.070317 | 233.068316 |
    | 2000-01-03 12:30:00 | 100.057539 | 100.057539 | 100.719833  103.068767  108.652385  119.077579  135.669998  159.223778  189.689272  226.025984 | 226.025984 | 1.578418  1.603138  1.661881  1.770292  1.939166  2.172795  2.467804  2.811927 |  100.0 |   0.057539 | 226.025984 |
    | 2000-01-03 13:00:00 | 100.047032 | 100.047032 | 100.620693  102.710144   107.77862  117.393137  132.888255  155.126904  184.208398  219.265319 | 219.265319 | 1.577385  1.599377  1.652758  1.752986  1.911274  2.132765  2.415457  2.748824 |  100.0 |   0.047032 | 219.265319 |
    | 2000-01-03 13:30:00 | 100.038402 | 100.038402 | 100.534879  102.391612   106.98763  115.846017  130.303413  151.278669  179.003291  212.781327 | 212.781327 | 1.576492  1.596035  1.644485  1.737041  1.885251   2.09501  2.365553  2.688024 |  100.0 |   0.038402 | 212.781327 |
    | 2000-01-03 14:00:00 | 100.031322 | 100.031322 | 100.460657  102.108944  106.272193  114.426097  127.903373   147.66757  174.066033  206.569081 | 206.569081 | 1.575719  1.593068  1.636991  1.722363  1.860992  2.059434  2.318036  2.629518 |  100.0 |   0.031322 | 206.569081 |
    | 2000-01-03 14:30:00 | 100.025521 | 100.025521 |  100.39651  101.858325  105.625656  113.123907   125.67657  144.282159  169.388332  200.623601 | 200.623601 | 1.575051  1.590437   1.63021  1.708867  1.838397  2.025944  2.272845  2.573295 |  100.0 |   0.025521 | 200.623601 |
    | 2000-01-03 15:00:00 | 100.020773 | 100.020773 | 100.341109  101.636317   105.04189  111.930594  123.611982  141.111122  164.961558  194.939755 | 194.939755 | 1.574475  1.588106   1.62408  1.696467  1.817371  1.994446  2.229914  2.519334 |  100.0 |   0.020773 | 194.939755 |
    | 2000-01-03 15:30:00 | 100.016892 | 100.016892 | 100.293296  101.439823  104.515259  110.837893  121.699139  138.143349  160.776792  189.512176 | 189.512176 | 1.573977  1.586042  1.618544  1.685086   1.79782  1.964846  2.189174   2.46761 |  100.0 |   0.016892 | 189.512176 |
    | 2000-01-03 16:00:00 | 100.013722 | 100.013722 | 100.252059  101.266058  104.040583  109.838093  119.928119  135.367995  156.824888  184.335201 | 184.335201 | 1.573548  1.584217   1.61355  1.674651  1.779658  1.937054  2.150551  2.418092 |  100.0 |   0.013722 | 184.335201 |
    | 2000-01-03 16:30:00 | 100.011136 | 100.011136 | 100.216516  101.112519  103.613106  108.924008  118.289533  132.774526  153.096544  179.402836 | 179.402836 | 1.573178  1.582604  1.609049  1.665091    1.7628  1.910981  2.113972  2.370741 |  100.0 |   0.011136 | 179.402836 |
    | 2000-01-03 17:00:00 | 100.009028 | 100.009028 |   100.1859  100.976962  103.228467  108.088944  116.774519  130.352757   149.58237  174.708744 | 174.708744 |  1.57286   1.58118  1.604997  1.656342  1.747166   1.88654  2.079358  2.325513 |  100.0 |   0.009028 | 174.708744 |
    | 2000-01-03 17:30:00 | 100.007313 | 100.007313 | 100.159544  100.857376  102.882668  107.326673  115.374718  128.092882  146.272961  170.246251 | 170.246251 | 1.572586  1.579924  1.601351  1.648343   1.73268  1.863648  2.046634  2.282361 |  100.0 |   0.007313 | 170.246251 |
    | 2000-01-03 18:00:00 | 100.005918 | 100.005918 | 100.136867  100.751961  102.572052  106.631402  114.082262  125.985489  143.158961  166.008373 | 166.008373 |  1.57235  1.578817  1.598075  1.641036  1.719268  1.842222  2.015722  2.241232 |  100.0 |   0.005918 | 166.008373 |
    | 2000-01-03 18:30:00 | 100.004785 | 100.004785 | 100.117366  100.659108  102.293273   105.99775  112.889746  124.021572   140.23112  161.987853 | 161.987853 | 1.572148  1.577842  1.595133  1.634368  1.706863  1.822185  1.986545   2.20207 |  100.0 |   0.004785 | 161.987853 |
    | 2000-01-03 19:00:00 | 100.003865 | 100.003865 | 100.100606   100.57738  102.043278   105.42072  111.790211  122.192539  137.480351   158.17721 |  158.17721 | 1.571973  1.576983  1.592495  1.628289  1.695398   1.80346  1.959027  2.164816 |  100.0 |   0.003865 |  158.17721 |
    | 2000-01-03 19:30:00 | 100.003119 | 100.003119 | 100.086208  100.505495  101.819278  104.895675  110.777126  120.490206  134.897767  154.568791 | 154.568791 | 1.571824  1.576229   1.59013  1.622751  1.684811  1.785977  1.933093  2.129408 |  100.0 |   0.003119 | 154.568791 |
    | 2000-01-03 20:00:00 | 100.002515 | 100.002515 | 100.073845  100.442313  101.618735  104.418318  109.844358  118.906796  132.474722  151.154832 | 151.154832 | 1.571695  1.575565  1.588012  1.617712  1.675044  1.769664  1.908671  2.095784 |  100.0 |   0.002515 | 151.154832 |
    | 2000-01-03 20:30:00 | 100.002026 | 100.002026 | 100.063234  100.386818  101.439336  103.984667   108.98616  117.434928  130.202836  147.927508 | 147.927508 | 1.571585  1.574983  1.586118   1.61313  1.666042  1.754457  1.885689  2.063879 |  100.0 |   0.002026 | 147.927508 |
    | 2000-01-03 21:00:00 | 100.001631 | 100.001631 | 100.054131  100.338106  101.278979   103.59104  108.197145  116.067609   128.07402  144.878992 | 144.878992 | 1.571491  1.574472  1.584424  1.608969  1.657751   1.74029  1.864077   2.03363 |  100.0 |   0.001631 | 144.878992 |
    | 2000-01-03 21:30:00 | 100.001312 | 100.001312 | 100.046325  100.295377  101.135755  103.234029  107.472268  114.798219  126.080482  142.001505 | 142.001505 |  1.57141  1.574023  1.582912  1.605192  1.650122  1.727103  1.843769  2.004973 |  100.0 |   0.001312 | 142.001505 |
    | 2000-01-03 22:00:00 | 100.001054 | 100.001054 | 100.039633  100.257919   101.00793  102.910488   106.80681  113.620498  124.214748  139.287354 | 139.287354 |  1.57134  1.573631  1.581562  1.601767  1.643108  1.714838  1.824699  1.977842 |  100.0 |   0.001054 | 139.287354 |
    | 2000-01-03 22:30:00 | 100.000846 | 100.000846 | 100.033899  100.225101  100.893936  102.617515  106.196356  112.528533  122.469654  136.728979 | 136.728979 | 1.571281  1.573286  1.580358  1.598665  1.636666  1.703441  1.806805  1.952175 |  100.0 |   0.000846 | 136.728979 |
    | 2000-01-03 23:00:00 | 100.000679 | 100.000679 | 100.028988  100.196366  100.792352  102.352431   105.63678   111.51674  120.838357   134.31898 |  134.31898 |  1.57123  1.572985  1.579285  1.595857  1.630755  1.692857  1.790026   1.92791 |  100.0 |   0.000679 |  134.31898 |
    | 2000-01-03 23:30:00 | 100.000544 | 100.000544 | 100.024782  100.171221  100.701891  102.112771   105.12423  110.579852  119.314325  132.050147 | 132.050147 | 1.571186  1.572722   1.57833  1.593317  1.625334  1.683037  1.774304  1.904985 |  100.0 |   0.000544 | 132.050147 |
    | 2000-01-04 00:00:00 | 100.000436 | 100.000436 | 100.021181  100.149229  100.621394  101.896267   104.65511  109.712903  117.891335   129.91548 |  129.91548 | 1.571149  1.572491   1.57748  1.591022  1.620368  1.673933  1.759583  1.883342 |  100.0 |   0.000436 |  129.91548 |
    | 2000-01-04 00:30:00 | 100.000349 | 100.000349 |   100.0181  100.130005  100.549813  101.700831  104.226066  108.911216  116.563468  127.908202 | 127.908202 | 1.571117   1.57229  1.576724   1.58895  1.615823  1.665499  1.745809  1.862921 |  100.0 |   0.000349 | 127.908202 |
    | 2000-01-04 01:00:00 | 100.000279 | 100.000279 | 100.015463   100.11321  100.486205  101.524551  103.833974  108.170385  115.325094  126.021778 | 126.021778 |  1.57109  1.572114  1.576053  1.587081  1.611666  1.657694  1.732931  1.843667 |  100.0 |   0.000279 | 126.021778 |
    | 2000-01-04 01:30:00 | 100.000223 | 100.000223 | 100.013209  100.098544  100.429719  101.365668  103.475923  107.486262   114.17087  124.249912 | 124.249912 | 1.571066   1.57196  1.575457  1.585397  1.607867  1.650475  1.720898  1.825525 |  100.0 |   0.000223 | 124.249912 |
    | 2000-01-04 02:00:00 | 100.000178 | 100.000178 | 100.011281  100.085744   100.37959  101.222573  103.149209  106.854947  113.095723  122.586562 | 122.586562 | 1.571046  1.571826  1.574928  1.583879  1.604399  1.643805  1.709665  1.808442 |  100.0 |   0.000178 | 122.586562 |
    | 2000-01-04 02:30:00 | 100.000142 | 100.000142 | 100.009632  100.074578  100.335133  101.093794  102.851314  106.272774  112.094846   121.02593 |  121.02593 | 1.571029  1.571709  1.574459  1.582514  1.601236  1.637646  1.699185  1.792366 |  100.0 |   0.000142 |  121.02593 |
    | 2000-01-04 03:00:00 | 100.000113 | 100.000113 | 100.008224  100.064842  100.295729  100.977981  102.579903  105.736299  111.163682  119.562469 | 119.562469 | 1.571015  1.571607  1.574043  1.581286  1.598352  1.631965  1.689416  1.777248 |  100.0 |   0.000113 | 119.562469 |
    | 2000-01-04 03:30:00 |  100.00009 |  100.00009 |  100.00702  100.056356  100.260827  100.873906  102.332807  105.242287  110.297915  118.190876 | 118.190876 | 1.571002  1.571518  1.573675  1.580182  1.595726  1.626728  1.680316  1.763041 |  100.0 |    0.00009 | 118.190876 |
    | 2000-01-04 04:00:00 | 100.000072 | 100.000072 | 100.005992  100.048963   100.22993  100.780444  102.108017  104.787703  109.493462  116.906087 | 116.906087 | 1.570991  1.571441  1.573349  1.579191  1.593337  1.621904  1.671845  1.749698 |  100.0 |   0.000072 | 116.906087 |
    | 2000-01-04 04:30:00 | 100.000057 | 100.000057 | 100.005113  100.042526  100.202596  100.696572  101.903672  104.369702  108.746457  115.703271 | 115.703271 | 1.570982  1.571374  1.573061  1.578302  1.591164  1.617465  1.663967  1.737176 |  100.0 |   0.000057 | 115.703271 |
    | 2000-01-04 05:00:00 | 100.000046 | 100.000046 | 100.004363  100.036922  100.178426  100.621358  101.718049  103.985618  108.053248  114.577825 | 114.577825 | 1.570975  1.571315  1.572806  1.577505   1.58919  1.613384  1.656646  1.725431 |  100.0 |   0.000046 | 114.577825 |
    | 2000-01-04 05:30:00 | 100.000036 | 100.000036 | 100.003722  100.032047  100.157067  100.553953  101.549558  103.632954   107.41038  113.525365 | 113.525365 | 1.570968  1.571264  1.572581   1.57679  1.587398  1.609634  1.649846  1.714424 |  100.0 |   0.000036 | 113.525365 |
    | 2000-01-04 06:00:00 | 100.000029 | 100.000029 | 100.003175  100.027806  100.138203  100.493587  101.396727  103.309375  106.814593  112.541718 | 112.541718 | 1.570962   1.57122  1.572383  1.576151  1.585772  1.606191  1.643537  1.704116 |  100.0 |   0.000029 | 112.541718 |
    | 2000-01-04 06:30:00 | 100.000023 | 100.000023 | 100.002709  100.024119   100.12155  100.439561  101.258203  103.012696  106.262807  111.622917 | 111.622917 | 1.570957  1.571181  1.572207  1.575578  1.584299  1.603033  1.637687  1.694467 |  100.0 |   0.000023 | 111.622917 |
    | 2000-01-04 07:00:00 | 100.000018 | 100.000018 |  100.00231  100.020915  100.106857  100.391239  101.132734  102.740879  105.752116  110.765189 | 110.765189 | 1.570953  1.571148  1.572053  1.575066  1.582964  1.600138  1.632266  1.685444 |  100.0 |   0.000018 | 110.765189 |
    | 2000-01-04 07:30:00 | 100.000014 | 100.000014 |  100.00197  100.018131  100.093901  100.348047  101.019171   102.49202   105.27978   109.96495 |  109.96495 |  1.57095  1.571119  1.571916  1.574609  1.581756  1.597487  1.627248   1.67701 |  100.0 |   0.000014 |  109.96495 |
    | 2000-01-04 08:00:00 | 100.000011 | 100.000011 |  100.00168  100.015713  100.082482  100.309464  100.916455  102.264343  104.843216  109.218794 | 109.218794 | 1.570947  1.571093  1.571796    1.5742  1.580663  1.595061  1.622606  1.669134 |  100.0 |   0.000011 | 109.218794 |
    | 2000-01-04 08:30:00 | 100.000009 | 100.000009 | 100.001433  100.013614  100.072422  100.275019  100.823615  102.056194  104.439992  108.523491 | 108.523491 | 1.570944  1.571071   1.57169  1.573836  1.579676  1.592842  1.618316  1.661784 |  100.0 |   0.000009 | 108.523491 |
    | 2000-01-04 09:00:00 | 100.000007 | 100.000007 | 100.001221  100.011792  100.063564  100.244286  100.739757  101.866035  104.067818  107.875974 | 107.875974 | 1.570942  1.571052  1.571597   1.57351  1.578784  1.590815  1.614352  1.654929 |  100.0 |   0.000007 | 107.875974 |
    | 2000-01-04 09:30:00 | 100.000006 | 100.000006 | 100.001041  100.010212  100.055767  100.216882  100.664064  101.692434   103.72454  107.273336 | 107.273336 |  1.57094  1.571036  1.571515   1.57322  1.577979  1.588964  1.610695  1.648542 |  100.0 |   0.000006 | 107.273336 |
    | 2000-01-04 10:00:00 | 100.000004 | 100.000004 | 100.000888  100.008841  100.048908   100.19246  100.595786  101.534059  103.408133  106.712818 | 106.712818 | 1.570939  1.571022  1.571443  1.572962  1.577253  1.587275  1.607321  1.642593 |  100.0 |   0.000004 | 106.712818 |
    | 2000-01-04 10:30:00 | 100.000004 | 100.000004 | 100.000757  100.007652  100.042877  100.170708  100.534236  101.389677  103.116696  106.191808 | 106.191808 | 1.570937  1.571009   1.57138  1.572732  1.576599  1.585735  1.604213  1.637058 |  100.0 |   0.000004 | 106.191808 |
    | 2000-01-04 11:00:00 | 100.000003 | 100.000003 | 100.000645  100.006621  100.037576  100.151345  100.478787  101.258143  102.848444  105.707833 | 105.707833 | 1.570936  1.570998  1.571324  1.572527  1.576009  1.584332   1.60135  1.631911 |  100.0 |   0.000003 | 105.707833 |
    | 2000-01-04 11:30:00 | 100.000002 | 100.000002 |  100.00055  100.005728  100.032919  100.134117  100.428866  101.138394  102.601704  105.258551 | 105.258551 | 1.570935  1.570989  1.571275  1.572345  1.575479  1.583055  1.598716  1.627129 |  100.0 |   0.000002 | 105.258551 |
    | 2000-01-04 12:00:00 | 100.000002 | 100.000002 | 100.000469  100.004954  100.028829  100.118797  100.383949   101.02945  102.374906  104.841747 | 104.841747 | 1.570934  1.570981  1.571232  1.572183  1.575001  1.581893  1.596294  1.622688 |  100.0 |   0.000002 | 104.841747 |
    | 2000-01-04 12:30:00 | 100.000001 | 100.000001 | 100.000399  100.004284  100.025239  100.105182  100.343561  100.930402  102.166583  104.455325 | 104.455325 | 1.570934  1.570974  1.571194  1.572039  1.574572  1.580837  1.594069  1.618569 |  100.0 |   0.000001 | 104.455325 |
    | 2000-01-04 13:00:00 | 100.000001 | 100.000001 |  100.00034  100.003704  100.022088  100.093087  100.307265  100.840411  101.975361  104.097307 | 104.097307 | 1.570933  1.570968  1.571161  1.571911  1.574187  1.579877  1.592026  1.614749 |  100.0 |   0.000001 | 104.097307 |
    | 2000-01-04 13:30:00 | 100.000001 | 100.000001 |  100.00029  100.003201  100.019324  100.082348  100.274668  100.758702  101.799955  103.765824 | 103.765824 | 1.570932  1.570963  1.571132  1.571798  1.573841  1.579006  1.590152  1.611211 |  100.0 |   0.000001 | 103.765824 |
    | 2000-01-04 14:00:00 | 100.000001 | 100.000001 | 100.000247  100.002767  100.016901  100.072818  100.245408  100.684561  101.639167  103.459113 | 103.459113 | 1.570932  1.570958  1.571107  1.571697   1.57353  1.578215  1.588434  1.607935 |  100.0 |   0.000001 | 103.459113 |
    | 2000-01-04 14:30:00 | 100.000001 | 100.000001 | 100.000211   100.00239  100.014777  100.064365   100.21916  100.617332  101.491878   103.17551 |  103.17551 | 1.570932  1.570954  1.571084  1.571608  1.573251  1.577499   1.58686  1.604904 |  100.0 |   0.000001 |  103.17551 |
    | 2000-01-04 15:00:00 |      100.0 |      100.0 |  100.00018  100.002065  100.012916  100.056871  100.195627  100.556407  101.357044  102.913449 | 102.913449 | 1.570931  1.570951  1.571065  1.571529  1.573002  1.576849  1.585418  1.602103 |  100.0 |        0.0 | 102.913449 |
    | 2000-01-04 15:30:00 |      100.0 |      100.0 | 100.000153  100.001783  100.011286   100.05023  100.174539  100.501231  101.233696  102.671453 | 102.671453 | 1.570931  1.570948  1.571048  1.571459  1.572778  1.576261    1.5841  1.599515 |  100.0 |        0.0 | 102.671453 |
    | 2000-01-04 16:00:00 |      100.0 |      100.0 |  100.00013   100.00154  100.009859  100.044347  100.155653  100.451291  101.120927  102.448135 | 102.448135 | 1.570931  1.570945  1.571033  1.571397  1.572578  1.575729  1.582895  1.597126 |  100.0 |        0.0 | 102.448135 |
    | 2000-01-04 16:30:00 |      100.0 |      100.0 | 100.000111   100.00133   100.00861  100.039138  100.138748  100.406118    101.0179  102.242189 | 102.242189 | 1.570931  1.570943   1.57102  1.571342  1.572398  1.575248  1.581793  1.594922 |  100.0 |        0.0 | 102.242189 |
    | 2000-01-04 17:00:00 |      100.0 |      100.0 | 100.000094  100.001143  100.007492  100.034435  100.123343  100.364596  100.922409  102.049826 | 102.049826 |  1.57093  1.570941  1.571008  1.571293  1.572238  1.574814  1.580789  1.592893 |  100.0 |        0.0 | 102.049826 |
    | 2000-01-04 17:30:00 |      100.0 |      100.0 |  100.00008  100.000987  100.006542  100.030378  100.109876  100.327827  100.836807  101.875381 | 101.875381 |  1.57093   1.57094  1.570998   1.57125  1.572095  1.574421  1.579873  1.591023 |  100.0 |        0.0 | 101.875381 |
    | 2000-01-04 18:00:00 |      100.0 |      100.0 | 100.000068  100.000853  100.005711   100.02679  100.097839  100.294624  100.758745  101.714821 | 101.714821 |  1.57093  1.570938  1.570989  1.571212  1.571967  1.574067  1.579038  1.589302 |  100.0 |        0.0 | 101.714821 |
    | 2000-01-04 18:30:00 |      100.0 |      100.0 | 100.000058  100.000736  100.004984  100.023618  100.087084  100.264657  100.687603  101.567135 | 101.567135 |  1.57093  1.570937  1.570982  1.571179  1.571853  1.573747  1.578276  1.587719 |  100.0 |        0.0 | 101.567135 |
    | 2000-01-04 19:00:00 |      100.0 |      100.0 |  100.00005  100.000635  100.004349  100.020814   100.07748  100.237626  100.622807  101.431378 | 101.431378 |  1.57093  1.570936  1.570975  1.571149  1.571751  1.573459  1.577583  1.586264 |  100.0 |        0.0 | 101.431378 |
    | 2000-01-04 19:30:00 |      100.0 |      100.0 | 100.000042  100.000548  100.003793  100.018337  100.068907  100.213257  100.563828  101.306666 | 101.306666 |  1.57093  1.570935  1.570969  1.571123   1.57166  1.573199  1.576952  1.584927 |  100.0 |        0.0 | 101.306666 |
    | 2000-01-04 20:00:00 |      100.0 |      100.0 | 100.000036  100.000473  100.003308   100.01615  100.061258    100.1913  100.510174  101.192173 | 101.192173 |  1.57093  1.570934  1.570964    1.5711  1.571579  1.572965  1.576378  1.583699 |  100.0 |        0.0 | 101.192173 |
    | 2000-01-04 20:30:00 |      100.0 |      100.0 | 100.000031  100.000408  100.002884  100.014219  100.054438  100.171526  100.461395  101.087128 | 101.087128 |  1.57093  1.570934   1.57096  1.571079  1.571506  1.572754  1.575856  1.582573 |  100.0 |        0.0 | 101.087128 |
    | 2000-01-04 21:00:00 |      100.0 |      100.0 | 100.000026  100.000352  100.002514  100.012515  100.048358  100.153728  100.417073  100.990812 | 100.990812 |  1.57093  1.570933  1.570956  1.571061  1.571442  1.572565  1.575382   1.58154 |  100.0 |        0.0 | 100.990812 |
    | 2000-01-04 21:30:00 |      100.0 |      100.0 | 100.000022  100.000304  100.002191  100.011012  100.042941  100.137716  100.376825  100.902553 | 100.902553 |  1.57093  1.570933  1.570952  1.571046  1.571384  1.572394  1.574952  1.580594 |  100.0 |        0.0 | 100.902553 |
    | 2000-01-04 22:00:00 |      100.0 |      100.0 | 100.000019  100.000262  100.001909  100.009686  100.038117  100.123319  100.340296  100.821727 | 100.821727 |  1.57093  1.570932  1.570949  1.571032  1.571333  1.572241  1.574561  1.579727 |  100.0 |        0.0 | 100.821727 |
    | 2000-01-04 22:30:00 |      100.0 |      100.0 | 100.000016  100.000226  100.001663  100.008518  100.033823  100.110381  100.307162  100.747753 | 100.747753 |  1.57093  1.570932  1.570947  1.571019  1.571288  1.572103  1.574207  1.578934 |  100.0 |        0.0 | 100.747753 |
    | 2000-01-04 23:00:00 |      100.0 |      100.0 | 100.000014  100.000195  100.001448  100.007488  100.030001  100.098758  100.277124   100.68009 |  100.68009 |  1.57093  1.570931  1.570945  1.571008  1.571247  1.571979  1.573886  1.578209 |  100.0 |        0.0 |  100.68009 |
    | 2000-01-04 23:30:00 |      100.0 |      100.0 | 100.000012  100.000168  100.001261  100.006581  100.026602  100.088323  100.249907  100.618238 | 100.618238 |  1.57093  1.570931  1.570943  1.570999  1.571211  1.571868  1.573595  1.577546 |  100.0 |        0.0 | 100.618238 |
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
# ...from lstream
from hydpy.models.lstream import lstream_fluxes
from hydpy.models.lstream import lstream_model
from hydpy.models.lstream import lstream_solver


class Model(lstream_model.Model):
    """Version 3 of HydPy-L-Stream."""
    SOLVERPARAMETERS = (
        lstream_solver.AbsErrorMax,
        lstream_solver.RelErrorMax,
        lstream_solver.RelDTMin,
        lstream_solver.RelDTMax,
    )
    SOLVERSEQUENCES = (
        lstream_fluxes.QG,
    )
    INLET_METHODS = (
        lstream_model.Pick_Q_V1,
    )
    RECEIVER_METHODS = ()
    ADD_METHODS = ()
    PART_ODE_METHODS = (
        lstream_model.Calc_QZA_V1,
        lstream_model.Calc_QG_V2,
        lstream_model.Calc_QA_V1,
    )
    FULL_ODE_METHODS = (
        lstream_model.Update_VG_V1,
    )
    OUTLET_METHODS = (
        lstream_model.Pass_Q_V1,
    )
    SENDER_METHODS = ()
    SUBMODELS = ()


tester = Tester()
cythonizer = Cythonizer()
cythonizer.finalise()
