import pytest
from gmsh_utils.gmsh_IO import GmshIO
import numpy.testing as npt
import numpy as np

from stem.geometry import *


class TestGeometry:

    @pytest.fixture
    def expected_geo_data_0D(self):
        """
        Expected geometry data for a 0D geometry group. The group is a geometry of a point

        Returns:
            - Dict[str, Any]: dictionary containing the geometry data as generated by the gmsh_io
        """
        expected_points = {1: [0, 0, 0], 2: [0.5, 0, 0]}
        return {"points": expected_points}

    @pytest.fixture
    def expected_geo_data_1D(self):
        """
        Expected geometry data for a 1D geometry group. The group is a geometry of a line

        Returns:
            - Dict[str, Any]: dictionary containing the geometry data as generated by the gmsh_io
        """
        expected_points = {4: [0, 1.0, 0], 11: [0, 2.0, 0], 12: [0.5, 2.0, 0]}
        expected_lines = {13: [4, 11], 14: [11, 12]}

        return {"points": expected_points, "lines": expected_lines}

    @pytest.fixture
    def expected_geo_data_2D(self):
        """
        Expected geometry data for a 2D geometry group. The group is a geometry of a square.

        Returns:
            - Dict[str, Any]: dictionary containing the geometry data as generated by the gmsh_io
        """
        expected_points = {3: [0.5, 1, 0], 4: [0, 1, 0], 11: [0, 2, 0], 12: [0.5, 2.0, 0]}
        expected_lines = {7: [3, 4], 13: [4, 11], 14: [11, 12], 15: [12, 3]}
        expected_surfaces = {17: [-13, -7, -15, -14]}

        return {"points": expected_points, "lines": expected_lines, "surfaces": expected_surfaces}

    @pytest.fixture
    def expected_geo_data_3D(self):
        """
        Expected geometry data for a 3D geometry group. The group is a geometry of a cubic block

        Returns:
            - Dict[str, Any]: dictionary containing the geometry data as generated by the gmsh_io
        """
        expected_points = {
            3: [0.5, 1., 0.],
            4: [0., 1., 0.],
            11: [0., 2., 0.],
            12: [0.5, 2., 0.],
            18: [0.5, 1., -0.5],
            22: [0., 1., -0.5],
            23: [0., 2., -0.5],
            32: [0.5, 2., -0.5]
        }
        expected_lines = {
            7: [3, 4],
            13: [4, 11],
            14: [11, 12],
            15: [12, 3],
            21: [18, 22],
            29: [3, 18],
            33: [4, 22],
            41: [23, 22],
            43: [18, 32],
            44: [32, 23],
            46: [11, 23],
            55: [12, 32]
        }
        expected_surfaces = {
            17: [-13, -7, -15, -14],
            34: [7, 33, -21, -29],
            48: [-13, 33, -41, -46],
            56: [-15, 55, -43, -29],
            60: [-14, 46, -44, -55],
            61: [41, -21, 43, 44]
        }
        expected_volumes = {2: [-17, 61, -48, -34, -56, -60]}

        return {
            "points": expected_points,
            "lines": expected_lines,
            "surfaces": expected_surfaces,
            "volumes": expected_volumes
        }

    def test_create_0d_geometry_from_gmsh_group(self, expected_geo_data_0D: Dict[str, Any]):
        """
        Test the creation of a 0D geometry from a gmsh group.

        Args:
            - expected_geo_data_0D (Dict[str, Any]): expected geometry data for a 0D geometry group.

        """
        # Read the gmsh geo file
        gmsh_io = GmshIO()
        gmsh_io.read_gmsh_geo(r"tests/test_data/gmsh_utils_column_2D.geo")
        geo_data = gmsh_io.geo_data

        # Create the geometry from the gmsh group
        geometry = Geometry.create_geometry_from_gmsh_group(geo_data, "point_group")

        # Assert that the geometry is created correctly
        assert len(geometry.points) == len(expected_geo_data_0D["points"])
        for point_id, point in geometry.points.items():
            assert point_id == point.id
            npt.assert_allclose(point.coordinates, expected_geo_data_0D["points"][point.id])

    def test_create_1d_geometry_from_gmsh_group(self, expected_geo_data_1D: Dict[str, Any]):
        """
        Test the creation of a 1D geometry from a gmsh group.

        Args:
            - expected_geo_data_1D (Dict[int, Any]): expected geometry data for a 1D geometry group.

        """

        # Read the gmsh geo file
        gmsh_io = GmshIO()
        gmsh_io.read_gmsh_geo(r"tests/test_data/gmsh_utils_column_2D.geo")
        geo_data = gmsh_io.geo_data

        # Create the geometry from the gmsh group
        geometry = Geometry.create_geometry_from_gmsh_group(geo_data, "line_group")

        # Assert that the geometry is created correctly
        assert len(geometry.points) == len(expected_geo_data_1D["points"])
        for point_id, point in geometry.points.items():
            assert point_id == point.id
            npt.assert_allclose(point.coordinates, expected_geo_data_1D["points"][point.id])

        assert len(geometry.lines) == len(expected_geo_data_1D["lines"])
        for line_id, line in geometry.lines.items():
            assert line_id == line.id
            npt.assert_equal(line.point_ids, expected_geo_data_1D["lines"][line.id])

    def test_create_2d_geometry_from_gmsh_group(self, expected_geo_data_2D: Dict[str, Any]):
        """
        Test the creation of a 2D geometry from a gmsh group.

        Args:
            - expected_geo_data_2D (Dict[int, Any]): expected geometry data for a 2D geometry group.

        """

        # Read the gmsh geo file
        gmsh_io = GmshIO()
        gmsh_io.read_gmsh_geo(r"tests/test_data/gmsh_utils_column_2D.geo")
        geo_data = gmsh_io.geo_data

        # Create the geometry from the gmsh group
        geometry = Geometry.create_geometry_from_gmsh_group(geo_data, "group_2")

        # Assert that the geometry is created correctly
        assert len(geometry.points) == len(expected_geo_data_2D["points"])
        for point_id, point in geometry.points.items():
            assert point_id == point.id
            npt.assert_allclose(point.coordinates, expected_geo_data_2D["points"][point.id])

        assert len(geometry.lines) == len(expected_geo_data_2D["lines"])
        for line_id, line in geometry.lines.items():
            assert line_id == line.id
            npt.assert_equal(line.point_ids, expected_geo_data_2D["lines"][line.id])

        assert len(geometry.surfaces) == len(expected_geo_data_2D["surfaces"])
        for surface_id, surface in geometry.surfaces.items():
            assert surface_id == surface.id
            npt.assert_equal(surface.line_ids, expected_geo_data_2D["surfaces"][surface.id])

    def test_create_3d_geometry_from_gmsh_group(self, expected_geo_data_3D: Dict[str, Any]):
        """
        Test the creation of a 3D geometry from a gmsh group.

        Args:
            - expected_geo_data_3D (Dict[str, Any]): expected geometry data for a 3D geometry group.

        """

        # Read the gmsh geo file
        gmsh_io = GmshIO()
        gmsh_io.read_gmsh_geo(r"tests/test_data/gmsh_utils_column_3D_tetra4.geo")
        geo_data = gmsh_io.geo_data

        # Create the geometry from the gmsh group
        geometry = Geometry.create_geometry_from_gmsh_group(geo_data, "group_2")

        # Assert that the geometry is created correctly
        assert len(geometry.points) == len(expected_geo_data_3D["points"])
        for point_id, point in geometry.points.items():
            assert point_id == point.id
            npt.assert_allclose(point.coordinates, expected_geo_data_3D["points"][point.id])

        assert len(geometry.lines) == len(expected_geo_data_3D["lines"])
        for line_id, line in geometry.lines.items():
            assert line_id == line.id
            npt.assert_equal(line.point_ids, expected_geo_data_3D["lines"][line.id])

        assert len(geometry.surfaces) == len(expected_geo_data_3D["surfaces"])
        for surface_id, surface in geometry.surfaces.items():
            assert surface_id == surface.id
            npt.assert_equal(surface.line_ids, expected_geo_data_3D["surfaces"][surface.id])

        assert len(geometry.volumes) == len(expected_geo_data_3D["volumes"])
        for volume_id, volume in geometry.volumes.items():
            assert volume_id == volume.id
            npt.assert_equal(volume.surface_ids, expected_geo_data_3D["volumes"][volume.id])

    def test_create_geometry_from_geo_data(self, expected_geo_data_3D: Dict[str, Any]):
        """
        Test the creation of a 3D geometry from a geo_data dictionary.

        Args:
            - expected_geo_data_3D (Dict[str, Any]): expected geometry data for a 3D geometry group.

        """

        geo_data = expected_geo_data_3D

        # Create the geometry from the gmsh group
        geometry = Geometry.create_geometry_from_geo_data(geo_data)

        # Assert that the geometry is created correctly
        assert len(geometry.points) == len(expected_geo_data_3D["points"])
        for point_id, point in geometry.points.items():
            assert point_id == point.id
            npt.assert_allclose(point.coordinates, expected_geo_data_3D["points"][point.id])

        assert len(geometry.lines) == len(expected_geo_data_3D["lines"])
        for line_id, line in geometry.lines.items():
            assert line_id == line.id
            npt.assert_equal(line.point_ids, expected_geo_data_3D["lines"][line.id])

        assert len(geometry.surfaces) == len(expected_geo_data_3D["surfaces"])
        for surface_id, surface in geometry.surfaces.items():
            assert surface_id == surface.id
            npt.assert_equal(surface.line_ids, expected_geo_data_3D["surfaces"][surface.id])

        assert len(geometry.volumes) == len(expected_geo_data_3D["volumes"])
        for volume_id, volume in geometry.volumes.items():
            assert volume_id == volume.id
            npt.assert_equal(volume.surface_ids, expected_geo_data_3D["volumes"][volume.id])

    def test_get_ordered_points_from_surface(self):
        """
        Test the function that returns the ordered points of a surface.

        """

        # Define the coordinates of the surface
        points = {
            1: Point.create([0, 5, 0], 1),
            2: Point.create([2, 5, 0], 2),
            3: Point.create([1, 7, -2], 3),
            4: Point.create([2, 6, -1], 4),
            5: Point.create([0, 6, -1], 5)
        }
        lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 4], 2),
            3: Line.create([4, 3], 3),
            4: Line.create([3, 5], 4),
            5: Line.create([5, 1], 5)
        }
        surfaces = {1: Surface.create([1, 2, 3, 4, 5], 1)}

        geometry = Geometry(points, lines, surfaces)
        ordered_points = geometry.get_ordered_points_from_surface(1)

        # Assert that the points are ordered correctly
        assert ordered_points == [points[1], points[2], points[4], points[3], points[5]]

    def test_length_line(self):
        """
        Test the calculation of the length of a line.

        """

        # Define the coordinates of the line
        points = {1: Point.create([0, 2, 2], 1), 2: Point.create([3, 5, -6], 2)}
        lines = {1: Line.create([2, 1], 1)}

        geometry = Geometry(points, lines)

        # calculate the length of the line
        calculated_length = geometry.calculate_length_line(1)

        # Assert that the length is calculated correctly
        assert calculated_length == pytest.approx(np.sqrt(3**2 + 3**2 + 8**2))

    def test_calculate_centroid_of_line(self):
        """
        Test the calculation of the centroid of a line.

        """

        # Define the coordinates of the line
        points = {1: Point.create([0, 2, 2], 1), 2: Point.create([3, 5, -6], 2)}
        lines = {1: Line.create([2, 1], 1)}

        geometry = Geometry(points, lines)
        centroid = geometry.calculate_centroid_of_line(1)

        # Assert that the centroid is calculated correctly
        npt.assert_array_almost_equal(centroid, np.array([1.5, 3.5, -2]))

    def test_calculate_centroid_of_surface(self):
        """
        Test the calculation of the centroid of a surface.

        """

        # Define the coordinates of the surface

        points = {
            1: Point.create([0, 5, 0], 1),
            2: Point.create([2, 5, 0], 2),
            3: Point.create([2, 6, -1], 3),
            4: Point.create([1, 7, -2], 4),
            5: Point.create([0, 6, -1], 5)
        }
        lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 5], 4),
            5: Line.create([5, 1], 5)
        }
        surfaces = {1: Surface.create([1, 2, 3, 4, 5], 1)}

        geometry = Geometry(points, lines, surfaces)
        centroid = geometry.calculate_centroid_of_surface(1)

        # Assert that the centroid is calculated correctly
        npt.assert_array_almost_equal(centroid, np.array([1, 5.8, -0.8]))

    def test_calculate_centre_of_mass_surface(self):
        """
        Test the calculation of the centre of mass of a surface.

        """

        # Define the coordinates of the surface

        points = {
            1: Point.create([0, 5, 0], 1),
            2: Point.create([2, 5, 0], 2),
            3: Point.create([2, 6, -1], 3),
            4: Point.create([1.5, 6, -1], 4),
            5: Point.create([0.5, 6, -1], 5),
            6: Point.create([0, 6, -1], 6)
        }
        lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 5], 4),
            5: Line.create([5, 6], 5),
            6: Line.create([6, 1], 6)
        }
        surfaces = {1: Surface.create([1, 2, 3, 4, 5, 6], 1)}

        geometry = Geometry(points, lines, surfaces)
        centre_of_mass = geometry.calculate_centre_of_mass_surface(1)

        # Assert that the centre of mass is calculated correctly
        npt.assert_array_almost_equal(centre_of_mass, np.array([1, 5.5, -0.5]))

    def test_calculate_area_convex_surface(self):
        """
        Test the calculation of the area of a convex surface.

        """

        # Define the coordinates of the surface

        points = {
            1: Point.create([0, 5, 0], 1),
            2: Point.create([2, 5, 0], 2),
            3: Point.create([2, 6, -1], 3),
            4: Point.create([1, 7, -2], 4),
            5: Point.create([0, 6, -1], 5)
        }
        lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 5], 4),
            5: Line.create([5, 1], 5)
        }
        surfaces = {1: Surface.create([1, 2, 3, 4, 5], 1)}

        geometry = Geometry(points, lines, surfaces)
        calculated_area = geometry.calculate_area_surface(1)

        # Assert that the area is calculated correctly
        assert pytest.approx(calculated_area) == np.sqrt(2) * 2 * 1.5

    def test_calculate_area_non_convex_surface(self):
        """
        Test the calculation of the area of a concave surface.

        """

        # Define the coordinates of the surface

        points = {
            1: Point.create([0, 5, 0], 1),
            2: Point.create([2, 5, 0], 2),
            3: Point.create([2, 6, -1], 3),
            4: Point.create([1, 5, 0], 4),
            5: Point.create([0, 6, -1], 5)
        }
        lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 5], 4),
            5: Line.create([5, 1], 5)
        }
        surfaces = {1: Surface.create([1, 2, 3, 4, 5], 1)}

        geometry = Geometry(points, lines, surfaces)
        calculated_area = geometry.calculate_area_surface(1)

        # Assert that the area is calculated correctly
        assert pytest.approx(calculated_area) == np.sqrt(2)

    def test_calculate_area_surface_with_collinear_points(self):
        """
        Test the calculation of the area of a surface, where the first 3 points are collinear.

        """

        # Define the coordinates of the surface

        points = {
            1: Point.create([0, 5, 0], 1),
            2: Point.create([1, 5, 0], 2),
            3: Point.create([2, 5, 0], 3),
            4: Point.create([2, 6, -1], 4),
            5: Point.create([0, 6, -1], 5)
        }
        lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 5], 4),
            5: Line.create([5, 1], 5)
        }
        surfaces = {1: Surface.create([1, 2, 3, 4, 5], 1)}

        geometry = Geometry(points, lines, surfaces)
        calculated_area = geometry.calculate_area_surface(1)

        # Assert that the area is calculated correctly
        assert pytest.approx(calculated_area) == 2 * np.sqrt(2)

    def test_calculate_area_surface_with_three_points(self):
        """
        Test the calculation of the area of a minimum surface.

        """

        # Define the coordinates of the surface
        points = {1: Point.create([0, 5, 0], 1), 2: Point.create([1, 5, 0], 2), 3: Point.create([0.5, 6, 1], 3)}
        lines = {1: Line.create([1, 2], 1), 2: Line.create([2, 3], 2), 3: Line.create([3, 1], 3)}
        surfaces = {1: Surface.create([1, 2, 3], 1)}

        geometry = Geometry(points, lines, surfaces)

        # calculate the area of the surface
        calculated_area = geometry.calculate_area_surface(1)

        # Assert that the area is calculated correctly
        assert pytest.approx(calculated_area) == np.sqrt(2) / 2

    def test_centre_of_mass_volume(self):
        """
        Test the calculation of the centre of mass of a cube with extra points.

        """
        # define a cube volume with extra points
        points = {
            1: Point.create([0, 0, 0], 1),
            2: Point.create([0, 1, 0], 2),
            3: Point.create([1, 1, 0], 3),
            4: Point.create([1, 0, 0], 4),
            5: Point.create([0, 0, 1], 5),
            6: Point.create([0, 1, 1], 6),
            7: Point.create([1, 1, 1], 7),
            8: Point.create([1, 0, 1], 8),
            9: Point.create([0.5, 0, 0], 9),
            10: Point.create([0.5, 0, 1], 10)
        }
        lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            # 4: Line.create([4, 1], 4), # -> 16,17
            5: Line.create([1, 5], 5),
            6: Line.create([2, 6], 6),
            7: Line.create([3, 7], 7),
            8: Line.create([4, 8], 8),
            9: Line.create([5, 6], 9),
            10: Line.create([6, 7], 10),
            11: Line.create([7, 8], 11),
            # 12: Line.create([8, 5], 12), # -> 14,15
            13: Line.create([9, 10], 13),
            14: Line.create([8, 10], 14),
            15: Line.create([10, 5], 15),
            16: Line.create([4, 9], 16),
            17: Line.create([9, 1], 17),
        }
        surfaces = {
            1: Surface.create([1, 2, 3, 16, 17], 1),
            2: Surface.create([5, 9, 6, 1], 2),
            3: Surface.create([6, 10, 7, 2], 3),
            4: Surface.create([7, 11, 8, 3], 4),
            5: Surface.create([8, 14, 13, 16], 5),
            6: Surface.create([5, 15, 13, 17], 6),
            7: Surface.create([14, 15, 9, 10, 11], 7)
        }
        volumes = {1: Volume.create([1, 2, 3, 4, 5, 6, 7], 1)}

        geometry = Geometry(points, lines, surfaces, volumes)

        # calculate the centre of mass of the volume
        centre_of_mass = geometry.calculate_centre_of_mass_volume(1)

        # Assert that the centre of mass is calculated correctly
        npt.assert_array_almost_equal(centre_of_mass, np.array([0.5, 0.5, 0.5]))
