import pickle
from copy import deepcopy
from typing import Tuple
import re
import sys
from pathlib import Path

import numpy.testing as npt
import pytest

from stem.geometry import *
from stem.model import *
from stem.output import NodalOutput, GiDOutputParameters, JsonOutputParameters
from stem.solver import *
from stem.boundary import RotationConstraint, DisplacementConstraint
from tests.utils import TestUtils

IS_LINUX = sys.platform == "linux"


class TestModel:

    @pytest.fixture
    def expected_geo_data_0D(self):
        """
        Expected geometry data for a 0D geometry group. The group is a geometry of a point

        Returns:
            - Dict[str, Any]: dictionary containing the geometry data as generated by the gmsh_io
        """
        expected_points = {1: [0, 0, 0], 2: [0.5, 0, 0]}
        return {"points": expected_points}

    @pytest.fixture
    def expected_geometry_single_layer_2D(self):
        """
        Sets expected geometry data for a 2D geometry group. The group is a geometry of a square.

        Returns:
            - :class:`stem.geometry.Geometry`: geometry of a 2D square
        """

        geometry = Geometry()

        geometry.points = {
            1: Point.create([0, 0, 0], 1),
            2: Point.create([1, 0, 0], 2),
            3: Point.create([1, 1, 0], 3),
            4: Point.create([0, 1, 0], 4)
        }

        geometry.lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 1], 4)
        }

        geometry.surfaces = {1: Surface.create([1, 2, 3, 4], 1)}

        geometry.volumes = {}

        return geometry

    @pytest.fixture
    def expected_geometry_single_layer_3D(self):
        """
        Sets expected geometry data for a 2D geometry group. The group is a geometry of a square.

        Returns:
            - :class:`stem.geometry.Geometry`: geometry of a 2D square
        """

        geometry = Geometry()

        geometry.points = {
            1: Point.create([0, 0, 0], 1),
            5: Point.create([0, 0, 1], 5),
            6: Point.create([1, 0, 1], 6),
            2: Point.create([1, 0, 0], 2),
            7: Point.create([1, 1, 1], 7),
            3: Point.create([1, 1, 0], 3),
            8: Point.create([0, 1, 1], 8),
            4: Point.create([0, 1, 0], 4)
        }

        geometry.lines = {
            5: Line.create([1, 5], 5),
            7: Line.create([5, 6], 7),
            6: Line.create([2, 6], 6),
            1: Line.create([1, 2], 1),
            9: Line.create([6, 7], 9),
            8: Line.create([3, 7], 8),
            2: Line.create([2, 3], 2),
            11: Line.create([7, 8], 11),
            10: Line.create([4, 8], 10),
            3: Line.create([3, 4], 3),
            12: Line.create([8, 5], 12),
            4: Line.create([4, 1], 4)
        }

        geometry.surfaces = {
            2: Surface.create([5, 7, -6, -1], 2),
            3: Surface.create([6, 9, -8, -2], 3),
            4: Surface.create([8, 11, -10, -3], 4),
            5: Surface.create([10, 12, -5, -4], 5),
            1: Surface.create([1, 2, 3, 4], 1),
            6: Surface.create([7, 9, 11, 12], 6)
        }

        # The volumes list converted to a dictionary
        geometry.volumes = {1: Volume.create([-2, -3, -4, -5, -1, 6], 1)}

        return geometry

    @pytest.fixture
    def expected_geometry_two_layers_2D(self):
        """
        Sets expected geometries for 2 attached 2D squares.

        Returns:
            - Tuple[:class:`stem.geometry.Geometry`,:class:`stem.geometry.Geometry`, :class:`stem.geometry.Geometry` ]:\
                geometries of 2 attached 2D squares

        """

        # geometry_1
        geometry_1 = Geometry()
        geometry_1.points = {
            1: Point.create([0, 0, 0], 1),
            2: Point.create([1, 0, 0], 2),
            3: Point.create([1, 1, 0], 3),
            4: Point.create([0, 1, 0], 4)
        }

        geometry_1.lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 1], 4)
        }

        geometry_1.surfaces = {1: Surface.create([1, 2, 3, 4], 1)}

        geometry_1.volumes = {}

        # geometry_2
        geometry_2 = Geometry()

        geometry_2.points = {
            5: Point.create([1, 2, 0], 5),
            6: Point.create([0, 2, 0], 6),
            4: Point.create([0, 1, 0], 4),
            3: Point.create([1, 1, 0], 3)
        }

        geometry_2.lines = {
            5: Line.create([5, 6], 5),
            6: Line.create([6, 4], 6),
            3: Line.create([3, 4], 3),
            7: Line.create([3, 5], 7)
        }

        geometry_2.surfaces = {2: Surface.create([5, 6, -3, 7], 2)}

        geometry_2.volumes = {}

        full_geometry = Geometry()
        full_geometry.points = {
            1: Point.create([0, 0, 0], 1),
            2: Point.create([1, 0, 0], 2),
            3: Point.create([1, 1, 0], 3),
            4: Point.create([0, 1, 0], 4),
            5: Point.create([1, 2, 0], 5),
            6: Point.create([0, 2, 0], 6)
        }

        full_geometry.lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 1], 4),
            5: Line.create([5, 6], 5),
            6: Line.create([6, 4], 6),
            7: Line.create([3, 5], 7)
        }

        full_geometry.surfaces = {1: Surface.create([1, 2, 3, 4], 1), 2: Surface.create([5, 6, -3, 7], 2)}

        full_geometry.volumes = {}

        return geometry_1, geometry_2, full_geometry

    @pytest.fixture
    def expected_geometry_two_layers_2D_after_sync(self):
        """
        Sets expected geometry of two model parts and the whole model after synchronising the geometry.

        Returns:
            - Tuple[:class:`stem.geometry.Geometry`,:class:`stem.geometry.Geometry`, \
                :class:`stem.geometry.Geometry`]: geometries of 2 attached 2D squares and the whole model
        """

        # create expected geometry layer 1
        geometry_1 = Geometry()

        geometry_1.points = {
            8: Point.create([0, 0, 0], 8),
            9: Point.create([1, 0, 0], 9),
            3: Point.create([1, 1, 0], 3),
            7: Point.create([0.5, 1, 0], 7),
            10: Point.create([0, 1, 0], 10)
        }

        geometry_1.lines = {
            9: Line.create([8, 9], 9),
            10: Line.create([9, 3], 10),
            7: Line.create([7, 3], 7),
            11: Line.create([7, 10], 11),
            12: Line.create([10, 8], 12)
        }

        geometry_1.surfaces = {1: Surface.create([9, 10, -7, 11, 12], 1)}

        geometry_2 = Geometry()
        geometry_2.points = {
            5: Point.create([1.0, 2.0, 0.0], 5),
            6: Point.create([0.5, 2.0, 0.0], 6),
            7: Point.create([0.5, 1, 0], 7),
            3: Point.create([1, 1, 0], 3)
        }

        geometry_2.lines = {
            5: Line.create([5, 6], 5),
            6: Line.create([6, 7], 6),
            7: Line.create([7, 3], 7),
            8: Line.create([3, 5], 8)
        }

        geometry_2.surfaces = {2: Surface.create([5, 6, 7, 8], 2)}

        geometry_2.volumes = {}

        # create expected full geometry
        full_geometry = Geometry()
        full_geometry.points = {
            3: Point.create([1, 1, 0], 3),
            5: Point.create([1, 2, 0], 5),
            6: Point.create([0.5, 2, 0], 6),
            7: Point.create([0.5, 1, 0], 7),
            8: Point.create([0, 0, 0], 8),
            9: Point.create([1, 0, 0], 9),
            10: Point.create([0, 1, 0], 10)
        }

        full_geometry.lines = {
            5: Line.create([5, 6], 5),
            6: Line.create([6, 7], 6),
            7: Line.create([7, 3], 7),
            8: Line.create([3, 5], 8),
            9: Line.create([8, 9], 9),
            10: Line.create([9, 3], 10),
            11: Line.create([7, 10], 11),
            12: Line.create([10, 8], 12)
        }

        full_geometry.surfaces = {1: Surface.create([9, 10, -7, 11, 12], 1), 2: Surface.create([5, 6, 7, 8], 2)}

        full_geometry.volumes = {}

        return geometry_1, geometry_2, full_geometry

    @pytest.fixture
    def expected_geometry_line_load(self):
        """
        Sets expected geometry data for a 1D geometry group. The group is a geometry of a multi-line.

        Returns:
            - :class:`stem.geometry.Geometry`: geometry of a 1D multi-line
        """

        geometry = Geometry()

        geometry.points = {
            1: Point.create([0, 0, 0], 1),
            2: Point.create([3, 0, 0], 2),
            3: Point.create([4, -1, 0], 3),
            4: Point.create([10, -1, 0], 4)
        }

        geometry.lines = {1: Line.create([1, 2], 1), 2: Line.create([2, 3], 2), 3: Line.create([3, 4], 3)}

        geometry.surfaces = {}

        geometry.volumes = {}

        return geometry

    @pytest.fixture
    def create_default_2d_soil_material(self):
        """
        Create a default soil material for a 2D geometry.

        Returns:
            - :class:`stem.soil_material.SoilMaterial`: default soil material

        """
        # define soil material
        ndim = 2
        soil_formulation = OnePhaseSoil(ndim, IS_DRAINED=True, DENSITY_SOLID=2650, POROSITY=0.3)
        constitutive_law = LinearElasticSoil(YOUNG_MODULUS=100e6, POISSON_RATIO=0.3)
        soil_material = SoilMaterial(name="soil",
                                     soil_formulation=soil_formulation,
                                     constitutive_law=constitutive_law,
                                     retention_parameters=SaturatedBelowPhreaticLevelLaw())
        return soil_material

    @pytest.fixture
    def create_default_3d_soil_material(self):
        """
        Create a default soil material for a 3D geometry.

        Returns:
            - :class:`stem.soil_material.SoilMaterial`: default soil material

        """
        # define soil material
        ndim = 3
        soil_formulation = OnePhaseSoil(ndim, IS_DRAINED=True, DENSITY_SOLID=2650, POROSITY=0.3)
        constitutive_law = LinearElasticSoil(YOUNG_MODULUS=100e6, POISSON_RATIO=0.3)
        soil_material = SoilMaterial(name="soil",
                                     soil_formulation=soil_formulation,
                                     constitutive_law=constitutive_law,
                                     retention_parameters=SaturatedBelowPhreaticLevelLaw())
        return soil_material

    @pytest.fixture
    def create_default_3d_beam(self):
        """
        Create a default beam material for a 3D geometry.
        """
        # Specify beam material model
        YOUNG_MODULUS = 210000000000
        POISSON_RATIO = 0.30000
        DENSITY = 7850
        CROSS_AREA = 0.01
        I22 = 0.0001
        I33 = 0.0001

        TORTIONAL_INERTIA = I22 + I33
        beam_material = EulerBeam(3, YOUNG_MODULUS, POISSON_RATIO, DENSITY, CROSS_AREA, I33, I22, TORTIONAL_INERTIA)
        name = "beam"
        return StructuralMaterial(name, beam_material)

    @pytest.fixture
    def create_default_point_load_parameters(self):
        """
        Create a default point load parameters.

        Returns:
            - :class:`stem.load.PointLoad`: default point load

        """
        # define soil material
        return PointLoad(active=[False, True, False], value=[0, -200, 0])

    @pytest.fixture
    def create_default_line_load_parameters(self):
        """
        Create a default line load parameters.

        Returns:
            - :class:`stem.load.PointLoad`: default point load

        """
        # define soil material
        return LineLoad(active=[False, True, False], value=[0, -20, 0])

    @pytest.fixture
    def create_default_surface_load_parameters(self):
        """
        Create a default surface load properties.

        Returns:
            - :class:`stem.load.SurfaceLoad`: default surface load

        """
        # define soil material
        return SurfaceLoad(active=[False, True, False], value=[0, -2, 0])

    @pytest.fixture
    def create_default_moving_load_parameters(self):
        """
        Create a default surface load properties.

        Returns:
            - :class:`stem.load.SurfaceLoad`: default surface load

        """
        # define soil material
        return MovingLoad(origin=[3.5, -0.5, 0.0],
                          load=[0.0, -10.0, 0.0],
                          velocity=5.0,
                          offset=3.0,
                          direction=[1, 1, 1])

    @pytest.fixture
    def create_default_outputs(self):
        """
        Sets default output parameters.

        Returns:
            - List[:class:`stem.output.Output`]: list of default output processes.
        """
        # Nodal results
        nodal_results = [NodalOutput.ACCELERATION]
        # Gauss point results
        # define output process

        output_process = Output(part_name="nodal_accelerations",
                                output_name="gid_nodal_accelerations_top",
                                output_dir="dir_test",
                                output_parameters=GiDOutputParameters(file_format="binary",
                                                                      output_interval=100,
                                                                      nodal_results=nodal_results))

        return output_process

    @pytest.fixture
    def expected_geometry_two_layers_3D_extruded(self) -> Tuple[Geometry, Geometry]:
        """
        Expected geometry data for a 3D geometry create from 2D extrusion. The geometry is 2 stacked blocks, where the
        top and bottom blocks are in different groups.

        Returns:
            - Tuple[:class:`stem.geometry.Geometry`,:class:`stem.geometry.Geometry`]: expected geometry data
        """

        geometry_1 = Geometry()
        geometry_1.points = {
            1: Point.create([0, 0, 0], 1),
            2: Point.create([0, 0, 1], 2),
            4: Point.create([1, 0, 1], 4),
            3: Point.create([1, 0, 0], 3),
            6: Point.create([1, 1, 1], 6),
            5: Point.create([1, 1, 0], 5),
            8: Point.create([0, 1, 1], 8),
            7: Point.create([0, 1, 0], 7)
        }

        geometry_1.lines = {
            1: Line.create([1, 2], 1),
            4: Line.create([2, 4], 4),
            2: Line.create([3, 4], 2),
            3: Line.create([1, 3], 3),
            7: Line.create([4, 6], 7),
            5: Line.create([5, 6], 5),
            6: Line.create([3, 5], 6),
            10: Line.create([6, 8], 10),
            8: Line.create([7, 8], 8),
            9: Line.create([5, 7], 9),
            12: Line.create([8, 2], 12),
            11: Line.create([7, 1], 11)
        }

        geometry_1.surfaces = {
            1: Surface.create([1, 4, -2, -3], 1),
            2: Surface.create([2, 7, -5, -6], 2),
            3: Surface.create([5, 10, -8, -9], 3),
            4: Surface.create([8, 12, -1, -11], 4),
            5: Surface.create([3, 6, 9, 11], 5),
            6: Surface.create([4, 7, 10, 12], 6)
        }

        geometry_1.volumes = {1: Volume.create([-1, -2, -3, -4, -5, 6], 1)}

        geometry_2 = Geometry()

        geometry_2.points = {
            9: Point.create([1.0, 2.0, 0.0], 9),
            10: Point.create([1., 2., 1.], 10),
            12: Point.create([0.0, 2., 1.], 12),
            11: Point.create([0, 2., 0.], 11),
            8: Point.create([0., 1., 1], 8),
            7: Point.create([0., 1., 0], 7),
            5: Point.create([1, 1., 0], 5),
            6: Point.create([1, 1., 1], 6)
        }

        geometry_2.lines = {
            13: Line.create([9, 10], 13),
            16: Line.create([10, 12], 16),
            14: Line.create([11, 12], 14),
            15: Line.create([9, 11], 15),
            18: Line.create([12, 8], 18),
            8: Line.create([7, 8], 8),
            17: Line.create([11, 7], 17),
            5: Line.create([5, 6], 5),
            10: Line.create([6, 8], 10),
            9: Line.create([5, 7], 9),
            20: Line.create([6, 10], 20),
            19: Line.create([5, 9], 19)
        }

        geometry_2.surfaces = {
            7: Surface.create([13, 16, -14, -15], 7),
            8: Surface.create([14, 18, -8, -17], 8),
            3: Surface.create([5, 10, -8, -9], 3),
            9: Surface.create([5, 20, -13, -19], 9),
            10: Surface.create([15, 17, -9, 19], 10),
            11: Surface.create([16, 18, -10, 20], 11)
        }

        geometry_2.volumes = {2: Volume.create([-7, -8, 3, -9, -10, 11], 2)}

        return geometry_1, geometry_2

    @pytest.fixture
    def expected_geometry_two_layers_3D_geo_file(self):
        """
        Expected geometry data for a 3D geometry create in a geo file. The geometry is 2 stacked blocks, where the top
        and bottom blocks are in different groups.

        Returns:
            - Tuple[:class:`stem.geometry.Geometry`,:class:`stem.geometry.Geometry`]: expected geometry data
        """

        geometry_1 = Geometry()
        geometry_1.volumes = {1: Volume.create([-10, 39, 26, 30, 34, 38], 1)}

        geometry_1.surfaces = {
            10: Surface.create([5, 6, 7, 8], 10),
            39: Surface.create([19, 20, 21, 22], 39),
            26: Surface.create([5, 25, -19, -24], 26),
            30: Surface.create([6, 29, -20, -25], 30),
            34: Surface.create([7, 33, -21, -29], 34),
            38: Surface.create([8, 24, -22, -33], 38)
        }

        geometry_1.lines = {
            5: Line.create([1, 2], 5),
            6: Line.create([2, 3], 6),
            7: Line.create([3, 4], 7),
            8: Line.create([4, 1], 8),
            19: Line.create([13, 14], 19),
            20: Line.create([14, 18], 20),
            21: Line.create([18, 22], 21),
            22: Line.create([22, 13], 22),
            25: Line.create([2, 14], 25),
            24: Line.create([1, 13], 24),
            29: Line.create([3, 18], 29),
            33: Line.create([4, 22], 33)
        }

        geometry_1.points = {
            1: Point.create([0., 0., 0.], 1),
            2: Point.create([0.5, 0., 0.], 2),
            3: Point.create([0.5, 1., 0.], 3),
            4: Point.create([0., 1., 0.], 4),
            13: Point.create([0., 0., -0.5], 13),
            14: Point.create([0.5, 0., -0.5], 14),
            18: Point.create([0.5, 1., -0.5], 18),
            22: Point.create([0., 1., -0.5], 22)
        }

        geometry_2 = Geometry()
        geometry_2.volumes = {2: Volume.create([-17, 61, -48, -34, -56, -60], 2)}

        geometry_2.surfaces = {
            17: Surface.create([-13, -7, -15, -14], 17),
            61: Surface.create([41, -21, 43, 44], 61),
            48: Surface.create([-13, 33, -41, -46], 48),
            34: Surface.create([7, 33, -21, -29], 34),
            56: Surface.create([-15, 55, -43, -29], 56),
            60: Surface.create([-14, 46, -44, -55], 60)
        }

        geometry_2.lines = {
            13: Line.create([4, 11], 13),
            7: Line.create([3, 4], 7),
            15: Line.create([12, 3], 15),
            14: Line.create([11, 12], 14),
            41: Line.create([23, 22], 41),
            21: Line.create([18, 22], 21),
            43: Line.create([18, 32], 43),
            44: Line.create([32, 23], 44),
            33: Line.create([4, 22], 33),
            46: Line.create([11, 23], 46),
            29: Line.create([3, 18], 29),
            55: Line.create([12, 32], 55)
        }

        geometry_2.points = {
            4: Point.create([0., 1., 0.], 4),
            11: Point.create([0., 2., 0.], 11),
            3: Point.create([0.5, 1., 0.], 3),
            12: Point.create([0.5, 2., 0.], 12),
            23: Point.create([0., 2., -0.5], 23),
            22: Point.create([0., 1., -0.5], 22),
            18: Point.create([0.5, 1., -0.5], 18),
            32: Point.create([0.5, 2., -0.5], 32)
        }

        return geometry_1, geometry_2

    @pytest.fixture(autouse=True)
    def close_gmsh(self):
        """
        Initializer to close gmsh if it was not closed before. In case a test fails, the destroyer method is not called
        on the Model object and gmsh keeps on running. Therefore, nodes, lines, surfaces and volumes ids are not
        reset to one. This causes also the next test after the failed one to fail as well, which has nothing to do
        the test itself.

        Returns:
            - None

        """
        gmsh_IO.GmshIO().finalize_gmsh()

    def test_add_single_soil_layer_2D(self, expected_geometry_single_layer_2D: Geometry,
                                      create_default_2d_soil_material: SoilMaterial):
        """
        Test if a single soil layer is added correctly to the model in a 2D space. A single soil layer is generated
        and a single soil material is created and added to the model.

        Args:
            - expected_geometry_single_layer_2D (:class:`stem.geometry.Geometry`): expected geometry of the model
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 2

        layer_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]

        # define soil material
        soil_material = create_default_2d_soil_material

        # create model
        model = Model(ndim)

        # add soil layer
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")

        # check if layer is added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "soil1"
        assert model.body_model_parts[0].material == soil_material

        # check if geometry is added correctly
        generated_geometry = model.body_model_parts[0].geometry
        expected_geometry = expected_geometry_single_layer_2D

        # check if points are added correctly
        TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_add_single_soil_layer_3D(self, expected_geometry_single_layer_3D: Geometry,
                                      create_default_3d_soil_material: SoilMaterial):
        """
        Test if a single soil layer is added correctly to the model in a 3D space. A single soil layer is generated
        and a single soil material is created and added to the model.

        Args:
            - expected_geometry_single_layer_3D (:class:`stem.geometry.Geometry`): expected geometry of the model
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 3

        layer_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]

        # define soil material
        soil_material = create_default_3d_soil_material

        # create model
        model = Model(ndim)
        model.extrusion_length = 1

        model.project_parameters = TestUtils.create_default_solver_settings()

        # add soil layer
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")

        # check if layer is added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "soil1"
        assert model.body_model_parts[0].material == soil_material

        # check if geometry is added correctly
        generated_geometry = model.body_model_parts[0].geometry
        expected_geometry = expected_geometry_single_layer_3D

        # check if points are added correctly
        TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_add_multiple_soil_layers_2D(self, expected_geometry_two_layers_2D: Tuple[Geometry, Geometry, Geometry],
                                         create_default_2d_soil_material: SoilMaterial):
        """
        Test if multiple soil layers are added correctly to the model in a 2D space. Multiple soil layers are generated
        and multiple soil materials are created and added to the model.

        Args:
            - expected_geometry_two_layers_2D (Tuple[:class:`stem.geometry.Geometry`, :class:`stem.geometry.Geometry`, \
              :class:`stem.geometry.Geometry`]): expected geometry of the model
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 2

        layer1_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        layer2_coordinates = [(1, 1, 0), (0, 1, 0), (0, 2, 0), (1, 2, 0)]

        # define soil materials
        soil_material1 = create_default_2d_soil_material
        soil_material1.name = "soil1"

        soil_material2 = create_default_2d_soil_material
        soil_material2.name = "soil2"

        # create model
        model = Model(ndim)

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")
        model.add_soil_layer_by_coordinates(layer2_coordinates, soil_material2, "layer2")

        # check if layers are added correctly
        assert len(model.body_model_parts) == 2
        assert model.body_model_parts[0].name == "layer1"
        assert model.body_model_parts[0].material == soil_material1
        assert model.body_model_parts[1].name == "layer2"
        assert model.body_model_parts[1].material == soil_material2

        # check if geometry is added correctly for each layer
        for i in range(len(model.body_model_parts)):
            generated_geometry = model.body_model_parts[i].geometry
            expected_geometry = expected_geometry_two_layers_2D[i]

            TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_add_multiple_soil_layers_3D(self, expected_geometry_two_layers_3D_extruded: Tuple[Geometry, Geometry],
                                         create_default_3d_soil_material: SoilMaterial):
        """
        Test if multiple soil layers are added correctly to the model in a 3D space. Multiple soil layers are generated
        and multiple soil materials are created and added to the model.

        Args:
            - expected_geometry_two_layers_3D_extruded (Tuple[:class:`stem.geometry.Geometry`, \
                :class:`stem.geometry.Geometry`]): expected geometry of the model which is created by extruding \
                a 2D geometry
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 3

        layer1_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        layer2_coordinates = [(1, 1, 0), (0, 1, 0), (0, 2, 0), (1, 2, 0)]

        # define soil materials
        soil_material1 = create_default_3d_soil_material
        soil_material1.name = "soil1"

        soil_material2 = create_default_3d_soil_material
        soil_material2.name = "soil2"

        # create model
        model = Model(ndim)
        model.extrusion_length = 1

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")
        model.add_soil_layer_by_coordinates(layer2_coordinates, soil_material2, "layer2")

        model.synchronise_geometry()

        # check if layers are added correctly
        assert len(model.body_model_parts) == 2
        assert model.body_model_parts[0].name == "layer1"
        assert model.body_model_parts[0].material == soil_material1
        assert model.body_model_parts[1].name == "layer2"
        assert model.body_model_parts[1].material == soil_material2

        # check if geometry is added correctly for each layer
        for i in range(len(model.body_model_parts)):
            generated_geometry = model.body_model_parts[i].geometry
            expected_geometry = expected_geometry_two_layers_3D_extruded[i]

            TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_validation_of_adding_soil_layers(self, create_default_3d_soil_material: SoilMaterial):
        """
        Tests that errors are raised when groups are not specified or added multiple times.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 3

        shape1 = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]

        # define soil materials
        soil_material1 = create_default_3d_soil_material
        soil_material1.name = "soil1"

        # create model
        model = Model(ndim)
        # add a valid group
        model.add_group_for_extrusion(group_name="Group1", reference_depth=0, extrusion_length=1)

        # expect it raises an error when adding a layer to a non-existing section
        with pytest.raises(
                ValueError,
                match="For 3D models either the extrusion length or the group name for the extrusion must be specified."
        ):
            model.add_soil_layer_by_coordinates(shape1, soil_material1, "layer1")

        # expect it raises an error when adding a layer to a non-existing section
        with pytest.raises(ValueError, match="Non-existent group specified `Group2`."):
            model.add_soil_layer_by_coordinates(shape1, soil_material1, "layer1", group_name="Group2")

        # add a soil layer that doesn't contain the reference point of the group
        shape2 = [(0, 0, 1), (1, 0, 1), (1, 1, 1), (0, 1, 1)]

        with pytest.raises(
                ValueError,
                match="The reference coordinate of group: Group1, does not lay on the same plane as soil layer: layer2"
        ):
            model.add_soil_layer_by_coordinates(shape2, soil_material1, "layer2", group_name="Group1")

        # add a soil layer which section is not planar
        shape3 = [(0, 0, 0), (1, 0, 0), (1, 1, 2), (0, 1, 3)]

        with pytest.raises(ValueError, match="Polygon for the soil layer are not on the same plane."):
            model.add_soil_layer_by_coordinates(shape3, soil_material1, "layer3", group_name="Group1")

    def test_validation_of_adding_groups(self):
        """
        Tests that errors are raised when groups are not specified or added multiple times.

        """

        ndim = 3

        # create model
        model = Model(ndim)
        # add a valid group
        model.add_group_for_extrusion(group_name="Group1", reference_depth=0, extrusion_length=1)

        # expect it raises an error when adding an already existing section
        with pytest.raises(ValueError, match="The group `Group1` already exists, but group names must be unique."):
            model.add_group_for_extrusion(group_name="Group1", reference_depth=0, extrusion_length=1)

    def test_adding_model_parts_to_groups(self):
        """
        Tests validation of adding model parts to groups.

        """

        ndim = 3

        # create model
        model = Model(ndim)
        # add a valid group
        model.add_group_for_extrusion(group_name="Group1", reference_depth=0, extrusion_length=1)

        # test if raises an error when adding a model part to a non existing group
        with pytest.raises(ValueError, match="The group specified `Group2` does not exist."):
            model.add_model_part_to_group(group_name="Group2", part_name="test_part")

        # test if raises an error when adding a non existing model part to a group
        with pytest.raises(ValueError, match="The model part specified `test_part` does not exist."):
            model.add_model_part_to_group(group_name="Group1", part_name="test_part")

    def test_add_multiple_sections_3D(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if two extruded sections are added correctly to the model in a 3D space. Two triangular sections are
        sequentially extruded and multiple soil materials are created and added to the model.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 3

        shape1 = [(0, 0, 0), (1, 0, 0), (0, 1, 0)]
        shape2 = [(0, 0.5, 1), (0.5, 0.5, 1), (0, 1, 1)]

        # define soil materials
        soil_material1 = create_default_3d_soil_material
        soil_material1.name = "soil1"

        soil_material2 = create_default_3d_soil_material
        soil_material2.name = "soil2"

        # create model
        model = Model(ndim)
        model.add_group_for_extrusion(group_name="Group1", reference_depth=0, extrusion_length=1)
        model.add_group_for_extrusion(group_name="Group2", reference_depth=1, extrusion_length=1)

        # add soil layers
        model.add_soil_layer_by_coordinates(shape1, soil_material1, "layer1", group_name="Group1")
        model.add_soil_layer_by_coordinates(shape2, soil_material2, "layer2", group_name="Group2")

        model.synchronise_geometry()

        # check if layers are added correctly
        assert len(model.body_model_parts) == 2
        assert model.body_model_parts[0].name == "layer1"
        assert model.body_model_parts[0].material == soil_material1
        assert model.body_model_parts[1].name == "layer2"
        assert model.body_model_parts[1].material == soil_material2

        # check if geometry is added correctly for each layer
        geometry_1 = Geometry()
        geometry_1.points = {
            12: Point.create([0.0, 0.0, 0.0], 12),
            13: Point.create([0.0, 0.0, 1.0], 13),
            15: Point.create([1.0, 0.0, 1.0], 15),
            14: Point.create([1.0, 0.0, 0.0], 14),
            8: Point.create([0.5, 0.5, 1.0], 8),
            6: Point.create([0.0, 1.0, 1.0], 6),
            16: Point.create([0.0, 1.0, 0.0], 16),
            7: Point.create([0.0, 0.5, 1.0], 7)
        }

        geometry_1.lines = {
            19: Line.create([12, 13], 19),
            22: Line.create([13, 15], 22),
            20: Line.create([14, 15], 20),
            21: Line.create([12, 14], 21),
            25: Line.create([15, 8], 25),
            11: Line.create([8, 6], 11),
            23: Line.create([16, 6], 23),
            24: Line.create([14, 16], 24),
            12: Line.create([6, 7], 12),
            27: Line.create([7, 13], 27),
            26: Line.create([16, 12], 26),
            10: Line.create([7, 8], 10)
        }

        geometry_1.surfaces = {
            11: Surface.create([19, 22, -20, -21], 11),
            12: Surface.create([20, 25, 11, -23, -24], 12),
            13: Surface.create([23, 12, 27, -19, -26], 13),
            14: Surface.create([21, 24, 26], 14),
            15: Surface.create([27, 22, 25, -10], 15),
            6: Surface.create([10, 11, 12], 6)
        }

        geometry_1.volumes = {1: Volume.create([-11, -12, -13, -14, 15, 6], 1)}

        geometry_2 = Geometry()

        geometry_2.points = {
            7: Point.create([0.0, 0.5, 1.0], 7),
            9: Point.create([0.0, 0.5, 2.0], 9),
            10: Point.create([0.5, 0.5, 2.0], 10),
            8: Point.create([0.5, 0.5, 1.0], 8),
            11: Point.create([0.0, 1.0, 2.0], 11),
            6: Point.create([0.0, 1.0, 1.0], 6)
        }

        geometry_2.lines = {
            13: Line.create([7, 9], 13),
            15: Line.create([9, 10], 15),
            14: Line.create([8, 10], 14),
            10: Line.create([7, 8], 10),
            17: Line.create([10, 11], 17),
            16: Line.create([6, 11], 16),
            11: Line.create([8, 6], 11),
            18: Line.create([11, 9], 18),
            12: Line.create([6, 7], 12)
        }

        geometry_2.surfaces = {
            7: Surface.create([13, 15, -14, -10], 7),
            8: Surface.create([14, 17, -16, -11], 8),
            9: Surface.create([16, 18, -13, -12], 9),
            6: Surface.create([10, 11, 12], 6),
            10: Surface.create([15, 17, 18], 10)
        }

        geometry_2.volumes = {2: Volume.create([-7, -8, -9, -6, 10], 2)}

        expected_geometries = [geometry_1, geometry_2]

        for model_part, expected_geometry in zip(model.body_model_parts, expected_geometries):

            TestUtils.assert_almost_equal_geometries(expected_geometry, model_part.geometry)

    def test_add_all_layers_from_geo_file_2D(self, expected_geometry_two_layers_2D: Tuple[Geometry, Geometry,
                                                                                          Geometry]):
        """
        Tests if all layers are added correctly to the model in a 2D space. A geo file is read and all layers are
        added to the model.

        Args:
            - expected_geometry_two_layers_2D (Tuple[:class:`stem.geometry.Geometry`, :class:`stem.geometry.Geometry`, \
                :class:`stem.geometry.Geometry`]): expected geometry of the model

        """

        geo_file_name = "tests/test_data/gmsh_utils_two_blocks_2D.geo"

        # create model
        model = Model(ndim=2)
        model.add_all_layers_from_geo_file(geo_file_name, ["group_1"])

        # check if body model parts are added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "group_1"

        # check if process model part is added correctly
        assert len(model.process_model_parts) == 1
        assert model.process_model_parts[0].name == "group_2"

        # check if geometry is added correctly for each layer
        for i in range(len(model.body_model_parts)):
            generated_geometry = model.body_model_parts[i].geometry
            expected_geometry = expected_geometry_two_layers_2D[i]

            TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_add_all_layers_from_geo_file_3D(self, expected_geometry_two_layers_3D_geo_file: Tuple[Geometry, Geometry]):
        """
        Tests if all layers are added correctly to the model in a 3D space. A geo file is read and all layers are
        added to the model.

        Args:
            - expected_geometry_two_layers_3D_geo_file (Tuple[:class:`stem.geometry.Geometry`, \
                :class:`stem.geometry.Geometry`]): expected geometry of the model

        """

        geo_file_name = "tests/test_data/gmsh_utils_column_3D_tetra4.geo"

        # create model
        model = Model(ndim=3)
        model.add_all_layers_from_geo_file(geo_file_name, ["group_1"])

        # check if body model parts are added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "group_1"

        # check if process model part is added correctly
        assert len(model.process_model_parts) == 1
        assert model.process_model_parts[0].name == "group_2"

        # check if geometry is added correctly
        all_model_parts = []
        all_model_parts.extend(model.body_model_parts)
        all_model_parts.extend(model.process_model_parts)

        # check if geometry is added correctly for each layer
        for i in range(len(all_model_parts)):
            generated_geometry = all_model_parts[i].geometry
            expected_geometry = expected_geometry_two_layers_3D_geo_file[i]

            TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_synchronise_geometry_2D(self, expected_geometry_two_layers_2D_after_sync: Tuple[Geometry, Geometry,
                                                                                             Geometry],
                                     create_default_2d_soil_material: SoilMaterial):
        """
        Test if the geometry is synchronised correctly in 2D after adding a new layer to the model. Where the new layer
        overlaps with the existing layer, the existing layer is cut and the overlapping part is removed.

        Args:
            - expected_geometry_two_layers_2D_after_sync (Tuple[:class:`stem.geometry.Geometry`, \
                :class:`stem.geometry.Geometry`, :class:`stem.geometry.Geometry`]): The expected geometry after \
                synchronising the geometry.
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # define layer coordinates
        ndim = 2
        layer1_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        layer2_coordinates = [(1, 1, 0), (0.5, 1, 0), (0.5, 2, 0), (1, 2, 0)]

        # define soil materials
        soil_material1 = create_default_2d_soil_material
        soil_material1.name = "soil1"

        soil_material2 = create_default_2d_soil_material
        soil_material2.name = "soil2"

        # create model
        model = Model(ndim)

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")
        model.add_soil_layer_by_coordinates(layer2_coordinates, soil_material2, "layer2")

        # synchronise geometry and recalculates the ids
        model.synchronise_geometry()

        # collect all generated geometries
        generated_geometries = [model.body_model_parts[0].geometry, model.body_model_parts[1].geometry, model.geometry]

        # check if geometry is added correctly for each layer
        for generated_geometry, expected_geometry in zip(generated_geometries,
                                                         expected_geometry_two_layers_2D_after_sync):

            TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_synchronise_geometry_3D(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if the geometry is synchronised correctly in 3D after adding a new layer to the model. Where the new layer
        overlaps with the existing layer, the existing layer is cut and the overlapping part is removed.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # define layer coordinates
        ndim = 3
        layer1_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        layer2_coordinates = [(1, 1, 0), (0.5, 1, 0), (0.5, 2, 0), (1, 2, 0)]

        # define soil materials
        soil_material1 = create_default_3d_soil_material
        soil_material1.name = "soil1"

        soil_material2 = create_default_3d_soil_material
        soil_material2.name = "soil2"

        # create model
        model = Model(ndim)
        model.extrusion_length = 1

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")
        model.add_soil_layer_by_coordinates(layer2_coordinates, soil_material2, "layer2")

        # synchronise geometry and recalculates the ids
        model.synchronise_geometry()

        with open("tests/test_data/expected_geometry_after_sync_3D.pickle", "rb") as f:
            expected_geometry_two_layers_3D_after_sync = pickle.load(f)

        # collect all generated geometries
        generated_geometries = [model.body_model_parts[0].geometry, model.body_model_parts[1].geometry, model.geometry]

        # check if geometry is added correctly for each layer
        for generated_geometry, expected_geometry in zip(generated_geometries,
                                                         expected_geometry_two_layers_3D_after_sync):

            TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_add_point_loads_to_2_points(self, create_default_point_load_parameters: PointLoad):
        """
        Test if a single soil point load is added correctly to the model. Two points are generated
        and a single load is created and added to the model.

        Args:
            - create_default_point_load_properties (:class:`stem.load.PointLoad`): default point load parameters

        """

        ndim = 3

        point_coordinates = [(-0.5, 0, 0), (0.5, 0, 0)]

        # define soil material
        load_parameters = create_default_point_load_parameters

        # create model
        model = Model(ndim)
        # add point load
        model.add_load_by_coordinates(point_coordinates, load_parameters, "point_load_1")

        # check if layer is added correctly
        assert len(model.process_model_parts) == 1
        assert model.process_model_parts[0].name == "point_load_1"
        TestUtils.assert_dictionary_almost_equal(model.process_model_parts[0].parameters.__dict__,
                                                 load_parameters.__dict__)

        # check if geometry is added correctly
        generated_geometry = model.process_model_parts[0].geometry
        expected_geometry = Geometry(points={1: Point.create([-0.5, 0, 0], 1), 2: Point.create([0.5, 0, 0], 2)})

        TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_add_line_load_to_3_edges(self, expected_geometry_line_load: Geometry,
                                      create_default_line_load_parameters: PointLoad):
        """
        Test if a line load is added correctly to the model when applied on 3 edges. 4 points are generated
        and a single soil material is created and added to the model.

        Args:
            - expected_geometry_line_load (:class:`stem.geometry.Geometry`): expected geometry of the model
            - create_default_line_load_parameters (:class:`stem.load.LineLoad`): default line load parameters

        """

        ndim = 3

        point_coordinates = [(0, 0, 0), (3, 0, 0), (4, -1, 0), (10, -1, 0)]

        # define soil material
        load_parameters = create_default_line_load_parameters

        # create model
        model = Model(ndim)
        # add line load
        model.add_load_by_coordinates(point_coordinates, load_parameters, "line_load_1")

        # check if layer is added correctly
        assert len(model.process_model_parts) == 1
        assert model.process_model_parts[0].name == "line_load_1"
        TestUtils.assert_dictionary_almost_equal(model.process_model_parts[0].parameters.__dict__,
                                                 load_parameters.__dict__)
        # check if geometry is added correctly
        generated_geometry = model.process_model_parts[0].geometry
        expected_geometry = expected_geometry_line_load

        TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_add_moving_point_load(self, expected_geometry_line_load: Geometry,
                                   create_default_moving_load_parameters: MovingLoad):
        """
        Test if a single soil point load is added correctly to the model. Two points are generated
        and a single load is created and added to the model.
        Args:
            - expected_geometry_line_load (:class:`stem.geometry.Geometry`): expected geometry of the model
            - create_default_moving_load_parameters (:class:`stem.load.MovingLoad`): default moving load parameters
        """

        ndim = 3

        point_coordinates = [(0, 0, 0), (3, 0, 0), (4, -1, 0), (10, -1, 0)]
        # origin is in (3.5, -0.5, 0) thus in the trajectory

        # define soil material
        load_parameters = create_default_moving_load_parameters

        # create model
        model = Model(ndim)
        # add moving load
        model.add_load_by_coordinates(point_coordinates, load_parameters, "moving_load_1")

        # check if layer is added correctly
        assert len(model.process_model_parts) == 1
        assert model.process_model_parts[0].name == "moving_load_1"
        TestUtils.assert_dictionary_almost_equal(model.process_model_parts[0].parameters.__dict__,
                                                 load_parameters.__dict__)

        # check if geometry is added correctly
        generated_geometry = model.process_model_parts[0].geometry
        expected_geometry = expected_geometry_line_load

        TestUtils.assert_almost_equal_geometries(expected_geometry, generated_geometry)

    def test_validation_moving_load(self, create_default_moving_load_parameters: MovingLoad):
        """
        Test validation of moving load when points is not collinear to the trajectory.

        Args:
            - create_default_moving_load_parameters (:class:`stem.load.MovingLoad`): default moving load parameters

        """

        ndim = 3

        point_coordinates = [(0.0, 0, 0), (1, 0, 0), (2, 0, 0), (4, 0, 0)]
        # origin is in (1.5, 0.5, 0) thus not in the trajectory

        # define soil material
        load_parameters = create_default_moving_load_parameters
        # create model
        model = Model(ndim)

        with pytest.raises(ValueError,
                           match="None of the lines are aligned with the origin of the moving load. Error."):
            model.add_load_by_coordinates(point_coordinates, load_parameters, "moving_load_1")

    def test_generate_mesh_with_only_a_body_model_part_2d(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if the mesh is generated correctly in 2D if there is only one body model part.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(2)

        # add soil material
        soil_material = create_default_2d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")
        model.synchronise_geometry()

        # generate mesh
        model.generate_mesh()

        mesh = model.body_model_parts[0].mesh

        assert mesh.ndim == 2

        unique_element_ids = []
        # check if mesh is generated correctly, i.e. if the number of elements is correct and if the element type is
        # correct and if the element ids are unique and if the number of nodes per element is correct
        assert len(mesh.elements) == 162
        for element_id, element in mesh.elements.items():
            assert element.element_type == "TRIANGLE_3N"
            assert element_id not in unique_element_ids
            assert len(element.node_ids) == 3
            unique_element_ids.append(element.id)

        # check if nodes are generated correctly, i.e. if there are nodes in the mesh and if the node ids are unique
        # and if the number of coordinates per node is correct
        unique_node_ids = []
        assert len(mesh.nodes) == 98
        for node_id, node in mesh.nodes.items():
            assert node_id not in unique_node_ids
            assert len(node.coordinates) == 3
            unique_node_ids.append(node.id)

    def test_add_output_to_non_existing_model_part(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if output nodes are correctly accounted for when meshing a surface.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # define layer coordinates
        ndim = 2
        layer1_coordinates = [(0, 0, 0), (4, 0, 0), (4, 1, 0), (0, 1, 0)]

        # define soil materials
        soil_material1 = create_default_2d_soil_material
        soil_material1.name = "soil1"

        # create model
        model = Model(ndim)

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")

        # synchronise geometry and recalculates the ids
        model.synchronise_geometry()
        # define output object
        # Nodal results
        nodal_results = [NodalOutput.ACCELERATION]
        # add outputs to existing model part
        model.add_output_settings(part_name="layer1",
                                  output_name="gid_nodal_accelerations_top",
                                  output_dir="dir_test",
                                  output_parameters=GiDOutputParameters(file_format="binary",
                                                                        output_interval=100,
                                                                        nodal_results=nodal_results))
        # add output to non-existing model part
        msg = "Model part for which output needs to be requested doesn't exist."
        with pytest.raises(ValueError, match=msg):
            model.add_output_settings(part_name="layer2",
                                      output_name="gid_nodal_accelerations_top",
                                      output_dir="dir_test",
                                      output_parameters=GiDOutputParameters(file_format="binary",
                                                                            output_interval=100,
                                                                            nodal_results=nodal_results))

    def test_add_output_to_a_surface_2d(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if output nodes are correctly accounted for when meshing a surface.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # define layer coordinates
        ndim = 2
        layer1_coordinates = [(0, 0, 0), (4, 0, 0), (4, 1, 0), (0, 1, 0)]

        # define soil materials
        soil_material1 = create_default_2d_soil_material
        soil_material1.name = "soil1"

        # create model
        model = Model(ndim)

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")

        # synchronise geometry and recalculates the ids
        model.synchronise_geometry()
        # Define nodal results
        nodal_results = [NodalOutput.ACCELERATION]
        # Define output coordinates
        output_coordinates = [(1.5, 1, 0), (1.5, 0.5, 0), (2.5, 0.5, 0), (2.5, 0, 0)]

        # add output settings
        model.add_output_settings_by_coordinates(output_coordinates,
                                                 part_name="nodal_accelerations",
                                                 output_name="json_nodal_accelerations_top",
                                                 output_dir="dir_test",
                                                 output_parameters=JsonOutputParameters(output_interval=100,
                                                                                        nodal_results=nodal_results))
        model.synchronise_geometry()
        model.generate_mesh()

        unique_element_ids = []
        unique_node_ids = []

        part = model.body_model_parts[0]
        assert part.mesh.ndim == 2

        # check if mesh is generated correctly, i.e. if the number of elements is correct and if the element type is
        # correct and if the element ids are unique and if the number of nodes per element is correct
        assert len(part.mesh.elements) == 98

        for element_id, element in part.mesh.elements.items():
            assert element.element_type == "TRIANGLE_3N"
            assert element_id not in unique_element_ids
            assert len(element.node_ids) == 3
            unique_element_ids.append(element.id)

        # check if nodes are generated correctly, i.e. if there are nodes in the mesh and if the node ids are unique
        # and if the number of coordinates per node is correct
        assert len(part.mesh.nodes) == 64
        for node_id, node in part.mesh.nodes.items():
            assert node_id not in unique_node_ids
            assert len(node.coordinates) == 3
            unique_node_ids.append(node.id)

        # assert the output parts
        part = model.process_model_parts[0]
        assert part.mesh.ndim == 1

        unique_node_ids = []
        # check if nodes are generated correctly, number of nodes are equal to the one requested in output,
        # no elements generated, unique node ids, and correct number of coordinates per node
        assert len(part.mesh.nodes) == len(output_coordinates)

        for (node_id, node), actual_output_coordinates in zip(part.mesh.nodes.items(), output_coordinates):
            assert node_id not in unique_node_ids
            assert len(node.coordinates) == 3
            # assert that the order of the nodes in the new model part is the same as the one in input
            # meaning, the coordinate of the output nodes has to match one-by-one with the requested output nodes
            npt.assert_almost_equal(actual_output_coordinates, node.coordinates)
            unique_node_ids.append(node.id)

        assert part.mesh.elements == {}

    def test_add_output_to_a_surface_3d(self, create_default_3d_soil_material: SoilMaterial,
                                        create_default_outputs: Output):
        """
        Test if output nodes are correctly accounted for when meshing a surface in 3d.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.
            - create_default_outputs (:class:`stem.output.Output`): the output object containing the \
                output info.
        """

        # define layer coordinates
        ndim = 3
        layer1_coordinates = [(0, 0, 0), (4, 0, 0), (4, 1, 0), (0, 1, 0)]

        # define soil materials
        soil_material1 = create_default_3d_soil_material
        soil_material1.name = "soil1"

        # create model
        model = Model(ndim)
        model.extrusion_length = 4

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "soil1")

        # synchronise geometry and recalculates the ids
        model.synchronise_geometry()
        # Define nodal results
        nodal_results = [NodalOutput.ACCELERATION]
        # Define output coordinates
        output_coordinates = [(0, 1, 2), (2, 1, 2), (4, 1, 2)]

        # add output settings
        model.add_output_settings_by_coordinates(output_coordinates,
                                                 part_name="nodal_accelerations",
                                                 output_name="json_nodal_accelerations_top",
                                                 output_dir="dir_test",
                                                 output_parameters=JsonOutputParameters(output_interval=100,
                                                                                        nodal_results=nodal_results))

        model.set_mesh_size(1)
        model.synchronise_geometry()

        model.generate_mesh()

        unique_element_ids = []
        unique_node_ids = []

        body_model_part = model.body_model_parts[0]
        assert body_model_part.mesh.ndim == 3

        # check if mesh is generated correctly, i.e. if the number of elements is correct and if the element type is
        # correct and if the element ids are unique and if the number of nodes per element is correct
        assert len(body_model_part.mesh.elements) == 187

        for element_id, element in body_model_part.mesh.elements.items():
            assert element.element_type == "TETRAHEDRON_4N"
            assert element_id not in unique_element_ids
            assert len(element.node_ids) == 4
            unique_element_ids.append(element.id)

        # check if nodes are generated correctly, i.e. if there are nodes in the mesh and if the node ids are unique
        # and if the number of coordinates per node is correct
        assert len(body_model_part.mesh.nodes) == 77
        for node_id, node in body_model_part.mesh.nodes.items():
            assert node_id not in unique_node_ids
            assert len(node.coordinates) == 3
            unique_node_ids.append(node.id)

        # assert the output parts
        output_model_part = model.process_model_parts[0]
        assert output_model_part.mesh.ndim == 1

        unique_node_ids = []
        # check if nodes are generated correctly, number of nodes are equal to the one requested in output,
        # no elements generated, unique node ids, and correct number of coordinates per node
        assert len(output_model_part.mesh.nodes) == len(output_coordinates)
        for (node_id, node), actual_output_coordinates in zip(output_model_part.mesh.nodes.items(), output_coordinates):
            assert node_id not in unique_node_ids
            assert len(node.coordinates) == 3
            # assert that the order of the nodes in the new model part is the same as the one in input
            # meaning, the coordinate of the output nodes has to match one-by-one with the requested output nodes
            npt.assert_almost_equal(actual_output_coordinates, node.coordinates)
            unique_node_ids.append(node.id)

        # No element outputs, so the element attribute of the mesh must be an empty dictionary
        assert output_model_part.mesh.elements == {}

    def test_generate_mesh_with_only_a_body_model_part_3d(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if the mesh is generated correctly in 3D if there is only one body model part.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(3)
        model.extrusion_length = 1

        # add soil material
        soil_material = create_default_3d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")
        model.synchronise_geometry()

        # generate mesh
        model.generate_mesh()

        mesh = model.body_model_parts[0].mesh

        assert mesh.ndim == 3

        unique_element_ids = []
        # check if mesh is generated correctly, i.e. if the number of elements is correct and if the element type is
        # correct and if the element ids are unique and if the number of nodes per element is correct
        assert len(mesh.elements) == 1120

        for element_id, element in mesh.elements.items():
            assert element.element_type == "TETRAHEDRON_4N"
            assert element_id not in unique_element_ids
            assert len(element.node_ids) == 4
            unique_element_ids.append(element.id)

        # check if nodes are generated correctly, i.e. if there are nodes in the mesh and if the node ids are unique
        # and if the number of coordinates per node is correct
        unique_node_ids = []
        assert len(mesh.nodes) == 340
        for node_id, node in mesh.nodes.items():
            assert node_id not in unique_node_ids
            assert len(node.coordinates) == 3
            unique_node_ids.append(node.id)

    def test_generate_mesh_with_body_and_process_model_part(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if the mesh is generated correctly in the body model part and a process model part.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(2)

        # add soil material
        soil_material = create_default_2d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")

        # add process geometry
        gmsh_process_input = {"process_0d": {"coordinates": [[0, 0.5, 0]], "ndim": 0}}
        model.gmsh_io.generate_geometry(gmsh_process_input, "")

        # create process model part
        process_model_part = ModelPart("process_0d")

        # set the geometry of the process model part
        process_model_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, "process_0d")

        # add process model part
        model.process_model_parts.append(process_model_part)

        # synchronise geometry and generate mesh
        model.synchronise_geometry()
        model.generate_mesh()

        # check mesh of body model part
        mesh_body = model.body_model_parts[0].mesh

        assert mesh_body.ndim == 2

        unique_element_ids = []
        # check if mesh is generated correctly, i.e. if the number of elements is correct and if the element type is
        # correct and if the element ids are unique and if the number of nodes per element is correct
        assert len(mesh_body.elements) == 162

        for element_id, element in mesh_body.elements.items():
            assert element.element_type == "TRIANGLE_3N"
            assert element_id not in unique_element_ids
            assert len(element.node_ids) == 3
            unique_element_ids.append(element_id)

        # check if nodes are generated correctly, i.e. if there are nodes in the mesh and if the node ids are unique
        # and if the number of coordinates per node is correct
        unique_body_node_ids = []
        assert len(mesh_body.nodes) == 98

        for node_id, node in mesh_body.nodes.items():
            assert node_id not in unique_body_node_ids
            assert len(node.coordinates) == 3
            unique_body_node_ids.append(node.id)

        # check process model part
        mesh_process = model.process_model_parts[0].mesh

        assert mesh_process.ndim == 0

        # check elements of process model part, i.e. if the number of elements is correct and if the element type is
        # correct and if the element ids are unique and if the number of nodes per element is correct
        assert len(mesh_process.elements) == 1
        for element_id, element in mesh_process.elements.items():
            assert element.element_type == "POINT_1N"
            assert element_id == 1
            assert element_id not in unique_element_ids
            assert len(element.node_ids) == 1
            unique_element_ids.append(element.id)

        # check nodes of process model part, i.e. if there is 1 node in the mesh and if the node ids are present in the
        # body mesh and if the number of coordinates per node is correct
        assert len(mesh_process.nodes) == 1
        for node_id, node in mesh_process.nodes.items():
            # check if node is also available in the body mesh
            assert node_id in unique_body_node_ids
            assert len(node.coordinates) == 3

    def test_generate_mesh_2d_2_layers_and_lineload(self, create_default_line_load_parameters: LineLoad,
                                                    create_default_2d_soil_material: SoilMaterial):
        """
        Test if the mesh is generated correctly in 2D for 2 layers plus lineload and fixed bottom.

        Args:
            - create_default_line_load_parameters (:class:`stem.load.LineLoad`): default line load parameters
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(2)

        # add soil material
        soil_material = create_default_2d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (4, 0, 0), (4, 1, 0), (0, 1, 0)], soil_material, "layer1")
        model.add_soil_layer_by_coordinates([(0, 1, 0), (4, 1, 0), (4, 2, 0), (0, 2, 0)], soil_material, "layer2")

        # add line load
        model.add_load_by_coordinates([(4, 2, 0), (0, 2, 0)], create_default_line_load_parameters, "line_load1")

        # add same line load in reversed order
        model.add_load_by_coordinates([(0, 2, 0), (4, 2, 0)], create_default_line_load_parameters, "line_load2")
        model.synchronise_geometry()

        # generate mesh
        model.generate_mesh()

        # check if mesh is generated correctly, i.e. if the number of elements is correct and if the element type is
        # correct, the elements are counterclockwise and the number of nodes per element is correct
        nodes = model.get_all_nodes()
        for bmp in model.body_model_parts:

            for element_id, element in bmp.mesh.elements.items():
                coordinates = [nodes[node_id].coordinates for node_id in element.node_ids]
                assert not Utils.are_2d_coordinates_clockwise(coordinates)
                assert element.element_type == "TRIANGLE_3N"
                assert len(element.node_ids) == 3

        # Check if all condition elements have a body element neighbour
        mapper_process_model_part_1 = model._Model__find_matching_body_elements_for_process_model_part(
            model.process_model_parts[0])

        mapper_process_model_part_2 = model._Model__find_matching_body_elements_for_process_model_part(
            model.process_model_parts[1])

        actual_element_ids_process_1 = [(process_element.id, body_element.id)
                                        for process_element, body_element in mapper_process_model_part_1]

        actual_element_ids_process_2 = [(process_element.id, body_element.id)
                                        for process_element, body_element in mapper_process_model_part_2]

        expected_ids = [(1, 85), (2, 116), (3, 125), (4, 95), (5, 96), (6, 124), (7, 98), (8, 100), (9, 83)]

        # check if the element ids are correct, process model part 1 and 2 should have the same element ids in the same
        # order
        np.testing.assert_equal(desired=expected_ids, actual=actual_element_ids_process_1)
        np.testing.assert_equal(desired=expected_ids, actual=actual_element_ids_process_2)

        # check order of nodes is consistent with what expected.
        node_ids_process_model_part_1 = np.array(
            [el.node_ids for el in model.process_model_parts[0].mesh.elements.values()])

        node_ids_process_model_part_2 = np.array(
            [el.node_ids for el in model.process_model_parts[1].mesh.elements.values()])

        expected_process_connectivities = np.array([[5, 29], [29, 30], [30, 31], [31, 32], [32, 33], [33, 34], [34, 35],
                                                    [35, 36], [36, 6]])

        # check if the node ids are correct, process model part 1 and 2 should have the same node ids in the same
        # order
        npt.assert_equal(node_ids_process_model_part_1, expected_process_connectivities)
        npt.assert_equal(node_ids_process_model_part_2, expected_process_connectivities)

    def test_adjusting_mesh_for_spring_dashpot_elements(self):
        """
        Test that the mesh is adjusted correctly when adding spring elements on nodes at the edges of a line.

        """
        model = Model(ndim=2)

        # add elastic spring damper element
        spring_damper = ElasticSpringDamper(NODAL_DISPLACEMENT_STIFFNESS=[1, 1, 1],
                                            NODAL_ROTATIONAL_STIFFNESS=[1, 1, 2],
                                            NODAL_DAMPING_COEFFICIENT=[1, 1, 3],
                                            NODAL_ROTATIONAL_DAMPING_COEFFICIENT=[1, 1, 4])

        # create model part
        # 3 lines, one broken with a mid-point, which should result in 4 springs
        # the lines are in different size so all the line are broken in smaller lines except the last.

        top_coordinates = [(0, 1, 0), (0, 2, 0), (1, 1, 0), (2, 0.3, 0)]
        bottom_coordinates = [(0, 0, 0), (0, 1, 0), (1, 0, 0), (2, 0, 0)]

        gmsh_input_top = {"top_coordinates": {"coordinates": top_coordinates, "ndim": 0}}
        gmsh_input_bottom = {"bottom_coordinates": {"coordinates": bottom_coordinates, "ndim": 0}}

        model.gmsh_io.generate_geometry(gmsh_input_top, "")
        model.gmsh_io.generate_geometry(gmsh_input_bottom, "")

        # create rail pad geometries
        top_point_ids = model.gmsh_io.make_points(top_coordinates)
        bot_point_ids = model.gmsh_io.make_points(bottom_coordinates)

        spring_line_ids = [
            model.gmsh_io.create_line([top_point_id, bot_point_id])
            for top_point_id, bot_point_id in zip(top_point_ids, bot_point_ids)
        ]

        model.gmsh_io.add_physical_group("spring_damper", 1, spring_line_ids)
        # assign spring damper to geometry
        spring_damper_model_part = BodyModelPart("spring_damper")
        spring_damper_model_part.material = StructuralMaterial("spring_damper", spring_damper)
        spring_damper_model_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, "spring_damper")

        # add model parts to model
        model.body_model_parts.append(spring_damper_model_part)
        model.synchronise_geometry()
        model.set_mesh_size(0.4)

        model.generate_mesh()

        # check the spring node ids are correct
        npt.assert_equal(list(model.body_model_parts[0].mesh.nodes), [2, 1, 5, 3, 6, 4, 7])
        # check that spring element ids are correct
        npt.assert_almost_equal(list(model.body_model_parts[0].mesh.elements), [18, 19, 20, 21])

        # remove mesh and check if raises are raised correctly
        model.body_model_parts[0].mesh = None
        # check that the function raises an error when the mesh is not generated
        expected_message = "Mesh not yet initialised. Please generate the mesh using Model.generate_mesh()"
        with pytest.raises(ValueError, match=expected_message):
            model._Model__adjust_mesh_spring_dampers()

    def test_adjusting_mesh_for_spring_dashpot_elements_with_soil_layer(self,
                                                                        create_default_2d_soil_material: SoilMaterial):
        """
        Test that the mesh is adjusted correctly when adding spring elements on top of a soil layer.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(ndim=2)

        # add soil material
        soil_material = create_default_2d_soil_material

        top_coordinates = [(0, 2, 0), (1, 2, 0), (2, 2, 0)]
        bottom_coordinates = [(0, 0, 0), (1, 0, 0), (2, 0, 0)]

        soil_coordinates_loop = top_coordinates + bottom_coordinates[::-1]

        # add soil layers
        model.add_soil_layer_by_coordinates(soil_coordinates_loop, soil_material, "layer1")

        # add elastic spring damper element
        spring_damper = ElasticSpringDamper(NODAL_DISPLACEMENT_STIFFNESS=[1, 1, 1],
                                            NODAL_ROTATIONAL_STIFFNESS=[1, 1, 2],
                                            NODAL_DAMPING_COEFFICIENT=[1, 1, 3],
                                            NODAL_ROTATIONAL_DAMPING_COEFFICIENT=[1, 1, 4])

        # generate geometries of bottom and top coordinates
        gmsh_input_top = {"top_coordinates": {"coordinates": top_coordinates, "ndim": 0}}
        gmsh_input_bottom = {"bottom_coordinates": {"coordinates": bottom_coordinates, "ndim": 0}}

        model.gmsh_io.generate_geometry(gmsh_input_top, "")
        model.gmsh_io.generate_geometry(gmsh_input_bottom, "")

        # create spring damper geometries and physical group
        top_point_ids = model.gmsh_io.make_points(top_coordinates)
        bot_point_ids = model.gmsh_io.make_points(bottom_coordinates)

        spring_line_ids = [
            model.gmsh_io.create_line([top_point_id, bot_point_id])
            for top_point_id, bot_point_id in zip(top_point_ids, bot_point_ids)
        ]

        model.gmsh_io.add_physical_group("spring_damper", 1, spring_line_ids)

        # assign spring damper to geometry
        spring_damper_model_part = BodyModelPart("spring_damper")
        spring_damper_model_part.material = StructuralMaterial("spring_damper", spring_damper)
        spring_damper_model_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, "spring_damper")

        # add model part to model
        model.body_model_parts.append(spring_damper_model_part)
        model.synchronise_geometry()
        model.set_mesh_size(1)

        # generate mesh
        model.generate_mesh(open_gmsh_gui=False)

        # check if the soil layer is meshed correctly
        assert len(model.body_model_parts[0].mesh.nodes) == 13
        assert len(model.body_model_parts[0].mesh.elements) == 16

        # check if the spring is meshed correctly
        assert len(model.body_model_parts[1].mesh.nodes) == 6
        for node in model.body_model_parts[1].mesh.nodes.values():
            # check if spring damper node is also in soil layer and if the coordinates are the same
            assert node.id in model.body_model_parts[0].mesh.nodes.keys()
            npt.assert_almost_equal(node.coordinates, model.body_model_parts[0].mesh.nodes[node.id].coordinates)

        # check if the spring element ids are correct and not in the soil layer
        assert len(model.body_model_parts[1].mesh.elements) == 3
        for element in model.body_model_parts[1].mesh.elements.values():
            assert element.id not in model.body_model_parts[0].mesh.elements.keys()

    def test__get_line_string_end_nodes_expected_raises(self):
        """
        Test that the function to get the spring end nodes and first element raises errors correctly.

        """

        # create empty modelpart
        model = Model(ndim=2)
        model_part = ModelPart("test")

        # check that the function raises an error when the geometry is not initialised
        with pytest.raises(ValueError, match=f"Geometry of model part `test` not yet initialised."):
            model._Model__get_line_string_end_nodes(model_part)

        # check that the function raises an error when the mesh is not initialised
        model_part.geometry = Geometry()
        with pytest.raises(ValueError, match=f"Mesh of model part `test` not yet initialised."):
            model._Model__get_line_string_end_nodes(model_part)

    def test_find_next_node_along_line_elements(self):
        """
        Test that the function to find the next node along the line elements works correctly. And that it raises
        errors correctly.

        """

        # create empty model
        model = Model(ndim=2)

        # create remaining element ids in random order
        remaining_element_ids = {4, 5, 3, 2, 1}

        # create remaining node ids in random order
        remaining_node_ids = {2, 6, 4, 3, 5}

        # fill in which elements are connected to which nodes
        node_to_elements = {1: [1], 2: [2, 3], 3: [1, 2], 4: [3, 4], 5: [4, 5], 6: [5]}

        # create 5 connected line elements
        line_elements = {
            1: Element(1, "LINE_2N", [1, 3]),
            2: Element(2, "LINE_2N", [3, 2]),
            3: Element(3, "LINE_2N", [2, 4]),
            4: Element(4, "LINE_2N", [4, 5]),
            5: Element(5, "LINE_2N", [5, 6])
        }
        target_node_ids = np.array([2, 3, 4, 5, 6])

        # define expected connected nodes in correct order
        expected_connected_nodes = [3, 2, 4, 5, 6]

        # first node is the start node
        first_node = 1

        # find next node along line elements
        for i in range(len(expected_connected_nodes)):
            next_node = model._Model__find_next_node_along_line_elements(first_node, remaining_element_ids,
                                                                         remaining_node_ids, node_to_elements,
                                                                         line_elements, target_node_ids)

            assert next_node == expected_connected_nodes[i]

            first_node = next_node

        # check if error is raised because the next node cannot be found
        target_node_ids = np.array([9])

        with pytest.raises(ValueError,
                           match=re.escape("Next node along the line cannot be found. "
                                           "As it is not included in the search space")):
            _ = model._Model__find_next_node_along_line_elements(first_node, remaining_element_ids, remaining_node_ids,
                                                                 node_to_elements, line_elements, target_node_ids)

        # create a fork
        line_elements[6] = Element(6, "LINE_2N", [3, 7])
        target_node_ids = np.array([3])
        remaining_node_ids = {2, 6, 4, 3, 5, 7}
        remaining_element_ids = {1, 2, 6}
        node_to_elements[3] = [1, 2, 6]
        node_to_elements[7] = [6]

        # check if fork is detected and error is raised
        first_node = 3
        with pytest.raises(ValueError,
                           match=re.escape("There is a fork in the mesh at elements: {1, 2, 6}, "
                                           "the next node along the line cannot be found.")):
            _ = model._Model__find_next_node_along_line_elements(first_node, remaining_element_ids, remaining_node_ids,
                                                                 node_to_elements, line_elements, target_node_ids)

        # check if error is raised when not all elements are line elements
        line_elements[7] = Element(7, "TRIANGLE_3N", [3, 7, 8])
        with pytest.raises(ValueError, match=re.escape("Not all elements are line elements.")):
            _ = model._Model__find_next_node_along_line_elements(first_node, remaining_element_ids, remaining_node_ids,
                                                                 node_to_elements, line_elements, target_node_ids)

    def test_add_field_raises_errors(self, create_default_2d_soil_material: SoilMaterial):
        """
        Checks that the function to add parameter field raises errors correctly.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(2)

        # add soil material
        soil_material = create_default_2d_soil_material

        # add fake body model part with no material
        model.body_model_parts.append(BodyModelPart(name="fake part"))
        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")
        line_load_parameters = LineLoad(active=[True, True, True], value=[0, -1000, 0])
        model.add_load_by_coordinates(name="line_load",
                                      coordinates=[(0, 0, 0), (0, 1, 0)],
                                      load_parameters=line_load_parameters)

        # Define the field generator
        correct_rf_generator = RandomFieldGenerator(cov=0.1,
                                                    model_name="Gaussian",
                                                    v_scale_fluctuation=5,
                                                    anisotropy=0.5,
                                                    angle=0,
                                                    seed=42)

        # define the field parameters
        correct_field_parameters_json = ParameterFieldParameters(property_names=["YOUNG_MODULUS"],
                                                                 function_type="json_file",
                                                                 field_file_names=["json_file.json"],
                                                                 field_generator=correct_rf_generator)

        # Define the field generator
        wrong_rf_generator = RandomFieldGenerator(cov=0.1,
                                                  model_name="Gaussian",
                                                  v_scale_fluctuation=5,
                                                  anisotropy=0.5,
                                                  angle=0,
                                                  seed=42)
        wrong_field_parameters_json = ParameterFieldParameters(property_names=["YOUNGS_MODULUS"],
                                                               function_type="json_file",
                                                               field_file_names=["json_file.json"],
                                                               field_generator=wrong_rf_generator)

        wrong_field_parameters_json_boolean = ParameterFieldParameters(property_names=["IS_DRAINED"],
                                                                       function_type="json_file",
                                                                       field_file_names=["json_file.json"],
                                                                       field_generator=wrong_rf_generator)

        # add random field to process model part
        msg = "The target part, `line_load`, is not a body model part."
        with pytest.raises(ValueError, match=msg):
            model.add_field(part_name="line_load", field_parameters=correct_field_parameters_json)

        # add random field to part with no material
        msg = "No material assigned to the body model part!"
        with pytest.raises(ValueError, match=msg):
            model.add_field(part_name="fake part", field_parameters=correct_field_parameters_json)

        # add random field to non-existing property
        msg = "Property YOUNGS_MODULUS is not one of the parameters of the soil material"
        with pytest.raises(ValueError, match=msg):
            model.add_field(part_name="layer1", field_parameters=wrong_field_parameters_json)

        # add random field to boolean property
        msg = "The property for which a random field needs to be generated, `IS_DRAINED` is not a numeric value."
        with pytest.raises(ValueError, match=msg):
            model.add_field(part_name="layer1", field_parameters=wrong_field_parameters_json_boolean)

    def test_random_field_generation_2d(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test the correct generation of the random field for a 2D model with one body model part.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(2)

        # add soil material
        soil_material = create_default_2d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")
        model.set_mesh_size(1)

        # Define the field generator
        random_field_generator = RandomFieldGenerator(cov=0.1,
                                                      model_name="Gaussian",
                                                      v_scale_fluctuation=1,
                                                      anisotropy=[0.5],
                                                      angle=[0],
                                                      seed=42)

        field_parameters_json = ParameterFieldParameters(property_names=["YOUNG_MODULUS"],
                                                         function_type="json_file",
                                                         field_generator=random_field_generator)

        model.add_field(part_name="layer1", field_parameters=field_parameters_json)
        model.synchronise_geometry()

        # generate mesh
        model.generate_mesh()

        actual_rf_values = model.process_model_parts[-1].parameters.field_generator.generated_fields[0]

        # assert the number of generated values to be equal to the amount of elements of the part
        assert len(actual_rf_values) == len(model.body_model_parts[0].mesh.elements)
        # assert the generated values against the expected values
        expected_rf_values = [104971256.1059345, 113280413.42177339, 105124797.09835173, 109686556.57934019]

        npt.assert_allclose(actual=actual_rf_values, desired=expected_rf_values)

    @pytest.mark.skipif(IS_LINUX,
                        reason="The 3D random field samples different values for linux and windows, "
                        "because the mesh is slightly different. See also the test for mdpa_file in "
                        "3d in test_kratos_io.py.")
    def test_random_field_generation_3d(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test the correct generation of the random field for a 3D model with one body model part.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(3)
        model.extrusion_length = 1

        # add soil material
        soil_material = create_default_3d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")
        model.set_mesh_size(1.0)

        # Define the field generator
        random_field_generator = RandomFieldGenerator(cov=0.1,
                                                      model_name="Gaussian",
                                                      v_scale_fluctuation=1,
                                                      anisotropy=[0.5, 0.5],
                                                      angle=[0, 0],
                                                      seed=42)

        field_parameters_json = ParameterFieldParameters(property_names=["YOUNG_MODULUS"],
                                                         function_type="json_file",
                                                         field_generator=random_field_generator)

        model.add_field(part_name="layer1", field_parameters=field_parameters_json)

        model.synchronise_geometry()

        # generate mesh
        model.generate_mesh()

        actual_rf_values = model.process_model_parts[0].parameters.field_generator.generated_fields[0]

        # TODO: make test for Unix  with different values

        # assert the number of generated values to be equal to the amount of elements of the part
        assert len(actual_rf_values) == len(model.body_model_parts[0].mesh.elements)

        expected_rf_values = [
            109219152.50312316, 103358912.90787594, 105339578.47289738, 107804266.66256714, 116674453.0103657,
            121205355.8771256, 117518624.66410118, 109641232.38516402, 108150391.42392428, 93740844.72077464,
            106608642.49695791, 111016462.96330133, 95787906.70407471, 109879617.69834961, 103724463.91386327,
            92715313.3744301, 115556177.86463425, 119222050.2452586, 112966908.38899206, 94554356.2203453,
            112709106.84842391, 93573278.00303535, 100680007.50177462, 105511523.87671089
        ]

        npt.assert_allclose(actual=actual_rf_values, desired=expected_rf_values)

    def test_validate_expected_success(self):
        """
        Test if the model is validated correctly. A model is created with two process model parts which both have
        a unique name.

        """

        model = Model(2)

        model_part1 = ModelPart("test1")
        model_part2 = ModelPart("test2")

        model.process_model_parts = [model_part1, model_part2]

        model.validate()

    def test_validate_expected_fail_non_unique_names(self):
        """
        Test if the model is validated correctly. A model is created with two process model parts which both have
        the same name. This should raise a ValueError.

        """

        model = Model(2)

        model_part1 = ModelPart("test")
        model_part2 = ModelPart("test")

        model.process_model_parts = [model_part1, model_part2]

        pytest.raises(ValueError, model.validate)

    def test_validate_expected_fail_no_name(self):
        """
        Test if the model is validated correctly. A model is created with a process model part which does not contain
        a name. This should raise a ValueError.

        """

        model = Model(2)

        model_part1 = ModelPart(None)
        model.process_model_parts = [model_part1]

        pytest.raises(ValueError, model.validate)

    def test_add_boundary_condition_by_geometry_ids(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if a boundary condition is added correctly to the model. A boundary condition is added to the model by
        specifying the geometry ids to which the boundary condition should be applied.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # create a 3D model
        model = Model(3)
        model.extrusion_length = 1

        # create multiple boundary condition parameters
        no_rotation_parameters = RotationConstraint(active=[True, True, True],
                                                    is_fixed=[True, True, True],
                                                    value=[0, 0, 0])

        absorbing_parameters = AbsorbingBoundary(absorbing_factors=[1, 1], virtual_thickness=0)

        no_displacement_parameters = DisplacementConstraint(active=[True, True, True],
                                                            is_fixed=[True, True, True],
                                                            value=[0, 0, 0])

        # add body model part
        soil_material = create_default_3d_soil_material
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "test_soil")

        # add boundary conditions in 0d, 1d and 2d
        model.add_boundary_condition_by_geometry_ids(0, [1, 2], no_rotation_parameters, "no_rotation")
        model.add_boundary_condition_by_geometry_ids(1, [8], absorbing_parameters, "absorbing")
        model.add_boundary_condition_by_geometry_ids(2, [1, 2], no_displacement_parameters, "no_displacement")

        model.synchronise_geometry()

        # set expected parameters of the boundary conditions
        expected_0d_model_part_parameters = RotationConstraint(active=[True, True, True],
                                                               is_fixed=[True, True, True],
                                                               value=[0, 0, 0])

        expected_1d_model_part_parameters = AbsorbingBoundary(absorbing_factors=[1, 1], virtual_thickness=0)

        expected_2d_model_part_parameters = DisplacementConstraint(active=[True, True, True],
                                                                   is_fixed=[True, True, True],
                                                                   value=[0, 0, 0])

        # set expected geometry 0d boundary condition
        expected_boundary_points = {1: Point.create([0, 0, 0], 1), 2: Point.create([1, 0, 0], 2)}
        expected_boundary_lines = {1: Line.create([1, 2], 1)}
        expected_boundary_surfaces = {}
        expected_boundary_volumes = {}

        expected_boundary_geometry_0d = Geometry(expected_boundary_points, expected_boundary_lines,
                                                 expected_boundary_surfaces, expected_boundary_volumes)

        # set expected geometry 1d boundary condition
        expected_boundary_points = {3: Point.create([1, 1, 0], 3), 7: Point.create([1, 1, 1], 7)}
        expected_boundary_lines = {8: Line.create([3, 7], 8)}
        expected_boundary_surfaces = {}
        expected_boundary_volumes = {}

        expected_boundary_geometry_1d = Geometry(expected_boundary_points, expected_boundary_lines,
                                                 expected_boundary_surfaces, expected_boundary_volumes)

        # set expected geometry 2d boundary condition

        expected_boundary_geometry_2d = Geometry()
        expected_boundary_geometry_2d.points = {
            1: Point.create([0, 0, 0], 1),
            2: Point.create([1, 0, 0], 2),
            3: Point.create([1, 1, 0], 3),
            4: Point.create([0, 1, 0], 4),
            5: Point.create([0, 0, 1], 5),
            6: Point.create([1, 0, 1], 6)
        }

        expected_boundary_geometry_2d.lines = {
            1: Line.create([1, 2], 1),
            2: Line.create([2, 3], 2),
            3: Line.create([3, 4], 3),
            4: Line.create([4, 1], 4),
            5: Line.create([1, 5], 5),
            7: Line.create([5, 6], 7),
            6: Line.create([2, 6], 6)
        }

        expected_boundary_geometry_2d.surfaces = {
            1: Surface.create([1, 2, 3, 4], 1),
            2: Surface.create([5, 7, -6, -1], 2)
        }

        expected_boundary_geometry_2d.volumes = {}

        # collect all expected geometries
        all_expected_geometries = [
            expected_boundary_geometry_0d, expected_boundary_geometry_1d, expected_boundary_geometry_2d
        ]

        # check 0d parameters
        npt.assert_allclose(model.process_model_parts[0].parameters.active, expected_0d_model_part_parameters.active)
        npt.assert_allclose(model.process_model_parts[0].parameters.is_fixed,
                            expected_0d_model_part_parameters.is_fixed)
        npt.assert_allclose(model.process_model_parts[0].parameters.value, expected_0d_model_part_parameters.value)

        # check 1d parameters
        npt.assert_allclose(model.process_model_parts[1].parameters.absorbing_factors,
                            expected_1d_model_part_parameters.absorbing_factors)
        npt.assert_allclose(model.process_model_parts[1].parameters.virtual_thickness,
                            expected_1d_model_part_parameters.virtual_thickness)

        # check 2d parameters
        npt.assert_allclose(model.process_model_parts[2].parameters.active, expected_2d_model_part_parameters.active)
        npt.assert_allclose(model.process_model_parts[2].parameters.is_fixed,
                            expected_2d_model_part_parameters.is_fixed)
        npt.assert_allclose(model.process_model_parts[2].parameters.value, expected_2d_model_part_parameters.value)

        for expected_geometry, model_part in zip(all_expected_geometries, model.process_model_parts):

            TestUtils.assert_almost_equal_geometries(expected_geometry, model_part.geometry)

    def test_add_load_by_geometry_ids(self, create_default_3d_soil_material: SoilMaterial,
                                      create_default_point_load_parameters: PointLoad,
                                      create_default_line_load_parameters: LineLoad,
                                      create_default_surface_load_parameters: SurfaceLoad):
        """
        Test if a load is added correctly to the model. Here the load is added to the model by
        specifying the geometry ids to which the load should be applied.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.
            - create_default_point_load_parameters (:class:`stem.load.PointLoad`): default point load parameters.
            - create_default_surface_load_parameters (:class:`stem.load.SurfaceLoad`): default surface load parameters.

        """

        ndim = 3

        # create a 3D model
        model = Model(ndim)
        model.extrusion_length = 1

        # set expected parameters of the load conditions
        expected_point_load_parameters = PointLoad(active=[False, True, False], value=[0, -200, 0])
        expected_line_load_parameters = LineLoad(active=[False, True, False], value=[0, -20, 0])
        expected_surface_load_parameters = SurfaceLoad(active=[False, True, False], value=[0, -2, 0])
        expected_moving_load_parameters = MovingLoad(origin=[0, 1, 0.5],
                                                     load=[0.0, -10.0, 0.0],
                                                     velocity=5.0,
                                                     offset=3.0,
                                                     direction=[1, 1, 1])

        # add body model part
        soil_material = create_default_3d_soil_material
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "test_soil")

        # add point, line, surface and moving loads by geometry id
        model.add_load_by_geometry_ids([3, 4, 7, 8], create_default_point_load_parameters, "point_loads")
        model.add_load_by_geometry_ids([3, 8, 10, 11], create_default_line_load_parameters, "line_loads")
        model.add_load_by_geometry_ids([4], create_default_surface_load_parameters, "surface_load")
        model.add_load_by_geometry_ids([3, 8, 10], expected_moving_load_parameters, "moving_load")

        # set expected geometry point load
        expected_load_points = {
            3: Point.create([1, 1, 0], 3),
            4: Point.create([0, 1, 0], 4),
            7: Point.create([1, 1, 1], 7),
            8: Point.create([0, 1, 1], 8)
        }
        expected_point_load_geometry = Geometry(expected_load_points, {}, {}, {})

        # set expected geometry line load
        expected_load_lines = {
            3: Line.create([3, 4], 3),
            8: Line.create([3, 7], 8),
            10: Line.create([4, 8], 10),
            11: Line.create([7, 8], 11)
        }

        expected_line_load_geometry = Geometry(expected_load_points, expected_load_lines, {}, {})

        # set expected geometry surface load
        expected_surface_load_geometry = Geometry()
        expected_surface_load_geometry.points = {
            3: Point.create([1, 1, 0], 3),
            7: Point.create([1, 1, 1], 7),
            8: Point.create([0, 1, 1], 8),
            4: Point.create([0, 1, 0], 4)
        }
        expected_surface_load_geometry.lines = {
            8: Line.create([3, 7], 8),
            11: Line.create([7, 8], 11),
            10: Line.create([4, 8], 10),
            3: Line.create([3, 4], 3)
        }
        expected_surface_load_geometry.surfaces = {4: Surface.create([8, 11, -10, -3], 4)}

        # set expected geometry moving load
        expected_surface_load_geometry = Geometry()
        expected_surface_load_geometry.points = {
            3: Point.create([1, 1, 0], 3),
            7: Point.create([1, 1, 1], 7),
            8: Point.create([0, 1, 1], 8),
            4: Point.create([0, 1, 0], 4)
        }
        expected_surface_load_geometry.lines = {
            8: Line.create([3, 7], 8),
            11: Line.create([7, 8], 11),
            10: Line.create([4, 8], 10),
            3: Line.create([3, 4], 3)
        }
        expected_surface_load_geometry.surfaces = {4: Surface.create([8, 11, -10, -3], 4)}

        # collect all expected geometriesl
        all_expected_geometries = [
            expected_point_load_geometry, expected_line_load_geometry, expected_surface_load_geometry
        ]

        for expected_geometry, model_part in zip(all_expected_geometries, model.process_model_parts):

            TestUtils.assert_almost_equal_geometries(expected_geometry, model_part.geometry)

        # check point load parameters
        npt.assert_allclose(model.process_model_parts[0].parameters.value, expected_point_load_parameters.value)
        npt.assert_allclose(model.process_model_parts[0].parameters.active, expected_point_load_parameters.active)

        # check line load parameters
        npt.assert_allclose(model.process_model_parts[1].parameters.value, expected_line_load_parameters.value)
        npt.assert_allclose(model.process_model_parts[1].parameters.active, expected_line_load_parameters.active)

        # check surface load parameters
        npt.assert_allclose(model.process_model_parts[2].parameters.value, expected_surface_load_parameters.value)
        npt.assert_allclose(model.process_model_parts[2].parameters.active, expected_surface_load_parameters.active)

        # check moving load parameters
        TestUtils.assert_dictionary_almost_equal(model.process_model_parts[3].parameters.__dict__,
                                                 expected_moving_load_parameters.__dict__)

    def test_add_load_by_geometry_ids_raises_error(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if a load is added correctly to the model. Here the load is added to the model by
        specifying the geometry ids to which the load should be applied.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        ndim = 3

        # create a 3D model
        model = Model(ndim)
        model.extrusion_length = 1

        soil_material = create_default_3d_soil_material
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "test_soil")

        moving_load_parameters = MovingLoad(origin=[0, 1, 0.5],
                                            load=[0.0, -10.0, 0.0],
                                            velocity=5.0,
                                            offset=3.0,
                                            direction=[1, 1, 1])

        # check raising of errors
        msg = "Load parameter provided is not supported: `GravityLoad`."
        with pytest.raises(NotImplementedError, match=msg):
            model.add_load_by_geometry_ids([1], GravityLoad(value=[0, -9.81, 0], active=[True, True, True]),
                                           "gravity load")
        # lines disconnected

        msg = ("The lines defined for the moving load are not aligned on a path."
               "Discontinuities or loops/branching points are found.")
        with pytest.raises(ValueError, match=msg):
            model.add_load_by_geometry_ids([8, 10], moving_load_parameters, "moving_load_wrong_1")

        # origin not in path
        # test for branching points
        msg = "None of the lines are aligned with the origin of the moving load. Error."
        with pytest.raises(ValueError, match=msg):
            model.add_load_by_geometry_ids([3, 8, 11], moving_load_parameters, "moving_load_wrong_2")

    def test_add_gravity_load_1d_and_2d(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if a gravity load is added correctly to the model in a 2d space containing 1d and 2d elements. A gravity
        load is generated and added to the model.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # create model
        model = Model(2)

        # add a 2d layer
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0)], create_default_2d_soil_material, "soil1")

        # add a 1d layer
        layer_settings = {"beam": {"ndim": 1, "element_size": -1, "coordinates": [[0, 0, 0], [1, 0, 0]]}}

        model.gmsh_io.generate_geometry(layer_settings, "")
        model.synchronise_geometry()

        # add 1d model part to model
        body_model_part = BodyModelPart("beam")
        body_model_part.material = EulerBeam(ndim=2,
                                             YOUNG_MODULUS=1e6,
                                             POISSON_RATIO=0.3,
                                             DENSITY=1,
                                             CROSS_AREA=1,
                                             I33=1)
        body_model_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, "beam")

        model.body_model_parts.append(body_model_part)

        # add gravity load
        model._Model__add_gravity_load()

        assert len(model.process_model_parts) == 2
        assert model.process_model_parts[0].name == "gravity_load_1d"
        assert model.process_model_parts[1].name == "gravity_load_2d"

        # setup expected geometries for 1d and 2d
        expected_geometry_points_1d = {1: Point.create([0, 0, 0], 1), 2: Point.create([1, 0, 0], 2)}
        expected_geometry_lines_1d = {1: Line.create([1, 2], 1)}
        expected_geometry_gravity_1d = Geometry(expected_geometry_points_1d, expected_geometry_lines_1d, {}, {})

        expected_geometry_points_2d = {
            1: Point.create([0, 0, 0], 1),
            2: Point.create([1, 0, 0], 2),
            3: Point.create([1, 1, 0], 3)
        }
        expected_geometry_lines_2d = {1: Line.create([1, 2], 1), 2: Line.create([2, 3], 2), 3: Line.create([3, 1], 3)}
        expected_geometry_surfaces_2d = {1: Surface.create([1, 2, 3], 1)}
        expected_geometry_gravity_2d = Geometry(expected_geometry_points_2d, expected_geometry_lines_2d,
                                                expected_geometry_surfaces_2d, {})

        expected_geometries = [expected_geometry_gravity_1d, expected_geometry_gravity_2d]

        # check if all process model parts are correct
        for model_part in model.process_model_parts:

            # check if parameters are added correctly
            npt.assert_allclose(model_part.parameters.value, [0, -9.81, 0])
            npt.assert_allclose(model_part.parameters.active, [True, True, True])

            # check if geometry is added correctly
            generated_model_part = model_part.geometry

            TestUtils.assert_almost_equal_geometries(expected_geometries[0], generated_model_part)

    def test_add_gravity_load_two_layers_same_dimension(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if a gravity load is added correctly to the model in a 2d space containing 2 layers. A gravity load is
        generated and added to the model.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # create model
        model = Model(2)

        # add a 2d layer
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0)], create_default_2d_soil_material, "soil1")
        model.add_soil_layer_by_coordinates([(1, 0, 0), (0, 0, 0), (1, -1, 0)], create_default_2d_soil_material,
                                            "soil2")

        model.synchronise_geometry()

        # add gravity load
        model._Model__add_gravity_load()

        assert len(model.process_model_parts) == 1

        generated_geometry = model.process_model_parts[0].geometry

        # check if number of points, lines, surfaces are correct, i.e. if the number of points, lines, surfaces are the
        # same as the number of points, lines, surfaces of the model geometry
        assert len(generated_geometry.points) == len(model.geometry.points) == 4
        assert len(generated_geometry.lines) == len(model.geometry.lines) == 5
        assert len(generated_geometry.surfaces) == len(model.geometry.surfaces) == 2

        assert model.process_model_parts[0].name == "gravity_load_2d"
        npt.assert_allclose(model.process_model_parts[0].parameters.value, [0, -9.81, 0])
        npt.assert_allclose(model.process_model_parts[0].parameters.active, [True, True, True])

    def test_add_gravity_load_3d(self, create_default_3d_soil_material):
        """
        Test if a gravity load is added correctly to the model in a 3d space. A gravity load is generated and added to
        the model.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # create model
        model = Model(3)
        model.extrusion_length = 1

        # add a 2d layer
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0)], create_default_3d_soil_material, "soil1")

        model.synchronise_geometry()

        # add gravity load
        model._Model__add_gravity_load()

        assert len(model.process_model_parts) == 1

        generated_geometry = model.process_model_parts[0].geometry

        # check if number of points, lines, surfaces are correct, i.e. if the number of points, lines, surfaces and
        # volumes are the same as the number of points, lines, surfaces and volumes of the model geometry
        assert len(generated_geometry.points) == len(model.geometry.points) == 6
        assert len(generated_geometry.lines) == len(model.geometry.lines) == 9
        assert len(generated_geometry.surfaces) == len(model.geometry.surfaces) == 5
        assert len(generated_geometry.volumes) == len(model.geometry.volumes) == 1

        assert model.process_model_parts[0].name == "gravity_load_3d"
        npt.assert_allclose(model.process_model_parts[0].parameters.value, [0, -9.81, 0])
        npt.assert_allclose(model.process_model_parts[0].parameters.active, [True, True, True])

    def test_setup_stress_initialisation(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if the stress initialisation is set up correctly. A model is created with a soil layer. It is checked if
        gravity is added in case the K0 procedure or gravity loading is used.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # set up solver settings
        analysis_type = AnalysisType.MECHANICAL_GROUNDWATER_FLOW

        solution_type = SolutionType.QUASI_STATIC

        time_integration = TimeIntegration(start_time=0.0,
                                           end_time=1.0,
                                           delta_time=0.1,
                                           reduction_factor=0.5,
                                           increase_factor=2.0,
                                           max_delta_time_factor=500)

        convergence_criterion = DisplacementConvergenceCriteria()

        stress_initialisation_type = StressInitialisationType.NONE

        solver_settings = SolverSettings(analysis_type=analysis_type,
                                         solution_type=solution_type,
                                         stress_initialisation_type=stress_initialisation_type,
                                         time_integration=time_integration,
                                         is_stiffness_matrix_constant=True,
                                         are_mass_and_damping_constant=True,
                                         convergence_criteria=convergence_criterion)

        # set up problem data
        problem_data = Problem(problem_name="test", number_of_threads=2, settings=solver_settings)

        model_no_gravity = Model(2)
        model_no_gravity.project_parameters = problem_data

        # set up soil material
        soil_material = create_default_2d_soil_material
        model_no_gravity.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0)], soil_material, "soil1")
        model_no_gravity.synchronise_geometry()

        # setup_stress_initialisation
        model_no_gravity._Model__setup_stress_initialisation()

        model_k0 = Model(2)
        model_k0.project_parameters = problem_data

        model_k0.project_parameters.settings.stress_initialisation_type = StressInitialisationType.K0_PROCEDURE
        model_k0.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0)], soil_material, "soil1")
        model_k0.synchronise_geometry()

        # setup_stress_initialisation
        model_k0._Model__setup_stress_initialisation()

        model_gravity_loading = Model(2)
        model_gravity_loading.project_parameters = problem_data

        model_gravity_loading.project_parameters.settings.stress_initialisation_type = \
            StressInitialisationType.GRAVITY_LOADING
        model_gravity_loading.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0)], soil_material, "soil1")
        model_gravity_loading.synchronise_geometry()

        # setup_stress_initialisation
        model_gravity_loading._Model__setup_stress_initialisation()

        assert len(model_no_gravity.process_model_parts) == 0
        assert len(model_k0.process_model_parts) == 1
        assert len(model_gravity_loading.process_model_parts) == 1

        assert model_k0.process_model_parts[0].name == "gravity_load_2d"
        assert model_gravity_loading.process_model_parts[0].name == "gravity_load_2d"

    def test_setup_stress_initialisation_without_project_parameters(self):
        """
        A model is created without project parameters. It is
        checked if a ValueError is raised while setting up the stress initialisation.

        """
        # create model
        model = Model(2)

        # test if value error is raised
        with pytest.raises(ValueError,
                           match=r"Project parameters must be set before setting up the stress initialisation"):
            model._Model__setup_stress_initialisation()

    def test_check_ordering_process_model_part_2d(self):
        """
        Test if the node order of the process model part is flipped, such that the nodes follow the same order as
        the neighbour element. After filling in the nodes of the process model part in reverse order.

        """

        # create model
        model = Model(2)

        # manually set mesh data nodes
        model.gmsh_io._GmshIO__mesh_data = {"nodes": {1: [0, 0, 0], 2: [1, 0, 0], 3: [1, 1, 0], 4: [0, 1, 0]}}

        # manually create process model part with nodes in reverse order
        process_element = Element(1, "LINE_2N", [2, 1])
        process_model_part = ModelPart("process")
        process_mesh = Mesh(1)
        process_mesh.elements = {1: process_element}
        process_mesh.nodes = {1: Node(1, [0, 0, 0]), 2: Node(2, [1, 0, 0])}
        process_model_part.mesh = process_mesh
        model.process_model_parts = [ModelPart("process")]

        # create body_model_part
        body_element = Element(2, "TRIANGLE_3N", [1, 2, 3])

        # check ordering of process model part connectivities
        mapper = [(process_element, body_element)]
        model._Model__check_ordering_process_model_part(mapper, process_model_part)

        # check if the node ids of the process model part are in the correct order
        assert process_model_part.mesh.elements[1].node_ids == [1, 2]

    def test_check_ordering_process_model_part_2d_multiple_elements(self):
        """
        Test if the node order of the first element in process model part is flipped in a 2D case. The second element
        should not be flipped. This test check for any order of the process element to body element mapping.

        """

        # create model
        model = Model(2)

        # manually set mesh data nodes
        model.gmsh_io._GmshIO__mesh_data = {"nodes": {1: [0, 0, 0], 2: [1, 0, 0], 3: [1, 1, 0], 4: [0, 1, 0]}}

        # manually create process model part with nodes in reverse order
        process_element1 = Element(1, "LINE_2N", [2, 1])
        process_element2 = Element(2, "LINE_2N", [2, 3])
        process_model_part = ModelPart("process")
        process_mesh = Mesh(1)
        process_mesh.elements = {1: process_element1, 2: process_element2}
        process_mesh.nodes = {1: Node(1, [0, 0, 0]), 2: Node(2, [1, 0, 0]), 3: Node(3, [1, 1, 0])}
        process_model_part.mesh = process_mesh
        model.process_model_parts = [ModelPart("process")]

        # create body_model_part
        body_element = Element(2, "TRIANGLE_3N", [1, 2, 3])

        # check ordering of process model part connectivities
        mapper = [(process_element1, body_element), (process_element2, body_element)]
        model._Model__check_ordering_process_model_part(mapper, process_model_part)

        # check if the node ids of the process model part are in the correct order, i.e. the node order of only the
        # first element should be flipped
        assert process_model_part.mesh.elements[1].node_ids == [1, 2]
        assert process_model_part.mesh.elements[2].node_ids == [2, 3]

        # redefine process elements and redefine mapper in reverse order
        # manually create process model part with nodes in outwards normal order
        process_element1 = Element(1, "LINE_2N", [2, 1])
        process_element2 = Element(2, "LINE_2N", [2, 3])
        process_mesh.elements = {1: process_element1, 2: process_element2}

        # add process_element and body_element to mapper
        mapper = [(process_element2, body_element), (process_element1, body_element)]

        # check ordering of process model part connectivities
        model._Model__check_ordering_process_model_part(mapper, process_model_part)

        # check if the node ids of the process model part are in the correct order, i.e. the node order of only the
        # first element should be flipped (the same order as before)
        assert process_model_part.mesh.elements[1].node_ids == [1, 2]
        assert process_model_part.mesh.elements[2].node_ids == [2, 3]

    def test_check_ordering_process_model_part_3d(self):
        """
        Test if the node order of the process model part is flipped, such that the normal is inwards. After filling in
        the nodes of the process model part in outwards normal order.

        """

        # create model
        model = Model(3)

        # manually set mesh data nodes
        model.gmsh_io._GmshIO__mesh_data = {"nodes": {1: [0, 0, 0], 2: [1, 0, 0], 3: [1, 1, 0], 4: [0, 0, 1]}}

        # manually create process model part with nodes in outwards normal order
        process_element = Element(1, "TRIANGLE_3N", [2, 1, 3])
        process_model_part = ModelPart("process")
        process_mesh = Mesh(1)
        process_mesh.elements = {1: process_element}
        process_mesh.nodes = {1: Node(1, [0, 0, 0]), 2: Node(2, [1, 0, 0])}
        process_model_part.mesh = process_mesh
        model.process_model_parts = [ModelPart("process")]

        # create body_model_part
        body_element = Element(2, "TETRAHEDRON_4N", [1, 2, 3, 4])

        # check ordering of process model part connectivities
        mapper = [(process_element, body_element)]
        model._Model__check_ordering_process_model_part(mapper, process_model_part)

        # check if the node ids of the process model part are in the correct order, i.e. the node order should be
        # flipped, such that the normal is inwards
        assert process_model_part.mesh.elements[1].node_ids == [2, 1, 3]

    def test_check_ordering_process_model_part_3d_multiple_elements(self):
        """
        Test if the node order of the first element in process model part is flipped, such that the normal is inwards.
        The second element should not be flipped. This test check for any order of the process element to body element
        mapping.

        """

        # create model
        model = Model(3)

        # manually set mesh data nodes
        model.gmsh_io._GmshIO__mesh_data = {"nodes": {1: [0, 0, 0], 2: [1, 0, 0], 3: [1, 1, 0], 4: [0, 0, 1]}}

        # manually create process model part with nodes in outwards normal order
        process_element1 = Element(1, "TRIANGLE_3N", [2, 1, 3])
        process_element2 = Element(2, "TRIANGLE_3N", [4, 3, 2])
        process_model_part = ModelPart("process")
        process_mesh = Mesh(2)
        process_mesh.elements = {1: process_element1, 2: process_element2}
        process_mesh.nodes = {
            1: Node(1, [0, 0, 0]),
            2: Node(2, [1, 0, 0]),
            3: Node(3, [1, 1, 0]),
            4: Node(4, [0, 0, 1])
        }
        process_model_part.mesh = process_mesh
        model.process_model_parts = [ModelPart("process")]

        # create body_model_part
        body_element = Element(2, "TETRAHEDRON_4N", [1, 2, 3, 4])

        # add process_element and body_element to mapper
        mapper = [(process_element1, body_element), (process_element2, body_element)]

        # check ordering of process model part connectivities
        model._Model__check_ordering_process_model_part(mapper, process_model_part)

        # check if the node ids of the process model part are in the correct order, i.e. the node order of only the
        # first element should be flipped, such that the normal is inwards
        assert process_model_part.mesh.elements[1].node_ids == [2, 1, 3]
        assert process_model_part.mesh.elements[2].node_ids == [2, 3, 4]

        # redefine process elements and redefine mapper in reverse order
        # manually create process model part with nodes in outwards normal order
        process_element1 = Element(1, "TRIANGLE_3N", [2, 1, 3])
        process_element2 = Element(2, "TRIANGLE_3N", [4, 3, 2])
        process_mesh.elements = {1: process_element1, 2: process_element2}

        # add process_element and body_element to mapper
        mapper = [(process_element2, body_element), (process_element1, body_element)]

        # check ordering of process model part connectivities
        model._Model__check_ordering_process_model_part(mapper, process_model_part)

        # check if the node ids of the process model part are in the correct order, i.e. the node order of only the
        # first element should be flipped, such that the normal is inwards (the same order as before)
        assert process_model_part.mesh.elements[1].node_ids == [2, 1, 3]
        assert process_model_part.mesh.elements[2].node_ids == [2, 3, 4]

    def test_show_geometry_file(self, create_default_3d_soil_material):
        """
        Test if the geometry html file is generated. A model is created with a soil layer. The geometry is plotted to a
         html file and the file is checked if it is created.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        # define soil material
        soil_material = create_default_3d_soil_material

        # create model
        model = Model(3)
        model.extrusion_length = 1

        # add soil layer
        layer_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")
        model.synchronise_geometry()

        model.show_geometry(file_name=r"tests/test_geometry.html", auto_open=False)

        # check if the file is created with pathlib
        assert Path(r"tests/test_geometry.html").exists()

        # remove file
        Path(r"tests/test_geometry.html").unlink()

    def test_post_setup_with_gravity_2D(self, expected_geometry_two_layers_2D: Tuple[Geometry, Geometry, Geometry],
                                        create_default_2d_soil_material: SoilMaterial):
        """
        Tests if gravity loading and zero water pressure is added correctly when using post setup. Gravity load and zero
        water pressure should be present on all nodes of the model.

        Args:
            - expected_geometry_single_layer_2D (Tuple[:class:`stem.geometry.Geometry`, \
              :class:`stem.geometry.Geometry`, :class:`stem.geometry.Geometry`]): expected geometry of the model
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 2

        layer1_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        layer2_coordinates = [(1, 1, 0), (0, 1, 0), (0, 2, 0), (1, 2, 0)]

        # define soil materials
        soil_material1 = create_default_2d_soil_material
        soil_material1.name = "soil1"

        soil_material2 = create_default_2d_soil_material
        soil_material2.name = "soil2"

        # create model
        model = Model(ndim)

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")
        model.add_soil_layer_by_coordinates(layer2_coordinates, soil_material2, "layer2")

        # set up gravity loading
        project_parameters = TestUtils.create_default_solver_settings()
        model.project_parameters = project_parameters
        model.project_parameters.settings.stress_initialisation_type = StressInitialisationType.GRAVITY_LOADING

        # add gravity through post setup
        model.post_setup()

        # get water and gravity model parts
        water_pressure_model_part = model.process_model_parts[0]
        gravity_model_part = model.process_model_parts[1]

        # assert if the water and gravity model part are the same as the expected model parts
        expected_water_pressure_geometry = expected_geometry_two_layers_2D[-1]
        expected_gravity_geometry = expected_geometry_two_layers_2D[-1]

        TestUtils.assert_almost_equal_geometries(expected_water_pressure_geometry, water_pressure_model_part.geometry)
        TestUtils.assert_almost_equal_geometries(expected_gravity_geometry, gravity_model_part.geometry)

        assert water_pressure_model_part.name == "zero_water_pressure"
        assert gravity_model_part.name == "gravity_load_2d"

        assert pytest.approx(water_pressure_model_part.parameters.water_pressure) == 0
        assert water_pressure_model_part.parameters.is_fixed

        npt.assert_allclose([0, -9.81, 0], gravity_model_part.parameters.value)
        npt.assert_allclose([True, True, True], gravity_model_part.parameters.active)

    def test_post_setup_with_water_pressure_3D(self, expected_geometry_two_layers_3D_extruded: Tuple[Geometry,
                                                                                                     Geometry],
                                               create_default_3d_soil_material: SoilMaterial):
        """
        Tests if gravity loading is not applied and zero water pressure is not added when using post setup. Water pressure
        should only be present on layer 1.

        Args:
            - expected_geometry_two_layers_3D_extruded (Tuple[:class:`stem.geometry.Geometry`, \
                :class:`stem.geometry.Geometry`]): expected geometry of the model
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 3

        layer1_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        layer2_coordinates = [(1, 1, 0), (0, 1, 0), (0, 2, 0), (1, 2, 0)]

        # define soil materials
        soil_material1 = create_default_3d_soil_material
        soil_material1.name = "soil1"

        soil_material2 = create_default_3d_soil_material
        soil_material2.name = "soil2"

        # create model
        model = Model(ndim)
        model.extrusion_length = 1

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")
        model.add_soil_layer_by_coordinates(layer2_coordinates, soil_material2, "layer2")

        # manually add water pressure model part
        water_pressure_model_part = ModelPart("water_pressure_part")
        water_pressure_model_part.geometry = model.body_model_parts[0].geometry
        water_pressure_model_part.parameters = UniformWaterPressure(water_pressure=100)
        model.process_model_parts.append(water_pressure_model_part)

        model.gmsh_io.add_physical_group(
            "water_pressure_part", 3, geometry_ids=model.gmsh_io.geo_data["physical_groups"]["layer1"]["geometry_ids"])

        # add project parameters
        project_parameters = TestUtils.create_default_solver_settings()

        model.project_parameters = project_parameters
        model.post_setup()

        # only 1 process model part should be present which is the water pressure on layer 1
        assert len(model.process_model_parts) == 1

        # check if the water pressure model part is the same as the expected model part
        assert model.process_model_parts[0].name == "water_pressure_part"
        assert pytest.approx(model.process_model_parts[0].parameters.water_pressure) == 100
        assert model.process_model_parts[0].parameters.is_fixed

        # check if the water pressure is only applied to the nodes of layer 1
        expected_water_pressure_geometry = expected_geometry_two_layers_3D_extruded[0]
        TestUtils.assert_almost_equal_geometries(expected_water_pressure_geometry,
                                                 model.process_model_parts[0].geometry)

    def test_post_setup_only_structural_material(self):
        """
        Test if the post setup is done correctly when only structural materials are present. Gravity loading should
        be applied, but no water pressures

        """

        ndim = 2

        model = Model(ndim)

        # Specify beam material model
        beam_material = EulerBeam(ndim, 210e9, 0.3, 7850, 0.01, 0.0001)
        name = "beam"
        structural_material = StructuralMaterial(name, beam_material)
        # Specify the coordinates for the beam: x:1m x y:0m
        beam_coordinates = [(0, 0, 0), (1, 0, 0)]
        # Create the beam
        gmsh_input = {name: {"coordinates": beam_coordinates, "ndim": 1}}
        # check if extrusion length is specified in 3D
        model.gmsh_io.generate_geometry(gmsh_input, "")
        #
        # create body model part
        body_model_part = BodyModelPart(name)
        body_model_part.material = structural_material

        # set the geometry of the body model part
        body_model_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, name)
        model.body_model_parts.append(body_model_part)

        # add project parameters and set up gravity loading
        project_parameters = TestUtils.create_default_solver_settings()
        project_parameters.settings.stress_initialisation_type = StressInitialisationType.GRAVITY_LOADING

        model.project_parameters = project_parameters

        # add gravity through post setup, do not add water pressure through post setup
        model.post_setup()

        # set expected geometry
        expected_geometry = Geometry()
        expected_geometry.points = {1: Point.create([0, 0, 0], 1), 2: Point.create([1, 0, 0], 2)}
        expected_geometry.lines = {1: Line.create([1, 2], 1)}

        # check if only gravity process model part is present, no water pressure should be present
        assert len(model.process_model_parts) == 1
        assert model.process_model_parts[0].name == "gravity_load_1d"
        assert model.process_model_parts[0].parameters.value == [0, -9.81, 0]
        assert model.process_model_parts[0].parameters.active == [True, True, True]

        # check if the geometry of the process model part is correct
        TestUtils.assert_almost_equal_geometries(expected_geometry, model.process_model_parts[0].geometry)

    def test_generate_straight_track_2d(self):
        """
        Test if a straight track is generated correctly in a 2d space. A straight track is generated and added to the
        model. The geometry and material of the rails, sleepers and rail pads are checked.
        """

        # initialise model
        model = Model(2)

        rail_parameters = EulerBeam(2, 1, 1, 1, 1, 1)
        rail_pad_parameters = ElasticSpringDamper([1, 1, 1], [1, 1, 1], [1, 1, 1], [1, 1, 1])
        sleeper_parameters = NodalConcentrated([1, 1, 1], 1, [1, 1, 1])

        origin_point = np.array([2.0, 3.0, 0])
        direction_vector = np.array([1, 0, 0])

        # create a straight track with rails, sleepers and rail pads
        model.generate_straight_track(0.6, 3, rail_parameters, sleeper_parameters, rail_pad_parameters, 0.02,
                                      origin_point, direction_vector, "track_1")

        # check geometry and material of the rail
        expected_rail_points = {
            4: Point.create([2.0, 3.02, 0], 4),
            5: Point.create([2.6, 3.02, 0], 5),
            6: Point.create([3.2, 3.02, 0], 6)
        }
        expected_rail_lines = {3: Line.create([4, 5], 3), 4: Line.create([5, 6], 4)}

        expected_rail_geometry = Geometry(expected_rail_points, expected_rail_lines)

        # check rail model part
        rail_model_part = model.body_model_parts[0]
        calculated_rail_geometry = rail_model_part.geometry
        calculated_rail_parameters = rail_model_part.material.material_parameters

        TestUtils.assert_almost_equal_geometries(expected_rail_geometry, calculated_rail_geometry)
        TestUtils.assert_dictionary_almost_equal(rail_parameters.__dict__, calculated_rail_parameters.__dict__)

        # check geometry and material of the sleepers
        expected_sleeper_points = {
            1: Point.create([2.0, 3.0, 0], 1),
            2: Point.create([2.6, 3.0, 0], 2),
            3: Point.create([3.2, 3.0, 0], 3)
        }
        expected_sleeper_geometry = Geometry(expected_sleeper_points)

        sleeper_model_part = model.body_model_parts[1]
        calculated_sleeper_geometry = sleeper_model_part.geometry
        calculated_sleeper_parameters = sleeper_model_part.material.material_parameters

        TestUtils.assert_almost_equal_geometries(expected_sleeper_geometry, calculated_sleeper_geometry)
        TestUtils.assert_dictionary_almost_equal(sleeper_parameters.__dict__, calculated_sleeper_parameters.__dict__)

        # check geometry and material of the rail pads
        rail_pad_model_part = model.body_model_parts[2]
        calculated_rail_pad_geometry = rail_pad_model_part.geometry
        calculated_rail_pad_parameters = rail_pad_model_part.material.material_parameters

        expected_rail_pad_points = {
            4: Point.create([2.0, 3.02, 0], 4),
            1: Point.create([2.0, 3.0, 0], 1),
            5: Point.create([2.6, 3.02, 0], 5),
            2: Point.create([2.6, 3.0, 0], 2),
            6: Point.create([3.2, 3.02, 0], 6),
            3: Point.create([3.2, 3.0, 0], 3)
        }

        expected_rail_pad_lines = {5: Line.create([4, 1], 5), 6: Line.create([5, 2], 6), 7: Line.create([6, 3], 7)}

        expected_rail_pad_geometry = Geometry(expected_rail_pad_points, expected_rail_pad_lines)

        TestUtils.assert_almost_equal_geometries(expected_rail_pad_geometry, calculated_rail_pad_geometry)
        TestUtils.assert_dictionary_almost_equal(rail_pad_parameters.__dict__, calculated_rail_pad_parameters.__dict__)

    def test_generate_straight_track_3d(self):
        """
        Tests if a straight track is generated correctly in a 3d space. A straight track is generated and added to the
        model. The geometry and material of the rails, sleepers and rail pads are checked.
        """

        model = Model(3)

        rail_parameters = EulerBeam(3, 1, 1, 1, 1, 1, 1, 1)
        rail_pad_parameters = ElasticSpringDamper([1, 1, 1], [1, 1, 1], [1, 1, 1], [1, 1, 1])
        sleeper_parameters = NodalConcentrated([1, 1, 1], 1, [1, 1, 1])

        origin_point = np.array([2.0, 3.0, 1.0])
        direction_vector = np.array([1, 1, -1])

        # create a straight track with rails, sleepers and rail pads
        model.generate_straight_track(0.6, 3, rail_parameters, sleeper_parameters, rail_pad_parameters, 0.02,
                                      origin_point, direction_vector, "track_1")

        distance_sleepers_xyz = 0.6 / 3**0.5

        # check geometry and material of the rail
        expected_rail_points = {
            4: Point.create([2.0, 3.02, 1.0], 4),
            5: Point.create([2.0 + distance_sleepers_xyz, 3.02 + distance_sleepers_xyz, 1.0 - distance_sleepers_xyz],
                            5),
            6: Point.create(
                [2.0 + 2 * distance_sleepers_xyz, 3.02 + 2 * distance_sleepers_xyz, 1.0 - 2 * distance_sleepers_xyz], 6)
        }
        expected_rail_lines = {3: Line.create([4, 5], 3), 4: Line.create([5, 6], 4)}

        expected_rail_geometry = Geometry(expected_rail_points, expected_rail_lines)

        # check rail model part
        rail_model_part = model.body_model_parts[0]
        calculated_rail_geometry = rail_model_part.geometry
        calculated_rail_parameters = rail_model_part.material.material_parameters

        TestUtils.assert_almost_equal_geometries(expected_rail_geometry, calculated_rail_geometry)
        TestUtils.assert_dictionary_almost_equal(rail_parameters.__dict__, calculated_rail_parameters.__dict__)

        # check geometry and material of the sleepers
        expected_sleeper_points = {
            1: Point.create([2.0, 3.0, 1.0], 1),
            2: Point.create([2.0 + distance_sleepers_xyz, 3.0 + distance_sleepers_xyz, 1.0 - distance_sleepers_xyz], 2),
            3: Point.create(
                [2.0 + 2 * distance_sleepers_xyz, 3.0 + 2 * distance_sleepers_xyz, 1.0 - 2 * distance_sleepers_xyz], 3)
        }

        expected_sleeper_geometry = Geometry(expected_sleeper_points)

        sleeper_model_part = model.body_model_parts[1]
        calculated_sleeper_geometry = sleeper_model_part.geometry
        calculated_sleeper_parameters = sleeper_model_part.material.material_parameters

        TestUtils.assert_almost_equal_geometries(expected_sleeper_geometry, calculated_sleeper_geometry)
        TestUtils.assert_dictionary_almost_equal(sleeper_parameters.__dict__, calculated_sleeper_parameters.__dict__)

        # check geometry and material of the rail pads
        rail_pad_model_part = model.body_model_parts[2]
        calculated_rail_pad_geometry = rail_pad_model_part.geometry
        calculated_rail_pad_parameters = rail_pad_model_part.material.material_parameters

        expected_rail_pad_points = {
            4: Point.create([2.0, 3.02, 1.0], 4),
            1: Point.create([2.0, 3.0, 1.0], 1),
            5: Point.create([2.0 + distance_sleepers_xyz, 3.02 + distance_sleepers_xyz, 1.0 - distance_sleepers_xyz],
                            5),
            2: Point.create([2.0 + distance_sleepers_xyz, 3.0 + distance_sleepers_xyz, 1.0 - distance_sleepers_xyz], 2),
            6: Point.create(
                [2.0 + 2 * distance_sleepers_xyz, 3.02 + 2 * distance_sleepers_xyz, 1.0 - 2 * distance_sleepers_xyz],
                6),
            3: Point.create(
                [2.0 + 2 * distance_sleepers_xyz, 3.0 + 2 * distance_sleepers_xyz, 1.0 - 2 * distance_sleepers_xyz], 3)
        }

        expected_rail_pad_lines = {5: Line.create([4, 1], 5), 6: Line.create([5, 2], 6), 7: Line.create([6, 3], 7)}

        expected_rail_pad_geometry = Geometry(expected_rail_pad_points, expected_rail_pad_lines)

        TestUtils.assert_almost_equal_geometries(expected_rail_pad_geometry, calculated_rail_pad_geometry)
        TestUtils.assert_dictionary_almost_equal(rail_pad_parameters.__dict__, calculated_rail_pad_parameters.__dict__)

        # check rotation constrain model part
        rotation_constrain_model_part = model.process_model_parts[1]
        calculated_rotation_constrain_geometry = rotation_constrain_model_part.geometry
        calculated_rotation_constrain_parameters = rotation_constrain_model_part.parameters

        expected_rotation_constrain_points = {4: Point.create([2.0, 3.02, 1.0], 4)}
        expected_rotation_constrain_geometry = Geometry(expected_rotation_constrain_points)

        expected_rotation_constraint_parameters = RotationConstraint(value=[0, 0, 0],
                                                                     is_fixed=[True, True, True],
                                                                     active=[True, True, True])

        TestUtils.assert_almost_equal_geometries(expected_rotation_constrain_geometry,
                                                 calculated_rotation_constrain_geometry)
        TestUtils.assert_dictionary_almost_equal(expected_rotation_constraint_parameters.__dict__,
                                                 calculated_rotation_constrain_parameters.__dict__)

    def test_set_element_size_of_group(self, create_default_2d_soil_material: SoilMaterial):
        """
        Tests setting the element size of a group. A model is created with a soil layer. The element size of the
        process model part is set to half the size of the rest of the mesh. The mesh is visually checked in Gmsh if it
        is as expected. In this test the only performed checks are the amount of elements and nodes per model part.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """
        model = Model(2)

        # add soil material
        soil_material = create_default_2d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")

        # add process geometry
        gmsh_process_input = {"process_1d": {"coordinates": [[0, 1, 0], [1, 1, 0]], "ndim": 1}}
        model.gmsh_io.generate_geometry(gmsh_process_input, "")

        # create process model part
        process_model_part = ModelPart("process_1d")

        # set the geometry of the process model part
        process_model_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, "process_1d")

        # add process geometry
        gmsh_process_input = {"process_1d_2": {"coordinates": [[0, 1, 0], [1, 1, 0]], "ndim": 1}}
        model.gmsh_io.generate_geometry(gmsh_process_input, "")

        # create process model part
        process_model_part2 = ModelPart("process_1d_2")

        # set the geometry of the process model part
        process_model_part2.get_geometry_from_geo_data(model.gmsh_io.geo_data, "process_1d_2")

        # add process model parts
        model.process_model_parts.append(process_model_part)
        model.process_model_parts.append(process_model_part2)

        # synchronise geometry and generate mesh
        model.synchronise_geometry()

        # set base mesh size
        model.set_mesh_size(1)

        # set element size of process model parts
        model.set_element_size_of_group(0.5, "process_1d")
        model.set_element_size_of_group(1, "process_1d_2")
        model.generate_mesh(open_gmsh_gui=False)

        # check mesh of body model part
        mesh_body = model.body_model_parts[0].mesh
        assert len(mesh_body.elements) == 13
        assert len(mesh_body.nodes) == 11

        # check process model parts, both process model parts should have the same amount of elements and nodes
        mesh_process = model.process_model_parts[0].mesh
        assert len(mesh_process.elements) == 2
        assert len(mesh_process.nodes) == 3

        mesh_process = model.process_model_parts[1].mesh
        assert len(mesh_process.elements) == 2
        assert len(mesh_process.nodes) == 3

    def test_set_element_size_of_group_expected_raise(self):
        """
        Tests if a ValueError is raised when the group name is not found while setting the element size of a group

        """

        model = Model(2)

        with pytest.raises(ValueError, match=f"Group name `non_existing_group` not found."):
            model.set_element_size_of_group(1, "non_existing_group")

    def test_split_body_model_part_3D(self, expected_geometry_two_layers_3D_extruded: Tuple[Geometry, Geometry],
                                      create_default_3d_soil_material: SoilMaterial):
        """
        Test if a body model part is split correctly in a 3D case. A model is created with two soil layers. The model is
        split in two parts and the geometry of the split model parts is checked.

        Args:
            - expected_geometry_two_layers_3D_extruded (Tuple[:class:`stem.geometry.Geometry`, \
                :class:`stem.geometry.Geometry`]): expected geometry of the model
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 3

        layer1_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        layer2_coordinates = [(1, 1, 0), (0, 1, 0), (0, 2, 0), (1, 2, 0)]

        # define soil materials
        soil_material1 = create_default_3d_soil_material
        soil_material1.name = "soil1"

        # create model
        model = Model(ndim)
        model.extrusion_length = 1

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")
        model.add_soil_layer_by_coordinates(layer2_coordinates, soil_material1, "layer2")

        model.gmsh_io.geo_data["physical_groups"]["layer1"]["geometry_ids"] = [1, 2]
        model.gmsh_io.geo_data["physical_groups"].pop("layer2")

        model.body_model_parts[0].geometry = Geometry.create_geometry_from_gmsh_group(model.gmsh_io.geo_data, "layer1")
        model.body_model_parts.pop(1)

        model.split_model_part("layer1", "split_layer1", [1], soil_material1)

        # check geometry of the split model part
        TestUtils.assert_almost_equal_geometries(expected_geometry_two_layers_3D_extruded[0],
                                                 model.body_model_parts[1].geometry)
        TestUtils.assert_almost_equal_geometries(expected_geometry_two_layers_3D_extruded[1],
                                                 model.body_model_parts[0].geometry)

        # split body model part expected error
        with pytest.raises(ValueError,
                           match="New parameters must have the same material type as in the original "
                           "body model part."):
            model.split_model_part("layer1", "second_split", [2],
                                   StructuralMaterial("beam", EulerBeam(3, 1, 1, 1, 1, 1, 1, 1)))

    def test_split_process_model_part_2D(self, expected_geometry_two_layers_2D: Tuple[Geometry, Geometry, Geometry],
                                         create_default_2d_soil_material: SoilMaterial):
        """
        Test if a process model part is split correctly in a 2D case. A model is created with two soil layers. The model
        is split in two parts and the geometry of the split model parts is checked.

        Args:
            - expected_geometry_two_layers_2D (Tuple[:class:`stem.geometry.Geometry`, \
                :class:`stem.geometry.Geometry`, :class:`stem.geometry.Geometry`]): expected geometry of the model
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        ndim = 2

        layer1_coordinates = [(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)]
        layer2_coordinates = [(1, 1, 0), (0, 1, 0), (0, 2, 0), (1, 2, 0)]

        # define soil materials
        soil_material1 = create_default_2d_soil_material
        soil_material1.name = "soil1"

        # create model
        model = Model(ndim)

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")
        model.add_soil_layer_by_coordinates(layer2_coordinates, soil_material1, "layer2")

        # add process model part on the whole geometry
        model._Model__add_gravity_load()

        # split process model part
        model.split_model_part("gravity_load_2d", "gravity_layer_2", [2],
                               GravityLoad(value=[20, 20, 20], active=[True, True, True]))

        # check geometry and parameters of the original and split model part
        assert model.process_model_parts[0].name == "gravity_load_2d"
        assert model.process_model_parts[0].parameters.value == [0, -9.81, 0]
        TestUtils.assert_almost_equal_geometries(expected_geometry_two_layers_2D[0],
                                                 model.process_model_parts[0].geometry)

        assert model.process_model_parts[1].name == "gravity_layer_2"
        assert model.process_model_parts[1].parameters.value == [20, 20, 20]
        TestUtils.assert_almost_equal_geometries(expected_geometry_two_layers_2D[1],
                                                 model.process_model_parts[1].geometry)

        # split process model part expected error
        with pytest.raises(ValueError,
                           match="New parameters must have the same process parameter type as in the "
                           "original process model part."):
            model.split_model_part("gravity_load_2d", "gravity_layer_3", [1],
                                   LineLoad(value=[20, 20, 20], active=[True, True, True]))

    def test_split_model_part_expected_raise(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if a ValueErrors are raised when:
        - the group name is not found while splitting a model part |
        - the geometry is not defined in the model part |
        - the model part type and new parameters type do not match

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        model = Model(2)

        model.process_model_parts.append(ModelPart("process_2d"))

        with pytest.raises(ValueError, match=f"Model part: non_existing_group not found."):
            model.split_model_part("non_existing_group", "split_group", [1], create_default_2d_soil_material)

        with pytest.raises(ValueError, match=f"Geometry is not defined in the model part: process_2d."):
            model.split_model_part("process_2d", "split_group", [1], create_default_2d_soil_material)

        model.process_model_parts[0].geometry = Geometry(surfaces={1: Surface.create([1, 2, 3], 1)},
                                                         lines={
                                                             1: Line.create([1, 2], 1),
                                                             2: Line.create([2, 3], 2),
                                                             3: Line.create([3, 1], 3)
                                                         },
                                                         points={
                                                             1: Point.create([0, 0, 0], 1),
                                                             2: Point.create([1, 0, 0], 2),
                                                             3: Point.create([1, 1, 0], 3)
                                                         })

        with pytest.raises(ValueError, match=f"Model part type and new parameters type must match."):
            model.split_model_part("process_2d", "split_group", [1], create_default_2d_soil_material)

    def test_finalise_json_output_raises_errors(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test that finalisation raises error correctly.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): A default soil material.

        """

        # define layer coordinates
        ndim = 2
        layer1_coordinates = [(0, 0, 0), (4, 0, 0), (4, 1, 0), (0, 1, 0)]

        # define soil materials
        soil_material1 = create_default_2d_soil_material
        soil_material1.name = "soil1"

        # create model
        model = Model(ndim)

        # add soil layers
        model.add_soil_layer_by_coordinates(layer1_coordinates, soil_material1, "layer1")

        # synchronise geometry and recalculates the ids
        model.synchronise_geometry()
        # Define nodal results
        nodal_results = [NodalOutput.ACCELERATION]
        # Define output coordinates
        output_coordinates = [(1.5, 1, 0), (1.5, 0.5, 0), (2.5, 0.5, 0), (2.5, 0, 0)]

        # add output settings
        model.add_output_settings_by_coordinates(output_coordinates,
                                                 part_name="nodal_accelerations",
                                                 output_name="json_nodal_accelerations",
                                                 output_dir="dir_test",
                                                 output_parameters=JsonOutputParameters(output_interval=100,
                                                                                        nodal_results=nodal_results))
        model.synchronise_geometry()
        model.generate_mesh()

        # set output name of json output to None
        model_copy = deepcopy(model)
        model_copy.output_settings[-1].output_name = None
        msg = "No name is specified for the json file."
        with pytest.raises(ValueError, match=msg):
            model_copy.finalise(input_folder="input_files")

        # set json filename to a wrong one
        model_copy = deepcopy(model)
        model_copy.output_settings[-1].output_name = "json_nodal_displacements"

        expected_path = Path(
            'input_dir') / model_copy.output_settings[-1].output_dir / model_copy.output_settings[-1].output_name
        expected_path = str(expected_path.with_suffix('.json'))
        msg = (f"No JSON file is found in the output directory for path: {expected_path}. "
               "Either the working folder is incorrectly specified or no simulation has been performed yet.")
        with pytest.raises(OSError, match=re.escape(msg)):
            model_copy.finalise(input_folder="input_dir")

        # set part name of the output settings to None
        model_copy = deepcopy(model)
        model_copy.output_settings[-1].part_name = None
        msg = "The output model part has no part name specified."
        with pytest.raises(ValueError, match=msg):
            model_copy.finalise(input_folder="input_files")

        # set part name of the output settings to non-existing part
        model_copy = deepcopy(model)
        model_copy.output_settings[-1].part_name = "part 404"
        msg = "No model part matches the part name specified in the output settings."
        with pytest.raises(ValueError, match=msg):
            model_copy.finalise(input_folder="input_files")

        # set part name of the output settings to non-existing part
        model_copy = deepcopy(model)
        model_copy.process_model_parts[-1].mesh = None
        msg = "process model part has not been meshed yet!"
        with pytest.raises(ValueError, match=msg):
            model_copy.finalise(input_folder="input_files")

    def test_add_boundary_condition_on_plane(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if a boundary condition is added correctly on a plane. A model is created with two soil layers. A boundary
        condition is added on a plane and the nodes of the boundary condition are checked.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """

        model = Model(3)
        model.extrusion_length = 1

        # add soil material
        soil_material = create_default_3d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")
        model.add_soil_layer_by_coordinates([(1, 1, 0), (0, 1, 0), (0, 2, 0), (1, 2, 0)], soil_material, "layer2")

        model.synchronise_geometry()

        no_displacement_boundary = DisplacementConstraint(active=[True, True, True],
                                                          is_fixed=[True, True, True],
                                                          value=[0, 0, 0])

        plane_coordinates = [(0, 0, 0), (0, 0, 1), (0, 1, 1)]
        model.add_boundary_condition_on_plane(plane_coordinates, no_displacement_boundary, "left_side_boundary")

        # check if the boundary condition is added to the model
        assert len(model.process_model_parts) == 1

        # check if the boundary condition is added to the correct model part
        assert model.process_model_parts[0].name == "left_side_boundary"
        assert model.process_model_parts[0].parameters == no_displacement_boundary

        # check if the boundary condition is added to the correct nodes
        expected_points = {
            7: Point.create([0, 1, 0], 7),
            8: Point.create([0, 1, 1], 8),
            2: Point.create([0, 0, 1], 2),
            1: Point.create([0, 0, 0], 1),
            11: Point.create([0, 2, 0], 11),
            12: Point.create([0, 2, 1], 12),
        }

        # get the generated points
        generated_points = model.process_model_parts[0].geometry.points

        # check if points are the same, these should be the all the points on the plane, i.e. points which are part
        # of layer 1 and layer 2
        for (generated_point_id, generated_point), (expected_point_id, expected_point) in \
                zip(generated_points.items(), expected_points.items()):
            assert generated_point_id == expected_point_id
            assert generated_point.id == expected_point.id
            npt.assert_allclose(generated_point.coordinates, expected_point.coordinates)

        # add the same boundary condition again, this should not raise an error
        model.add_boundary_condition_on_plane(plane_coordinates, no_displacement_boundary, "left_side_boundary")

        # check if the boundary condition is added to the model
        assert len(model.process_model_parts) == 1

        # check if the boundary condition is added to the correct model part
        assert model.process_model_parts[0].name == "left_side_boundary"
        assert model.process_model_parts[0].parameters == no_displacement_boundary

        # get the generated points
        generated_points = model.process_model_parts[0].geometry.points

        # check if points are the same, these should be the all the points on the plane, i.e. points which are part
        # of layer 1 and layer 2
        for (generated_point_id, generated_point), (expected_point_id, expected_point) in \
                zip(generated_points.items(), expected_points.items()):
            assert generated_point_id == expected_point_id
            assert generated_point.id == expected_point.id
            npt.assert_allclose(generated_point.coordinates, expected_point.coordinates)

        # expect it raises an error when the less than 3 points are given to define a plane
        with pytest.raises(ValueError, match="At least 3 vertices are required to define a plane."):
            model.add_boundary_condition_on_plane([(0, 0, 0), (0, 0, 1)], no_displacement_boundary, "wrong_plane")

    def test_add_boundary_condition_on_polygon(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if a boundary condition is added correctly on a polygon. A model is created with two soil layers. A boundary
        condition is added on a polygon and the nodes of the boundary condition are checked.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material
        """

        model = Model(3)
        model.extrusion_length = 1

        # add soil material
        soil_material = create_default_3d_soil_material

        # add soil layers
        model.add_soil_layer_by_coordinates([(0, 0, 0), (1, 0, 0), (1, 1, 0), (0, 1, 0)], soil_material, "layer1")
        model.add_soil_layer_by_coordinates([(1, 1, 0), (0, 1, 0), (0, 2, 0), (1, 2, 0)], soil_material, "layer2")

        model.synchronise_geometry()

        no_displacement_boundary = DisplacementConstraint(active=[True, True, True],
                                                          is_fixed=[True, True, True],
                                                          value=[0, 0, 0])

        polygon_coordinates = [(0, 0, 0), (0, 0, 1), (0, 1, 1), (0, 1, 0)]
        model.add_boundary_condition_on_polygon(polygon_coordinates, no_displacement_boundary, "left_bottom_boundary")

        # check if the boundary condition is added to the model
        assert len(model.process_model_parts) == 1

        # check if the boundary condition is added to the correct model part
        assert model.process_model_parts[0].name == "left_bottom_boundary"
        assert model.process_model_parts[0].parameters == no_displacement_boundary

        # check if the boundary condition is added to the correct nodes
        expected_points = {
            7: Point.create([0, 1, 0], 7),
            8: Point.create([0, 1, 1], 8),
            2: Point.create([0, 0, 1], 2),
            1: Point.create([0, 0, 0], 1),
        }

        # Get the generated points
        generated_points = model.process_model_parts[0].geometry.points

        # check if points are the same, these should be only the points which are part of layer 1
        for (generated_point_id, generated_point), (expected_point_id, expected_point) in \
                zip(generated_points.items(), expected_points.items()):
            assert generated_point_id == expected_point_id
            assert generated_point.id == expected_point.id
            npt.assert_allclose(generated_point.coordinates, expected_point.coordinates)

    def test_get_bounding_box_soil(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if the bounding box of all soil volumes is correctly calculated. A model is created with a soil layer and
        the bounding box of the soil volume is checked.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material
        """
        ndim = 3

        layer_coordinates = [(0, 0, 0), (1, 0, 0), (1, 2, 0), (0, 2, 0)]

        # define soil material
        soil_material = create_default_3d_soil_material

        # create model
        model = Model(ndim)
        model.extrusion_length = 4

        model.project_parameters = TestUtils.create_default_solver_settings()

        # add soil layer
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")

        # check if layer is added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "soil1"
        assert model.body_model_parts[0].material == soil_material

        # run the tests
        min_coords, max_coords = model.get_bounding_box_soil()

        assert min_coords[0] == 0
        assert min_coords[1] == 0
        assert min_coords[2] == 0
        assert max_coords[0] == 1
        assert max_coords[1] == 2
        assert max_coords[2] == 4

    def test_get_bounding_box_soil_error(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if an error is raised when the model part has no geometry. A model is created with a soil layer and the
        bounding box of the soil volume is checked. The geometry of the model part is set to None and an error is
        expected.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material
        """
        ndim = 3

        layer_coordinates = [(0, 0, 0), (1, 0, 0), (1, 2, 3), (0, 2, 3)]

        # define soil material
        soil_material = create_default_3d_soil_material

        # create model
        model = Model(ndim)
        model.extrusion_length = 1

        model.project_parameters = TestUtils.create_default_solver_settings()

        # add soil layer
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")

        # check if layer is added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "soil1"
        assert model.body_model_parts[0].material == soil_material
        model.body_model_parts[0].geometry = None

        with pytest.raises(ValueError, match="Model part has no geometry."):
            model.get_bounding_box_soil()

    def test_get_points_outside_soil_volume(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if the points outside the soil volume are correctly found. A model is created with a soil layer and
        points outside the soil volume are checked. Some points are outside the soil volume and some are inside.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material

        """
        ndim = 3

        layer_coordinates = [(0, 0, 0), (1, 0, 0), (1, 2, 0), (0, 2, 0)]

        # define soil material
        soil_material = create_default_3d_soil_material

        # create model
        model = Model(ndim)
        model.extrusion_length = 4

        model.project_parameters = TestUtils.create_default_solver_settings()

        # add soil layer
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")

        # check if layer is added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "soil1"
        assert model.body_model_parts[0].material == soil_material

        points_outside_test = [(1, 0, -1), (1, 0, 0), (0, 0, 0), (1, 0, 0), (1, 0, 5), (1, 0, 6)]
        outside_name = f"points_outside"
        points_outside_settings = {outside_name: {"coordinates": points_outside_test, "ndim": 0}}
        model.gmsh_io.generate_geometry(points_outside_settings, "")
        points_outside_part = BodyModelPart(outside_name)
        points_outside_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, outside_name)
        rail_pad_parameters = ElasticSpringDamper(NODAL_DISPLACEMENT_STIFFNESS=[1, 750e6, 1],
                                                  NODAL_ROTATIONAL_STIFFNESS=[0, 0, 0],
                                                  NODAL_DAMPING_COEFFICIENT=[1, 750e3, 1],
                                                  NODAL_ROTATIONAL_DAMPING_COEFFICIENT=[0, 0, 0])
        points_outside_part.material = StructuralMaterial(name=outside_name, material_parameters=rail_pad_parameters)
        model.body_model_parts.append(points_outside_part)
        # run the test
        points_outside_volume = model.get_points_outside_soil(outside_name)
        # check if the points are outside the soil volume
        assert len(points_outside_volume) == 3
        assert points_outside_volume[0].coordinates == [1., 0., -1.]
        assert points_outside_volume[1].coordinates == [1., 0., 5.]
        assert points_outside_volume[2].coordinates == [1., 0., 6.]

    def test_get_points_outside_soil_volume_error(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if an error is raised when the model part is not found. A model is created with a soil layer and points
        outside the soil volume are checked. The model part is not found and an error is expected.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material
        """
        ndim = 3

        layer_coordinates = [(0, 0, 0), (1, 0, 0), (1, 2, 0), (0, 2, 0)]

        # define soil material
        soil_material = create_default_3d_soil_material

        # create model
        model = Model(ndim)
        model.extrusion_length = 4

        model.project_parameters = TestUtils.create_default_solver_settings()

        # add soil layer
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")

        # check if layer is added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "soil1"
        assert model.body_model_parts[0].material == soil_material

        points_outside_test = [(1, 0, -1), (1, 0, 0), (0, 0, 0), (1, 0, 0), (1, 0, 5), (1, 0, 6)]
        outside_name = f"fake_name"
        with pytest.raises(ValueError, match="Model part fake_name not found."):
            model.get_points_outside_soil(outside_name)

    def test_get_points_outside_soil_volume_error_geometry(self, create_default_3d_soil_material: SoilMaterial):
        """
        Test if an error is raised when the model part has no geometry. A model is created with a soil layer and points
        outside the soil volume are checked. The geometry of the model part is set to None and an error is expected.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material
        """
        ndim = 3

        # create model
        model = Model(ndim)
        model.extrusion_length = 1

        model.project_parameters = TestUtils.create_default_solver_settings()

        beam = StructuralMaterial(name="soil1", material_parameters=EulerBeam(2, 1, 1, 1, 1, 1))
        beam_part = BodyModelPart("soil1")
        beam_part.material = beam
        model.body_model_parts.append(beam_part)

        # check if layer is added correctly
        assert len(model.body_model_parts) == 1
        assert model.body_model_parts[0].name == "soil1"

        outside_name = f"soil1"
        with pytest.raises(ValueError, match="Model part soil1 has no geometry."):
            model.get_points_outside_soil(outside_name)

    def test_generate_extended_straight_track_2d(self, create_default_2d_soil_material: SoilMaterial):
        """
        Test if a straight track is generated correctly in a 2d space. A straight track is generated and added to the
        model. The geometry and material of the rails, sleepers and rail pads are checked.

        Args:
            - create_default_2d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material
        """

        # initialise model
        model = Model(2)
        # add soil material 2d
        soil_material = create_default_2d_soil_material
        layer_coordinates = [(2, 3, 0), (4, 3, 0), (4, 1, 0), (2, 1, 0)]
        # add soil layer
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")

        rail_parameters = EulerBeam(2, 1, 1, 1, 1, 1)
        extended_soil_parameters = ElasticSpringDamper([1, 1, 1], [1, 1, 1], [1, 1, 1], [1, 1, 1])
        rail_pad_parameters = ElasticSpringDamper([1, 1, 1], [1, 1, 1], [1, 1, 1], [1, 1, 1])
        sleeper_parameters = NodalConcentrated([1, 1, 1], 1, [1, 1, 1])

        origin_point = np.array([1.0, 3.0, -1])
        direction_vector = np.array([1, 0, 0])

        # create a straight track with rails, sleepers and rail pads
        model.generate_extended_straight_track(0.6, 8, rail_parameters, sleeper_parameters, rail_pad_parameters, 0.02,
                                               origin_point, extended_soil_parameters, 5, direction_vector, "track_1")

        # check geometry and material of the rail
        expected_rail_points = {
            13: Point.create([1.0, 3.02, -1.0], 13),
            14: Point.create([1.6, 3.02, -1.0], 14),
            15: Point.create([2.2, 3.02, -1.0], 15),
            16: Point.create([2.8, 3.02, -1.0], 16),
            17: Point.create([3.4, 3.02, -1.0], 17),
            18: Point.create([4.0, 3.02, -1.0], 18),
            19: Point.create([4.6, 3.02, -1.0], 19),
            20: Point.create([5.2, 3.02, -1.0], 20)
        }
        expected_rail_lines = {
            12: Line.create([13, 14], 12),
            13: Line.create([14, 15], 13),
            14: Line.create([15, 16], 14),
            15: Line.create([16, 17], 15),
            16: Line.create([17, 18], 16),
            17: Line.create([18, 19], 17),
            18: Line.create([19, 20], 18),
        }

        expected_rail_geometry = Geometry(expected_rail_points, expected_rail_lines)

        # check rail model part
        rail_model_part = model.body_model_parts[1]
        calculated_rail_geometry = rail_model_part.geometry
        calculated_rail_parameters = rail_model_part.material.material_parameters

        TestUtils.assert_almost_equal_geometries(expected_rail_geometry, calculated_rail_geometry)
        TestUtils.assert_dictionary_almost_equal(rail_parameters.__dict__, calculated_rail_parameters.__dict__)

        # check geometry and material of the sleepers
        expected_sleeper_points = {
            5: Point.create([1.0, 3.00, -1.0], 5),
            6: Point.create([1.6, 3.00, -1.0], 6),
            7: Point.create([2.2, 3.00, -1.0], 7),
            8: Point.create([2.8, 3.00, -1.0], 8),
            9: Point.create([3.4, 3.00, -1.0], 9),
            10: Point.create([4.0, 3.00, -1.0], 10),
            11: Point.create([4.6, 3.00, -1.0], 11),
            12: Point.create([5.2, 3.00, -1.0], 12)
        }
        expected_sleeper_geometry = Geometry(expected_sleeper_points)

        sleeper_model_part = model.body_model_parts[2]
        calculated_sleeper_geometry = sleeper_model_part.geometry
        calculated_sleeper_parameters = sleeper_model_part.material.material_parameters

        TestUtils.assert_almost_equal_geometries(expected_sleeper_geometry, calculated_sleeper_geometry)
        TestUtils.assert_dictionary_almost_equal(sleeper_parameters.__dict__, calculated_sleeper_parameters.__dict__)

        # check geometry and material of the rail pads
        rail_pad_model_part = model.body_model_parts[3]
        calculated_rail_pad_geometry = rail_pad_model_part.geometry
        calculated_rail_pad_parameters = rail_pad_model_part.material.material_parameters

        expected_rail_pad_points = {
            13: Point.create([1.0, 3.02, -1.0], 13),
            5: Point.create([1.0, 3.0, -1.0], 5),
            14: Point.create([1.6, 3.02, -1.0], 14),
            6: Point.create([1.6, 3.0, -1.0], 6),
            15: Point.create([2.2, 3.02, -1.0], 15),
            7: Point.create([2.2, 3.0, -1.0], 7),
            16: Point.create([2.8, 3.02, -1.0], 16),
            8: Point.create([2.8, 3.0, -1.0], 8),
            17: Point.create([3.4, 3.02, -1.0], 17),
            9: Point.create([3.4, 3.0, -1.0], 9),
            18: Point.create([4.0, 3.02, -1.0], 18),
            10: Point.create([4.0, 3.0, -1.0], 10),
            19: Point.create([4.6, 3.02, -1.0], 19),
            11: Point.create([4.6, 3.0, -1.0], 11),
            20: Point.create([5.2, 3.02, -1.0], 20),
            12: Point.create([5.2, 3.0, -1.0], 12),
        }

        expected_rail_pad_lines = {
            19: Line.create([13, 5], 19),
            20: Line.create([14, 6], 20),
            21: Line.create([15, 7], 21),
            22: Line.create([16, 8], 22),
            23: Line.create([17, 9], 23),
            24: Line.create([18, 10], 24),
            25: Line.create([19, 11], 25),
            26: Line.create([20, 12], 26),
        }

        expected_rail_pad_geometry = Geometry(expected_rail_pad_points, expected_rail_pad_lines)

        TestUtils.assert_almost_equal_geometries(expected_rail_pad_geometry, calculated_rail_pad_geometry)
        TestUtils.assert_dictionary_almost_equal(rail_pad_parameters.__dict__, calculated_rail_pad_parameters.__dict__)

        # check geometry and material of soil equivalent
        soil_equivalent_model_part = model.body_model_parts[4]
        calculated_soil_equivalent_geometry = soil_equivalent_model_part.geometry
        calculated_soil_equivalent_parameters = soil_equivalent_model_part.material.material_parameters

        expected_soil_equivalent_points = {
            5: Point.create([1.0, 3.0, -1.0], 5),
            21: Point.create([1.0, -2.0, -1.0], 21),
            6: Point.create([1.6, 3.0, -1.0], 6),
            22: Point.create([1.6, -2.0, -1.0], 22),
            11: Point.create([4.6, 3.0, -1.0], 11),
            23: Point.create([4.6, -2.0, -1.0], 23),
            12: Point.create([5.2, 3.0, -1.0], 12),
            24: Point.create([5.2, -2.0, -1.0], 24),
        }

        expected_soil_equivalent_lines = {
            27: Line.create([5, 21], 27),
            28: Line.create([6, 22], 28),
            29: Line.create([11, 23], 29),
            30: Line.create([12, 24], 30),
        }

        expected_soil_equivalent_geometry = Geometry(expected_soil_equivalent_points, expected_soil_equivalent_lines)

        TestUtils.assert_almost_equal_geometries(expected_soil_equivalent_geometry, calculated_soil_equivalent_geometry)
        TestUtils.assert_dictionary_almost_equal(extended_soil_parameters.__dict__,
                                                 calculated_soil_equivalent_parameters.__dict__)

    def test_generate_extended_straight_track_3d(self, create_default_3d_soil_material: SoilMaterial):
        """
        Tests if a straight track is generated correctly in a 3d space. A straight track is generated and added to the
        model. The geometry and material of the rails, sleepers and rail pads are checked.

        Args:
            - create_default_3d_soil_material (:class:`stem.soil_material.SoilMaterial`): default soil material
        """

        model = Model(3)
        model.extrusion_length = 1
        # add soil material 2d
        soil_material = create_default_3d_soil_material
        layer_coordinates = [(0.0, 0.0, 0.0), (5.0, 0.0, 0.0), (5.0, 3.0, 0.0), (0.0, 3.0, 0.0)]
        # add soil layer
        model.add_soil_layer_by_coordinates(layer_coordinates, soil_material, "soil1")

        rail_parameters = EulerBeam(3, 1, 1, 1, 1, 1, 1, 1)
        rail_pad_parameters = ElasticSpringDamper([1, 1, 1], [1, 1, 1], [1, 1, 1], [1, 1, 1])
        extended_soil_parameters = ElasticSpringDamper([1, 1, 1], [1, 1, 1], [1, 1, 1], [1, 1, 1])
        sleeper_parameters = NodalConcentrated([1, 1, 1], 1, [1, 1, 1])

        origin_point = np.array([2.0, 3.0, -0.5])
        direction_vector = np.array([0, 0, 1])

        # create a straight track with rails, sleepers and rail pads
        # create a straight track with rails, sleepers and rail pads
        model.generate_extended_straight_track(0.5, 5, rail_parameters, sleeper_parameters, rail_pad_parameters, 0.02,
                                               origin_point, extended_soil_parameters, 5, direction_vector, "track_1")

        # check geometry and material of the rail
        expected_rail_points = {
            14: Point.create([2.0, 3.02, -0.5], 14),
            15: Point.create([2.0, 3.02, 0.0], 15),
            16: Point.create([2.0, 3.02, 0.5], 16),
            17: Point.create([2.0, 3.02, 1.0], 17),
            18: Point.create([2.0, 3.02, 1.5], 18),
        }
        expected_rail_lines = {
            21: Line.create([14, 15], 21),
            22: Line.create([15, 16], 22),
            23: Line.create([16, 17], 23),
            24: Line.create([17, 18], 24),
        }

        expected_rail_geometry = Geometry(expected_rail_points, expected_rail_lines)

        # check rail model part
        rail_model_part = model.body_model_parts[1]
        calculated_rail_geometry = rail_model_part.geometry
        calculated_rail_parameters = rail_model_part.material.material_parameters

        TestUtils.assert_almost_equal_geometries(expected_rail_geometry, calculated_rail_geometry)
        TestUtils.assert_dictionary_almost_equal(rail_parameters.__dict__, calculated_rail_parameters.__dict__)

        # check geometry and material of the sleepers
        expected_sleeper_points = {
            9: Point.create([2.0, 3.0, -0.5], 9),
            10: Point.create([2.0, 3.0, 0.0], 10),
            11: Point.create([2.0, 3.0, 0.5], 11),
            12: Point.create([2.0, 3.0, 1.0], 12),
        }

        expected_sleeper_geometry = Geometry(expected_sleeper_points)

        sleeper_model_part = model.body_model_parts[2]
        calculated_sleeper_geometry = sleeper_model_part.geometry
        calculated_sleeper_parameters = sleeper_model_part.material.material_parameters

        TestUtils.assert_almost_equal_geometries(expected_sleeper_geometry, calculated_sleeper_geometry)
        TestUtils.assert_dictionary_almost_equal(sleeper_parameters.__dict__, calculated_sleeper_parameters.__dict__)

        # check geometry and material of the rail pads
        rail_pad_model_part = model.body_model_parts[3]
        calculated_rail_pad_geometry = rail_pad_model_part.geometry
        calculated_rail_pad_parameters = rail_pad_model_part.material.material_parameters

        expected_rail_pad_points = {
            14: Point.create([2.0, 3.02, -0.5], 14),
            9: Point.create([2.0, 3.0, -0.5], 9),
            15: Point.create([2.0, 3.02, 0.0], 15),
            10: Point.create([2.0, 3.0, 0.0], 10),
            16: Point.create([2.0, 3.02, 0.5], 16),
            11: Point.create([2.0, 3.0, 0.5], 11),
            17: Point.create([2.0, 3.02, 1.0], 17),
            12: Point.create([2.0, 3.0, 1.0], 12),
            18: Point.create([2.0, 3.02, 1.5], 18),
            13: Point.create([2.0, 3.0, 1.5], 13),
        }

        expected_rail_pad_lines = {
            25: Line.create([14, 9], 25),
            26: Line.create([15, 10], 26),
            27: Line.create([16, 11], 27),
            28: Line.create([17, 12], 28),
            29: Line.create([18, 13], 29),
        }

        expected_rail_pad_geometry = Geometry(expected_rail_pad_points, expected_rail_pad_lines)

        TestUtils.assert_almost_equal_geometries(expected_rail_pad_geometry, calculated_rail_pad_geometry)
        TestUtils.assert_dictionary_almost_equal(rail_pad_parameters.__dict__, calculated_rail_pad_parameters.__dict__)

        # check rotation constrain model part
        rotation_constrain_model_part = model.process_model_parts[1]
        calculated_rotation_constrain_geometry = rotation_constrain_model_part.geometry
        calculated_rotation_constrain_parameters = rotation_constrain_model_part.parameters

        expected_rotation_constrain_points = {14: Point.create([2.0, 3.02, -0.5], 14)}
        expected_rotation_constrain_geometry = Geometry(expected_rotation_constrain_points)

        expected_rotation_constraint_parameters = RotationConstraint(value=[0, 0, 0],
                                                                     is_fixed=[True, True, True],
                                                                     active=[True, True, True])

        TestUtils.assert_almost_equal_geometries(expected_rotation_constrain_geometry,
                                                 calculated_rotation_constrain_geometry)
        TestUtils.assert_dictionary_almost_equal(expected_rotation_constraint_parameters.__dict__,
                                                 calculated_rotation_constrain_parameters.__dict__)

        # check geometry and material of soil equivalent
        soil_equivalent_model_part = model.body_model_parts[4]
        calculated_soil_equivalent_geometry = soil_equivalent_model_part.geometry
        calculated_soil_equivalent_parameters = soil_equivalent_model_part.material.material_parameters

        expected_soil_equivalent_points = {
            9: Point.create([2.0, 3.0, -0.5], 9),
            19: Point.create([2.0, -2.0, -0.5], 19),
            13: Point.create([2.0, 3.0, 1.5], 13),
            20: Point.create([2.0, -2.0, 1.5], 20),
        }

        expected_soil_equivalent_lines = {
            30: Line.create([9, 19], 30),
            31: Line.create([13, 20], 31),
        }

        expected_soil_equivalent_geometry = Geometry(expected_soil_equivalent_points, expected_soil_equivalent_lines)

        TestUtils.assert_almost_equal_geometries(expected_soil_equivalent_geometry, calculated_soil_equivalent_geometry)
        TestUtils.assert_dictionary_almost_equal(extended_soil_parameters.__dict__,
                                                 calculated_soil_equivalent_parameters.__dict__)

    def test_add_hinge_on_beam(self, create_default_3d_beam):
        """
        Test if a hinge is added correctly on a beam. A model is created with a beam and a hinge is added on the beam.
        This test also checks for errors.
        """

        model = Model(3)
        beam_material = create_default_3d_beam
        # Specify the coordinates for the beam: x:1m x y:0m
        beam_coordinates = [(0, 0, 0), (1, 0, 0)]
        # Create the beam
        gmsh_input = {beam_material.name: {"coordinates": beam_coordinates, "ndim": 1}}

        # check if extrusion length is specified in 3D
        model.gmsh_io.generate_geometry(gmsh_input, "")
        #
        # create body model part
        body_model_part = BodyModelPart(beam_material.name)
        body_model_part.material = beam_material

        # set the geometry of the body model part
        body_model_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, beam_material.name)
        model.body_model_parts.append(body_model_part)

        hinge_parameters = HingeParameters(1e9, 1e9)

        model.add_hinge_on_beam(beam_material.name, [(0.2, 0.0, 0.0)], hinge_parameters, "hinge_1")

        # check if hinge is added to the model
        assert model.process_model_parts[0].name == "hinge_1"
        TestUtils.assert_almost_equal_geometries(model.process_model_parts[0].geometry,
                                                 Geometry({3: Point.create([0.2, 0.0, 0.0], 3)}))
        assert model.process_model_parts[0].parameters == hinge_parameters

        # check that the hinge node is part of the beam model part
        assert 3 in model.body_model_parts[0].geometry.points.keys()

        # try to add hinge outside of beam
        with pytest.raises(ValueError, match="The hinge points are not part of the beam model part `beam`."):
            model.add_hinge_on_beam(beam_material.name, [(0.2, 0.2, 0.0)], hinge_parameters, "hinge_2")

        # try to add hinge on a spring damper
        spring_damper_parameters = ElasticSpringDamper([1, 1, 1], [1, 1, 1], [1, 1, 1], [1, 1, 1])
        spring_damper_material = StructuralMaterial(name="spring_damper", material_parameters=spring_damper_parameters)
        gmsh_input = {spring_damper_material.name: {"coordinates": [(-1, 0, 0), (0, 0, 0)], "ndim": 1}}
        model.gmsh_io.generate_geometry(gmsh_input, "")

        spring_damper_part = BodyModelPart("spring_damper")
        spring_damper_part.material = spring_damper_material
        spring_damper_part.get_geometry_from_geo_data(model.gmsh_io.geo_data, spring_damper_material.name)
        model.body_model_parts.append(spring_damper_part)

        with pytest.raises(ValueError, match="Hinges can only be applied to beam model parts"):
            model.add_hinge_on_beam(spring_damper_part.name, [(-0.2, 0.0, 0.0)], hinge_parameters, "hinge_3")

        # try to add hinge on non existing model part
        with pytest.raises(ValueError, match="Model part `non_existing_part` not found."):
            model.add_hinge_on_beam("non_existing_part", [(-0.2, 0.0, 0.0)], hinge_parameters, "hinge_4")

        # try to add hinge on 2D model
        model.ndim = 2
        with pytest.raises(NotImplementedError, match="Hinges can only be applied in 3D models"):
            model.add_hinge_on_beam(beam_material.name, [(-0.2, 0.0, 0.0)], hinge_parameters, "hinge_5")
