from bson import ObjectId
from bson.errors import InvalidId
from pydantic.json import ENCODERS_BY_TYPE
from pymongo import ASCENDING


def Indexed(typ, index_type=ASCENDING):
    """
    Returns a subclass of `typ` with an extra attribute `_indexed` et to True.
    When instantiated the type of the result will actually be `typ`.
    """

    class NewType(typ):
        _indexed = index_type

        def __new__(cls, *args, **kwargs):
            return typ.__new__(typ, *args, **kwargs)

    NewType.__name__ = f"Indexed {typ.__name__}"
    return NewType


class PydanticObjectId(ObjectId):
    @classmethod
    def __get_validators__(cls):
        yield cls.validate

    @classmethod
    def validate(cls, v):
        if isinstance(v, bytes):
            v = v.decode("utf-8")
        try:
            return PydanticObjectId(v)
        except InvalidId:
            raise TypeError("Id must be of type PydanticObjectId")

    @classmethod
    def __modify_schema__(cls, field_schema):
        field_schema.update_dict(
            type="string",
            examples=["5eb7cf5a86d9755df3a6c593", "5eb7cfb05e32e07750a1756a"],
        )


ENCODERS_BY_TYPE[
    PydanticObjectId
] = str  # it is a workaround to force pydantic make json schema for this field
