import numpy as np
import pandas as pd


def read_tigrfam(path: str) -> pd.DataFrame:
    """Read the TIGRFAM file.

    Args:
        path: The path to the TIGRFAM file.
    """
    dtype = {
        'seq_id': object,
        'hmm_acc': object,
        'full_seq_e_value': np.float64,
        'full_seq_score': np.float64,
        'full_seq_bias': np.float64,
        'best_domain_e_value': np.float64,
        'best_domain_score': np.float64,
        'best_domain_bias': np.float64,
        'exp': np.float64,
        'reg': np.float64,
        'clu': np.float64,
        'ov': np.float64,
        'env': np.float64,
        'dom': np.float64,
        'rep': np.float64,
        'inc': np.float64,
        'description': object,
    }
    lines = list()
    with open(path, 'r') as f:
        for line in f.readlines():
            line = line.strip()
            if line.startswith('#') or line == '':
                continue
            cols = line.split()
            cur_line = [cols[0], cols[3]]
            cur_line.extend(cols[4:18])
            cur_line.append(' '.join(cols[18:]))
            lines.append(cur_line)
    return pd.DataFrame(lines, columns=dtype)


def read_tigrfam_tophit(path: str) -> pd.DataFrame:
    """Read the TIGRFAM tophit file.

    Args:
        path: The path to the TIGRFAM tophit file.
    """
    dtype = {
        'seq_id': object,
        'marker': object,
        'e_value': np.float64,
        'bit_score': np.float64,
    }
    lines = list()
    with open(path, 'r') as f:
        f.readline()
        for line in f.readlines():
            line = line.strip()
            gene_id, hits = line.split('\t')
            for hit in hits.split(';'):
                pfam_acc, e_val, bit_score = hit.split(',')
                lines.append([gene_id, pfam_acc, e_val, bit_score])
    return pd.DataFrame(lines, columns=dtype)

#: Markers used in the TIGRFAM 15.0 release.
TIGRFAM_15 = frozenset(
    {'TIGR00001', 'TIGR00471', 'TIGR00938', 'TIGR01396', 'TIGR01853', 'TIGR02308', 'TIGR02763', 'TIGR03218',
     'TIGR03673', 'TIGR04124', 'TIGR00002', 'TIGR00472', 'TIGR00939', 'TIGR01397', 'TIGR01854', 'TIGR02309',
     'TIGR02764', 'TIGR03219', 'TIGR03674', 'TIGR04125', 'TIGR00003', 'TIGR00473', 'TIGR00940', 'TIGR01398',
     'TIGR01855', 'TIGR02310', 'TIGR02765', 'TIGR03220', 'TIGR03675', 'TIGR04126', 'TIGR00004', 'TIGR00474',
     'TIGR00941', 'TIGR01399', 'TIGR01856', 'TIGR02311', 'TIGR02766', 'TIGR03221', 'TIGR03676', 'TIGR04127',
     'TIGR00005', 'TIGR00475', 'TIGR00942', 'TIGR01400', 'TIGR01857', 'TIGR02312', 'TIGR02767', 'TIGR03222',
     'TIGR03677', 'TIGR04128', 'TIGR00006', 'TIGR00476', 'TIGR00943', 'TIGR01401', 'TIGR01858', 'TIGR02313',
     'TIGR02768', 'TIGR03223', 'TIGR03678', 'TIGR04129', 'TIGR00007', 'TIGR00477', 'TIGR00944', 'TIGR01402',
     'TIGR01859', 'TIGR02314', 'TIGR02769', 'TIGR03224', 'TIGR03679', 'TIGR04130', 'TIGR00008', 'TIGR00478',
     'TIGR00945', 'TIGR01403', 'TIGR01860', 'TIGR02315', 'TIGR02770', 'TIGR03225', 'TIGR03680', 'TIGR04131',
     'TIGR00009', 'TIGR00479', 'TIGR00946', 'TIGR01404', 'TIGR01861', 'TIGR02316', 'TIGR02771', 'TIGR03226',
     'TIGR03682', 'TIGR04132', 'TIGR00010', 'TIGR00481', 'TIGR00947', 'TIGR01405', 'TIGR01862', 'TIGR02317',
     'TIGR02772', 'TIGR03227', 'TIGR03683', 'TIGR04133', 'TIGR00011', 'TIGR00482', 'TIGR00948', 'TIGR01406',
     'TIGR01863', 'TIGR02318', 'TIGR02773', 'TIGR03228', 'TIGR03684', 'TIGR04134', 'TIGR00012', 'TIGR00483',
     'TIGR00949', 'TIGR01407', 'TIGR01865', 'TIGR02319', 'TIGR02774', 'TIGR03229', 'TIGR03685', 'TIGR04135',
     'TIGR00013', 'TIGR00484', 'TIGR00950', 'TIGR01408', 'TIGR01866', 'TIGR02320', 'TIGR02775', 'TIGR03230',
     'TIGR03686', 'TIGR04136', 'TIGR00014', 'TIGR00485', 'TIGR00951', 'TIGR01409', 'TIGR01868', 'TIGR02321',
     'TIGR02776', 'TIGR03231', 'TIGR03687', 'TIGR04137', 'TIGR00016', 'TIGR00486', 'TIGR00952', 'TIGR01410',
     'TIGR01869', 'TIGR02322', 'TIGR02777', 'TIGR03232', 'TIGR03688', 'TIGR04138', 'TIGR00017', 'TIGR00487',
     'TIGR00954', 'TIGR01411', 'TIGR01870', 'TIGR02323', 'TIGR02778', 'TIGR03233', 'TIGR03689', 'TIGR04139',
     'TIGR00018', 'TIGR00488', 'TIGR00955', 'TIGR01412', 'TIGR01873', 'TIGR02324', 'TIGR02779', 'TIGR03234',
     'TIGR03690', 'TIGR04140', 'TIGR00019', 'TIGR00489', 'TIGR00956', 'TIGR01413', 'TIGR01874', 'TIGR02325',
     'TIGR02780', 'TIGR03235', 'TIGR03691', 'TIGR04141', 'TIGR00020', 'TIGR00490', 'TIGR00957', 'TIGR01414',
     'TIGR01875', 'TIGR02326', 'TIGR02781', 'TIGR03236', 'TIGR03692', 'TIGR04142', 'TIGR00021', 'TIGR00491',
     'TIGR00958', 'TIGR01415', 'TIGR01876', 'TIGR02327', 'TIGR02782', 'TIGR03237', 'TIGR03693', 'TIGR04143',
     'TIGR00022', 'TIGR00492', 'TIGR00959', 'TIGR01416', 'TIGR01877', 'TIGR02328', 'TIGR02783', 'TIGR03238',
     'TIGR03694', 'TIGR04144', 'TIGR00023', 'TIGR00493', 'TIGR00962', 'TIGR01417', 'TIGR01878', 'TIGR02329',
     'TIGR02784', 'TIGR03239', 'TIGR03695', 'TIGR04145', 'TIGR00024', 'TIGR00494', 'TIGR00963', 'TIGR01418',
     'TIGR01879', 'TIGR02330', 'TIGR02785', 'TIGR03240', 'TIGR03696', 'TIGR04146', 'TIGR00025', 'TIGR00495',
     'TIGR00964', 'TIGR01419', 'TIGR01880', 'TIGR02331', 'TIGR02786', 'TIGR03241', 'TIGR03697', 'TIGR04147',
     'TIGR00026', 'TIGR00496', 'TIGR00965', 'TIGR01420', 'TIGR01881', 'TIGR02332', 'TIGR02787', 'TIGR03242',
     'TIGR03698', 'TIGR04148', 'TIGR00027', 'TIGR00497', 'TIGR00966', 'TIGR01421', 'TIGR01882', 'TIGR02333',
     'TIGR02788', 'TIGR03243', 'TIGR03699', 'TIGR04149', 'TIGR00028', 'TIGR00498', 'TIGR00967', 'TIGR01422',
     'TIGR01883', 'TIGR02334', 'TIGR02789', 'TIGR03244', 'TIGR03700', 'TIGR04150', 'TIGR00029', 'TIGR00499',
     'TIGR00968', 'TIGR01423', 'TIGR01884', 'TIGR02335', 'TIGR02790', 'TIGR03245', 'TIGR03701', 'TIGR04151',
     'TIGR00030', 'TIGR00500', 'TIGR00969', 'TIGR01424', 'TIGR01885', 'TIGR02336', 'TIGR02791', 'TIGR03246',
     'TIGR03702', 'TIGR04152', 'TIGR00031', 'TIGR00501', 'TIGR00970', 'TIGR01425', 'TIGR01886', 'TIGR02337',
     'TIGR02792', 'TIGR03247', 'TIGR03703', 'TIGR04153', 'TIGR00032', 'TIGR00502', 'TIGR00971', 'TIGR01426',
     'TIGR01887', 'TIGR02338', 'TIGR02793', 'TIGR03248', 'TIGR03704', 'TIGR04154', 'TIGR00033', 'TIGR00503',
     'TIGR00972', 'TIGR01427', 'TIGR01888', 'TIGR02339', 'TIGR02794', 'TIGR03249', 'TIGR03705', 'TIGR04155',
     'TIGR00034', 'TIGR00504', 'TIGR00973', 'TIGR01428', 'TIGR01889', 'TIGR02340', 'TIGR02795', 'TIGR03250',
     'TIGR03706', 'TIGR04156', 'TIGR00035', 'TIGR00505', 'TIGR00974', 'TIGR01429', 'TIGR01890', 'TIGR02341',
     'TIGR02796', 'TIGR03251', 'TIGR03707', 'TIGR04157', 'TIGR00036', 'TIGR00506', 'TIGR00975', 'TIGR01430',
     'TIGR01891', 'TIGR02342', 'TIGR02797', 'TIGR03252', 'TIGR03708', 'TIGR04158', 'TIGR00037', 'TIGR00507',
     'TIGR00976', 'TIGR01431', 'TIGR01892', 'TIGR02343', 'TIGR02798', 'TIGR03253', 'TIGR03709', 'TIGR04159',
     'TIGR00038', 'TIGR00508', 'TIGR00977', 'TIGR01432', 'TIGR01893', 'TIGR02344', 'TIGR02799', 'TIGR03254',
     'TIGR03710', 'TIGR04160', 'TIGR00039', 'TIGR00509', 'TIGR00978', 'TIGR01433', 'TIGR01894', 'TIGR02345',
     'TIGR02800', 'TIGR03255', 'TIGR03711', 'TIGR04161', 'TIGR00040', 'TIGR00510', 'TIGR00979', 'TIGR01434',
     'TIGR01895', 'TIGR02346', 'TIGR02801', 'TIGR03256', 'TIGR03712', 'TIGR04162', 'TIGR00041', 'TIGR00511',
     'TIGR00980', 'TIGR01435', 'TIGR01896', 'TIGR02347', 'TIGR02802', 'TIGR03257', 'TIGR03713', 'TIGR04163',
     'TIGR00042', 'TIGR00512', 'TIGR00981', 'TIGR01436', 'TIGR01897', 'TIGR02348', 'TIGR02803', 'TIGR03258',
     'TIGR03714', 'TIGR04164', 'TIGR00043', 'TIGR00513', 'TIGR00982', 'TIGR01437', 'TIGR01898', 'TIGR02349',
     'TIGR02804', 'TIGR03259', 'TIGR03715', 'TIGR04165', 'TIGR00044', 'TIGR00514', 'TIGR00983', 'TIGR01438',
     'TIGR01899', 'TIGR02350', 'TIGR02805', 'TIGR03260', 'TIGR03716', 'TIGR04166', 'TIGR00045', 'TIGR00515',
     'TIGR00984', 'TIGR01439', 'TIGR01900', 'TIGR02351', 'TIGR02806', 'TIGR03261', 'TIGR03717', 'TIGR04167',
     'TIGR00046', 'TIGR00516', 'TIGR00985', 'TIGR01440', 'TIGR01901', 'TIGR02352', 'TIGR02807', 'TIGR03262',
     'TIGR03718', 'TIGR04168', 'TIGR00048', 'TIGR00517', 'TIGR00986', 'TIGR01441', 'TIGR01902', 'TIGR02353',
     'TIGR02808', 'TIGR03263', 'TIGR03719', 'TIGR04169', 'TIGR00049', 'TIGR00518', 'TIGR00987', 'TIGR01442',
     'TIGR01903', 'TIGR02354', 'TIGR02809', 'TIGR03264', 'TIGR03720', 'TIGR04170', 'TIGR00050', 'TIGR00519',
     'TIGR00988', 'TIGR01443', 'TIGR01904', 'TIGR02355', 'TIGR02810', 'TIGR03265', 'TIGR03721', 'TIGR04171',
     'TIGR00051', 'TIGR00520', 'TIGR00989', 'TIGR01444', 'TIGR01905', 'TIGR02356', 'TIGR02811', 'TIGR03266',
     'TIGR03722', 'TIGR04172', 'TIGR00052', 'TIGR00521', 'TIGR00990', 'TIGR01445', 'TIGR01906', 'TIGR02357',
     'TIGR02812', 'TIGR03267', 'TIGR03723', 'TIGR04173', 'TIGR00053', 'TIGR00522', 'TIGR00991', 'TIGR01446',
     'TIGR01907', 'TIGR02358', 'TIGR02813', 'TIGR03268', 'TIGR03724', 'TIGR04174', 'TIGR00054', 'TIGR00523',
     'TIGR00992', 'TIGR01447', 'TIGR01908', 'TIGR02359', 'TIGR02814', 'TIGR03269', 'TIGR03725', 'TIGR04175',
     'TIGR00055', 'TIGR00524', 'TIGR00993', 'TIGR01448', 'TIGR01909', 'TIGR02360', 'TIGR02815', 'TIGR03270',
     'TIGR03726', 'TIGR04176', 'TIGR00056', 'TIGR00525', 'TIGR00994', 'TIGR01449', 'TIGR01910', 'TIGR02361',
     'TIGR02816', 'TIGR03271', 'TIGR03727', 'TIGR04177', 'TIGR00057', 'TIGR00526', 'TIGR00995', 'TIGR01450',
     'TIGR01911', 'TIGR02362', 'TIGR02817', 'TIGR03272', 'TIGR03728', 'TIGR04178', 'TIGR00058', 'TIGR00527',
     'TIGR00996', 'TIGR01451', 'TIGR01912', 'TIGR02363', 'TIGR02818', 'TIGR03274', 'TIGR03729', 'TIGR04179',
     'TIGR00059', 'TIGR00528', 'TIGR00997', 'TIGR01452', 'TIGR01913', 'TIGR02364', 'TIGR02819', 'TIGR03275',
     'TIGR03730', 'TIGR04180', 'TIGR00060', 'TIGR00529', 'TIGR00998', 'TIGR01453', 'TIGR01914', 'TIGR02365',
     'TIGR02820', 'TIGR03276', 'TIGR03731', 'TIGR04181', 'TIGR00061', 'TIGR00530', 'TIGR00999', 'TIGR01454',
     'TIGR01915', 'TIGR02366', 'TIGR02821', 'TIGR03277', 'TIGR03732', 'TIGR04182', 'TIGR00062', 'TIGR00531',
     'TIGR01000', 'TIGR01455', 'TIGR01916', 'TIGR02367', 'TIGR02822', 'TIGR03278', 'TIGR03733', 'TIGR04183',
     'TIGR00063', 'TIGR00532', 'TIGR01001', 'TIGR01456', 'TIGR01917', 'TIGR02368', 'TIGR02823', 'TIGR03279',
     'TIGR03734', 'TIGR04184', 'TIGR00064', 'TIGR00533', 'TIGR01002', 'TIGR01457', 'TIGR01918', 'TIGR02369',
     'TIGR02824', 'TIGR03280', 'TIGR03735', 'TIGR04185', 'TIGR00065', 'TIGR00534', 'TIGR01003', 'TIGR01458',
     'TIGR01919', 'TIGR02370', 'TIGR02825', 'TIGR03281', 'TIGR03736', 'TIGR04186', 'TIGR00066', 'TIGR00535',
     'TIGR01004', 'TIGR01459', 'TIGR01920', 'TIGR02371', 'TIGR02826', 'TIGR03282', 'TIGR03737', 'TIGR04187',
     'TIGR00067', 'TIGR00536', 'TIGR01005', 'TIGR01460', 'TIGR01921', 'TIGR02372', 'TIGR02827', 'TIGR03283',
     'TIGR03738', 'TIGR04188', 'TIGR00068', 'TIGR00537', 'TIGR01006', 'TIGR01461', 'TIGR01922', 'TIGR02373',
     'TIGR02828', 'TIGR03284', 'TIGR03739', 'TIGR04189', 'TIGR00069', 'TIGR00538', 'TIGR01007', 'TIGR01462',
     'TIGR01923', 'TIGR02374', 'TIGR02829', 'TIGR03285', 'TIGR03740', 'TIGR04190', 'TIGR00070', 'TIGR00539',
     'TIGR01008', 'TIGR01463', 'TIGR01924', 'TIGR02375', 'TIGR02830', 'TIGR03286', 'TIGR03741', 'TIGR04191',
     'TIGR00071', 'TIGR00540', 'TIGR01009', 'TIGR01464', 'TIGR01925', 'TIGR02376', 'TIGR02831', 'TIGR03287',
     'TIGR03742', 'TIGR04192', 'TIGR00072', 'TIGR00541', 'TIGR01010', 'TIGR01465', 'TIGR01926', 'TIGR02377',
     'TIGR02832', 'TIGR03288', 'TIGR03743', 'TIGR04193', 'TIGR00073', 'TIGR00542', 'TIGR01011', 'TIGR01466',
     'TIGR01927', 'TIGR02378', 'TIGR02833', 'TIGR03289', 'TIGR03744', 'TIGR04194', 'TIGR00074', 'TIGR00543',
     'TIGR01012', 'TIGR01467', 'TIGR01928', 'TIGR02379', 'TIGR02834', 'TIGR03290', 'TIGR03745', 'TIGR04195',
     'TIGR00075', 'TIGR00544', 'TIGR01013', 'TIGR01469', 'TIGR01929', 'TIGR02380', 'TIGR02835', 'TIGR03291',
     'TIGR03746', 'TIGR04196', 'TIGR00077', 'TIGR00545', 'TIGR01015', 'TIGR01470', 'TIGR01930', 'TIGR02381',
     'TIGR02836', 'TIGR03292', 'TIGR03747', 'TIGR04197', 'TIGR00078', 'TIGR00546', 'TIGR01016', 'TIGR01472',
     'TIGR01931', 'TIGR02382', 'TIGR02837', 'TIGR03293', 'TIGR03748', 'TIGR04198', 'TIGR00079', 'TIGR00547',
     'TIGR01017', 'TIGR01473', 'TIGR01932', 'TIGR02383', 'TIGR02838', 'TIGR03294', 'TIGR03749', 'TIGR04199',
     'TIGR00080', 'TIGR00548', 'TIGR01018', 'TIGR01474', 'TIGR01933', 'TIGR02384', 'TIGR02839', 'TIGR03295',
     'TIGR03750', 'TIGR04200', 'TIGR00081', 'TIGR00549', 'TIGR01019', 'TIGR01475', 'TIGR01934', 'TIGR02385',
     'TIGR02840', 'TIGR03296', 'TIGR03751', 'TIGR04201', 'TIGR00082', 'TIGR00550', 'TIGR01020', 'TIGR01476',
     'TIGR01935', 'TIGR02386', 'TIGR02841', 'TIGR03297', 'TIGR03752', 'TIGR04202', 'TIGR00083', 'TIGR00551',
     'TIGR01021', 'TIGR01477', 'TIGR01936', 'TIGR02387', 'TIGR02842', 'TIGR03298', 'TIGR03753', 'TIGR04203',
     'TIGR00084', 'TIGR00552', 'TIGR01022', 'TIGR01478', 'TIGR01937', 'TIGR02388', 'TIGR02843', 'TIGR03299',
     'TIGR03754', 'TIGR04204', 'TIGR00086', 'TIGR00553', 'TIGR01023', 'TIGR01479', 'TIGR01938', 'TIGR02389',
     'TIGR02844', 'TIGR03300', 'TIGR03755', 'TIGR04205', 'TIGR00087', 'TIGR00554', 'TIGR01024', 'TIGR01480',
     'TIGR01939', 'TIGR02390', 'TIGR02845', 'TIGR03301', 'TIGR03756', 'TIGR04206', 'TIGR00088', 'TIGR00555',
     'TIGR01025', 'TIGR01481', 'TIGR01940', 'TIGR02391', 'TIGR02846', 'TIGR03302', 'TIGR03757', 'TIGR04207',
     'TIGR00089', 'TIGR00556', 'TIGR01026', 'TIGR01482', 'TIGR01941', 'TIGR02392', 'TIGR02847', 'TIGR03303',
     'TIGR03758', 'TIGR04209', 'TIGR00090', 'TIGR00557', 'TIGR01027', 'TIGR01484', 'TIGR01942', 'TIGR02393',
     'TIGR02848', 'TIGR03304', 'TIGR03759', 'TIGR04210', 'TIGR00091', 'TIGR00558', 'TIGR01028', 'TIGR01485',
     'TIGR01943', 'TIGR02394', 'TIGR02849', 'TIGR03305', 'TIGR03760', 'TIGR04211', 'TIGR00092', 'TIGR00559',
     'TIGR01029', 'TIGR01486', 'TIGR01944', 'TIGR02395', 'TIGR02850', 'TIGR03306', 'TIGR03761', 'TIGR04212',
     'TIGR00093', 'TIGR00560', 'TIGR01030', 'TIGR01487', 'TIGR01945', 'TIGR02396', 'TIGR02851', 'TIGR03307',
     'TIGR03762', 'TIGR04213', 'TIGR00094', 'TIGR00561', 'TIGR01031', 'TIGR01488', 'TIGR01946', 'TIGR02397',
     'TIGR02852', 'TIGR03308', 'TIGR03763', 'TIGR04214', 'TIGR00095', 'TIGR00562', 'TIGR01032', 'TIGR01489',
     'TIGR01947', 'TIGR02398', 'TIGR02853', 'TIGR03309', 'TIGR03764', 'TIGR04215', 'TIGR00096', 'TIGR00563',
     'TIGR01033', 'TIGR01490', 'TIGR01948', 'TIGR02399', 'TIGR02854', 'TIGR03310', 'TIGR03765', 'TIGR04216',
     'TIGR00097', 'TIGR00564', 'TIGR01034', 'TIGR01491', 'TIGR01949', 'TIGR02400', 'TIGR02855', 'TIGR03311',
     'TIGR03766', 'TIGR04217', 'TIGR00099', 'TIGR00565', 'TIGR01035', 'TIGR01492', 'TIGR01950', 'TIGR02401',
     'TIGR02856', 'TIGR03312', 'TIGR03767', 'TIGR04218', 'TIGR00100', 'TIGR00566', 'TIGR01036', 'TIGR01493',
     'TIGR01951', 'TIGR02402', 'TIGR02857', 'TIGR03313', 'TIGR03768', 'TIGR04219', 'TIGR00101', 'TIGR00567',
     'TIGR01037', 'TIGR01494', 'TIGR01952', 'TIGR02403', 'TIGR02858', 'TIGR03314', 'TIGR03769', 'TIGR04220',
     'TIGR00103', 'TIGR00568', 'TIGR01038', 'TIGR01495', 'TIGR01953', 'TIGR02404', 'TIGR02859', 'TIGR03315',
     'TIGR03770', 'TIGR04221', 'TIGR00104', 'TIGR00569', 'TIGR01039', 'TIGR01496', 'TIGR01954', 'TIGR02405',
     'TIGR02860', 'TIGR03316', 'TIGR03771', 'TIGR04222', 'TIGR00105', 'TIGR00570', 'TIGR01040', 'TIGR01497',
     'TIGR01955', 'TIGR02406', 'TIGR02861', 'TIGR03317', 'TIGR03772', 'TIGR04223', 'TIGR00106', 'TIGR00571',
     'TIGR01041', 'TIGR01498', 'TIGR01956', 'TIGR02407', 'TIGR02862', 'TIGR03318', 'TIGR03773', 'TIGR04224',
     'TIGR00107', 'TIGR00573', 'TIGR01042', 'TIGR01499', 'TIGR01957', 'TIGR02408', 'TIGR02863', 'TIGR03319',
     'TIGR03774', 'TIGR04225', 'TIGR00109', 'TIGR00574', 'TIGR01043', 'TIGR01500', 'TIGR01958', 'TIGR02409',
     'TIGR02864', 'TIGR03321', 'TIGR03775', 'TIGR04226', 'TIGR00110', 'TIGR00575', 'TIGR01044', 'TIGR01501',
     'TIGR01959', 'TIGR02410', 'TIGR02865', 'TIGR03322', 'TIGR03776', 'TIGR04227', 'TIGR00111', 'TIGR00576',
     'TIGR01045', 'TIGR01502', 'TIGR01960', 'TIGR02411', 'TIGR02866', 'TIGR03323', 'TIGR03777', 'TIGR04228',
     'TIGR00112', 'TIGR00577', 'TIGR01046', 'TIGR01503', 'TIGR01961', 'TIGR02412', 'TIGR02867', 'TIGR03324',
     'TIGR03778', 'TIGR04229', 'TIGR00113', 'TIGR00578', 'TIGR01047', 'TIGR01504', 'TIGR01962', 'TIGR02413',
     'TIGR02868', 'TIGR03325', 'TIGR03779', 'TIGR04230', 'TIGR00114', 'TIGR00580', 'TIGR01048', 'TIGR01505',
     'TIGR01963', 'TIGR02414', 'TIGR02869', 'TIGR03326', 'TIGR03780', 'TIGR04231', 'TIGR00115', 'TIGR00581',
     'TIGR01049', 'TIGR01506', 'TIGR01964', 'TIGR02415', 'TIGR02870', 'TIGR03327', 'TIGR03781', 'TIGR04232',
     'TIGR00116', 'TIGR00583', 'TIGR01050', 'TIGR01507', 'TIGR01965', 'TIGR02416', 'TIGR02871', 'TIGR03328',
     'TIGR03782', 'TIGR04233', 'TIGR00117', 'TIGR00584', 'TIGR01051', 'TIGR01508', 'TIGR01966', 'TIGR02417',
     'TIGR02872', 'TIGR03329', 'TIGR03783', 'TIGR04234', 'TIGR00118', 'TIGR00585', 'TIGR01052', 'TIGR01509',
     'TIGR01967', 'TIGR02418', 'TIGR02873', 'TIGR03330', 'TIGR03784', 'TIGR04235', 'TIGR00119', 'TIGR00586',
     'TIGR01053', 'TIGR01510', 'TIGR01968', 'TIGR02419', 'TIGR02874', 'TIGR03331', 'TIGR03785', 'TIGR04236',
     'TIGR00120', 'TIGR00587', 'TIGR01054', 'TIGR01511', 'TIGR01969', 'TIGR02420', 'TIGR02875', 'TIGR03332',
     'TIGR03786', 'TIGR04237', 'TIGR00121', 'TIGR00588', 'TIGR01055', 'TIGR01512', 'TIGR01970', 'TIGR02421',
     'TIGR02876', 'TIGR03333', 'TIGR03787', 'TIGR04238', 'TIGR00122', 'TIGR00589', 'TIGR01056', 'TIGR01513',
     'TIGR01971', 'TIGR02422', 'TIGR02877', 'TIGR03334', 'TIGR03788', 'TIGR04239', 'TIGR00123', 'TIGR00590',
     'TIGR01057', 'TIGR01514', 'TIGR01972', 'TIGR02423', 'TIGR02878', 'TIGR03335', 'TIGR03789', 'TIGR04240',
     'TIGR00124', 'TIGR00591', 'TIGR01058', 'TIGR01515', 'TIGR01973', 'TIGR02424', 'TIGR02880', 'TIGR03336',
     'TIGR03790', 'TIGR04241', 'TIGR00125', 'TIGR00592', 'TIGR01059', 'TIGR01517', 'TIGR01974', 'TIGR02425',
     'TIGR02881', 'TIGR03337', 'TIGR03791', 'TIGR04242', 'TIGR00126', 'TIGR00593', 'TIGR01060', 'TIGR01518',
     'TIGR01975', 'TIGR02426', 'TIGR02882', 'TIGR03338', 'TIGR03792', 'TIGR04243', 'TIGR00127', 'TIGR00594',
     'TIGR01061', 'TIGR01519', 'TIGR01976', 'TIGR02427', 'TIGR02883', 'TIGR03339', 'TIGR03793', 'TIGR04244',
     'TIGR00128', 'TIGR00595', 'TIGR01062', 'TIGR01520', 'TIGR01977', 'TIGR02428', 'TIGR02884', 'TIGR03340',
     'TIGR03794', 'TIGR04245', 'TIGR00129', 'TIGR00596', 'TIGR01063', 'TIGR01521', 'TIGR01978', 'TIGR02429',
     'TIGR02885', 'TIGR03341', 'TIGR03795', 'TIGR04246', 'TIGR00130', 'TIGR00597', 'TIGR01064', 'TIGR01522',
     'TIGR01979', 'TIGR02430', 'TIGR02886', 'TIGR03342', 'TIGR03796', 'TIGR04247', 'TIGR00131', 'TIGR00598',
     'TIGR01065', 'TIGR01523', 'TIGR01980', 'TIGR02431', 'TIGR02887', 'TIGR03343', 'TIGR03797', 'TIGR04248',
     'TIGR00132', 'TIGR00599', 'TIGR01066', 'TIGR01524', 'TIGR01981', 'TIGR02432', 'TIGR02888', 'TIGR03344',
     'TIGR03798', 'TIGR04249', 'TIGR00133', 'TIGR00600', 'TIGR01067', 'TIGR01525', 'TIGR01982', 'TIGR02433',
     'TIGR02889', 'TIGR03345', 'TIGR03799', 'TIGR04250', 'TIGR00134', 'TIGR00601', 'TIGR01068', 'TIGR01526',
     'TIGR01983', 'TIGR02434', 'TIGR02890', 'TIGR03346', 'TIGR03800', 'TIGR04251', 'TIGR00135', 'TIGR00602',
     'TIGR01069', 'TIGR01527', 'TIGR01984', 'TIGR02435', 'TIGR02891', 'TIGR03347', 'TIGR03801', 'TIGR04252',
     'TIGR00136', 'TIGR00603', 'TIGR01070', 'TIGR01528', 'TIGR01985', 'TIGR02436', 'TIGR02892', 'TIGR03348',
     'TIGR03802', 'TIGR04253', 'TIGR00137', 'TIGR00604', 'TIGR01071', 'TIGR01529', 'TIGR01986', 'TIGR02437',
     'TIGR02893', 'TIGR03349', 'TIGR03803', 'TIGR04254', 'TIGR00138', 'TIGR00605', 'TIGR01072', 'TIGR01530',
     'TIGR01987', 'TIGR02438', 'TIGR02894', 'TIGR03350', 'TIGR03804', 'TIGR04255', 'TIGR00139', 'TIGR00606',
     'TIGR01073', 'TIGR01531', 'TIGR01988', 'TIGR02439', 'TIGR02895', 'TIGR03351', 'TIGR03805', 'TIGR04256',
     'TIGR00140', 'TIGR00607', 'TIGR01074', 'TIGR01532', 'TIGR01989', 'TIGR02440', 'TIGR02896', 'TIGR03352',
     'TIGR03806', 'TIGR04257', 'TIGR00142', 'TIGR00608', 'TIGR01075', 'TIGR01533', 'TIGR01990', 'TIGR02441',
     'TIGR02897', 'TIGR03353', 'TIGR03807', 'TIGR04258', 'TIGR00143', 'TIGR00609', 'TIGR01076', 'TIGR01534',
     'TIGR01991', 'TIGR02442', 'TIGR02898', 'TIGR03354', 'TIGR03808', 'TIGR04259', 'TIGR00144', 'TIGR00611',
     'TIGR01077', 'TIGR01535', 'TIGR01992', 'TIGR02443', 'TIGR02899', 'TIGR03355', 'TIGR03809', 'TIGR04260',
     'TIGR00145', 'TIGR00612', 'TIGR01078', 'TIGR01536', 'TIGR01993', 'TIGR02444', 'TIGR02900', 'TIGR03356',
     'TIGR03810', 'TIGR04261', 'TIGR00147', 'TIGR00613', 'TIGR01079', 'TIGR01537', 'TIGR01994', 'TIGR02445',
     'TIGR02901', 'TIGR03357', 'TIGR03811', 'TIGR04262', 'TIGR00148', 'TIGR00614', 'TIGR01080', 'TIGR01538',
     'TIGR01995', 'TIGR02446', 'TIGR02902', 'TIGR03358', 'TIGR03812', 'TIGR04263', 'TIGR00149', 'TIGR00615',
     'TIGR01081', 'TIGR01539', 'TIGR01996', 'TIGR02447', 'TIGR02903', 'TIGR03359', 'TIGR03813', 'TIGR04264',
     'TIGR00150', 'TIGR00616', 'TIGR01082', 'TIGR01540', 'TIGR01997', 'TIGR02448', 'TIGR02904', 'TIGR03360',
     'TIGR03814', 'TIGR04265', 'TIGR00151', 'TIGR00617', 'TIGR01083', 'TIGR01541', 'TIGR01998', 'TIGR02449',
     'TIGR02905', 'TIGR03361', 'TIGR03815', 'TIGR04266', 'TIGR00152', 'TIGR00618', 'TIGR01084', 'TIGR01542',
     'TIGR01999', 'TIGR02450', 'TIGR02906', 'TIGR03362', 'TIGR03816', 'TIGR04267', 'TIGR00153', 'TIGR00619',
     'TIGR01085', 'TIGR01543', 'TIGR02000', 'TIGR02451', 'TIGR02907', 'TIGR03363', 'TIGR03817', 'TIGR04268',
     'TIGR00154', 'TIGR00621', 'TIGR01086', 'TIGR01544', 'TIGR02001', 'TIGR02452', 'TIGR02908', 'TIGR03364',
     'TIGR03818', 'TIGR04269', 'TIGR00155', 'TIGR00622', 'TIGR01087', 'TIGR01545', 'TIGR02002', 'TIGR02453',
     'TIGR02909', 'TIGR03365', 'TIGR03819', 'TIGR04270', 'TIGR00156', 'TIGR00623', 'TIGR01088', 'TIGR01546',
     'TIGR02003', 'TIGR02454', 'TIGR02910', 'TIGR03366', 'TIGR03820', 'TIGR04271', 'TIGR00157', 'TIGR00624',
     'TIGR01089', 'TIGR01547', 'TIGR02004', 'TIGR02455', 'TIGR02911', 'TIGR03367', 'TIGR03821', 'TIGR04272',
     'TIGR00158', 'TIGR00625', 'TIGR01090', 'TIGR01548', 'TIGR02005', 'TIGR02456', 'TIGR02912', 'TIGR03368',
     'TIGR03822', 'TIGR04273', 'TIGR00159', 'TIGR00627', 'TIGR01091', 'TIGR01549', 'TIGR02006', 'TIGR02457',
     'TIGR02913', 'TIGR03369', 'TIGR03823', 'TIGR04274', 'TIGR00160', 'TIGR00628', 'TIGR01092', 'TIGR01550',
     'TIGR02007', 'TIGR02458', 'TIGR02914', 'TIGR03370', 'TIGR03824', 'TIGR04275', 'TIGR00161', 'TIGR00629',
     'TIGR01093', 'TIGR01551', 'TIGR02008', 'TIGR02459', 'TIGR02915', 'TIGR03371', 'TIGR03825', 'TIGR04276',
     'TIGR00162', 'TIGR00630', 'TIGR01096', 'TIGR01552', 'TIGR02009', 'TIGR02460', 'TIGR02916', 'TIGR03372',
     'TIGR03826', 'TIGR04277', 'TIGR00163', 'TIGR00631', 'TIGR01097', 'TIGR01553', 'TIGR02010', 'TIGR02461',
     'TIGR02917', 'TIGR03373', 'TIGR03827', 'TIGR04278', 'TIGR00164', 'TIGR00632', 'TIGR01098', 'TIGR01554',
     'TIGR02011', 'TIGR02462', 'TIGR02918', 'TIGR03374', 'TIGR03828', 'TIGR04279', 'TIGR00165', 'TIGR00633',
     'TIGR01099', 'TIGR01555', 'TIGR02012', 'TIGR02463', 'TIGR02919', 'TIGR03375', 'TIGR03829', 'TIGR04280',
     'TIGR00166', 'TIGR00634', 'TIGR01100', 'TIGR01556', 'TIGR02013', 'TIGR02464', 'TIGR02920', 'TIGR03376',
     'TIGR03830', 'TIGR04281', 'TIGR00167', 'TIGR00635', 'TIGR01101', 'TIGR01557', 'TIGR02014', 'TIGR02465',
     'TIGR02921', 'TIGR03377', 'TIGR03831', 'TIGR04282', 'TIGR00168', 'TIGR00636', 'TIGR01102', 'TIGR01558',
     'TIGR02015', 'TIGR02466', 'TIGR02922', 'TIGR03378', 'TIGR03832', 'TIGR04283', 'TIGR00169', 'TIGR00637',
     'TIGR01103', 'TIGR01559', 'TIGR02016', 'TIGR02467', 'TIGR02923', 'TIGR03379', 'TIGR03833', 'TIGR04284',
     'TIGR00170', 'TIGR00638', 'TIGR01104', 'TIGR01560', 'TIGR02017', 'TIGR02468', 'TIGR02924', 'TIGR03380',
     'TIGR03834', 'TIGR04285', 'TIGR00171', 'TIGR00639', 'TIGR01105', 'TIGR01561', 'TIGR02018', 'TIGR02469',
     'TIGR02925', 'TIGR03381', 'TIGR03835', 'TIGR04286', 'TIGR00172', 'TIGR00640', 'TIGR01106', 'TIGR01562',
     'TIGR02019', 'TIGR02470', 'TIGR02926', 'TIGR03382', 'TIGR03836', 'TIGR04287', 'TIGR00173', 'TIGR00641',
     'TIGR01107', 'TIGR01563', 'TIGR02020', 'TIGR02471', 'TIGR02927', 'TIGR03383', 'TIGR03837', 'TIGR04288',
     'TIGR00174', 'TIGR00642', 'TIGR01108', 'TIGR01564', 'TIGR02021', 'TIGR02472', 'TIGR02928', 'TIGR03384',
     'TIGR03838', 'TIGR04289', 'TIGR00175', 'TIGR00643', 'TIGR01109', 'TIGR01565', 'TIGR02022', 'TIGR02473',
     'TIGR02929', 'TIGR03385', 'TIGR03839', 'TIGR04290', 'TIGR00176', 'TIGR00644', 'TIGR01110', 'TIGR01566',
     'TIGR02023', 'TIGR02474', 'TIGR02930', 'TIGR03388', 'TIGR03840', 'TIGR04291', 'TIGR00177', 'TIGR00645',
     'TIGR01111', 'TIGR01567', 'TIGR02024', 'TIGR02475', 'TIGR02931', 'TIGR03389', 'TIGR03841', 'TIGR04292',
     'TIGR00178', 'TIGR00646', 'TIGR01112', 'TIGR01568', 'TIGR02025', 'TIGR02476', 'TIGR02932', 'TIGR03390',
     'TIGR03842', 'TIGR04293', 'TIGR00179', 'TIGR00647', 'TIGR01113', 'TIGR01569', 'TIGR02026', 'TIGR02477',
     'TIGR02933', 'TIGR03391', 'TIGR03843', 'TIGR04294', 'TIGR00180', 'TIGR00648', 'TIGR01114', 'TIGR01570',
     'TIGR02027', 'TIGR02478', 'TIGR02934', 'TIGR03392', 'TIGR03844', 'TIGR04295', 'TIGR00181', 'TIGR00649',
     'TIGR01115', 'TIGR01571', 'TIGR02028', 'TIGR02479', 'TIGR02935', 'TIGR03393', 'TIGR03845', 'TIGR04296',
     'TIGR00182', 'TIGR00651', 'TIGR01116', 'TIGR01572', 'TIGR02029', 'TIGR02480', 'TIGR02936', 'TIGR03394',
     'TIGR03846', 'TIGR04297', 'TIGR00183', 'TIGR00652', 'TIGR01117', 'TIGR01573', 'TIGR02030', 'TIGR02481',
     'TIGR02937', 'TIGR03395', 'TIGR03847', 'TIGR04298', 'TIGR00184', 'TIGR00653', 'TIGR01118', 'TIGR01574',
     'TIGR02031', 'TIGR02482', 'TIGR02938', 'TIGR03396', 'TIGR03848', 'TIGR04299', 'TIGR00185', 'TIGR00654',
     'TIGR01119', 'TIGR01575', 'TIGR02032', 'TIGR02483', 'TIGR02939', 'TIGR03397', 'TIGR03849', 'TIGR04300',
     'TIGR00186', 'TIGR00655', 'TIGR01120', 'TIGR01577', 'TIGR02033', 'TIGR02484', 'TIGR02940', 'TIGR03398',
     'TIGR03850', 'TIGR04301', 'TIGR00187', 'TIGR00656', 'TIGR01121', 'TIGR01578', 'TIGR02034', 'TIGR02485',
     'TIGR02941', 'TIGR03399', 'TIGR03851', 'TIGR04302', 'TIGR00188', 'TIGR00657', 'TIGR01122', 'TIGR01579',
     'TIGR02035', 'TIGR02486', 'TIGR02943', 'TIGR03400', 'TIGR03852', 'TIGR04303', 'TIGR00189', 'TIGR00658',
     'TIGR01123', 'TIGR01580', 'TIGR02036', 'TIGR02487', 'TIGR02944', 'TIGR03401', 'TIGR03853', 'TIGR04304',
     'TIGR00190', 'TIGR00659', 'TIGR01124', 'TIGR01581', 'TIGR02037', 'TIGR02488', 'TIGR02945', 'TIGR03402',
     'TIGR03854', 'TIGR04305', 'TIGR00191', 'TIGR00661', 'TIGR01125', 'TIGR01582', 'TIGR02038', 'TIGR02489',
     'TIGR02946', 'TIGR03403', 'TIGR03855', 'TIGR04306', 'TIGR00192', 'TIGR00663', 'TIGR01126', 'TIGR01583',
     'TIGR02039', 'TIGR02490', 'TIGR02947', 'TIGR03404', 'TIGR03856', 'TIGR04307', 'TIGR00193', 'TIGR00664',
     'TIGR01127', 'TIGR01584', 'TIGR02040', 'TIGR02491', 'TIGR02948', 'TIGR03405', 'TIGR03857', 'TIGR04308',
     'TIGR00194', 'TIGR00665', 'TIGR01128', 'TIGR01586', 'TIGR02041', 'TIGR02492', 'TIGR02949', 'TIGR03406',
     'TIGR03858', 'TIGR04309', 'TIGR00195', 'TIGR00666', 'TIGR01129', 'TIGR01587', 'TIGR02042', 'TIGR02493',
     'TIGR02950', 'TIGR03407', 'TIGR03859', 'TIGR04310', 'TIGR00196', 'TIGR00667', 'TIGR01130', 'TIGR01588',
     'TIGR02043', 'TIGR02494', 'TIGR02951', 'TIGR03408', 'TIGR03860', 'TIGR04311', 'TIGR00197', 'TIGR00668',
     'TIGR01131', 'TIGR01589', 'TIGR02044', 'TIGR02495', 'TIGR02952', 'TIGR03409', 'TIGR03861', 'TIGR04312',
     'TIGR00198', 'TIGR00669', 'TIGR01132', 'TIGR01590', 'TIGR02045', 'TIGR02497', 'TIGR02953', 'TIGR03410',
     'TIGR03862', 'TIGR04313', 'TIGR00199', 'TIGR00670', 'TIGR01133', 'TIGR01591', 'TIGR02046', 'TIGR02498',
     'TIGR02954', 'TIGR03411', 'TIGR03863', 'TIGR04314', 'TIGR00200', 'TIGR00671', 'TIGR01134', 'TIGR01592',
     'TIGR02047', 'TIGR02499', 'TIGR02955', 'TIGR03412', 'TIGR03864', 'TIGR04315', 'TIGR00201', 'TIGR00672',
     'TIGR01135', 'TIGR01593', 'TIGR02048', 'TIGR02500', 'TIGR02956', 'TIGR03413', 'TIGR03865', 'TIGR04316',
     'TIGR00202', 'TIGR00673', 'TIGR01136', 'TIGR01594', 'TIGR02049', 'TIGR02501', 'TIGR02957', 'TIGR03414',
     'TIGR03866', 'TIGR04317', 'TIGR00203', 'TIGR00674', 'TIGR01137', 'TIGR01595', 'TIGR02050', 'TIGR02502',
     'TIGR02959', 'TIGR03415', 'TIGR03867', 'TIGR04318', 'TIGR00204', 'TIGR00675', 'TIGR01138', 'TIGR01596',
     'TIGR02051', 'TIGR02503', 'TIGR02960', 'TIGR03416', 'TIGR03868', 'TIGR04319', 'TIGR00205', 'TIGR00676',
     'TIGR01139', 'TIGR01597', 'TIGR02052', 'TIGR02504', 'TIGR02961', 'TIGR03417', 'TIGR03869', 'TIGR04320',
     'TIGR00206', 'TIGR00677', 'TIGR01140', 'TIGR01598', 'TIGR02053', 'TIGR02505', 'TIGR02962', 'TIGR03418',
     'TIGR03870', 'TIGR04321', 'TIGR00207', 'TIGR00678', 'TIGR01141', 'TIGR01599', 'TIGR02054', 'TIGR02506',
     'TIGR02963', 'TIGR03419', 'TIGR03871', 'TIGR04322', 'TIGR00208', 'TIGR00679', 'TIGR01142', 'TIGR01600',
     'TIGR02055', 'TIGR02507', 'TIGR02964', 'TIGR03420', 'TIGR03872', 'TIGR04323', 'TIGR00209', 'TIGR00680',
     'TIGR01143', 'TIGR01601', 'TIGR02056', 'TIGR02508', 'TIGR02965', 'TIGR03421', 'TIGR03873', 'TIGR04324',
     'TIGR00210', 'TIGR00681', 'TIGR01144', 'TIGR01602', 'TIGR02057', 'TIGR02509', 'TIGR02966', 'TIGR03422',
     'TIGR03874', 'TIGR04325', 'TIGR00211', 'TIGR00682', 'TIGR01145', 'TIGR01603', 'TIGR02058', 'TIGR02510',
     'TIGR02967', 'TIGR03423', 'TIGR03875', 'TIGR04326', 'TIGR00212', 'TIGR00683', 'TIGR01146', 'TIGR01604',
     'TIGR02059', 'TIGR02511', 'TIGR02968', 'TIGR03424', 'TIGR03876', 'TIGR04327', 'TIGR00213', 'TIGR00684',
     'TIGR01147', 'TIGR01605', 'TIGR02060', 'TIGR02512', 'TIGR02969', 'TIGR03425', 'TIGR03877', 'TIGR04328',
     'TIGR00214', 'TIGR00685', 'TIGR01148', 'TIGR01606', 'TIGR02061', 'TIGR02513', 'TIGR02970', 'TIGR03426',
     'TIGR03878', 'TIGR04329', 'TIGR00215', 'TIGR00686', 'TIGR01149', 'TIGR01607', 'TIGR02062', 'TIGR02514',
     'TIGR02971', 'TIGR03427', 'TIGR03879', 'TIGR04330', 'TIGR00216', 'TIGR00687', 'TIGR01150', 'TIGR01608',
     'TIGR02063', 'TIGR02515', 'TIGR02972', 'TIGR03428', 'TIGR03880', 'TIGR04331', 'TIGR00217', 'TIGR00688',
     'TIGR01151', 'TIGR01609', 'TIGR02064', 'TIGR02516', 'TIGR02973', 'TIGR03429', 'TIGR03881', 'TIGR04332',
     'TIGR00218', 'TIGR00689', 'TIGR01152', 'TIGR01610', 'TIGR02065', 'TIGR02517', 'TIGR02974', 'TIGR03430',
     'TIGR03882', 'TIGR04333', 'TIGR00219', 'TIGR00690', 'TIGR01153', 'TIGR01611', 'TIGR02066', 'TIGR02518',
     'TIGR02975', 'TIGR03431', 'TIGR03883', 'TIGR04334', 'TIGR00220', 'TIGR00691', 'TIGR01156', 'TIGR01612',
     'TIGR02067', 'TIGR02519', 'TIGR02976', 'TIGR03432', 'TIGR03884', 'TIGR04335', 'TIGR00221', 'TIGR00692',
     'TIGR01157', 'TIGR01613', 'TIGR02068', 'TIGR02520', 'TIGR02977', 'TIGR03433', 'TIGR03885', 'TIGR04336',
     'TIGR00222', 'TIGR00693', 'TIGR01158', 'TIGR01614', 'TIGR02069', 'TIGR02521', 'TIGR02978', 'TIGR03434',
     'TIGR03886', 'TIGR04337', 'TIGR00223', 'TIGR00694', 'TIGR01159', 'TIGR01615', 'TIGR02070', 'TIGR02522',
     'TIGR02979', 'TIGR03435', 'TIGR03887', 'TIGR04338', 'TIGR00224', 'TIGR00695', 'TIGR01160', 'TIGR01616',
     'TIGR02071', 'TIGR02523', 'TIGR02980', 'TIGR03436', 'TIGR03888', 'TIGR04339', 'TIGR00225', 'TIGR00696',
     'TIGR01161', 'TIGR01617', 'TIGR02072', 'TIGR02524', 'TIGR02981', 'TIGR03437', 'TIGR03889', 'TIGR04340',
     'TIGR00227', 'TIGR00697', 'TIGR01162', 'TIGR01618', 'TIGR02073', 'TIGR02525', 'TIGR02982', 'TIGR03438',
     'TIGR03890', 'TIGR04341', 'TIGR00228', 'TIGR00698', 'TIGR01163', 'TIGR01619', 'TIGR02074', 'TIGR02526',
     'TIGR02983', 'TIGR03439', 'TIGR03891', 'TIGR04342', 'TIGR00229', 'TIGR00699', 'TIGR01164', 'TIGR01620',
     'TIGR02075', 'TIGR02527', 'TIGR02984', 'TIGR03440', 'TIGR03892', 'TIGR04343', 'TIGR00230', 'TIGR00700',
     'TIGR01165', 'TIGR01621', 'TIGR02076', 'TIGR02528', 'TIGR02985', 'TIGR03441', 'TIGR03893', 'TIGR04344',
     'TIGR00231', 'TIGR00701', 'TIGR01166', 'TIGR01622', 'TIGR02077', 'TIGR02529', 'TIGR02986', 'TIGR03442',
     'TIGR03894', 'TIGR04345', 'TIGR00232', 'TIGR00702', 'TIGR01167', 'TIGR01623', 'TIGR02078', 'TIGR02530',
     'TIGR02987', 'TIGR03443', 'TIGR03895', 'TIGR04346', 'TIGR00233', 'TIGR00703', 'TIGR01168', 'TIGR01624',
     'TIGR02079', 'TIGR02531', 'TIGR02988', 'TIGR03444', 'TIGR03896', 'TIGR04347', 'TIGR00234', 'TIGR00704',
     'TIGR01169', 'TIGR01625', 'TIGR02080', 'TIGR02532', 'TIGR02989', 'TIGR03445', 'TIGR03897', 'TIGR04348',
     'TIGR00235', 'TIGR00705', 'TIGR01170', 'TIGR01626', 'TIGR02081', 'TIGR02533', 'TIGR02990', 'TIGR03446',
     'TIGR03898', 'TIGR04349', 'TIGR00236', 'TIGR00706', 'TIGR01171', 'TIGR01627', 'TIGR02082', 'TIGR02534',
     'TIGR02991', 'TIGR03447', 'TIGR03899', 'TIGR04350', 'TIGR00237', 'TIGR00707', 'TIGR01172', 'TIGR01628',
     'TIGR02083', 'TIGR02535', 'TIGR02992', 'TIGR03448', 'TIGR03900', 'TIGR04351', 'TIGR00238', 'TIGR00708',
     'TIGR01173', 'TIGR01629', 'TIGR02084', 'TIGR02536', 'TIGR02993', 'TIGR03449', 'TIGR03901', 'TIGR04352',
     'TIGR00239', 'TIGR00709', 'TIGR01174', 'TIGR01630', 'TIGR02085', 'TIGR02537', 'TIGR02994', 'TIGR03450',
     'TIGR03902', 'TIGR04353', 'TIGR00240', 'TIGR00710', 'TIGR01175', 'TIGR01631', 'TIGR02086', 'TIGR02538',
     'TIGR02995', 'TIGR03451', 'TIGR03903', 'TIGR04354', 'TIGR00241', 'TIGR00711', 'TIGR01176', 'TIGR01632',
     'TIGR02087', 'TIGR02539', 'TIGR02996', 'TIGR03452', 'TIGR03904', 'TIGR04355', 'TIGR00242', 'TIGR00712',
     'TIGR01177', 'TIGR01633', 'TIGR02088', 'TIGR02540', 'TIGR02997', 'TIGR03453', 'TIGR03905', 'TIGR04356',
     'TIGR00243', 'TIGR00713', 'TIGR01178', 'TIGR01634', 'TIGR02089', 'TIGR02541', 'TIGR02998', 'TIGR03454',
     'TIGR03906', 'TIGR04357', 'TIGR00244', 'TIGR00714', 'TIGR01179', 'TIGR01635', 'TIGR02090', 'TIGR02542',
     'TIGR02999', 'TIGR03455', 'TIGR03907', 'TIGR04358', 'TIGR00245', 'TIGR00715', 'TIGR01180', 'TIGR01636',
     'TIGR02091', 'TIGR02543', 'TIGR03000', 'TIGR03457', 'TIGR03908', 'TIGR04359', 'TIGR00246', 'TIGR00716',
     'TIGR01181', 'TIGR01637', 'TIGR02092', 'TIGR02544', 'TIGR03001', 'TIGR03458', 'TIGR03909', 'TIGR04360',
     'TIGR00247', 'TIGR00717', 'TIGR01182', 'TIGR01638', 'TIGR02093', 'TIGR02546', 'TIGR03002', 'TIGR03459',
     'TIGR03910', 'TIGR04361', 'TIGR00249', 'TIGR00718', 'TIGR01183', 'TIGR01639', 'TIGR02094', 'TIGR02547',
     'TIGR03003', 'TIGR03460', 'TIGR03911', 'TIGR04362', 'TIGR00250', 'TIGR00719', 'TIGR01184', 'TIGR01640',
     'TIGR02095', 'TIGR02548', 'TIGR03004', 'TIGR03461', 'TIGR03912', 'TIGR04363', 'TIGR00251', 'TIGR00720',
     'TIGR01185', 'TIGR01641', 'TIGR02096', 'TIGR02549', 'TIGR03005', 'TIGR03462', 'TIGR03913', 'TIGR04364',
     'TIGR00252', 'TIGR00721', 'TIGR01186', 'TIGR01642', 'TIGR02097', 'TIGR02550', 'TIGR03006', 'TIGR03463',
     'TIGR03914', 'TIGR04365', 'TIGR00253', 'TIGR00722', 'TIGR01187', 'TIGR01643', 'TIGR02098', 'TIGR02551',
     'TIGR03007', 'TIGR03464', 'TIGR03915', 'TIGR04366', 'TIGR00254', 'TIGR00723', 'TIGR01188', 'TIGR01644',
     'TIGR02099', 'TIGR02552', 'TIGR03008', 'TIGR03465', 'TIGR03916', 'TIGR04367', 'TIGR00255', 'TIGR00724',
     'TIGR01189', 'TIGR01645', 'TIGR02100', 'TIGR02553', 'TIGR03009', 'TIGR03466', 'TIGR03917', 'TIGR04368',
     'TIGR00256', 'TIGR00725', 'TIGR01190', 'TIGR01646', 'TIGR02101', 'TIGR02554', 'TIGR03010', 'TIGR03467',
     'TIGR03918', 'TIGR04369', 'TIGR00257', 'TIGR00726', 'TIGR01191', 'TIGR01647', 'TIGR02102', 'TIGR02555',
     'TIGR03011', 'TIGR03468', 'TIGR03919', 'TIGR04370', 'TIGR00258', 'TIGR00727', 'TIGR01192', 'TIGR01648',
     'TIGR02103', 'TIGR02556', 'TIGR03012', 'TIGR03469', 'TIGR03920', 'TIGR04371', 'TIGR00259', 'TIGR00728',
     'TIGR01193', 'TIGR01649', 'TIGR02104', 'TIGR02557', 'TIGR03013', 'TIGR03470', 'TIGR03921', 'TIGR04372',
     'TIGR00260', 'TIGR00729', 'TIGR01194', 'TIGR01650', 'TIGR02105', 'TIGR02558', 'TIGR03014', 'TIGR03471',
     'TIGR03922', 'TIGR04373', 'TIGR00261', 'TIGR00730', 'TIGR01195', 'TIGR01651', 'TIGR02106', 'TIGR02559',
     'TIGR03015', 'TIGR03472', 'TIGR03923', 'TIGR04374', 'TIGR00262', 'TIGR00731', 'TIGR01196', 'TIGR01652',
     'TIGR02107', 'TIGR02560', 'TIGR03016', 'TIGR03473', 'TIGR03924', 'TIGR04375', 'TIGR00263', 'TIGR00732',
     'TIGR01197', 'TIGR01653', 'TIGR02108', 'TIGR02561', 'TIGR03017', 'TIGR03474', 'TIGR03925', 'TIGR04376',
     'TIGR00264', 'TIGR00733', 'TIGR01198', 'TIGR01654', 'TIGR02109', 'TIGR02562', 'TIGR03018', 'TIGR03475',
     'TIGR03926', 'TIGR04377', 'TIGR00266', 'TIGR00734', 'TIGR01200', 'TIGR01655', 'TIGR02110', 'TIGR02563',
     'TIGR03019', 'TIGR03476', 'TIGR03927', 'TIGR04378', 'TIGR00267', 'TIGR00735', 'TIGR01201', 'TIGR01656',
     'TIGR02111', 'TIGR02564', 'TIGR03020', 'TIGR03477', 'TIGR03928', 'TIGR04379', 'TIGR00268', 'TIGR00736',
     'TIGR01202', 'TIGR01657', 'TIGR02112', 'TIGR02565', 'TIGR03021', 'TIGR03478', 'TIGR03929', 'TIGR04380',
     'TIGR00269', 'TIGR00737', 'TIGR01203', 'TIGR01658', 'TIGR02113', 'TIGR02566', 'TIGR03022', 'TIGR03479',
     'TIGR03930', 'TIGR04381', 'TIGR00270', 'TIGR00738', 'TIGR01204', 'TIGR01659', 'TIGR02114', 'TIGR02567',
     'TIGR03023', 'TIGR03480', 'TIGR03931', 'TIGR04382', 'TIGR00271', 'TIGR00739', 'TIGR01205', 'TIGR01660',
     'TIGR02115', 'TIGR02568', 'TIGR03024', 'TIGR03481', 'TIGR03932', 'TIGR04383', 'TIGR00272', 'TIGR00740',
     'TIGR01206', 'TIGR01661', 'TIGR02116', 'TIGR02569', 'TIGR03025', 'TIGR03482', 'TIGR03933', 'TIGR04384',
     'TIGR00273', 'TIGR00741', 'TIGR01207', 'TIGR01662', 'TIGR02117', 'TIGR02570', 'TIGR03026', 'TIGR03483',
     'TIGR03934', 'TIGR04385', 'TIGR00274', 'TIGR00742', 'TIGR01208', 'TIGR01663', 'TIGR02118', 'TIGR02571',
     'TIGR03027', 'TIGR03485', 'TIGR03935', 'TIGR04386', 'TIGR00275', 'TIGR00743', 'TIGR01209', 'TIGR01664',
     'TIGR02119', 'TIGR02572', 'TIGR03028', 'TIGR03486', 'TIGR03936', 'TIGR04387', 'TIGR00276', 'TIGR00744',
     'TIGR01210', 'TIGR01665', 'TIGR02120', 'TIGR02573', 'TIGR03029', 'TIGR03487', 'TIGR03937', 'TIGR04388',
     'TIGR00277', 'TIGR00745', 'TIGR01211', 'TIGR01666', 'TIGR02121', 'TIGR02574', 'TIGR03030', 'TIGR03488',
     'TIGR03938', 'TIGR04389', 'TIGR00278', 'TIGR00746', 'TIGR01212', 'TIGR01667', 'TIGR02122', 'TIGR02577',
     'TIGR03031', 'TIGR03489', 'TIGR03939', 'TIGR04390', 'TIGR00279', 'TIGR00747', 'TIGR01213', 'TIGR01668',
     'TIGR02123', 'TIGR02578', 'TIGR03032', 'TIGR03490', 'TIGR03940', 'TIGR04391', 'TIGR00280', 'TIGR00748',
     'TIGR01214', 'TIGR01669', 'TIGR02124', 'TIGR02579', 'TIGR03033', 'TIGR03491', 'TIGR03941', 'TIGR04392',
     'TIGR00281', 'TIGR00749', 'TIGR01215', 'TIGR01670', 'TIGR02125', 'TIGR02580', 'TIGR03034', 'TIGR03492',
     'TIGR03942', 'TIGR04393', 'TIGR00282', 'TIGR00750', 'TIGR01216', 'TIGR01671', 'TIGR02126', 'TIGR02581',
     'TIGR03035', 'TIGR03493', 'TIGR03943', 'TIGR04394', 'TIGR00283', 'TIGR00751', 'TIGR01217', 'TIGR01672',
     'TIGR02127', 'TIGR02582', 'TIGR03036', 'TIGR03494', 'TIGR03944', 'TIGR04395', 'TIGR00284', 'TIGR00752',
     'TIGR01218', 'TIGR01673', 'TIGR02128', 'TIGR02583', 'TIGR03037', 'TIGR03495', 'TIGR03945', 'TIGR04396',
     'TIGR00285', 'TIGR00753', 'TIGR01219', 'TIGR01674', 'TIGR02129', 'TIGR02584', 'TIGR03038', 'TIGR03496',
     'TIGR03946', 'TIGR04397', 'TIGR00286', 'TIGR00754', 'TIGR01220', 'TIGR01675', 'TIGR02130', 'TIGR02585',
     'TIGR03039', 'TIGR03497', 'TIGR03947', 'TIGR04398', 'TIGR00287', 'TIGR00755', 'TIGR01221', 'TIGR01676',
     'TIGR02131', 'TIGR02586', 'TIGR03041', 'TIGR03498', 'TIGR03948', 'TIGR04399', 'TIGR00288', 'TIGR00756',
     'TIGR01222', 'TIGR01677', 'TIGR02132', 'TIGR02587', 'TIGR03042', 'TIGR03499', 'TIGR03949', 'TIGR04400',
     'TIGR00289', 'TIGR00757', 'TIGR01223', 'TIGR01678', 'TIGR02133', 'TIGR02588', 'TIGR03043', 'TIGR03500',
     'TIGR03950', 'TIGR04401', 'TIGR00290', 'TIGR00758', 'TIGR01224', 'TIGR01679', 'TIGR02134', 'TIGR02589',
     'TIGR03044', 'TIGR03501', 'TIGR03951', 'TIGR04402', 'TIGR00291', 'TIGR00759', 'TIGR01225', 'TIGR01680',
     'TIGR02135', 'TIGR02590', 'TIGR03045', 'TIGR03502', 'TIGR03952', 'TIGR04403', 'TIGR00292', 'TIGR00760',
     'TIGR01226', 'TIGR01681', 'TIGR02136', 'TIGR02591', 'TIGR03046', 'TIGR03503', 'TIGR03953', 'TIGR04404',
     'TIGR00293', 'TIGR00761', 'TIGR01227', 'TIGR01682', 'TIGR02137', 'TIGR02592', 'TIGR03047', 'TIGR03504',
     'TIGR03954', 'TIGR04405', 'TIGR00294', 'TIGR00762', 'TIGR01228', 'TIGR01683', 'TIGR02138', 'TIGR02593',
     'TIGR03048', 'TIGR03505', 'TIGR03955', 'TIGR04406', 'TIGR00295', 'TIGR00763', 'TIGR01229', 'TIGR01684',
     'TIGR02139', 'TIGR02594', 'TIGR03049', 'TIGR03506', 'TIGR03956', 'TIGR04407', 'TIGR00296', 'TIGR00764',
     'TIGR01230', 'TIGR01685', 'TIGR02140', 'TIGR02595', 'TIGR03050', 'TIGR03507', 'TIGR03957', 'TIGR04408',
     'TIGR00297', 'TIGR00765', 'TIGR01231', 'TIGR01686', 'TIGR02141', 'TIGR02596', 'TIGR03051', 'TIGR03508',
     'TIGR03958', 'TIGR04409', 'TIGR00298', 'TIGR00766', 'TIGR01232', 'TIGR01687', 'TIGR02142', 'TIGR02597',
     'TIGR03052', 'TIGR03509', 'TIGR03959', 'TIGR04410', 'TIGR00299', 'TIGR00767', 'TIGR01233', 'TIGR01688',
     'TIGR02143', 'TIGR02598', 'TIGR03053', 'TIGR03510', 'TIGR03960', 'TIGR04411', 'TIGR00300', 'TIGR00768',
     'TIGR01234', 'TIGR01689', 'TIGR02144', 'TIGR02599', 'TIGR03054', 'TIGR03511', 'TIGR03961', 'TIGR04412',
     'TIGR00302', 'TIGR00769', 'TIGR01235', 'TIGR01690', 'TIGR02145', 'TIGR02600', 'TIGR03055', 'TIGR03512',
     'TIGR03962', 'TIGR04413', 'TIGR00303', 'TIGR00770', 'TIGR01236', 'TIGR01691', 'TIGR02146', 'TIGR02601',
     'TIGR03056', 'TIGR03513', 'TIGR03963', 'TIGR04414', 'TIGR00304', 'TIGR00771', 'TIGR01237', 'TIGR01692',
     'TIGR02147', 'TIGR02602', 'TIGR03057', 'TIGR03514', 'TIGR03964', 'TIGR04415', 'TIGR00305', 'TIGR00773',
     'TIGR01238', 'TIGR01693', 'TIGR02148', 'TIGR02603', 'TIGR03058', 'TIGR03515', 'TIGR03965', 'TIGR04416',
     'TIGR00306', 'TIGR00774', 'TIGR01239', 'TIGR01694', 'TIGR02149', 'TIGR02604', 'TIGR03059', 'TIGR03516',
     'TIGR03966', 'TIGR04417', 'TIGR00307', 'TIGR00775', 'TIGR01240', 'TIGR01695', 'TIGR02150', 'TIGR02605',
     'TIGR03060', 'TIGR03517', 'TIGR03967', 'TIGR04418', 'TIGR00308', 'TIGR00776', 'TIGR01241', 'TIGR01696',
     'TIGR02151', 'TIGR02606', 'TIGR03061', 'TIGR03518', 'TIGR03968', 'TIGR04419', 'TIGR00309', 'TIGR00777',
     'TIGR01242', 'TIGR01697', 'TIGR02152', 'TIGR02607', 'TIGR03062', 'TIGR03519', 'TIGR03969', 'TIGR04420',
     'TIGR00310', 'TIGR00778', 'TIGR01243', 'TIGR01698', 'TIGR02153', 'TIGR02608', 'TIGR03063', 'TIGR03520',
     'TIGR03970', 'TIGR04421', 'TIGR00311', 'TIGR00779', 'TIGR01244', 'TIGR01699', 'TIGR02154', 'TIGR02609',
     'TIGR03064', 'TIGR03521', 'TIGR03971', 'TIGR04422', 'TIGR00312', 'TIGR00780', 'TIGR01245', 'TIGR01700',
     'TIGR02155', 'TIGR02610', 'TIGR03065', 'TIGR03522', 'TIGR03972', 'TIGR04423', 'TIGR00313', 'TIGR00781',
     'TIGR01246', 'TIGR01701', 'TIGR02156', 'TIGR02611', 'TIGR03066', 'TIGR03523', 'TIGR03973', 'TIGR04424',
     'TIGR00314', 'TIGR00782', 'TIGR01247', 'TIGR01702', 'TIGR02157', 'TIGR02612', 'TIGR03067', 'TIGR03524',
     'TIGR03974', 'TIGR04425', 'TIGR00315', 'TIGR00783', 'TIGR01248', 'TIGR01703', 'TIGR02158', 'TIGR02613',
     'TIGR03068', 'TIGR03525', 'TIGR03975', 'TIGR04426', 'TIGR00316', 'TIGR00784', 'TIGR01249', 'TIGR01704',
     'TIGR02159', 'TIGR02614', 'TIGR03069', 'TIGR03526', 'TIGR03976', 'TIGR04427', 'TIGR00317', 'TIGR00785',
     'TIGR01250', 'TIGR01705', 'TIGR02160', 'TIGR02615', 'TIGR03070', 'TIGR03527', 'TIGR03977', 'TIGR04428',
     'TIGR00318', 'TIGR00786', 'TIGR01251', 'TIGR01706', 'TIGR02161', 'TIGR02616', 'TIGR03071', 'TIGR03528',
     'TIGR03978', 'TIGR04429', 'TIGR00319', 'TIGR00787', 'TIGR01252', 'TIGR01707', 'TIGR02162', 'TIGR02617',
     'TIGR03072', 'TIGR03529', 'TIGR03979', 'TIGR04430', 'TIGR00320', 'TIGR00788', 'TIGR01253', 'TIGR01708',
     'TIGR02163', 'TIGR02618', 'TIGR03073', 'TIGR03530', 'TIGR03980', 'TIGR04431', 'TIGR00321', 'TIGR00789',
     'TIGR01254', 'TIGR01709', 'TIGR02164', 'TIGR02619', 'TIGR03074', 'TIGR03531', 'TIGR03981', 'TIGR04432',
     'TIGR00322', 'TIGR00790', 'TIGR01255', 'TIGR01710', 'TIGR02165', 'TIGR02620', 'TIGR03075', 'TIGR03532',
     'TIGR03982', 'TIGR04433', 'TIGR00323', 'TIGR00791', 'TIGR01256', 'TIGR01711', 'TIGR02166', 'TIGR02621',
     'TIGR03076', 'TIGR03533', 'TIGR03983', 'TIGR04434', 'TIGR00324', 'TIGR00792', 'TIGR01257', 'TIGR01712',
     'TIGR02167', 'TIGR02622', 'TIGR03077', 'TIGR03534', 'TIGR03984', 'TIGR04435', 'TIGR00325', 'TIGR00793',
     'TIGR01258', 'TIGR01713', 'TIGR02168', 'TIGR02623', 'TIGR03078', 'TIGR03535', 'TIGR03985', 'TIGR04436',
     'TIGR00326', 'TIGR00794', 'TIGR01259', 'TIGR01714', 'TIGR02169', 'TIGR02624', 'TIGR03079', 'TIGR03536',
     'TIGR03986', 'TIGR04437', 'TIGR00327', 'TIGR00795', 'TIGR01260', 'TIGR01715', 'TIGR02170', 'TIGR02625',
     'TIGR03080', 'TIGR03537', 'TIGR03987', 'TIGR04438', 'TIGR00328', 'TIGR00796', 'TIGR01261', 'TIGR01716',
     'TIGR02171', 'TIGR02627', 'TIGR03081', 'TIGR03538', 'TIGR03988', 'TIGR04439', 'TIGR00329', 'TIGR00797',
     'TIGR01262', 'TIGR01717', 'TIGR02172', 'TIGR02628', 'TIGR03082', 'TIGR03539', 'TIGR03989', 'TIGR04440',
     'TIGR00330', 'TIGR00798', 'TIGR01263', 'TIGR01718', 'TIGR02173', 'TIGR02629', 'TIGR03083', 'TIGR03540',
     'TIGR03990', 'TIGR04441', 'TIGR00331', 'TIGR00799', 'TIGR01264', 'TIGR01719', 'TIGR02174', 'TIGR02630',
     'TIGR03084', 'TIGR03541', 'TIGR03991', 'TIGR04442', 'TIGR00332', 'TIGR00800', 'TIGR01265', 'TIGR01720',
     'TIGR02175', 'TIGR02631', 'TIGR03085', 'TIGR03542', 'TIGR03992', 'TIGR04443', 'TIGR00333', 'TIGR00801',
     'TIGR01266', 'TIGR01721', 'TIGR02176', 'TIGR02632', 'TIGR03086', 'TIGR03543', 'TIGR03993', 'TIGR04444',
     'TIGR00334', 'TIGR00802', 'TIGR01267', 'TIGR01722', 'TIGR02177', 'TIGR02633', 'TIGR03087', 'TIGR03544',
     'TIGR03994', 'TIGR04445', 'TIGR00335', 'TIGR00803', 'TIGR01268', 'TIGR01723', 'TIGR02178', 'TIGR02634',
     'TIGR03088', 'TIGR03545', 'TIGR03995', 'TIGR04446', 'TIGR00336', 'TIGR00804', 'TIGR01269', 'TIGR01724',
     'TIGR02179', 'TIGR02635', 'TIGR03089', 'TIGR03546', 'TIGR03996', 'TIGR04447', 'TIGR00337', 'TIGR00805',
     'TIGR01270', 'TIGR01725', 'TIGR02180', 'TIGR02636', 'TIGR03090', 'TIGR03547', 'TIGR03997', 'TIGR04448',
     'TIGR00338', 'TIGR00806', 'TIGR01271', 'TIGR01726', 'TIGR02181', 'TIGR02637', 'TIGR03091', 'TIGR03548',
     'TIGR03998', 'TIGR04449', 'TIGR00339', 'TIGR00807', 'TIGR01272', 'TIGR01727', 'TIGR02182', 'TIGR02638',
     'TIGR03092', 'TIGR03549', 'TIGR03999', 'TIGR04450', 'TIGR00340', 'TIGR00808', 'TIGR01273', 'TIGR01728',
     'TIGR02183', 'TIGR02639', 'TIGR03093', 'TIGR03550', 'TIGR04000', 'TIGR04451', 'TIGR00341', 'TIGR00809',
     'TIGR01274', 'TIGR01729', 'TIGR02184', 'TIGR02640', 'TIGR03094', 'TIGR03551', 'TIGR04001', 'TIGR04452',
     'TIGR00342', 'TIGR00810', 'TIGR01275', 'TIGR01730', 'TIGR02185', 'TIGR02641', 'TIGR03095', 'TIGR03552',
     'TIGR04002', 'TIGR04453', 'TIGR00343', 'TIGR00811', 'TIGR01276', 'TIGR01731', 'TIGR02186', 'TIGR02642',
     'TIGR03096', 'TIGR03553', 'TIGR04003', 'TIGR04454', 'TIGR00344', 'TIGR00813', 'TIGR01277', 'TIGR01732',
     'TIGR02187', 'TIGR02643', 'TIGR03097', 'TIGR03554', 'TIGR04004', 'TIGR04455', 'TIGR00345', 'TIGR00814',
     'TIGR01278', 'TIGR01733', 'TIGR02188', 'TIGR02644', 'TIGR03098', 'TIGR03555', 'TIGR04005', 'TIGR04456',
     'TIGR00346', 'TIGR00815', 'TIGR01279', 'TIGR01734', 'TIGR02189', 'TIGR02645', 'TIGR03099', 'TIGR03556',
     'TIGR04006', 'TIGR04457', 'TIGR00347', 'TIGR00816', 'TIGR01280', 'TIGR01735', 'TIGR02190', 'TIGR02646',
     'TIGR03100', 'TIGR03557', 'TIGR04007', 'TIGR04458', 'TIGR00348', 'TIGR00817', 'TIGR01281', 'TIGR01736',
     'TIGR02191', 'TIGR02647', 'TIGR03101', 'TIGR03558', 'TIGR04008', 'TIGR04459', 'TIGR00350', 'TIGR00819',
     'TIGR01282', 'TIGR01737', 'TIGR02192', 'TIGR02648', 'TIGR03102', 'TIGR03559', 'TIGR04009', 'TIGR04460',
     'TIGR00351', 'TIGR00820', 'TIGR01283', 'TIGR01738', 'TIGR02193', 'TIGR02649', 'TIGR03103', 'TIGR03560',
     'TIGR04010', 'TIGR04461', 'TIGR00353', 'TIGR00821', 'TIGR01284', 'TIGR01739', 'TIGR02194', 'TIGR02650',
     'TIGR03104', 'TIGR03561', 'TIGR04011', 'TIGR04462', 'TIGR00354', 'TIGR00822', 'TIGR01285', 'TIGR01740',
     'TIGR02195', 'TIGR02651', 'TIGR03105', 'TIGR03562', 'TIGR04012', 'TIGR04463', 'TIGR00355', 'TIGR00823',
     'TIGR01286', 'TIGR01741', 'TIGR02196', 'TIGR02652', 'TIGR03106', 'TIGR03563', 'TIGR04013', 'TIGR04464',
     'TIGR00357', 'TIGR00824', 'TIGR01287', 'TIGR01742', 'TIGR02197', 'TIGR02653', 'TIGR03107', 'TIGR03564',
     'TIGR04014', 'TIGR04465', 'TIGR00358', 'TIGR00825', 'TIGR01288', 'TIGR01743', 'TIGR02198', 'TIGR02654',
     'TIGR03108', 'TIGR03565', 'TIGR04015', 'TIGR04466', 'TIGR00359', 'TIGR00826', 'TIGR01289', 'TIGR01744',
     'TIGR02199', 'TIGR02655', 'TIGR03109', 'TIGR03566', 'TIGR04016', 'TIGR04467', 'TIGR00360', 'TIGR00827',
     'TIGR01290', 'TIGR01745', 'TIGR02200', 'TIGR02656', 'TIGR03110', 'TIGR03567', 'TIGR04017', 'TIGR04468',
     'TIGR00361', 'TIGR00828', 'TIGR01291', 'TIGR01746', 'TIGR02201', 'TIGR02657', 'TIGR03111', 'TIGR03568',
     'TIGR04018', 'TIGR04469', 'TIGR00362', 'TIGR00829', 'TIGR01292', 'TIGR01747', 'TIGR02202', 'TIGR02658',
     'TIGR03112', 'TIGR03569', 'TIGR04019', 'TIGR04470', 'TIGR00363', 'TIGR00830', 'TIGR01293', 'TIGR01748',
     'TIGR02203', 'TIGR02659', 'TIGR03113', 'TIGR03570', 'TIGR04020', 'TIGR04471', 'TIGR00364', 'TIGR00831',
     'TIGR01294', 'TIGR01749', 'TIGR02204', 'TIGR02660', 'TIGR03114', 'TIGR03571', 'TIGR04021', 'TIGR04472',
     'TIGR00365', 'TIGR00832', 'TIGR01295', 'TIGR01750', 'TIGR02205', 'TIGR02661', 'TIGR03115', 'TIGR03572',
     'TIGR04022', 'TIGR04473', 'TIGR00366', 'TIGR00833', 'TIGR01296', 'TIGR01751', 'TIGR02206', 'TIGR02662',
     'TIGR03116', 'TIGR03573', 'TIGR04023', 'TIGR04474', 'TIGR00367', 'TIGR00834', 'TIGR01297', 'TIGR01752',
     'TIGR02207', 'TIGR02663', 'TIGR03117', 'TIGR03574', 'TIGR04024', 'TIGR04475', 'TIGR00368', 'TIGR00835',
     'TIGR01298', 'TIGR01753', 'TIGR02208', 'TIGR02664', 'TIGR03118', 'TIGR03575', 'TIGR04025', 'TIGR04476',
     'TIGR00369', 'TIGR00836', 'TIGR01299', 'TIGR01754', 'TIGR02209', 'TIGR02665', 'TIGR03119', 'TIGR03576',
     'TIGR04026', 'TIGR04477', 'TIGR00370', 'TIGR00837', 'TIGR01300', 'TIGR01755', 'TIGR02210', 'TIGR02666',
     'TIGR03120', 'TIGR03577', 'TIGR04027', 'TIGR04478', 'TIGR00372', 'TIGR00838', 'TIGR01301', 'TIGR01756',
     'TIGR02211', 'TIGR02667', 'TIGR03121', 'TIGR03578', 'TIGR04028', 'TIGR04479', 'TIGR00373', 'TIGR00839',
     'TIGR01302', 'TIGR01757', 'TIGR02212', 'TIGR02668', 'TIGR03122', 'TIGR03579', 'TIGR04029', 'TIGR04480',
     'TIGR00374', 'TIGR00840', 'TIGR01303', 'TIGR01758', 'TIGR02213', 'TIGR02669', 'TIGR03123', 'TIGR03580',
     'TIGR04030', 'TIGR04481', 'TIGR00375', 'TIGR00841', 'TIGR01304', 'TIGR01759', 'TIGR02214', 'TIGR02670',
     'TIGR03124', 'TIGR03581', 'TIGR04031', 'TIGR04482', 'TIGR00376', 'TIGR00842', 'TIGR01305', 'TIGR01760',
     'TIGR02215', 'TIGR02671', 'TIGR03125', 'TIGR03582', 'TIGR04032', 'TIGR04483', 'TIGR00377', 'TIGR00843',
     'TIGR01306', 'TIGR01761', 'TIGR02216', 'TIGR02672', 'TIGR03126', 'TIGR03583', 'TIGR04033', 'TIGR04484',
     'TIGR00378', 'TIGR00844', 'TIGR01307', 'TIGR01762', 'TIGR02217', 'TIGR02673', 'TIGR03127', 'TIGR03584',
     'TIGR04034', 'TIGR04485', 'TIGR00379', 'TIGR00845', 'TIGR01308', 'TIGR01763', 'TIGR02218', 'TIGR02674',
     'TIGR03128', 'TIGR03585', 'TIGR04035', 'TIGR04486', 'TIGR00380', 'TIGR00846', 'TIGR01309', 'TIGR01764',
     'TIGR02219', 'TIGR02675', 'TIGR03129', 'TIGR03586', 'TIGR04036', 'TIGR04487', 'TIGR00381', 'TIGR00847',
     'TIGR01310', 'TIGR01765', 'TIGR02220', 'TIGR02677', 'TIGR03130', 'TIGR03587', 'TIGR04037', 'TIGR04488',
     'TIGR00382', 'TIGR00848', 'TIGR01311', 'TIGR01766', 'TIGR02221', 'TIGR02678', 'TIGR03131', 'TIGR03588',
     'TIGR04038', 'TIGR04489', 'TIGR00383', 'TIGR00849', 'TIGR01312', 'TIGR01767', 'TIGR02222', 'TIGR02679',
     'TIGR03132', 'TIGR03589', 'TIGR04039', 'TIGR04490', 'TIGR00384', 'TIGR00851', 'TIGR01313', 'TIGR01768',
     'TIGR02223', 'TIGR02680', 'TIGR03133', 'TIGR03590', 'TIGR04040', 'TIGR04491', 'TIGR00385', 'TIGR00852',
     'TIGR01314', 'TIGR01769', 'TIGR02224', 'TIGR02681', 'TIGR03134', 'TIGR03591', 'TIGR04041', 'TIGR04492',
     'TIGR00387', 'TIGR00853', 'TIGR01315', 'TIGR01770', 'TIGR02225', 'TIGR02682', 'TIGR03135', 'TIGR03592',
     'TIGR04042', 'TIGR04493', 'TIGR00388', 'TIGR00854', 'TIGR01316', 'TIGR01771', 'TIGR02226', 'TIGR02683',
     'TIGR03136', 'TIGR03593', 'TIGR04043', 'TIGR04494', 'TIGR00389', 'TIGR00855', 'TIGR01317', 'TIGR01772',
     'TIGR02227', 'TIGR02684', 'TIGR03137', 'TIGR03594', 'TIGR04044', 'TIGR04495', 'TIGR00390', 'TIGR00856',
     'TIGR01318', 'TIGR01773', 'TIGR02228', 'TIGR02685', 'TIGR03138', 'TIGR03595', 'TIGR04045', 'TIGR04496',
     'TIGR00391', 'TIGR00857', 'TIGR01319', 'TIGR01774', 'TIGR02229', 'TIGR02686', 'TIGR03139', 'TIGR03596',
     'TIGR04046', 'TIGR04497', 'TIGR00392', 'TIGR00858', 'TIGR01320', 'TIGR01776', 'TIGR02230', 'TIGR02687',
     'TIGR03140', 'TIGR03597', 'TIGR04047', 'TIGR04498', 'TIGR00393', 'TIGR00859', 'TIGR01321', 'TIGR01777',
     'TIGR02231', 'TIGR02688', 'TIGR03141', 'TIGR03598', 'TIGR04048', 'TIGR04499', 'TIGR00394', 'TIGR00860',
     'TIGR01322', 'TIGR01778', 'TIGR02232', 'TIGR02689', 'TIGR03142', 'TIGR03599', 'TIGR04049', 'TIGR04500',
     'TIGR00395', 'TIGR00861', 'TIGR01323', 'TIGR01779', 'TIGR02234', 'TIGR02690', 'TIGR03143', 'TIGR03600',
     'TIGR04050', 'TIGR04501', 'TIGR00396', 'TIGR00862', 'TIGR01324', 'TIGR01780', 'TIGR02235', 'TIGR02691',
     'TIGR03144', 'TIGR03601', 'TIGR04051', 'TIGR04502', 'TIGR00397', 'TIGR00863', 'TIGR01325', 'TIGR01781',
     'TIGR02236', 'TIGR02692', 'TIGR03145', 'TIGR03602', 'TIGR04052', 'TIGR04503', 'TIGR00398', 'TIGR00864',
     'TIGR01326', 'TIGR01782', 'TIGR02237', 'TIGR02693', 'TIGR03146', 'TIGR03603', 'TIGR04053', 'TIGR04504',
     'TIGR00399', 'TIGR00865', 'TIGR01327', 'TIGR01783', 'TIGR02238', 'TIGR02694', 'TIGR03147', 'TIGR03604',
     'TIGR04054', 'TIGR04505', 'TIGR00400', 'TIGR00867', 'TIGR01328', 'TIGR01784', 'TIGR02239', 'TIGR02695',
     'TIGR03148', 'TIGR03605', 'TIGR04055', 'TIGR04506', 'TIGR00401', 'TIGR00868', 'TIGR01329', 'TIGR01785',
     'TIGR02240', 'TIGR02696', 'TIGR03149', 'TIGR03606', 'TIGR04056', 'TIGR04507', 'TIGR00402', 'TIGR00869',
     'TIGR01330', 'TIGR01786', 'TIGR02241', 'TIGR02697', 'TIGR03150', 'TIGR03607', 'TIGR04057', 'TIGR04508',
     'TIGR00403', 'TIGR00870', 'TIGR01331', 'TIGR01787', 'TIGR02242', 'TIGR02698', 'TIGR03151', 'TIGR03608',
     'TIGR04058', 'TIGR04509', 'TIGR00405', 'TIGR00871', 'TIGR01332', 'TIGR01788', 'TIGR02243', 'TIGR02699',
     'TIGR03152', 'TIGR03609', 'TIGR04059', 'TIGR04510', 'TIGR00406', 'TIGR00872', 'TIGR01333', 'TIGR01789',
     'TIGR02244', 'TIGR02700', 'TIGR03153', 'TIGR03610', 'TIGR04060', 'TIGR04511', 'TIGR00407', 'TIGR00873',
     'TIGR01334', 'TIGR01790', 'TIGR02245', 'TIGR02701', 'TIGR03154', 'TIGR03611', 'TIGR04061', 'TIGR04512',
     'TIGR00408', 'TIGR00874', 'TIGR01335', 'TIGR01791', 'TIGR02246', 'TIGR02702', 'TIGR03155', 'TIGR03612',
     'TIGR04062', 'TIGR04513', 'TIGR00409', 'TIGR00875', 'TIGR01336', 'TIGR01792', 'TIGR02247', 'TIGR02703',
     'TIGR03156', 'TIGR03613', 'TIGR04063', 'TIGR04514', 'TIGR00410', 'TIGR00876', 'TIGR01337', 'TIGR01793',
     'TIGR02248', 'TIGR02704', 'TIGR03157', 'TIGR03614', 'TIGR04064', 'TIGR04515', 'TIGR00411', 'TIGR00877',
     'TIGR01338', 'TIGR01795', 'TIGR02249', 'TIGR02705', 'TIGR03158', 'TIGR03615', 'TIGR04065', 'TIGR04516',
     'TIGR00412', 'TIGR00878', 'TIGR01339', 'TIGR01796', 'TIGR02250', 'TIGR02706', 'TIGR03159', 'TIGR03616',
     'TIGR04066', 'TIGR04517', 'TIGR00413', 'TIGR00879', 'TIGR01340', 'TIGR01797', 'TIGR02251', 'TIGR02707',
     'TIGR03160', 'TIGR03617', 'TIGR04067', 'TIGR04518', 'TIGR00414', 'TIGR00880', 'TIGR01341', 'TIGR01798',
     'TIGR02252', 'TIGR02708', 'TIGR03161', 'TIGR03618', 'TIGR04068', 'TIGR04519', 'TIGR00415', 'TIGR00881',
     'TIGR01342', 'TIGR01799', 'TIGR02253', 'TIGR02709', 'TIGR03162', 'TIGR03619', 'TIGR04069', 'TIGR04520',
     'TIGR00416', 'TIGR00882', 'TIGR01343', 'TIGR01800', 'TIGR02254', 'TIGR02710', 'TIGR03164', 'TIGR03620',
     'TIGR04070', 'TIGR04521', 'TIGR00417', 'TIGR00883', 'TIGR01344', 'TIGR01801', 'TIGR02256', 'TIGR02711',
     'TIGR03165', 'TIGR03621', 'TIGR04071', 'TIGR04522', 'TIGR00418', 'TIGR00884', 'TIGR01345', 'TIGR01802',
     'TIGR02257', 'TIGR02712', 'TIGR03166', 'TIGR03622', 'TIGR04072', 'TIGR04523', 'TIGR00419', 'TIGR00885',
     'TIGR01346', 'TIGR01803', 'TIGR02258', 'TIGR02713', 'TIGR03167', 'TIGR03623', 'TIGR04073', 'TIGR04524',
     'TIGR00420', 'TIGR00886', 'TIGR01347', 'TIGR01804', 'TIGR02259', 'TIGR02714', 'TIGR03168', 'TIGR03624',
     'TIGR04074', 'TIGR04525', 'TIGR00421', 'TIGR00887', 'TIGR01348', 'TIGR01805', 'TIGR02260', 'TIGR02715',
     'TIGR03169', 'TIGR03625', 'TIGR04075', 'TIGR04526', 'TIGR00422', 'TIGR00888', 'TIGR01349', 'TIGR01806',
     'TIGR02261', 'TIGR02716', 'TIGR03170', 'TIGR03626', 'TIGR04076', 'TIGR04527', 'TIGR00423', 'TIGR00889',
     'TIGR01350', 'TIGR01807', 'TIGR02262', 'TIGR02717', 'TIGR03171', 'TIGR03627', 'TIGR04077', 'TIGR04528',
     'TIGR00424', 'TIGR00890', 'TIGR01351', 'TIGR01808', 'TIGR02263', 'TIGR02718', 'TIGR03172', 'TIGR03628',
     'TIGR04078', 'TIGR04529', 'TIGR00425', 'TIGR00891', 'TIGR01352', 'TIGR01809', 'TIGR02264', 'TIGR02719',
     'TIGR03173', 'TIGR03629', 'TIGR04079', 'TIGR04530', 'TIGR00426', 'TIGR00892', 'TIGR01353', 'TIGR01810',
     'TIGR02265', 'TIGR02720', 'TIGR03174', 'TIGR03630', 'TIGR04080', 'TIGR04531', 'TIGR00427', 'TIGR00893',
     'TIGR01354', 'TIGR01811', 'TIGR02266', 'TIGR02721', 'TIGR03175', 'TIGR03631', 'TIGR04081', 'TIGR04532',
     'TIGR00430', 'TIGR00894', 'TIGR01355', 'TIGR01812', 'TIGR02267', 'TIGR02722', 'TIGR03176', 'TIGR03632',
     'TIGR04082', 'TIGR04533', 'TIGR00431', 'TIGR00895', 'TIGR01356', 'TIGR01813', 'TIGR02268', 'TIGR02723',
     'TIGR03177', 'TIGR03633', 'TIGR04083', 'TIGR04534', 'TIGR00432', 'TIGR00896', 'TIGR01357', 'TIGR01814',
     'TIGR02269', 'TIGR02724', 'TIGR03178', 'TIGR03634', 'TIGR04084', 'TIGR04535', 'TIGR00433', 'TIGR00897',
     'TIGR01358', 'TIGR01815', 'TIGR02270', 'TIGR02725', 'TIGR03180', 'TIGR03635', 'TIGR04085', 'TIGR04536',
     'TIGR00434', 'TIGR00898', 'TIGR01359', 'TIGR01816', 'TIGR02271', 'TIGR02726', 'TIGR03181', 'TIGR03636',
     'TIGR04086', 'TIGR04537', 'TIGR00435', 'TIGR00899', 'TIGR01360', 'TIGR01817', 'TIGR02272', 'TIGR02727',
     'TIGR03182', 'TIGR03637', 'TIGR04087', 'TIGR04538', 'TIGR00436', 'TIGR00900', 'TIGR01361', 'TIGR01818',
     'TIGR02273', 'TIGR02728', 'TIGR03183', 'TIGR03638', 'TIGR04088', 'TIGR04539', 'TIGR00437', 'TIGR00901',
     'TIGR01362', 'TIGR01819', 'TIGR02274', 'TIGR02729', 'TIGR03184', 'TIGR03639', 'TIGR04089', 'TIGR04540',
     'TIGR00438', 'TIGR00902', 'TIGR01363', 'TIGR01820', 'TIGR02275', 'TIGR02730', 'TIGR03185', 'TIGR03640',
     'TIGR04090', 'TIGR04541', 'TIGR00439', 'TIGR00903', 'TIGR01364', 'TIGR01821', 'TIGR02276', 'TIGR02731',
     'TIGR03186', 'TIGR03641', 'TIGR04091', 'TIGR04542', 'TIGR00440', 'TIGR00904', 'TIGR01365', 'TIGR01822',
     'TIGR02277', 'TIGR02732', 'TIGR03187', 'TIGR03642', 'TIGR04092', 'TIGR04543', 'TIGR00441', 'TIGR00905',
     'TIGR01366', 'TIGR01823', 'TIGR02278', 'TIGR02733', 'TIGR03188', 'TIGR03643', 'TIGR04093', 'TIGR04544',
     'TIGR00442', 'TIGR00906', 'TIGR01367', 'TIGR01824', 'TIGR02279', 'TIGR02734', 'TIGR03189', 'TIGR03644',
     'TIGR04094', 'TIGR04545', 'TIGR00443', 'TIGR00907', 'TIGR01368', 'TIGR01825', 'TIGR02280', 'TIGR02735',
     'TIGR03190', 'TIGR03645', 'TIGR04095', 'TIGR04546', 'TIGR00444', 'TIGR00908', 'TIGR01369', 'TIGR01826',
     'TIGR02281', 'TIGR02736', 'TIGR03191', 'TIGR03646', 'TIGR04096', 'TIGR04547', 'TIGR00445', 'TIGR00909',
     'TIGR01370', 'TIGR01827', 'TIGR02282', 'TIGR02737', 'TIGR03192', 'TIGR03647', 'TIGR04098', 'TIGR04548',
     'TIGR00446', 'TIGR00910', 'TIGR01371', 'TIGR01828', 'TIGR02283', 'TIGR02738', 'TIGR03193', 'TIGR03648',
     'TIGR04099', 'TIGR04549', 'TIGR00447', 'TIGR00911', 'TIGR01372', 'TIGR01829', 'TIGR02284', 'TIGR02739',
     'TIGR03194', 'TIGR03649', 'TIGR04100', 'TIGR04550', 'TIGR00448', 'TIGR00912', 'TIGR01373', 'TIGR01830',
     'TIGR02285', 'TIGR02740', 'TIGR03195', 'TIGR03650', 'TIGR04101', 'TIGR04551', 'TIGR00449', 'TIGR00913',
     'TIGR01374', 'TIGR01831', 'TIGR02286', 'TIGR02741', 'TIGR03196', 'TIGR03651', 'TIGR04102', 'TIGR04552',
     'TIGR00450', 'TIGR00914', 'TIGR01375', 'TIGR01832', 'TIGR02287', 'TIGR02742', 'TIGR03197', 'TIGR03652',
     'TIGR04103', 'TIGR04553', 'TIGR00451', 'TIGR00915', 'TIGR01376', 'TIGR01833', 'TIGR02288', 'TIGR02743',
     'TIGR03198', 'TIGR03653', 'TIGR04104', 'TIGR04554', 'TIGR00452', 'TIGR00916', 'TIGR01377', 'TIGR01834',
     'TIGR02289', 'TIGR02744', 'TIGR03199', 'TIGR03654', 'TIGR04105', 'TIGR04555', 'TIGR00453', 'TIGR00917',
     'TIGR01378', 'TIGR01835', 'TIGR02290', 'TIGR02745', 'TIGR03200', 'TIGR03655', 'TIGR04106', 'TIGR04556',
     'TIGR00454', 'TIGR00918', 'TIGR01379', 'TIGR01836', 'TIGR02291', 'TIGR02746', 'TIGR03201', 'TIGR03656',
     'TIGR04107', 'TIGR04557', 'TIGR00455', 'TIGR00920', 'TIGR01380', 'TIGR01837', 'TIGR02292', 'TIGR02747',
     'TIGR03202', 'TIGR03657', 'TIGR04108', 'TIGR04558', 'TIGR00456', 'TIGR00921', 'TIGR01381', 'TIGR01838',
     'TIGR02293', 'TIGR02748', 'TIGR03203', 'TIGR03658', 'TIGR04109', 'TIGR04559', 'TIGR00457', 'TIGR00922',
     'TIGR01382', 'TIGR01839', 'TIGR02294', 'TIGR02749', 'TIGR03204', 'TIGR03659', 'TIGR04110', 'TIGR04560',
     'TIGR00458', 'TIGR00924', 'TIGR01383', 'TIGR01840', 'TIGR02295', 'TIGR02750', 'TIGR03205', 'TIGR03660',
     'TIGR04111', 'TIGR04561', 'TIGR00459', 'TIGR00926', 'TIGR01384', 'TIGR01841', 'TIGR02296', 'TIGR02751',
     'TIGR03206', 'TIGR03661', 'TIGR04112', 'TIGR04562', 'TIGR00460', 'TIGR00927', 'TIGR01385', 'TIGR01842',
     'TIGR02297', 'TIGR02752', 'TIGR03207', 'TIGR03662', 'TIGR04113', 'TIGR04563', 'TIGR00461', 'TIGR00928',
     'TIGR01386', 'TIGR01843', 'TIGR02298', 'TIGR02753', 'TIGR03208', 'TIGR03663', 'TIGR04114', 'TIGR04564',
     'TIGR00462', 'TIGR00929', 'TIGR01387', 'TIGR01844', 'TIGR02299', 'TIGR02754', 'TIGR03209', 'TIGR03664',
     'TIGR04115', 'TIGR04565', 'TIGR00463', 'TIGR00930', 'TIGR01388', 'TIGR01845', 'TIGR02300', 'TIGR02755',
     'TIGR03210', 'TIGR03665', 'TIGR04116', 'TIGR04566', 'TIGR00464', 'TIGR00931', 'TIGR01389', 'TIGR01846',
     'TIGR02301', 'TIGR02756', 'TIGR03211', 'TIGR03666', 'TIGR04117', 'TIGR04567', 'TIGR00465', 'TIGR00932',
     'TIGR01390', 'TIGR01847', 'TIGR02302', 'TIGR02757', 'TIGR03212', 'TIGR03667', 'TIGR04118', 'TIGR04568',
     'TIGR00466', 'TIGR00933', 'TIGR01391', 'TIGR01848', 'TIGR02303', 'TIGR02758', 'TIGR03213', 'TIGR03668',
     'TIGR04119', 'TIGR04569', 'TIGR00467', 'TIGR00934', 'TIGR01392', 'TIGR01849', 'TIGR02304', 'TIGR02759',
     'TIGR03214', 'TIGR03669', 'TIGR04120', 'TIGR04570', 'TIGR00468', 'TIGR00935', 'TIGR01393', 'TIGR01850',
     'TIGR02305', 'TIGR02760', 'TIGR03215', 'TIGR03670', 'TIGR04121', 'TIGR04571', 'TIGR00469', 'TIGR00936',
     'TIGR01394', 'TIGR01851', 'TIGR02306', 'TIGR02761', 'TIGR03216', 'TIGR03671', 'TIGR04122', 'TIGR00470',
     'TIGR00937', 'TIGR01395', 'TIGR01852', 'TIGR02307', 'TIGR02762', 'TIGR03217', 'TIGR03672', 'TIGR04123'})
