import math

def nominal_to_continuous(R: float, m: int) -> float:
    """
    Convert nominal (discrete) interest rate to continuous compounding rate.
    
    This function converts an interest rate that compounds m times per year
    to an equivalent rate under continuous compounding.
    
    Args:
        R: Nominal annual interest rate (as decimal, e.g., 0.05 for 5%)
        m: Number of compounding periods per year
        
    Returns:
        Equivalent continuous compounding rate
        
    Example:
        >>> nominal_to_continuous(0.05, 12)
        0.048889...
    """
    return m * math.log(1 + R / m)

def continuous_to_nominal(r: float, m: int) -> float:
    """
    Convert continuous compounding rate to nominal (discrete) interest rate.
    
    This function converts a continuously compounded interest rate to an
    equivalent rate that compounds m times per year.
    
    Args:
        r: Continuous compounding rate (as decimal, e.g., 0.05 for 5%)
        m: Number of compounding periods per year
        
    Returns:
        Equivalent nominal interest rate
        
    Example:
        >>> continuous_to_nominal(0.05, 12)
        0.051162...
    """
    return m *(math.exp(r / m)-1)