"""A module for implementing interaction between MIDI and SequenceGenerators."""

import abc
import threading
import time

# internal imports
import tensorflow as tf

from magenta.protobuf import generator_pb2
from magenta.protobuf import music_pb2


class MidiInteractionException(Exception):
  """Base class for exceptions in this module."""
  pass


# TODO(adarob): Move to sequence_utils.
def merge_sequence_notes(sequence_1, sequence_2):
  """Returns a new NoteSequence combining the notes from both inputs.

  All fields aside from `notes` and `total_time` are copied from the first
  input.

  Args:
    sequence_1: A NoteSequence to merge. All fields aside from `notes` and
        `total_time` are copied directly from this sequence in the merged
        sequence.
    sequence_2: A NoteSequence to merge.

  Returns:
    A new NoteSequence combining the notes from the input sequences.
  """
  merged_sequence = music_pb2.NoteSequence()
  merged_sequence.CopyFrom(sequence_1)
  merged_sequence.notes.extend(sequence_2.notes)
  merged_sequence.total_time = max(sequence_1.total_time, sequence_2.total_time)
  return merged_sequence


# TODO(adarob): Move to sequence_utils.
def filter_instrument(sequence, instrument, from_time=0):
  """Returns a new NoteSequence with notes from the given instrument removed.

  Only notes that start on or after `from_time` will be completely removed.
  Those that start before and end after `from_time` will be truncated to end
  at `from_time`.

  Args:
    sequence: The NoteSequence to created the filtered sequence from.
    instrument: The instrument number to remove notes of.
    from_time: The time on or after which to remove or truncate notes.

  Returns:
    A new NoteSequence with notes from the given instrument removed or truncated
    after `from_time`.
  """
  filtered_sequence = music_pb2.NoteSequence()
  filtered_sequence.CopyFrom(sequence)
  del filtered_sequence.notes[:]
  for note in sequence.notes:
    if note.instrument == instrument:
      if note.start_time >= from_time:
        continue
      if note.end_time >= from_time:
        note.end_time = from_time
    filtered_sequence.notes.add().CopyFrom(note)
  return filtered_sequence


# TODO(adarob): Move to sequence_utils.
def adjust_times(sequence, delta_seconds):
  """Adjusts times in NoteSequence by given amount."""
  for note in sequence.notes:
    note.start_time += delta_seconds
    note.end_time += delta_seconds
  sequence.total_time += delta_seconds


class MidiInteraction(threading.Thread):
  """Base class for handling interaction between MIDI and SequenceGenerator.

  Child classes will provided the "main loop" of an interactive session between
  a MidiHub used for MIDI I/O and sequences generated by a SequenceGenerator in
  their `run` methods.

  Should be started by calling `start` to launch in a separate thread.

  Args:
    midi_hub: The MidiHub to use for MIDI I/O.
    qpm: The quarters per minute to use for this interaction.
  """
  _metaclass__ = abc.ABCMeta

  def __init__(self, midi_hub, qpm):
    self._midi_hub = midi_hub
    self._qpm = qpm
    # A signal to tell the main loop when to stop.
    self._stop_signal = threading.Event()
    # A signal to specify when the main loop has stopped.
    self._stopped = threading.Event()
    super(MidiInteraction, self).__init__()

  @abc.abstractmethod
  def run(self):
    """The main loop for the interaction.

    Must exit shortly after `self._stop_signal` is set.
    """
    pass

  def stop(self):
    """Stops the main loop, and blocks until the interaction is stopped."""
    self._stop_signal.set()
    self.join()


class CallAndResponseMidiInteraction(MidiInteraction):
  """Implementation of a MidiInteraction for real-time "call and response".

  Alternates between receiving input from the MidiHub ("call" phase) and
  playing generated sequences ("response" phase). During the call phase, the
  input is captured and used to generate the response, which is then played back
  during the response phase.

  Args:
    midi_hub: The MidiHub to use for MIDI I/O.
    qpm: The quarters per minute to use for this interaction.
    sequence_generator: The SequenceGenerator to use to generate the responses
        in this interaction.
    quarters_per_bar: The number of quarter notes in each bar/measure.
    phase_bars: The optional number of bars in each phase. `end_call_signal`
        must be provided if None.
    end_call_signal: The optional midi_hub.MidiSignal to use as a signal to stop
        the call phase at the end of the current bar. `phase_bars` must be
        provided if None.
  """

  def __init__(self,
               midi_hub,
               qpm,
               sequence_generator,
               quarters_per_bar=4,
               phase_bars=None,
               end_call_signal=None):
    super(CallAndResponseMidiInteraction, self).__init__(midi_hub, qpm)
    self._sequence_generator = sequence_generator
    self._quarters_per_bar = quarters_per_bar
    self._phase_bars = phase_bars
    self._end_call_signal = end_call_signal

  def run(self):
    """The main loop for a real-time call and response interaction.

    Alternates between receiving input from the MidiHub ("call" phase) and
    playing generated sequences ("response" phase). During the call phase, the
    input is captured and used to generate the response, which is then played
    back during the response phase.
    """
    # We measure time in units of quarter notes.
    quarter_duration = 60.0 / self._qpm
    # Start time in quarter notes from the epoch.
    start_quarters = (time.time() + 1.0) // quarter_duration

    # The number of notes before call phase ends to start generation for
    # response phase. Will be automatically adjusted to be as small as possible
    # while avoiding late response starts.
    predictahead_quarters = 1

    # Offset to beginning of call phase from start_quarters.
    call_offset_quarters = 0

    while not self._stop_signal.is_set():
      # Call phase.

      # Call phase start in quarter notes from the epoch.
      call_start_quarters = start_quarters + call_offset_quarters
      # Start the metronome at the beginning of the call phase.
      self._midi_hub.start_metronome(
          self._qpm, call_start_quarters * quarter_duration)

      # Start a captor at the beginning of the call phase.
      captor = self._midi_hub.start_capture(
          self._qpm, call_start_quarters * quarter_duration)

      if self._phase_bars is not None:
        # The duration of the call phase in quarter notes.
        call_quarters = self._phase_bars * self._quarters_per_bar
        # The duration of the capture in quarter notes.
        capture_quarters = call_quarters - predictahead_quarters
      else:
        # Wait for end signal.
        self._midi_hub.wait_for_event(self._end_call_signal)
        # The duration of the call phase in quarter notes.
        # We end the call phase at the end of the next bar that is at least
        # `predicathead_quarters` in the future.
        call_quarters = time.time() // quarter_duration - call_start_quarters
        remaining_call_quarters = -call_quarters % self._quarters_per_bar
        if remaining_call_quarters < predictahead_quarters:
          remaining_call_quarters += self._quarters_per_bar
        call_quarters += remaining_call_quarters
        # The duration of the capture in quarter notes.
        capture_quarters = call_quarters - predictahead_quarters

      # Set the metronome to stop at the appropriate time.
      self._midi_hub.stop_metronome(
          (call_quarters + call_start_quarters) * quarter_duration,
          block=False)

      # Stop the and captor at the appropriate time.
      captor.stop(stop_time=(
          (call_start_quarters + capture_quarters) * quarter_duration))
      captured_sequence = captor.captured_sequence()

      # Check to see if a stop has been requested during capture.
      if self._stop_signal.is_set():
        break

      # Set times in `captured_sequence` so that the call start is at 0.
      adjust_times(captured_sequence, -(call_start_quarters * quarter_duration))

      # Generate sequence.
      response_start_quarters = call_quarters
      response_end_quarters = 2 * call_quarters

      generator_options = generator_pb2.GeneratorOptions()
      generator_options.generate_sections.add(
          start_time_seconds=response_start_quarters * quarter_duration,
          end_time_seconds=response_end_quarters * quarter_duration)

      # Generate response.
      response_sequence = self._sequence_generator.generate(
          captured_sequence, generator_options)

      # Set times in `captured_sequence` back to the wall times.
      adjust_times(response_sequence, call_start_quarters * quarter_duration)

      # Check to see if a stop has been requested during generation.
      if self._stop_signal.is_set():
        break

      # Response phase.
      # Start response playback.
      self._midi_hub.start_playback(response_sequence)

      # Compute and log delta time between end of accompaniment before update
      # when the extension generation completed..
      remaining_time = (
          (response_start_quarters + call_start_quarters) * quarter_duration -
          time.time())
      if remaining_time > (predictahead_quarters * quarter_duration):
        predictahead_quarters -= 1
        tf.logging.info('Generator is ahead by %.3f seconds. '
                        'Decreasing predictahead_quarters to %d.',
                        remaining_time, predictahead_quarters)
      elif remaining_time < 0:
        predictahead_quarters += 1
        tf.logging.info('Generator is lagging by %.3f seconds. '
                        'Increasing predictahead_quarters to %d.',
                        -remaining_time, predictahead_quarters)

      call_offset_quarters += response_end_quarters

  def stop(self):
    if self._end_call_signal is not None:
      self._midi_hub.wake_signal_waiters(self._end_call_signal)
    super(CallAndResponseMidiInteraction, self).stop()
