"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const function_base_1 = require("./function-base");
const lambda_generated_1 = require("./lambda.generated");
/**
 * A new alias to a particular version of a Lambda function.
 */
class Alias extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.canCreatePermissions = true;
        this.underlyingLambda = props.version.lambda;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: props.aliasName,
            description: props.description,
            functionName: this.underlyingLambda.functionName,
            functionVersion: props.version.functionVersion,
            routingConfig: this.determineRoutingConfig(props)
        });
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.node.stack.parseArn(alias.aliasArn, ":").resourceName}:${props.aliasName}`;
        this.functionArn = alias.aliasArn;
    }
    /**
     * Role associated with this alias
     */
    get role() {
        return this.underlyingLambda.role;
    }
    get grantPrincipal() {
        return this.underlyingLambda.grantPrincipal;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, Object.assign({ dimensions: {
                FunctionName: this.underlyingLambda.functionName,
                Resource: this.functionArn
            } }, props));
    }
    export() {
        return {
            functionArn: new cdk.CfnOutput(this, 'AliasArn', { value: this.functionArn }).makeImportValue().toString()
        };
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.functionVersion,
                    functionWeight: vw.weight
                };
            })
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxpYXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbGlhcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLG9DQUFxQztBQUNyQyxtREFBK0U7QUFFL0UseURBQThDO0FBNkM5Qzs7R0FFRztBQUNILE1BQWEsS0FBTSxTQUFRLDRCQUFZO0lBd0JyQyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQWlCO1FBQzdELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFSQSx5QkFBb0IsR0FBWSxJQUFJLENBQUM7UUFVdEQsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDO1FBRTdDLE1BQU0sS0FBSyxHQUFHLElBQUksMkJBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzNDLElBQUksRUFBRSxLQUFLLENBQUMsU0FBUztZQUNyQixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsWUFBWSxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZO1lBQ2hELGVBQWUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLGVBQWU7WUFDOUMsYUFBYSxFQUFFLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLENBQUM7U0FDbEQsQ0FBQyxDQUFDO1FBRUgsb0dBQW9HO1FBQ3BHLGdIQUFnSDtRQUNoSCwwQ0FBMEM7UUFDMUMsSUFBSSxDQUFDLFlBQVksR0FBRyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDLFlBQWEsSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDeEcsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsSUFBSTtRQUNiLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQztJQUNwQyxDQUFDO0lBRUQsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQztJQUM5QyxDQUFDO0lBRU0sTUFBTSxDQUFDLFVBQWtCLEVBQUUsUUFBd0MsRUFBRTtRQUMxRSw4R0FBOEc7UUFDOUcsT0FBTyxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsa0JBQzVCLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFlBQVk7Z0JBQ2hELFFBQVEsRUFBRSxJQUFJLENBQUMsV0FBVzthQUMzQixJQUNFLEtBQUssRUFDUixDQUFDO0lBQ0wsQ0FBQztJQUVNLE1BQU07UUFDWCxPQUFPO1lBQ0wsV0FBVyxFQUFFLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDLFFBQVEsRUFBRTtTQUMzRyxDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0ssc0JBQXNCLENBQUMsS0FBaUI7UUFDOUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN0RSxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUV6RCxPQUFPO1lBQ0wsd0JBQXdCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRTtnQkFDMUQsT0FBTztvQkFDTCxlQUFlLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxlQUFlO29CQUMzQyxjQUFjLEVBQUUsRUFBRSxDQUFDLE1BQU07aUJBQzFCLENBQUM7WUFDSixDQUFDLENBQUM7U0FDSCxDQUFDO0lBQ0osQ0FBQztJQUVEOzs7O09BSUc7SUFDSyx5QkFBeUIsQ0FBQyxPQUF3QjtRQUN4RCxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzVCLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQUUsTUFBTSxJQUFJLEtBQUssQ0FBQywyREFBMkQsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7YUFBRTtZQUM3SCxPQUFPLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFDbEIsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRTNCLElBQUksS0FBSyxHQUFHLENBQUMsRUFBRTtZQUNiLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELEtBQUssRUFBRSxDQUFDLENBQUM7U0FDdkY7SUFDSCxDQUFDO0NBQ0Y7QUEzR0Qsc0JBMkdDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IGNsb3Vkd2F0Y2ggPSByZXF1aXJlKCdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCcpO1xuaW1wb3J0IGNkayA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2NkaycpO1xuaW1wb3J0IHsgRnVuY3Rpb25CYXNlLCBGdW5jdGlvbkltcG9ydFByb3BzLCBJRnVuY3Rpb24gfSBmcm9tICcuL2Z1bmN0aW9uLWJhc2UnO1xuaW1wb3J0IHsgVmVyc2lvbiB9IGZyb20gJy4vbGFtYmRhLXZlcnNpb24nO1xuaW1wb3J0IHsgQ2ZuQWxpYXMgfSBmcm9tICcuL2xhbWJkYS5nZW5lcmF0ZWQnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbmV3IExhbWJkYSBhbGlhc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEFsaWFzUHJvcHMge1xuICAvKipcbiAgICogRGVzY3JpcHRpb24gZm9yIHRoZSBhbGlhc1xuICAgKlxuICAgKiBAZGVmYXVsdCBObyBkZXNjcmlwdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEZ1bmN0aW9uIHZlcnNpb24gdGhpcyBhbGlhcyByZWZlcnMgdG9cbiAgICpcbiAgICogVXNlIGxhbWJkYS5hZGRWZXJzaW9uKCkgdG8gb2J0YWluIGEgbmV3IGxhbWJkYSB2ZXJzaW9uIHRvIHJlZmVyIHRvLlxuICAgKi9cbiAgcmVhZG9ubHkgdmVyc2lvbjogVmVyc2lvbjtcblxuICAvKipcbiAgICogTmFtZSBvZiB0aGlzIGFsaWFzXG4gICAqL1xuICByZWFkb25seSBhbGlhc05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogQWRkaXRpb25hbCB2ZXJzaW9ucyB3aXRoIGluZGl2aWR1YWwgd2VpZ2h0cyB0aGlzIGFsaWFzIHBvaW50cyB0b1xuICAgKlxuICAgKiBJbmRpdmlkdWFsIGFkZGl0aW9uYWwgdmVyc2lvbiB3ZWlnaHRzIHNwZWNpZmllZCBoZXJlIHNob3VsZCBhZGQgdXAgdG9cbiAgICogKGxlc3MgdGhhbikgb25lLiBBbGwgcmVtYWluaW5nIHdlaWdodCBpcyByb3V0ZWQgdG8gdGhlIGRlZmF1bHRcbiAgICogdmVyc2lvbi5cbiAgICpcbiAgICogRm9yIGV4YW1wbGUsIHRoZSBjb25maWcgaXNcbiAgICpcbiAgICogICAgdmVyc2lvbjogXCIxXCJcbiAgICogICAgYWRkaXRpb25hbFZlcnNpb25zOiBbeyB2ZXJzaW9uOiBcIjJcIiwgd2VpZ2h0OiAwLjA1IH1dXG4gICAqXG4gICAqIFRoZW4gNSUgb2YgdHJhZmZpYyB3aWxsIGJlIHJvdXRlZCB0byBmdW5jdGlvbiB2ZXJzaW9uIDIsIHdoaWxlXG4gICAqIHRoZSByZW1haW5pbmcgOTUlIG9mIHRyYWZmaWMgd2lsbCBiZSByb3V0ZWQgdG8gZnVuY3Rpb24gdmVyc2lvbiAxLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBhZGRpdGlvbmFsIHZlcnNpb25zXG4gICAqL1xuICByZWFkb25seSBhZGRpdGlvbmFsVmVyc2lvbnM/OiBWZXJzaW9uV2VpZ2h0W107XG59XG5cbi8qKlxuICogQSBuZXcgYWxpYXMgdG8gYSBwYXJ0aWN1bGFyIHZlcnNpb24gb2YgYSBMYW1iZGEgZnVuY3Rpb24uXG4gKi9cbmV4cG9ydCBjbGFzcyBBbGlhcyBleHRlbmRzIEZ1bmN0aW9uQmFzZSB7XG4gIC8qKlxuICAgKiBBUk4gb2YgdGhpcyBhbGlhc1xuICAgKlxuICAgKiBVc2VkIHRvIGJlIGFibGUgdG8gdXNlIEFsaWFzIGluIHBsYWNlIG9mIGEgcmVndWxhciBMYW1iZGEuIExhbWJkYSBhY2NlcHRzXG4gICAqIEFSTnMgZXZlcnl3aGVyZSBpdCBhY2NlcHRzIGZ1bmN0aW9uIG5hbWVzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGZ1bmN0aW9uTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBUk4gb2YgdGhpcyBhbGlhc1xuICAgKlxuICAgKiBVc2VkIHRvIGJlIGFibGUgdG8gdXNlIEFsaWFzIGluIHBsYWNlIG9mIGEgcmVndWxhciBMYW1iZGEuIExhbWJkYSBhY2NlcHRzXG4gICAqIEFSTnMgZXZlcnl3aGVyZSBpdCBhY2NlcHRzIGZ1bmN0aW9uIG5hbWVzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGZ1bmN0aW9uQXJuOiBzdHJpbmc7XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGNhbkNyZWF0ZVBlcm1pc3Npb25zOiBib29sZWFuID0gdHJ1ZTtcblxuICAvKipcbiAgICogVGhlIGFjdHVhbCBMYW1iZGEgZnVuY3Rpb24gb2JqZWN0IHRoYXQgdGhpcyBBbGlhcyBpcyBwb2ludGluZyB0b1xuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSB1bmRlcmx5aW5nTGFtYmRhOiBJRnVuY3Rpb247XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBbGlhc1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMudW5kZXJseWluZ0xhbWJkYSA9IHByb3BzLnZlcnNpb24ubGFtYmRhO1xuXG4gICAgY29uc3QgYWxpYXMgPSBuZXcgQ2ZuQWxpYXModGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgbmFtZTogcHJvcHMuYWxpYXNOYW1lLFxuICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uLFxuICAgICAgZnVuY3Rpb25OYW1lOiB0aGlzLnVuZGVybHlpbmdMYW1iZGEuZnVuY3Rpb25OYW1lLFxuICAgICAgZnVuY3Rpb25WZXJzaW9uOiBwcm9wcy52ZXJzaW9uLmZ1bmN0aW9uVmVyc2lvbixcbiAgICAgIHJvdXRpbmdDb25maWc6IHRoaXMuZGV0ZXJtaW5lUm91dGluZ0NvbmZpZyhwcm9wcylcbiAgICB9KTtcblxuICAgIC8vIEFSTiBwYXJzaW5nIHNwbGl0cyBvbiBgOmAsIHNvIHdlIGNhbiBvbmx5IGdldCB0aGUgZnVuY3Rpb24ncyBuYW1lIGZyb20gdGhlIEFSTiBhcyByZXNvdXJjZU5hbWUuLi5cbiAgICAvLyBBbmQgd2UncmUgcGFyc2luZyBpdCBvdXQgKGluc3RlYWQgb2YgdXNpbmcgdGhlIHVuZGVybHlpbmcgZnVuY3Rpb24gZGlyZWN0bHkpIGluIG9yZGVyIHRvIGhhdmUgdXNlIG9mIGl0IGluY3VyXG4gICAgLy8gYW4gaW1wbGljaXQgZGVwZW5kZW5jeSBvbiB0aGUgcmVzb3VyY2UuXG4gICAgdGhpcy5mdW5jdGlvbk5hbWUgPSBgJHt0aGlzLm5vZGUuc3RhY2sucGFyc2VBcm4oYWxpYXMuYWxpYXNBcm4sIFwiOlwiKS5yZXNvdXJjZU5hbWUhfToke3Byb3BzLmFsaWFzTmFtZX1gO1xuICAgIHRoaXMuZnVuY3Rpb25Bcm4gPSBhbGlhcy5hbGlhc0FybjtcbiAgfVxuXG4gIC8qKlxuICAgKiBSb2xlIGFzc29jaWF0ZWQgd2l0aCB0aGlzIGFsaWFzXG4gICAqL1xuICBwdWJsaWMgZ2V0IHJvbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudW5kZXJseWluZ0xhbWJkYS5yb2xlO1xuICB9XG5cbiAgcHVibGljIGdldCBncmFudFByaW5jaXBhbCgpIHtcbiAgICByZXR1cm4gdGhpcy51bmRlcmx5aW5nTGFtYmRhLmdyYW50UHJpbmNpcGFsO1xuICB9XG5cbiAgcHVibGljIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzOiBjbG91ZHdhdGNoLk1ldHJpY0N1c3RvbWl6YXRpb24gPSB7fSk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAvLyBNZXRyaWNzIG9uIEFsaWFzZXMgbmVlZCB0aGUgXCJiYXJlXCIgZnVuY3Rpb24gbmFtZSwgYW5kIHRoZSBhbGlhcycgQVJOLCB0aGlzIGRpZmZlcmVzIGZyb20gdGhlIGJhc2UgYmVoYXZpb3IuXG4gICAgcmV0dXJuIHN1cGVyLm1ldHJpYyhtZXRyaWNOYW1lLCB7XG4gICAgICBkaW1lbnNpb25zOiB7XG4gICAgICAgIEZ1bmN0aW9uTmFtZTogdGhpcy51bmRlcmx5aW5nTGFtYmRhLmZ1bmN0aW9uTmFtZSxcbiAgICAgICAgUmVzb3VyY2U6IHRoaXMuZnVuY3Rpb25Bcm5cbiAgICAgIH0sXG4gICAgICAuLi5wcm9wc1xuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIGV4cG9ydCgpOiBGdW5jdGlvbkltcG9ydFByb3BzIHtcbiAgICByZXR1cm4ge1xuICAgICAgZnVuY3Rpb25Bcm46IG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdBbGlhc0FybicsIHsgdmFsdWU6IHRoaXMuZnVuY3Rpb25Bcm4gfSkubWFrZUltcG9ydFZhbHVlKCkudG9TdHJpbmcoKVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQ2FsY3VsYXRlIHRoZSByb3V0aW5nQ29uZmlnIHBhcmFtZXRlciBmcm9tIHRoZSBpbnB1dCBwcm9wc1xuICAgKi9cbiAgcHJpdmF0ZSBkZXRlcm1pbmVSb3V0aW5nQ29uZmlnKHByb3BzOiBBbGlhc1Byb3BzKSB7XG4gICAgaWYgKCFwcm9wcy5hZGRpdGlvbmFsVmVyc2lvbnMgfHwgcHJvcHMuYWRkaXRpb25hbFZlcnNpb25zLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICB0aGlzLnZhbGlkYXRlQWRkaXRpb25hbFdlaWdodHMocHJvcHMuYWRkaXRpb25hbFZlcnNpb25zKTtcblxuICAgIHJldHVybiB7XG4gICAgICBhZGRpdGlvbmFsVmVyc2lvbldlaWdodHM6IHByb3BzLmFkZGl0aW9uYWxWZXJzaW9ucy5tYXAodncgPT4ge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIGZ1bmN0aW9uVmVyc2lvbjogdncudmVyc2lvbi5mdW5jdGlvblZlcnNpb24sXG4gICAgICAgICAgZnVuY3Rpb25XZWlnaHQ6IHZ3LndlaWdodFxuICAgICAgICB9O1xuICAgICAgfSlcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlIHRoYXQgdGhlIGFkZGl0aW9uYWwgdmVyc2lvbiB3ZWlnaHRzIG1ha2Ugc2Vuc2VcbiAgICpcbiAgICogV2UgdmFsaWRhdGUgdGhhdCB0aGV5IGFyZSBwb3NpdGl2ZSBhbmQgYWRkIHVwIHRvIHNvbWV0aGluZyA8PSAxLlxuICAgKi9cbiAgcHJpdmF0ZSB2YWxpZGF0ZUFkZGl0aW9uYWxXZWlnaHRzKHdlaWdodHM6IFZlcnNpb25XZWlnaHRbXSkge1xuICAgIGNvbnN0IHRvdGFsID0gd2VpZ2h0cy5tYXAodyA9PiB7XG4gICAgICBpZiAody53ZWlnaHQgPCAwIHx8IHcud2VpZ2h0ID4gMSkgeyB0aHJvdyBuZXcgRXJyb3IoYEFkZGl0aW9uYWwgdmVyc2lvbiB3ZWlnaHQgbXVzdCBiZSBiZXR3ZWVuIDAgYW5kIDEsIGdvdDogJHt3LndlaWdodH1gKTsgfVxuICAgICAgcmV0dXJuIHcud2VpZ2h0O1xuICAgIH0pLnJlZHVjZSgoYSwgeCkgPT4gYSArIHgpO1xuXG4gICAgaWYgKHRvdGFsID4gMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBTdW0gb2YgYWRkaXRpb25hbCB2ZXJzaW9uIHdlaWdodHMgbXVzdCBub3QgZXhjZWVkIDEsIGdvdDogJHt0b3RhbH1gKTtcbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBBIHZlcnNpb24vd2VpZ2h0IHBhaXIgZm9yIHJvdXRpbmcgdHJhZmZpYyB0byBMYW1iZGEgZnVuY3Rpb25zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVmVyc2lvbldlaWdodCB7XG4gIC8qKlxuICAgKiBUaGUgdmVyc2lvbiB0byByb3V0ZSB0cmFmZmljIHRvXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9uOiBWZXJzaW9uO1xuXG4gIC8qKlxuICAgKiBIb3cgbXVjaCB3ZWlnaHQgdG8gYXNzaWduIHRvIHRoaXMgdmVyc2lvbiAoMC4uMSlcbiAgICovXG4gIHJlYWRvbmx5IHdlaWdodDogbnVtYmVyO1xufVxuIl19