import assets = require('@aws-cdk/assets');
import s3 = require('@aws-cdk/aws-s3');
import cdk = require('@aws-cdk/cdk');
export declare abstract class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static bucket(bucket: s3.IBucket, key: string, objectVersion?: string): S3Code;
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static inline(code: string): InlineCode;
    /**
     * Loads the function code from a local disk asset.
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static asset(path: string): AssetCode;
    /**
     * @returns Zip archives the contents of a directory on disk and uses this
     * as the lambda handler's code.
     * @param directoryToZip The directory to zip
     * @deprecated use `lambda.Code.asset(path)` (no need to specify if it's a file or a directory)
     */
    static directory(directoryToZip: string): AssetCode;
    /**
     * @returns Uses a file on disk as a lambda handler's code.
     * @param filePath The file path
     * @deprecated use `lambda.Code.asset(path)` (no need to specify if it's a file or a directory)
     */
    static file(filePath: string): AssetCode;
    /**
     * Determines whether this Code is inline code or not.
     */
    abstract readonly isInline: boolean;
    /**
     * Called when the lambda or layer is initialized to allow this object to
     * bind to the stack, add resources and have fun.
     */
    bind(_construct: cdk.Construct): void;
}
/**
 * Lambda code from an S3 archive.
 */
export declare class S3Code extends Code {
    private key;
    private objectVersion?;
    readonly isInline = false;
    private bucketName;
    constructor(bucket: s3.IBucket, key: string, objectVersion?: string | undefined);
}
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
export declare class InlineCode extends Code {
    private code;
    readonly isInline = true;
    constructor(code: string);
    bind(construct: cdk.Construct): void;
}
/**
 * Lambda code from a local directory.
 */
export declare class AssetCode extends Code {
    readonly path: string;
    readonly isInline = false;
    /**
     * The asset packaging.
     */
    readonly packaging: assets.AssetPackaging;
    private asset?;
    /**
     * @param path The path to the asset file or directory.
     * @param packaging The asset packaging format (optional, determined automatically)
     */
    constructor(path: string, packaging?: assets.AssetPackaging);
    bind(construct: cdk.Construct): void;
}
