"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assets = require("@aws-cdk/assets");
const fs = require("fs");
class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static bucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static inline(code) {
        return new InlineCode(code);
    }
    /**
     * Loads the function code from a local disk asset.
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static asset(path) {
        return new AssetCode(path);
    }
    /**
     * @returns Zip archives the contents of a directory on disk and uses this
     * as the lambda handler's code.
     * @param directoryToZip The directory to zip
     * @deprecated use `lambda.Code.asset(path)` (no need to specify if it's a file or a directory)
     */
    static directory(directoryToZip) {
        return new AssetCode(directoryToZip, assets.AssetPackaging.ZipDirectory);
    }
    /**
     * @returns Uses a file on disk as a lambda handler's code.
     * @param filePath The file path
     * @deprecated use `lambda.Code.asset(path)` (no need to specify if it's a file or a directory)
     */
    static file(filePath) {
        return new AssetCode(filePath, assets.AssetPackaging.File);
    }
    /**
     * Called when the lambda or layer is initialized to allow this object to
     * bind to the stack, add resources and have fun.
     */
    bind(_construct) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * @internal
     */
    _toJSON(_) {
        return {
            s3Bucket: this.bucketName,
            s3Key: this.key,
            s3ObjectVersion: this.objectVersion
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length > 4096) {
            throw new Error("Lambda source is too large, must be <= 4096 but is " + code.length);
        }
    }
    bind(construct) {
        const runtime = construct.runtime;
        if (!runtime.supportsInlineCode) {
            throw new Error(`Inline source not allowed for ${runtime && runtime.name}`);
        }
    }
    /**
     * @internal
     */
    _toJSON(_) {
        return {
            zipFile: this.code
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     * @param packaging The asset packaging format (optional, determined automatically)
     */
    constructor(path, packaging) {
        super();
        this.path = path;
        this.isInline = false;
        if (packaging !== undefined) {
            this.packaging = packaging;
        }
        else {
            this.packaging = fs.lstatSync(path).isDirectory()
                ? assets.AssetPackaging.ZipDirectory
                : assets.AssetPackaging.File;
        }
    }
    bind(construct) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new assets.Asset(construct, 'Code', {
                path: this.path,
                packaging: this.packaging
            });
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
    }
    /**
     * @internal
     */
    _toJSON(resource) {
        if (resource) {
            // https://github.com/awslabs/aws-cdk/issues/1432
            this.asset.addResourceMetadata(resource, 'Code');
        }
        return {
            s3Bucket: this.asset.s3BucketName,
            s3Key: this.asset.s3ObjectKey
        };
    }
}
exports.AssetCode = AssetCode;
//# sourceMappingURL=data:application/json;base64,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