import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import ec2 = require('@aws-cdk/aws-ec2');
import events = require('@aws-cdk/aws-events');
import iam = require('@aws-cdk/aws-iam');
import logs = require('@aws-cdk/aws-logs');
import s3n = require('@aws-cdk/aws-s3-notifications');
import stepfunctions = require('@aws-cdk/aws-stepfunctions');
import cdk = require('@aws-cdk/cdk');
import { IEventSource } from './event-source';
import { Permission } from './permission';
export interface IFunction extends cdk.IConstruct, events.IEventRuleTarget, logs.ILogSubscriptionDestination, s3n.IBucketNotificationDestination, ec2.IConnectable, stepfunctions.IStepFunctionsTaskResource, iam.IGrantable {
    /**
     * Logical ID of this Function.
     */
    readonly id: string;
    /**
     * The name of the function.
     */
    readonly functionName: string;
    /**
     * The ARN fo the function.
     */
    readonly functionArn: string;
    /**
     * The IAM role associated with this function.
     */
    readonly role?: iam.IRole;
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    readonly isBoundToVpc: boolean;
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     */
    addPermission(id: string, permission: Permission): void;
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(identity: iam.IGrantable): iam.Grant;
    /**
     * Return the given named metric for this Lambda
     */
    metric(metricName: string, props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the Duration of this Lambda
     *
     * @default average over 5 minutes
     */
    metricDuration(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the number of invocations of this Lambda
     *
     * @default sum over 5 minutes
     */
    metricInvocations(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the number of throttled invocations of this Lambda
     *
     * @default sum over 5 minutes
     */
    metricThrottles(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Export this Function (without the role)
     */
    export(): FunctionImportProps;
    addEventSource(source: IEventSource): void;
}
/**
 * Represents a Lambda function defined outside of this stack.
 */
export interface FunctionImportProps {
    /**
     * The ARN of the Lambda function.
     *
     * Format: arn:<partition>:lambda:<region>:<account-id>:function:<function-name>
     */
    readonly functionArn: string;
    /**
     * The IAM execution role associated with this function.
     *
     * If the role is not specified, any role-related operations will no-op.
     */
    readonly role?: iam.IRole;
    /**
     * Id of the securityGroup for this Lambda, if in a VPC.
     *
     * This needs to be given in order to support allowing connections
     * to this Lambda.
     */
    readonly securityGroupId?: string;
}
export declare abstract class FunctionBase extends cdk.Construct implements IFunction {
    /**
     * The principal this Lambda Function is running as
     */
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * The name of the function.
     */
    abstract readonly functionName: string;
    /**
     * The ARN fo the function.
     */
    abstract readonly functionArn: string;
    /**
     * The IAM role associated with this function.
     *
     * Undefined if the function was imported without a role.
     */
    abstract readonly role?: iam.IRole;
    /**
     * Whether the addPermission() call adds any permissions
     *
     * True for new Lambdas, false for imported Lambdas (they might live in different accounts).
     */
    protected abstract readonly canCreatePermissions: boolean;
    /**
     * Indicates if the policy that allows CloudWatch logs to publish to this lambda has been added.
     */
    private logSubscriptionDestinationPolicyAddedFor;
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     */
    addPermission(id: string, permission: Permission): void;
    readonly id: string;
    addToRolePolicy(statement: iam.PolicyStatement): void;
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    readonly connections: ec2.Connections;
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    readonly isBoundToVpc: boolean;
    /**
     * Returns a RuleTarget that can be used to trigger this Lambda as a
     * result from a CloudWatch event.
     */
    asEventRuleTarget(ruleArn: string, ruleId: string): events.EventRuleTargetProps;
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee: iam.IGrantable): iam.Grant;
    logSubscriptionDestination(sourceLogGroup: logs.ILogGroup): logs.LogSubscriptionDestination;
    /**
     * Export this Function (without the role)
     */
    abstract export(): FunctionImportProps;
    /**
     * Allows this Lambda to be used as a destination for bucket notifications.
     * Use `bucket.onEvent(lambda)` to subscribe.
     */
    asBucketNotificationDestination(bucketArn: string, bucketId: string): s3n.BucketNotificationDestinationProps;
    asStepFunctionsTaskResource(_callingTask: stepfunctions.Task): stepfunctions.StepFunctionsTaskResourceProps;
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source: IEventSource): void;
    private parsePermissionPrincipal;
}
