"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const s3n = require("@aws-cdk/aws-s3-notifications");
const cdk = require("@aws-cdk/cdk");
const lambda_generated_1 = require("./lambda.generated");
class FunctionBase extends cdk.Construct {
    constructor() {
        super(...arguments);
        /**
         * Indicates if the policy that allows CloudWatch logs to publish to this lambda has been added.
         */
        this.logSubscriptionDestinationPolicyAddedFor = [];
    }
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     */
    addPermission(id, permission) {
        if (!this.canCreatePermissions) {
            // FIXME: Report metadata
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const action = permission.action || 'lambda:InvokeFunction';
        new lambda_generated_1.CfnPermission(this, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount,
            sourceArn: permission.sourceArn,
        });
    }
    get id() {
        return this.node.id;
    }
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // tslint:disable-next-line:max-line-length
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    /**
     * Returns a RuleTarget that can be used to trigger this Lambda as a
     * result from a CloudWatch event.
     */
    asEventRuleTarget(ruleArn, ruleId) {
        const permissionId = `AllowEventRule${ruleId}`;
        if (!this.node.tryFindChild(permissionId)) {
            this.addPermission(permissionId, {
                action: 'lambda:InvokeFunction',
                principal: new iam.ServicePrincipal('events.amazonaws.com'),
                sourceArn: ruleArn
            });
        }
        return {
            id: this.node.id,
            arn: this.functionArn,
        };
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['lambda:InvokeFunction'],
            resourceArns: [this.functionArn],
            // Fake resource-like object on which to call addToResourcePolicy(), which actually
            // calls addPermission()
            resource: {
                addToResourcePolicy: (_statement) => {
                    // Couldn't add permissions to the principal, so add them locally.
                    const identifier = 'Invoke' + JSON.stringify(grantee.grantPrincipal.policyFragment.principalJson);
                    this.addPermission(identifier, {
                        principal: grantee.grantPrincipal,
                        action: 'lambda:InvokeFunction',
                    });
                },
                dependencyRoots: [],
                node: this.node,
            },
        });
    }
    logSubscriptionDestination(sourceLogGroup) {
        const arn = sourceLogGroup.logGroupArn;
        if (this.logSubscriptionDestinationPolicyAddedFor.indexOf(arn) === -1) {
            // NOTE: the use of {AWS::Region} limits this to the same region, which shouldn't really be an issue,
            // since the Lambda must be in the same region as the SubscriptionFilter anyway.
            //
            // (Wildcards in principals are unfortunately not supported.
            this.addPermission('InvokedByCloudWatchLogs', {
                principal: new iam.ServicePrincipal(`logs.${this.node.stack.region}.amazonaws.com`),
                sourceArn: arn
            });
            this.logSubscriptionDestinationPolicyAddedFor.push(arn);
        }
        return { arn: this.functionArn };
    }
    /**
     * Allows this Lambda to be used as a destination for bucket notifications.
     * Use `bucket.onEvent(lambda)` to subscribe.
     */
    asBucketNotificationDestination(bucketArn, bucketId) {
        const permissionId = `AllowBucketNotificationsFrom${bucketId}`;
        if (!this.node.tryFindChild(permissionId)) {
            this.addPermission(permissionId, {
                sourceAccount: this.node.stack.accountId,
                principal: new iam.ServicePrincipal('s3.amazonaws.com'),
                sourceArn: bucketArn,
            });
        }
        // if we have a permission resource for this relationship, add it as a dependency
        // to the bucket notifications resource, so it will be created first.
        const permission = this.node.tryFindChild(permissionId);
        return {
            type: s3n.BucketNotificationDestinationType.Lambda,
            arn: this.functionArn,
            dependencies: permission ? [permission] : undefined
        };
    }
    asStepFunctionsTaskResource(_callingTask) {
        return {
            resourceArn: this.functionArn,
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.functionArn },
            policyStatements: [new iam.PolicyStatement()
                    .addResource(this.functionArn)
                    .addActions("lambda:InvokeFunction")
            ]
        };
    }
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source) {
        source.bind(this);
    }
    parsePermissionPrincipal(principal) {
        if (!principal) {
            return undefined;
        }
        // use duck-typing, not instance of
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if (`service` in principal) {
            return principal.service;
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${this.node.resolve(principal.toString())}. ` +
            'Supported: AccountPrincipal, ServicePrincipal');
    }
}
exports.FunctionBase = FunctionBase;
//# sourceMappingURL=data:application/json;base64,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