import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import logs = require('@aws-cdk/aws-logs');
import sqs = require('@aws-cdk/aws-sqs');
import cdk = require('@aws-cdk/cdk');
import { Code } from './code';
import { IEventSource } from './event-source';
import { FunctionBase, FunctionImportProps, IFunction } from './function-base';
import { Version } from './lambda-version';
import { ILayerVersion } from './layers';
import { Runtime } from './runtime';
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
export declare enum Tracing {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Active = 0,
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    PassThrough = 1,
    /**
     * Lambda will not trace any request.
     */
    Disabled = 2
}
export interface FunctionProps {
    /**
     * The source code of your Lambda function. You can point to a file in an
     * Amazon Simple Storage Service (Amazon S3) bucket or specify your source
     * code as inline text.
     */
    readonly code: Code;
    /**
     * A description of the function.
     */
    readonly description?: string;
    /**
     * The name of the function (within your source code) that Lambda calls to
     * start running your code. For more information, see the Handler property
     * in the AWS Lambda Developer Guide.
     *
     * NOTE: If you specify your source code as inline text by specifying the
     * ZipFile property within the Code property, specify index.function_name as
     * the handler.
     */
    readonly handler: string;
    /**
     * The function execution time (in seconds) after which Lambda terminates
     * the function. Because the execution time affects cost, set this value
     * based on the function's expected execution time.
     *
     * @default 3 seconds.
     */
    readonly timeout?: number;
    /**
     * Key-value pairs that Lambda caches and makes available for your Lambda
     * functions. Use environment variables to apply configuration changes, such
     * as test and production environment configurations, without changing your
     * Lambda function source code.
     */
    readonly environment?: {
        [key: string]: any;
    };
    /**
     * The runtime environment for the Lambda function that you are uploading.
     * For valid values, see the Runtime property in the AWS Lambda Developer
     * Guide.
     */
    readonly runtime: Runtime;
    /**
     * A name for the function. If you don't specify a name, AWS CloudFormation
     * generates a unique physical ID and uses that ID for the function's name.
     * For more information, see Name Type.
     */
    readonly functionName?: string;
    /**
     * The amount of memory, in MB, that is allocated to your Lambda function.
     * Lambda uses this value to proportionally allocate the amount of CPU
     * power. For more information, see Resource Model in the AWS Lambda
     * Developer Guide.
     *
     * @default The default value is 128 MB
     */
    readonly memorySize?: number;
    /**
     * Initial policy statements to add to the created Lambda Role.
     *
     * You can call `addToRolePolicy` to the created lambda to add statements post creation.
     */
    readonly initialPolicy?: iam.PolicyStatement[];
    /**
     * Lambda execution role.
     *
     * This is the role that will be assumed by the function upon execution.
     * It controls the permissions that the function will have. The Role must
     * be assumable by the 'lambda.amazonaws.com' service principal.
     *
     * @default a unique role will be generated for this lambda function.
     * Both supplied and generated roles can always be changed by calling `addToRolePolicy`.
     */
    readonly role?: iam.IRole;
    /**
     * VPC network to place Lambda network interfaces
     *
     * Specify this if the Lambda function needs to access resources in a VPC.
     */
    readonly vpc?: ec2.IVpcNetwork;
    /**
     * Where to place the network interfaces within the VPC.
     *
     * Only used if 'vpc' is supplied. Note: internet access for Lambdas
     * requires a NAT gateway, so picking Public subnets is not allowed.
     *
     * @default All private subnets
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * What security group to associate with the Lambda's network interfaces.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default If the function is placed within a VPC and a security group is
     * not specified, a dedicated security group will be created for this
     * function.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Whether to allow the Lambda to send all network traffic
     *
     * If set to false, you must individually add traffic rules to allow the
     * Lambda to connect to network targets.
     *
     * @default true
     */
    readonly allowAllOutbound?: boolean;
    /**
     * Enabled DLQ. If `deadLetterQueue` is undefined,
     * an SQS queue with default options will be defined for your Function.
     *
     * @default false unless `deadLetterQueue` is set, which implies DLQ is enabled
     */
    readonly deadLetterQueueEnabled?: boolean;
    /**
     * The SQS queue to use if DLQ is enabled.
     *
     * @default SQS queue with 14 day retention period if `deadLetterQueueEnabled` is `true`
     */
    readonly deadLetterQueue?: sqs.IQueue;
    /**
     * Enable AWS X-Ray Tracing for Lambda Function.
     *
     * @default undefined X-Ray tracing disabled
     */
    readonly tracing?: Tracing;
    /**
     * A list of layers to add to the function's execution environment. You can configure your Lambda function to pull in
     * additional code during initialization in the form of layers. Layers are packages of libraries or other dependencies
     * that can be used by mulitple functions.
     *
     * @default no layers
     */
    readonly layers?: ILayerVersion[];
    /**
     * The maximum of concurrent executions you want to reserve for the function.
     *
     * @default no specific limit - account limit
     * @see https://docs.aws.amazon.com/lambda/latest/dg/concurrent-executions.html
     */
    readonly reservedConcurrentExecutions?: number;
    /**
     * Event sources for this function.
     *
     * You can also add event sources using `addEventSource`.
     */
    readonly events?: IEventSource[];
    /**
     * The number of days log events are kept in CloudWatch Logs. When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `Infinity`.
     *
     * @default logs never expire
     */
    readonly logRetentionDays?: logs.RetentionDays;
}
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
export declare class Function extends FunctionBase {
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     *    Lambda.import(this, 'MyImportedFunction', { lambdaArn: new LambdaArn('arn:aws:...') });
     *
     * @param parent The parent construct
     * @param id The name of the lambda construct
     * @param props A reference to a Lambda function. Can be created manually (see
     * example above) or obtained through a call to `lambda.export()`.
     */
    static import(scope: cdk.Construct, id: string, props: FunctionImportProps): IFunction;
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName: string, props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props?: cloudwatch.MetricCustomization): cloudwatch.Metric;
    /**
     * Name of this function
     */
    readonly functionName: string;
    /**
     * ARN of this function
     */
    readonly functionArn: string;
    /**
     * Execution role associated with this function
     */
    readonly role?: iam.IRole;
    /**
     * The runtime configured for this lambda.
     */
    readonly runtime: Runtime;
    /**
     * The name of the handler configured for this lambda.
     */
    readonly handler: string;
    /**
     * The principal this Lambda Function is running as
     */
    readonly grantPrincipal: iam.IPrincipal;
    protected readonly canCreatePermissions = true;
    private readonly layers;
    /**
     * Environment variables for this function
     */
    private readonly environment?;
    constructor(scope: cdk.Construct, id: string, props: FunctionProps);
    /**
     * Export this Function (without the role)
     */
    export(): FunctionImportProps;
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     */
    addEnvironment(key: string, value: any): this;
    /**
     * Adds a Lambda Layer to this Lambda function.
     *
     * @param layer the layer to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayer(layer: ILayerVersion): this;
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want
     * to deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda source code, or
     *  omit to skip validation.
     * @param description A description for this version.
     * @returns A new Version object.
     */
    addVersion(name: string, codeSha256?: string, description?: string): Version;
    /**
     * Add a new version for this Lambda, always with a different name.
     *
     * This is similar to the {@link addVersion} method,
     * but useful when deploying this Lambda through CodePipeline with blue/green deployments.
     * When using {@link addVersion},
     * your Alias will not be updated until you change the name passed to {@link addVersion} in your CDK code.
     * When deploying through a Pipeline,
     * that might lead to a situation where a change to your Lambda application code will never be activated,
     * even though it traveled through the entire Pipeline,
     * because the Alias is still pointing to an old Version.
     * This method creates a new, unique Version every time the CDK code is executed,
     * and so prevents that from happening.
     */
    newVersion(): Version;
    private renderEnvironment;
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    private configureVpc;
    private buildDeadLetterConfig;
    private buildTracingConfig;
}
export declare class ImportedFunction extends FunctionBase {
    private readonly props;
    readonly grantPrincipal: iam.IPrincipal;
    readonly functionName: string;
    readonly functionArn: string;
    readonly role?: iam.IRole;
    protected readonly canCreatePermissions = false;
    constructor(scope: cdk.Construct, id: string, props: FunctionImportProps);
    export(): FunctionImportProps;
}
