"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/cdk");
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const log_retention_1 = require("./log-retention");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing[Tracing["Active"] = 0] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing[Tracing["PassThrough"] = 1] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing[Tracing["Disabled"] = 2] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.canCreatePermissions = true;
        this.layers = [];
        this.environment = props.environment || {};
        const managedPolicyArns = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicyArns.push(new iam.AwsManagedPolicy("service-role/AWSLambdaBasicExecutionRole", this).policyArn);
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicyArns.push(new iam.AwsManagedPolicy("service-role/AWSLambdaVPCAccessExecutionRole", this).policyArn);
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicyArns,
        });
        this.grantPrincipal = this.role;
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const isChina = this.node.stack.env.region && this.node.stack.env.region.startsWith('cn-');
        if (isChina && props.environment && Object.keys(props.environment).length > 0) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Environment variables are not supported in this region (${this.node.stack.env.region}); consider using tags or SSM parameters instead`);
        }
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: props.functionName,
            description: props.description,
            code: new cdk.Token(() => props.code._toJSON(resource)),
            layers: new cdk.Token(() => this.layers.length > 0 ? this.layers.map(layer => layer.layerVersionArn) : undefined).toList(),
            handler: props.handler,
            timeout: props.timeout,
            runtime: props.runtime.name,
            role: this.role.roleArn,
            environment: new cdk.Token(() => this.renderEnvironment()),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(props),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions
        });
        resource.node.addDependency(this.role);
        this.functionName = resource.ref;
        this.functionArn = resource.functionArn;
        this.handler = props.handler;
        this.runtime = props.runtime;
        // allow code to bind to stack.
        props.code.bind(this);
        for (const layer of props.layers || []) {
            this.addLayer(layer);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetentionDays) {
            new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retentionDays: props.logRetentionDays
            });
        }
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     *    Lambda.import(this, 'MyImportedFunction', { lambdaArn: new LambdaArn('arn:aws:...') });
     *
     * @param parent The parent construct
     * @param id The name of the lambda construct
     * @param props A reference to a Lambda function. Can be created manually (see
     * example above) or obtained through a call to `lambda.export()`.
     */
    static import(scope, id, props) {
        return new ImportedFunction(scope, id, props);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric(Object.assign({ namespace: 'AWS/Lambda', metricName }, props));
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', Object.assign({ statistic: 'sum' }, props));
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', Object.assign({ statistic: 'sum' }, props));
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', Object.assign({ statistic: 'sum' }, props));
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', Object.assign({ statistic: 'max' }, props));
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', Object.assign({ statistic: 'max' }, props));
    }
    /**
     * Export this Function (without the role)
     */
    export() {
        return {
            functionArn: new cdk.CfnOutput(this, 'FunctionArn', { value: this.functionArn }).makeImportValue().toString(),
            securityGroupId: this._connections && this._connections.securityGroups[0]
                ? new cdk.CfnOutput(this, 'SecurityGroupId', { value: this._connections.securityGroups[0].securityGroupId }).makeImportValue().toString()
                : undefined
        };
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     */
    addEnvironment(key, value) {
        if (!this.environment) {
            // TODO: add metadata
            return this;
        }
        this.environment[key] = value;
        return this;
    }
    /**
     * Adds a Lambda Layer to this Lambda function.
     *
     * @param layer the layer to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayer(layer) {
        if (this.layers.length === 5) {
            throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
        }
        if (layer.compatibleRuntimes && layer.compatibleRuntimes.indexOf(this.runtime) === -1) {
            const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
            throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
        }
        this.layers.push(layer);
        return this;
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want
     * to deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda source code, or
     *  omit to skip validation.
     * @param description A description for this version.
     * @returns A new Version object.
     */
    addVersion(name, codeSha256, description) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
        });
    }
    /**
     * Add a new version for this Lambda, always with a different name.
     *
     * This is similar to the {@link addVersion} method,
     * but useful when deploying this Lambda through CodePipeline with blue/green deployments.
     * When using {@link addVersion},
     * your Alias will not be updated until you change the name passed to {@link addVersion} in your CDK code.
     * When deploying through a Pipeline,
     * that might lead to a situation where a change to your Lambda application code will never be activated,
     * even though it traveled through the entire Pipeline,
     * because the Alias is still pointing to an old Version.
     * This method creates a new, unique Version every time the CDK code is executed,
     * and so prevents that from happening.
     */
    newVersion() {
        const now = new Date();
        return this.addVersion(now.toISOString());
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        return {
            variables: this.environment
        };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error(`Cannot configure 'securityGroup' or 'allowAllOutbound' without configuring a VPC`);
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error(`Configure 'allowAllOutbound' directly on the supplied SecurityGroup.`);
        }
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
            description: 'Automatic security group for Lambda Function ' + this.node.uniqueId,
            allowAllOutbound: props.allowAllOutbound
        });
        this._connections = new ec2.Connections({ securityGroups: [securityGroup] });
        // Pick subnets, make sure they're not Public. Routing through an IGW
        // won't work because the ENIs don't get a Public IP.
        // Why are we not simply forcing vpcSubnets? Because you might still be choosing
        // Isolated networks or selecting among 2 sets of Private subnets by name.
        const subnetIds = props.vpc.subnetIds(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId)) {
                throw new Error('Not possible to place Lambda Functions in a Public subnet');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: [securityGroup.securityGroupId]
        };
    }
    buildDeadLetterConfig(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriodSec: 1209600
        });
        this.addToRolePolicy(new iam.PolicyStatement()
            .addAction('sqs:SendMessage')
            .addResource(deadLetterQueue.queueArn));
        return {
            targetArn: deadLetterQueue.queueArn
        };
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.Disabled) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement()
            .addActions('xray:PutTraceSegments', 'xray:PutTelemetryRecords')
            .addAllResources());
        return {
            mode: Tracing[props.tracing]
        };
    }
}
exports.Function = Function;
class ImportedFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.canCreatePermissions = false;
        this.functionArn = props.functionArn;
        this.functionName = extractNameFromArn(props.functionArn);
        this.role = props.role;
        this.grantPrincipal = this.role || new iam.ImportedResourcePrincipal({ resource: this });
        if (props.securityGroupId) {
            this._connections = new ec2.Connections({
                securityGroups: [
                    ec2.SecurityGroup.import(this, 'SecurityGroup', { securityGroupId: props.securityGroupId })
                ]
            });
        }
    }
    export() {
        return this.props;
    }
}
exports.ImportedFunction = ImportedFunction;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return cdk.Fn.select(6, cdk.Fn.split(':', arn));
}
//# sourceMappingURL=data:application/json;base64,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