import cdk = require('@aws-cdk/cdk');
import { Code } from './code';
import { Runtime } from './runtime';
export interface LayerVersionProps {
    /**
     * The runtimes that this layer is compatible with.
     *
     * @default All runtimes are supported
     */
    readonly compatibleRuntimes?: Runtime[];
    /**
     * The content of this Layer. Using *inline* (per ``code.isInline``) code is not permitted.
     */
    readonly code: Code;
    /**
     * The description the this Lambda Layer.
     */
    readonly description?: string;
    /**
     * The SPDX licence identifier or URL to the license file for this layer.
     *
     * @default no license information will be recorded.
     */
    readonly license?: string;
    /**
     * The name of the layer.
     * @default a name will be generated.
     */
    readonly name?: string;
}
export interface ILayerVersion extends cdk.IConstruct {
    /**
     * The ARN of the Lambda Layer version that this Layer defines.
     */
    readonly layerVersionArn: string;
    /**
     * The runtimes compatible with this Layer.
     */
    readonly compatibleRuntimes?: Runtime[];
    /**
     * Exports this layer for use in another Stack. The resulting object can be passed to the ``LayerVersion.import``
     * function to obtain an ``ILayerVersion`` in the user stack.
     */
    export(): LayerVersionImportProps;
    /**
     * Grants usage of this layer to specific entities. Usage within the same account where the layer is defined is always
     * allowed and does not require calling this method. Note that the principal that creates the Lambda function using
     * the layer (for example, a CloudFormation changeset execution role) also needs to have the
     * ``lambda:GetLayerVersion`` permission on the layer version.
     *
     * @param id the ID of the grant in the construct tree.
     * @param grantee the identification of the grantee.
     */
    grantUsage(id: string, grantee: LayerVersionUsageGrantee): ILayerVersion;
}
/**
 * A reference to a Lambda Layer version.
 */
export declare abstract class LayerVersionBase extends cdk.Construct implements ILayerVersion {
    abstract readonly layerVersionArn: string;
    abstract readonly compatibleRuntimes?: Runtime[];
    grantUsage(id: string, grantee: LayerVersionUsageGrantee): ILayerVersion;
    export(): LayerVersionImportProps;
}
/**
 * Identification of an account (or organization) that is allowed to access a Lambda Layer Version.
 */
export interface LayerVersionUsageGrantee {
    /**
     * The AWS Account id of the account that is authorized to use a Lambda Layer Version. The wild-card ``'*'`` can be
     * used to grant access to "any" account (or any account in an organization when ``organizationId`` is specified).
     */
    readonly accountId: string;
    /**
     * The ID of the AWS Organization to hwich the grant is restricted.
     *
     * Can only be specified if ``accountId`` is ``'*'``
     */
    readonly organizationId?: string;
}
/**
 * Properties necessary to import a LayerVersion.
 */
export interface LayerVersionImportProps {
    /**
     * The ARN of the LayerVersion.
     */
    readonly layerVersionArn: string;
    /**
     * The list of compatible runtimes with this Layer.
     */
    readonly compatibleRuntimes?: Runtime[];
}
/**
 * Defines a new Lambda Layer version.
 */
export declare class LayerVersion extends LayerVersionBase {
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id    the id of the imported layer in the construct tree.
     * @param props the properties of the imported layer.
     */
    static import(scope: cdk.Construct, id: string, props: LayerVersionImportProps): ILayerVersion;
    readonly layerVersionArn: string;
    readonly compatibleRuntimes?: Runtime[];
    constructor(scope: cdk.Construct, id: string, props: LayerVersionProps);
}
/**
 * Properties of a Singleton Lambda Layer Version.
 */
export interface SingletonLayerVersionProps extends LayerVersionProps {
    /**
     * A unique identifier to identify this lambda layer version.
     *
     * The identifier should be unique across all layer providers.
     * We recommend generating a UUID per provider.
     */
    readonly uuid: string;
}
/**
 * A Singleton Lambda Layer Version. The construct gurantees exactly one LayerVersion will be created in a given Stack
 * for the provided ``uuid``. It is recommended to use ``uuidgen`` to create a new ``uuid`` each time a new singleton
 * layer is created.
 */
export declare class SingletonLayerVersion extends cdk.Construct implements ILayerVersion {
    private readonly layerVersion;
    constructor(scope: cdk.Construct, id: string, props: SingletonLayerVersionProps);
    readonly layerVersionArn: string;
    readonly compatibleRuntimes: Runtime[] | undefined;
    export(): LayerVersionImportProps;
    grantUsage(id: string, grantee: LayerVersionUsageGrantee): ILayerVersion;
    private ensureLayerVersion;
}
