"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const lambda_generated_1 = require("./lambda.generated");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends cdk.Construct {
    grantUsage(id, grantee) {
        if (grantee.organizationId != null && grantee.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${grantee.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: grantee.accountId,
            organizationId: grantee.organizationId,
        });
        return this;
    }
    export() {
        return {
            layerVersionArn: new cdk.CfnOutput(this, 'LayerVersionArn', { value: this.layerVersionArn }).makeImportValue().toString(),
            compatibleRuntimes: this.compatibleRuntimes,
        };
    }
}
exports.LayerVersionBase = LayerVersionBase;
/**
 * Defines a new Lambda Layer version.
 */
class LayerVersion extends LayerVersionBase {
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id    the id of the imported layer in the construct tree.
     * @param props the properties of the imported layer.
     */
    static import(scope, id, props) {
        return new ImportedLayerVersion(scope, id, props);
    }
    constructor(scope, id, props) {
        super(scope, id);
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        props.code.bind(this);
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: new cdk.Token(() => props.code._toJSON(resource)),
            description: props.description,
            layerName: props.name,
            licenseInfo: props.license,
        });
        this.layerVersionArn = resource.layerVersionArn;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
}
exports.LayerVersion = LayerVersion;
class ImportedLayerVersion extends LayerVersionBase {
    constructor(parent, id, props) {
        super(parent, id);
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        this.layerVersionArn = props.layerVersionArn;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
}
/**
 * A Singleton Lambda Layer Version. The construct gurantees exactly one LayerVersion will be created in a given Stack
 * for the provided ``uuid``. It is recommended to use ``uuidgen`` to create a new ``uuid`` each time a new singleton
 * layer is created.
 */
class SingletonLayerVersion extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.layerVersion = this.ensureLayerVersion(props);
    }
    get layerVersionArn() {
        return this.layerVersion.layerVersionArn;
    }
    get compatibleRuntimes() {
        return this.layerVersion.compatibleRuntimes;
    }
    export() {
        return {
            layerVersionArn: this.layerVersionArn,
            compatibleRuntimes: this.compatibleRuntimes,
        };
    }
    grantUsage(id, grantee) {
        this.layerVersion.grantUsage(id, grantee);
        return this;
    }
    ensureLayerVersion(props) {
        const singletonId = `SingletonLayer-${props.uuid}`;
        const existing = this.node.stack.node.tryFindChild(singletonId);
        if (existing) {
            return existing;
        }
        return new LayerVersion(this.node.stack, singletonId, props);
    }
}
exports.SingletonLayerVersion = SingletonLayerVersion;
//# sourceMappingURL=data:application/json;base64,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