"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
const path = require("path");
const code_1 = require("./code");
const runtime_1 = require("./runtime");
const singleton_lambda_1 = require("./singleton-lambda");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 */
class LogRetention extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Custom resource provider
        const provider = new singleton_lambda_1.SingletonFunction(this, 'Provider', {
            code: code_1.Code.asset(path.join(__dirname, 'log-retention-provider')),
            runtime: runtime_1.Runtime.NodeJS810,
            handler: 'index.handler',
            uuid: 'aae0aa3c-5b4d-4f87-b02d-85b201efdd8a',
            lambdaPurpose: 'LogRetention',
        });
        if (provider.role && !provider.role.node.tryFindChild('DefaultPolicy')) { // Avoid duplicate statements
            provider.role.addToPolicy(new iam.PolicyStatement()
                .addActions('logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy')
                // We need '*' here because we will also put a retention policy on
                // the log group of the provider function. Referencing it's name
                // creates a CF circular dependency.
                .addAllResources());
        }
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                RetentionInDays: props.retentionDays === Infinity ? undefined : props.retentionDays
            }
        });
    }
}
exports.LogRetention = LogRetention;
//# sourceMappingURL=data:application/json;base64,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