"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk_1 = require("@aws-cdk/cdk");
const lambda = require("../lib");
module.exports = {
    'version and aliases'(test) {
        const stack = new cdk_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NodeJS610,
        });
        const version = fn.addVersion('1');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        assert_1.expect(stack).to(assert_1.beASupersetOfTemplate({
            MyLambdaVersion16CDE3C40: {
                Type: "AWS::Lambda::Version",
                Properties: {
                    FunctionName: { Ref: "MyLambdaCCE802FB" }
                }
            },
            Alias325C5727: {
                Type: "AWS::Lambda::Alias",
                Properties: {
                    FunctionName: { Ref: "MyLambdaCCE802FB" },
                    FunctionVersion: stack.node.resolve(version.functionVersion),
                    Name: "prod"
                }
            }
        }));
        test.done();
    },
    'can use newVersion to create a new Version'(test) {
        const stack = new cdk_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NodeJS610,
        });
        const version = fn.newVersion();
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Version', {
            FunctionName: { Ref: "MyLambdaCCE802FB" },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
            FunctionName: { Ref: "MyLambdaCCE802FB" },
            Name: "prod"
        }));
        test.done();
    },
    'can add additional versions to alias'(test) {
        const stack = new cdk_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NodeJS610,
        });
        const version1 = fn.addVersion('1');
        const version2 = fn.addVersion('2');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version: version1,
            additionalVersions: [{ version: version2, weight: 0.1 }]
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
            FunctionVersion: stack.node.resolve(version1.functionVersion),
            RoutingConfig: {
                AdditionalVersionWeights: [
                    {
                        FunctionVersion: stack.node.resolve(version2.functionVersion),
                        FunctionWeight: 0.1
                    }
                ]
            }
        }));
        test.done();
    },
    'sanity checks on version weights'(test) {
        const stack = new cdk_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NodeJS610,
        });
        const version = fn.addVersion('1');
        // WHEN: Individual weight too high
        test.throws(() => {
            new lambda.Alias(stack, 'Alias1', {
                aliasName: 'prod', version,
                additionalVersions: [{ version, weight: 5 }]
            });
        });
        // WHEN: Sum too high
        test.throws(() => {
            new lambda.Alias(stack, 'Alias2', {
                aliasName: 'prod', version,
                additionalVersions: [{ version, weight: 0.5 }, { version, weight: 0.6 }]
            });
        });
        test.done();
    },
    'metric adds Resource: aliasArn to dimensions'(test) {
        const stack = new cdk_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NodeJS610,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // WHEN
        new cloudwatch.Alarm(stack, 'Alarm', {
            metric: alias.metric('Test'),
            alarmName: 'Test',
            threshold: 1,
            evaluationPeriods: 1
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            Dimensions: [{
                    Name: "FunctionName",
                    Value: {
                        Ref: "MyLambdaCCE802FB"
                    }
                }, {
                    Name: "Resource",
                    Value: {
                        Ref: "Alias325C5727"
                    }
                }]
        }));
        test.done();
    },
    'alias exposes real Lambdas role'(test) {
        const stack = new cdk_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NodeJS610,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // THEN
        test.equals(alias.role, fn.role);
        test.done();
    },
    'functionName is derived from the aliasArn so that dependencies are sound'(test) {
        const stack = new cdk_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NodeJS610,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // WHEN
        test.deepEqual(stack.node.resolve(alias.functionName), {
            "Fn::Join": [
                "",
                [
                    {
                        "Fn::Select": [
                            6,
                            {
                                "Fn::Split": [
                                    ":",
                                    {
                                        Ref: "Alias325C5727"
                                    }
                                ]
                            }
                        ]
                    },
                    ":prod"
                ]
            ]
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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