"use strict";
const assert_1 = require("@aws-cdk/assert");
const assets = require("@aws-cdk/assets");
const cdk = require("@aws-cdk/cdk");
const cxapi = require("@aws-cdk/cx-api");
const path = require("path");
const lambda = require("../lib");
function defineFunction(code, runtime = lambda.Runtime.NodeJS810) {
    const stack = new cdk.Stack();
    return new lambda.Function(stack, 'Func', {
        handler: 'foom',
        code, runtime
    });
}
function generateRandomString(bytes) {
    let s = '';
    for (let i = 0; i < bytes; ++i) {
        s += String.fromCharCode(Math.round(Math.random() * 256));
    }
    return s;
}
module.exports = {
    'lambda.Code.inline': {
        'fails if used with unsupported runtimes'(test) {
            test.throws(() => defineFunction(lambda.Code.inline('boom'), lambda.Runtime.Go1x), /Inline source not allowed for go1\.x/);
            test.throws(() => defineFunction(lambda.Code.inline('boom'), lambda.Runtime.Java8), /Inline source not allowed for java8/);
            test.done();
        },
        'fails if larger than 4096 bytes'(test) {
            test.throws(() => defineFunction(lambda.Code.inline(generateRandomString(4097)), lambda.Runtime.NodeJS610), /Lambda source is too large, must be <= 4096 but is 4097/);
            test.done();
        }
    },
    'lambda.Code.asset': {
        'determines packaging type from file type'(test) {
            // WHEN
            const fileAsset = lambda.Code.asset(path.join(__dirname, 'handler.zip'));
            const directoryAsset = lambda.Code.asset(path.join(__dirname, 'my-lambda-handler'));
            // THEN
            test.deepEqual(fileAsset.packaging, assets.AssetPackaging.File);
            test.deepEqual(directoryAsset.packaging, assets.AssetPackaging.ZipDirectory);
            test.done();
        },
        'fails if a non-zip asset is used'(test) {
            // GIVEN
            const fileAsset = lambda.Code.asset(path.join(__dirname, 'my-lambda-handler', 'index.py'));
            // THEN
            test.throws(() => defineFunction(fileAsset), /Asset must be a \.zip file or a directory/);
            test.done();
        },
        'only one Asset object gets created even if multiple functions use the same AssetCode'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'MyStack');
            const directoryAsset = lambda.Code.asset(path.join(__dirname, 'my-lambda-handler'));
            // WHEN
            new lambda.Function(stack, 'Func1', {
                handler: 'foom',
                runtime: lambda.Runtime.NodeJS810,
                code: directoryAsset
            });
            new lambda.Function(stack, 'Func2', {
                handler: 'foom',
                runtime: lambda.Runtime.NodeJS810,
                code: directoryAsset
            });
            // THEN
            const synthesized = app.synthesizeStack('MyStack');
            // Func1 has an asset, Func2 does not
            test.deepEqual(synthesized.metadata['/MyStack/Func1/Code'][0].type, 'aws:cdk:asset');
            test.deepEqual(synthesized.metadata['/MyStack/Func2/Code'], undefined);
            test.done();
        },
        'adds code asset metadata'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
            const location = path.join(__dirname, 'my-lambda-handler');
            // WHEN
            new lambda.Function(stack, 'Func1', {
                code: lambda.Code.asset(location),
                runtime: lambda.Runtime.NodeJS810,
                handler: 'foom',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Metadata: {
                    [cxapi.ASSET_RESOURCE_METADATA_PATH_KEY]: location,
                    [cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY]: 'Code'
                }
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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