"use strict";
const assert_1 = require("@aws-cdk/assert");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/cdk");
const path = require("path");
const lambda = require("../lib");
function newTestLambda(scope) {
    return new lambda.Function(scope, 'MyLambda', {
        code: new lambda.InlineCode('foo'),
        handler: 'bar',
        runtime: lambda.Runtime.Python27
    });
}
module.exports = {
    'default function'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs6.10' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'adds policy permissions'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            initialPolicy: [new iam.PolicyStatement().addAction("*").addAllResources()]
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaServiceRoleDefaultPolicy5BBC6F68: {
                    Type: "AWS::IAM::Policy",
                    Properties: {
                        PolicyDocument: {
                            Statement: [
                                {
                                    Action: "*",
                                    Effect: "Allow",
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        PolicyName: "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        Roles: [
                            {
                                Ref: "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs6.10' },
                    DependsOn: ['MyLambdaServiceRoleDefaultPolicy5BBC6F68', 'MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'fails if inline code is used for an invalid runtime'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'bar',
            runtime: lambda.Runtime.DotNetCore2
        }));
        test.done();
    },
    'addToResourcePolicy': {
        'can be used to add permissions to the Lambda function'(test) {
            const stack = new cdk.Stack();
            const fn = newTestLambda(stack);
            fn.addPermission('S3Permission', {
                action: 'lambda:*',
                principal: new iam.ServicePrincipal('s3.amazonaws.com'),
                sourceAccount: stack.accountId,
                sourceArn: 'arn:aws:s3:::my_bucket'
            });
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyLambdaServiceRole4539ECB6": {
                        "Type": "AWS::IAM::Role",
                        "Properties": {
                            "AssumeRolePolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "sts:AssumeRole",
                                        "Effect": "Allow",
                                        "Principal": {
                                            "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "ManagedPolicyArns": 
                            // tslint:disable-next-line:max-line-length
                            [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                        }
                    },
                    "MyLambdaCCE802FB": {
                        "Type": "AWS::Lambda::Function",
                        "Properties": {
                            "Code": {
                                "ZipFile": "foo"
                            },
                            "Handler": "bar",
                            "Role": {
                                "Fn::GetAtt": [
                                    "MyLambdaServiceRole4539ECB6",
                                    "Arn"
                                ]
                            },
                            "Runtime": "python2.7"
                        },
                        "DependsOn": [
                            "MyLambdaServiceRole4539ECB6"
                        ]
                    },
                    "MyLambdaS3Permission99D0EA08": {
                        "Type": "AWS::Lambda::Permission",
                        "Properties": {
                            "Action": "lambda:*",
                            "FunctionName": {
                                "Fn::GetAtt": [
                                    "MyLambdaCCE802FB",
                                    "Arn"
                                ]
                            },
                            "Principal": "s3.amazonaws.com",
                            "SourceAccount": {
                                "Ref": "AWS::AccountId"
                            },
                            "SourceArn": "arn:aws:s3:::my_bucket"
                        }
                    }
                }
            });
            test.done();
        },
        'fails if the principal is not a service or account principals'(test) {
            const stack = new cdk.Stack();
            const fn = newTestLambda(stack);
            test.throws(() => fn.addPermission('F1', { principal: new iam.ArnPrincipal('just:arn') }), /Invalid principal type for Lambda permission statement/);
            fn.addPermission('S1', { principal: new iam.ServicePrincipal('my-service') });
            fn.addPermission('S2', { principal: new iam.AccountPrincipal('account') });
            test.done();
        },
        'BYORole'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const role = new iam.Role(stack, 'SomeRole', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            role.addToPolicy(new iam.PolicyStatement().addAction('confirm:itsthesame'));
            // WHEN
            const fn = new lambda.Function(stack, 'Function', {
                code: new lambda.InlineCode('test'),
                runtime: lambda.Runtime.Python36,
                handler: 'index.test',
                role,
                initialPolicy: [
                    new iam.PolicyStatement().addAction('inline:inline')
                ]
            });
            fn.addToRolePolicy(new iam.PolicyStatement().addAction('explicit:explicit'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                "PolicyDocument": {
                    "Version": "2012-10-17",
                    "Statement": [
                        { "Action": "confirm:itsthesame", "Effect": "Allow" },
                        { "Action": "inline:inline", "Effect": "Allow" },
                        { "Action": "explicit:explicit", "Effect": "Allow" }
                    ],
                },
            }));
            test.done();
        }
    },
    'import/export': {
        'lambda.export() can be used to add Outputs to the stack and returns an IFunction object'(test) {
            // GIVEN
            const stack1 = new cdk.Stack();
            const stack2 = new cdk.Stack();
            const fn = newTestLambda(stack1);
            // WHEN
            const props = fn.export();
            const imported = lambda.Function.import(stack2, 'Imported', props);
            // Can call addPermission() but it won't do anything
            imported.addPermission('Hello', {
                principal: new iam.ServicePrincipal('harry')
            });
            test.done();
        },
    },
    'Lambda can serve as EventRule target, permission gets added'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = newTestLambda(stack);
        const rule1 = new events.EventRule(stack, 'Rule', { scheduleExpression: 'rate(1 minute)' });
        const rule2 = new events.EventRule(stack, 'Rule2', { scheduleExpression: 'rate(5 minutes)' });
        // WHEN
        rule1.addTarget(fn);
        rule2.addTarget(fn);
        // THEN
        const lambdaId = "MyLambdaCCE802FB";
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            "Action": "lambda:InvokeFunction",
            "FunctionName": {
                "Fn::GetAtt": [
                    lambdaId,
                    "Arn"
                ]
            },
            "Principal": "events.amazonaws.com",
            "SourceArn": { "Fn::GetAtt": ["Rule4C995B7F", "Arn"] }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            "Action": "lambda:InvokeFunction",
            "FunctionName": {
                "Fn::GetAtt": [
                    lambdaId,
                    "Arn"
                ]
            },
            "Principal": "events.amazonaws.com",
            "SourceArn": { "Fn::GetAtt": ["Rule270732244", "Arn"] }
        }));
        assert_1.expect(stack).to(assert_1.countResources('AWS::Events::Rule', 2));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            "Targets": [
                {
                    "Arn": { "Fn::GetAtt": [lambdaId, "Arn"] },
                    "Id": "MyLambda"
                }
            ]
        }));
        test.done();
    },
    'Lambda code can be read from a local directory via an asset'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: lambda.Code.directory(path.join(__dirname, 'my-lambda-handler')),
            handler: 'index.handler',
            runtime: lambda.Runtime.Python36
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Code": {
                "S3Bucket": {
                    "Ref": "MyLambdaCodeS3BucketC82A5870"
                },
                "S3Key": { "Fn::Join": ["", [
                            { "Fn::Select": [0, { "Fn::Split": ["||", { "Ref": "MyLambdaCodeS3VersionKey47762537" }] }] },
                            { "Fn::Select": [1, { "Fn::Split": ["||", { "Ref": "MyLambdaCodeS3VersionKey47762537" }] }] },
                        ]] }
            },
            "Handler": "index.handler",
            "Role": {
                "Fn::GetAtt": [
                    "MyLambdaServiceRole4539ECB6",
                    "Arn"
                ]
            },
            "Runtime": "python3.6"
        }));
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to true and functionName defined by client'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            functionName: 'OneFunctionToRuleThemAll',
            deadLetterQueueEnabled: true
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "MyLambdaDeadLetterQueue399EEA2D",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaDeadLetterQueue399EEA2D": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "MessageRetentionPeriod": 1209600
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs6.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "MyLambdaDeadLetterQueue399EEA2D",
                                    "Arn"
                                ]
                            }
                        },
                        "FunctionName": "OneFunctionToRuleThemAll"
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to true and functionName not defined by client'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            deadLetterQueueEnabled: true,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "MyLambdaDeadLetterQueue399EEA2D",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaDeadLetterQueue399EEA2D": {
                    "Type": "AWS::SQS::Queue",
                    "Properties": {
                        "MessageRetentionPeriod": 1209600
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs6.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "MyLambdaDeadLetterQueue399EEA2D",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client sets deadLetterQueueEnabled to false'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            deadLetterQueueEnabled: false,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs6.10"
                    },
                    "DependsOn": [
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        });
        test.done();
    },
    'default function with SQS DLQ when client provides Queue to be used as DLQ'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriodSec: 1209600
        });
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            deadLetterQueue: dlQueue,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "DeadLetterQueue9F481546",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs6.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "DeadLetterQueue9F481546",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6"
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'default function with SQS DLQ when client provides Queue to be used as DLQ and deadLetterQueueEnabled set to true'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriodSec: 1209600
        });
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            deadLetterQueueEnabled: true,
            deadLetterQueue: dlQueue,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyLambdaServiceRole4539ECB6": {
                    "Type": "AWS::IAM::Role",
                    "Properties": {
                        "AssumeRolePolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sts:AssumeRole",
                                    "Effect": "Allow",
                                    "Principal": {
                                        "Service": { "Fn::Join": ["", ["lambda.", { Ref: "AWS::URLSuffix" }]] }
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "ManagedPolicyArns": [
                            {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            "Ref": "AWS::Partition"
                                        },
                                        ":iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
                                    ]
                                ]
                            }
                        ]
                    }
                },
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": "sqs:SendMessage",
                                    "Effect": "Allow",
                                    "Resource": {
                                        "Fn::GetAtt": [
                                            "DeadLetterQueue9F481546",
                                            "Arn"
                                        ]
                                    }
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "Roles": [
                            {
                                "Ref": "MyLambdaServiceRole4539ECB6"
                            }
                        ]
                    }
                },
                "MyLambdaCCE802FB": {
                    "Type": "AWS::Lambda::Function",
                    "Properties": {
                        "Code": {
                            "ZipFile": "foo"
                        },
                        "Handler": "index.handler",
                        "Role": {
                            "Fn::GetAtt": [
                                "MyLambdaServiceRole4539ECB6",
                                "Arn"
                            ]
                        },
                        "Runtime": "nodejs6.10",
                        "DeadLetterConfig": {
                            "TargetArn": {
                                "Fn::GetAtt": [
                                    "DeadLetterQueue9F481546",
                                    "Arn"
                                ]
                            }
                        }
                    },
                    "DependsOn": [
                        "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                        "MyLambdaServiceRole4539ECB6",
                    ]
                }
            }
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'error when default function with SQS DLQ when client provides Queue to be used as DLQ and deadLetterQueueEnabled set to false'(test) {
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriodSec: 1209600
        });
        test.throws(() => new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            deadLetterQueueEnabled: false,
            deadLetterQueue: dlQueue,
        }), /deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false/);
        test.done();
    },
    'default function with Active tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            tracing: lambda.Tracing.Active
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs6.10",
                "TracingConfig": {
                    "Mode": "Active"
                }
            },
            "DependsOn": [
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                "MyLambdaServiceRole4539ECB6",
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default function with PassThrough tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            tracing: lambda.Tracing.PassThrough
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs6.10",
                "TracingConfig": {
                    "Mode": "PassThrough"
                }
            },
            "DependsOn": [
                "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
                "MyLambdaServiceRole4539ECB6",
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default function with Disabled tracing'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
            tracing: lambda.Tracing.Disabled
        });
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "xray:PutTraceSegments",
                            "xray:PutTelemetryRecords"
                        ],
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "MyLambdaServiceRoleDefaultPolicy5BBC6F68",
            "Roles": [
                {
                    "Ref": "MyLambdaServiceRole4539ECB6"
                }
            ]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            "Properties": {
                "Code": {
                    "ZipFile": "foo"
                },
                "Handler": "index.handler",
                "Role": {
                    "Fn::GetAtt": [
                        "MyLambdaServiceRole4539ECB6",
                        "Arn"
                    ]
                },
                "Runtime": "nodejs6.10"
            },
            "DependsOn": [
                "MyLambdaServiceRole4539ECB6"
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'grantInvoke adds iam:InvokeFunction'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.AccountPrincipal('1234'),
        });
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        // WHEN
        fn.grantInvoke(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'lambda:InvokeFunction',
                        Effect: 'Allow',
                        Resource: { "Fn::GetAtt": ["Function76856677", "Arn"] }
                    }
                ]
            }
        }));
        test.done();
    },
    'Can use metricErrors on a lambda Function'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        // THEN
        test.deepEqual(stack.node.resolve(fn.metricErrors()), {
            dimensions: { FunctionName: { Ref: 'Function76856677' } },
            namespace: 'AWS/Lambda',
            metricName: 'Errors',
            periodSec: 300,
            statistic: 'Sum',
        });
        test.done();
    },
    'addEventSource calls bind'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'Function', {
            code: lambda.Code.inline('xxx'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS810,
        });
        let bindTarget;
        class EventSourceMock {
            bind(target) {
                bindTarget = target;
            }
        }
        // WHEN
        fn.addEventSource(new EventSourceMock());
        // THEN
        test.same(bindTarget, fn);
        test.done();
    },
    'support inline code for Ruby runtime'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.Ruby25,
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'ruby2.5' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'using an incompatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const layer = lambda.LayerVersion.import(stack, 'TestLayer', {
            layerVersionArn: 'arn:aws:...',
            compatibleRuntimes: [lambda.Runtime.NodeJS810],
        });
        // THEN
        test.throws(() => new lambda.Function(stack, 'Function', {
            layers: [layer],
            runtime: lambda.Runtime.NodeJS610,
            code: lambda.Code.inline('exports.main = function() { console.log("DONE"); }'),
            handler: 'index.main'
        }), /nodejs6.10 is not in \[nodejs8.10\]/);
        test.done();
    },
    'using more than 5 layers'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const layers = new Array(6).fill(lambda.LayerVersion.import(stack, 'TestLayer', {
            layerVersionArn: 'arn:aws:...',
            compatibleRuntimes: [lambda.Runtime.NodeJS810],
        }));
        // THEN
        test.throws(() => new lambda.Function(stack, 'Function', {
            layers,
            runtime: lambda.Runtime.NodeJS810,
            code: lambda.Code.inline('exports.main = function() { console.log("DONE"); }'),
            handler: 'index.main'
        }), /Unable to add layer:/);
        test.done();
    },
    'environment variables are prohibited in China'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'MyLambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NodeJS,
                environment: {
                    SOME: 'Variable'
                }
            });
        }, /Environment variables are not supported/);
        test.done();
    },
    'environment variables work in an unspecified region'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS,
            environment: {
                SOME: 'Variable'
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            Environment: {
                Variables: {
                    SOME: "Variable"
                }
            }
        }));
        test.done();
    },
    'support reserved concurrent executions'(test) {
        const stack = new cdk.Stack();
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS,
            reservedConcurrentExecutions: 10
        });
        assert_1.expect(stack).toMatch({ Resources: { MyLambdaServiceRole4539ECB6: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: { "Fn::Join": ["", ['lambda.', { Ref: "AWS::URLSuffix" }]] } } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: 
                        // arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
                        // tslint:disable-next-line:max-line-length
                        [{ 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']] }],
                    } },
                MyLambdaCCE802FB: { Type: 'AWS::Lambda::Function',
                    Properties: { Code: { ZipFile: 'foo' },
                        Handler: 'index.handler',
                        ReservedConcurrentExecutions: 10,
                        Role: { 'Fn::GetAtt': ['MyLambdaServiceRole4539ECB6', 'Arn'] },
                        Runtime: 'nodejs' },
                    DependsOn: ['MyLambdaServiceRole4539ECB6'] } } });
        test.done();
    },
    'its possible to specify event sources upon creation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        let bindCount = 0;
        class EventSource {
            bind(_) {
                bindCount++;
            }
        }
        // WHEN
        new lambda.Function(stack, 'fn', {
            code: lambda.Code.inline('boom'),
            runtime: lambda.Runtime.NodeJS810,
            handler: 'index.bam',
            events: [
                new EventSource(),
                new EventSource(),
            ]
        });
        // THEN
        test.deepEqual(bindCount, 2);
        test.done();
    },
    'Provided Runtime returns the right values'(test) {
        const rt = lambda.Runtime.Provided;
        test.equal(rt.name, 'provided');
        test.equal(rt.supportsInlineCode, false);
        test.done();
    },
    'specify log retention'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS,
            logRetentionDays: logs.RetentionDays.OneMonth
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            'LogGroupName': {
                'Fn::Join': [
                    '',
                    [
                        '/aws/lambda/',
                        {
                            Ref: 'MyLambdaCCE802FB'
                        }
                    ]
                ]
            },
            'RetentionInDays': 30
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5sYW1iZGEuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmxhbWJkYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQWlHO0FBQ2pHLDhDQUErQztBQUMvQyx3Q0FBeUM7QUFDekMsMENBQTJDO0FBQzNDLHdDQUF5QztBQUN6QyxvQ0FBcUM7QUFFckMsNkJBQThCO0FBQzlCLGlDQUFrQztBQTB6Q2xDLFNBQVMsYUFBYSxDQUFDLEtBQW9CO0lBQ3pDLE9BQU8sSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDNUMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7UUFDbEMsT0FBTyxFQUFFLEtBQUs7UUFDZCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRO0tBQ2pDLENBQUMsQ0FBQztBQUNMLENBQUM7QUE1ekNELGlCQUFTO0lBQ1Asa0JBQWtCLENBQUMsSUFBVTtRQUMzQixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUNyQyxJQUFJLEVBQUUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNsQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTO1NBQ2xDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsRUFBRSxTQUFTLEVBQy9CLEVBQUUsMkJBQTJCLEVBQzFCLEVBQUUsSUFBSSxFQUFFLGdCQUFnQjtvQkFDeEIsVUFBVSxFQUNULEVBQUUsd0JBQXdCLEVBQ3pCLEVBQUUsU0FBUyxFQUNWLENBQUUsRUFBRSxNQUFNLEVBQUUsZ0JBQWdCO29DQUMxQixNQUFNLEVBQUUsT0FBTztvQ0FDZixTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUU7NEJBQ3pGLE9BQU8sRUFBRSxZQUFZLEVBQUU7d0JBQzFCLGlCQUFpQjt3QkFDakIsbUVBQW1FO3dCQUNsRSwyQ0FBMkM7d0JBQzNDLENBQUMsRUFBQyxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBQyxHQUFHLEVBQUUsZ0JBQWdCLEVBQUMsRUFBRSwyREFBMkQsQ0FBQyxDQUFDLEVBQUMsQ0FBQztxQkFDcEgsRUFBQztnQkFDSixnQkFBZ0IsRUFDZixFQUFFLElBQUksRUFBRSx1QkFBdUI7b0JBQy9CLFVBQVUsRUFDVCxFQUFFLElBQUksRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUU7d0JBQzFCLE9BQU8sRUFBRSxlQUFlO3dCQUN4QixJQUFJLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSw2QkFBNkIsRUFBRSxLQUFLLENBQUUsRUFBRTt3QkFDaEUsT0FBTyxFQUFFLFlBQVksRUFBRTtvQkFDeEIsU0FBUyxFQUFFLENBQUUsNkJBQTZCLENBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx5QkFBeUIsQ0FBQyxJQUFVO1FBQ2xDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ3JDLElBQUksRUFBRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ2xDLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVM7WUFDakMsYUFBYSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxFQUFFLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDO1NBQzVFLENBQUMsQ0FBQztRQUNILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsRUFBRSxTQUFTLEVBQy9CLEVBQUUsMkJBQTJCLEVBQzFCLEVBQUUsSUFBSSxFQUFFLGdCQUFnQjtvQkFDeEIsVUFBVSxFQUNULEVBQUUsd0JBQXdCLEVBQ3pCLEVBQUUsU0FBUyxFQUNWLENBQUUsRUFBRSxNQUFNLEVBQUUsZ0JBQWdCO29DQUMxQixNQUFNLEVBQUUsT0FBTztvQ0FDZixTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUU7NEJBQ3pGLE9BQU8sRUFBRSxZQUFZLEVBQUU7d0JBQzFCLGlCQUFpQjt3QkFDakIsMkNBQTJDO3dCQUMzQyxDQUFDLEVBQUMsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsTUFBTSxFQUFFLEVBQUMsR0FBRyxFQUFFLGdCQUFnQixFQUFDLEVBQUUsMkRBQTJELENBQUMsQ0FBQyxFQUFDLENBQUM7cUJBQ3JILEVBQUM7Z0JBQ0Ysd0NBQXdDLEVBQUU7b0JBQ3hDLElBQUksRUFBRSxrQkFBa0I7b0JBQ3hCLFVBQVUsRUFBRTt3QkFDVixjQUFjLEVBQUU7NEJBQ2hCLFNBQVMsRUFBRTtnQ0FDVDtvQ0FDQSxNQUFNLEVBQUUsR0FBRztvQ0FDWCxNQUFNLEVBQUUsT0FBTztvQ0FDZixRQUFRLEVBQUUsR0FBRztpQ0FDWjs2QkFDRjs0QkFDRCxPQUFPLEVBQUUsWUFBWTt5QkFDcEI7d0JBQ0QsVUFBVSxFQUFFLDBDQUEwQzt3QkFDdEQsS0FBSyxFQUFFOzRCQUNQO2dDQUNFLEdBQUcsRUFBRSw2QkFBNkI7NkJBQ25DO3lCQUNBO3FCQUNGO2lCQUNGO2dCQUNELGdCQUFnQixFQUNmLEVBQUUsSUFBSSxFQUFFLHVCQUF1QjtvQkFDL0IsVUFBVSxFQUNULEVBQUUsSUFBSSxFQUFFLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRTt3QkFDMUIsT0FBTyxFQUFFLGVBQWU7d0JBQ3hCLElBQUksRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLDZCQUE2QixFQUFFLEtBQUssQ0FBRSxFQUFFO3dCQUNoRSxPQUFPLEVBQUUsWUFBWSxFQUFFO29CQUN4QixTQUFTLEVBQUUsQ0FBRSwwQ0FBMEMsRUFBRSw2QkFBNkIsQ0FBRSxFQUFFLEVBQUUsRUFBRSxDQUFFLENBQUM7UUFDdEcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBRWQsQ0FBQztJQUVELHFEQUFxRCxDQUFDLElBQVU7UUFDOUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUN2RCxJQUFJLEVBQUUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNsQyxPQUFPLEVBQUUsS0FBSztZQUNkLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7U0FDcEMsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQscUJBQXFCLEVBQUU7UUFDckIsdURBQXVELENBQUMsSUFBVTtZQUNoRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLEVBQUUsR0FBRyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7WUFFaEMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxjQUFjLEVBQUU7Z0JBQy9CLE1BQU0sRUFBRSxVQUFVO2dCQUNsQixTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLENBQUM7Z0JBQ3ZELGFBQWEsRUFBRSxLQUFLLENBQUMsU0FBUztnQkFDOUIsU0FBUyxFQUFFLHdCQUF3QjthQUNwQyxDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUNwQixXQUFXLEVBQUU7b0JBQ2IsNkJBQTZCLEVBQUU7d0JBQzdCLE1BQU0sRUFBRSxnQkFBZ0I7d0JBQ3hCLFlBQVksRUFBRTs0QkFDZCwwQkFBMEIsRUFBRTtnQ0FDMUIsV0FBVyxFQUFFO29DQUNiO3dDQUNFLFFBQVEsRUFBRSxnQkFBZ0I7d0NBQzFCLFFBQVEsRUFBRSxPQUFPO3dDQUNqQixXQUFXLEVBQUU7NENBQ2IsU0FBUyxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQyxFQUFFO3lDQUN0RTtxQ0FDRjtpQ0FDQTtnQ0FDRCxTQUFTLEVBQUUsWUFBWTs2QkFDeEI7NEJBQ0QsbUJBQW1COzRCQUNuQiwyQ0FBMkM7NEJBQzNDLENBQUMsRUFBQyxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBQyxHQUFHLEVBQUUsZ0JBQWdCLEVBQUMsRUFBRSwyREFBMkQsQ0FBQyxDQUFDLEVBQUMsQ0FBQzt5QkFDbkg7cUJBQ0Y7b0JBQ0Qsa0JBQWtCLEVBQUU7d0JBQ2xCLE1BQU0sRUFBRSx1QkFBdUI7d0JBQy9CLFlBQVksRUFBRTs0QkFDZCxNQUFNLEVBQUU7Z0NBQ04sU0FBUyxFQUFFLEtBQUs7NkJBQ2pCOzRCQUNELFNBQVMsRUFBRSxLQUFLOzRCQUNoQixNQUFNLEVBQUU7Z0NBQ04sWUFBWSxFQUFFO29DQUNaLDZCQUE2QjtvQ0FDN0IsS0FBSztpQ0FDTjs2QkFDRjs0QkFDRCxTQUFTLEVBQUUsV0FBVzt5QkFDckI7d0JBQ0QsV0FBVyxFQUFFOzRCQUNiLDZCQUE2Qjt5QkFDNUI7cUJBQ0Y7b0JBQ0QsOEJBQThCLEVBQUU7d0JBQzlCLE1BQU0sRUFBRSx5QkFBeUI7d0JBQ2pDLFlBQVksRUFBRTs0QkFDZCxRQUFRLEVBQUUsVUFBVTs0QkFDcEIsY0FBYyxFQUFFO2dDQUNkLFlBQVksRUFBRTtvQ0FDWixrQkFBa0I7b0NBQ2xCLEtBQUs7aUNBQ047NkJBQ0Y7NEJBQ0QsV0FBVyxFQUFFLGtCQUFrQjs0QkFDL0IsZUFBZSxFQUFFO2dDQUNmLEtBQUssRUFBRSxnQkFBZ0I7NkJBQ3hCOzRCQUNELFdBQVcsRUFBRSx3QkFBd0I7eUJBQ3BDO3FCQUNGO2lCQUNBO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELCtEQUErRCxDQUFDLElBQVU7WUFDeEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxFQUFFLEdBQUcsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBRWhDLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsRUFDdkYsd0RBQXdELENBQUMsQ0FBQztZQUU1RCxFQUFFLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLENBQUM7WUFDOUUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBRTNFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxTQUFTLENBQUMsSUFBVTtZQUNsQixRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7Z0JBQzNDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQzthQUM1RCxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsRUFBRSxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUM7WUFFNUUsT0FBTztZQUNQLE1BQU0sRUFBRSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO2dCQUNoRCxJQUFJLEVBQUUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQztnQkFDbkMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsUUFBUTtnQkFDaEMsT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLElBQUk7Z0JBQ0osYUFBYSxFQUFFO29CQUNiLElBQUksR0FBRyxDQUFDLGVBQWUsRUFBRSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUM7aUJBQ3JEO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsRUFBRSxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxTQUFTLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDO1lBRTdFLE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ2hELGdCQUFnQixFQUFFO29CQUNoQixTQUFTLEVBQUUsWUFBWTtvQkFDdkIsV0FBVyxFQUFFO3dCQUNYLEVBQUUsUUFBUSxFQUFFLG9CQUFvQixFQUFFLFFBQVEsRUFBRSxPQUFPLEVBQUU7d0JBQ3JELEVBQUUsUUFBUSxFQUFFLGVBQWUsRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFO3dCQUNoRCxFQUFFLFFBQVEsRUFBRSxtQkFBbUIsRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFO3FCQUNyRDtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsZUFBZSxFQUFFO1FBQ2YseUZBQXlGLENBQUMsSUFBVTtZQUNsRyxRQUFRO1lBQ1IsTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDL0IsTUFBTSxNQUFNLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDL0IsTUFBTSxFQUFFLEdBQUcsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1lBRWpDLE9BQU87WUFDUCxNQUFNLEtBQUssR0FBRyxFQUFFLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDMUIsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUVuRSxvREFBb0Q7WUFDcEQsUUFBUSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUU7Z0JBQzlCLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUM7YUFDN0MsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsNkRBQTZELENBQUMsSUFBVTtRQUN0RSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxFQUFFLEdBQUcsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2hDLE1BQU0sS0FBSyxHQUFHLElBQUksTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsa0JBQWtCLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBQzVGLE1BQU0sS0FBSyxHQUFHLElBQUksTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsa0JBQWtCLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO1FBRTlGLE9BQU87UUFDUCxLQUFLLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ3BCLEtBQUssQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFcEIsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLGtCQUFrQixDQUFDO1FBRXBDLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxRQUFRLEVBQUUsdUJBQXVCO1lBQ2pDLGNBQWMsRUFBRTtnQkFDZCxZQUFZLEVBQUU7b0JBQ1osUUFBUTtvQkFDUixLQUFLO2lCQUNOO2FBQ0Y7WUFDRCxXQUFXLEVBQUUsc0JBQXNCO1lBQ25DLFdBQVcsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGNBQWMsRUFBRSxLQUFLLENBQUUsRUFBRTtTQUN6RCxDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx5QkFBeUIsRUFBRTtZQUN2RCxRQUFRLEVBQUUsdUJBQXVCO1lBQ2pDLGNBQWMsRUFBRTtnQkFDZCxZQUFZLEVBQUU7b0JBQ1osUUFBUTtvQkFDUixLQUFLO2lCQUNOO2FBQ0Y7WUFDRCxXQUFXLEVBQUUsc0JBQXNCO1lBQ25DLFdBQVcsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGVBQWUsRUFBRSxLQUFLLENBQUUsRUFBRTtTQUMxRCxDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsdUJBQWMsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3pELGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxtQkFBbUIsRUFBRTtZQUNqRCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0EsS0FBSyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsUUFBUSxFQUFFLEtBQUssQ0FBRSxFQUFFO29CQUM1QyxJQUFJLEVBQUUsVUFBVTtpQkFDZjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkRBQTZELENBQUMsSUFBVTtRQUN0RSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDckMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLG1CQUFtQixDQUFDLENBQUM7WUFDdEUsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsUUFBUTtTQUNqQyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO1lBQ3JELE1BQU0sRUFBRTtnQkFDTixVQUFVLEVBQUU7b0JBQ1osS0FBSyxFQUFFLDhCQUE4QjtpQkFDcEM7Z0JBQ0QsT0FBTyxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUUsRUFBRSxFQUFFOzRCQUM3QixFQUFDLFlBQVksRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFDLFdBQVcsRUFBRSxDQUFDLElBQUksRUFBRSxFQUFDLEtBQUssRUFBRSxrQ0FBa0MsRUFBQyxDQUFDLEVBQUMsQ0FBQyxFQUFDOzRCQUN2RixFQUFDLFlBQVksRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFDLFdBQVcsRUFBRSxDQUFDLElBQUksRUFBRSxFQUFDLEtBQUssRUFBRSxrQ0FBa0MsRUFBQyxDQUFDLEVBQUMsQ0FBQyxFQUFDO3lCQUN0RixDQUFDLEVBQUM7YUFDSjtZQUNELFNBQVMsRUFBRSxlQUFlO1lBQzFCLE1BQU0sRUFBRTtnQkFDTixZQUFZLEVBQUU7b0JBQ2QsNkJBQTZCO29CQUM3QixLQUFLO2lCQUNKO2FBQ0Y7WUFDRCxTQUFTLEVBQUUsV0FBVztTQUN2QixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrSEFBa0gsQ0FBQyxJQUFVO1FBQzNILE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ3JDLElBQUksRUFBRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ2xDLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVM7WUFDakMsWUFBWSxFQUFFLDBCQUEwQjtZQUN4QyxzQkFBc0IsRUFBRSxJQUFJO1NBQzdCLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQ25CO1lBQ0UsV0FBVyxFQUFFO2dCQUNYLDZCQUE2QixFQUFFO29CQUMvQixNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixZQUFZLEVBQUU7d0JBQ1osMEJBQTBCLEVBQUU7NEJBQzVCLFdBQVcsRUFBRTtnQ0FDWDtvQ0FDQSxRQUFRLEVBQUUsZ0JBQWdCO29DQUMxQixRQUFRLEVBQUUsT0FBTztvQ0FDakIsV0FBVyxFQUFFO3dDQUNYLFNBQVMsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsRUFBRTtxQ0FDeEU7aUNBQ0E7NkJBQ0Y7NEJBQ0QsU0FBUyxFQUFFLFlBQVk7eUJBQ3RCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNyQjtnQ0FDRSxVQUFVLEVBQUU7b0NBQ1osRUFBRTtvQ0FDRjt3Q0FDRSxNQUFNO3dDQUNOOzRDQUNBLEtBQUssRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELDJEQUEyRDtxQ0FDNUQ7aUNBQ0E7NkJBQ0Y7eUJBQ0E7cUJBQ0Y7aUJBQ0E7Z0JBQ0QsMENBQTBDLEVBQUU7b0JBQzVDLE1BQU0sRUFBRSxrQkFBa0I7b0JBQzFCLFlBQVksRUFBRTt3QkFDWixnQkFBZ0IsRUFBRTs0QkFDbEIsV0FBVyxFQUFFO2dDQUNYO29DQUNBLFFBQVEsRUFBRSxpQkFBaUI7b0NBQzNCLFFBQVEsRUFBRSxPQUFPO29DQUNqQixVQUFVLEVBQUU7d0NBQ1YsWUFBWSxFQUFFOzRDQUNkLGlDQUFpQzs0Q0FDakMsS0FBSzt5Q0FDSjtxQ0FDRjtpQ0FDQTs2QkFDRjs0QkFDRCxTQUFTLEVBQUUsWUFBWTt5QkFDdEI7d0JBQ0QsWUFBWSxFQUFFLDBDQUEwQzt3QkFDeEQsT0FBTyxFQUFFOzRCQUNUO2dDQUNFLEtBQUssRUFBRSw2QkFBNkI7NkJBQ3JDO3lCQUNBO3FCQUNGO2lCQUNBO2dCQUNELGlDQUFpQyxFQUFFO29CQUNuQyxNQUFNLEVBQUUsaUJBQWlCO29CQUN6QixZQUFZLEVBQUU7d0JBQ1osd0JBQXdCLEVBQUUsT0FBTztxQkFDbEM7aUJBQ0E7Z0JBQ0Qsa0JBQWtCLEVBQUU7b0JBQ3BCLE1BQU0sRUFBRSx1QkFBdUI7b0JBQy9CLFlBQVksRUFBRTt3QkFDWixNQUFNLEVBQUU7NEJBQ1IsU0FBUyxFQUFFLEtBQUs7eUJBQ2Y7d0JBQ0QsU0FBUyxFQUFFLGVBQWU7d0JBQzFCLE1BQU0sRUFBRTs0QkFDUixZQUFZLEVBQUU7Z0NBQ1osNkJBQTZCO2dDQUM3QixLQUFLOzZCQUNOO3lCQUNBO3dCQUNELFNBQVMsRUFBRSxZQUFZO3dCQUN2QixrQkFBa0IsRUFBRTs0QkFDcEIsV0FBVyxFQUFFO2dDQUNYLFlBQVksRUFBRTtvQ0FDZCxpQ0FBaUM7b0NBQ2pDLEtBQUs7aUNBQ0o7NkJBQ0Y7eUJBQ0E7d0JBQ0QsY0FBYyxFQUFFLDBCQUEwQjtxQkFDM0M7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLDBDQUEwQzt3QkFDMUMsNkJBQTZCO3FCQUM5QjtpQkFDQTthQUNGO1NBQ0EsQ0FDSixDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNIQUFzSCxDQUFDLElBQVU7UUFDL0gsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDckMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztZQUNqQyxzQkFBc0IsRUFBRSxJQUFJO1NBQzdCLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQ25CO1lBQ0UsV0FBVyxFQUFFO2dCQUNYLDZCQUE2QixFQUFFO29CQUMvQixNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixZQUFZLEVBQUU7d0JBQ1osMEJBQTBCLEVBQUU7NEJBQzVCLFdBQVcsRUFBRTtnQ0FDWDtvQ0FDQSxRQUFRLEVBQUUsZ0JBQWdCO29DQUMxQixRQUFRLEVBQUUsT0FBTztvQ0FDakIsV0FBVyxFQUFFO3dDQUNYLFNBQVMsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsRUFBRTtxQ0FDeEU7aUNBQ0E7NkJBQ0Y7NEJBQ0QsU0FBUyxFQUFFLFlBQVk7eUJBQ3RCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNyQjtnQ0FDRSxVQUFVLEVBQUU7b0NBQ1osRUFBRTtvQ0FDRjt3Q0FDRSxNQUFNO3dDQUNOOzRDQUNBLEtBQUssRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELDJEQUEyRDtxQ0FDNUQ7aUNBQ0E7NkJBQ0Y7eUJBQ0E7cUJBQ0Y7aUJBQ0E7Z0JBQ0QsMENBQTBDLEVBQUU7b0JBQzVDLE1BQU0sRUFBRSxrQkFBa0I7b0JBQzFCLFlBQVksRUFBRTt3QkFDWixnQkFBZ0IsRUFBRTs0QkFDbEIsV0FBVyxFQUFFO2dDQUNYO29DQUNBLFFBQVEsRUFBRSxpQkFBaUI7b0NBQzNCLFFBQVEsRUFBRSxPQUFPO29DQUNqQixVQUFVLEVBQUU7d0NBQ1YsWUFBWSxFQUFFOzRDQUNkLGlDQUFpQzs0Q0FDakMsS0FBSzt5Q0FDSjtxQ0FDRjtpQ0FDQTs2QkFDRjs0QkFDRCxTQUFTLEVBQUUsWUFBWTt5QkFDdEI7d0JBQ0QsWUFBWSxFQUFFLDBDQUEwQzt3QkFDeEQsT0FBTyxFQUFFOzRCQUNUO2dDQUNFLEtBQUssRUFBRSw2QkFBNkI7NkJBQ3JDO3lCQUNBO3FCQUNGO2lCQUNBO2dCQUNELGlDQUFpQyxFQUFFO29CQUNuQyxNQUFNLEVBQUUsaUJBQWlCO29CQUN6QixZQUFZLEVBQUU7d0JBQ1osd0JBQXdCLEVBQUUsT0FBTztxQkFDbEM7aUJBQ0E7Z0JBQ0Qsa0JBQWtCLEVBQUU7b0JBQ3BCLE1BQU0sRUFBRSx1QkFBdUI7b0JBQy9CLFlBQVksRUFBRTt3QkFDWixNQUFNLEVBQUU7NEJBQ1IsU0FBUyxFQUFFLEtBQUs7eUJBQ2Y7d0JBQ0QsU0FBUyxFQUFFLGVBQWU7d0JBQzFCLE1BQU0sRUFBRTs0QkFDUixZQUFZLEVBQUU7Z0NBQ1osNkJBQTZCO2dDQUM3QixLQUFLOzZCQUNOO3lCQUNBO3dCQUNELFNBQVMsRUFBRSxZQUFZO3dCQUN2QixrQkFBa0IsRUFBRTs0QkFDcEIsV0FBVyxFQUFFO2dDQUNYLFlBQVksRUFBRTtvQ0FDZCxpQ0FBaUM7b0NBQ2pDLEtBQUs7aUNBQ0o7NkJBQ0Y7eUJBQ0E7cUJBQ0Y7b0JBQ0QsV0FBVyxFQUFFO3dCQUNYLDBDQUEwQzt3QkFDMUMsNkJBQTZCO3FCQUM5QjtpQkFDQTthQUNGO1NBQ0EsQ0FDSixDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdGQUFnRixDQUFDLElBQVU7UUFDekYsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDckMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztZQUNqQyxzQkFBc0IsRUFBRSxLQUFLO1NBQzlCLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQ25CO1lBQ0UsV0FBVyxFQUFFO2dCQUNiLDZCQUE2QixFQUFFO29CQUM3QixNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixZQUFZLEVBQUU7d0JBQ2QsMEJBQTBCLEVBQUU7NEJBQzFCLFdBQVcsRUFBRTtnQ0FDYjtvQ0FDRSxRQUFRLEVBQUUsZ0JBQWdCO29DQUMxQixRQUFRLEVBQUUsT0FBTztvQ0FDakIsV0FBVyxFQUFFO3dDQUNiLFNBQVMsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsRUFBRTtxQ0FDdEU7aUNBQ0Y7NkJBQ0E7NEJBQ0QsU0FBUyxFQUFFLFlBQVk7eUJBQ3hCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNuQjtnQ0FDQSxVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDQSxNQUFNO3dDQUNOOzRDQUNFLEtBQUssRUFBRSxnQkFBZ0I7eUNBQ3hCO3dDQUNELDJEQUEyRDtxQ0FDMUQ7aUNBQ0Y7NkJBQ0E7eUJBQ0Y7cUJBQ0E7aUJBQ0Y7Z0JBQ0Qsa0JBQWtCLEVBQUU7b0JBQ2xCLE1BQU0sRUFBRSx1QkFBdUI7b0JBQy9CLFlBQVksRUFBRTt3QkFDZCxNQUFNLEVBQUU7NEJBQ04sU0FBUyxFQUFFLEtBQUs7eUJBQ2pCO3dCQUNELFNBQVMsRUFBRSxlQUFlO3dCQUMxQixNQUFNLEVBQUU7NEJBQ04sWUFBWSxFQUFFO2dDQUNkLDZCQUE2QjtnQ0FDN0IsS0FBSzs2QkFDSjt5QkFDRjt3QkFDRCxTQUFTLEVBQUUsWUFBWTtxQkFDdEI7b0JBQ0QsV0FBVyxFQUFFO3dCQUNiLDZCQUE2QjtxQkFDNUI7aUJBQ0Y7YUFDQTtTQUNGLENBQ0YsQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0RUFBNEUsQ0FBQyxJQUFVO1FBQ3JGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7WUFDdEQsU0FBUyxFQUFFLGNBQWM7WUFDekIsa0JBQWtCLEVBQUUsT0FBTztTQUM1QixDQUFDLENBQUM7UUFFSCxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUNyQyxJQUFJLEVBQUUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNsQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTO1lBQ2pDLGVBQWUsRUFBRSxPQUFPO1NBQ3pCLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQ25CO1lBQ0UsV0FBVyxFQUFFO2dCQUNYLDZCQUE2QixFQUFFO29CQUMvQixNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixZQUFZLEVBQUU7d0JBQ1osMEJBQTBCLEVBQUU7NEJBQzVCLFdBQVcsRUFBRTtnQ0FDWDtvQ0FDQSxRQUFRLEVBQUUsZ0JBQWdCO29DQUMxQixRQUFRLEVBQUUsT0FBTztvQ0FDakIsV0FBVyxFQUFFO3dDQUNYLFNBQVMsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsRUFBRTtxQ0FDeEU7aUNBQ0E7NkJBQ0Y7NEJBQ0QsU0FBUyxFQUFFLFlBQVk7eUJBQ3RCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNyQjtnQ0FDRSxVQUFVLEVBQUU7b0NBQ1osRUFBRTtvQ0FDRjt3Q0FDRSxNQUFNO3dDQUNOOzRDQUNBLEtBQUssRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELDJEQUEyRDtxQ0FDNUQ7aUNBQ0E7NkJBQ0Y7eUJBQ0E7cUJBQ0Y7aUJBQ0E7Z0JBQ0QsMENBQTBDLEVBQUU7b0JBQzVDLE1BQU0sRUFBRSxrQkFBa0I7b0JBQzFCLFlBQVksRUFBRTt3QkFDWixnQkFBZ0IsRUFBRTs0QkFDbEIsV0FBVyxFQUFFO2dDQUNYO29DQUNBLFFBQVEsRUFBRSxpQkFBaUI7b0NBQzNCLFFBQVEsRUFBRSxPQUFPO29DQUNqQixVQUFVLEVBQUU7d0NBQ1YsWUFBWSxFQUFFOzRDQUNkLHlCQUF5Qjs0Q0FDekIsS0FBSzt5Q0FDSjtxQ0FDRjtpQ0FDQTs2QkFDRjs0QkFDRCxTQUFTLEVBQUUsWUFBWTt5QkFDdEI7d0JBQ0QsWUFBWSxFQUFFLDBDQUEwQzt3QkFDeEQsT0FBTyxFQUFFOzRCQUNUO2dDQUNFLEtBQUssRUFBRSw2QkFBNkI7NkJBQ3JDO3lCQUNBO3FCQUNGO2lCQUNBO2dCQUNELGtCQUFrQixFQUFFO29CQUNwQixNQUFNLEVBQUUsdUJBQXVCO29CQUMvQixZQUFZLEVBQUU7d0JBQ1osTUFBTSxFQUFFOzRCQUNSLFNBQVMsRUFBRSxLQUFLO3lCQUNmO3dCQUNELFNBQVMsRUFBRSxlQUFlO3dCQUMxQixNQUFNLEVBQUU7NEJBQ1IsWUFBWSxFQUFFO2dDQUNaLDZCQUE2QjtnQ0FDN0IsS0FBSzs2QkFDTjt5QkFDQTt3QkFDRCxTQUFTLEVBQUUsWUFBWTt3QkFDdkIsa0JBQWtCLEVBQUU7NEJBQ3BCLFdBQVcsRUFBRTtnQ0FDWCxZQUFZLEVBQUU7b0NBQ2QseUJBQXlCO29DQUN6QixLQUFLO2lDQUNKOzZCQUNGO3lCQUNBO3FCQUNGO29CQUNELFdBQVcsRUFBRTt3QkFDWCwwQ0FBMEM7d0JBQzFDLDZCQUE2QjtxQkFDOUI7aUJBQ0E7YUFDRjtTQUNBLEVBQ0gsbUJBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN2QixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbUhBQW1ILENBQUMsSUFBVTtRQUM1SCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixNQUFNLE9BQU8sR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO1lBQ3RELFNBQVMsRUFBRSxjQUFjO1lBQ3pCLGtCQUFrQixFQUFFLE9BQU87U0FDNUIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDckMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztZQUNqQyxzQkFBc0IsRUFBRSxJQUFJO1lBQzVCLGVBQWUsRUFBRSxPQUFPO1NBQ3pCLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQ25CO1lBQ0UsV0FBVyxFQUFFO2dCQUNiLDZCQUE2QixFQUFFO29CQUM3QixNQUFNLEVBQUUsZ0JBQWdCO29CQUN4QixZQUFZLEVBQUU7d0JBQ2QsMEJBQTBCLEVBQUU7NEJBQzFCLFdBQVcsRUFBRTtnQ0FDYjtvQ0FDRSxRQUFRLEVBQUUsZ0JBQWdCO29DQUMxQixRQUFRLEVBQUUsT0FBTztvQ0FDakIsV0FBVyxFQUFFO3dDQUNiLFNBQVMsRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsRUFBRTtxQ0FDdEU7aUNBQ0Y7NkJBQ0E7NEJBQ0QsU0FBUyxFQUFFLFlBQVk7eUJBQ3hCO3dCQUNELG1CQUFtQixFQUFFOzRCQUNuQjtnQ0FDQSxVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDQSxNQUFNO3dDQUNOOzRDQUNFLEtBQUssRUFBRSxnQkFBZ0I7eUNBQ3hCO3dDQUNELDJEQUEyRDtxQ0FDMUQ7aUNBQ0Y7NkJBQ0E7eUJBQ0Y7cUJBQ0E7aUJBQ0Y7Z0JBQ0QsMENBQTBDLEVBQUU7b0JBQzFDLE1BQU0sRUFBRSxrQkFBa0I7b0JBQzFCLFlBQVksRUFBRTt3QkFDZCxnQkFBZ0IsRUFBRTs0QkFDaEIsV0FBVyxFQUFFO2dDQUNiO29DQUNFLFFBQVEsRUFBRSxpQkFBaUI7b0NBQzNCLFFBQVEsRUFBRSxPQUFPO29DQUNqQixVQUFVLEVBQUU7d0NBQ1osWUFBWSxFQUFFOzRDQUNaLHlCQUF5Qjs0Q0FDekIsS0FBSzt5Q0FDTjtxQ0FDQTtpQ0FDRjs2QkFDQTs0QkFDRCxTQUFTLEVBQUUsWUFBWTt5QkFDeEI7d0JBQ0QsWUFBWSxFQUFFLDBDQUEwQzt3QkFDeEQsT0FBTyxFQUFFOzRCQUNQO2dDQUNBLEtBQUssRUFBRSw2QkFBNkI7NkJBQ25DO3lCQUNGO3FCQUNBO2lCQUNGO2dCQUNELGtCQUFrQixFQUFFO29CQUNsQixNQUFNLEVBQUUsdUJBQXVCO29CQUMvQixZQUFZLEVBQUU7d0JBQ2QsTUFBTSxFQUFFOzRCQUNOLFNBQVMsRUFBRSxLQUFLO3lCQUNqQjt3QkFDRCxTQUFTLEVBQUUsZUFBZTt3QkFDMUIsTUFBTSxFQUFFOzRCQUNOLFlBQVksRUFBRTtnQ0FDZCw2QkFBNkI7Z0NBQzdCLEtBQUs7NkJBQ0o7eUJBQ0Y7d0JBQ0QsU0FBUyxFQUFFLFlBQVk7d0JBQ3ZCLGtCQUFrQixFQUFFOzRCQUNsQixXQUFXLEVBQUU7Z0NBQ2IsWUFBWSxFQUFFO29DQUNaLHlCQUF5QjtvQ0FDekIsS0FBSztpQ0FDTjs2QkFDQTt5QkFDRjtxQkFDQTtvQkFDRCxXQUFXLEVBQUU7d0JBQ1gsMENBQTBDO3dCQUMxQyw2QkFBNkI7cUJBQzlCO2lCQUNGO2FBQ0E7U0FDRixFQUNELG1CQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDdkIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtIQUErSCxDQUFDLElBQVU7UUFDeEksTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtZQUN0RCxTQUFTLEVBQUUsY0FBYztZQUN6QixrQkFBa0IsRUFBRSxPQUFPO1NBQzVCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDekQsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztZQUNqQyxzQkFBc0IsRUFBRSxLQUFLO1lBQzdCLGVBQWUsRUFBRSxPQUFPO1NBQ3ZCLENBQUMsRUFBRSw0RUFBNEUsQ0FBQyxDQUFDO1FBRWxGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxzQ0FBc0MsQ0FBQyxJQUFVO1FBQy9DLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ3JDLElBQUksRUFBRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ2xDLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVM7WUFDakMsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsTUFBTTtTQUMvQixDQUFDLENBQUM7UUFFSCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7WUFDbEQsZ0JBQWdCLEVBQUU7Z0JBQ2hCLFdBQVcsRUFBRTtvQkFDYjt3QkFDRSxRQUFRLEVBQUU7NEJBQ1YsdUJBQXVCOzRCQUN2QiwwQkFBMEI7eUJBQ3pCO3dCQUNELFFBQVEsRUFBRSxPQUFPO3dCQUNqQixVQUFVLEVBQUUsR0FBRztxQkFDaEI7aUJBQ0E7Z0JBQ0QsU0FBUyxFQUFFLFlBQVk7YUFDeEI7WUFDRCxZQUFZLEVBQUUsMENBQTBDO1lBQ3hELE9BQU8sRUFBRTtnQkFDUDtvQkFDQSxLQUFLLEVBQUUsNkJBQTZCO2lCQUNuQzthQUNGO1NBQ0EsQ0FBQyxDQUFDLENBQUM7UUFFSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7WUFDdkQsWUFBWSxFQUFFO2dCQUNaLE1BQU0sRUFBRTtvQkFDUixTQUFTLEVBQUUsS0FBSztpQkFDZjtnQkFDRCxTQUFTLEVBQUUsZUFBZTtnQkFDMUIsTUFBTSxFQUFFO29CQUNSLFlBQVksRUFBRTt3QkFDWiw2QkFBNkI7d0JBQzdCLEtBQUs7cUJBQ047aUJBQ0E7Z0JBQ0QsU0FBUyxFQUFFLFlBQVk7Z0JBQ3ZCLGVBQWUsRUFBRTtvQkFDakIsTUFBTSxFQUFFLFFBQVE7aUJBQ2Y7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCwwQ0FBMEM7Z0JBQzFDLDZCQUE2QjthQUM5QjtTQUNBLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7UUFFckMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJDQUEyQyxDQUFDLElBQVU7UUFDcEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDckMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztZQUNqQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1NBQ3BDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtZQUNsRCxnQkFBZ0IsRUFBRTtnQkFDaEIsV0FBVyxFQUFFO29CQUNiO3dCQUNFLFFBQVEsRUFBRTs0QkFDVix1QkFBdUI7NEJBQ3ZCLDBCQUEwQjt5QkFDekI7d0JBQ0QsUUFBUSxFQUFFLE9BQU87d0JBQ2pCLFVBQVUsRUFBRSxHQUFHO3FCQUNoQjtpQkFDQTtnQkFDRCxTQUFTLEVBQUUsWUFBWTthQUN4QjtZQUNELFlBQVksRUFBRSwwQ0FBMEM7WUFDeEQsT0FBTyxFQUFFO2dCQUNQO29CQUNBLEtBQUssRUFBRSw2QkFBNkI7aUJBQ25DO2FBQ0Y7U0FDQSxDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtZQUN2RCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFO29CQUNSLFNBQVMsRUFBRSxLQUFLO2lCQUNmO2dCQUNELFNBQVMsRUFBRSxlQUFlO2dCQUMxQixNQUFNLEVBQUU7b0JBQ1IsWUFBWSxFQUFFO3dCQUNaLDZCQUE2Qjt3QkFDN0IsS0FBSztxQkFDTjtpQkFDQTtnQkFDRCxTQUFTLEVBQUUsWUFBWTtnQkFDdkIsZUFBZSxFQUFFO29CQUNqQixNQUFNLEVBQUUsYUFBYTtpQkFDcEI7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCwwQ0FBMEM7Z0JBQzFDLDZCQUE2QjthQUM5QjtTQUNBLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7UUFFckMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHdDQUF3QyxDQUFDLElBQVU7UUFDakQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDckMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztZQUNqQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRO1NBQ2pDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtZQUNyRCxnQkFBZ0IsRUFBRTtnQkFDaEIsV0FBVyxFQUFFO29CQUNiO3dCQUNFLFFBQVEsRUFBRTs0QkFDVix1QkFBdUI7NEJBQ3ZCLDBCQUEwQjt5QkFDekI7d0JBQ0QsUUFBUSxFQUFFLE9BQU87d0JBQ2pCLFVBQVUsRUFBRSxHQUFHO3FCQUNoQjtpQkFDQTtnQkFDRCxTQUFTLEVBQUUsWUFBWTthQUN4QjtZQUNELFlBQVksRUFBRSwwQ0FBMEM7WUFDeEQsT0FBTyxFQUFFO2dCQUNQO29CQUNBLEtBQUssRUFBRSw2QkFBNkI7aUJBQ25DO2FBQ0Y7U0FDQSxDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyx1QkFBdUIsRUFBRTtZQUN2RCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFO29CQUNSLFNBQVMsRUFBRSxLQUFLO2lCQUNmO2dCQUNELFNBQVMsRUFBRSxlQUFlO2dCQUMxQixNQUFNLEVBQUU7b0JBQ1IsWUFBWSxFQUFFO3dCQUNaLDZCQUE2Qjt3QkFDN0IsS0FBSztxQkFDTjtpQkFDQTtnQkFDRCxTQUFTLEVBQUUsWUFBWTthQUN4QjtZQUNELFdBQVcsRUFBRTtnQkFDWCw2QkFBNkI7YUFDOUI7U0FDQSxFQUFFLHFCQUFZLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1FBRXJDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxQ0FBcUMsQ0FBQyxJQUFVO1FBQzlDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDO1NBQzVDLENBQUMsQ0FBQztRQUNILE1BQU0sRUFBRSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ2hELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUM7WUFDL0IsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztTQUNsQyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVyQixPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO1lBQ2hELGNBQWMsRUFBRTtnQkFDZCxPQUFPLEVBQUUsWUFBWTtnQkFDckIsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRSx1QkFBdUI7d0JBQy9CLE1BQU0sRUFBRSxPQUFPO3dCQUNmLFFBQVEsRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLGtCQUFrQixFQUFFLEtBQUssQ0FBRSxFQUFFO3FCQUMxRDtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMkNBQTJDLENBQUMsSUFBVTtRQUNwRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxFQUFFLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDaEQsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQztZQUMvQixPQUFPLEVBQUUsZUFBZTtZQUN4QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTO1NBQ2xDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxFQUFFO1lBQ3BELFVBQVUsRUFBRSxFQUFFLFlBQVksRUFBRSxFQUFFLEdBQUcsRUFBRSxrQkFBa0IsRUFBRSxFQUFDO1lBQ3hELFNBQVMsRUFBRSxZQUFZO1lBQ3ZCLFVBQVUsRUFBRSxRQUFRO1lBQ3BCLFNBQVMsRUFBRSxHQUFHO1lBQ2QsU0FBUyxFQUFFLEtBQUs7U0FDakIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJCQUEyQixDQUFDLElBQVU7UUFDcEMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sRUFBRSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ2hELElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUM7WUFDL0IsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztTQUNsQyxDQUFDLENBQUM7UUFFSCxJQUFJLFVBQVUsQ0FBQztRQUVmLE1BQU0sZUFBZTtZQUNaLElBQUksQ0FBQyxNQUF3QjtnQkFDbEMsVUFBVSxHQUFHLE1BQU0sQ0FBQztZQUN0QixDQUFDO1NBQ0Y7UUFFRCxPQUFPO1FBQ1AsRUFBRSxDQUFDLGNBQWMsQ0FBQyxJQUFJLGVBQWUsRUFBRSxDQUFDLENBQUM7UUFFekMsT0FBTztRQUNQLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQzFCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCxzQ0FBc0MsQ0FBQyxJQUFVO1FBQy9DLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ3JDLElBQUksRUFBRSxJQUFJLE1BQU0sQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ2xDLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU07U0FDL0IsQ0FBQyxDQUFDO1FBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLFNBQVMsRUFDL0IsRUFBRSwyQkFBMkIsRUFDekIsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCO29CQUN4QixVQUFVLEVBQ1YsRUFBRSx3QkFBd0IsRUFDeEIsRUFBRSxTQUFTLEVBQ1gsQ0FBRSxFQUFFLE1BQU0sRUFBRSxnQkFBZ0I7b0NBQzFCLE1BQU0sRUFBRSxPQUFPO29DQUNmLFNBQVMsRUFBRSxFQUFFLE9BQU8sRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBRTs0QkFDeEYsT0FBTyxFQUFFLFlBQVksRUFBRTt3QkFDM0IsaUJBQWlCO3dCQUNqQixtRUFBbUU7d0JBQ2pFLDJDQUEyQzt3QkFDM0MsQ0FBQyxFQUFDLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFDLEdBQUcsRUFBRSxnQkFBZ0IsRUFBQyxFQUFFLDJEQUEyRCxDQUFDLENBQUMsRUFBQyxDQUFDO3FCQUNySCxFQUFDO2dCQUNKLGdCQUFnQixFQUNkLEVBQUUsSUFBSSxFQUFFLHVCQUF1QjtvQkFDL0IsVUFBVSxFQUNWLEVBQUUsSUFBSSxFQUFFLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRTt3QkFDMUIsT0FBTyxFQUFFLGVBQWU7d0JBQ3hCLElBQUksRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFFLDZCQUE2QixFQUFFLEtBQUssQ0FBRSxFQUFFO3dCQUNoRSxPQUFPLEVBQUUsU0FBUyxFQUFFO29CQUNwQixTQUFTLEVBQUUsQ0FBRSw2QkFBNkIsQ0FBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDMUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZCQUE2QixDQUFDLElBQVU7UUFDdEMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDcEQsTUFBTSxLQUFLLEdBQUcsTUFBTSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtZQUMzRCxlQUFlLEVBQUUsYUFBYTtZQUM5QixrQkFBa0IsRUFBRSxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDO1NBQy9DLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQzNDLE1BQU0sRUFBRSxDQUFDLEtBQUssQ0FBQztZQUNmLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVM7WUFDakMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLG9EQUFvRCxDQUFDO1lBQzlFLE9BQU8sRUFBRSxZQUFZO1NBQ3RCLENBQUMsRUFDRixxQ0FBcUMsQ0FBQyxDQUFDO1FBRW5ELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwwQkFBMEIsQ0FBQyxJQUFVO1FBQ25DLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ3BELE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1lBQzlFLGVBQWUsRUFBRSxhQUFhO1lBQzlCLGtCQUFrQixFQUFFLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUM7U0FDL0MsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUMzQyxNQUFNO1lBQ04sT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsU0FBUztZQUNqQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsb0RBQW9ELENBQUM7WUFDOUUsT0FBTyxFQUFFLFlBQVk7U0FDdEIsQ0FBQyxFQUNGLHNCQUFzQixDQUFDLENBQUM7UUFFcEMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtDQUErQyxDQUFDLElBQVU7UUFDeEQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxFQUFDLENBQUMsQ0FBQztRQUVwRixPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtnQkFDckMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7Z0JBQ2xDLE9BQU8sRUFBRSxlQUFlO2dCQUN4QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNO2dCQUM5QixXQUFXLEVBQUU7b0JBQ1gsSUFBSSxFQUFFLFVBQVU7aUJBQ2pCO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxFQUFFLHlDQUF5QyxDQUFDLENBQUM7UUFFOUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFEQUFxRCxDQUFDLElBQVU7UUFDOUQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUNyQyxJQUFJLEVBQUUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNsQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNO1lBQzlCLFdBQVcsRUFBRTtnQkFDWCxJQUFJLEVBQUUsVUFBVTthQUNqQjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7WUFDckQsV0FBVyxFQUFFO2dCQUNYLFNBQVMsRUFBRTtvQkFDVCxJQUFJLEVBQUUsVUFBVTtpQkFDakI7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBRWQsQ0FBQztJQUVELHdDQUF3QyxDQUFDLElBQVU7UUFDakQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDckMsSUFBSSxFQUFFLElBQUksTUFBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDbEMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsTUFBTTtZQUM5Qiw0QkFBNEIsRUFBRSxFQUFFO1NBQ2pDLENBQUMsQ0FBQztRQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsRUFBRSxTQUFTLEVBQy9CLEVBQUUsMkJBQTJCLEVBQ3pCLEVBQUUsSUFBSSxFQUFFLGdCQUFnQjtvQkFDeEIsVUFBVSxFQUNWLEVBQUUsd0JBQXdCLEVBQ3hCLEVBQUUsU0FBUyxFQUNYLENBQUUsRUFBRSxNQUFNLEVBQUUsZ0JBQWdCO29DQUMxQixNQUFNLEVBQUUsT0FBTztvQ0FDZixTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUU7NEJBQ3hGLE9BQU8sRUFBRSxZQUFZLEVBQUU7d0JBQzNCLGlCQUFpQjt3QkFDakIsbUVBQW1FO3dCQUNqRSwyQ0FBMkM7d0JBQzNDLENBQUMsRUFBQyxVQUFVLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBQyxHQUFHLEVBQUUsZ0JBQWdCLEVBQUMsRUFBRSwyREFBMkQsQ0FBQyxDQUFDLEVBQUMsQ0FBQztxQkFDckgsRUFBQztnQkFDSixnQkFBZ0IsRUFDZCxFQUFFLElBQUksRUFBRSx1QkFBdUI7b0JBQy9CLFVBQVUsRUFDVixFQUFFLElBQUksRUFBRSxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUU7d0JBQzFCLE9BQU8sRUFBRSxlQUFlO3dCQUN4Qiw0QkFBNEIsRUFBRSxFQUFFO3dCQUNoQyxJQUFJLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBRSw2QkFBNkIsRUFBRSxLQUFLLENBQUUsRUFBRTt3QkFDaEUsT0FBTyxFQUFFLFFBQVEsRUFBRTtvQkFDbkIsU0FBUyxFQUFFLENBQUUsNkJBQTZCLENBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQzFELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxREFBcUQsQ0FBQyxJQUFVO1FBQzlELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixJQUFJLFNBQVMsR0FBRyxDQUFDLENBQUM7UUFFbEIsTUFBTSxXQUFXO1lBQ1IsSUFBSSxDQUFDLENBQXNCO2dCQUNoQyxTQUFTLEVBQUUsQ0FBQztZQUNkLENBQUM7U0FDRjtRQUVELE9BQU87UUFDUCxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtZQUMvQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDO1lBQ2hDLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVM7WUFDakMsT0FBTyxFQUFFLFdBQVc7WUFDcEIsTUFBTSxFQUFFO2dCQUNOLElBQUksV0FBVyxFQUFFO2dCQUNqQixJQUFJLFdBQVcsRUFBRTthQUNsQjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUM3QixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMkNBQTJDLENBQUMsSUFBVTtRQUNwRCxNQUFNLEVBQUUsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztRQUVuQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDaEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFekMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHVCQUF1QixDQUFDLElBQVU7UUFDaEMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUNyQyxJQUFJLEVBQUUsSUFBSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztZQUNsQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNO1lBQzlCLGdCQUFnQixFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUTtTQUM5QyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHNCQUFzQixFQUFFO1lBQ3BELGNBQWMsRUFBRTtnQkFDZCxVQUFVLEVBQUU7b0JBQ1YsRUFBRTtvQkFDRjt3QkFDRSxjQUFjO3dCQUNkOzRCQUNFLEdBQUcsRUFBRSxrQkFBa0I7eUJBQ3hCO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxpQkFBaUIsRUFBRSxFQUFFO1NBQ3RCLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2IsQ0FBQztDQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBjb3VudFJlc291cmNlcywgZXhwZWN0LCBoYXZlUmVzb3VyY2UsIE1hdGNoU3R5bGUsIFJlc291cmNlUGFydCB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgZXZlbnRzID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWV2ZW50cycpO1xuaW1wb3J0IGlhbSA9IHJlcXVpcmUoJ0Bhd3MtY2RrL2F3cy1pYW0nKTtcbmltcG9ydCBsb2dzID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWxvZ3MnKTtcbmltcG9ydCBzcXMgPSByZXF1aXJlKCdAYXdzLWNkay9hd3Mtc3FzJyk7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY2RrJyk7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHBhdGggPSByZXF1aXJlKCdwYXRoJyk7XG5pbXBvcnQgbGFtYmRhID0gcmVxdWlyZSgnLi4vbGliJyk7XG5cbi8vIHRzbGludDpkaXNhYmxlOm9iamVjdC1saXRlcmFsLWtleS1xdW90ZXNcblxuZXhwb3J0ID0ge1xuICAnZGVmYXVsdCBmdW5jdGlvbicodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ015TGFtYmRhJywge1xuICAgICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdmb28nKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5vZGVKUzYxMCxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7IFJlc291cmNlczpcbiAgICAgIHsgTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2OlxuICAgICAgICAgeyBUeXBlOiAnQVdTOjpJQU06OlJvbGUnLFxuICAgICAgICAgUHJvcGVydGllczpcbiAgICAgICAgICB7IEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDpcbiAgICAgICAgICAgeyBTdGF0ZW1lbnQ6XG4gICAgICAgICAgICBbIHsgQWN0aW9uOiAnc3RzOkFzc3VtZVJvbGUnLFxuICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgIFByaW5jaXBhbDogeyBTZXJ2aWNlOiB7IFwiRm46OkpvaW5cIjogW1wiXCIsIFsnbGFtYmRhLicsIHsgUmVmOiBcIkFXUzo6VVJMU3VmZml4XCIgfV1dIH0gfSB9IF0sXG4gICAgICAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnIH0sXG4gICAgICAgICAgTWFuYWdlZFBvbGljeUFybnM6XG4gICAgICAgICAgLy8gYXJuOmF3czppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZVxuICAgICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgICAgIFt7J0ZuOjpKb2luJzogWycnLCBbJ2FybjonLCB7UmVmOiAnQVdTOjpQYXJ0aXRpb24nfSwgJzppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZSddXX1dLFxuICAgICAgICAgIH19LFxuICAgICAgICBNeUxhbWJkYUNDRTgwMkZCOlxuICAgICAgICAgeyBUeXBlOiAnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJyxcbiAgICAgICAgIFByb3BlcnRpZXM6XG4gICAgICAgICAgeyBDb2RlOiB7IFppcEZpbGU6ICdmb28nIH0sXG4gICAgICAgICAgSGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgICAgIFJvbGU6IHsgJ0ZuOjpHZXRBdHQnOiBbICdNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjYnLCAnQXJuJyBdIH0sXG4gICAgICAgICAgUnVudGltZTogJ25vZGVqczYuMTAnIH0sXG4gICAgICAgICBEZXBlbmRzT246IFsgJ015TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNicgXSB9IH0gfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2FkZHMgcG9saWN5IHBlcm1pc3Npb25zJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ015TGFtYmRhJywge1xuICAgICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdmb28nKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5vZGVKUzYxMCxcbiAgICAgIGluaXRpYWxQb2xpY3k6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCgpLmFkZEFjdGlvbihcIipcIikuYWRkQWxsUmVzb3VyY2VzKCldXG4gICAgfSk7XG4gICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHsgUmVzb3VyY2VzOlxuICAgICAgeyBNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjY6XG4gICAgICAgICB7IFR5cGU6ICdBV1M6OklBTTo6Um9sZScsXG4gICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgIHsgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OlxuICAgICAgICAgICB7IFN0YXRlbWVudDpcbiAgICAgICAgICAgIFsgeyBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6IHsgXCJGbjo6Sm9pblwiOiBbXCJcIiwgWydsYW1iZGEuJywgeyBSZWY6IFwiQVdTOjpVUkxTdWZmaXhcIiB9XV0gfSB9IH0gXSxcbiAgICAgICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycgfSxcbiAgICAgICAgICBNYW5hZ2VkUG9saWN5QXJuczpcbiAgICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgICAgW3snRm46OkpvaW4nOiBbJycsIFsnYXJuOicsIHtSZWY6ICdBV1M6OlBhcnRpdGlvbid9LCAnOmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJ11dfV0sXG4gICAgICAgIH19LFxuICAgICAgICBNeUxhbWJkYVNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTVCQkM2RjY4OiB7XG4gICAgICAgICAgVHlwZTogXCJBV1M6OklBTTo6UG9saWN5XCIsXG4gICAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgIEFjdGlvbjogXCIqXCIsXG4gICAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBSZXNvdXJjZTogXCIqXCJcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFZlcnNpb246IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgUG9saWN5TmFtZTogXCJNeUxhbWJkYVNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTVCQkM2RjY4XCIsXG4gICAgICAgICAgICBSb2xlczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZWY6IFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIE15TGFtYmRhQ0NFODAyRkI6XG4gICAgICAgICB7IFR5cGU6ICdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLFxuICAgICAgICAgUHJvcGVydGllczpcbiAgICAgICAgICB7IENvZGU6IHsgWmlwRmlsZTogJ2ZvbycgfSxcbiAgICAgICAgICBIYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgICAgUm9sZTogeyAnRm46OkdldEF0dCc6IFsgJ015TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNicsICdBcm4nIF0gfSxcbiAgICAgICAgICBSdW50aW1lOiAnbm9kZWpzNi4xMCcgfSxcbiAgICAgICAgIERlcGVuZHNPbjogWyAnTXlMYW1iZGFTZXJ2aWNlUm9sZURlZmF1bHRQb2xpY3k1QkJDNkY2OCcsICdNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjYnIF0gfSB9IH0gKTtcbiAgICB0ZXN0LmRvbmUoKTtcblxuICB9LFxuXG4gICdmYWlscyBpZiBpbmxpbmUgY29kZSBpcyB1c2VkIGZvciBhbiBpbnZhbGlkIHJ1bnRpbWUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgICBjb2RlOiBuZXcgbGFtYmRhLklubGluZUNvZGUoJ2ZvbycpLFxuICAgICAgaGFuZGxlcjogJ2JhcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Eb3ROZXRDb3JlMlxuICAgIH0pKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYWRkVG9SZXNvdXJjZVBvbGljeSc6IHtcbiAgICAnY2FuIGJlIHVzZWQgdG8gYWRkIHBlcm1pc3Npb25zIHRvIHRoZSBMYW1iZGEgZnVuY3Rpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgZm4gPSBuZXdUZXN0TGFtYmRhKHN0YWNrKTtcblxuICAgICAgZm4uYWRkUGVybWlzc2lvbignUzNQZXJtaXNzaW9uJywge1xuICAgICAgICBhY3Rpb246ICdsYW1iZGE6KicsXG4gICAgICAgIHByaW5jaXBhbDogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzMy5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIHNvdXJjZUFjY291bnQ6IHN0YWNrLmFjY291bnRJZCxcbiAgICAgICAgc291cmNlQXJuOiAnYXJuOmF3czpzMzo6Om15X2J1Y2tldCdcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goe1xuICAgICAgICBcIlJlc291cmNlc1wiOiB7XG4gICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OklBTTo6Um9sZVwiLFxuICAgICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgICAgXCJBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkFjdGlvblwiOiBcInN0czpBc3N1bWVSb2xlXCIsXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJQcmluY2lwYWxcIjoge1xuICAgICAgICAgICAgICBcIlNlcnZpY2VcIjogeyBcIkZuOjpKb2luXCI6IFtcIlwiLCBbXCJsYW1iZGEuXCIsIHsgUmVmOiBcIkFXUzo6VVJMU3VmZml4XCIgfV1dIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJNYW5hZ2VkUG9saWN5QXJuc1wiOlxuICAgICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgICBbeydGbjo6Sm9pbic6IFsnJywgWydhcm46Jywge1JlZjogJ0FXUzo6UGFydGl0aW9uJ30sICc6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUnXV19XSxcbiAgICAgICAgICB9XG4gICAgICAgIH0sXG4gICAgICAgIFwiTXlMYW1iZGFDQ0U4MDJGQlwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICBcIkNvZGVcIjoge1xuICAgICAgICAgICAgXCJaaXBGaWxlXCI6IFwiZm9vXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiSGFuZGxlclwiOiBcImJhclwiLFxuICAgICAgICAgIFwiUm9sZVwiOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiLFxuICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIlJ1bnRpbWVcIjogXCJweXRob24yLjdcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJEZXBlbmRzT25cIjogW1xuICAgICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIFwiTXlMYW1iZGFTM1Blcm1pc3Npb245OUQwRUEwOFwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpMYW1iZGE6OlBlcm1pc3Npb25cIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiQWN0aW9uXCI6IFwibGFtYmRhOipcIixcbiAgICAgICAgICBcIkZ1bmN0aW9uTmFtZVwiOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcIk15TGFtYmRhQ0NFODAyRkJcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJQcmluY2lwYWxcIjogXCJzMy5hbWF6b25hd3MuY29tXCIsXG4gICAgICAgICAgXCJTb3VyY2VBY2NvdW50XCI6IHtcbiAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpBY2NvdW50SWRcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJTb3VyY2VBcm5cIjogXCJhcm46YXdzOnMzOjo6bXlfYnVja2V0XCJcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnZmFpbHMgaWYgdGhlIHByaW5jaXBhbCBpcyBub3QgYSBzZXJ2aWNlIG9yIGFjY291bnQgcHJpbmNpcGFscycodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBmbiA9IG5ld1Rlc3RMYW1iZGEoc3RhY2spO1xuXG4gICAgICB0ZXN0LnRocm93cygoKSA9PiBmbi5hZGRQZXJtaXNzaW9uKCdGMScsIHsgcHJpbmNpcGFsOiBuZXcgaWFtLkFyblByaW5jaXBhbCgnanVzdDphcm4nKSB9KSxcbiAgICAgICAgL0ludmFsaWQgcHJpbmNpcGFsIHR5cGUgZm9yIExhbWJkYSBwZXJtaXNzaW9uIHN0YXRlbWVudC8pO1xuXG4gICAgICBmbi5hZGRQZXJtaXNzaW9uKCdTMScsIHsgcHJpbmNpcGFsOiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ215LXNlcnZpY2UnKSB9KTtcbiAgICAgIGZuLmFkZFBlcm1pc3Npb24oJ1MyJywgeyBwcmluY2lwYWw6IG5ldyBpYW0uQWNjb3VudFByaW5jaXBhbCgnYWNjb3VudCcpIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ0JZT1JvbGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdTb21lUm9sZScsIHtcbiAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICB9KTtcbiAgICAgIHJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoKS5hZGRBY3Rpb24oJ2NvbmZpcm06aXRzdGhlc2FtZScpKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgZm4gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnRnVuY3Rpb24nLCB7XG4gICAgICAgIGNvZGU6IG5ldyBsYW1iZGEuSW5saW5lQ29kZSgndGVzdCcpLFxuICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QeXRob24zNixcbiAgICAgICAgaGFuZGxlcjogJ2luZGV4LnRlc3QnLFxuICAgICAgICByb2xlLFxuICAgICAgICBpbml0aWFsUG9saWN5OiBbXG4gICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoKS5hZGRBY3Rpb24oJ2lubGluZTppbmxpbmUnKVxuICAgICAgICBdXG4gICAgICB9KTtcblxuICAgICAgZm4uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KCkuYWRkQWN0aW9uKCdleHBsaWNpdDpleHBsaWNpdCcpKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIixcbiAgICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAgICB7IFwiQWN0aW9uXCI6IFwiY29uZmlybTppdHN0aGVzYW1lXCIsIFwiRWZmZWN0XCI6IFwiQWxsb3dcIiB9LFxuICAgICAgICAgICAgeyBcIkFjdGlvblwiOiBcImlubGluZTppbmxpbmVcIiwgXCJFZmZlY3RcIjogXCJBbGxvd1wiIH0sXG4gICAgICAgICAgICB7IFwiQWN0aW9uXCI6IFwiZXhwbGljaXQ6ZXhwbGljaXRcIiwgXCJFZmZlY3RcIjogXCJBbGxvd1wiIH1cbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH0sXG5cbiAgJ2ltcG9ydC9leHBvcnQnOiB7XG4gICAgJ2xhbWJkYS5leHBvcnQoKSBjYW4gYmUgdXNlZCB0byBhZGQgT3V0cHV0cyB0byB0aGUgc3RhY2sgYW5kIHJldHVybnMgYW4gSUZ1bmN0aW9uIG9iamVjdCcodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGZuID0gbmV3VGVzdExhbWJkYShzdGFjazEpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBwcm9wcyA9IGZuLmV4cG9ydCgpO1xuICAgICAgY29uc3QgaW1wb3J0ZWQgPSBsYW1iZGEuRnVuY3Rpb24uaW1wb3J0KHN0YWNrMiwgJ0ltcG9ydGVkJywgcHJvcHMpO1xuXG4gICAgICAvLyBDYW4gY2FsbCBhZGRQZXJtaXNzaW9uKCkgYnV0IGl0IHdvbid0IGRvIGFueXRoaW5nXG4gICAgICBpbXBvcnRlZC5hZGRQZXJtaXNzaW9uKCdIZWxsbycsIHtcbiAgICAgICAgcHJpbmNpcGFsOiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2hhcnJ5JylcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gICdMYW1iZGEgY2FuIHNlcnZlIGFzIEV2ZW50UnVsZSB0YXJnZXQsIHBlcm1pc3Npb24gZ2V0cyBhZGRlZCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgZm4gPSBuZXdUZXN0TGFtYmRhKHN0YWNrKTtcbiAgICBjb25zdCBydWxlMSA9IG5ldyBldmVudHMuRXZlbnRSdWxlKHN0YWNrLCAnUnVsZScsIHsgc2NoZWR1bGVFeHByZXNzaW9uOiAncmF0ZSgxIG1pbnV0ZSknIH0pO1xuICAgIGNvbnN0IHJ1bGUyID0gbmV3IGV2ZW50cy5FdmVudFJ1bGUoc3RhY2ssICdSdWxlMicsIHsgc2NoZWR1bGVFeHByZXNzaW9uOiAncmF0ZSg1IG1pbnV0ZXMpJyB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBydWxlMS5hZGRUYXJnZXQoZm4pO1xuICAgIHJ1bGUyLmFkZFRhcmdldChmbik7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3QgbGFtYmRhSWQgPSBcIk15TGFtYmRhQ0NFODAyRkJcIjtcblxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6UGVybWlzc2lvbicsIHtcbiAgICAgIFwiQWN0aW9uXCI6IFwibGFtYmRhOkludm9rZUZ1bmN0aW9uXCIsXG4gICAgICBcIkZ1bmN0aW9uTmFtZVwiOiB7XG4gICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgbGFtYmRhSWQsXG4gICAgICAgICAgXCJBcm5cIlxuICAgICAgICBdXG4gICAgICB9LFxuICAgICAgXCJQcmluY2lwYWxcIjogXCJldmVudHMuYW1hem9uYXdzLmNvbVwiLFxuICAgICAgXCJTb3VyY2VBcm5cIjogeyBcIkZuOjpHZXRBdHRcIjogWyBcIlJ1bGU0Qzk5NUI3RlwiLCBcIkFyblwiIF0gfVxuICAgIH0pKTtcblxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6UGVybWlzc2lvbicsIHtcbiAgICAgIFwiQWN0aW9uXCI6IFwibGFtYmRhOkludm9rZUZ1bmN0aW9uXCIsXG4gICAgICBcIkZ1bmN0aW9uTmFtZVwiOiB7XG4gICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgbGFtYmRhSWQsXG4gICAgICAgICAgXCJBcm5cIlxuICAgICAgICBdXG4gICAgICB9LFxuICAgICAgXCJQcmluY2lwYWxcIjogXCJldmVudHMuYW1hem9uYXdzLmNvbVwiLFxuICAgICAgXCJTb3VyY2VBcm5cIjogeyBcIkZuOjpHZXRBdHRcIjogWyBcIlJ1bGUyNzA3MzIyNDRcIiwgXCJBcm5cIiBdIH1cbiAgICB9KSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGNvdW50UmVzb3VyY2VzKCdBV1M6OkV2ZW50czo6UnVsZScsIDIpKTtcbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFdmVudHM6OlJ1bGUnLCB7XG4gICAgICBcIlRhcmdldHNcIjogW1xuICAgICAgICB7XG4gICAgICAgIFwiQXJuXCI6IHsgXCJGbjo6R2V0QXR0XCI6IFsgbGFtYmRhSWQsIFwiQXJuXCIgXSB9LFxuICAgICAgICBcIklkXCI6IFwiTXlMYW1iZGFcIlxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0xhbWJkYSBjb2RlIGNhbiBiZSByZWFkIGZyb20gYSBsb2NhbCBkaXJlY3RvcnkgdmlhIGFuIGFzc2V0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5kaXJlY3RvcnkocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LWxhbWJkYS1oYW5kbGVyJykpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUHl0aG9uMzZcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgICAgXCJDb2RlXCI6IHtcbiAgICAgICAgXCJTM0J1Y2tldFwiOiB7XG4gICAgICAgIFwiUmVmXCI6IFwiTXlMYW1iZGFDb2RlUzNCdWNrZXRDODJBNTg3MFwiXG4gICAgICAgIH0sXG4gICAgICAgIFwiUzNLZXlcIjogeyBcIkZuOjpKb2luXCI6IFsgXCJcIiwgW1xuICAgICAgICB7XCJGbjo6U2VsZWN0XCI6IFswLCB7XCJGbjo6U3BsaXRcIjogW1wifHxcIiwge1wiUmVmXCI6IFwiTXlMYW1iZGFDb2RlUzNWZXJzaW9uS2V5NDc3NjI1MzdcIn1dfV19LFxuICAgICAgICB7XCJGbjo6U2VsZWN0XCI6IFsxLCB7XCJGbjo6U3BsaXRcIjogW1wifHxcIiwge1wiUmVmXCI6IFwiTXlMYW1iZGFDb2RlUzNWZXJzaW9uS2V5NDc3NjI1MzdcIn1dfV19LFxuICAgICAgICBdXX1cbiAgICAgIH0sXG4gICAgICBcIkhhbmRsZXJcIjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICBcIlJvbGVcIjoge1xuICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiLFxuICAgICAgICBcIkFyblwiXG4gICAgICAgIF1cbiAgICAgIH0sXG4gICAgICBcIlJ1bnRpbWVcIjogXCJweXRob24zLjZcIlxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZWZhdWx0IGZ1bmN0aW9uIHdpdGggU1FTIERMUSB3aGVuIGNsaWVudCBzZXRzIGRlYWRMZXR0ZXJRdWV1ZUVuYWJsZWQgdG8gdHJ1ZSBhbmQgZnVuY3Rpb25OYW1lIGRlZmluZWQgYnkgY2xpZW50Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgICBjb2RlOiBuZXcgbGFtYmRhLklubGluZUNvZGUoJ2ZvbycpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTNjEwLFxuICAgICAgZnVuY3Rpb25OYW1lOiAnT25lRnVuY3Rpb25Ub1J1bGVUaGVtQWxsJyxcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZUVuYWJsZWQ6IHRydWVcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaChcbiAgICAgIHtcbiAgICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OklBTTo6Um9sZVwiLFxuICAgICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgICAgICBcIkFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiUHJpbmNpcGFsXCI6IHtcbiAgICAgICAgICAgICAgICBcIlNlcnZpY2VcIjogeyBcIkZuOjpKb2luXCI6IFtcIlwiLCBbXCJsYW1iZGEuXCIsIHsgUmVmOiBcIkFXUzo6VVJMU3VmZml4XCIgfV1dIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgXCJWZXJzaW9uXCI6IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgXCJNYW5hZ2VkUG9saWN5QXJuc1wiOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFwiUmVmXCI6IFwiQVdTOjpQYXJ0aXRpb25cIlxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXCI6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGVcIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6SUFNOjpQb2xpY3lcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICAgICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogXCJzcXM6U2VuZE1lc3NhZ2VcIixcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IHtcbiAgICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICAgIFwiTXlMYW1iZGFEZWFkTGV0dGVyUXVldWUzOTlFRUEyRFwiLFxuICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFwiUG9saWN5TmFtZVwiOiBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIixcbiAgICAgICAgICAgIFwiUm9sZXNcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIlJlZlwiOiBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJNeUxhbWJkYURlYWRMZXR0ZXJRdWV1ZTM5OUVFQTJEXCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OlNRUzo6UXVldWVcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgICAgXCJNZXNzYWdlUmV0ZW50aW9uUGVyaW9kXCI6IDEyMDk2MDBcbiAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIk15TGFtYmRhQ0NFODAyRkJcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6TGFtYmRhOjpGdW5jdGlvblwiLFxuICAgICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgICAgICBcIkNvZGVcIjoge1xuICAgICAgICAgICAgXCJaaXBGaWxlXCI6IFwiZm9vXCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIkhhbmRsZXJcIjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICAgICAgICBcIlJvbGVcIjoge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjZcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFwiUnVudGltZVwiOiBcIm5vZGVqczYuMTBcIixcbiAgICAgICAgICAgIFwiRGVhZExldHRlckNvbmZpZ1wiOiB7XG4gICAgICAgICAgICBcIlRhcmdldEFyblwiOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiTXlMYW1iZGFEZWFkTGV0dGVyUXVldWUzOTlFRUEyRFwiLFxuICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIkZ1bmN0aW9uTmFtZVwiOiBcIk9uZUZ1bmN0aW9uVG9SdWxlVGhlbUFsbFwiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIkRlcGVuZHNPblwiOiBbXG4gICAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIixcbiAgICAgICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIH1cbiAgICApO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZWZhdWx0IGZ1bmN0aW9uIHdpdGggU1FTIERMUSB3aGVuIGNsaWVudCBzZXRzIGRlYWRMZXR0ZXJRdWV1ZUVuYWJsZWQgdG8gdHJ1ZSBhbmQgZnVuY3Rpb25OYW1lIG5vdCBkZWZpbmVkIGJ5IGNsaWVudCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ015TGFtYmRhJywge1xuICAgICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdmb28nKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5vZGVKUzYxMCxcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZUVuYWJsZWQ6IHRydWUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goXG4gICAgICB7XG4gICAgICAgIFwiUmVzb3VyY2VzXCI6IHtcbiAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpJQU06OlJvbGVcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgICAgXCJBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFwic3RzOkFzc3VtZVJvbGVcIixcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlByaW5jaXBhbFwiOiB7XG4gICAgICAgICAgICAgICAgXCJTZXJ2aWNlXCI6IHsgXCJGbjo6Sm9pblwiOiBbXCJcIiwgW1wibGFtYmRhLlwiLCB7IFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiIH1dXSB9XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFwiTWFuYWdlZFBvbGljeUFybnNcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgIFwiYXJuOlwiLFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIlJlZlwiOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIFwiOmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTVCQkM2RjY4XCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OklBTTo6UG9saWN5XCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFwic3FzOlNlbmRNZXNzYWdlXCIsXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiB7XG4gICAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICBcIk15TGFtYmRhRGVhZExldHRlclF1ZXVlMzk5RUVBMkRcIixcbiAgICAgICAgICAgICAgICBcIkFyblwiXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIlBvbGljeU5hbWVcIjogXCJNeUxhbWJkYVNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTVCQkM2RjY4XCIsXG4gICAgICAgICAgICBcIlJvbGVzXCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJSZWZcIjogXCJNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjZcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiTXlMYW1iZGFEZWFkTGV0dGVyUXVldWUzOTlFRUEyRFwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpTUVM6OlF1ZXVlXCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICAgIFwiTWVzc2FnZVJldGVudGlvblBlcmlvZFwiOiAxMjA5NjAwXG4gICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJNeUxhbWJkYUNDRTgwMkZCXCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OkxhbWJkYTo6RnVuY3Rpb25cIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgICAgXCJDb2RlXCI6IHtcbiAgICAgICAgICAgIFwiWmlwRmlsZVwiOiBcImZvb1wiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgXCJIYW5kbGVyXCI6IFwiaW5kZXguaGFuZGxlclwiLFxuICAgICAgICAgICAgXCJSb2xlXCI6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCIsXG4gICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIlJ1bnRpbWVcIjogXCJub2RlanM2LjEwXCIsXG4gICAgICAgICAgICBcIkRlYWRMZXR0ZXJDb25maWdcIjoge1xuICAgICAgICAgICAgXCJUYXJnZXRBcm5cIjoge1xuICAgICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcIk15TGFtYmRhRGVhZExldHRlclF1ZXVlMzk5RUVBMkRcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIkRlcGVuZHNPblwiOiBbXG4gICAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIixcbiAgICAgICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIH1cbiAgICApO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZWZhdWx0IGZ1bmN0aW9uIHdpdGggU1FTIERMUSB3aGVuIGNsaWVudCBzZXRzIGRlYWRMZXR0ZXJRdWV1ZUVuYWJsZWQgdG8gZmFsc2UnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdNeUxhbWJkYScsIHtcbiAgICAgIGNvZGU6IG5ldyBsYW1iZGEuSW5saW5lQ29kZSgnZm9vJyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Ob2RlSlM2MTAsXG4gICAgICBkZWFkTGV0dGVyUXVldWVFbmFibGVkOiBmYWxzZSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaChcbiAgICAgIHtcbiAgICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpJQU06OlJvbGVcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiQXNzdW1lUm9sZVBvbGljeURvY3VtZW50XCI6IHtcbiAgICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiUHJpbmNpcGFsXCI6IHtcbiAgICAgICAgICAgICAgXCJTZXJ2aWNlXCI6IHsgXCJGbjo6Sm9pblwiOiBbXCJcIiwgW1wibGFtYmRhLlwiLCB7IFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiIH1dXSB9XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiTWFuYWdlZFBvbGljeUFybnNcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIlJlZlwiOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXCI6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGVcIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAgXCJNeUxhbWJkYUNDRTgwMkZCXCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OkxhbWJkYTo6RnVuY3Rpb25cIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiQ29kZVwiOiB7XG4gICAgICAgICAgICBcIlppcEZpbGVcIjogXCJmb29cIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJIYW5kbGVyXCI6IFwiaW5kZXguaGFuZGxlclwiLFxuICAgICAgICAgIFwiUm9sZVwiOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjZcIixcbiAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiUnVudGltZVwiOiBcIm5vZGVqczYuMTBcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJEZXBlbmRzT25cIjogW1xuICAgICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgICk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2RlZmF1bHQgZnVuY3Rpb24gd2l0aCBTUVMgRExRIHdoZW4gY2xpZW50IHByb3ZpZGVzIFF1ZXVlIHRvIGJlIHVzZWQgYXMgRExRJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBjb25zdCBkbFF1ZXVlID0gbmV3IHNxcy5RdWV1ZShzdGFjaywgJ0RlYWRMZXR0ZXJRdWV1ZScsIHtcbiAgICAgIHF1ZXVlTmFtZTogJ015TGFtYmRhX0RMUScsXG4gICAgICByZXRlbnRpb25QZXJpb2RTZWM6IDEyMDk2MDBcbiAgICB9KTtcblxuICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdNeUxhbWJkYScsIHtcbiAgICAgIGNvZGU6IG5ldyBsYW1iZGEuSW5saW5lQ29kZSgnZm9vJyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Ob2RlSlM2MTAsXG4gICAgICBkZWFkTGV0dGVyUXVldWU6IGRsUXVldWUsXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2goXG4gICAgICB7XG4gICAgICAgIFwiUmVzb3VyY2VzXCI6IHtcbiAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpJQU06OlJvbGVcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgICAgXCJBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFwic3RzOkFzc3VtZVJvbGVcIixcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlByaW5jaXBhbFwiOiB7XG4gICAgICAgICAgICAgICAgXCJTZXJ2aWNlXCI6IHsgXCJGbjo6Sm9pblwiOiBbXCJcIiwgW1wibGFtYmRhLlwiLCB7IFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiIH1dXSB9XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFwiTWFuYWdlZFBvbGljeUFybnNcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgIFwiYXJuOlwiLFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIlJlZlwiOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIFwiOmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTVCQkM2RjY4XCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OklBTTo6UG9saWN5XCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICAgIFwiUG9saWN5RG9jdW1lbnRcIjoge1xuICAgICAgICAgICAgXCJTdGF0ZW1lbnRcIjogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiQWN0aW9uXCI6IFwic3FzOlNlbmRNZXNzYWdlXCIsXG4gICAgICAgICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgICAgICAgXCJSZXNvdXJjZVwiOiB7XG4gICAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICBcIkRlYWRMZXR0ZXJRdWV1ZTlGNDgxNTQ2XCIsXG4gICAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgXCJWZXJzaW9uXCI6IFwiMjAxMi0xMC0xN1wiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgXCJQb2xpY3lOYW1lXCI6IFwiTXlMYW1iZGFTZXJ2aWNlUm9sZURlZmF1bHRQb2xpY3k1QkJDNkY2OFwiLFxuICAgICAgICAgICAgXCJSb2xlc1wiOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFwiUmVmXCI6IFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIk15TGFtYmRhQ0NFODAyRkJcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6TGFtYmRhOjpGdW5jdGlvblwiLFxuICAgICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgICAgICBcIkNvZGVcIjoge1xuICAgICAgICAgICAgXCJaaXBGaWxlXCI6IFwiZm9vXCJcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBcIkhhbmRsZXJcIjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICAgICAgICBcIlJvbGVcIjoge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjZcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFwiUnVudGltZVwiOiBcIm5vZGVqczYuMTBcIixcbiAgICAgICAgICAgIFwiRGVhZExldHRlckNvbmZpZ1wiOiB7XG4gICAgICAgICAgICBcIlRhcmdldEFyblwiOiB7XG4gICAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICAgIFwiRGVhZExldHRlclF1ZXVlOUY0ODE1NDZcIixcbiAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIkRlcGVuZHNPblwiOiBbXG4gICAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIixcbiAgICAgICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIH1cbiAgICAsIE1hdGNoU3R5bGUuU1VQRVJTRVQpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZWZhdWx0IGZ1bmN0aW9uIHdpdGggU1FTIERMUSB3aGVuIGNsaWVudCBwcm92aWRlcyBRdWV1ZSB0byBiZSB1c2VkIGFzIERMUSBhbmQgZGVhZExldHRlclF1ZXVlRW5hYmxlZCBzZXQgdG8gdHJ1ZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgY29uc3QgZGxRdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdEZWFkTGV0dGVyUXVldWUnLCB7XG4gICAgICBxdWV1ZU5hbWU6ICdNeUxhbWJkYV9ETFEnLFxuICAgICAgcmV0ZW50aW9uUGVyaW9kU2VjOiAxMjA5NjAwXG4gICAgfSk7XG5cbiAgICBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgICBjb2RlOiBuZXcgbGFtYmRhLklubGluZUNvZGUoJ2ZvbycpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTNjEwLFxuICAgICAgZGVhZExldHRlclF1ZXVlRW5hYmxlZDogdHJ1ZSxcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZTogZGxRdWV1ZSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaChcbiAgICAgIHtcbiAgICAgICAgXCJSZXNvdXJjZXNcIjoge1xuICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiOiB7XG4gICAgICAgICAgXCJUeXBlXCI6IFwiQVdTOjpJQU06OlJvbGVcIixcbiAgICAgICAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgICAgIFwiQXNzdW1lUm9sZVBvbGljeURvY3VtZW50XCI6IHtcbiAgICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgICAgICAgIFwiUHJpbmNpcGFsXCI6IHtcbiAgICAgICAgICAgICAgXCJTZXJ2aWNlXCI6IHsgXCJGbjo6Sm9pblwiOiBbXCJcIiwgW1wibGFtYmRhLlwiLCB7IFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiIH1dXSB9XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiTWFuYWdlZFBvbGljeUFybnNcIjogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIlJlZlwiOiBcIkFXUzo6UGFydGl0aW9uXCJcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXCI6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGVcIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTVCQkM2RjY4XCI6IHtcbiAgICAgICAgICBcIlR5cGVcIjogXCJBV1M6OklBTTo6UG9saWN5XCIsXG4gICAgICAgICAgXCJQcm9wZXJ0aWVzXCI6IHtcbiAgICAgICAgICBcIlBvbGljeURvY3VtZW50XCI6IHtcbiAgICAgICAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgXCJBY3Rpb25cIjogXCJzcXM6U2VuZE1lc3NhZ2VcIixcbiAgICAgICAgICAgICAgXCJFZmZlY3RcIjogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBcIlJlc291cmNlXCI6IHtcbiAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICBcIkRlYWRMZXR0ZXJRdWV1ZTlGNDgxNTQ2XCIsXG4gICAgICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFwiUG9saWN5TmFtZVwiOiBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIixcbiAgICAgICAgICBcIlJvbGVzXCI6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiUmVmXCI6IFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICB9LFxuICAgICAgICBcIk15TGFtYmRhQ0NFODAyRkJcIjoge1xuICAgICAgICAgIFwiVHlwZVwiOiBcIkFXUzo6TGFtYmRhOjpGdW5jdGlvblwiLFxuICAgICAgICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICAgICAgXCJDb2RlXCI6IHtcbiAgICAgICAgICAgIFwiWmlwRmlsZVwiOiBcImZvb1wiXG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIkhhbmRsZXJcIjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICAgICAgXCJSb2xlXCI6IHtcbiAgICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiLFxuICAgICAgICAgICAgXCJBcm5cIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgXCJSdW50aW1lXCI6IFwibm9kZWpzNi4xMFwiLFxuICAgICAgICAgIFwiRGVhZExldHRlckNvbmZpZ1wiOiB7XG4gICAgICAgICAgICBcIlRhcmdldEFyblwiOiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcIkRlYWRMZXR0ZXJRdWV1ZTlGNDgxNTQ2XCIsXG4gICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgIF1cbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgICAgfSxcbiAgICAgICAgICBcIkRlcGVuZHNPblwiOiBbXG4gICAgICAgICAgICBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIixcbiAgICAgICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCIsXG4gICAgICAgICAgXVxuICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAsIE1hdGNoU3R5bGUuU1VQRVJTRVQpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdlcnJvciB3aGVuIGRlZmF1bHQgZnVuY3Rpb24gd2l0aCBTUVMgRExRIHdoZW4gY2xpZW50IHByb3ZpZGVzIFF1ZXVlIHRvIGJlIHVzZWQgYXMgRExRIGFuZCBkZWFkTGV0dGVyUXVldWVFbmFibGVkIHNldCB0byBmYWxzZScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgY29uc3QgZGxRdWV1ZSA9IG5ldyBzcXMuUXVldWUoc3RhY2ssICdEZWFkTGV0dGVyUXVldWUnLCB7XG4gICAgICBxdWV1ZU5hbWU6ICdNeUxhbWJkYV9ETFEnLFxuICAgICAgcmV0ZW50aW9uUGVyaW9kU2VjOiAxMjA5NjAwXG4gICAgfSk7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdmb28nKSxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTNjEwLFxuICAgIGRlYWRMZXR0ZXJRdWV1ZUVuYWJsZWQ6IGZhbHNlLFxuICAgIGRlYWRMZXR0ZXJRdWV1ZTogZGxRdWV1ZSxcbiAgICB9KSwgL2RlYWRMZXR0ZXJRdWV1ZSBkZWZpbmVkIGJ1dCBkZWFkTGV0dGVyUXVldWVFbmFibGVkIGV4cGxpY2l0bHkgc2V0IHRvIGZhbHNlLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZGVmYXVsdCBmdW5jdGlvbiB3aXRoIEFjdGl2ZSB0cmFjaW5nJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgICBjb2RlOiBuZXcgbGFtYmRhLklubGluZUNvZGUoJ2ZvbycpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTNjEwLFxuICAgICAgdHJhY2luZzogbGFtYmRhLlRyYWNpbmcuQWN0aXZlXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICBcIlBvbGljeURvY3VtZW50XCI6IHtcbiAgICAgIFwiU3RhdGVtZW50XCI6IFtcbiAgICAgIHtcbiAgICAgICAgXCJBY3Rpb25cIjogW1xuICAgICAgICBcInhyYXk6UHV0VHJhY2VTZWdtZW50c1wiLFxuICAgICAgICBcInhyYXk6UHV0VGVsZW1ldHJ5UmVjb3Jkc1wiXG4gICAgICAgIF0sXG4gICAgICAgIFwiRWZmZWN0XCI6IFwiQWxsb3dcIixcbiAgICAgICAgXCJSZXNvdXJjZVwiOiBcIipcIlxuICAgICAgfVxuICAgICAgXSxcbiAgICAgIFwiVmVyc2lvblwiOiBcIjIwMTItMTAtMTdcIlxuICAgIH0sXG4gICAgXCJQb2xpY3lOYW1lXCI6IFwiTXlMYW1iZGFTZXJ2aWNlUm9sZURlZmF1bHRQb2xpY3k1QkJDNkY2OFwiLFxuICAgIFwiUm9sZXNcIjogW1xuICAgICAge1xuICAgICAgXCJSZWZcIjogXCJNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjZcIlxuICAgICAgfVxuICAgIF1cbiAgICB9KSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgIFwiUHJvcGVydGllc1wiOiB7XG4gICAgICBcIkNvZGVcIjoge1xuICAgICAgXCJaaXBGaWxlXCI6IFwiZm9vXCJcbiAgICAgIH0sXG4gICAgICBcIkhhbmRsZXJcIjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICBcIlJvbGVcIjoge1xuICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjZcIixcbiAgICAgICAgXCJBcm5cIlxuICAgICAgXVxuICAgICAgfSxcbiAgICAgIFwiUnVudGltZVwiOiBcIm5vZGVqczYuMTBcIixcbiAgICAgIFwiVHJhY2luZ0NvbmZpZ1wiOiB7XG4gICAgICBcIk1vZGVcIjogXCJBY3RpdmVcIlxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXBlbmRzT25cIjogW1xuICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTVCQkM2RjY4XCIsXG4gICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiLFxuICAgIF1cbiAgICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZGVmYXVsdCBmdW5jdGlvbiB3aXRoIFBhc3NUaHJvdWdoIHRyYWNpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdNeUxhbWJkYScsIHtcbiAgICAgIGNvZGU6IG5ldyBsYW1iZGEuSW5saW5lQ29kZSgnZm9vJyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Ob2RlSlM2MTAsXG4gICAgICB0cmFjaW5nOiBsYW1iZGEuVHJhY2luZy5QYXNzVGhyb3VnaFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICB7XG4gICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgXCJ4cmF5OlB1dFRyYWNlU2VnbWVudHNcIixcbiAgICAgICAgXCJ4cmF5OlB1dFRlbGVtZXRyeVJlY29yZHNcIlxuICAgICAgICBdLFxuICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgIFwiUmVzb3VyY2VcIjogXCIqXCJcbiAgICAgIH1cbiAgICAgIF0sXG4gICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICB9LFxuICAgIFwiUG9saWN5TmFtZVwiOiBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIixcbiAgICBcIlJvbGVzXCI6IFtcbiAgICAgIHtcbiAgICAgIFwiUmVmXCI6IFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgIH1cbiAgICBdXG4gICAgfSkpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgXCJDb2RlXCI6IHtcbiAgICAgIFwiWmlwRmlsZVwiOiBcImZvb1wiXG4gICAgICB9LFxuICAgICAgXCJIYW5kbGVyXCI6IFwiaW5kZXguaGFuZGxlclwiLFxuICAgICAgXCJSb2xlXCI6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCIsXG4gICAgICAgIFwiQXJuXCJcbiAgICAgIF1cbiAgICAgIH0sXG4gICAgICBcIlJ1bnRpbWVcIjogXCJub2RlanM2LjEwXCIsXG4gICAgICBcIlRyYWNpbmdDb25maWdcIjoge1xuICAgICAgXCJNb2RlXCI6IFwiUGFzc1Rocm91Z2hcIlxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXBlbmRzT25cIjogW1xuICAgICAgXCJNeUxhbWJkYVNlcnZpY2VSb2xlRGVmYXVsdFBvbGljeTVCQkM2RjY4XCIsXG4gICAgICBcIk15TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNlwiLFxuICAgIF1cbiAgICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZGVmYXVsdCBmdW5jdGlvbiB3aXRoIERpc2FibGVkIHRyYWNpbmcnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdNeUxhbWJkYScsIHtcbiAgICAgIGNvZGU6IG5ldyBsYW1iZGEuSW5saW5lQ29kZSgnZm9vJyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Ob2RlSlM2MTAsXG4gICAgICB0cmFjaW5nOiBsYW1iZGEuVHJhY2luZy5EaXNhYmxlZFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgXCJQb2xpY3lEb2N1bWVudFwiOiB7XG4gICAgICBcIlN0YXRlbWVudFwiOiBbXG4gICAgICB7XG4gICAgICAgIFwiQWN0aW9uXCI6IFtcbiAgICAgICAgXCJ4cmF5OlB1dFRyYWNlU2VnbWVudHNcIixcbiAgICAgICAgXCJ4cmF5OlB1dFRlbGVtZXRyeVJlY29yZHNcIlxuICAgICAgICBdLFxuICAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAgICAgIFwiUmVzb3VyY2VcIjogXCIqXCJcbiAgICAgIH1cbiAgICAgIF0sXG4gICAgICBcIlZlcnNpb25cIjogXCIyMDEyLTEwLTE3XCJcbiAgICB9LFxuICAgIFwiUG9saWN5TmFtZVwiOiBcIk15TGFtYmRhU2VydmljZVJvbGVEZWZhdWx0UG9saWN5NUJCQzZGNjhcIixcbiAgICBcIlJvbGVzXCI6IFtcbiAgICAgIHtcbiAgICAgIFwiUmVmXCI6IFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICAgIH1cbiAgICBdXG4gICAgfSkpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICBcIlByb3BlcnRpZXNcIjoge1xuICAgICAgXCJDb2RlXCI6IHtcbiAgICAgIFwiWmlwRmlsZVwiOiBcImZvb1wiXG4gICAgICB9LFxuICAgICAgXCJIYW5kbGVyXCI6IFwiaW5kZXguaGFuZGxlclwiLFxuICAgICAgXCJSb2xlXCI6IHtcbiAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCIsXG4gICAgICAgIFwiQXJuXCJcbiAgICAgIF1cbiAgICAgIH0sXG4gICAgICBcIlJ1bnRpbWVcIjogXCJub2RlanM2LjEwXCJcbiAgICB9LFxuICAgIFwiRGVwZW5kc09uXCI6IFtcbiAgICAgIFwiTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2XCJcbiAgICBdXG4gICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2dyYW50SW52b2tlIGFkZHMgaWFtOkludm9rZUZ1bmN0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5BY2NvdW50UHJpbmNpcGFsKCcxMjM0JyksXG4gICAgfSk7XG4gICAgY29uc3QgZm4gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnRnVuY3Rpb24nLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5pbmxpbmUoJ3h4eCcpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTODEwLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGZuLmdyYW50SW52b2tlKHJvbGUpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICdsYW1iZGE6SW52b2tlRnVuY3Rpb24nLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHsgXCJGbjo6R2V0QXR0XCI6IFsgXCJGdW5jdGlvbjc2ODU2Njc3XCIsIFwiQXJuXCIgXSB9XG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9XG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0NhbiB1c2UgbWV0cmljRXJyb3JzIG9uIGEgbGFtYmRhIEZ1bmN0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBmbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdGdW5jdGlvbicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmlubGluZSgneHh4JyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Ob2RlSlM4MTAsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2subm9kZS5yZXNvbHZlKGZuLm1ldHJpY0Vycm9ycygpKSwge1xuICAgICAgZGltZW5zaW9uczogeyBGdW5jdGlvbk5hbWU6IHsgUmVmOiAnRnVuY3Rpb243Njg1NjY3NycgfX0sXG4gICAgICBuYW1lc3BhY2U6ICdBV1MvTGFtYmRhJyxcbiAgICAgIG1ldHJpY05hbWU6ICdFcnJvcnMnLFxuICAgICAgcGVyaW9kU2VjOiAzMDAsXG4gICAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2FkZEV2ZW50U291cmNlIGNhbGxzIGJpbmQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGZuID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ0Z1bmN0aW9uJywge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuaW5saW5lKCd4eHgnKSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5vZGVKUzgxMCxcbiAgICB9KTtcblxuICAgIGxldCBiaW5kVGFyZ2V0O1xuXG4gICAgY2xhc3MgRXZlbnRTb3VyY2VNb2NrIGltcGxlbWVudHMgbGFtYmRhLklFdmVudFNvdXJjZSB7XG4gICAgICBwdWJsaWMgYmluZCh0YXJnZXQ6IGxhbWJkYS5JRnVuY3Rpb24pIHtcbiAgICAgICAgYmluZFRhcmdldCA9IHRhcmdldDtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBXSEVOXG4gICAgZm4uYWRkRXZlbnRTb3VyY2UobmV3IEV2ZW50U291cmNlTW9jaygpKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnNhbWUoYmluZFRhcmdldCwgZm4pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnc3VwcG9ydCBpbmxpbmUgY29kZSBmb3IgUnVieSBydW50aW1lJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgICBjb2RlOiBuZXcgbGFtYmRhLklubGluZUNvZGUoJ2ZvbycpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUnVieTI1LFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b01hdGNoKHsgUmVzb3VyY2VzOlxuICAgICAgeyBNeUxhbWJkYVNlcnZpY2VSb2xlNDUzOUVDQjY6XG4gICAgICAgICAgeyBUeXBlOiAnQVdTOjpJQU06OlJvbGUnLFxuICAgICAgICAgIFByb3BlcnRpZXM6XG4gICAgICAgICAgeyBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6XG4gICAgICAgICAgICB7IFN0YXRlbWVudDpcbiAgICAgICAgICAgIFsgeyBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6IHsgXCJGbjo6Sm9pblwiOiBbXCJcIiwgWydsYW1iZGEuJywgeyBSZWY6IFwiQVdTOjpVUkxTdWZmaXhcIiB9XV0gfSB9IH0gXSxcbiAgICAgICAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnIH0sXG4gICAgICAgICAgTWFuYWdlZFBvbGljeUFybnM6XG4gICAgICAgICAgLy8gYXJuOmF3czppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZVxuICAgICAgICAgICAgLy8gdHNsaW50OmRpc2FibGUtbmV4dC1saW5lOm1heC1saW5lLWxlbmd0aFxuICAgICAgICAgICAgW3snRm46OkpvaW4nOiBbJycsIFsnYXJuOicsIHtSZWY6ICdBV1M6OlBhcnRpdGlvbid9LCAnOmlhbTo6YXdzOnBvbGljeS9zZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJ11dfV0sXG4gICAgICAgICAgfX0sXG4gICAgICAgIE15TGFtYmRhQ0NFODAyRkI6XG4gICAgICAgICAgeyBUeXBlOiAnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJyxcbiAgICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgIHsgQ29kZTogeyBaaXBGaWxlOiAnZm9vJyB9LFxuICAgICAgICAgIEhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgICAgICBSb2xlOiB7ICdGbjo6R2V0QXR0JzogWyAnTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2JywgJ0FybicgXSB9LFxuICAgICAgICAgIFJ1bnRpbWU6ICdydWJ5Mi41JyB9LFxuICAgICAgICAgIERlcGVuZHNPbjogWyAnTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2JyBdIH0gfSB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndXNpbmcgYW4gaW5jb21wYXRpYmxlIGxheWVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCAnVGVzdFN0YWNrJyk7XG4gICAgY29uc3QgbGF5ZXIgPSBsYW1iZGEuTGF5ZXJWZXJzaW9uLmltcG9ydChzdGFjaywgJ1Rlc3RMYXllcicsIHtcbiAgICAgIGxheWVyVmVyc2lvbkFybjogJ2Fybjphd3M6Li4uJyxcbiAgICAgIGNvbXBhdGlibGVSdW50aW1lczogW2xhbWJkYS5SdW50aW1lLk5vZGVKUzgxMF0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ0Z1bmN0aW9uJywge1xuICAgICAgICAgICAgICAgICAgbGF5ZXJzOiBbbGF5ZXJdLFxuICAgICAgICAgICAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTNjEwLFxuICAgICAgICAgICAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuaW5saW5lKCdleHBvcnRzLm1haW4gPSBmdW5jdGlvbigpIHsgY29uc29sZS5sb2coXCJET05FXCIpOyB9JyksXG4gICAgICAgICAgICAgICAgICBoYW5kbGVyOiAnaW5kZXgubWFpbidcbiAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgICAvbm9kZWpzNi4xMCBpcyBub3QgaW4gXFxbbm9kZWpzOC4xMFxcXS8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3VzaW5nIG1vcmUgdGhhbiA1IGxheWVycycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgJ1Rlc3RTdGFjaycpO1xuICAgIGNvbnN0IGxheWVycyA9IG5ldyBBcnJheSg2KS5maWxsKGxhbWJkYS5MYXllclZlcnNpb24uaW1wb3J0KHN0YWNrLCAnVGVzdExheWVyJywge1xuICAgICAgbGF5ZXJWZXJzaW9uQXJuOiAnYXJuOmF3czouLi4nLFxuICAgICAgY29tcGF0aWJsZVJ1bnRpbWVzOiBbbGFtYmRhLlJ1bnRpbWUuTm9kZUpTODEwXSxcbiAgICB9KSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ0Z1bmN0aW9uJywge1xuICAgICAgICAgICAgICAgICAgbGF5ZXJzLFxuICAgICAgICAgICAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTODEwLFxuICAgICAgICAgICAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuaW5saW5lKCdleHBvcnRzLm1haW4gPSBmdW5jdGlvbigpIHsgY29uc29sZS5sb2coXCJET05FXCIpOyB9JyksXG4gICAgICAgICAgICAgICAgICBoYW5kbGVyOiAnaW5kZXgubWFpbidcbiAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgICAvVW5hYmxlIHRvIGFkZCBsYXllcjovKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdlbnZpcm9ubWVudCB2YXJpYWJsZXMgYXJlIHByb2hpYml0ZWQgaW4gQ2hpbmEnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwgeyBlbnY6IHsgcmVnaW9uOiAnY24tbm9ydGgtMScgfX0pO1xuXG4gICAgLy8gV0hFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdNeUxhbWJkYScsIHtcbiAgICAgICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdmb28nKSxcbiAgICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Ob2RlSlMsXG4gICAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgICAgU09NRTogJ1ZhcmlhYmxlJ1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9LCAvRW52aXJvbm1lbnQgdmFyaWFibGVzIGFyZSBub3Qgc3VwcG9ydGVkLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZW52aXJvbm1lbnQgdmFyaWFibGVzIHdvcmsgaW4gYW4gdW5zcGVjaWZpZWQgcmVnaW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgICBjb2RlOiBuZXcgbGFtYmRhLklubGluZUNvZGUoJ2ZvbycpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgU09NRTogJ1ZhcmlhYmxlJ1xuICAgICAgfVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7XG4gICAgICBFbnZpcm9ubWVudDoge1xuICAgICAgICBWYXJpYWJsZXM6IHtcbiAgICAgICAgICBTT01FOiBcIlZhcmlhYmxlXCJcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuXG4gIH0sXG5cbiAgJ3N1cHBvcnQgcmVzZXJ2ZWQgY29uY3VycmVudCBleGVjdXRpb25zJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTXlMYW1iZGEnLCB7XG4gICAgICBjb2RlOiBuZXcgbGFtYmRhLklubGluZUNvZGUoJ2ZvbycpLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTLFxuICAgICAgcmVzZXJ2ZWRDb25jdXJyZW50RXhlY3V0aW9uczogMTBcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9NYXRjaCh7IFJlc291cmNlczpcbiAgICAgIHsgTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2OlxuICAgICAgICAgIHsgVHlwZTogJ0FXUzo6SUFNOjpSb2xlJyxcbiAgICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgIHsgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OlxuICAgICAgICAgICAgeyBTdGF0ZW1lbnQ6XG4gICAgICAgICAgICBbIHsgQWN0aW9uOiAnc3RzOkFzc3VtZVJvbGUnLFxuICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgIFByaW5jaXBhbDogeyBTZXJ2aWNlOiB7IFwiRm46OkpvaW5cIjogW1wiXCIsIFsnbGFtYmRhLicsIHsgUmVmOiBcIkFXUzo6VVJMU3VmZml4XCIgfV1dIH0gfSB9IF0sXG4gICAgICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyB9LFxuICAgICAgICAgIE1hbmFnZWRQb2xpY3lBcm5zOlxuICAgICAgICAgIC8vIGFybjphd3M6aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGVcbiAgICAgICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgICAgIFt7J0ZuOjpKb2luJzogWycnLCBbJ2FybjonLCB7UmVmOiAnQVdTOjpQYXJ0aXRpb24nfSwgJzppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FXU0xhbWJkYUJhc2ljRXhlY3V0aW9uUm9sZSddXX1dLFxuICAgICAgICAgIH19LFxuICAgICAgICBNeUxhbWJkYUNDRTgwMkZCOlxuICAgICAgICAgIHsgVHlwZTogJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsXG4gICAgICAgICAgUHJvcGVydGllczpcbiAgICAgICAgICB7IENvZGU6IHsgWmlwRmlsZTogJ2ZvbycgfSxcbiAgICAgICAgICBIYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgICAgUmVzZXJ2ZWRDb25jdXJyZW50RXhlY3V0aW9uczogMTAsXG4gICAgICAgICAgUm9sZTogeyAnRm46OkdldEF0dCc6IFsgJ015TGFtYmRhU2VydmljZVJvbGU0NTM5RUNCNicsICdBcm4nIF0gfSxcbiAgICAgICAgICBSdW50aW1lOiAnbm9kZWpzJyB9LFxuICAgICAgICAgIERlcGVuZHNPbjogWyAnTXlMYW1iZGFTZXJ2aWNlUm9sZTQ1MzlFQ0I2JyBdIH0gfSB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaXRzIHBvc3NpYmxlIHRvIHNwZWNpZnkgZXZlbnQgc291cmNlcyB1cG9uIGNyZWF0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIGxldCBiaW5kQ291bnQgPSAwO1xuXG4gICAgY2xhc3MgRXZlbnRTb3VyY2UgaW1wbGVtZW50cyBsYW1iZGEuSUV2ZW50U291cmNlIHtcbiAgICAgIHB1YmxpYyBiaW5kKF86IGxhbWJkYS5GdW5jdGlvbkJhc2UpOiB2b2lkIHtcbiAgICAgICAgYmluZENvdW50Kys7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdmbicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmlubGluZSgnYm9vbScpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTm9kZUpTODEwLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmJhbScsXG4gICAgICBldmVudHM6IFtcbiAgICAgICAgbmV3IEV2ZW50U291cmNlKCksXG4gICAgICAgIG5ldyBFdmVudFNvdXJjZSgpLFxuICAgICAgXVxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKGJpbmRDb3VudCwgMik7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1Byb3ZpZGVkIFJ1bnRpbWUgcmV0dXJucyB0aGUgcmlnaHQgdmFsdWVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgcnQgPSBsYW1iZGEuUnVudGltZS5Qcm92aWRlZDtcblxuICAgIHRlc3QuZXF1YWwocnQubmFtZSwgJ3Byb3ZpZGVkJyk7XG4gICAgdGVzdC5lcXVhbChydC5zdXBwb3J0c0lubGluZUNvZGUsIGZhbHNlKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzcGVjaWZ5IGxvZyByZXRlbnRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdNeUxhbWJkYScsIHtcbiAgICAgIGNvZGU6IG5ldyBsYW1iZGEuSW5saW5lQ29kZSgnZm9vJyksXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5Ob2RlSlMsXG4gICAgICBsb2dSZXRlbnRpb25EYXlzOiBsb2dzLlJldGVudGlvbkRheXMuT25lTW9udGhcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQ3VzdG9tOjpMb2dSZXRlbnRpb24nLCB7XG4gICAgICAnTG9nR3JvdXBOYW1lJzoge1xuICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgJycsXG4gICAgICAgICAgW1xuICAgICAgICAgICAgJy9hd3MvbGFtYmRhLycsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogJ015TGFtYmRhQ0NFODAyRkInXG4gICAgICAgICAgICB9XG4gICAgICAgICAgXVxuICAgICAgICBdXG4gICAgICB9LFxuICAgICAgJ1JldGVudGlvbkluRGF5cyc6IDMwXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gICB9XG59O1xuXG5mdW5jdGlvbiBuZXdUZXN0TGFtYmRhKHNjb3BlOiBjZGsuQ29uc3RydWN0KSB7XG4gIHJldHVybiBuZXcgbGFtYmRhLkZ1bmN0aW9uKHNjb3BlLCAnTXlMYW1iZGEnLCB7XG4gICAgY29kZTogbmV3IGxhbWJkYS5JbmxpbmVDb2RlKCdmb28nKSxcbiAgICBoYW5kbGVyOiAnYmFyJyxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QeXRob24yN1xuICB9KTtcbn1cbiJdfQ==