"use strict";
const AWS = require("aws-sdk-mock");
const nock = require("nock");
const sinon = require("sinon");
const provider = require("../lib/log-retention-provider");
const eventCommon = {
    ServiceToken: 'token',
    ResponseURL: 'https://localhost',
    StackId: 'stackId',
    RequestId: 'requestId',
    LogicalResourceId: 'logicalResourceId',
    PhysicalResourceId: 'group',
    ResourceType: 'Custom::LogRetention',
};
const context = {
    functionName: 'provider'
};
function createRequest(type) {
    return nock('https://localhost')
        .put('/', (body) => body.Status === type && body.PhysicalResourceId === 'group')
        .reply(200);
}
module.exports = {
    'tearDown'(callback) {
        AWS.restore();
        nock.cleanAll();
        callback();
    },
    async 'create event'(test) {
        const createLogGroupFake = sinon.fake.resolves({});
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = Object.assign({}, eventCommon, { RequestType: 'Create', ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group'
            } });
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group'
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: 'group',
            retentionInDays: 30
        });
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: '/aws/lambda/provider'
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: '/aws/lambda/provider',
            retentionInDays: 1
        });
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'update event with new log retention'(test) {
        const error = new Error();
        error.code = 'ResourceAlreadyExistsException';
        const createLogGroupFake = sinon.fake.rejects(error);
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = Object.assign({}, eventCommon, { RequestType: 'Update', ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '365',
                LogGroupName: 'group'
            }, OldResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
                RetentionInDays: '30'
            } });
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group'
        });
        sinon.assert.calledWith(putRetentionPolicyFake, {
            logGroupName: 'group',
            retentionInDays: 365
        });
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'update event with log retention undefined'(test) {
        const error = new Error();
        error.code = 'ResourceAlreadyExistsException';
        const createLogGroupFake = sinon.fake.rejects(error);
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = Object.assign({}, eventCommon, { RequestType: 'Update', PhysicalResourceId: 'group', ResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group'
            }, OldResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group',
                RetentionInDays: '365'
            } });
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.calledWith(createLogGroupFake, {
            logGroupName: 'group'
        });
        sinon.assert.calledWith(deleteRetentionPolicyFake, {
            logGroupName: 'group'
        });
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'delete event'(test) {
        const createLogGroupFake = sinon.fake.resolves({});
        const putRetentionPolicyFake = sinon.fake.resolves({});
        const deleteRetentionPolicyFake = sinon.fake.resolves({});
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        AWS.mock('CloudWatchLogs', 'putRetentionPolicy', putRetentionPolicyFake);
        AWS.mock('CloudWatchLogs', 'deleteRetentionPolicy', deleteRetentionPolicyFake);
        const event = Object.assign({}, eventCommon, { RequestType: 'Delete', PhysicalResourceId: 'group', ResourceProperties: {
                ServiceToken: 'token',
                LogGroupName: 'group'
            } });
        const request = createRequest('SUCCESS');
        await provider.handler(event, context);
        sinon.assert.notCalled(createLogGroupFake);
        sinon.assert.notCalled(putRetentionPolicyFake);
        sinon.assert.notCalled(deleteRetentionPolicyFake);
        test.equal(request.isDone(), true);
        test.done();
    },
    async 'responds with FAILED on error'(test) {
        const createLogGroupFake = sinon.fake.rejects(new Error('UnkownError'));
        AWS.mock('CloudWatchLogs', 'createLogGroup', createLogGroupFake);
        const event = Object.assign({}, eventCommon, { RequestType: 'Create', ResourceProperties: {
                ServiceToken: 'token',
                RetentionInDays: '30',
                LogGroupName: 'group'
            } });
        const request = createRequest('FAILED');
        await provider.handler(event, context);
        test.equal(request.isDone(), true);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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