"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/cdk");
const lambda = require("../lib");
/**
 * Use a class as test fixture
 *
 * setUp() will be mapped to the (synchronous) constructor. tearDown(cb) will be called if available.
 */
function classFixture(klass) {
    let fixture;
    const ret = {
        setUp(cb) {
            fixture = new klass();
            cb();
        },
        tearDown(cb) {
            if (fixture.tearDown) {
                fixture.tearDown(cb);
            }
            else {
                cb();
            }
        }
    };
    const testNames = Reflect.ownKeys(klass.prototype).filter(m => m !== 'tearDown' && m !== 'constructor');
    for (const testName of testNames) {
        ret[testName] = (test) => fixture[testName](test);
    }
    return ret;
}
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
module.exports = {
    'lambda in a VPC': classFixture(class Henk {
        constructor() {
            // GIVEN
            this.stack = new cdk.Stack();
            this.vpc = new ec2.VpcNetwork(this.stack, 'VPC');
            // WHEN
            this.lambda = new lambda.Function(this.stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NodeJS610,
                vpc: this.vpc,
                allowAllOutbound: false
            });
        }
        'has subnet and securitygroup'(test) {
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] }
                    ],
                    SubnetIds: [
                        { Ref: "VPCPrivateSubnet1Subnet8BCA10E0" },
                        { Ref: "VPCPrivateSubnet2SubnetCFCDAA7A" },
                        { Ref: "VPCPrivateSubnet3Subnet3EDCD457" }
                    ]
                }
            }));
            test.done();
        }
        'participates in Connections objects'(test) {
            // GIVEN
            const securityGroup = new ec2.SecurityGroup(this.stack, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            this.lambda.connections.allowTo(somethingConnectable, new ec2.TcpAllPorts(), 'Lambda can call connectable');
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(this.stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupEgress", {
                GroupId: { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] },
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                DestinationSecurityGroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
                FromPort: 0,
                ToPort: 65535
            }));
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(this.stack).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                FromPort: 0,
                GroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
                SourceSecurityGroupId: { "Fn::GetAtt": ["LambdaSecurityGroupE74659A1", "GroupId"] },
                ToPort: 65535
            }));
            test.done();
        }
        'can still make Connections after export/import'(test) {
            // GIVEN
            const stack2 = new cdk.Stack();
            const securityGroup = new ec2.SecurityGroup(stack2, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            const importedLambda = lambda.Function.import(stack2, 'Lambda', this.lambda.export());
            importedLambda.connections.allowTo(somethingConnectable, new ec2.TcpAllPorts(), 'Lambda can call connectable');
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(stack2).to(assert_1.haveResource("AWS::EC2::SecurityGroupEgress", {
                GroupId: { "Fn::ImportValue": "Stack:LambdaSecurityGroupId9A2717B3" },
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                DestinationSecurityGroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
                FromPort: 0,
                ToPort: 65535
            }));
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(stack2).to(assert_1.haveResource("AWS::EC2::SecurityGroupIngress", {
                IpProtocol: "tcp",
                Description: "Lambda can call connectable",
                FromPort: 0,
                GroupId: { "Fn::GetAtt": ["SomeSecurityGroupEF219AD6", "GroupId"] },
                SourceSecurityGroupId: { "Fn::ImportValue": "Stack:LambdaSecurityGroupId9A2717B3" },
                ToPort: 65535
            }));
            test.done();
        }
    }),
    'lambda without VPC throws Error upon accessing connections'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const lambdaFn = new lambda.Function(stack, 'Lambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NodeJS610,
        });
        // WHEN
        test.throws(() => {
            lambdaFn.connections.allowToAnyIPv4(new ec2.TcpAllPorts(), 'Reach for the world Lambda!');
        });
        test.done();
    },
    'picking public subnets is not allowed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.VpcNetwork(stack, 'VPC');
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NodeJS610,
                vpc,
                vpcSubnets: { subnetType: ec2.SubnetType.Public }
            });
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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