#!/usr/bin/env python
#  -*- coding: utf-8 -*-

import argparse
import os
import sys

import numpy as np

import mpolar

from datetime import datetime
import xarray as xr

__version__ = '0.1'


def main():
    parser = argparse.ArgumentParser(
        description=""" -- CSV2NC: Converts CSV Satori format to 5D polar NetCDF file --""",
        formatter_class=argparse.RawTextHelpFormatter
    )

    parser.add_argument('control_mode', type=str,
                        help="""control mode of the polar: possible values are p (power control) or s (speed control)""",
                        choices=['speed', 'power'])
    parser.add_argument('csv_file', type=str, help="""Satori CSV polar file to cconvert""")
    parser.add_argument('-vn', '--vessel_name', type=str, default='', help="""Vessel name""")
    parser.add_argument('-lc', '--loading_case', type=str, default='', help="""Loading case""")

    args = parser.parse_args()

    csv_filename = args.csv_file
    csv_filename = os.path.abspath(csv_filename)
    basename, _ = os.path.splitext(csv_filename)

    nc_filename = basename + ".nc"

    if args.control_mode == 'speed':
        print("SPEED CONTROL")
        control_name = "STW_kt"
        control_unit = "kt"

        variable_name = "PB_kW"
        variable_unit = "kW"
    elif args.control_mode == 'power':
        print("POWER CONTROL")
        control_name = "PB_kW"
        control_unit = "kW"

        variable_name = "STW_kt"
        variable_unit = "kt"
    else:
        raise RuntimeError("Unknown control mode, this should not happen...")

    print(csv_filename, "\t-->\t", nc_filename)

    ds_tmp = mpolar.parse(csv_filename, sep=';',
                          column_name='TWS_kt', column_unit='kt',
                          row_name='TWA_deg', row_unit='deg',
                          variable_name=variable_name, variable_unit=variable_unit,
                          control_name=control_name, control_unit=control_unit)

    ds_tmp = ds_tmp.rename({"stw_kt": "STW_kt"})
    ds_tmp = ds_tmp.rename({"pb_kw": "PB_kW"})
    ds_tmp = ds_tmp.rename({"tws_kt": "TWS_kt"})
    ds_tmp = ds_tmp.rename({"twa_deg": "TWA_deg"})

    # ds_tmp.to_netcdf("essai.nc")

    # sys.exit(1)

    ds_tmp = ds_tmp.transpose(control_name, "TWS_kt", "TWA_deg")

    coords = {
        control_name: ds_tmp.coords[control_name].to_numpy(),
        "TWS_kt": ds_tmp.coords["TWS_kt"].to_numpy(),
        "TWA_deg": ds_tmp.coords["TWA_deg"].to_numpy(),
        "WA_deg": np.array([0., 180.]),
        "Hs_m": np.zeros(1)
    }

    dims = [control_name, "TWS_kt", "TWA_deg", "WA_deg", "Hs_m"]

    var_array = np.zeros((coords[control_name].size,
                          coords["TWS_kt"].size,
                          coords["TWA_deg"].size,
                          coords["WA_deg"].size,
                          coords["Hs_m"].size,
                          ))

    for i, _ in enumerate(coords["WA_deg"]):
        var_array[:, :, :, i, 0] = ds_tmp.variables[variable_name]

    data_vars = {variable_name: (dims, var_array)}

    attrs = dict(
        info="Generated by CSV2NC tool",
        version=__version__,
        copyright="D-ICE ENGINEERING (C)",
        date=datetime.today().strftime('%Y %m %d %Hh%Mm%Ss'),
        vessel_type="HYBRID MOTOR AND SAIL VESSEL",
        control_variable=control_name,
        polar_type="ND",
        note="Wind and Waves are on starboard tack and enclosed data are consistent with this, in NED convention",
        vessel_name=args.vessel_name,
        loading_case=args.loading_case,
        json_input_data='',
    )

    ds_out = xr.Dataset(
        data_vars=data_vars,
        coords=coords,
        attrs=attrs
    )

    ds_out.coords[control_name].attrs["unit"] = ds_tmp.coords[control_name].attrs["units"]
    if control_name == "STW_kt":
        ds_out.coords[control_name].attrs["description"] = "SPEED THROUGH WATER"
    else:
        ds_out.coords[control_name].attrs["description"] = "BRAKE POWER"

    ds_out.coords["TWS_kt"].attrs["unit"] = "kt"
    ds_out.coords["TWS_kt"].attrs["description"] = "TRUE WIND SPEED"

    ds_out.coords["TWA_deg"].attrs["unit"] = "deg"
    ds_out.coords["TWA_deg"].attrs["description"] = "TRUE WIND ANGLE"

    ds_out.coords["WA_deg"].attrs["unit"] = "deg"
    ds_out.coords["WA_deg"].attrs["description"] = "WAVES ANGLE"

    ds_out.coords["Hs_m"].attrs["unit"] = "m"
    ds_out.coords["Hs_m"].attrs["description"] = "SIGNIFICANT WAVE HEIGHT"

    ds_out.variables[variable_name].attrs["unit"] = variable_unit
    if variable_name == "PB_kW":
        ds_out.variables[variable_name].attrs["description"] = "BRAKE_POWER"
    else:
        ds_out.variables[variable_name].attrs["description"] = "SPEED THROUGH WATER"

    ds_out.to_netcdf(nc_filename)


if __name__ == '__main__':
    main()
