"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const s3notifications = require("@aws-cdk/aws-s3-notifications");
const cdk = require("@aws-cdk/cdk");
/**
 * Since we can't take a dependency on @aws-cdk/sns, this is a simple wrapper
 * for AWS::SNS::Topic which implements IBucketNotificationDestination.
 */
class Topic extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.policy = new iam.PolicyDocument();
        this.notifyingBucketPaths = new Set();
        const resource = new cdk.CfnResource(this, 'Resource', { type: 'AWS::SNS::Topic' });
        const topicArn = resource.ref;
        new cdk.CfnResource(this, 'Policy', {
            type: 'AWS::SNS::TopicPolicy',
            properties: {
                Topics: [topicArn],
                PolicyDocument: this.policy
            }
        });
        this.topicArn = topicArn;
    }
    asBucketNotificationDestination(bucketArn, bucketId) {
        // add permission to each source bucket
        if (!this.notifyingBucketPaths.has(bucketId)) {
            this.policy.addStatement(new iam.PolicyStatement()
                .describe(`sid${this.policy.statementCount}`)
                .addServicePrincipal('s3.amazonaws.com')
                .addAction('sns:Publish')
                .addResource(this.topicArn)
                .addCondition('ArnLike', { "aws:SourceArn": bucketArn }));
            this.notifyingBucketPaths.add(bucketId);
        }
        return {
            arn: this.topicArn,
            type: s3notifications.BucketNotificationDestinationType.Topic
        };
    }
}
exports.Topic = Topic;
//# sourceMappingURL=data:application/json;base64,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