"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_2 = require("aws-cdk-lib");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const defaults = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
(0, core_1.SetConsistentFeatureFlags)(stack);
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.hours(8),
    minTtl: aws_cdk_lib_2.Duration.hours(5),
    maxTtl: aws_cdk_lib_2.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.minutes(0),
    minTtl: aws_cdk_lib_2.Duration.minutes(0),
    maxTtl: aws_cdk_lib_2.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.CUSTOM,
            authorizer: (0, core_1.CreateApiAuthorizer)(stack, `${(0, core_1.generateIntegStackName)(__filename)}-authorizer`)
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    },
    cloudFrontLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    },
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
(0, core_1.suppressCustomHandlerCfnNagWarnings)(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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