"""
Data loading logic, after web scraping process is completed.
author: Jose Vicente Nunez <kodegeek.com@protonmail.com>
"""
import datetime
import logging
from collections import defaultdict
from enum import Enum
from pathlib import Path
from typing import Any

import pandas
import tomlkit
from pandas import DataFrame, Series
from tomlkit import TOMLDocument

logging.basicConfig(format='%(asctime)s %(message)s', encoding='utf-8', level=logging.INFO)

"""
Runners started on waves, but for basic analysis we will assume all runners were able to run
at the same time.
"""
BASE_RACE_DATETIME = {
    2023: datetime.datetime(year=2023, month=9, day=4, hour=20, minute=0, second=0, microsecond=0),
    2024: datetime.datetime(year=2024, month=9, day=9, hour=20, minute=0, second=0, microsecond=0)
}


class Level(Enum):
    """
    Course levels
    Interested only in people who completed the 86 floors. So is either full course or dnf
    """
    FULL = "Full Course"
    DNF = "DNF"


class RaceFields(Enum):
    """
    Fields
    """
    BIB = "bib"
    NAME = "name"
    GENDER = "gender"
    AGE = "age"
    COUNTRY = "country"
    STATE = "state"
    CITY = "locality"
    TIME = "full course"
    TWENTY_FLOOR_TIME = '20th floor'
    SIXTY_FIVE_FLOOR_TIME = '65th floor'


FIELD_NAMES = [x.value for x in RaceFields]
FIELD_NAMES_AND_POS: dict[RaceFields, int] = {field: idx for idx, field in zip(range(0, len(RaceFields)), RaceFields, strict=False)}

RACE_RESULTS_JSON_FULL_LEVEL = {
    2023: Path(__file__).parent.joinpath("results-2023.json"),
    2024: Path(__file__).parent.joinpath("results-2024.json")
}
DEFAULT_YEAR = 2024
COUNTRY_DETAILS = Path(__file__).parent.joinpath("country_codes.toml")


def load_json_data(
        data_file: Path = None,
        remove_dnf: bool = True,
        default_year: int = 2023,
        use_pretty: bool = False
) -> DataFrame:
    """
    Load the JSON lines as a dataframe.
    Args:
        data_file: JSON file, generated by a run from https://github.com/josevnz/athlinks-races
        remove_dnf: Whether to remove the DNF column from the dataframe. By default, remove non-finishers to avoid skewing the results.
        default_year: Default year to load the data from
        use_pretty: Whether to use pretty formatting class for the race durations or not.
    The split_data contains more nested details. When each row is converted to a dict it looks like this:
    [
    {'distance_m': 320, 'gun_time_ms': {'timeInMillis': 788000, 'timeUnit': 'm'},
    'interval_full': True,
    'name': 'Full Course',
    'number': 1,
    'time_ms': 788000,
    'time_with_penalties_ms': 788000},
   {'distance_m': 61,
    'gun_time_ms': {'timeInMillis': 157000, 'timeUnit': 'm'},
    'interval_full': False,
    'name': '20th Floor',
    'number': 2,
    'time_ms': 157000,
    'time_with_penalties_ms': 157000},
   {'distance_m': 229,
    'gun_time_ms': {'timeInMillis': 578000, 'timeUnit': 'm'},
    'interval_full': False,
    'name': '65th Floor',
    'number': 3,
    'time_ms': 578000,
    'time_with_penalties_ms': 578000}]

    To better process as a dataframe, these are flattened too
    """
    def_file = RACE_RESULTS_JSON_FULL_LEVEL[default_year] if data_file is None else data_file
    df = pandas.read_json(def_file, lines=True, encoding='utf-8')

    if remove_dnf:
        df = df.loc[df.racer_has_finished, :]

    # Normalize Age
    median_age = df[RaceFields.AGE.value].median()
    df[RaceFields.AGE.value] = df[RaceFields.AGE.value].fillna(median_age)
    df[RaceFields.AGE.value] = df[RaceFields.AGE.value].apply(lambda x: median_age if x == 0 else x)
    df[RaceFields.AGE.value] = df[RaceFields.AGE.value].astype(int)

    # Normalize state and city
    df.replace({RaceFields.STATE.value: {'-': ''}}, inplace=True)
    df[RaceFields.STATE.value] = df[RaceFields.STATE.value].fillna('')
    df[RaceFields.CITY.value] = df[RaceFields.CITY.value].fillna('')
    for col in [
        RaceFields.NAME.value,
        RaceFields.CITY.value,
    ]:
        df[col] = df[col].apply(lambda x: x.title())

    # Flatten inner keys, ignore others
    new_cols = defaultdict(list)
    for _, split_details in pandas.Series(df['split_data']).to_dict().items():
        for split_detail in split_details:
            name = split_detail['name'].lower()
            time = split_detail['time_ms']
            new_cols[name].append(time)
    df.drop('split_data', axis=1, inplace=True)
    for name, rows in new_cols.items():
        df[name] = rows
    del new_cols

    # Uppercase
    for col in [
        RaceFields.COUNTRY.value,
        RaceFields.GENDER.value,
    ]:
        df[col] = df[col].apply(lambda x: x.upper())
    country_data = load_country_details()
    df[RaceFields.COUNTRY.value] = df[RaceFields.COUNTRY.value].apply(
        lambda x: lookup_country_by_code(country_data=country_data, letter_code=x)[0]
    )

    # Normalize BIB and make it the index
    df[RaceFields.BIB.value] = df[RaceFields.BIB.value].astype(int)
    df.set_index(RaceFields.BIB.value, inplace=True)

    # Normalize timestamps
    for time_field in [
        RaceFields.TIME.value,
        RaceFields.TWENTY_FLOOR_TIME.value,
        RaceFields.SIXTY_FIVE_FLOOR_TIME.value
    ]:
        try:
            df[time_field] = pandas.to_timedelta(df[time_field], unit="milliseconds")
            if use_pretty:
                df[time_field] = df[time_field].apply(PrettyDuration)
        except ValueError as ve:
            raise ValueError(f'{time_field}={df[time_field]}', ve) from ve
    # df['finishtimestamp'] = BASE_RACE_DATETIME[default_year] + df[RaceFields.TIME.value]

    return df


def df_to_list_of_tuples(
        df: DataFrame,
        bibs: list[int] = None
) -> tuple | list[tuple]:
    """
    Take a DataFrame and return a more friendly structure to be used by a DataTable, for example.
    :param df DataFrame to convert
    :param bibs List of racing BIB to filter
    :return list of Tuple of rows, Tuple with columns
    """
    bib_as_column = df.reset_index(level=0, inplace=False)
    if not bibs:
        filtered = bib_as_column
    else:
        filtered = bib_as_column[bib_as_column[RaceFields.BIB.value].isin(bibs)]
    rows = []
    for _, r in filtered.iterrows():
        ind_row: list[Any] = []
        for col in FIELD_NAMES:
            ind_row.append(r[col])
        tpl = tuple(ind_row)
        rows.append(tpl)

    return tuple(FIELD_NAMES), rows


def series_to_list_of_tuples(series: Series) -> list[tuple]:
    """
    Helper series to list of tuples
    """
    dct = series.to_dict()
    rows = []
    for key, value in dct.items():
        rows.append(tuple([key, value]))
    return rows


def load_country_details(data_file: Path = None) -> TOMLDocument:
    """
    Args:
        data_file (Path): Path to data file in TOML format
    [ISOCountryCodes]
    name = "United States of America"
    alpha-2 = "US"
    alpha-3 = "USA"
    country-code = "840"
    iso_3166-2 = "ISO 3166-2:US"
    region = "Americas"
    sub-region = "Northern America"
    intermediate-region = ""
    region-code = "019"
    sub-region-code = "021"
    intermediate-region-code = ""

    """
    def_file = COUNTRY_DETAILS if data_file is None else data_file
    with open(def_file, encoding='utf-8') as f:
        data = tomlkit.load(fp=f)
        return data


class CountryColumns(Enum):
    """
    Country columns
    """
    NAME = "name"
    ALPHA_2 = "alpha-2"
    ALPHA_3 = "alpha-3"
    COUNTRY_CODE = "country-code"
    ISO_3166_2 = "iso_3166-2"
    REGION = "region"
    SUB_REGION = "sub-region"
    INTERMEDIATE_REGION = "intermediate-region"
    REGION_CODE = "region-code"
    SUB_REGION_CODE = "sub-region-code"
    INTERMEDIATE_REGION_CODE = "intermediate-region-code"


COUNTRY_COLUMNS = [country.value for country in CountryColumns]


def lookup_country_by_code(country_data: TOMLDocument, letter_code: str) -> tuple[str, TOMLDocument] | None:
    """
    Args:
        country_data (TOMLDocument): TOML document with country details
        letter_code: 2,3-letter ISO code used to filter country
    Returns:
        TOML document with country details, none if the lookup fails
    """
    if not isinstance(letter_code, str):
        raise ValueError(f"Invalid type for three letter country code: '{letter_code}'")
    if len(letter_code) == 3:
        for country_name, country_details in country_data.items():
            if letter_code == country_details['alpha-3']:
                return country_name, country_details
    elif len(letter_code) == 2:
        for country_name, country_details in country_data.items():
            if letter_code == country_details['alpha-2']:
                return country_name, country_details
    else:
        raise ValueError(f"Invalid letter country code: '{letter_code}'")
    return None


def get_times(df: DataFrame) -> DataFrame:
    """
    Get times from dataframe
    """
    return df.select_dtypes(include=['timedelta64', 'datetime64'])


def get_positions(df: DataFrame) -> DataFrame:
    """
    Get positions from dataframe
    """
    return df.select_dtypes(include=['int64'])


def get_categories(df: DataFrame) -> DataFrame:
    """
    Get categories from dataframe
    """
    return df.select_dtypes(include=['object'])


def beautify_race_times(time: datetime.timedelta) -> str:
    """
    Formatting for provided time
    """
    mm, ss = divmod(time.total_seconds(), 60)
    hh, mm = divmod(mm, 60)  # Ignore days part as the race doesn't last more than 24 hours
    return f"{int(hh)}:{int(mm)}:{int(ss)}"


class PrettyDuration:

    """
    The Empire State Building is a race that will never take more than 24 hours to complete.
    This class has special formatting to display the duration as M:SS
    """

    def __init__(self, duration: datetime.timedelta):
        self.duration = duration

    def __convert_timedelta__(self):
        seconds = self.duration.seconds
        minutes = (seconds % 3600) // 60
        seconds = seconds % 60
        return minutes, seconds

    def __str__(self):
        minutes, seconds = self.__convert_timedelta__()
        return f"{minutes}:{seconds}"

    def __lt__(self, other) -> bool:
        if other is not None:
            return self.duration.total_seconds() < other.duration.total_seconds()
        raise ValueError(f"Not a PrettyDuration: {other}")
