import re
from pathlib import Path
import logging as lg
from importlib.metadata import version

from mako.template import Template
from caseconverter import snakecase, pascalcase
import inspect
import gray_formatter

from prismarine.prisma_common import get_cluster


type ExtraImport = list[tuple[str, str]]


HEADER = '''\
# This file is generated by prismarine {version}. Do not modify it.

from prismarine.runtime.dynamo_crud import (
    _query,
    _get_item,
    _update,
    _put_item,
    _delete,
    _scan,
    _save,
    Model,
    without,
    DbNotFound
)

from typing import TypedDict, List, Literal, NotRequired  # noqa
from types import EllipsisType

from {access_module} import get_dynamo_access
'''

DYNAMO_ACCESS = '''\
# Create a DynamoDB resource
dynamo = get_dynamo_access()
'''


def build_client(
    cluster, base_dir: Path, runtime: str | None, access_module: str | None,
    extra_imports: ExtraImport = []
):
    '''
    Build a Prismarine client for a given cluster package.

    Args:
        cluster: The Cluster object.
        base_dir: The base directory of the project.
        runtime: The runtime package where the models are defined.
        access_module: The database access module to use.
        extra_imports: Extra imports to add to the client in format:
            [('path.to.module', 'ClassName')]
    '''
    r_base_dir = base_dir.resolve()
    this_dir = Path(__file__).resolve().parent
    template_file = Path(this_dir, 'model.mako')
    model_template = Template(template_file.read_text(encoding='utf-8'))
    module_body = ''
    imports = set()

    for name, model in sorted(cluster.models.items(), key=lambda x: x[0]):
        file_path = Path(inspect.getfile(model['cls']))
        relative_arr = file_path.relative_to(r_base_dir).as_posix().split('/')
        relative_arr[-1] = relative_arr[-1].replace('.py', '')
        imports.add(('.'.join(relative_arr), model['class_name']))
        source = inspect.getsource(model['cls'])
        source = re.sub(r'@.+?\n', '', source)

        source = re.sub(
            r'[a-zA-Z\d]+\(TypedDict\)', 'UpdateDTO(TypedDict, total=False)', source
        )

        module_body += '\n\n\n' + str(
            model_template.render(
                snake=snakecase,
                pascal=pascalcase,
                ModelClass=f'{name}Model',
                Model=model['class_name'],
                PartitionKey=model['main']['PK'],
                SortKey=model['main']['SK'],
                Indexes=[
                    {
                        'name': index_name,
                        'PartitionKey': index['PK'],
                        'SortKey': index['SK']
                    }
                    for index_name, index
                    in sorted(model['indexes'].items(), key=lambda x: x[0])
                ],
                TableName=model['table'],
                DtoLines=source.split('\n')
            )
        )

    # Additional exported classes (but not models)
    for export in cluster.exports.values():
        file_path = Path(inspect.getfile(export['cls']))
        relative_arr = file_path.relative_to(r_base_dir).as_posix().split('/')
        relative_arr[-1] = relative_arr[-1].replace('.py', '')
        imports.add(('.'.join(relative_arr), export['class_name']))

    access_module = access_module or 'prismarine.runtime.dynamo_default'
    header = HEADER.format(
        access_module=access_module,
        version=version('prismarine')
    )

    runtime_prefix = f'{runtime}.' if runtime else ''

    header += '# Model classes imports\n'
    for i in sorted(imports, key=lambda x: x[1]):
        header += f'from {runtime_prefix}{i[0]} import {i[1]}\n'

    if extra_imports:
        header += '# Extra imports\n'
        for i in sorted(extra_imports, key=lambda x: x[1]):
            header += f'from {i[0]} import {i[1]}\n'

    content = header + DYNAMO_ACCESS + module_body
    content = gray_formatter.fix_content(content)
    return content


def write_client(content, base_dir: Path, cluster_package: str):
    client_path = Path(base_dir, cluster_package, 'prismarine_client.py')
    lg.info('Writing ' + str(client_path))
    client_path.write_text(content)


def generate_client(
    base_dir: Path,
    cluster_package: str,
    *,
    runtime: str | None,
    access_module: str | None,
    extra_imports: ExtraImport = []
):
    '''
    Generate a Prismarine client for a given cluster package.

    Args:
        base_dir: The base directory of the project.
        cluster_package: The name of the cluster package.
        runtime: The runtime package where the models are defined.
        access_module: The database access module to use.
        extra_imports: Extra imports to add to the client in format:
            [('path.to.module', 'ClassName')]
    '''
    cluster = get_cluster(base_dir, cluster_package)

    content = build_client(
        cluster, base_dir, runtime, access_module,
        extra_imports=extra_imports
    )

    write_client(content, base_dir, cluster_package)
