"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile || (exports.DependenciesFile = DependenciesFile = {}));
class Packaging {
    /**
     * Standard packaging with `pip`.
     */
    static withPip() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIP,
        });
    }
    /**
     * Packaging with `pipenv`.
     */
    static withPipenv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIPENV,
            // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
            // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
            exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv requirements > ${DependenciesFile.PIP} && rm -rf .venv`,
        });
    }
    /**
     * Packaging with `poetry`.
     */
    static withPoetry(props) {
        return new Packaging({
            dependenciesFile: DependenciesFile.POETRY,
            // Export dependencies with credentials available in the bundling image.
            exportCommand: [
                'poetry', 'export',
                ...props?.poetryIncludeHashes ? [] : ['--without-hashes'],
                ...props?.poetryWithoutUrls ? ['--without-urls'] : [],
                '--with-credentials',
                '--format', DependenciesFile.PIP,
                '--output', DependenciesFile.PIP,
            ].join(' '),
        });
    }
    /**
     * No dependencies or packaging.
     */
    static withNoPackaging() {
        return new Packaging({ dependenciesFile: DependenciesFile.NONE });
    }
    static fromEntry(entry, poetryIncludeHashes, poetryWithoutUrls) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return this.withPipenv();
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return this.withPoetry({ poetryIncludeHashes, poetryWithoutUrls });
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return this.withPip();
        }
        else {
            return this.withNoPackaging();
        }
    }
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
}
exports.Packaging = Packaging;
//# sourceMappingURL=data:application/json;base64,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