r"""
**VICE Developer's Documentation**

This file implements a simple generator object with which many unit tests can
be systematically generated by subclassing and adding a __call__ function with
a @unittest decorator.
"""

from .._globals import _VERSION_ERROR_
import sys
if sys.version_info[:2] == (2, 7):
	strcomp = basestring
elif sys.version_info[:2] >= (3, 5):
	strcomp = str
else:
	_VERSION_ERROR_()


class generator:

	r"""
	**VICE Developer's Documentation**

	A simple object which can be used to systematically generate unit test
	objects. To do so, simply subclass this object and add a ``__call__``
	function with a ``@unittest`` decorator. The ``__call__`` function should
	be written the same as any other function with a ``@unittest`` decorator.

	**Signature**: generator(msg, **kwargs)

	Attributes & Parameters
	-----------------------
	msg : ``str``
		The ``name`` attribute of the resulting ``unittest`` object.
	kwargs : ``dict``
		Keyword arguments to be passed along to the function being tested.

	Example
	-------
	class unittest_generator(generator):
		@unittest
		def __call__(self):
			def test():
				result = True
				# Run the test, switching result to False if some condition
				# is not met.
				return result
			return [self.msg, test]
	"""

	def __init__(self, msg, **kwargs):
		self.msg = msg
		self.kwargs = kwargs

	@property
	def msg(self):
		r"""
		**VICE** Developer's Documentation

		Type : ``str``

		The message to print with the unit test (i.e. it's ``name`` attribute).
		"""
		return self._msg

	@msg.setter
	def msg(self, value):
		if isinstance(value, strcomp):
			self._msg = value
		else:
			raise TypeError("Attribute 'msg' must be of type str. Got: %s" % (
				type(value)))

	@property
	def kwargs(self):
		r"""
		**VICE Developer's Documentation**

		Type : ``dict``

		Keyword arguments to be passed along to the function being tested,
		though at its core this is simply a container that can be used to
		store additional data.
		"""
		return self._kwargs

	@kwargs.setter
	def kwargs(self, value):
		if isinstance(value, dict):
			self._kwargs = {}
			for key in value.keys(): self._kwargs[key] = value[key]
		else:
			raise TypeError("""\
Attribute 'kwargs' must be of type dict. Got: %s""" % (type(value)))

