from enum import Enum
from typing import Sequence
from functools import total_ordering

@total_ordering
class Protocol(Enum):
    # Keep protocols in order of preference.
    TLS1_3 = b"\x03\x04"
    TLS1_2 = b"\x03\x03"
    TLS1_1 = b"\x03\x02"
    TLS1_0 = b"\x03\x01"
    SSLv3 = b"\x03\x00"

    def __repr__(self):
        return self.name
    def __lt__(self, other):
        if self.__class__ != other.__class__:
            return NotImplemented
        return self.value < other.value

class RecordType(Enum):
    INVALID = b'\x00' # Unused in this script.
    CHANGE_CIPHER_SPEC = b'\x14' # Unused in this script.
    ALERT = b'\x15'
    HANDSHAKE = b'\x16'
    APPLICATION_DATA = b'\x17' # Unused in this script.

class HandshakeType(Enum):
    client_hello = b'\x01'
    server_hello = b'\x02'
    new_session_ticket = b'\x04'
    end_of_early_data = b'\x05'
    encrypted_extensions = b'\x08'
    certificate = b'\x0B'
    server_key_exchange = b'\x0C'
    certificate_request = b'\x0D'
    server_hello_done = b'\x0E'
    certificate_verify = b'\x0F'
    finished = b'\x14'
    certificate_status = b'\x16'
    key_update = b'\x18'
    message_hash = b'\x19'

class CompressionMethod(Enum):
    NULL = b'\x00'
    DEFLATE = b'\x01'

class PskKeyExchangeMode(Enum):
    psk_ke = b'\x00'
    psk_dhe_ke = b'\x01'

class Group(Enum):
    def __new__(cls, value, *rest, **kwds):
        obj = object.__new__(cls)
        obj._value_ = value
        return obj
    # Annotate each group with whether it's a PQ group.
    def __init__(self, _: bytes, is_pq: bool = False):
        self.is_pq = is_pq
    def __repr__(self):
        return self.name
    
    sect163k1 = b'\x00\x01'
    sect163r1 = b'\x00\x02'
    sect163r2 = b'\x00\x03'
    sect193r1 = b'\x00\x04'
    sect193r2 = b'\x00\x05'
    sect233k1 = b'\x00\x06'
    sect233r1 = b'\x00\x07'
    sect239k1 = b'\x00\x08'
    sect283k1 = b'\x00\x09'
    sect283r1 = b'\x00\x0a'
    sect409k1 = b'\x00\x0b'
    sect409r1 = b'\x00\x0c'
    sect571k1 = b'\x00\x0d'
    sect571r1 = b'\x00\x0e'
    secp160k1 = b'\x00\x0f'
    secp160r1 = b'\x00\x10'
    secp160r2 = b'\x00\x11'
    secp192k1 = b'\x00\x12'
    secp192r1 = b'\x00\x13'
    secp224k1 = b'\x00\x14'
    secp224r1 = b'\x00\x15'
    secp256k1 = b'\x00\x16'
    secp256r1 = b'\x00\x17'
    secp384r1 = b'\x00\x18'
    secp521r1 = b'\x00\x19'
    brainpoolP256r1 = b'\x00\x1a'
    brainpoolP384r1 = b'\x00\x1b'
    brainpoolP512r1 = b'\x00\x1c'
    x25519 = b'\x00\x1d'
    x448 = b'\x00\x1e'
    brainpoolP256r1tls13 = b'\x00\x1f'
    brainpoolP384r1tls13 = b'\x00\x20'
    brainpoolP512r1tls13 = b'\x00\x21'
    GC256A = b'\x00\x22'
    GC256B = b'\x00\x23'
    GC256C = b'\x00\x24'
    GC256D = b'\x00\x25'
    GC512A = b'\x00\x26'
    GC512B = b'\x00\x27'
    GC512C = b'\x00\x28'
    curveSM2 = b'\x00\x29'
    ffdhe2048 = b'\x01\x00'
    ffdhe3072 = b'\x01\x01'
    ffdhe4096 = b'\x01\x02'
    ffdhe6144 = b'\x01\x03'
    ffdhe8192 = b'\x01\x04'
    arbitrary_explicit_prime_curves = b'\xff\x01'
    arbitrary_explicit_char2_curves = b'\xff\x02'

    # Somewhat common post-quantum groups, not yet standardized:
    X25519Kyber768Draft00 = b'\x63\x99', True
    X25519Kyber768Draft00_obsolete = b'\xfe\x31', True
    X25519Kyber512Draft00 = b'\xfe\x30', True
    SecP256r1Kyber768Draft00 = b'\x63\x9a', True

    # Long list of unusual post-quantum groups from liboqs:
    # https://github.com/open-quantum-safe/oqs-provider/blob/main/ALGORITHMS.md?plain=1#L13
    frodo640aes = b'\x02\x00', True
    p256_frodo640aes = b'\x2F\x00', True
    x25519_frodo640aes = b'\x2F\x80', True
    frodo640shake = b'\x02\x01', True
    p256_frodo640shake = b'\x2F\x01', True
    x25519_frodo640shake = b'\x2F\x81', True
    frodo976aes = b'\x02\x02', True
    p384_frodo976aes = b'\x2F\x02', True
    x448_frodo976aes = b'\x2F\x82', True
    frodo976shake = b'\x02\x03', True
    p384_frodo976shake = b'\x2F\x03', True
    x448_frodo976shake = b'\x2F\x83', True
    frodo1344aes = b'\x02\x04', True
    p521_frodo1344aes = b'\x2F\x04', True
    frodo1344shake = b'\x02\x05', True
    p521_frodo1344shake = b'\x2F\x05', True
    kyber512 = b'\x02\x3A', True
    p256_kyber512 = b'\x2F\x3A', True
    x25519_kyber512 = b'\x2F\x39', True
    kyber768 = b'\x02\x3C', True
    p384_kyber768 = b'\x2F\x3C', True
    x448_kyber768 = b'\x2F\x90', True
    kyber1024 = b'\x02\x3D', True
    p521_kyber1024 = b'\x2F\x3D', True
    bikel1 = b'\x02\x41', True
    p256_bikel1 = b'\x2F\x41', True
    x25519_bikel1 = b'\x2F\xAE', True
    bikel3 = b'\x02\x42', True
    p384_bikel3 = b'\x2F\x42', True
    x448_bikel3 = b'\x2F\xAF', True
    bikel5 = b'\x02\x43', True
    p521_bikel5 = b'\x2F\x43', True
    hqc128 = b'\x02\x2C', True
    p256_hqc128 = b'\x2F\x2C', True
    x25519_hqc128 = b'\x2F\xAC', True
    hqc192 = b'\x02\x2D', True
    p384_hqc192 = b'\x2F\x2D', True
    x448_hqc192 = b'\x2F\xAD', True
    hqc256 = b'\x02\x2E', True
    p521_hqc256 = b'\x2F\x2E', True
    dilithium2 = b'\xfe\xa0', True
    p256_dilithium2 = b'\xfe\xa1', True
    rsa3072_dilithium2 = b'\xfe\xa2', True
    dilithium3 = b'\xfe\xa3', True
    p384_dilithium3 = b'\xfe\xa4', True
    dilithium5 = b'\xfe\xa5', True
    p521_dilithium5 = b'\xfe\xa6', True
    falcon512 = b'\xfe\xae', True
    p256_falcon512 = b'\xfe\xaf', True
    rsa3072_falcon512 = b'\xfe\xb0', True
    falcon1024 = b'\xfe\xb1', True
    p521_falcon1024 = b'\xfe\xb2', True
    sphincssha2128fsimple = b'\xfe\xb3', True
    p256_sphincssha2128fsimple = b'\xfe\xb4', True
    rsa3072_sphincssha2128fsimple = b'\xfe\xb5', True
    sphincssha2128ssimple = b'\xfe\xb6', True
    p256_sphincssha2128ssimple = b'\xfe\xb7', True
    rsa3072_sphincssha2128ssimple = b'\xfe\xb8', True
    sphincssha2192fsimple = b'\xfe\xb9', True
    p384_sphincssha2192fsimple = b'\xfe\xba', True
    sphincssha2192ssimple = b'\xfe\xbb', True
    p384_sphincssha2192ssimple = b'\xfe\xbc', True
    sphincssha2256fsimple = b'\xfe\xbd', True
    p521_sphincssha2256fsimple = b'\xfe\xbe', True
    sphincssha2256ssimple = b'\xfe\xc0', True
    p521_sphincssha2256ssimple = b'\xfe\xc1', True
    sphincsshake128fsimple = b'\xfe\xc2', True
    p256_sphincsshake128fsimple = b'\xfe\xc3', True
    rsa3072_sphincsshake128fsimple = b'\xfe\xc4', True
    sphincsshake128ssimple = b'\xfe\xc5', True
    p256_sphincsshake128ssimple = b'\xfe\xc6', True
    rsa3072_sphincsshake128ssimple = b'\xfe\xc7', True
    sphincsshake192fsimple = b'\xfe\xc8', True
    p384_sphincsshake192fsimple = b'\xfe\xc9', True
    sphincsshake192ssimple = b'\xfe\xca', True
    p384_sphincsshake192ssimple = b'\xfe\xcb', True
    sphincsshake256fsimple = b'\xfe\xcc', True
    p521_sphincsshake256fsimple = b'\xfe\xcd', True
    sphincsshake256ssimple = b'\xfe\xce', True
    p521_sphincsshake256ssimple = b'\xfe\xcf', True

class CipherSuite(Enum):
    def __repr__(self):
        return self.name
    def __new__(cls, value, *rest, **kwds):
        obj = object.__new__(cls)
        obj._value_ = value
        return obj
    # Annotate each cipher suite with the protocols it's supported at.
    # Default to all but TLS 1.3, because that's the most common.
    def __init__(self, _: bytes, protocols: Sequence[Protocol] = (Protocol.SSLv3, Protocol.TLS1_0, Protocol.TLS1_1, Protocol.TLS1_2)):
        self.protocols = protocols

    # Pseudo cipher suite, not actually picked.
    #TLS_EMPTY_RENEGOTIATION_INFO_SCSV = b"\x00\xff"

    # TLS 1.3 cipher suites.
    TLS_AES_128_GCM_SHA256 = b"\x13\x01", (Protocol.TLS1_3,)
    TLS_AES_256_GCM_SHA384 = b"\x13\x02", (Protocol.TLS1_3,)
    TLS_CHACHA20_POLY1305_SHA256 = b"\x13\x03", (Protocol.TLS1_3,)
    TLS_AES_128_CCM_SHA256 = b"\x13\x04", (Protocol.TLS1_3,)
    TLS_AES_128_CCM_8_SHA256 = b"\x13\x05", (Protocol.TLS1_3,)

    # Cipher suite that had its number reassigned.
    OLD_TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256  = b'\xcc\x13'
    
    # Cipher suites adapted from IANA assignments:
    # https://www.iana.org/assignments/tls-parameters/tls-parameters.xhtml#tls-parameters-4
    TLS_AEGIS_128L_SHA256 = b'\x13\x07' # [draft-irtf-cfrg-aegis-aead-00]
    TLS_AEGIS_256_SHA384 = b'\x13\x06' # [draft-irtf-cfrg-aegis-aead-00]
    TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA = b'\x00\x19' # [RFC4346]
    TLS_DH_anon_EXPORT_WITH_RC4_40_MD5 = b'\x00\x17' # [RFC4346][RFC6347]
    TLS_DH_anon_WITH_3DES_EDE_CBC_SHA = b'\x00\x1B' # [RFC5246]
    TLS_DH_anon_WITH_AES_128_CBC_SHA = b'\x00\x34' # [RFC5246]
    TLS_DH_anon_WITH_AES_128_CBC_SHA256 = b'\x00\x6C' # [RFC5246]
    TLS_DH_anon_WITH_AES_128_GCM_SHA256 = b'\x00\xA6' # [RFC5288]
    TLS_DH_anon_WITH_AES_256_CBC_SHA = b'\x00\x3A' # [RFC5246]
    TLS_DH_anon_WITH_AES_256_CBC_SHA256 = b'\x00\x6D' # [RFC5246]
    TLS_DH_anon_WITH_AES_256_GCM_SHA384 = b'\x00\xA7' # [RFC5288]
    TLS_DH_anon_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x46' # [RFC6209]
    TLS_DH_anon_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x5A' # [RFC6209]
    TLS_DH_anon_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x47' # [RFC6209]
    TLS_DH_anon_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x5B' # [RFC6209]
    TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA = b'\x00\x46' # [RFC5932]
    TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256 = b'\x00\xBF' # [RFC5932]
    TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x84' # [RFC6367]
    TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA = b'\x00\x89' # [RFC5932]
    TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256 = b'\x00\xC5' # [RFC5932]
    TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x85' # [RFC6367]
    TLS_DH_anon_WITH_DES_CBC_SHA = b'\x00\x1A' # [RFC8996]
    TLS_DH_anon_WITH_RC4_128_MD5 = b'\x00\x18' # [RFC5246][RFC6347]
    TLS_DH_anon_WITH_SEED_CBC_SHA = b'\x00\x9B' # [RFC4162]
    TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA = b'\x00\x0B' # [RFC4346]
    TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA = b'\x00\x0D' # [RFC5246]
    TLS_DH_DSS_WITH_AES_128_CBC_SHA = b'\x00\x30' # [RFC5246]
    TLS_DH_DSS_WITH_AES_128_CBC_SHA256 = b'\x00\x3E' # [RFC5246]
    TLS_DH_DSS_WITH_AES_128_GCM_SHA256 = b'\x00\xA4' # [RFC5288]
    TLS_DH_DSS_WITH_AES_256_CBC_SHA = b'\x00\x36' # [RFC5246]
    TLS_DH_DSS_WITH_AES_256_CBC_SHA256 = b'\x00\x68' # [RFC5246]
    TLS_DH_DSS_WITH_AES_256_GCM_SHA384 = b'\x00\xA5' # [RFC5288]
    TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x3E' # [RFC6209]
    TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x58' # [RFC6209]
    TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x3F' # [RFC6209]
    TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x59' # [RFC6209]
    TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA = b'\x00\x42' # [RFC5932]
    TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256 = b'\x00\xBB' # [RFC5932]
    TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x82' # [RFC6367]
    TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA = b'\x00\x85' # [RFC5932]
    TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256 = b'\x00\xC1' # [RFC5932]
    TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x83' # [RFC6367]
    TLS_DH_DSS_WITH_DES_CBC_SHA = b'\x00\x0C' # [RFC8996]
    TLS_DH_DSS_WITH_SEED_CBC_SHA = b'\x00\x97' # [RFC4162]
    TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA = b'\x00\x0E' # [RFC4346]
    TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA = b'\x00\x10' # [RFC5246]
    TLS_DH_RSA_WITH_AES_128_CBC_SHA = b'\x00\x31' # [RFC5246]
    TLS_DH_RSA_WITH_AES_128_CBC_SHA256 = b'\x00\x3F' # [RFC5246]
    TLS_DH_RSA_WITH_AES_128_GCM_SHA256 = b'\x00\xA0' # [RFC5288]
    TLS_DH_RSA_WITH_AES_256_CBC_SHA = b'\x00\x37' # [RFC5246]
    TLS_DH_RSA_WITH_AES_256_CBC_SHA256 = b'\x00\x69' # [RFC5246]
    TLS_DH_RSA_WITH_AES_256_GCM_SHA384 = b'\x00\xA1' # [RFC5288]
    TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x40' # [RFC6209]
    TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x54' # [RFC6209]
    TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x41' # [RFC6209]
    TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x55' # [RFC6209]
    TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA = b'\x00\x43' # [RFC5932]
    TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = b'\x00\xBC' # [RFC5932]
    TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x7E' # [RFC6367]
    TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA = b'\x00\x86' # [RFC5932]
    TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256 = b'\x00\xC2' # [RFC5932]
    TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x7F' # [RFC6367]
    TLS_DH_RSA_WITH_DES_CBC_SHA = b'\x00\x0F' # [RFC8996]
    TLS_DH_RSA_WITH_SEED_CBC_SHA = b'\x00\x98' # [RFC4162]
    TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA = b'\x00\x11' # [RFC4346]
    TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA = b'\x00\x13' # [RFC5246]
    TLS_DHE_DSS_WITH_AES_128_CBC_SHA = b'\x00\x32' # [RFC5246]
    TLS_DHE_DSS_WITH_AES_128_CBC_SHA256 = b'\x00\x40' # [RFC5246]
    TLS_DHE_DSS_WITH_AES_128_GCM_SHA256 = b'\x00\xA2' # [RFC5288]
    TLS_DHE_DSS_WITH_AES_256_CBC_SHA = b'\x00\x38' # [RFC5246]
    TLS_DHE_DSS_WITH_AES_256_CBC_SHA256 = b'\x00\x6A' # [RFC5246]
    TLS_DHE_DSS_WITH_AES_256_GCM_SHA384 = b'\x00\xA3' # [RFC5288]
    TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x42' # [RFC6209]
    TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x56' # [RFC6209]
    TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x43' # [RFC6209]
    TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x57' # [RFC6209]
    TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA = b'\x00\x44' # [RFC5932]
    TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256 = b'\x00\xBD' # [RFC5932]
    TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x80' # [RFC6367]
    TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA = b'\x00\x87' # [RFC5932]
    TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256 = b'\x00\xC3' # [RFC5932]
    TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x81' # [RFC6367]
    TLS_DHE_DSS_WITH_DES_CBC_SHA = b'\x00\x12' # [RFC8996]
    TLS_DHE_DSS_WITH_SEED_CBC_SHA = b'\x00\x99' # [RFC4162]
    TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA = b'\x00\x8F' # [RFC4279]
    TLS_DHE_PSK_WITH_AES_128_CBC_SHA = b'\x00\x90' # [RFC4279]
    TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 = b'\x00\xB2' # [RFC5487]
    TLS_DHE_PSK_WITH_AES_128_CCM = b'\xC0\xA6' # [RFC6655]
    TLS_DHE_PSK_WITH_AES_128_GCM_SHA256 = b'\x00\xAA' # [RFC5487]
    TLS_DHE_PSK_WITH_AES_256_CBC_SHA = b'\x00\x91' # [RFC4279]
    TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 = b'\x00\xB3' # [RFC5487]
    TLS_DHE_PSK_WITH_AES_256_CCM = b'\xC0\xA7' # [RFC6655]
    TLS_DHE_PSK_WITH_AES_256_GCM_SHA384 = b'\x00\xAB' # [RFC5487]
    TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x66' # [RFC6209]
    TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x6C' # [RFC6209]
    TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x67' # [RFC6209]
    TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x6D' # [RFC6209]
    TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 = b'\xC0\x96' # [RFC6367]
    TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x90' # [RFC6367]
    TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 = b'\xC0\x97' # [RFC6367]
    TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x91' # [RFC6367]
    TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256 = b'\xCC\xAD' # [RFC7905]
    TLS_DHE_PSK_WITH_NULL_SHA = b'\x00\x2D' # [RFC4785]
    TLS_DHE_PSK_WITH_NULL_SHA256 = b'\x00\xB4' # [RFC5487]
    TLS_DHE_PSK_WITH_NULL_SHA384 = b'\x00\xB5' # [RFC5487]
    TLS_DHE_PSK_WITH_RC4_128_SHA = b'\x00\x8E' # [RFC4279][RFC6347]
    TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA = b'\x00\x14' # [RFC4346]
    TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA = b'\x00\x16' # [RFC5246]
    TLS_DHE_RSA_WITH_AES_128_CBC_SHA = b'\x00\x33' # [RFC5246]
    TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 = b'\x00\x67' # [RFC5246]
    TLS_DHE_RSA_WITH_AES_128_CCM = b'\xC0\x9E' # [RFC6655]
    TLS_DHE_RSA_WITH_AES_128_CCM_8 = b'\xC0\xA2' # [RFC6655]
    TLS_DHE_RSA_WITH_AES_128_GCM_SHA256 = b'\x00\x9E' # [RFC5288]
    TLS_DHE_RSA_WITH_AES_256_CBC_SHA = b'\x00\x39' # [RFC5246]
    TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 = b'\x00\x6B' # [RFC5246]
    TLS_DHE_RSA_WITH_AES_256_CCM = b'\xC0\x9F' # [RFC6655]
    TLS_DHE_RSA_WITH_AES_256_CCM_8 = b'\xC0\xA3' # [RFC6655]
    TLS_DHE_RSA_WITH_AES_256_GCM_SHA384 = b'\x00\x9F' # [RFC5288]
    TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x44' # [RFC6209]
    TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x52' # [RFC6209]
    TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x45' # [RFC6209]
    TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x53' # [RFC6209]
    TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA = b'\x00\x45' # [RFC5932]
    TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = b'\x00\xBE' # [RFC5932]
    TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x7C' # [RFC6367]
    TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA = b'\x00\x88' # [RFC5932]
    TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 = b'\x00\xC4' # [RFC5932]
    TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x7D' # [RFC6367]
    TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256 = b'\xCC\xAA' # [RFC7905]
    TLS_DHE_RSA_WITH_DES_CBC_SHA = b'\x00\x15' # [RFC8996]
    TLS_DHE_RSA_WITH_SEED_CBC_SHA = b'\x00\x9A' # [RFC4162]
    TLS_ECCPWD_WITH_AES_128_CCM_SHA256 = b'\xC0\xB2' # [RFC8492]
    TLS_ECCPWD_WITH_AES_128_GCM_SHA256 = b'\xC0\xB0' # [RFC8492]
    TLS_ECCPWD_WITH_AES_256_CCM_SHA384 = b'\xC0\xB3' # [RFC8492]
    TLS_ECCPWD_WITH_AES_256_GCM_SHA384 = b'\xC0\xB1' # [RFC8492]
    TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA = b'\xC0\x17' # [RFC8422]
    TLS_ECDH_anon_WITH_AES_128_CBC_SHA = b'\xC0\x18' # [RFC8422]
    TLS_ECDH_anon_WITH_AES_256_CBC_SHA = b'\xC0\x19' # [RFC8422]
    TLS_ECDH_anon_WITH_NULL_SHA = b'\xC0\x15' # [RFC8422]
    TLS_ECDH_anon_WITH_RC4_128_SHA = b'\xC0\x16' # [RFC8422][RFC6347]
    TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA = b'\xC0\x03' # [RFC8422]
    TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA = b'\xC0\x04' # [RFC8422]
    TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 = b'\xC0\x25' # [RFC5289]
    TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256 = b'\xC0\x2D' # [RFC5289]
    TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA = b'\xC0\x05' # [RFC8422]
    TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 = b'\xC0\x26' # [RFC5289]
    TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384 = b'\xC0\x2E' # [RFC5289]
    TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x4A' # [RFC6209]
    TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x5E' # [RFC6209]
    TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x4B' # [RFC6209]
    TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x5F' # [RFC6209]
    TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 = b'\xC0\x74' # [RFC6367]
    TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x88' # [RFC6367]
    TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 = b'\xC0\x75' # [RFC6367]
    TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x89' # [RFC6367]
    TLS_ECDH_ECDSA_WITH_NULL_SHA = b'\xC0\x01' # [RFC8422]
    TLS_ECDH_ECDSA_WITH_RC4_128_SHA = b'\xC0\x02' # [RFC8422][RFC6347]
    TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA = b'\xC0\x0D' # [RFC8422]
    TLS_ECDH_RSA_WITH_AES_128_CBC_SHA = b'\xC0\x0E' # [RFC8422]
    TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256 = b'\xC0\x29' # [RFC5289]
    TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256 = b'\xC0\x31' # [RFC5289]
    TLS_ECDH_RSA_WITH_AES_256_CBC_SHA = b'\xC0\x0F' # [RFC8422]
    TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384 = b'\xC0\x2A' # [RFC5289]
    TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384 = b'\xC0\x32' # [RFC5289]
    TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x4E' # [RFC6209]
    TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x62' # [RFC6209]
    TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x4F' # [RFC6209]
    TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x63' # [RFC6209]
    TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = b'\xC0\x78' # [RFC6367]
    TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x8C' # [RFC6367]
    TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384 = b'\xC0\x79' # [RFC6367]
    TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x8D' # [RFC6367]
    TLS_ECDH_RSA_WITH_NULL_SHA = b'\xC0\x0B' # [RFC8422]
    TLS_ECDH_RSA_WITH_RC4_128_SHA = b'\xC0\x0C' # [RFC8422][RFC6347]
    TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA = b'\xC0\x08' # [RFC8422]
    TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA = b'\xC0\x09' # [RFC8422]
    TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256 = b'\xC0\x23' # [RFC5289]
    TLS_ECDHE_ECDSA_WITH_AES_128_CCM = b'\xC0\xAC' # [RFC7251]
    TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8 = b'\xC0\xAE' # [RFC7251]
    TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 = b'\xC0\x2B' # [RFC5289]
    TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA = b'\xC0\x0A' # [RFC8422]
    TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384 = b'\xC0\x24' # [RFC5289]
    TLS_ECDHE_ECDSA_WITH_AES_256_CCM = b'\xC0\xAD' # [RFC7251]
    TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8 = b'\xC0\xAF' # [RFC7251]
    TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 = b'\xC0\x2C' # [RFC5289]
    TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x48' # [RFC6209]
    TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x5C' # [RFC6209]
    TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x49' # [RFC6209]
    TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x5D' # [RFC6209]
    TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 = b'\xC0\x72' # [RFC6367]
    TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x86' # [RFC6367]
    TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 = b'\xC0\x73' # [RFC6367]
    TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x87' # [RFC6367]
    TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 = b'\xCC\xA9' # [RFC7905]
    TLS_ECDHE_ECDSA_WITH_NULL_SHA = b'\xC0\x06' # [RFC8422]
    TLS_ECDHE_ECDSA_WITH_RC4_128_SHA = b'\xC0\x07' # [RFC8422][RFC6347]
    TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA = b'\xC0\x34' # [RFC5489]
    TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA = b'\xC0\x35' # [RFC5489]
    TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256 = b'\xC0\x37' # [RFC5489]
    TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256 = b'\xD0\x03' # [RFC8442]
    TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256 = b'\xD0\x05' # [RFC8442]
    TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256 = b'\xD0\x01' # [RFC8442]
    TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA = b'\xC0\x36' # [RFC5489]
    TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384 = b'\xC0\x38' # [RFC5489]
    TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384 = b'\xD0\x02' # [RFC8442]
    TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x70' # [RFC6209]
    TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x71' # [RFC6209]
    TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 = b'\xC0\x9A' # [RFC6367]
    TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 = b'\xC0\x9B' # [RFC6367]
    TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256 = b'\xCC\xAC' # [RFC7905]
    TLS_ECDHE_PSK_WITH_NULL_SHA = b'\xC0\x39' # [RFC5489]
    TLS_ECDHE_PSK_WITH_NULL_SHA256 = b'\xC0\x3A' # [RFC5489]
    TLS_ECDHE_PSK_WITH_NULL_SHA384 = b'\xC0\x3B' # [RFC5489]
    TLS_ECDHE_PSK_WITH_RC4_128_SHA = b'\xC0\x33' # [RFC5489][RFC6347]
    TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA = b'\xC0\x12' # [RFC8422]
    TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA = b'\xC0\x13' # [RFC8422]
    TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256 = b'\xC0\x27' # [RFC5289]
    TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256 = b'\xC0\x2F' # [RFC5289]
    TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA = b'\xC0\x14' # [RFC8422]
    TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384 = b'\xC0\x28' # [RFC5289]
    TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384 = b'\xC0\x30' # [RFC5289]
    TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x4C' # [RFC6209]
    TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x60' # [RFC6209]
    TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x4D' # [RFC6209]
    TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x61' # [RFC6209]
    TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = b'\xC0\x76' # [RFC6367]
    TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x8A' # [RFC6367]
    TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384 = b'\xC0\x77' # [RFC6367]
    TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x8B' # [RFC6367]
    TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256 = b'\xCC\xA8' # [RFC7905]
    TLS_ECDHE_RSA_WITH_NULL_SHA = b'\xC0\x10' # [RFC8422]
    TLS_ECDHE_RSA_WITH_RC4_128_SHA = b'\xC0\x11' # [RFC8422][RFC6347]
    TLS_GOSTR341112_256_WITH_28147_CNT_IMIT = b'\xC1\x02' # [RFC9189]
    TLS_GOSTR341112_256_WITH_KUZNYECHIK_CTR_OMAC = b'\xC1\x00' # [RFC9189]
    TLS_GOSTR341112_256_WITH_KUZNYECHIK_MGM_L = b'\xC1\x03' # [RFC9367]
    TLS_GOSTR341112_256_WITH_KUZNYECHIK_MGM_S = b'\xC1\x05' # [RFC9367]
    TLS_GOSTR341112_256_WITH_MAGMA_CTR_OMAC = b'\xC1\x01' # [RFC9189]
    TLS_GOSTR341112_256_WITH_MAGMA_MGM_L = b'\xC1\x04' # [RFC9367]
    TLS_GOSTR341112_256_WITH_MAGMA_MGM_S = b'\xC1\x06' # [RFC9367]
    TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5 = b'\x00\x29' # [RFC2712]
    TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA = b'\x00\x26' # [RFC2712]
    TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5 = b'\x00\x2A' # [RFC2712]
    TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA = b'\x00\x27' # [RFC2712]
    TLS_KRB5_EXPORT_WITH_RC4_40_MD5 = b'\x00\x2B' # [RFC2712][RFC6347]
    TLS_KRB5_EXPORT_WITH_RC4_40_SHA = b'\x00\x28' # [RFC2712][RFC6347]
    TLS_KRB5_WITH_3DES_EDE_CBC_MD5 = b'\x00\x23' # [RFC2712]
    TLS_KRB5_WITH_3DES_EDE_CBC_SHA = b'\x00\x1F' # [RFC2712]
    TLS_KRB5_WITH_DES_CBC_MD5 = b'\x00\x22' # [RFC2712]
    TLS_KRB5_WITH_DES_CBC_SHA = b'\x00\x1E' # [RFC2712]
    TLS_KRB5_WITH_IDEA_CBC_MD5 = b'\x00\x25' # [RFC2712]
    TLS_KRB5_WITH_IDEA_CBC_SHA = b'\x00\x21' # [RFC2712]
    TLS_KRB5_WITH_RC4_128_MD5 = b'\x00\x24' # [RFC2712][RFC6347]
    TLS_KRB5_WITH_RC4_128_SHA = b'\x00\x20' # [RFC2712][RFC6347]
    TLS_NULL_WITH_NULL_NULL = b'\x00\x00' # [RFC5246]
    TLS_PSK_DHE_WITH_AES_128_CCM_8 = b'\xC0\xAA' # [RFC6655]
    TLS_PSK_DHE_WITH_AES_256_CCM_8 = b'\xC0\xAB' # [RFC6655]
    TLS_PSK_WITH_3DES_EDE_CBC_SHA = b'\x00\x8B' # [RFC4279]
    TLS_PSK_WITH_AES_128_CBC_SHA = b'\x00\x8C' # [RFC4279]
    TLS_PSK_WITH_AES_128_CBC_SHA256 = b'\x00\xAE' # [RFC5487]
    TLS_PSK_WITH_AES_128_CCM = b'\xC0\xA4' # [RFC6655]
    TLS_PSK_WITH_AES_128_CCM_8 = b'\xC0\xA8' # [RFC6655]
    TLS_PSK_WITH_AES_128_GCM_SHA256 = b'\x00\xA8' # [RFC5487]
    TLS_PSK_WITH_AES_256_CBC_SHA = b'\x00\x8D' # [RFC4279]
    TLS_PSK_WITH_AES_256_CBC_SHA384 = b'\x00\xAF' # [RFC5487]
    TLS_PSK_WITH_AES_256_CCM = b'\xC0\xA5' # [RFC6655]
    TLS_PSK_WITH_AES_256_CCM_8 = b'\xC0\xA9' # [RFC6655]
    TLS_PSK_WITH_AES_256_GCM_SHA384 = b'\x00\xA9' # [RFC5487]
    TLS_PSK_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x64' # [RFC6209]
    TLS_PSK_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x6A' # [RFC6209]
    TLS_PSK_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x65' # [RFC6209]
    TLS_PSK_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x6B' # [RFC6209]
    TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256 = b'\xC0\x94' # [RFC6367]
    TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x8E' # [RFC6367]
    TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384 = b'\xC0\x95' # [RFC6367]
    TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x8F' # [RFC6367]
    TLS_PSK_WITH_CHACHA20_POLY1305_SHA256 = b'\xCC\xAB' # [RFC7905]
    TLS_PSK_WITH_NULL_SHA = b'\x00\x2C' # [RFC4785]
    TLS_PSK_WITH_NULL_SHA256 = b'\x00\xB0' # [RFC5487]
    TLS_PSK_WITH_NULL_SHA384 = b'\x00\xB1' # [RFC5487]
    TLS_PSK_WITH_RC4_128_SHA = b'\x00\x8A' # [RFC4279][RFC6347]
    TLS_RSA_EXPORT_WITH_DES40_CBC_SHA = b'\x00\x08' # [RFC4346]
    TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5 = b'\x00\x06' # [RFC4346]
    TLS_RSA_EXPORT_WITH_RC4_40_MD5 = b'\x00\x03' # [RFC4346][RFC6347]
    TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA = b'\x00\x93' # [RFC4279]
    TLS_RSA_PSK_WITH_AES_128_CBC_SHA = b'\x00\x94' # [RFC4279]
    TLS_RSA_PSK_WITH_AES_128_CBC_SHA256 = b'\x00\xB6' # [RFC5487]
    TLS_RSA_PSK_WITH_AES_128_GCM_SHA256 = b'\x00\xAC' # [RFC5487]
    TLS_RSA_PSK_WITH_AES_256_CBC_SHA = b'\x00\x95' # [RFC4279]
    TLS_RSA_PSK_WITH_AES_256_CBC_SHA384 = b'\x00\xB7' # [RFC5487]
    TLS_RSA_PSK_WITH_AES_256_GCM_SHA384 = b'\x00\xAD' # [RFC5487]
    TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x68' # [RFC6209]
    TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x6E' # [RFC6209]
    TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x69' # [RFC6209]
    TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x6F' # [RFC6209]
    TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256 = b'\xC0\x98' # [RFC6367]
    TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x92' # [RFC6367]
    TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384 = b'\xC0\x99' # [RFC6367]
    TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x93' # [RFC6367]
    TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256 = b'\xCC\xAE' # [RFC7905]
    TLS_RSA_PSK_WITH_NULL_SHA = b'\x00\x2E' # [RFC4785]
    TLS_RSA_PSK_WITH_NULL_SHA256 = b'\x00\xB8' # [RFC5487]
    TLS_RSA_PSK_WITH_NULL_SHA384 = b'\x00\xB9' # [RFC5487]
    TLS_RSA_PSK_WITH_RC4_128_SHA = b'\x00\x92' # [RFC4279][RFC6347]
    TLS_RSA_WITH_3DES_EDE_CBC_SHA = b'\x00\x0A' # [RFC5246]
    TLS_RSA_WITH_AES_128_CBC_SHA = b'\x00\x2F' # [RFC5246]
    TLS_RSA_WITH_AES_128_CBC_SHA256 = b'\x00\x3C' # [RFC5246]
    TLS_RSA_WITH_AES_128_CCM = b'\xC0\x9C' # [RFC6655]
    TLS_RSA_WITH_AES_128_CCM_8 = b'\xC0\xA0' # [RFC6655]
    TLS_RSA_WITH_AES_128_GCM_SHA256 = b'\x00\x9C' # [RFC5288]
    TLS_RSA_WITH_AES_256_CBC_SHA = b'\x00\x35' # [RFC5246]
    TLS_RSA_WITH_AES_256_CBC_SHA256 = b'\x00\x3D' # [RFC5246]
    TLS_RSA_WITH_AES_256_CCM = b'\xC0\x9D' # [RFC6655]
    TLS_RSA_WITH_AES_256_CCM_8 = b'\xC0\xA1' # [RFC6655]
    TLS_RSA_WITH_AES_256_GCM_SHA384 = b'\x00\x9D' # [RFC5288]
    TLS_RSA_WITH_ARIA_128_CBC_SHA256 = b'\xC0\x3C' # [RFC6209]
    TLS_RSA_WITH_ARIA_128_GCM_SHA256 = b'\xC0\x50' # [RFC6209]
    TLS_RSA_WITH_ARIA_256_CBC_SHA384 = b'\xC0\x3D' # [RFC6209]
    TLS_RSA_WITH_ARIA_256_GCM_SHA384 = b'\xC0\x51' # [RFC6209]
    TLS_RSA_WITH_CAMELLIA_128_CBC_SHA = b'\x00\x41' # [RFC5932]
    TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256 = b'\x00\xBA' # [RFC5932]
    TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256 = b'\xC0\x7A' # [RFC6367]
    TLS_RSA_WITH_CAMELLIA_256_CBC_SHA = b'\x00\x84' # [RFC5932]
    TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256 = b'\x00\xC0' # [RFC5932]
    TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384 = b'\xC0\x7B' # [RFC6367]
    TLS_RSA_WITH_DES_CBC_SHA = b'\x00\x09' # [RFC8996]
    TLS_RSA_WITH_IDEA_CBC_SHA = b'\x00\x07' # [RFC8996]
    TLS_RSA_WITH_NULL_MD5 = b'\x00\x01' # [RFC5246]
    TLS_RSA_WITH_NULL_SHA = b'\x00\x02' # [RFC5246]
    TLS_RSA_WITH_NULL_SHA256 = b'\x00\x3B' # [RFC5246]
    TLS_RSA_WITH_RC4_128_MD5 = b'\x00\x04' # [RFC5246][RFC6347]
    TLS_RSA_WITH_RC4_128_SHA = b'\x00\x05' # [RFC5246][RFC6347]
    TLS_RSA_WITH_SEED_CBC_SHA = b'\x00\x96' # [RFC4162]
    TLS_SHA256_SHA256 = b'\xC0\xB4' # [RFC9150]
    TLS_SHA384_SHA384 = b'\xC0\xB5' # [RFC9150]
    TLS_SM4_CCM_SM3 = b'\x00\xC7' # [RFC8998]
    TLS_SM4_GCM_SM3 = b'\x00\xC6' # [RFC8998]
    TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA = b'\xC0\x1C' # [RFC5054]
    TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA = b'\xC0\x1F' # [RFC5054]
    TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA = b'\xC0\x22' # [RFC5054]
    TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA = b'\xC0\x1B' # [RFC5054]
    TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA = b'\xC0\x1E' # [RFC5054]
    TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA = b'\xC0\x21' # [RFC5054]
    TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA = b'\xC0\x1A' # [RFC5054]
    TLS_SRP_SHA_WITH_AES_128_CBC_SHA = b'\xC0\x1D' # [RFC5054]
    TLS_SRP_SHA_WITH_AES_256_CBC_SHA = b'\xC0\x20' # [RFC5054]

class AlertLevel(Enum):
    """ Different alert levels that can be sent by the server. """
    WARNING = b'\x01'
    FATAL = b'\x02'

class AlertDescription(Enum):
    """ Different alert messages that can be sent by the server. """
    close_notify = b'\x00'
    unexpected_message = b'\x0a'
    bad_record_mac = b'\x14'
    record_overflow = b'\x16'
    handshake_failure = b'\x28'
    bad_certificate = b'\x2a'
    unsupported_certificate = b'\x2b'
    certificate_revoked = b'\x2c'
    certificate_expired = b'\x2d'
    certificate_unknown = b'\x2e'
    illegal_parameter = b'\x2f'
    unknown_ca = b'\x30'
    access_denied = b'\x31'
    decode_error = b'\x32'
    decrypt_error = b'\x33'
    protocol_version = b'\x46'
    insufficient_security = b'\x47'
    internal_error = b'\x50'
    inappropriate_fallback = b'\x56'
    user_canceled = b'\x5a'
    missing_extension = b'\x6d'
    unsupported_extension = b'\x6e'
    unrecognized_name = b'\x70'
    bad_certificate_status_response = b'\x71'
    unknown_psk_identity = b'\x73'
    certificate_required = b'\x74'
    no_application_protocol = b'\x78'

class ExtensionType(Enum):
    server_name = b'\x00\x00'
    max_fragment_length = b'\x00\x01'
    client_certificate_url = b'\x00\x02'
    trusted_ca_keys = b'\x00\x03'
    truncated_hmac = b'\x00\x04'
    status_request = b'\x00\x05'
    user_mapping = b'\x00\x06'
    client_authz = b'\x00\x07'
    server_authz = b'\x00\x08'
    cert_type = b'\x00\x09'
    supported_groups = b'\x00\x0a'
    ec_point_formats = b'\x00\x0b'
    srp = b'\x00\x0c'
    signature_algorithms = b'\x00\x0d'
    use_srtp = b'\x00\x0e'
    heartbeat = b'\x00\x0f'
    application_layer_protocol_negotiation = b'\x00\x10'
    status_request_v2 = b'\x00\x11'
    signed_certificate_timestamp = b'\x00\x12'
    client_certificate_type = b'\x00\x13'
    server_certificate_type = b'\x00\x14'
    padding = b'\x00\x15'
    encrypt_then_mac = b'\x00\x16'
    extended_master_secret = b'\x00\x17'
    token_binding = b'\x00\x18'
    cached_info = b'\x00\x19'
    tls_lts = b'\x00\x1a'
    compress_certificate = b'\x00\x1b'
    record_size_limit = b'\x00\x1c'
    pwd_protect = b'\x00\x1d'
    pwd_clear = b'\x00\x1e'
    password_salt = b'\x00\x1f'
    ticket_pinning = b'\x00\x20'
    tls_cert_with_extern_psk = b'\x00\x21'
    delegated_credential = b'\x00\x22'
    session_ticket = b'\x00\x23'
    TLMSP = b'\x00\x24'
    TLMSP_proxying = b'\x00\x25'
    TLMSP_delegate = b'\x00\x26'
    supported_ekt_ciphers = b'\x00\x27'
    pre_shared_key = b'\x00\x29'
    early_data = b'\x00\x2a'
    supported_versions = b'\x00\x2b'
    cookie = b'\x00\x2c'
    psk_key_exchange_modes = b'\x00\x2d'
    certificate_authorities = b'\x00\x2f'
    oid_filters = b'\x00\x30'
    post_handshake_auth = b'\x00\x31'
    signature_algorithms_cert = b'\x00\x32'
    key_share = b'\x00\x33'
    transparency_info = b'\x00\x34'
    connection_id_deprecated = b'\x00\x35'
    connection_id = b'\x00\x36'
    external_id_hash = b'\x00\x37'
    external_session_id = b'\x00\x38'
    quic_transport_parameters = b'\x00\x39'
    ticket_request = b'\x00\x3a'
    dnssec_chain = b'\x00\x3b'
    sequence_number_encryption_algorithms = b'\x00\x3c'